/*!\file Numericalflux.c
 * \brief: implementation of the Numericalflux object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../objects.h"
/*}}}*/	

/*Load macros*/
#define NUMVERTICES_INTERNAL 4
#define NUMVERTICES_BOUNDARY 2
#define NDOF1 1

/*Numericalflux constructors and destructor*/
/*FUNCTION Numericalflux::Numericalflux(){{{1*/
Numericalflux::Numericalflux(){
	this->inputs=NULL;
	this->parameters=NULL;
	this->helement=NULL;
	this->element=NULL;
	this->hnodes=NULL;
	this->nodes=NULL;
}
/*}}}*/
/*}}}*//*FUNCTION Numericalflux::Numericalflux(int id, int i, IoModel* iomodel, int analysis_type) {{{1*/
Numericalflux::Numericalflux(int numericalflux_id,int i, IoModel* iomodel, int in_analysis_type){

	/* Intermediary */
	int  e1,e2;
	int  i1,i2;
	int  j;
	int  pos1,pos2,pos3,pos4;
	int  num_nodes;
	int  num_elems;

	/*numericalflux constructor data: */
	int   numericalflux_elem_ids[2];
	int   numericalflux_mparid;
	int   numericalflux_node_ids[4];
	int   numericalflux_type;

	/* Get MatPar id */
	numericalflux_mparid=iomodel->numberofelements+1; //matlab indexing

	/*First, see wether this is an internal or boundary edge (if e2=NaN)*/
	if (isnan((double)iomodel->edges[4*i+3])){ //edges are [node1 node2 elem1 elem2]
		/* Boundary edge, only one element */
		e1=(int)iomodel->edges[4*i+2];
		e2=(int)UNDEF;
		num_elems=1;
		num_nodes=2;
		numericalflux_type=BoundaryEnum;
		numericalflux_elem_ids[0]=(int)e1;
	}
	else{
		/* internal edge: connected to 2 elements */
		e1=(int)iomodel->edges[4*i+2];
		e2=(int)iomodel->edges[4*i+3];
		num_elems=2;
		num_nodes=4;
		numericalflux_type=InternalEnum;
		numericalflux_elem_ids[0]=(int)e1;
		numericalflux_elem_ids[1]=(int)e2;
	}

	/*1: Get vertices ids*/
	i1=(int)iomodel->edges[4*i+0];
	i2=(int)iomodel->edges[4*i+1];

	if (numericalflux_type==InternalEnum){

		/*Now, we must get the nodes of the 4 nodes located on the edge*/

		/*2: Get the column where these ids are located in the index*/
		pos1=pos2=pos3=pos4=UNDEF;
		for(j=0;j<3;j++){
			if (iomodel->elements[3*(e1-1)+j]==i1) pos1=j+1;
			if (iomodel->elements[3*(e1-1)+j]==i2) pos2=j+1;
			if (iomodel->elements[3*(e2-1)+j]==i1) pos3=j+1;
			if (iomodel->elements[3*(e2-1)+j]==i2) pos4=j+1;
		}
		ISSMASSERT(pos1!=UNDEF && pos2!=UNDEF && pos3!=UNDEF && pos4!=UNDEF);

		/*3: We have the id of the elements and the position of the vertices in the index
		 * we can compute their dofs!*/
		numericalflux_node_ids[0]=iomodel->nodecounter+3*(e1-1)+pos1;
		numericalflux_node_ids[1]=iomodel->nodecounter+3*(e1-1)+pos2;
		numericalflux_node_ids[2]=iomodel->nodecounter+3*(e2-1)+pos3;
		numericalflux_node_ids[3]=iomodel->nodecounter+3*(e2-1)+pos4;
	}
	else{

		/*2: Get the column where these ids are located in the index*/
		pos1=pos2=UNDEF;
		for(j=0;j<3;j++){
			if (iomodel->elements[3*(e1-1)+j]==i1) pos1=j+1;
			if (iomodel->elements[3*(e1-1)+j]==i2) pos2=j+1;
		}
		ISSMASSERT(pos1!=UNDEF && pos2!=UNDEF);

		/*3: We have the id of the elements and the position of the vertices in the index
		 * we can compute their dofs!*/
		numericalflux_node_ids[0]=iomodel->nodecounter+3*(e1-1)+pos1;
		numericalflux_node_ids[1]=iomodel->nodecounter+3*(e1-1)+pos2;
	}

	/*Ok, we have everything to build the object: */
	this->id=numericalflux_id;
	this->analysis_type=in_analysis_type;

	/*Hooks: */
	this->hnodes  =new Hook(numericalflux_node_ids,num_nodes);
	this->helement=new Hook(numericalflux_elem_ids,1); // take only the first element for now

	//intialize  and add as many inputs per element as requested: 
	this->inputs=new Inputs();
	this->inputs->AddInput(new IntInput(TypeEnum,numericalflux_type));

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;
	this->element=NULL;
	this->nodes=NULL;
}
/*}}}*/
/*FUNCTION Numericalflux::~Numericalflux(){{{1*/
Numericalflux::~Numericalflux(){
	delete inputs;
	this->parameters=NULL;
	delete helement;
	delete hnodes;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION Numericalflux::Echo {{{1*/
void Numericalflux::Echo(void){
	printf("Numericalflux:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumToString(analysis_type));
	hnodes->Echo();
	helement->Echo();
	printf("   parameters: %p\n",parameters);
	printf("   inputs: %p\n",inputs);
}
/*}}}*/
/*FUNCTION Numericalflux::DeepEcho {{{1*/
void Numericalflux::DeepEcho(void){

	printf("Numericalflux:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumToString(analysis_type));
	hnodes->DeepEcho();
	helement->DeepEcho();
	printf("   parameters\n");
	if(parameters)
	 parameters->DeepEcho();
	else
	 printf("      NULL\n");
	printf("   inputs\n");
	inputs->DeepEcho();
	
}		
/*}}}*/
/*FUNCTION Numericalflux::Id {{{1*/
int    Numericalflux::Id(void){
	return id;
}
/*}}}*/
/*FUNCTION Numericalflux::MyRank {{{1*/
int    Numericalflux::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Numericalflux::Marshall {{{1*/
void  Numericalflux::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Numericalflux: */
	enum_type=NumericalfluxEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Numericalflux data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&analysis_type,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);

	/*Marshall hooks: */
	hnodes->Marshall(&marshalled_dataset);
	helement->Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::MarshallSize{{{1*/
int   Numericalflux::MarshallSize(){

	return sizeof(id)
		+sizeof(analysis_type)
		+hnodes->MarshallSize()
		+helement->MarshallSize()
		+inputs->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Numericalflux::Demarshall {{{1*/
void  Numericalflux::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&analysis_type,marshalled_dataset,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);

	/*demarshall hooks: */
	hnodes=new Hook(); hnodes->Demarshall(&marshalled_dataset);
	helement=new Hook(); helement->Demarshall(&marshalled_dataset);
	
	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;
	this->element=NULL;
	this->nodes=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::Enum {{{1*/
int Numericalflux::Enum(void){

	return NumericalfluxEnum;

}
/*}}}*/
/*FUNCTION Numericalflux::copy {{{1*/
Object* Numericalflux::copy() {
	
	Numericalflux* numericalflux=NULL;

	numericalflux=new Numericalflux();

	/*copy fields: */
	numericalflux->id=this->id;
	numericalflux->analysis_type=this->analysis_type;
	if(this->inputs){
		numericalflux->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		numericalflux->inputs=new Inputs();
	}
	/*point parameters: */
	numericalflux->parameters=this->parameters;

	/*now deal with hooks and objects: */
	numericalflux->hnodes=(Hook*)this->hnodes->copy();
	numericalflux->helement=(Hook*)this->helement->copy();

	/*corresponding fields*/
	numericalflux->nodes  =(Node**)numericalflux->hnodes->deliverp();
	numericalflux->element=(Element*)numericalflux->helement->delivers();

	return numericalflux;
}
/*}}}*/

/*Load virtual functions definitions:*/
/*FUNCTION Numericalflux::Configure {{{1*/
void  Numericalflux::Configure(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes->configure(nodesin);
	helement->configure(elementsin);

	/*Initialize hooked fields*/
	this->nodes  =(Node**)hnodes->deliverp();
	this->element=(Element*)helement->delivers();

	/*point parameters to real dataset: */
	this->parameters=parametersin;

}
/*}}}*/
/*FUNCTION Numericalflux::SetCurrentConfiguration {{{1*/
void  Numericalflux::SetCurrentConfiguration(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

}
/*}}}*/
/*FUNCTION Numericalflux::CreateKMatrix {{{1*/
void  Numericalflux::CreateKMatrix(Mat Kgg,Mat Kff, Mat Kfs){

	int type;
	int analysis_type;
	ElementMatrix* Ke=NULL;

	/*recover some parameters*/
	inputs->GetParameterValue(&type,TypeEnum);
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	switch(analysis_type){
		case PrognosticAnalysisEnum: case BalancedthicknessAnalysisEnum: case AdjointBalancedthicknessAnalysisEnum:
			switch(type){
				case InternalEnum:
					Ke=CreateKMatrixInternal();
					break;
				case BoundaryEnum:
					Ke=CreateKMatrixBoundary();
					break;
				default:
					ISSMERROR("type not supported yet");
			}
			break;
		default:
			ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}

	/*Add to global matrix*/
	if(Ke){
		Ke->AddToGlobal(Kgg,Kff,Kfs);
		delete Ke;
	}

}
/*}}}*/
/*FUNCTION Numericalflux::CreatePVector {{{1*/
void  Numericalflux::CreatePVector(Vec pg,Vec pf){

	int type;
	int analysis_type;
	ElementVector* pe=NULL;

	/*recover some parameters*/
	inputs->GetParameterValue(&type,TypeEnum);
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	switch(analysis_type){
		case PrognosticAnalysisEnum: case BalancedthicknessAnalysisEnum: case AdjointBalancedthicknessAnalysisEnum:
			switch(type){
				case InternalEnum:
					pe=CreatePVectorInternal();
					break;
				case BoundaryEnum:
					pe=CreatePVectorBoundary();
					break;
				default:
					ISSMERROR("type not supported yet");
			}
			break;
		default:
			ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}

	/*Add to global matrix*/
	if(pe){
		pe->AddToGlobal(pg,pf);
		delete pe;
	}

}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreateKMatrix {{{1*/
void  Numericalflux::PenaltyCreateKMatrix(Mat Kgg,Mat Kff, Mat Kfs,double kmax){

	/*No stiffness loads applied, do nothing: */
	return;

}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreatePVector{{{1*/
void  Numericalflux::PenaltyCreatePVector(Vec pg,Vec pf,double kmax){

	/*No penalty loads applied, do nothing: */
	return;

}
/*}}}*/
/*FUNCTION Numericalflux::InAnalysis{{{1*/
bool Numericalflux::InAnalysis(int in_analysis_type){
	if (in_analysis_type==this->analysis_type) return true;
	else return false;
}
/*}}}*/

/*Numericalflux management*/
/*FUNCTION Numericalflux::CreateKMatrixInternal {{{1*/
ElementMatrix* Numericalflux::CreateKMatrixInternal(void){

	/* constants*/
	const int numdof=NDOF1*NUMVERTICES_INTERNAL;

	/* Intermediaries*/
	int        i,j,ig,index1,index2,analysis_type;
	double     DL1,DL2,Jdet,dt,vx,vy,UdotN;
	double     xyz_list[NUMVERTICES_INTERNAL][3];
	double     normal[2];
	double     B[numdof];
	double     Bprime[numdof];
	double     Ke_g1[numdof][numdof];
	double     Ke_g2[numdof][numdof];
	GaussTria *gauss;

	/*Initialize Element matrix and return if necessary*/
	Tria*  tria=(Tria*)element;
	if(tria->IsOnWater()) return NULL;
	ElementMatrix* Ke=NewElementMatrix(nodes,NUMVERTICES_INTERNAL,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes,NUMVERTICES_INTERNAL);
	Input* vxaverage_input=tria->inputs->GetInput(VxEnum);
	Input* vyaverage_input=tria->inputs->GetInput(VyEnum);
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	GetNormal(&normal[0],xyz_list);
	switch(analysis_type){
		case PrognosticAnalysisEnum:
			parameters->FindParam(&dt,DtEnum); break;
		case BalancedthicknessAnalysisEnum: case AdjointBalancedthicknessAnalysisEnum:
			dt=1; break;/*No transient term is involved*/
		default:
			ISSMERROR("analysis_type %s not supported yet",EnumToString(analysis_type));
	}

	/* Start  looping on the number of gaussian points: */
	index1=tria->GetNodeIndex(nodes[0]);
	index2=tria->GetNodeIndex(nodes[1]);
	gauss=new GaussTria(index1,index2,2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		tria->GetSegmentBFlux(&B[0],gauss,index1,index2);
		tria->GetSegmentBprimeFlux(&Bprime[0],gauss,index1,index2);

		vxaverage_input->GetParameterValue(&vx,gauss);
		vyaverage_input->GetParameterValue(&vy,gauss);
		UdotN=vx*normal[0]+vy*normal[1];
		tria->GetSegmentJacobianDeterminant(&Jdet,&xyz_list[0][0],gauss);
		DL1=gauss->weight*Jdet*dt*UdotN/2;
		DL2=gauss->weight*Jdet*dt*fabs(UdotN)/2;

		TripleMultiply(&B[0],1,numdof,1,
					&DL1,1,1,0,
					&Bprime[0],1,numdof,0,
					&Ke_g1[0][0],0);
		TripleMultiply(&B[0],1,numdof,1,
					&DL2,1,1,0,
					&B[0],1,numdof,0,
					&Ke_g2[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_g1[i][j];
		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_g2[i][j];
	}
	
	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Numericalflux::CreateKMatrixBoundary {{{1*/
ElementMatrix* Numericalflux::CreateKMatrixBoundary(void){

	/* constants*/
	const int numdof=NDOF1*NUMVERTICES_BOUNDARY;

	/* Intermediaries*/
	int        i,j,ig,index1,index2,analysis_type;
	double     DL,Jdet,dt,vx,vy,mean_vx,mean_vy,UdotN;
	double     xyz_list[NUMVERTICES_BOUNDARY][3];
	double     normal[2];
	double     L[numdof];
	double     Ke_g[numdof][numdof];
	GaussTria *gauss;

	/*Initialize Element matrix and return if necessary*/
	Tria*  tria=(Tria*)element;
	if(tria->IsOnWater()) return NULL;
	ElementMatrix* Ke=NewElementMatrix(nodes,NUMVERTICES_BOUNDARY,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0],nodes,NUMVERTICES_BOUNDARY);
	Input* vxaverage_input=tria->inputs->GetInput(VxEnum);
	Input* vyaverage_input=tria->inputs->GetInput(VyEnum);
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	GetNormal(&normal[0],xyz_list);
	switch(analysis_type){
		case PrognosticAnalysisEnum:
			parameters->FindParam(&dt,DtEnum); break;
		case BalancedthicknessAnalysisEnum: case AdjointBalancedthicknessAnalysisEnum:
			dt=1; break;/*No transient term is involved*/
		default:
			ISSMERROR("analysis_type %s not supported yet",EnumToString(analysis_type));
	}

	/*Check wether it is an inflow or outflow BC (0 is the middle of the segment)*/
	index1=tria->GetNodeIndex(nodes[0]);
	index2=tria->GetNodeIndex(nodes[1]);

	gauss=new GaussTria();
	gauss->GaussEdgeCenter(index1,index2);
	vxaverage_input->GetParameterValue(&mean_vx,gauss);
	vyaverage_input->GetParameterValue(&mean_vy,gauss);
	delete gauss;

	UdotN=mean_vx*normal[0]+mean_vy*normal[1];
	if (UdotN<=0){
		/*(u,n)<0 -> inflow, PenaltyCreatePVector will take care of it*/
		return NULL;
	}

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(index1,index2,2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		tria->GetSegmentNodalFunctions(&L[0],gauss,index1,index2);

		vxaverage_input->GetParameterValue(&vx,gauss);
		vyaverage_input->GetParameterValue(&vy,gauss);
		UdotN=vx*normal[0]+vy*normal[1];
		tria->GetSegmentJacobianDeterminant(&Jdet,&xyz_list[0][0],gauss);
		DL=gauss->weight*Jdet*dt*UdotN;

		TripleMultiply(&L[0],1,numdof,1,
					&DL,1,1,0,
					&L[0],1,numdof,0,
					&Ke_g[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_g[i][j];
	} 

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Numericalflux::CreatePVectorInternal{{{1*/
ElementVector* Numericalflux::CreatePVectorInternal(void){

	/*Nothing added to PVector*/
	return NULL;

}
/*}}}*/
/*FUNCTION Numericalflux::CreatePVectorBoundary{{{1*/
ElementVector* Numericalflux::CreatePVectorBoundary(void){

	/* constants*/
	const int numdof=NDOF1*NUMVERTICES_BOUNDARY;

	/* Intermediaries*/
	int        i,j,ig,index1,index2,analysis_type;
	double     DL,Jdet,dt,vx,vy,mean_vx,mean_vy,UdotN,thickness;
	double     xyz_list[NUMVERTICES_BOUNDARY][3];
	double     normal[2];
	double     L[numdof];
	GaussTria *gauss;

	/*Initialize Load Vectorand return if necessary*/
	Tria*  tria=(Tria*)element;
	if(tria->IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES_BOUNDARY,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0],nodes,NUMVERTICES_BOUNDARY);
	Input* vxaverage_input=tria->inputs->GetInput(VxEnum); ISSMASSERT(vxaverage_input); 
	Input* vyaverage_input=tria->inputs->GetInput(VyEnum); ISSMASSERT(vyaverage_input);
	Input* thickness_input=tria->inputs->GetInput(ThicknessObsEnum);

	/*Here, as it is a forcing, we have H=Hobs by default (for control methods)*/
	if (!thickness_input){
		thickness_input=tria->inputs->GetInput(ThicknessEnum); ISSMASSERT(thickness_input);
	}

	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	GetNormal(&normal[0],xyz_list);
	switch(analysis_type){
		case PrognosticAnalysisEnum:
			parameters->FindParam(&dt,DtEnum); break;
		case BalancedthicknessAnalysisEnum: case AdjointBalancedthicknessAnalysisEnum:
			dt=1; break;/*No transient term is involved*/
		default:
			ISSMERROR("analysis_type %s not supported yet",EnumToString(analysis_type));
	}

	/*Check wether it is an inflow or outflow BC (0 is the middle of the segment)*/
	index1=tria->GetNodeIndex(nodes[0]);
	index2=tria->GetNodeIndex(nodes[1]);

	gauss=new GaussTria();
	gauss->GaussEdgeCenter(index1,index2);
	vxaverage_input->GetParameterValue(&mean_vx,gauss);
	vyaverage_input->GetParameterValue(&mean_vy,gauss);
	delete gauss;
	UdotN=mean_vx*normal[0]+mean_vy*normal[1];
	if (UdotN>0){
		/*(u,n)>0 -> outflow, PenaltyCreateKMatrix will take care of it*/
		return NULL;
	}

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(index1,index2,2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		tria->GetSegmentNodalFunctions(&L[0],gauss,index1,index2);

		vxaverage_input->GetParameterValue(&vx,gauss);
		vyaverage_input->GetParameterValue(&vy,gauss);
		thickness_input->GetParameterValue(&thickness,gauss);
		UdotN=vx*normal[0]+vy*normal[1];
		tria->GetSegmentJacobianDeterminant(&Jdet,&xyz_list[0][0],gauss);
		DL= - gauss->weight*Jdet*dt*UdotN*thickness;

		for(i=0;i<numdof;i++) pe->values[i] += DL*L[i];
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Numericalflux::GetNormal {{{1*/
void Numericalflux:: GetNormal(double* normal,double xyz_list[4][3]){

	/*Build unit outward pointing vector*/
	double vector[2];
	double norm;

	vector[0]=xyz_list[1][0] - xyz_list[0][0];
	vector[1]=xyz_list[1][1] - xyz_list[0][1];

	norm=sqrt(pow(vector[0],2.0)+pow(vector[1],2.0));

	normal[0]= + vector[1]/norm;
	normal[1]= - vector[0]/norm;
}
/*}}}*/
