/*!\file Penta.cpp
 * \brief: implementation of the Penta object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../Container/Container.h"
/*}}}*/

/*Element macros*/
#define NUMVERTICES   6
#define NUMVERTICES2D 3
#define NDOF1 1
#define NDOF2 2
#define NDOF3 3
#define NDOF4 4

/*Penta constructors and destructor*/
/*FUNCTION Penta::Penta(){{{1*/
Penta::Penta(){
	this->nodes=NULL;
	this->matice=NULL;
	this->matpar=NULL;
	this->neighbors=NULL;
	this->inputs=NULL;
	this->parameters=NULL;
	this->results=NULL;
}
/*}}}*/
/*FUNCTION Penta::~Penta(){{{1*/
Penta::~Penta(){
	delete inputs;
	delete results;
	this->parameters=NULL;
}
/*}}}*/
/*FUNCTION Penta::Penta(int id, int index, IoModel* iomodel,int nummodels) {{{1*/
Penta::Penta(int penta_id, int index, IoModel* iomodel,int nummodels)
	:PentaRef(nummodels)
	,PentaHook(nummodels,index+1,iomodel->numberofelements+1) //index+1: matice id, iomodel->numberofelements+1: matpar id
                                                                      { //i is the element index

	int i;
	int penta_elements_ids[2];

	/*Checks in debugging mode*/
	/*{{{2*/
	ISSMASSERT(iomodel->upperelements);
	ISSMASSERT(iomodel->lowerelements);
	/*}}}*/

	/*id: */
	this->id=penta_id;

	/*Build neighbors list*/
	if isnan(iomodel->upperelements[index]) penta_elements_ids[1]=this->id; //upper penta is the same penta
	else                                    penta_elements_ids[1]=(int)(iomodel->upperelements[index]);
	if isnan(iomodel->lowerelements[index]) penta_elements_ids[0]=this->id; //lower penta is the same penta
	else                                    penta_elements_ids[0]=(int)(iomodel->lowerelements[index]);
	this->InitHookNeighbors(penta_elements_ids);

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;

	/*intialize inputs and results: */
	this->inputs=new Inputs();
	this->results=new Results();
	
	/*initialize pointers:*/
	this->nodes=NULL;
	this->matice=NULL;
	this->matpar=NULL;
	this->neighbors=NULL;
}
/*}}}*/

/*Object virtual functions definitions: */
/*FUNCTION Penta::copy {{{1*/
Object* Penta::copy() {

	int i;

	Penta* penta=NULL;

	penta=new Penta();

	//deal with PentaRef mother class
	penta->element_type_list=(int*)xmalloc(this->numanalyses*sizeof(int));
	for(i=0;i<this->numanalyses;i++) penta->element_type_list[i]=this->element_type_list[i];

	//deal with PentaHook mother class
	penta->numanalyses=this->numanalyses;
	penta->hnodes=new Hook*[penta->numanalyses];
	for(i=0;i<penta->numanalyses;i++)penta->hnodes[i]=(Hook*)this->hnodes[i]->copy();
	penta->hmatice=(Hook*)this->hmatice->copy();
	penta->hmatpar=(Hook*)this->hmatpar->copy();
	penta->hneighbors=(Hook*)this->hneighbors->copy();

	/*deal with Penta  copy fields: */
	penta->id=this->id;
	if(this->inputs){
		penta->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		penta->inputs=new Inputs();
	}
	if(this->results){
		penta->results=(Results*)this->results->Copy();
	}
	else{
		penta->results=new Results();
	}
	/*point parameters: */
	penta->parameters=this->parameters;

	/*recover objects: */
	penta->nodes=(Node**)xmalloc(6*sizeof(Node*)); //we cannot rely on an analysis_counter to tell us which analysis_type we are running, so we just copy the nodes.
	for(i=0;i<6;i++)penta->nodes[i]=this->nodes[i];
	penta->matice=(Matice*)penta->hmatice->delivers();
	penta->matpar=(Matpar*)penta->hmatpar->delivers();
	penta->neighbors=(Penta**)penta->hneighbors->deliverp();

	return penta;
}
/*}}}*/
/*FUNCTION Penta::DeepEcho{{{1*/
void Penta::DeepEcho(void){

	int i;
	
	printf("Penta:\n");
	printf("   id: %i\n",id);
	nodes[0]->DeepEcho();
	nodes[1]->DeepEcho();
	nodes[2]->DeepEcho();
	nodes[3]->DeepEcho();
	nodes[4]->DeepEcho();
	nodes[5]->DeepEcho();
	matice->DeepEcho();
	matpar->DeepEcho();
	printf("   neighbor ids: %i-%i\n",neighbors[0]->Id(),neighbors[1]->Id());
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();
	printf("   results\n");
	results->DeepEcho();
	return;
}
/*}}}*/
/*FUNCTION Penta::Demarshall {{{1*/
void  Penta::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;
	int flaghook;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&numanalyses,marshalled_dataset,sizeof(numanalyses));marshalled_dataset+=sizeof(numanalyses);

	/*demarshall Ref: */
	this->element_type_list=(int*)xmalloc(this->numanalyses*sizeof(int));
	for(i=0;i<numanalyses;i++){ memcpy(&element_type_list[i],marshalled_dataset,sizeof(int));marshalled_dataset+=sizeof(int);}

	/*allocate dynamic memory: */
	this->hnodes=new Hook*[this->numanalyses];
	/*demarshall hooks: */
	for(i=0;i<numanalyses;i++){
		memcpy(&flaghook,marshalled_dataset,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
		if(flaghook){ // there is a hook so demarshall it
			hnodes[i]=new Hook();
			hnodes[i]->Demarshall(&marshalled_dataset);
		}
		else hnodes[i]=NULL; //There is no hook so it is NULL
	}
	hmatice=new Hook(); hmatice->Demarshall(&marshalled_dataset);
	hmatpar=new Hook(); hmatpar->Demarshall(&marshalled_dataset);
	hneighbors=new Hook(); hneighbors->Demarshall(&marshalled_dataset);

	/*pointers are garbage, until configuration is carried out: */
	nodes=NULL;
	matice=NULL;
	matpar=NULL;
	neighbors=NULL;
	
	/*demarshall inputs and results: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 
	results=(Results*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Penta::Echo{{{1*/

void Penta::Echo(void){
	this->DeepEcho();
}
/*}}}*/
/*FUNCTION Penta::Enum {{{1*/
int Penta::Enum(void){

	return PentaEnum;

}
/*}}}*/
/*FUNCTION Penta::Id {{{1*/
int    Penta::Id(void){
	return id; 
}
/*}}}*/
/*FUNCTION Penta::Marshall {{{1*/
void  Penta::Marshall(char** pmarshalled_dataset){

	int   i;
	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;
	char* marshalled_results=NULL;
	int   marshalled_results_size;
	int   flaghook; //to indicate if hook is NULL or exists

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Penta: */
	enum_type=PentaEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Penta data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&numanalyses,sizeof(numanalyses));marshalled_dataset+=sizeof(numanalyses);

	/*Mershall Ref: */
	for(i=0;i<numanalyses;i++){
		memcpy(marshalled_dataset,&element_type_list[i],sizeof(element_type_list[i]));marshalled_dataset+=sizeof(element_type_list[i]);
	}

	/*Marshall hooks: */
	for(i=0;i<numanalyses;i++){
		if(hnodes[i]){
			/*Set flag to 1 as there is a hook */
			flaghook=1;
			memcpy(marshalled_dataset,&flaghook,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
			hnodes[i]->Marshall(&marshalled_dataset);
		}
		else{
			/*Set flag to 0 and do not marshall flag as there is no Hook */
			flaghook=0;
			memcpy(marshalled_dataset,&flaghook,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
		}
	}
	hmatice->Marshall(&marshalled_dataset);
	hmatpar->Marshall(&marshalled_dataset);
	hneighbors->Marshall(&marshalled_dataset);

	/*Marshall inputs and results: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	marshalled_results_size=results->MarshallSize();
	marshalled_results=results->Marshall();
	memcpy(marshalled_dataset,marshalled_results,marshalled_results_size*sizeof(char));
	marshalled_dataset+=marshalled_results_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);
	xfree((void**)&marshalled_results);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Penta::MarshallSize {{{1*/
int   Penta::MarshallSize(){

	int i;
	int hnodes_size=0;;

	for(i=0;i<numanalyses;i++){
		hnodes_size+=sizeof(int); //Flag 0 or 1
		if (hnodes[i]) hnodes_size+=hnodes[i]->MarshallSize();
	}

	return sizeof(id)
		+hnodes_size
		+sizeof(numanalyses)
		+numanalyses*sizeof(int) //element_type_lists
		+hmatice->MarshallSize()
		+hmatpar->MarshallSize()
		+hneighbors->MarshallSize()
		+inputs->MarshallSize()
		+results->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Penta::MyRank {{{1*/
int    Penta::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/

/*Update virtual functions definitions: */
/*FUNCTION Penta::InputUpdateFromConstant(bool value, int name);{{{1*/
void  Penta::InputUpdateFromConstant(bool constant, int name){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new BoolInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromConstant(double value, int name);{{{1*/
void  Penta::InputUpdateFromConstant(double constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new DoubleInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromConstant(int value, int name);{{{1*/
void  Penta::InputUpdateFromConstant(int constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new IntInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolution {{{1*/
void  Penta::InputUpdateFromSolution(double* solution){

	int analysis_type;

	/*retreive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Just branch to the correct InputUpdateFromSolution generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		InputUpdateFromSolutionDiagnosticHoriz( solution);
	}
	else if (analysis_type==DiagnosticHutterAnalysisEnum){
		InputUpdateFromSolutionDiagnosticHutter( solution);
	}
	else if (analysis_type==DiagnosticVertAnalysisEnum){
		InputUpdateFromSolutionDiagnosticVert( solution);
	}
	else if (analysis_type==AdjointHorizAnalysisEnum){
		int approximation;
		inputs->GetParameterValue(&approximation,ApproximationEnum);
		if(approximation==StokesApproximationEnum || approximation==NoneApproximationEnum){
			InputUpdateFromSolutionAdjointStokes( solution);
		}
		else{
			InputUpdateFromSolutionAdjointHoriz( solution);
		}
	}
	else if (analysis_type==BedSlopeXAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,BedSlopeXEnum);
	}
	else if (analysis_type==BedSlopeYAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,BedSlopeYEnum);
	}
	else if (analysis_type==SurfaceSlopeXAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,SurfaceSlopeXEnum);
	}
	else if (analysis_type==SurfaceSlopeYAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,SurfaceSlopeYEnum);
	}
	else if (analysis_type==PrognosticAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,ThicknessEnum);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,ThicknessEnum);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,VelEnum);
	}
	else if (analysis_type==ThermalAnalysisEnum){
		InputUpdateFromSolutionThermal( solution);
	}
	else if (analysis_type==MeltingAnalysisEnum){
		InputUpdateFromSolutionOneDof(solution,MeltingRateEnum);
	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVector(double* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVector(double* vector, int name, int type){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*Penta update B in InputUpdateFromSolutionThermal, so don't look for B update here.*/

	switch(type){

		case VertexEnum:

			/*New PentaVertexInpu*/
			double values[6];

			/*Get values on the 6 vertices*/
			for (int i=0;i<6;i++){
				values[i]=vector[this->nodes[i]->GetVertexDof()];
			}

			/*update input*/
			this->inputs->AddInput(new PentaVertexInput(name,values));
			return;

		default:

			ISSMERROR("type %i (%s) not implemented yet",type,EnumToString(type));
	}
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVector(int* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVector(int* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVector(bool* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVector(bool* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVectorDakota(double* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVectorDakota(double* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVectorDakota(int* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVectorDakota(int* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVectorDakota(bool* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVectorDakota(bool* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/

/*Element virtual functions definitions: */
/*FUNCTION Penta::AverageOntoPartition {{{1*/
void  Penta::AverageOntoPartition(Vec partition_contributions,Vec partition_areas,double* vertex_response,double* qmu_part){
	ISSMERROR("Not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::ComputeBasalStress {{{1*/
void  Penta::ComputeBasalStress(Vec sigma_b){

	int i,j;
	int    doflist[NUMVERTICES];
	double xyz_list[NUMVERTICES][3];
	double xyz_list_tria[3][3];

	/*Parameters*/
	double rho_ice,gravity;
	double bed_normal[3];
	double bed;
	double basalforce[3];
	double epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double devstresstensor[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double stresstensor[6]={0.0};
	double viscosity;
	int analysis_type;

	int  dofv[3]={0,1,2};
	int  dofp[1]={3};
	double Jdet2d;

	/*Gauss*/
	int     ig;
	GaussPenta* gauss;

	/*Output*/
	double pressure;
	double sigma_xx,sigma_yy,sigma_zz;
	double sigma_xy,sigma_xz,sigma_yz;
	double surface=0;
	double value=0;
	
	/*flags: */
	int  approximation;

	/*parameters: */
	double stokesreconditioning;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*Check analysis_types*/
	if (analysis_type!=DiagnosticHorizAnalysisEnum) ISSMERROR("Not supported yet!");
	if (approximation!=StokesApproximationEnum) ISSMERROR("Not supported yet!");

	/*retrieve some parameters: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	
	if(!IsOnBed()){
		//put zero
		VecSetValue(sigma_b,id-1,0.0,INSERT_VALUES);
		return;
	}

	/*recovre material parameters: */
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	for(i=0;i<3;i++){
		for(j=0;j<3;j++){
			xyz_list_tria[i][j]=xyz_list[i][j];
		}
	}

	/*Retrieve all inputs we will be needing: */
	Input* pressure_input=inputs->GetInput(PressureEnum); ISSMASSERT(pressure_input);
	Input* vx_input=inputs->GetInput(VxEnum);             ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);             ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);             ISSMASSERT(vz_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(0,1,2,2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Compute strain rate viscosity and pressure: */
		this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);
		matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);
		pressure_input->GetParameterValue(&pressure,gauss);

		/*Compute Stress*/
		sigma_xx=2*viscosity*epsilon[0]-pressure*stokesreconditioning; // sigma = nu eps - pressure
		sigma_yy=2*viscosity*epsilon[1]-pressure*stokesreconditioning;
		sigma_zz=2*viscosity*epsilon[2]-pressure*stokesreconditioning;
		sigma_xy=2*viscosity*epsilon[3];
		sigma_xz=2*viscosity*epsilon[4];
		sigma_yz=2*viscosity*epsilon[5];

		/*Get normal vector to the bed */
		BedNormal(&bed_normal[0],xyz_list_tria);

		/*basalforce*/
		basalforce[0] += sigma_xx*bed_normal[0] + sigma_xy*bed_normal[1] + sigma_xz*bed_normal[2];
		basalforce[1] += sigma_xy*bed_normal[0] + sigma_yy*bed_normal[1] + sigma_yz*bed_normal[2];
		basalforce[2] += sigma_xz*bed_normal[0] + sigma_yz*bed_normal[1] + sigma_zz*bed_normal[2];

		GetTriaJacobianDeterminant(&Jdet2d, &xyz_list_tria[0][0],gauss);
		value+=sigma_zz*Jdet2d*gauss->weight;
		surface+=Jdet2d*gauss->weight;
	}
	value=value/surface;

	/*Add value to output*/
	VecSetValue(sigma_b,id-1,(const double)value,INSERT_VALUES);
}
/*}}}*/
/*FUNCTION Penta::ComputeStrainRate {{{1*/
void  Penta::ComputeStrainRate(Vec eps){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
		/*FUNCTION Penta::Configure {{{1*/
void  Penta::Configure(Elements* elementsin, Loads* loadsin, DataSet* nodesin, Materials* materialsin, Parameters* parametersin){

	int analysis_counter;
	
	/*go into parameters and get the analysis_counter: */
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Get Element type*/
	this->element_type=this->element_type_list[analysis_counter];

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	if (this->hnodes[analysis_counter]) this->hnodes[analysis_counter]->configure(nodesin);
	this->hmatice->configure(materialsin);
	this->hmatpar->configure(materialsin);
	this->hneighbors->configure(elementsin);

	/*Now, go pick up the objects inside the hooks: */
	if (this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;
	this->matice=(Matice*)this->hmatice->delivers();
	this->matpar=(Matpar*)this->hmatpar->delivers();
	this->neighbors=(Penta**)this->hneighbors->deliverp();

	/*point parameters to real dataset: */
	this->parameters=parametersin;
}
/*}}}*/
/*FUNCTION Penta::RegularizeInversion {{{1*/
double Penta::RegularizeInversion(void){

	double J;
	Tria* tria=NULL;

	/*recover some inputs: */
	int  approximation;
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/*Bail out if this element if:
	 * -> Not MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !IsOnSurface()) || (approximation==MacAyealApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute RegularizeInversion*/

		/*Depth Average B*/
		this->InputDepthAverageAtBase(RheologyBEnum,RheologyBbarEnum,MaterialsEnum);

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->RegularizeInversion();
		delete tria->matice; delete tria;

		/*delete B average*/
		this->matice->inputs->DeleteInput(RheologyBbarEnum);

		return J;
	}
	else{

		/*Depth Average B and put it in inputs*/
		Penta* penta=GetBasalElement();
		penta->InputDepthAverageAtBase(RheologyBEnum,RheologyBbarEnum,MaterialsEnum);
		Input* B_input=penta->matice->inputs->GetInput(RheologyBbarEnum);
		Input* B_copy=(Input*)B_input->copy();
		this->matice->inputs->AddInput((Input*)B_copy);

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->RegularizeInversion();
		delete tria->matice; delete tria;

		/*delete B average*/
		this->matice->inputs->DeleteInput(RheologyBbarEnum);
		penta->matice->inputs->DeleteInput(RheologyBbarEnum);

		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrix {{{1*/
void  Penta::CreateKMatrix(Mat Kgg, Mat Kff, Mat Kfs){

	/*retrieve parameters: */
	int analysis_type;
	ElementMatrix* Ke=NULL;
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Checks in debugging {{{2*/
	ISSMASSERT(this->nodes && this->matice && this->matpar && this->neighbors && this->parameters && this->inputs);
	/*}}}*/

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	switch(analysis_type){
		case DiagnosticHorizAnalysisEnum: case AdjointHorizAnalysisEnum:
			Ke=CreateKMatrixDiagnosticHoriz();
			break;
		case DiagnosticHutterAnalysisEnum:
			Ke=CreateKMatrixDiagnosticHutter();
			break;
		case DiagnosticVertAnalysisEnum:
			Ke=CreateKMatrixDiagnosticVert();
			break;
		case BedSlopeXAnalysisEnum: case SurfaceSlopeXAnalysisEnum: case BedSlopeYAnalysisEnum: case SurfaceSlopeYAnalysisEnum:
			Ke=CreateKMatrixSlope();
			break;
		case PrognosticAnalysisEnum:
			Ke=CreateKMatrixPrognostic();
			break;
		case BalancedthicknessAnalysisEnum:
			Ke=CreateKMatrixBalancedthickness();
			break;
		case BalancedvelocitiesAnalysisEnum:
			Ke=CreateKMatrixBalancedvelocities();
			break;
		case ThermalAnalysisEnum:
			Ke=CreateKMatrixThermal();
			break;
		case MeltingAnalysisEnum:
			Ke=CreateKMatrixMelting();
			break;
		default:
			ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}

	/*Add to global matrix*/
	if(Ke){
		Ke->AddToGlobal(Kgg,Kff,Kfs);
		delete Ke;
	}
}
/*}}}*/
/*FUNCTION Penta::CreatePVector {{{1*/
void  Penta::CreatePVector(Vec pg, Vec pf){

	/*retrive parameters: */
	ElementVector* pe=NULL;
	int analysis_type;
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*if debugging mode, check that all pointers exist {{{2*/
	ISSMASSERT(this->nodes && this->matice && this->matpar && this->neighbors && this->parameters && this->inputs);
	/*}}}*/

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	switch(analysis_type){
		case DiagnosticHorizAnalysisEnum:
			pe=CreatePVectorDiagnosticHoriz();
			if(pe) pe->AddToGlobal(pg,pf);
			if(pe) delete pe;
			break;
		case AdjointHorizAnalysisEnum:
			pe=CreatePVectorAdjointHoriz();
			if(pe) pe->AddToGlobal(pg,pf);
			if(pe) delete pe;
			break;
		case DiagnosticHutterAnalysisEnum:
			pe=CreatePVectorDiagnosticHutter();
			if(pe) pe->AddToGlobal(pg,pf);
			if(pe) delete pe;
			break;
		case DiagnosticVertAnalysisEnum:
			CreatePVectorDiagnosticVert( pg);
			break;
		case BedSlopeXAnalysisEnum: case SurfaceSlopeXAnalysisEnum: case BedSlopeYAnalysisEnum: case SurfaceSlopeYAnalysisEnum:
			pe=CreatePVectorSlope();
			if(pe) pe->AddToGlobal(pg,pf);
			if(pe) delete pe;
			break;
		case PrognosticAnalysisEnum:
			CreatePVectorPrognostic( pg);
			break;
		case BalancedthicknessAnalysisEnum:
			CreatePVectorBalancedthickness( pg);
			break;
		case BalancedvelocitiesAnalysisEnum:
			CreatePVectorBalancedvelocities( pg);
			break;
		case ThermalAnalysisEnum:
			CreatePVectorThermal( pg);
			break;
		case MeltingAnalysisEnum:
			CreatePVectorMelting( pg);
			break;
		default:
			ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}

}
/*}}}*/
/*FUNCTION Penta::DeleteResults {{{1*/
void  Penta::DeleteResults(void){

	/*Delete and reinitialize results*/
	delete this->results;
	this->results=new Results();

}
/*}}}*/
/*FUNCTION Penta::GetNodeIndex {{{1*/
int Penta::GetNodeIndex(Node* node){

	ISSMASSERT(nodes);
	for(int i=0;i<NUMVERTICES;i++){
		if(node==nodes[i])
		 return i;
	}
	ISSMERROR("Node provided not found among element nodes");

}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputs{{{1*/
void  Penta::GetSolutionFromInputs(Vec solution){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Just branch to the correct InputUpdateFromSolution generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		int approximation;
		inputs->GetParameterValue(&approximation,ApproximationEnum);
		if(approximation==StokesApproximationEnum || approximation==NoneApproximationEnum){
			GetSolutionFromInputsDiagnosticStokes(solution);
		}
		else if (approximation==MacAyealApproximationEnum || approximation==PattynApproximationEnum || approximation==HutterApproximationEnum){
			GetSolutionFromInputsDiagnosticHoriz(solution);
		}
		else if (approximation==MacAyealPattynApproximationEnum || approximation==PattynStokesApproximationEnum){
			return; //the elements around will create the solution
		}
	}
	else if(analysis_type==DiagnosticHutterAnalysisEnum){
		GetSolutionFromInputsDiagnosticHutter(solution);
	}
	else if(analysis_type==DiagnosticVertAnalysisEnum){
		GetSolutionFromInputsDiagnosticVert(solution);
	}
	else if(analysis_type==ThermalAnalysisEnum){
		GetSolutionFromInputsThermal(solution);
	}
	else{
		ISSMERROR("analysis: %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}
}
/*}}}*/
/*FUNCTION Penta::GetVectorFromInputs{{{1*/
void  Penta::GetVectorFromInputs(Vec vector,int NameEnum){

	int i;
	int doflist1[NUMVERTICES];

	/*Find NameEnum input in the inputs dataset, and get it to fill in the vector: */
	for(i=0;i<this->inputs->Size();i++){
		Input* input=(Input*)this->inputs->GetObjectByOffset(i);
		if(input->EnumType()==NameEnum){
			/*We found the enum.  Use its values to fill into the vector, using the vertices ids: */
			this->GetDofList1(&doflist1[0]);
			input->GetVectorFromInputs(vector,&doflist1[0]);
			break;
		}
	}
}
/*}}}*/
/*FUNCTION Penta::Gradj {{{1*/
void  Penta::Gradj(Vec gradient,int control_type){

	/*If on water, skip grad (=0): */
	if(IsOnWater())return;

	if (control_type==DragCoefficientEnum){
		GradjDrag(gradient);
	}
	else if (control_type==RheologyBbarEnum){
		GradjB(gradient);
	}
	else ISSMERROR("control type %s not supported yet: ",EnumToString(control_type));
}
/*}}}*/
/*FUNCTION Penta::InputConvergence{{{1*/
bool Penta::InputConvergence(double* eps, int* enums,int num_enums,int* criterionenums,double* criterionvalues,int num_criterionenums){

	int i;
	Input** new_inputs=NULL;
	Input** old_inputs=NULL;
	bool    converged=true;

	new_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the new inputs
	old_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the old inputs

	for(i=0;i<num_enums/2;i++){
		new_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+0]);
		old_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+1]);
		if(!new_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumToString(enums[2*i+0]));
		if(!old_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumToString(enums[2*i+0]));
	}

	/*ok, we've got the inputs (new and old), now loop throught the number of criterions and fill the eps array:*/
	for(i=0;i<num_criterionenums;i++){
		IsInputConverged(eps+i,new_inputs,old_inputs,num_enums/2,criterionenums[i]);
		if(eps[i]>criterionvalues[i]) converged=false; 
	}

	/*clean up*/
	xfree((void**)&new_inputs);
	xfree((void**)&old_inputs);

	/*Return output*/
	return converged;

}
/*}}}*/
/*FUNCTION Penta::InputControlUpdate{{{1*/
void  Penta::InputControlUpdate(double scalar,bool save_parameter){

	/*Intermediary*/
	Input* input=NULL;
	double cm_min,cm_max;
	int    control_type;

	/*retrieve some parameters: */
	this->parameters->FindParam(&cm_min,CmMinEnum);
	this->parameters->FindParam(&cm_max,CmMaxEnum);
	this->parameters->FindParam(&control_type,ControlTypeEnum);

	/*Rheology*/
	if(control_type==RheologyBbarEnum){

		/*The update is done by the element on bed only*/
		if (!IsOnBed()) return;

		/*First, get revert to previous parameter value (kept in ControlParameter input)*/
		matice->inputs->DuplicateInput(ControlParameterEnum,RheologyBEnum);

		/*Now, update using the gradient new = old + scalar * gradient*/
		//matice->inputs->AXPY(RheologyBbarEnum,scalar,GradientEnum);
		// For now: Gradient is in element (TO BE CHANGED) and parameter in matice
		Input* input_B   =(Input*)matice->inputs->GetInput(RheologyBEnum); ISSMASSERT(input_B);
		Input* input_Grad=(Input*)this->inputs->GetInput(GradientEnum); ISSMASSERT(input_Grad);
		input_B->AXPY(input_Grad,scalar);

		/*Constrain constrain input*/
		input=(Input*)matice->inputs->GetInput(RheologyBEnum); ISSMASSERT(input);
		input->Constrain(cm_min,cm_max);

		/*OK, now we can propagate the new parameter up*/
		this->InputExtrude(RheologyBEnum,MaterialsEnum);

		/*Finally: save input if requested*/
		if (save_parameter){
			matice->inputs->DuplicateInput(RheologyBEnum,ControlParameterEnum);
			this->InputExtrude(ControlParameterEnum,MaterialsEnum);
		}

	}
	else if(control_type==DragCoefficientEnum){

		/*First, get revert to previous parameter value (kept in ControlParameter input)*/
		this->inputs->DuplicateInput(ControlParameterEnum,DragCoefficientEnum);

		/*Now, update using the gradient new = old + scalar * gradient*/
		this->inputs->AXPY(DragCoefficientEnum,scalar,GradientEnum);

		/*Constrain input*/
		input=(Input*)this->inputs->GetInput(DragCoefficientEnum); ISSMASSERT(input);
		input->Constrain(cm_min,cm_max);

		/*Finally: save input if requested*/
		if (save_parameter) inputs->DuplicateInput(DragCoefficientEnum,ControlParameterEnum);

	}
	else{
		ISSMERROR("control type %s not implemented yet",EnumToString(control_type));
	}
}
/*}}}*/
/*FUNCTION Penta::InputDepthAverageAtBase{{{1*/
void  Penta::InputDepthAverageAtBase(int enum_type,int average_enum_type,int object_enum){

	/*Intermediaries*/
	int  step,i;

	Penta* penta=NULL;
	Input* original_input=NULL;
	Input* element_integrated_input=NULL;
	Input* total_integrated_input=NULL;
	Input* element_thickness_input=NULL;
	Input* total_thickness_input=NULL;
	Input* depth_averaged_input=NULL;

	double  xyz_list[NUMVERTICES][3];
	double  Helem_list[NUMVERTICES];
	double  zeros_list[NUMVERTICES]={0.0};

	/*recover parameters: */

	/*Are we on the base? If not, return*/
	if(!IsOnBed()) return;

	/*OK, we are on bed. Initialize global inputs as 0*/
	total_thickness_input =new PentaVertexInput(ThicknessEnum,zeros_list);

	/*Now follow all the upper element from the base to the surface to integrate the input*/
	penta=this;
	step =0;
	for(;;){

		/*Step1: Get original input (to be depth avegaged): */
		if (object_enum==ElementsEnum)
		 original_input=(Input*)penta->inputs->GetInput(enum_type);
		else if (object_enum==MaterialsEnum)
		 original_input=(Input*)penta->matice->inputs->GetInput(enum_type);
		else
		 ISSMERROR("object %s not supported yet",EnumToString(object_enum));
		if(!original_input) ISSMERROR("could not find input with enum %s",EnumToString(enum_type));

		/*If first time, initialize total_integrated_input*/
		if (step==0){
			if (original_input->Enum()==PentaVertexInputEnum)
			 total_integrated_input=new PentaVertexInput(average_enum_type,zeros_list);
			else if (original_input->Enum()==DoubleInputEnum)
			 total_integrated_input=new DoubleInput(average_enum_type,0.0);
			else
			 ISSMERROR("object %s not supported yet",EnumToString(original_input->Enum()));
		}

		/*Step2: Create element thickness input*/
		GetVerticesCoordinates(&xyz_list[0][0],penta->nodes,NUMVERTICES);
		for(i=0;i<3;i++){
			Helem_list[i]=xyz_list[i+3][2]-xyz_list[i][2];
			Helem_list[i+3]=Helem_list[i];
		}
		element_thickness_input=new PentaVertexInput(ThicknessEnum,Helem_list);

		/*Step3: Vertically integrate A COPY of the original*/
		element_integrated_input=(Input*)original_input->copy();
		element_integrated_input->VerticallyIntegrate(element_thickness_input);

		/*Add contributions to global inputs*/
		total_integrated_input->AXPY(element_integrated_input,1.0);
		total_thickness_input ->AXPY(element_thickness_input,1.0);

		/*Clean up*/
		delete element_thickness_input;
		delete element_integrated_input;

		/*Stop if we have reached the surface, else, take upper penta*/
		if (penta->IsOnSurface()) break;

		/* get upper Penta*/
		penta=penta->GetUpperElement();
		ISSMASSERT(penta->Id()!=this->id);

		/*increase couter*/
		step++;
	}

	/*OK, now we only need to divide the depth integrated input by the total thickness!*/
	depth_averaged_input=total_integrated_input->PointwiseDivide(total_thickness_input);
	depth_averaged_input->ChangeEnum(average_enum_type);

	/*Clean up*/
	delete total_thickness_input;
	delete total_integrated_input;

	/*Finally, add to inputs*/
	if (object_enum==ElementsEnum)
	 this->inputs->AddInput((Input*)depth_averaged_input);
	else if (object_enum==MaterialsEnum)
	 this->matice->inputs->AddInput((Input*)depth_averaged_input);
	else
	 ISSMERROR("object %s not supported yet",EnumToString(object_enum));
}
/*}}}*/
/*FUNCTION Penta::InputDuplicate{{{1*/
void  Penta::InputDuplicate(int original_enum,int new_enum){

	/*Call inputs method*/
	if (IsInput(original_enum)) inputs->DuplicateInput(original_enum,new_enum);

}
/*}}}*/
/*FUNCTION Penta::InputArtificialNoise{{{1*/
void  Penta::InputArtificialNoise(int enum_type,double min,double max){

	Input* input=NULL;

	/*Make a copy of the original input: */
	input=(Input*)this->inputs->GetInput(enum_type);
	if(!input)ISSMERROR(" could not find old input with enum: %s",EnumToString(enum_type));

	/*ArtificialNoise: */
	input->ArtificialNoise(min,max);
}
/*}}}*/
/*FUNCTION Penta::InputScale{{{1*/
void  Penta::InputScale(int enum_type,double scale_factor){

	Input* input=NULL;

	/*Make a copy of the original input: */
	input=(Input*)this->inputs->GetInput(enum_type);
	if(!input)ISSMERROR(" could not find old input with enum: %s",EnumToString(enum_type));

	/*Scale: */
	input->Scale(scale_factor);
}
/*}}}*/
/*FUNCTION Penta::InputToResult{{{1*/
void  Penta::InputToResult(int enum_type,int step,double time){

	int    i;
	bool   found = false;
	Input *input = NULL;

	/*Go through all the input objects, and find the one corresponding to enum_type, if it exists: */
	if (enum_type==RheologyBbarEnum){
		input=this->matice->inputs->GetInput(RheologyBEnum);
	}
	else{
		input=this->inputs->GetInput(enum_type);
	}
	if (!input) ISSMERROR("Input %s not found in penta->inputs",EnumToString(enum_type));

	/*If we don't find it, no big deal, just don't do the transfer. Otherwise, build a new Result 
	 * object out of the input, with the additional step and time information: */
	this->results->AddObject((Object*)input->SpawnResult(step,time));

}
/*}}}*/
/*FUNCTION Penta::MassFlux {{{1*/
double Penta::MassFlux( double* segment,bool process_units){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::MaxAbsVx{{{1*/
void  Penta::MaxAbsVx(double* pmaxabsvx, bool process_units){

	/*Get maximum:*/
	double maxabsvx=this->inputs->MaxAbs(VxEnum);

	/*process units if requested: */
	if(process_units) maxabsvx=UnitConversion(maxabsvx,IuToExtEnum,VxEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxabsvx=maxabsvx;
}
/*}}}*/
/*FUNCTION Penta::MaxAbsVy{{{1*/
void  Penta::MaxAbsVy(double* pmaxabsvy, bool process_units){

	/*Get maximum:*/
	double maxabsvy=this->inputs->MaxAbs(VyEnum);

	/*process units if requested: */
	if(process_units) maxabsvy=UnitConversion(maxabsvy,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxabsvy=maxabsvy;
}
/*}}}*/
/*FUNCTION Penta::MaxAbsVz{{{1*/
void  Penta::MaxAbsVz(double* pmaxabsvz, bool process_units){

	/*Get maximum:*/
	double maxabsvz=this->inputs->MaxAbs(VzEnum);

	/*process units if requested: */
	if(process_units) maxabsvz=UnitConversion(maxabsvz,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxabsvz=maxabsvz;
}
/*}}}*/
/*FUNCTION Penta::MaxVel{{{1*/
void  Penta::MaxVel(double* pmaxvel, bool process_units){

	/*Get maximum:*/
	double maxvel=this->inputs->Max(VelEnum);

	/*process units if requested: */
	if(process_units) maxvel=UnitConversion(maxvel,IuToExtEnum,VelEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvel=maxvel;

}
/*}}}*/
/*FUNCTION Penta::MaxVx{{{1*/
void  Penta::MaxVx(double* pmaxvx, bool process_units){

	/*Get maximum:*/
	double maxvx=this->inputs->Max(VxEnum);

	/*process units if requested: */
	if(process_units) maxvx=UnitConversion(maxvx,IuToExtEnum,VxEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvx=maxvx;

}
/*}}}*/
/*FUNCTION Penta::MaxVy{{{1*/
void  Penta::MaxVy(double* pmaxvy, bool process_units){

	/*Get maximum:*/
	double maxvy=this->inputs->Max(VyEnum);

	/*process units if requested: */
	if(process_units) maxvy=UnitConversion(maxvy,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvy=maxvy;

}
/*}}}*/
/*FUNCTION Penta::MaxVz{{{1*/
void  Penta::MaxVz(double* pmaxvz, bool process_units){

	/*Get maximum:*/
	double maxvz=this->inputs->Max(VzEnum);

	/*process units if requested: */
	if(process_units) maxvz=UnitConversion(maxvz,IuToExtEnum,VzEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvz=maxvz;

}
/*}}}*/
/*FUNCTION Penta::MinVel{{{1*/
void  Penta::MinVel(double* pminvel, bool process_units){

	/*Get minimum:*/
	double minvel=this->inputs->Min(VelEnum);

	/*process units if requested: */
	if(process_units) minvel=UnitConversion(minvel,IuToExtEnum,VelEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvel=minvel;

}
/*}}}*/
/*FUNCTION Penta::MinVx{{{1*/
void  Penta::MinVx(double* pminvx, bool process_units){

	/*Get minimum:*/
	double minvx=this->inputs->Min(VxEnum);

	/*process units if requested: */
	if(process_units) minvx=UnitConversion(minvx,IuToExtEnum,VxEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvx=minvx;

}
/*}}}*/
/*FUNCTION Penta::MinVy{{{1*/
void  Penta::MinVy(double* pminvy, bool process_units){

	/*Get minimum:*/
	double minvy=this->inputs->Min(VyEnum);

	/*process units if requested: */
	if(process_units) minvy=UnitConversion(minvy,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvy=minvy;

}
/*}}}*/
/*FUNCTION Penta::MinVz{{{1*/
void  Penta::MinVz(double* pminvz, bool process_units){

	/*Get minimum:*/
	double minvz=this->inputs->Min(VzEnum);

	/*process units if requested: */
	if(process_units) minvz=UnitConversion(minvz,IuToExtEnum,VzEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvz=minvz;

}
/*}}}*/
/*FUNCTION Penta::ThicknessAbsMisfit {{{1*/
double Penta::ThicknessAbsMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(IsOnWater())return 0;
	ISSMERROR("Not implemented yet");

	tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
	J=tria->ThicknessAbsMisfit(process_units);
	delete tria->matice; delete tria;
	return J;
}
/*}}}*/
/*FUNCTION Penta::SurfaceAbsVelMisfit {{{1*/
double Penta::SurfaceAbsVelMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !IsOnSurface()) || (approximation==MacAyealApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceAbsVelMisfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->SurfaceAbsVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->SurfaceAbsVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceRelVelMisfit {{{1*/
double Penta::SurfaceRelVelMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !IsOnSurface()) || (approximation==MacAyealApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceRelVelMisfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->SurfaceRelVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->SurfaceRelVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceLogVelMisfit {{{1*/
double Penta::SurfaceLogVelMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !IsOnSurface()) || (approximation==MacAyealApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceLogVelMisfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->SurfaceLogVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->SurfaceLogVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceLogVxVyMisfit {{{1*/
double Penta::SurfaceLogVxVyMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !IsOnSurface()) || (approximation==MacAyealApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceLogVxVyMisfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->SurfaceLogVxVyMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->SurfaceLogVxVyMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceAverageVelMisfit {{{1*/
double Penta::SurfaceAverageVelMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !IsOnSurface()) || (approximation==MacAyealApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceAverageVelMisfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->SurfaceAverageVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->SurfaceAverageVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::PatchFill{{{1*/
void  Penta::PatchFill(int* pcount, Patch* patch){

	int i;
	int count;
	int vertices_ids[6];


	/*recover pointer: */
	count=*pcount;
		
	/*will be needed later: */
	for(i=0;i<6;i++) vertices_ids[i]=nodes[i]->GetVertexId(); //vertices id start at column 3 of the patch.

	for(i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);

		/*For this result,fill the information in the Patch object (element id + vertices ids), and then hand 
		 *it to the result object, to fill the rest: */
		patch->fillelementinfo(count,this->id,vertices_ids,6);
		elementresult->PatchFill(count,patch);

		/*increment counter: */
		count++;
	}

	/*Assign output pointers:*/
	*pcount=count;
}/*}}}*/
/*FUNCTION Penta::PatchSize{{{1*/
void  Penta::PatchSize(int* pnumrows, int* pnumvertices,int* pnumnodes){

	int     i;
	
	/*output: */
	int     numrows     = 0;
	int     numnodes    = 0;

	/*Go through all the results objects, and update the counters: */
	for (i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);
		/*first, we have one more result: */
		numrows++;
		/*now, how many vertices and how many nodal values for this result? :*/
		numnodes=elementresult->NumberOfNodalValues(); //ask result object.
	}

	/*Assign output pointers:*/
	*pnumrows=numrows;
	*pnumvertices=NUMVERTICES;
	*pnumnodes=numnodes;
	
}
/*}}}*/
/*FUNCTION Penta::ProcessResultsUnits{{{1*/
void  Penta::ProcessResultsUnits(void){

	int i;

	for(i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);
		elementresult->ProcessUnits(this->parameters);
	}

}
/*}}}*/
/*FUNCTION Penta::SetCurrentConfiguration {{{1*/
void  Penta::SetCurrentConfiguration(Elements* elementsin, Loads* loadsin, DataSet* nodesin, Materials* materialsin, Parameters* parametersin){

	int analysis_counter;

	/*go into parameters and get the analysis_counter: */
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Get Element type*/
	this->element_type=this->element_type_list[analysis_counter];

	/*Pick up nodes */
	if (this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;
}
/*}}}*/
/*FUNCTION Penta::SurfaceArea {{{1*/
double Penta::SurfaceArea(void){

	double S;
	Tria* tria=NULL;

	/*inputs: */
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !IsOnSurface()) || (approximation==MacAyealApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceArea*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		S=tria->SurfaceArea();
		delete tria->matice; delete tria;
		return S;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		S=tria->SurfaceArea();
		delete tria->matice; delete tria;
		return S;
	}
}
/*}}}*/
/*FUNCTION Penta::Update {{{1*/
void Penta::Update(int index,IoModel* iomodel,int analysis_counter,int analysis_type){ 

	/*Intermediaries*/
	IssmInt i;
	int     penta_type;
	int     penta_node_ids[6];
	int     penta_vertex_ids[6];
	double  nodeinputs[6];

	/*Checks if debuging*/
	/*{{{2*/
	ISSMASSERT(iomodel->elements);
	/*}}}*/

	/*Recover element type*/
	if ((analysis_type==PrognosticAnalysisEnum || analysis_type==BalancedthicknessAnalysisEnum) && iomodel->prognostic_DG){
		/*P1 Discontinuous Galerkin*/
		penta_type=P1DGEnum;
	}
	else if (analysis_type==StokesAnalysisEnum){
		/*MINI Continuous Galerkin*/
		penta_type=MiniEnum;
	}
	else{
		/*P1 Continuous Galerkin*/
		penta_type=P1Enum;
	}
	this->SetElementType(penta_type,analysis_counter);

	/*Recover vertices ids needed to initialize inputs*/
	for(i=0;i<6;i++){ 
		penta_vertex_ids[i]=(int)iomodel->elements[6*index+i]; //ids for vertices are in the elements array from Matlab
	}

	/*Recover nodes ids needed to initialize the node hook.*/
	for(i=0;i<6;i++){ 
		//go recover node ids, needed to initialize the node hook.
		//WARNING: We assume P1 elements here!!!!!
		penta_node_ids[i]=iomodel->nodecounter+(int)iomodel->elements[6*index+i]; //ids for vertices are in the elements array from Matlab
	}

	/*hooks: */
	this->SetHookNodes(penta_node_ids,analysis_counter); this->nodes=NULL; //set hook to nodes, for this analysis type

	//add as many inputs per element as requested: 
	if (iomodel->thickness) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->thickness[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(ThicknessEnum,nodeinputs));
	}
	if (iomodel->surface) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->surface[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(SurfaceEnum,nodeinputs));
	}
	if (iomodel->bed) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->bed[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(BedEnum,nodeinputs));
	}
	if (iomodel->drag_coefficient) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->drag_coefficient[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(DragCoefficientEnum,nodeinputs));

		if (iomodel->drag_p) this->inputs->AddInput(new DoubleInput(DragPEnum,iomodel->drag_p[index]));
		if (iomodel->drag_q) this->inputs->AddInput(new DoubleInput(DragQEnum,iomodel->drag_q[index]));
		this->inputs->AddInput(new IntInput(DragTypeEnum,iomodel->drag_type));

	}
	if (iomodel->control_parameter) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->control_parameter[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(ControlParameterEnum,nodeinputs));
	}
	if (iomodel->melting_rate) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->melting_rate[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(MeltingRateEnum,nodeinputs));
	}
	if (iomodel->accumulation_rate) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->accumulation_rate[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(AccumulationRateEnum,nodeinputs));
	}
	if (iomodel->geothermalflux) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->geothermalflux[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(GeothermalFluxEnum,nodeinputs));
	}	
	if (iomodel->pressure) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->pressure[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(PressureEnum,nodeinputs));
	}
	if (iomodel->temperature) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->temperature[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(TemperatureEnum,nodeinputs));
	}
	if (iomodel->dhdt) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->dhdt[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(DhDtEnum,nodeinputs));
	}
	/*vx,vy and vz: */
	if (iomodel->vx) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vx[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VxEnum,nodeinputs));
		this->inputs->AddInput(new PentaVertexInput(VxOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new PentaVertexInput(QmuVxEnum,nodeinputs));
	}
	if (iomodel->vy) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vy[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VyEnum,nodeinputs));
		this->inputs->AddInput(new PentaVertexInput(VyOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new PentaVertexInput(QmuVyEnum,nodeinputs));
	}
	if (iomodel->vz) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vz[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VzEnum,nodeinputs));
		this->inputs->AddInput(new PentaVertexInput(VzOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new PentaVertexInput(QmuVzEnum,nodeinputs));
	}
	if (iomodel->vx_obs) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vx_obs[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VxObsEnum,nodeinputs));
	}
	if (iomodel->vy_obs) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vy_obs[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VyObsEnum,nodeinputs));
	}
	if (iomodel->vz_obs) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vz_obs[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VzObsEnum,nodeinputs));
	}
	if (iomodel->weights) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->weights[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(WeightsEnum,nodeinputs));
	}
	if (iomodel->elementoniceshelf) this->inputs->AddInput(new BoolInput(ElementOnIceShelfEnum,(IssmBool)iomodel->elementoniceshelf[index]));
	if (iomodel->elementonbed) this->inputs->AddInput(new BoolInput(ElementOnBedEnum,(IssmBool)iomodel->elementonbed[index]));
	if (iomodel->elementonwater) this->inputs->AddInput(new BoolInput(ElementOnWaterEnum,(IssmBool)iomodel->elementonwater[index]));
	if (iomodel->elementonsurface) this->inputs->AddInput(new BoolInput(ElementOnSurfaceEnum,(IssmBool)iomodel->elementonsurface[index]));

	/*Defaults if not provided in iomodel*/
	switch(analysis_type){

		case DiagnosticHorizAnalysisEnum:

			/*default vx,vy and vz: either observation or 0 */
			if(!iomodel->vx){
				if (iomodel->vx_obs) for(i=0;i<6;i++)nodeinputs[i]=iomodel->vx_obs[penta_vertex_ids[i]-1]/iomodel->yts;
				else                 for(i=0;i<6;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new PentaVertexInput(VxEnum,nodeinputs));
				this->inputs->AddInput(new PentaVertexInput(VxOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new PentaVertexInput(QmuVxEnum,nodeinputs));
			}
			if(!iomodel->vy){
				if (iomodel->vy_obs) for(i=0;i<6;i++)nodeinputs[i]=iomodel->vy_obs[penta_vertex_ids[i]-1]/iomodel->yts;
				else                 for(i=0;i<6;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new PentaVertexInput(VyEnum,nodeinputs));
				this->inputs->AddInput(new PentaVertexInput(VyOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new PentaVertexInput(QmuVyEnum,nodeinputs));
			}
			if(!iomodel->vz){
				if (iomodel->vz_obs) for(i=0;i<6;i++)nodeinputs[i]=iomodel->vz_obs[penta_vertex_ids[i]-1]/iomodel->yts;
				else                 for(i=0;i<6;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new PentaVertexInput(VzEnum,nodeinputs));
				this->inputs->AddInput(new PentaVertexInput(VzOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new PentaVertexInput(QmuVzEnum,nodeinputs));
			}
			if(!iomodel->pressure){
				for(i=0;i<6;i++)nodeinputs[i]=0;
				if(iomodel->qmu_analysis){
					this->inputs->AddInput(new PentaVertexInput(PressureEnum,nodeinputs));
					this->inputs->AddInput(new PentaVertexInput(QmuPressureEnum,nodeinputs));
				}
				if(iomodel->isstokes){
					this->inputs->AddInput(new PentaVertexInput(PressureEnum,nodeinputs));
					this->inputs->AddInput(new PentaVertexInput(PressureOldEnum,nodeinputs));
				}
			}
			if(*(iomodel->elements_type+index)==PattynStokesApproximationEnum){
				/*Create VzPattyn and VzStokes Enums*/
				if(iomodel->vz && iomodel->gridonstokes){
					for(i=0;i<6;i++) nodeinputs[i]=iomodel->vz[penta_vertex_ids[i]-1]/iomodel->yts*iomodel->gridonstokes[penta_vertex_ids[i]-1];
					this->inputs->AddInput(new PentaVertexInput(VzStokesEnum,nodeinputs));
					for(i=0;i<6;i++) nodeinputs[i]=iomodel->vz[penta_vertex_ids[i]-1]/iomodel->yts*(1-iomodel->gridonstokes[penta_vertex_ids[i]-1]);
					this->inputs->AddInput(new PentaVertexInput(VzPattynEnum,nodeinputs));
				}
				else{
					for(i=0;i<6;i++)nodeinputs[i]=0;
					this->inputs->AddInput(new PentaVertexInput(VzStokesEnum,nodeinputs));
					this->inputs->AddInput(new PentaVertexInput(VzPattynEnum,nodeinputs));
				}
			}
			break;

		default:
			/*No update for other solution types*/
			break;

	}

	//Need to know the type of approximation for this element
	if(iomodel->elements_type){
		if (*(iomodel->elements_type+index)==MacAyealApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,MacAyealApproximationEnum));
		}
		else if (*(iomodel->elements_type+index)==PattynApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,PattynApproximationEnum));
		}
		else if (*(iomodel->elements_type+index)==MacAyealPattynApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,MacAyealPattynApproximationEnum));
		}
		else if (*(iomodel->elements_type+index)==HutterApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,HutterApproximationEnum));
		}
		else if (*(iomodel->elements_type+index)==StokesApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,StokesApproximationEnum));
		}
		else if (*(iomodel->elements_type+index)==PattynStokesApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,PattynStokesApproximationEnum));
		}
		else if (*(iomodel->elements_type+index)==NoneApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,NoneApproximationEnum));
		}
		else{
			ISSMERROR("Approximation type %s not supported yet",EnumToString((int)*(iomodel->elements_type+index)));
		}
	}

}
/*}}}*/
/*FUNCTION Penta::UpdateGeometry{{{1*/
void  Penta::UpdateGeometry(void){

	/*Intermediaries*/
	int i;
	double rho_ice,rho_water;

	/*Correct thickness*/
	this->inputs->ConstrainMin(ThicknessEnum,1.0);

	/*If shelf: hydrostatic equilibrium*/
	if (this->IsOnShelf()){

		/*recover material parameters: */
		rho_ice=matpar->GetRhoIce();
		rho_water=matpar->GetRhoWater();

		/*Create New Surface: s = (1-rho_ice/rho_water) h*/
		this->inputs->DuplicateInput(ThicknessEnum,SurfaceEnum);     //1: copy thickness into surface
		InputScale(SurfaceEnum,(1-rho_ice/rho_water)); //2: surface = surface * (1-di)

		/*Create New Bed b = -rho_ice/rho_water h*/
		this->inputs->DuplicateInput(ThicknessEnum,BedEnum);         //1: copy thickness into bed
		InputScale(BedEnum, -rho_ice/rho_water);       //2: bed = bed * (-di)
	}

	/*If sheet: surface = bed + thickness*/
	else{

		/*The bed does not change, update surface only s = b + h*/
		this->inputs->DuplicateInput(BedEnum,SurfaceEnum);          //1: copy bed into surface
		this->inputs->AXPY(SurfaceEnum,1.0,ThicknessEnum);     //2: surface = surface + 1 * thickness
	}

}
/*}}}*/

/*Penta specific routines: */
/*FUNCTION Penta::BedNormal {{{1*/
void Penta::BedNormal(double* bed_normal, double xyz_list[3][3]){

	int i;
	double v13[3];
	double v23[3];
	double normal[3];
	double normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	normal_norm=sqrt( pow(normal[0],2)+pow(normal[1],2)+pow(normal[2],2) );

	/*Bed normal is opposite to surface normal*/
	*(bed_normal)=-normal[0]/normal_norm;
	*(bed_normal+1)=-normal[1]/normal_norm;
	*(bed_normal+2)=-normal[2]/normal_norm;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixBalancedthickness {{{1*/
ElementMatrix* Penta::CreateKMatrixBalancedthickness(void){

	/*Figure out if this penta is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the stiffness matrix. */
	if (!IsOnBed() || IsOnWater()) return NULL;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Spawn Tria element from the base of the Penta: */
	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementMatrix* Ke=tria->CreateKMatrixBalancedthickness();
	delete tria->matice; delete tria;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	/*clean up and return*/
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixBalancedvelocities {{{1*/
ElementMatrix* Penta::CreateKMatrixBalancedvelocities(void){

	/*Figure out if this penta is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the stiffness matrix. */
	if (!IsOnBed() || IsOnWater()) return NULL;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Spawn Tria element from the base of the Penta: */
	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementMatrix* Ke=tria->CreateKMatrixBalancedvelocities();
	delete tria->matice; delete tria;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	/*clean up and return*/
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixCouplingMacAyealPattyn{{{1*/
ElementMatrix* Penta::CreateKMatrixCouplingMacAyealPattyn(void){

	/*compute all stiffness matrices for this element*/
	ElementMatrix* Ke1=CreateKMatrixCouplingMacAyealPattynViscous();
	ElementMatrix* Ke2=CreateKMatrixCouplingMacAyealPattynFriction();
	ElementMatrix* Ke =new ElementMatrix(Ke1,Ke2);

	/*clean-up and return*/
	delete Ke1;
	delete Ke2;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixCouplingMacAyealPattynViscous{{{1*/
ElementMatrix* Penta::CreateKMatrixCouplingMacAyealPattynViscous(void){

	const int    NUMVERTICESm=3;  //MacAyealNUMVERTICES
	const int    numdofm=2*NUMVERTICESm;
	const int    NUMVERTICESp=6; //Pattyn NUMVERTICES
	const int    numdofp=2*NUMVERTICESp;
	const int    numdoftotal=2*NDOF2*NUMVERTICES;
	int             i,j,ig;
	double       xyz_list[NUMVERTICESp][3];
	GaussPenta *gauss=NULL;
	GaussTria  *gauss_tria=NULL;
	double viscosity; //viscosity
	double oldviscosity; //viscosity
	double newviscosity; //viscosity
	double epsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double oldepsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double B[3][numdofp];
	double Bprime[3][numdofm];
	double L[2][numdofp];
	double D[3][3]={0.0};            // material matrix, simple scalar matrix.
	double D_scalar;
	double DL[2][2]={0.0}; //for basal drag
	double DL_scalar;
	double Ke_gg[numdofp][numdofm]={0.0}; //local element stiffness matrix 
	double Ke_gg_gaussian[numdofp][numdofm]; //stiffness matrix evaluated at the gaussian point.
	double Jdet;
	double  alpha2_list[3];
	double  alpha2;
	double viscosity_overshoot;

	/*Find penta on bed as pattyn must be coupled to the dofs on the bed: */
	Penta* pentabase=GetBasalElement();
	Tria* tria=pentabase->SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke1=NewElementMatrix(pentabase->nodes,NUMVERTICES,this->parameters,MacAyealApproximationEnum);
	ElementMatrix* Ke2=NewElementMatrix(this->nodes     ,NUMVERTICES,this->parameters,PattynApproximationEnum);
	ElementMatrix* Ke=new ElementMatrix(Ke1,Ke2);
	delete Ke1; delete Ke2;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICESp);
	this->parameters->FindParam(&viscosity_overshoot,ViscosityOvershootEnum);
	Input* vx_input=inputs->GetInput(VxEnum);       ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);       ISSMASSERT(vy_input);
	Input* vxold_input=inputs->GetInput(VxOldEnum); ISSMASSERT(vxold_input);
	Input* vyold_input=inputs->GetInput(VyOldEnum); ISSMASSERT(vyold_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(5,5);
	gauss_tria=new GaussTria();
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);
		gauss->SynchronizeGaussTria(gauss_tria);

		this->GetStrainRate3dPattyn(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input);
		this->GetStrainRate3dPattyn(&oldepsilon[0],&xyz_list[0][0],gauss,vxold_input,vyold_input);
		matice->GetViscosity3d(&viscosity, &epsilon[0]);
		matice->GetViscosity3d(&oldviscosity, &oldepsilon[0]);

		GetBMacAyealPattyn(&B[0][0], &xyz_list[0][0], gauss);
		tria->GetBprimeMacAyeal(&Bprime[0][0], &xyz_list[0][0], gauss_tria);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		newviscosity=viscosity+viscosity_overshoot*(viscosity-oldviscosity);
		D_scalar=2*newviscosity*gauss->weight*Jdet;
		for (i=0;i<3;i++) D[i][i]=D_scalar;

		TripleMultiply( &B[0][0],3,numdofp,1,
					&D[0][0],3,3,0,
					&Bprime[0][0],3,numdofm,0,
					&Ke_gg_gaussian[0][0],0);

		for( i=0; i<numdofp; i++) for(j=0;j<numdofm; j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
	} 
	for(i=0;i<numdofp;i++) for(j=0;j<numdofm;j++) Ke->values[(i+2*numdofm)*numdoftotal+j]+=Ke_gg[i][j];
	for(i=0;i<numdofm;i++) for(j=0;j<numdofp;j++) Ke->values[i*numdoftotal+(j+2*numdofm)]+=Ke_gg[j][i];

	/*Clean-up and return*/
	delete tria->matice; delete tria;
	delete gauss;
	delete gauss_tria;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixCouplingMacAyealPattynFriction{{{1*/
ElementMatrix* Penta::CreateKMatrixCouplingMacAyealPattynFriction(void){

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater() || IsOnShelf() || !IsOnBed()) return NULL;

	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementMatrix* Ke=tria->CreateKMatrixCouplingMacAyealPattynFriction();
	delete tria->matice; delete tria;

	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixCouplingPattynStokes{{{1*/
ElementMatrix* Penta::CreateKMatrixCouplingPattynStokes(void){

	/*compute all stiffness matrices for this element*/
	ElementMatrix* Ke1=NewElementMatrix(this->nodes,NUMVERTICES,this->parameters,PattynApproximationEnum);
	ElementMatrix* Ke2=NewElementMatrix(this->nodes,NUMVERTICES,this->parameters,StokesApproximationEnum);
	ElementMatrix* Ke=new ElementMatrix(Ke1,Ke2);
	delete Ke1;
	delete Ke2;
	Ke1=CreateKMatrixDiagnosticPattyn();
	Ke2=CreateKMatrixDiagnosticStokes();

	/*Constants*/
	const int    numdofp=2*NUMVERTICES;
	const int    numdofs=4*NUMVERTICES;
	const int    numdoftotal=(4+2)*NUMVERTICES;
	int          i,j;

	for(i=0;i<numdofs;i++) for(j=0;j<NUMVERTICES;j++){
		Ke->values[(i+numdofp)*numdoftotal+NDOF2*j+0]+=Ke2->values[i*numdofs+NDOF4*j+0];
		Ke->values[(i+numdofp)*numdoftotal+NDOF2*j+1]+=Ke2->values[i*numdofs+NDOF4*j+1];
	}
	for(i=0;i<numdofp;i++) for(j=0;j<NUMVERTICES;j++){
		Ke->values[i*numdoftotal+numdofp+NDOF4*j+0]+=Ke1->values[i*numdofp+NDOF2*j+0];
		Ke->values[i*numdoftotal+numdofp+NDOF4*j+1]+=Ke1->values[i*numdofp+NDOF2*j+1];
	}

	/*clean-up and return*/
	delete Ke1;
	delete Ke2;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticHoriz {{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticHoriz(void){

	int approximation;
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	switch(approximation){
		case MacAyealApproximationEnum:
			return CreateKMatrixDiagnosticMacAyeal2d();
		case PattynApproximationEnum:
			return CreateKMatrixDiagnosticPattyn();
		case StokesApproximationEnum:
			return CreateKMatrixDiagnosticStokes();
		case HutterApproximationEnum:
			return NULL;
		case NoneApproximationEnum:
			return NULL;
		case MacAyealPattynApproximationEnum:
			return CreateKMatrixDiagnosticMacAyealPattyn();
		case PattynStokesApproximationEnum:
			return CreateKMatrixDiagnosticPattynStokes();
		default:
			ISSMERROR("Approximation %s not supported yet",EnumToString(approximation));
	}
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticHutter{{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticHutter(void){

	/*Intermediaries*/
	const int numdof=NDOF2*NUMVERTICES;
	int       connectivity[2];
	double    one0,one1;
	int       i,i0,i1,j0,j1;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=NewElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Spawn 3 beam elements: */
	for(i=0;i<3;i++){
		/*2 dofs of first node*/
		i0=2*i;
		i1=2*i+1;
		/*2 dofs of second node*/
		j0=2*(i+3);
		j1=2*(i+3)+1;

		/*Find connectivity for the two nodes*/
		connectivity[0]=nodes[i]->GetConnectivity();
		connectivity[1]=nodes[i+3]->GetConnectivity();
		one0=1/(double)connectivity[0];
		one1=1/(double)connectivity[1];

		/*Create matrix for these two nodes*/
		if (IsOnBed() && IsOnSurface()){
			Ke->values[i0*numdof+i0]=one0;
			Ke->values[i1*numdof+i1]=one0;
			Ke->values[j0*numdof+i0]=-one1;
			Ke->values[j0*numdof+j0]=one1;
			Ke->values[j1*numdof+i1]=-one1;
			Ke->values[j1*numdof+j1]=one1;
		}
		else if (IsOnBed()){
			Ke->values[i0*numdof+i0]=one0;
			Ke->values[i1*numdof+i1]=one0;
			Ke->values[j0*numdof+i0]=-2*one1;
			Ke->values[j0*numdof+j0]=2*one1;
			Ke->values[j1*numdof+i1]=-2*one1;
			Ke->values[j1*numdof+j1]=2*one1;
		}
		else if (IsOnSurface()){
			Ke->values[j0*numdof+i0]=-one1;
			Ke->values[j0*numdof+j0]=one1;
			Ke->values[j1*numdof+i1]=-one1;
			Ke->values[j1*numdof+j1]=one1;
		}
		else{ //node is on two horizontal layers and beams include the values only once, so the have to use half of the connectivity
			Ke->values[j0*numdof+i0]=-2*one1;
			Ke->values[j0*numdof+j0]=2*one1;
			Ke->values[j1*numdof+i1]=-2*one1;
			Ke->values[j1*numdof+j1]=2*one1;
		}
	}

	/*Clean up and return*/
	return Ke;
}
/*FUNCTION Penta::CreateKMatrixDiagnosticMacAyeal2d{{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticMacAyeal2d(void){
	
	/*Figure out if this penta is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the stiffness matrix. */
	if (!IsOnBed() || IsOnWater()) return NULL;

	/*Depth Averaging B*/
	this->InputDepthAverageAtBase(RheologyBEnum,RheologyBbarEnum,MaterialsEnum);

	/*Call Tria function*/
	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementMatrix* Ke=tria->CreateKMatrixDiagnosticMacAyeal();
	delete tria->matice; delete tria;

	/*Delete B averaged*/
	this->matice->inputs->DeleteInput(RheologyBbarEnum);

	/*clean up and return*/
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticMacAyeal3d{{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticMacAyeal3d(void){

	/*compute all stiffness matrices for this element*/
	ElementMatrix* Ke1=CreateKMatrixDiagnosticMacAyeal3dViscous();
	ElementMatrix* Ke2=CreateKMatrixDiagnosticMacAyeal3dFriction();
	ElementMatrix* Ke =new ElementMatrix(Ke1,Ke2);

	/*clean-up and return*/
	delete Ke1;
	delete Ke2;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticMacAyeal3dViscous{{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticMacAyeal3dViscous(void){

	const int    numdof2d=2*NUMVERTICES2D;
	int             i,j,ig;
	double       xyz_list[NUMVERTICES][3];
	GaussPenta *gauss=NULL;
	GaussTria  *gauss_tria=NULL;
	double viscosity, oldviscosity, newviscosity, viscosity_overshoot;
	double epsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double oldepsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double B[3][numdof2d];
	double Bprime[3][numdof2d];
	double L[2][numdof2d];
	double D[3][3]={0.0};            // material matrix, simple scalar matrix.
	double D_scalar;
	double DL[2][2]={0.0}; //for basal drag
	double DL_scalar;
	double Ke_gg_gaussian[numdof2d][numdof2d]; //stiffness matrix evaluated at the gaussian point.
	double Jdet;
	double  slope[2]={0.0};
	double  slope_magnitude;
	double  alpha2_list[3];
	double  alpha2;
	double MAXSLOPE=.06; // 6 %
	double MOUNTAINKEXPONENT=10;
	Tria*  tria=NULL;
	Penta* pentabase=NULL;

	/*Find penta on bed as this is a macayeal elements: */
	pentabase=GetBasalElement();
	tria=pentabase->SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=NewElementMatrix(tria->nodes,NUMVERTICES2D,this->parameters,MacAyealApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes,NUMVERTICES);
	this->parameters->FindParam(&viscosity_overshoot,ViscosityOvershootEnum);
	Input* vx_input=inputs->GetInput(VxEnum);       ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);       ISSMASSERT(vy_input);
	Input* vxold_input=inputs->GetInput(VxOldEnum); ISSMASSERT(vxold_input);
	Input* vyold_input=inputs->GetInput(VyOldEnum); ISSMASSERT(vyold_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(5,5);
	gauss_tria=new GaussTria();
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);
		gauss->SynchronizeGaussTria(gauss_tria);

		tria->GetBMacAyeal(&B[0][0], &xyz_list[0][0], gauss_tria);
		tria->GetBprimeMacAyeal(&Bprime[0][0], &xyz_list[0][0], gauss_tria);


		this->GetStrainRate3dPattyn(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input);
		this->GetStrainRate3dPattyn(&oldepsilon[0],&xyz_list[0][0],gauss,vxold_input,vyold_input);
		matice->GetViscosity3d(&viscosity, &epsilon[0]);
		matice->GetViscosity3d(&oldviscosity, &oldepsilon[0]);
		newviscosity=viscosity+viscosity_overshoot*(viscosity-oldviscosity);
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		D_scalar=2*newviscosity*gauss->weight*Jdet;
		for (i=0;i<3;i++) D[i][i]=D_scalar;

		TripleMultiply( &B[0][0],3,numdof2d,1,
					&D[0][0],3,3,0,
					&Bprime[0][0],3,numdof2d,0,
					&Ke_gg_gaussian[0][0],0);

		for(i=0;i<numdof2d;i++) for(j=0;j<numdof2d;j++) Ke->values[i*numdof2d+j]+=Ke_gg_gaussian[i][j];
	}

	/*Clean up and return*/
	delete tria->matice;
	delete tria;
	delete gauss_tria;
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticMacAyeal3dFriction{{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticMacAyeal3dFriction(void){

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater() || IsOnShelf() || !IsOnBed()) return NULL;

	/*Build a tria element using the 3 grids of the base of the penta. Then use 
	 * the tria functionality to build a friction stiffness matrix on these 3
	 * grids: */
	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementMatrix* Ke=tria->CreateKMatrixDiagnosticMacAyealFriction();
	delete tria->matice; delete tria;

	/*clean-up and return*/
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticMacAyealPattyn{{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticMacAyealPattyn(void){

	/*compute all stiffness matrices for this element*/
	ElementMatrix* Ke1=CreateKMatrixDiagnosticMacAyeal3d();
	ElementMatrix* Ke2=CreateKMatrixDiagnosticPattyn();
	ElementMatrix* Ke3=CreateKMatrixCouplingMacAyealPattyn();
	ElementMatrix* Ke =new ElementMatrix(Ke1,Ke2,Ke3);

	/*clean-up and return*/
	delete Ke1;
	delete Ke2;
	delete Ke3;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticPattyn{{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticPattyn(void){

	/*compute all stiffness matrices for this element*/
	ElementMatrix* Ke1=CreateKMatrixDiagnosticPattynViscous();
	ElementMatrix* Ke2=CreateKMatrixDiagnosticPattynFriction();
	ElementMatrix* Ke =new ElementMatrix(Ke1,Ke2);

	/*clean-up and return*/
	delete Ke1;
	delete Ke2;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticPattynViscous{{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticPattynViscous(void){

	/* local declarations */
	const int    numdof=2*NUMVERTICES;
	int             i,j,ig;
	double       xyz_list[NUMVERTICES][3];
	GaussPenta *gauss=NULL;
	double viscosity; //viscosity
	double oldviscosity; //viscosity
	double newviscosity; //viscosity
	double viscosity_overshoot;
	double epsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double oldepsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double B[5][numdof];
	double Bprime[5][numdof];
	double L[2][numdof];
	double D[5][5]={0.0};            // material matrix, simple scalar matrix.
	double D_scalar;
	double DL[2][2]={0.0}; //for basal drag
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0}; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.
	double Jdet;
	double  slope[2]={0.0};
	double  slope_magnitude;
	double  alpha2_list[3];
	double  alpha2;
	double MAXSLOPE=.06; // 6 %
	double MOUNTAINKEXPONENT=10;
	Tria*  tria=NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=NewElementMatrix(nodes,NUMVERTICES,this->parameters,PattynApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	this->parameters->FindParam(&viscosity_overshoot,ViscosityOvershootEnum);
	Input* vx_input=inputs->GetInput(VxEnum);       ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);       ISSMASSERT(vy_input);
	Input* vxold_input=inputs->GetInput(VxOldEnum); ISSMASSERT(vxold_input);
	Input* vyold_input=inputs->GetInput(VyOldEnum); ISSMASSERT(vyold_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(5,5);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		this->GetStrainRate3dPattyn(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input);
		this->GetStrainRate3dPattyn(&oldepsilon[0],&xyz_list[0][0],gauss,vxold_input,vyold_input);
		matice->GetViscosity3d(&viscosity, &epsilon[0]);
		matice->GetViscosity3d(&oldviscosity, &oldepsilon[0]);

		GetBPattyn(&B[0][0], &xyz_list[0][0], gauss);
		GetBprimePattyn(&Bprime[0][0], &xyz_list[0][0], gauss);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		newviscosity=viscosity+viscosity_overshoot*(viscosity-oldviscosity);
		D_scalar=2*newviscosity*gauss->weight*Jdet;
		for (i=0;i<5;i++) D[i][i]=D_scalar;

		TripleMultiply( &B[0][0],5,numdof,1,
					&D[0][0],5,5,0,
					&Bprime[0][0],5,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg_gaussian[i][j];
	}

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticPattynFriction{{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticPattynFriction(void){

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater() || IsOnShelf() || !IsOnBed()) return NULL;

	/*Build a tria element using the 3 grids of the base of the penta. Then use 
	 * the tria functionality to build a friction stiffness matrix on these 3
	 * grids: */
	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementMatrix* Ke=tria->CreateKMatrixDiagnosticPattynFriction();
	delete tria->matice; delete tria;

	/*clean-up and return*/
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticPattynStokes{{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticPattynStokes(void){

	/*compute all stiffness matrices for this element*/
	ElementMatrix* Ke1=CreateKMatrixDiagnosticPattyn();
	ElementMatrix* Ke2=CreateKMatrixDiagnosticStokes();
	ElementMatrix* Ke3=CreateKMatrixCouplingPattynStokes();
	ElementMatrix* Ke =new ElementMatrix(Ke1,Ke2,Ke3);

	/*clean-up and return*/
	delete Ke1;
	delete Ke2;
	delete Ke3;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticStokes{{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticStokes(void){

	/*compute all stiffness matrices for this element*/
	ElementMatrix* Ke1=CreateKMatrixDiagnosticStokesViscous();
	ElementMatrix* Ke2=CreateKMatrixDiagnosticStokesFriction();
	ElementMatrix* Ke =new ElementMatrix(Ke1,Ke2);

	/*clean-up and return*/
	delete Ke1;
	delete Ke2;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticStokesViscous {{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticStokesViscous(void){

	const int numdof=NUMVERTICES*NDOF4;
	int i,j;
	int     ig;
	double     xyz_list[NUMVERTICES][3];
	double	  xyz_list_tria[NUMVERTICES2D][3];
	double	  bed_normal[3];
	double     Ke_temp[27][27]={0.0}; //for the six nodes and the bubble 
	double     Ke_reduced[numdof][numdof]; //for the six nodes only
	double     Ke_gaussian[27][27];
	double     B[8][27];
	double     B_prime[8][27];
	double     Jdet;
	double     D[8][8]={0.0};
	double     D_scalar;
	double     DLStokes[14][14]={0.0};
	GaussPenta *gauss=NULL;
	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity;
	double  alpha2_gauss;
	Friction* friction=NULL;
	double stokesreconditioning;
	int analysis_type;
	int approximation;

	/*If on water or not Stokes, skip stiffness: */
	inputs->GetParameterValue(&approximation,ApproximationEnum);
	if(IsOnWater() || (approximation!=StokesApproximationEnum && approximation!=PattynStokesApproximationEnum)) return NULL;
	ElementMatrix* Ke=NewElementMatrix(nodes,NUMVERTICES,this->parameters,StokesApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	Input* vx_input=inputs->GetInput(VxEnum); ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum); ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum); ISSMASSERT(vz_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(5,5);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);
		matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

		GetBStokes(&B[0][0],&xyz_list[0][0],gauss); 
		GetBprimeStokes(&B_prime[0][0],&xyz_list[0][0],gauss); 

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
		 * onto this scalar matrix, so that we win some computational time: */
		D_scalar=gauss->weight*Jdet;
		for (i=0;i<6;i++) D[i][i]=D_scalar*2*viscosity;
		for (i=6;i<8;i++) D[i][i]=-D_scalar*stokesreconditioning;

		TripleMultiply( &B[0][0],8,27,1,
					&D[0][0],8,8,0,
					&B_prime[0][0],8,27,0,
					&Ke_gaussian[0][0],0);

		for(i=0;i<27;i++) for(j=0;j<27;j++) Ke_temp[i][j]+=Ke_gaussian[i][j];
	}

	/*Condensation*/
	ReduceMatrixStokes(Ke->values, &Ke_temp[0][0]);

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticStokesFriction {{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticStokesFriction(void){

	const int numdof=NUMVERTICES*NDOF4;
	const int numdof2d=NUMVERTICES2D*NDOF4;
	int i,j;
	int     ig;
	double     xyz_list[NUMVERTICES][3];
	double	  xyz_list_tria[NUMVERTICES2D][3];
	double	  bed_normal[3];
	double     Ke_temp[27][27]={0.0}; //for the six nodes and the bubble 
	double     Ke_reduced[numdof][numdof]; //for the six nodes only
	double     Ke_gaussian[27][27];
	double     Ke_drag_gaussian[numdof2d][numdof2d];
	double     B[8][27];
	double     B_prime[8][27];
	double     LStokes[14][numdof2d];
	double     LprimeStokes[14][numdof2d];
	double     Jdet;
	double     Jdet2d;
	double     D[8][8]={0.0};
	double     D_scalar;
	double     DLStokes[14][14]={0.0};
	GaussPenta *gauss=NULL;
	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity;
	double  alpha2_gauss;
	Friction* friction=NULL;
	double stokesreconditioning;
	int analysis_type;
	int approximation;

	/*If on water or not Stokes, skip stiffness: */
	inputs->GetParameterValue(&approximation,ApproximationEnum);
	if(IsOnWater() || IsOnShelf() || !IsOnBed() || (approximation!=StokesApproximationEnum && approximation!=PattynStokesApproximationEnum)) return NULL;
	ElementMatrix* Ke=NewElementMatrix(nodes,NUMVERTICES,this->parameters,StokesApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	Input* vx_input=inputs->GetInput(VxEnum); ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum); ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum); ISSMASSERT(vz_input);
	for(i=0;i<NUMVERTICES2D;i++) for(j=0;j<3;j++) xyz_list_tria[i][j]=xyz_list[i][j];

	/*build friction object, used later on: */
	friction=new Friction("3d",inputs,matpar,analysis_type);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(0,1,2,2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetLStokes(&LStokes[0][0], gauss);
		GetLprimeStokes(&LprimeStokes[0][0], &xyz_list[0][0], gauss);

		this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);
		matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

		BedNormal(&bed_normal[0],xyz_list_tria);
		GetTriaJacobianDeterminant(&Jdet2d, &xyz_list_tria[0][0],gauss);
		friction->GetAlpha2(&alpha2_gauss, gauss,VxEnum,VyEnum,VzEnum);

		DLStokes[0][0]=alpha2_gauss*gauss->weight*Jdet2d;
		DLStokes[1][1]=alpha2_gauss*gauss->weight*Jdet2d;
		DLStokes[2][2]=-alpha2_gauss*gauss->weight*Jdet2d*bed_normal[0]*bed_normal[2];
		DLStokes[3][3]=-alpha2_gauss*gauss->weight*Jdet2d*bed_normal[1]*bed_normal[2];
		DLStokes[4][4]=-alpha2_gauss*gauss->weight*Jdet2d*bed_normal[0]*bed_normal[2];
		DLStokes[5][5]=-alpha2_gauss*gauss->weight*Jdet2d*bed_normal[1]*bed_normal[2];
		DLStokes[6][6]=-2*viscosity*gauss->weight*Jdet2d*bed_normal[0];
		DLStokes[7][7]=-2*viscosity*gauss->weight*Jdet2d*bed_normal[1];
		DLStokes[8][8]=-2*viscosity*gauss->weight*Jdet2d*bed_normal[2];
		DLStokes[9][8]=-2*viscosity*gauss->weight*Jdet2d*bed_normal[0]/2.0;
		DLStokes[10][10]=-2*viscosity*gauss->weight*Jdet2d*bed_normal[1]/2.0;
		DLStokes[11][11]=stokesreconditioning*gauss->weight*Jdet2d*bed_normal[0];
		DLStokes[12][12]=stokesreconditioning*gauss->weight*Jdet2d*bed_normal[1];
		DLStokes[13][13]=stokesreconditioning*gauss->weight*Jdet2d*bed_normal[2];

		TripleMultiply( &LStokes[0][0],14,numdof2d,1,
					&DLStokes[0][0],14,14,0,
					&LprimeStokes[0][0],14,numdof2d,0,
					&Ke_drag_gaussian[0][0],0);

		for(i=0;i<numdof2d;i++) for(j=0;j<numdof2d;j++) Ke_temp[i][j]+=Ke_drag_gaussian[i][j];
	}

	for(i=0;i<numdof2d;i++) for(j=0;j<numdof2d;j++) Ke->values[i*numdof+j]+=Ke_temp[i][j];

	/*Clean up and return*/
	delete gauss;
	delete friction;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticVert {{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticVert(void){

	/*compute all stiffness matrices for this element*/
	ElementMatrix* Ke1=CreateKMatrixDiagnosticVertVolume();
	ElementMatrix* Ke2=CreateKMatrixDiagnosticVertSurface();
	ElementMatrix* Ke =new ElementMatrix(Ke1,Ke2);

	/*clean-up and return*/
	delete Ke1;
	delete Ke2;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticVertVolume {{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticVertVolume(void){

	/* local declarations */
	const int    numdof=NDOF1*NUMVERTICES;
	int             i,j,ig;
	double       xyz_list[NUMVERTICES][3];
	GaussPenta *gauss=NULL;
	double  Ke_gg[numdof][numdof]={0.0};
	double  Jdet;
	double  B[NDOF1][NUMVERTICES];
	double  Bprime[NDOF1][NUMVERTICES];
	double  DL_scalar;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=NewElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(2,2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetBVert(&B[0][0], &xyz_list[0][0], gauss);
		GetBprimeVert(&Bprime[0][0], &xyz_list[0][0], gauss);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		DL_scalar=gauss->weight*Jdet;

		TripleMultiply( &B[0][0],1,NUMVERTICES,1,
					&DL_scalar,1,1,0,
					&Bprime[0][0],1,NUMVERTICES,0,
					&Ke_gg[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg[i][j];
	} 

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticVertSurface {{{1*/
ElementMatrix* Penta::CreateKMatrixDiagnosticVertSurface(void){

	if (!IsOnSurface() || IsOnWater()) return NULL;

	/*Call Tria function*/
	Tria* tria=(Tria*)SpawnTria(3,4,5); //nodes 3,4 and 5 are on the surface
	ElementMatrix* Ke=tria->CreateKMatrixDiagnosticVertSurface();
	delete tria->matice; delete tria;

	/*clean up and return*/
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixMelting {{{1*/
ElementMatrix* Penta::CreateKMatrixMelting(void){

	if (!IsOnBed() || IsOnWater()) return NULL;

	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementMatrix* Ke=tria->CreateKMatrixMelting();

	delete tria->matice; delete tria;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixPrognostic {{{1*/
ElementMatrix* Penta::CreateKMatrixPrognostic(void){

	if (!IsOnBed() || IsOnWater()) return NULL;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementMatrix* Ke=tria->CreateKMatrixPrognostic();
	delete tria->matice; delete tria;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	/*clean up and return*/
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixSlope {{{1*/
ElementMatrix* Penta::CreateKMatrixSlope(void){

	if (!IsOnBed() || IsOnWater()) return NULL;

	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementMatrix* Ke=tria->CreateKMatrixSlope();
	delete tria->matice; delete tria;

	/*clean up and return*/
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixThermal {{{1*/
ElementMatrix* Penta::CreateKMatrixThermal(void){

	/*compute all stiffness matrices for this element*/
	ElementMatrix* Ke1=CreateKMatrixThermalVolume();
	ElementMatrix* Ke2=CreateKMatrixThermalShelf();
	ElementMatrix* Ke =new ElementMatrix(Ke1,Ke2);

	/*clean-up and return*/
	delete Ke1;
	delete Ke2;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixThermalVolume {{{1*/
ElementMatrix* Penta::CreateKMatrixThermalVolume(void){

	int i,j;
	int     ig;
	int found=0;
	const int    numdof=NDOF1*NUMVERTICES;
	double       xyz_list[NUMVERTICES][3];
	int*         doflist=NULL;
	GaussPenta *gauss=NULL;
	double  K[2][2]={0.0};
	double  u,v,w;
	double     Ke_gaussian_conduct[numdof][numdof];
	double     Ke_gaussian_advec[numdof][numdof];
	double     Ke_gaussian_artdiff[numdof][numdof];
	double     Ke_gaussian_transient[numdof][numdof];
	double     B[3][numdof];
	double     Bprime[3][numdof];
	double     B_conduct[3][numdof];
	double     B_advec[3][numdof];
	double     B_artdiff[2][numdof];
	double     Bprime_advec[3][numdof];
	double     L[numdof];
	double     D_scalar;
	double     D[3][3];
	double     l1l2l3[3];
	double     tl1l2l3D[3];
	double     tBD[3][numdof];
	double     tBD_conduct[3][numdof];
	double     tBD_advec[3][numdof];
	double     tBD_artdiff[3][numdof];
	double     tLD[numdof];
	double     Jdet;
	double     gravity,rho_ice,rho_water;
	double     heatcapacity,thermalconductivity;
	double     mixed_layer_capacity,thermal_exchange_velocity;
	double dt,epsvel;
	bool   artdiff;
	Tria*  tria=NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=NewElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	heatcapacity=matpar->GetHeatCapacity();
	thermalconductivity=matpar->GetThermalConductivity();
	this->parameters->FindParam(&dt,DtEnum);
	this->parameters->FindParam(&artdiff,ArtDiffEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);
	Input* vx_input=inputs->GetInput(VxEnum); ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum); ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum); ISSMASSERT(vz_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(2,2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/*Conduction: */

		GetBConduct(&B_conduct[0][0],&xyz_list[0][0],gauss); 

		D_scalar=gauss->weight*Jdet*(thermalconductivity/(rho_ice*heatcapacity));
		if(dt) D_scalar=D_scalar*dt;

		D[0][0]=D_scalar; D[0][1]=0; D[0][2]=0;
		D[1][0]=0; D[1][1]=D_scalar; D[1][2]=0;
		D[2][0]=0; D[2][1]=0; D[2][2]=D_scalar;

		MatrixMultiply(&B_conduct[0][0],3,numdof,1,&D[0][0],3,3,0,&tBD_conduct[0][0],0);
		MatrixMultiply(&tBD_conduct[0][0],numdof,3,0,&B_conduct[0][0],3,numdof,0,&Ke_gaussian_conduct[0][0],0);

		/*Advection: */

		GetBAdvec(&B_advec[0][0],&xyz_list[0][0],gauss); 
		GetBprimeAdvec(&Bprime_advec[0][0],&xyz_list[0][0],gauss); 

		vx_input->GetParameterValue(&u, gauss);
		vy_input->GetParameterValue(&v, gauss);
		vz_input->GetParameterValue(&w, gauss);

		D_scalar=gauss->weight*Jdet;
		if(dt) D_scalar=D_scalar*dt;

		D[0][0]=D_scalar*u;D[0][1]=0;         D[0][2]=0;
		D[1][0]=0;         D[1][1]=D_scalar*v;D[1][2]=0;
		D[2][0]=0;         D[2][1]=0;         D[2][2]=D_scalar*w;

		MatrixMultiply(&B_advec[0][0],3,numdof,1,&D[0][0],3,3,0,&tBD_advec[0][0],0);
		MatrixMultiply(&tBD_advec[0][0],numdof,3,0,&Bprime_advec[0][0],3,numdof,0,&Ke_gaussian_advec[0][0],0);

		/*Transient: */

		if(dt){
			GetNodalFunctionsP1(&L[0], gauss);
			D_scalar=gauss->weight*Jdet;
			D_scalar=D_scalar;

			MatrixMultiply(&L[0],numdof,1,0,&D_scalar,1,1,0,&tLD[0],0);
			MatrixMultiply(&tLD[0],numdof,1,0,&L[0],1,numdof,0,&Ke_gaussian_transient[0][0],0);
		}
		else{
			for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke_gaussian_transient[i][j]=0;
		}

		/*Artifficial diffusivity*/

		if(artdiff){
			/*Build K: */
			D_scalar=gauss->weight*Jdet/(pow(u,2)+pow(v,2)+epsvel);
			if(dt) D_scalar=D_scalar*dt;
			K[0][0]=D_scalar*pow(u,2);       K[0][1]=D_scalar*fabs(u)*fabs(v);
			K[1][0]=D_scalar*fabs(u)*fabs(v);K[1][1]=D_scalar*pow(v,2);

			GetBArtdiff(&B_artdiff[0][0],&xyz_list[0][0],gauss); 

			MatrixMultiply(&B_artdiff[0][0],2,numdof,1,&K[0][0],2,2,0,&tBD_artdiff[0][0],0);
			MatrixMultiply(&tBD_artdiff[0][0],numdof,2,0,&B_artdiff[0][0],2,numdof,0,&Ke_gaussian_artdiff[0][0],0);
		}
		else{
			for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke_gaussian_artdiff[i][j]=0;
		}

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gaussian_conduct[i][j]+Ke_gaussian_advec[i][j]+Ke_gaussian_transient[i][j]+Ke_gaussian_artdiff[i][j];
	}

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixThermalShelf {{{1*/
ElementMatrix* Penta::CreateKMatrixThermalShelf(void){

	if (!IsOnBed() || !IsOnShelf() || IsOnWater()) return NULL;

	/*Call Tria function*/
	Tria* tria=(Tria*)SpawnTria(0,1,2);
	ElementMatrix* Ke=tria->CreateKMatrixThermal();
	delete tria->matice; delete tria;

	return Ke;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorAdjointHoriz{{{1*/
ElementVector* Penta::CreatePVectorAdjointHoriz(void){

	int approximation;
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	switch(approximation){
		case MacAyealApproximationEnum:
			return CreatePVectorAdjointMacAyeal();
		case PattynApproximationEnum:
			return CreatePVectorAdjointPattyn();
		case NoneApproximationEnum:
			return NULL;
		case StokesApproximationEnum:
			return CreatePVectorAdjointStokes();
		default:
			ISSMERROR("Approximation %s not supported yet",EnumToString(approximation));
	}
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorAdjointMacAyeal{{{1*/
ElementVector* Penta::CreatePVectorAdjointMacAyeal(){

	if (!IsOnBed() || IsOnWater()) return NULL;

	/*Call Tria function*/
	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementVector* pe=tria->CreatePVectorAdjointHoriz();
	delete tria->matice; delete tria;

	/*clean up and return*/
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorAdjointPattyn{{{1*/
ElementVector* Penta::CreatePVectorAdjointPattyn(void){

	if (!IsOnSurface() || IsOnWater()) return NULL;

	/*Call Tria function*/
	Tria* tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
	ElementVector* pe=tria->CreatePVectorAdjointHoriz();
	delete tria->matice; delete tria;

	/*clean up and return*/
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorBalancedthickness {{{1*/
void Penta::CreatePVectorBalancedthickness( Vec pg){

	if (!IsOnBed() || IsOnWater()) return;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Call Tria function*/
	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementVector* pe=tria->CreatePVectorBalancedthickness();
	delete tria->matice; delete tria;
	pe->AddToGlobal(pg,NULL);
	delete pe;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	/*Clean up and return*/
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorBalancedvelocities {{{1*/
void Penta::CreatePVectorBalancedvelocities( Vec pg){

	if (!IsOnBed() || IsOnWater()) return;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Call Tria function*/
	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementVector* pe=tria->CreatePVectorBalancedvelocities();
	delete tria->matice; delete tria;
	pe->AddToGlobal(pg,NULL);
	delete pe;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	/*Clean up and return*/
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorCouplingPattynStokes {{{1*/
ElementVector* Penta::CreatePVectorCouplingPattynStokes(void){

	/*compute all load vectorsfor this element*/
	ElementVector* pe1=CreatePVectorCouplingPattynStokesViscous();
	ElementVector* pe2=CreatePVectorCouplingPattynStokesFriction();
	ElementVector* pe =new ElementVector(pe1,pe2);

	/*clean-up and return*/
	delete pe1;
	delete pe2;
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorCouplingPattynStokesViscous {{{1*/
ElementVector* Penta::CreatePVectorCouplingPattynStokesViscous(void){

	const int numdof=NUMVERTICES*NDOF4;
	int i,j;
	int     ig;
	double		   xyz_list_tria[NUMVERTICES2D][3];
	double         xyz_list[NUMVERTICES][3];
	double		   bed_normal[3];
	GaussPenta *gauss=NULL;
	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity, w, alpha2_gauss;
	double  dw[3];
	double     Pe_gaussian[24]={0.0}; //for the six nodes
	double     l1l6[6]; //for the six nodes of the penta
	double     dh1dh6[3][6]; //for the six nodes of the penta
	double     Jdet;
	double     Jdet2d;
	Tria*     tria=NULL;
	Friction* friction=NULL;
	double stokesreconditioning;
	int    approximation;
	int    analysis_type;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	inputs->GetParameterValue(&approximation,ApproximationEnum);
	if(approximation!=PattynStokesApproximationEnum) return NULL;
	ElementVector* pe=NewElementVector(nodes,NUMVERTICES,this->parameters,StokesApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	Input* vx_input=inputs->GetInput(VxEnum);               ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);               ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);               ISSMASSERT(vz_input);
	Input* vzpattyn_input=inputs->GetInput(VzPattynEnum);   ISSMASSERT(vzpattyn_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(5,5);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);
		matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		GetNodalFunctionsP1(&l1l6[0], gauss);
		GetNodalFunctionsP1Derivatives(&dh1dh6[0][0],&xyz_list[0][0], gauss);

		vzpattyn_input->GetParameterDerivativeValue(&dw[0],&xyz_list[0][0],gauss);

		for(i=0;i<NUMVERTICES;i++){
			pe->values[i*NDOF4+0]+=-Jdet*gauss->weight*viscosity*dw[0]*dh1dh6[2][i]/2;
			pe->values[i*NDOF4+1]+=-Jdet*gauss->weight*viscosity*dw[1]*dh1dh6[2][i]/2;
			pe->values[i*NDOF4+2]+=-Jdet*gauss->weight*viscosity*(dw[0]*dh1dh6[0][i]+dw[1]*dh1dh6[1][i]+dw[2]*dh1dh6[2][i])/2;
			pe->values[i*NDOF4+3]+=Jdet*gauss->weight*stokesreconditioning*dw[2]*l1l6[i];
		}
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorCouplingPattynStokesFriction{{{1*/
ElementVector* Penta::CreatePVectorCouplingPattynStokesFriction(void){

	const int numdof=NUMVERTICES*NDOF4;
	int i,j;
	int     ig;
	double		   xyz_list_tria[NUMVERTICES2D][3];
	double         xyz_list[NUMVERTICES][3];
	double		   bed_normal[3];
	GaussPenta *gauss=NULL;
	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity, w, alpha2_gauss;
	double  dw[3];
	double     Pe_gaussian[24]={0.0}; //for the six nodes
	double     l1l6[6]; //for the six nodes of the penta
	double     dh1dh6[3][6]; //for the six nodes of the penta
	double     Jdet;
	double     Jdet2d;
	Tria*     tria=NULL;
	Friction* friction=NULL;
	double stokesreconditioning;
	int    approximation;
	int    analysis_type;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater() || !IsOnBed() || IsOnShelf()) return NULL;
	inputs->GetParameterValue(&approximation,ApproximationEnum);
	if(approximation!=PattynStokesApproximationEnum) return NULL;
	ElementVector* pe=NewElementVector(nodes,NUMVERTICES,this->parameters,StokesApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	Input* vx_input=inputs->GetInput(VxEnum);               ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);               ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);               ISSMASSERT(vz_input);
	Input* vzpattyn_input=inputs->GetInput(VzPattynEnum);   ISSMASSERT(vzpattyn_input);


	for(i=0;i<NUMVERTICES2D;i++){
		for(j=0;j<3;j++){
			xyz_list_tria[i][j]=xyz_list[i][j];
		}
	}

	/*build friction object, used later on: */
	friction=new Friction("3d",inputs,matpar,analysis_type);

	/* Start looping on the number of gauss 2d (nodes on the bedrock) */
	gauss=new GaussPenta(0,1,2,2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Get the Jacobian determinant */
		GetTriaJacobianDeterminant(&Jdet2d, &xyz_list_tria[0][0], gauss);

		/*Get L matrix : */
		GetNodalFunctionsP1(l1l6, gauss);

		/*Get normal vecyor to the bed */
		BedNormal(&bed_normal[0],xyz_list_tria);

		/*Get Viscosity*/
		this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);
		matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

		/*Get friction*/
		friction->GetAlpha2(&alpha2_gauss, gauss,VxEnum,VyEnum,VzEnum);

		/*Get w and its derivatives*/
		vzpattyn_input->GetParameterValue(&w, gauss);
		vzpattyn_input->GetParameterDerivativeValue(&dw[0],&xyz_list[0][0],gauss);

		for(i=0;i<NUMVERTICES2D;i++){
			pe->values[i*NDOF4+0]+=Jdet2d*gauss->weight*(alpha2_gauss*w*bed_normal[0]*bed_normal[2]+viscosity*dw[2]*bed_normal[0])*l1l6[i];
			pe->values[i*NDOF4+1]+=Jdet2d*gauss->weight*(alpha2_gauss*w*bed_normal[1]*bed_normal[2]+viscosity*dw[2]*bed_normal[1])*l1l6[i];
			pe->values[i*NDOF4+2]+=Jdet2d*gauss->weight*viscosity*(dw[0]*bed_normal[0]+dw[1]*bed_normal[1]+dw[2]*bed_normal[2])*l1l6[i];
		}
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticHoriz{{{1*/
ElementVector* Penta::CreatePVectorDiagnosticHoriz(void){

	int approximation;
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	switch(approximation){
		case MacAyealApproximationEnum:
			return CreatePVectorDiagnosticMacAyeal();
		case PattynApproximationEnum:
			return CreatePVectorDiagnosticPattyn();
		case HutterApproximationEnum:
			return NULL;
		case NoneApproximationEnum:
			return NULL;
		case StokesApproximationEnum:
			return CreatePVectorDiagnosticStokes();
		case MacAyealPattynApproximationEnum:
			return CreatePVectorDiagnosticMacAyealPattyn();
		case PattynStokesApproximationEnum:
			return CreatePVectorDiagnosticPattynStokes();
		default:
			ISSMERROR("Approximation %s not supported yet",EnumToString(approximation));
	}
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticMacAyealPattyn{{{1*/
ElementVector* Penta::CreatePVectorDiagnosticMacAyealPattyn(void){

	/*compute all load vectorsfor this element*/
	ElementVector* pe1=CreatePVectorDiagnosticMacAyeal();
	ElementVector* pe2=CreatePVectorDiagnosticPattyn();
	ElementVector* pe =new ElementVector(pe1,pe2);

	/*clean-up and return*/
	delete pe1;
	delete pe2;
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticPattynStokes{{{1*/
ElementVector* Penta::CreatePVectorDiagnosticPattynStokes(void){

	/*compute all load vectorsfor this element*/
	ElementVector* pe1=CreatePVectorDiagnosticPattyn();
	ElementVector* pe2=CreatePVectorDiagnosticStokes();
	ElementVector* pe3=CreatePVectorCouplingPattynStokes();
	ElementVector* pe =new ElementVector(pe1,pe2,pe3);

	/*clean-up and return*/
	delete pe1;
	delete pe2;
	delete pe3;
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticHutter{{{1*/
ElementVector* Penta::CreatePVectorDiagnosticHutter(void){

	int i,j,k;
	int      ig;
	const int numdofs=NDOF2*NUMVERTICES;
	double    xyz_list[NUMVERTICES][3];
	double    xyz_list_segment[2][3];
	double    z_list[NUMVERTICES];
	double    z_segment[2];
	double    slope2,constant_part;
	int       node0,node1;
	GaussPenta* gauss=NULL;
	double   slope[2];
	double   z_g;
	double   rho_ice,gravity,n,B;
	double   Jdet;
	double   ub,vb;
	double   surface,thickness;
	int  connectivity[2];

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=NewElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	n=matice->GetN();
	B=matice->GetB();
	Input* thickness_input=inputs->GetInput(ThicknessEnum);  ISSMASSERT(thickness_input);
	Input* surface_input=inputs->GetInput(SurfaceEnum);      ISSMASSERT(surface_input);
	Input* slopex_input=inputs->GetInput(SurfaceSlopeXEnum); ISSMASSERT(slopex_input);
	Input* slopey_input=inputs->GetInput(SurfaceSlopeYEnum); ISSMASSERT(slopey_input);
	for(i=0;i<NUMVERTICES;i++)z_list[i]=xyz_list[i][2];

	/*Loop on the three segments*/
	for(i=0;i<3;i++){

		node0=i;
		node1=i+3;

		for(j=0;j<3;j++){
			xyz_list_segment[0][j]=xyz_list[node0][j];
			xyz_list_segment[1][j]=xyz_list[node1][j];
		}

		connectivity[0]=nodes[node0]->GetConnectivity();
		connectivity[1]=nodes[node1]->GetConnectivity();

		/*Loop on the Gauss points: */
		gauss=new GaussPenta(node0,node1,3);
		for(ig=gauss->begin();ig<gauss->end();ig++){
			gauss->GaussPoint(ig);

			slopex_input->GetParameterValue(&slope[0],gauss);
			slopey_input->GetParameterValue(&slope[1],gauss);
			surface_input->GetParameterValue(&surface,gauss);
			thickness_input->GetParameterValue(&thickness,gauss);

			slope2=pow(slope[0],2)+pow(slope[1],2);
			constant_part=-2*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2));

			PentaRef::GetParameterValue(&z_g,&z_list[0],gauss);
			GetSegmentJacobianDeterminant(&Jdet,&xyz_list_segment[0][0],gauss);

			if (IsOnSurface()){
				for(j=0;j<NDOF2;j++) pe->values[2*node1+j]+=constant_part*pow((surface-z_g)/B,n)*slope[j]*Jdet*gauss->weight/(double)connectivity[1];
			}
			else{//connectivity is too large, should take only half on it
				for(j=0;j<NDOF2;j++) pe->values[2*node1+j]+=constant_part*pow((surface-z_g)/B,n)*slope[j]*Jdet*gauss->weight*2/(double)connectivity[1];
			}
		}
		delete gauss;

		//Deal with lower surface
		if (IsOnBed()){
			constant_part=-1.58*pow((double)10.0,-(double)10.0)*rho_ice*gravity*thickness;
			ub=constant_part*slope[0];
			vb=constant_part*slope[1];

			pe->values[2*node0]+=ub/(double)connectivity[0];
			pe->values[2*node0+1]+=vb/(double)connectivity[0];
		}
	}

	/*Clean up and return*/
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticMacAyeal{{{1*/
ElementVector* Penta::CreatePVectorDiagnosticMacAyeal(void){

	if (!IsOnBed() || IsOnWater()) return NULL;

	/*Call Tria function*/
	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementVector* pe=tria->CreatePVectorDiagnosticMacAyeal();
	delete tria->matice; delete tria;

	/*Clean up and return*/
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticPattyn{{{1*/
ElementVector* Penta::CreatePVectorDiagnosticPattyn(void){

	/* node data: */
	const int    numdof=NDOF2*NUMVERTICES;
	int i,j;
	int     ig;
	double       xyz_list[NUMVERTICES][3];
	double  slope[3]; //do not put 2! this goes into GetParameterDerivativeValue, which addresses slope[3] also!
	double  driving_stress_baseline;
	double  thickness;
	GaussPenta *gauss=NULL;
	double Jdet;
	double l1l6[6];
	double  pe_g_gaussian[numdof];

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=NewElementVector(nodes,NUMVERTICES,this->parameters,PattynApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	Input* thickness_input=inputs->GetInput(ThicknessEnum); ISSMASSERT(thickness_input);
	Input* surface_input=inputs->GetInput(SurfaceEnum);     ISSMASSERT(surface_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(2,3);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		thickness_input->GetParameterValue(&thickness, gauss);
		surface_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],gauss);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctionsP1(l1l6, gauss);

		driving_stress_baseline=matpar->GetRhoIce()*matpar->GetG();

		for(i=0;i<NUMVERTICES;i++) for(j=0;j<NDOF2;j++) pe->values[i*NDOF2+j]+= -driving_stress_baseline*slope[j]*Jdet*gauss->weight*l1l6[i];
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticStokes {{{1*/
ElementVector* Penta::CreatePVectorDiagnosticStokes(void){

	/*compute all load vectorsfor this element*/
	ElementVector* pe1=CreatePVectorDiagnosticStokesViscous();
	ElementVector* pe2=CreatePVectorDiagnosticStokesShelf();
	ElementVector* pe =new ElementVector(pe1,pe2);

	/*clean-up and return*/
	delete pe1;
	delete pe2;
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticStokesViscous {{{1*/
ElementVector* Penta::CreatePVectorDiagnosticStokesViscous(void){

	int i,j;
	int     ig;
	const int numdof=NUMVERTICES*NDOF4;
	int       numdof2d=NUMVERTICES2D*NDOF4;
	double         gravity,rho_ice,rho_water;
	double		   xyz_list_tria[NUMVERTICES2D][3];
	double         xyz_list[NUMVERTICES][3];
	double		   bed_normal[3];
	double         bed;
	GaussPenta *gauss=NULL;
	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity;
	double  water_pressure;
	double     Pe_temp[27]={0.0}; //for the six nodes and the bubble 
	double     Pe_gaussian[27]={0.0}; //for the six nodes and the bubble 
	double     Ke_temp[27][3]={0.0}; //for the six nodes and the bubble 
	double     Pe_reduced[numdof]; //for the six nodes only
	double     Ke_gaussian[27][3];
	double     l1l6[6]; //for the six nodes of the penta
	double     l1l7[7]; //for the six nodes and the bubble 
	double     B[8][27];
	double     B_prime[8][27];
	double     B_prime_bubble[8][3];
	double     Jdet;
	double     Jdet2d;
	double     D[8][8]={0.0};
	double     D_scalar;
	double     tBD[27][8];
	Tria*            tria=NULL;
	double stokesreconditioning;
	int  approximation;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	inputs->GetParameterValue(&approximation,ApproximationEnum);
	if(approximation!=StokesApproximationEnum) return NULL;
	ElementVector* pe=NewElementVector(nodes,NUMVERTICES,this->parameters,StokesApproximationEnum);

	/*Retrieve all inputs and parameters*/
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	Input* vx_input=inputs->GetInput(VxEnum);   ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);   ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);   ISSMASSERT(vz_input);
	Input* bed_input=inputs->GetInput(BedEnum); ISSMASSERT(bed_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(5,5);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);
		matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctionsMINI(&l1l7[0], gauss);

		/* Build gaussian vector */
		for(i=0;i<NUMVERTICES+1;i++){
			Pe_gaussian[i*NDOF4+2]=-rho_ice*gravity*Jdet*gauss->weight*l1l7[i];
		}

		/*Add Pe_gaussian to terms in Pe_temp. Watch out for column orientation from matlab: */
		for(i=0;i<27;i++) Pe_temp[i]+=Pe_gaussian[i];

		/*Get B and Bprime matrices: */
		GetBStokes(&B[0][0],&xyz_list[0][0],gauss); 
		GetBprimeStokes(&B_prime[0][0],&xyz_list[0][0], gauss); 

		/*Get bubble part of Bprime */
		for(i=0;i<8;i++) for(j=0;j<3;j++) B_prime_bubble[i][j]=B_prime[i][j+24];

		/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
		 * onto this scalar matrix, so that we win some computational time: */
		D_scalar=gauss->weight*Jdet;
		for (i=0;i<6;i++) D[i][i]=D_scalar*2*viscosity;
		for (i=6;i<8;i++) D[i][i]=-D_scalar*stokesreconditioning;

		/*  Do the triple product tB*D*Bprime: */
		MatrixMultiply(&B[0][0],8,27,1,&D[0][0],8,8,0,&tBD[0][0],0);
		MatrixMultiply(&tBD[0][0],27,8,0,&B_prime_bubble[0][0],8,3,0,&Ke_gaussian[0][0],0);

		/*Add Ke_gaussian and Ke_gaussian to terms in pKe. Watch out for column orientation from matlab: */
		for(i=0;i<27;i++) for(j=0;j<3;j++) Ke_temp[i][j]+=Ke_gaussian[i][j];
	}

	/*Condensation*/
	ReduceVectorStokes(pe->values, &Ke_temp[0][0], &Pe_temp[0]);

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticStokesShelf{{{1*/
ElementVector* Penta::CreatePVectorDiagnosticStokesShelf(void){

	int i,j;
	int     ig;
	const int numdof=NUMVERTICES*NDOF4;
	int       numdof2d=NUMVERTICES2D*NDOF4;
	double         gravity,rho_ice,rho_water;
	double		   xyz_list_tria[NUMVERTICES2D][3];
	double         xyz_list[NUMVERTICES][3];
	double		   bed_normal[3];
	double         bed;
	GaussPenta *gauss=NULL;
	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity;
	double  water_pressure;
	double     Pe_temp[27]={0.0}; //for the six nodes and the bubble 
	double     Pe_gaussian[27]={0.0}; //for the six nodes and the bubble 
	double     Ke_temp[27][3]={0.0}; //for the six nodes and the bubble 
	double     Pe_reduced[numdof]; //for the six nodes only
	double     Ke_gaussian[27][3];
	double     l1l6[6]; //for the six nodes of the penta
	double     l1l7[7]; //for the six nodes and the bubble 
	double     B[8][27];
	double     B_prime[8][27];
	double     B_prime_bubble[8][3];
	double     Jdet;
	double     Jdet2d;
	double     D[8][8]={0.0};
	double     D_scalar;
	double     tBD[27][8];
	Tria*            tria=NULL;
	double stokesreconditioning;
	int  approximation;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater() || !IsOnBed() || !IsOnShelf()) return NULL;
	inputs->GetParameterValue(&approximation,ApproximationEnum);
	if(approximation!=StokesApproximationEnum) return NULL;
	ElementVector* pe=NewElementVector(nodes,NUMVERTICES,this->parameters,StokesApproximationEnum);

	/*Retrieve all inputs and parameters*/
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	Input* vx_input=inputs->GetInput(VxEnum);   ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);   ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);   ISSMASSERT(vz_input);
	Input* bed_input=inputs->GetInput(BedEnum); ISSMASSERT(bed_input);


	for(i=0;i<NUMVERTICES2D;i++){
		for(j=0;j<3;j++){
			xyz_list_tria[i][j]=xyz_list[i][j];
		}
	}

	/* Start looping on the number of gauss 2d (nodes on the bedrock) */
	gauss=new GaussPenta(0,1,2,2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetTriaJacobianDeterminant(&Jdet2d, &xyz_list_tria[0][0], gauss);
		bed_input->GetParameterValue(&bed, gauss);
		GetNodalFunctionsP1(l1l6, gauss);

		water_pressure=gravity*rho_water*bed;

		BedNormal(&bed_normal[0],xyz_list_tria);

		for(i=0;i<NUMVERTICES2D;i++) for(j=0;j<3;j++) pe->values[i*NDOF4+j]+=water_pressure*gauss->weight*Jdet2d*l1l6[i]*bed_normal[j];
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorAdjointStokes{{{1*/
ElementVector* Penta::CreatePVectorAdjointStokes(void){

	if (!IsOnSurface() || IsOnWater()) return NULL;

	/*Call Tria function*/
	Tria* tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
	ElementVector* pe=tria->CreatePVectorAdjointStokes();
	delete tria->matice; delete tria;

	/*clean up and return*/
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticVert {{{1*/
void  Penta::CreatePVectorDiagnosticVert( Vec pg){

	int i;

	/* node data: */
	const int    numdof=NDOF1*NUMVERTICES;
	double       xyz_list[NUMVERTICES][3];
	int*         doflist=NULL;

	/* gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;

	/* Jacobian: */
	double Jdet;

	/*element vector at the gaussian points: */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];
	double l1l6[6];

	/*Spawning: */
	Tria* tria=NULL;

	/*input parameters for structural analysis (diagnostic): */
	double du[3];
	double dv[3];
	double dw[3];
	double dudx,dvdy,dwdz;

	/*Get some parameters*/
	int approximation;

	/*If on water, skip: */
	if(IsOnWater())return;
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If we are on the bedrock, spawn a tria on the bedrock, and use it to build the 
	 *diagnostic base vertical stifness: */
	if(IsOnBed()){
		tria=(Tria*)SpawnTria(0,1,2); //nodes 0, 1 and 2 are on the bedrock
		tria->CreatePVectorDiagnosticBaseVert(pg);
		delete tria->matice; delete tria;
	}

	/*Now, handle the standard penta element: */
	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Retrieve all inputs we will be needing: */
	Input* vx_input=inputs->GetInput(VxEnum); ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum); ISSMASSERT(vy_input);
	Input* vzstokes_input=NULL;
	if(approximation==PattynStokesApproximationEnum){
		vzstokes_input=inputs->GetInput(VzStokesEnum); ISSMASSERT(vzstokes_input);
	}

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(2,2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Get velocity derivative, with respect to x and y: */
		vx_input->GetParameterDerivativeValue(&du[0],&xyz_list[0][0],gauss);
		vy_input->GetParameterDerivativeValue(&dv[0],&xyz_list[0][0],gauss);
		if(approximation==PattynStokesApproximationEnum){
			vzstokes_input->GetParameterDerivativeValue(&dw[0],&xyz_list[0][0],gauss);
			dwdz=dw[2];
		}
		else dwdz=0;
		dudx=du[0];
		dvdy=dv[1];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/*Get nodal functions: */
		GetNodalFunctionsP1(l1l6, gauss);

		/*Build pe_g_gaussian vector: */
		for (i=0;i<NUMVERTICES;i++){
			pe_g_gaussian[i]=(dudx+dvdy+dwdz)*Jdet*gauss->weight*l1l6[i];
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	xfree((void**)&doflist);
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorMelting {{{1*/
void Penta::CreatePVectorMelting( Vec pg){
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorPrognostic {{{1*/
void Penta::CreatePVectorPrognostic( Vec pg){

	if (!IsOnBed() || IsOnWater()) return;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Call Tria function*/
	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementVector* pe=tria->CreatePVectorPrognostic();
	delete tria->matice; delete tria;
	pe->AddToGlobal(pg,NULL);
	delete pe;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	/*Clean up and return*/
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorSlope {{{1*/
ElementVector* Penta::CreatePVectorSlope(void){

	if (!IsOnBed() || IsOnWater()) return NULL;

	/*Call Tria function*/
	Tria* tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	ElementVector* pe=tria->CreatePVectorSlope();
	delete tria->matice; delete tria;

	/*clean up and return*/
	return pe;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorThermal {{{1*/
void Penta::CreatePVectorThermal( Vec pg){

	/*indexing: */
	int i,j;
	int found=0;

	const int  numdof=NUMVERTICES*NDOF1;
	int*       doflist=NULL;

	/*Grid data: */
	double        xyz_list[NUMVERTICES][3];

	/* gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;

	double temperature_list[NUMVERTICES];
	double temperature;

	/*Material properties: */
	double gravity,rho_ice,rho_water;
	double mixed_layer_capacity,heatcapacity;
	double beta,meltingpoint,thermal_exchange_velocity;

	/* element parameters: */
	int    friction_type;

	/*matrices: */
	double P_terms[numdof]={0.0};
	double L[numdof];
	double l1l2l3[3];
	double basalfriction;
	double epsilon[6];
	double epsilon_sqr[3][3];
	double epsilon_matrix[3][3];

	double Jdet;
	double viscosity;
	double epsilon_eff;
	double phi;
	double t_pmp;
	double scalar;
	double scalar_def;
	double scalar_ocean;
	double scalar_transient;

	/*Collapsed formulation: */
	Tria*  tria=NULL;
	double dt;

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/*If on water, skip: */
	if(IsOnWater())return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	heatcapacity=matpar->GetHeatCapacity();
	beta=matpar->GetBeta();
	meltingpoint=matpar->GetMeltingPoint();

	/*Retrieve all inputs we will be needing: */
	Input* vx_input=inputs->GetInput(VxEnum); ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum); ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum); ISSMASSERT(vz_input);
	Input* temperature_input=NULL;
	if (dt){
		temperature_input=inputs->GetInput(TemperatureEnum); ISSMASSERT(inputs);
	}

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(2,3);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Compute strain rate and viscosity: */
		this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);
		matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/* Get nodal functions */
		GetNodalFunctionsP1(&L[0], gauss);

		/*Build deformational heating: */
		GetPhi(&phi, &epsilon[0], viscosity);

		/*Build pe_gaussian */
		scalar_def=phi/(rho_ice*heatcapacity)*Jdet*gauss->weight;
		if(dt) scalar_def=scalar_def*dt;

		for(i=0;i<NUMVERTICES;i++) P_terms[i]+=scalar_def*L[i];

		/* Build transient now */
		if(dt){
			temperature_input->GetParameterValue(&temperature, gauss);
			scalar_transient=temperature*Jdet*gauss->weight;
			for(i=0;i<NUMVERTICES;i++) P_terms[i]+=scalar_transient*L[i];
		}
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	/* Ice/ocean heat exchange flux on ice shelf base */
	if(IsOnBed() && IsOnShelf()){

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreatePVectorThermalShelf(pg);
		delete tria->matice; delete tria;
	}

	/* Geothermal flux on ice sheet base and basal friction */
	if(IsOnBed() && !IsOnShelf()){

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreatePVectorThermalSheet(pg);
		delete tria->matice; delete tria;
	}

	xfree((void**)&doflist);
	delete gauss;

}
/*}}}*/
/*FUNCTION Penta::GetDofList {{{1*/
void  Penta::GetDofList(int** pdoflist,int approximation_enum,int setenum){

	int i,j;
	int numberofdofs=0;
	int count=0;

	/*output: */
	int* doflist=NULL;

	/*First, figure out size of doflist: */
	for(i=0;i<6;i++){
		numberofdofs+=nodes[i]->GetNumberOfDofs(approximation_enum,setenum);
	}

	/*Allocate: */
	doflist=(int*)xmalloc(numberofdofs*sizeof(int));

	/*Populate: */
	count=0;
	for(i=0;i<6;i++){
		nodes[i]->GetDofList(doflist+count,approximation_enum,setenum);
		count+=nodes[i]->GetNumberOfDofs(approximation_enum,setenum);
	}

	/*Assign output pointers:*/
	*pdoflist=doflist;

}
/*}}}*/
/*FUNCTION Penta::GetDofList1 {{{1*/
void  Penta::GetDofList1(int* doflist){
	
	int i;

	for(i=0;i<6;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}
/*}}}*/
/*FUNCTION Penta::GetElementType {{{1*/
int Penta::GetElementType(){

	/*return PentaRef field*/
	return this->element_type;

}
/*}}}*/
/*FUNCTION Penta::GetParameterListOnVertices(double* pvalue,int enumtype) {{{1*/
void Penta::GetParameterListOnVertices(double* pvalue,int enumtype){

	/*Intermediaries*/
	double     value[NUMVERTICES];
	GaussPenta *gauss              = NULL;

	/*Recover input*/
	Input* input=inputs->GetInput(enumtype);
	if (!input) ISSMERROR("Input %s not found in element",EnumToString(enumtype));

	/*Checks in debugging mode*/
	ISSMASSERT(pvalue);

	/* Start looping on the number of vertices: */
	gauss=new GaussPenta();
	for (int iv=0;iv<NUMVERTICES;iv++){
		gauss->GaussVertex(iv);
		input->GetParameterValue(&pvalue[iv],gauss);
	}

	/*clean-up*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::GetParameterListOnVertices(double* pvalue,int enumtype,double defaultvalue) {{{1*/
void Penta::GetParameterListOnVertices(double* pvalue,int enumtype,double defaultvalue){

	/*Intermediaries*/
	double     value[NUMVERTICES];
	GaussPenta *gauss              = NULL;

	/*Recover input*/
	Input* input=inputs->GetInput(enumtype);

	/*Checks in debugging mode*/
	ISSMASSERT(pvalue);

	/* Start looping on the number of vertices: */
	if (input){
		gauss=new GaussPenta();
		for (int iv=0;iv<NUMVERTICES;iv++){
			gauss->GaussVertex(iv);
			input->GetParameterValue(&pvalue[iv],gauss);
		}
	}
	else{
		for (int iv=0;iv<NUMVERTICES;iv++) pvalue[iv]=defaultvalue;
	}

	/*clean-up*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::GetParameterValue(double* pvalue,Node* node,int enumtype) {{{1*/
void Penta::GetParameterValue(double* pvalue,Node* node,int enumtype){

	Input* input=inputs->GetInput(enumtype);
	if(!input) ISSMERROR("No input of type %s found in tria",EnumToString(enumtype));

	GaussPenta* gauss=new GaussPenta();
	gauss->GaussVertex(this->GetNodeIndex(node));

	input->GetParameterValue(pvalue,gauss);
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::GetPhi {{{1*/
void Penta::GetPhi(double* phi, double*  epsilon, double viscosity){
	/*Compute deformational heating from epsilon and viscosity */

	double epsilon_matrix[3][3];
	double epsilon_eff;
	double epsilon_sqr[3][3];

	/* Build epsilon matrix */
	epsilon_matrix[0][0]=*(epsilon+0);
	epsilon_matrix[1][0]=*(epsilon+3);
	epsilon_matrix[2][0]=*(epsilon+4);
	epsilon_matrix[0][1]=*(epsilon+3);
	epsilon_matrix[1][1]=*(epsilon+1);
	epsilon_matrix[2][1]=*(epsilon+5);
	epsilon_matrix[0][2]=*(epsilon+4);
	epsilon_matrix[1][2]=*(epsilon+5);
	epsilon_matrix[2][2]=*(epsilon+2);

	/* Effective value of epsilon_matrix */
	epsilon_sqr[0][0]=pow(epsilon_matrix[0][0],2);
	epsilon_sqr[1][0]=pow(epsilon_matrix[1][0],2);
	epsilon_sqr[2][0]=pow(epsilon_matrix[2][0],2);
	epsilon_sqr[0][1]=pow(epsilon_matrix[0][1],2);
	epsilon_sqr[1][1]=pow(epsilon_matrix[1][1],2);
	epsilon_sqr[2][1]=pow(epsilon_matrix[2][1],2);
	epsilon_sqr[0][2]=pow(epsilon_matrix[0][2],2);
	epsilon_sqr[1][2]=pow(epsilon_matrix[1][2],2);
	epsilon_sqr[2][2]=pow(epsilon_matrix[2][2],2);
	epsilon_eff=1/pow(2,0.5)*pow((epsilon_sqr[0][0]+epsilon_sqr[0][1]+ epsilon_sqr[0][2]+ epsilon_sqr[1][0]+ epsilon_sqr[1][1]+ epsilon_sqr[1][2]+ epsilon_sqr[2][0]+ epsilon_sqr[2][1]+ epsilon_sqr[2][2]),0.5);

	/*Phi = Tr(sigma * eps) 
	 *    = Tr(sigma'* eps)
	 *    = 2 * eps_eff * sigma'_eff
	 *    = 4 * eps_eff ^2*/
	*phi=4*pow(epsilon_eff,2.0)*viscosity;
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticHoriz{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticHoriz(Vec solution){

	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	GaussPenta*  gauss;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx;
	double       vy;
	int          approximation;

	/*Get approximation enum and dof list: */
	inputs->GetParameterValue(&approximation,ApproximationEnum);
	Input* vx_input=inputs->GetInput(VxEnum); ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum); ISSMASSERT(vy_input);

	/*If the element is a coupling, do nothing: every grid is also on an other elements 
	 * (as coupling is between MacAyeal and Pattyn) so the other element will take care of it*/
	GetDofList(&doflist,approximation,GsetEnum);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	gauss=new GaussPenta();
	for(i=0;i<NUMVERTICES;i++){

		/*Recover vx and vy*/
		gauss->GaussVertex(i);
		vx_input->GetParameterValue(&vx,gauss);
		vy_input->GetParameterValue(&vy,gauss);
		values[i*numdofpervertex+0]=vx;
		values[i*numdofpervertex+1]=vy;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

	/*Free ressources:*/
	delete gauss;
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticHutter{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticHutter(Vec solution){

	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	GaussPenta*  gauss=NULL;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx;
	double       vy;

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	Input* vx_input=inputs->GetInput(VxEnum); ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum); ISSMASSERT(vy_input);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	gauss=new GaussPenta();
	for(i=0;i<NUMVERTICES;i++){

		/*Recover vx and vy*/
		gauss->GaussVertex(i);
		vx_input->GetParameterValue(&vx,gauss);
		vy_input->GetParameterValue(&vy,gauss);
		values[i*numdofpervertex+0]=vx;
		values[i*numdofpervertex+1]=vy;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

	/*Free ressources:*/
	delete gauss;
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticVert{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticVert(Vec solution){

	int i;

	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*NUMVERTICES;
	GaussPenta*  gauss=NULL;
	int*         doflist=NULL;
	double       values[numdof];
	double       vz;

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	Input* vz_input=inputs->GetInput(VzEnum); ISSMASSERT(vz_input);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	gauss=new GaussPenta();
	for(i=0;i<NUMVERTICES;i++){

		/*Recover vz */
		gauss->GaussVertex(i);
		vz_input->GetParameterValue(&vz,gauss);
		values[i]=vz;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

	/*Free ressources:*/
	delete gauss;
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticStokes{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticStokes(Vec solution){

	int i;

	const int    numdofpervertex=4;
	const int    numdof=numdofpervertex*NUMVERTICES;
	GaussPenta  *gauss;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx,vy,vz,p;
	double       stokesreconditioning;

	/*Get dof list: */
	GetDofList(&doflist,StokesApproximationEnum,GsetEnum);
	Input* vx_input=inputs->GetInput(VxEnum);       ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);       ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);       ISSMASSERT(vz_input);
	Input* p_input =inputs->GetInput(PressureEnum); ISSMASSERT(p_input);

	/*Recondition pressure: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);

	/*Ok, we have vx vy vz and P in values, fill in vx vy vz P arrays: */
	/*P1 element only for now*/
	gauss=new GaussPenta();
	for(i=0;i<NUMVERTICES;i++){
		gauss->GaussVertex(i);
		vx_input->GetParameterValue(&vx,gauss);
		vy_input->GetParameterValue(&vy,gauss);
		vz_input->GetParameterValue(&vz,gauss);
		p_input ->GetParameterValue(&p ,gauss);
		values[i*numdofpervertex+0]=vx;
		values[i*numdofpervertex+1]=vy;
		values[i*numdofpervertex+2]=vz;
		values[i*numdofpervertex+3]=p/stokesreconditioning;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

	/*Free ressources:*/
	delete gauss;
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsThermal{{{1*/
void  Penta::GetSolutionFromInputsThermal(Vec solution){

	int i;

	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*NUMVERTICES;
	GaussPenta  *gauss=NULL;
	int*         doflist=NULL;
	double       values[numdof];
	double       temp;


	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	Input* t_input=inputs->GetInput(TemperatureEnum); ISSMASSERT(t_input);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	gauss=new GaussPenta();
	for(i=0;i<NUMVERTICES;i++){

		/*Recover vz */
		gauss->GaussVertex(i);
		t_input->GetParameterValue(&temp,gauss);
		values[i]=temp;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);
	
	/*Free ressources:*/
	delete gauss;
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::GetStrainRate3dPattyn{{{1*/
void Penta::GetStrainRate3dPattyn(double* epsilon,double* xyz_list, GaussPenta* gauss, Input* vx_input, Input* vy_input){
	/*Compute the 3d Blatter/PattynStrain Rate (5 components):
	 *
	 * epsilon=[exx eyy exy exz eyz]
	 *
	 * with exz=1/2 du/dz
	 *      eyz=1/2 dv/dz
	 *
	 * the contribution of vz is neglected
	 */

	int i;

	double epsilonvx[5];
	double epsilonvy[5];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input){
		ISSMERROR("Input missing. Here are the input pointers we have for vx: %p, vy: %p\n",vx_input,vy_input);
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vx_input->GetVxStrainRate3dPattyn(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate3dPattyn(epsilonvy,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<5;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i];

}
/*}}}*/
/*FUNCTION Penta::GetStrainRate3d{{{1*/
void Penta::GetStrainRate3d(double* epsilon,double* xyz_list, GaussPenta* gauss, Input* vx_input, Input* vy_input, Input* vz_input){
	/*Compute the 3d Strain Rate (6 components):
	 *
	 * epsilon=[exx eyy ezz exy exz eyz]
	 */

	int i;

	double epsilonvx[6];
	double epsilonvy[6];
	double epsilonvz[6];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input || !vz_input){
		ISSMERROR("Input missing. Here are the input pointers we have for vx: %p, vy: %p, vz: %p\n",vx_input,vy_input,vz_input);
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vx_input->GetVxStrainRate3d(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate3d(epsilonvy,xyz_list,gauss);
	vz_input->GetVzStrainRate3d(epsilonvz,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<6;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i]+epsilonvz[i];

}
/*}}}*/
/*FUNCTION Penta::GetBasalElement{{{1*/
Penta* Penta::GetBasalElement(void){

	/*Output*/
	Penta* penta=NULL;

	/*Go through all elements till the bed is reached*/
	penta=this;
	for(;;){

		/*Stop if we have reached the surface, else, take lower penta*/
		if (penta->IsOnBed()) break;

		/* get lower Penta*/
		penta=penta->GetLowerElement();
		ISSMASSERT(penta->Id()!=this->id);
	}

	/*return output*/
	return penta;
}
/*}}}*/
/*FUNCTION Penta::GetLowerElement{{{1*/
Penta* Penta::GetLowerElement(void){

	Penta* upper_penta=NULL;
	upper_penta=(Penta*)neighbors[0]; //first one (0) under, second one (1) above
	return upper_penta;

}
/*}}}*/
/*FUNCTION Penta::GetUpperElement{{{1*/
Penta* Penta::GetUpperElement(void){

	Penta* upper_penta=NULL;
	upper_penta=(Penta*)neighbors[1]; //first one under, second one above
	return upper_penta;

}
/*}}}*/
/*FUNCTION Penta::GetZcoord {{{1*/
double Penta::GetZcoord(GaussPenta* gauss){

	int    i;
	double xyz_list[NUMVERTICES][3];
	double z_list[NUMVERTICES];
	double z;

	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	for(i=0;i<NUMVERTICES;i++){
		z_list[i]=xyz_list[i][2];
	}
	PentaRef::GetParameterValue(&z,z_list,gauss);

	return z;
}
/*}}}*/
/*FUNCTION Penta::GradjB {{{1*/
void  Penta::GradjB(Vec gradient){

	int i;
	Tria* tria=NULL;
	TriaVertexInput* triavertexinput=NULL;

	/*retrieve inputs :*/
	int  approximation;
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, skip: */
	if(IsOnWater())return;

	if (approximation==MacAyealApproximationEnum){
		/*Bail out element if MacAyeal (2d) and not on bed*/
		if (!IsOnBed()) return;

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute gardj*/

		/*Depth Average B*/
		this->InputDepthAverageAtBase(RheologyBEnum,RheologyBbarEnum,MaterialsEnum);

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		tria->GradjB(gradient);
		delete tria->matice; delete tria;

		/*delete B average*/
		this->matice->inputs->DeleteInput(RheologyBbarEnum);
	}
	else{
		/*Gradient is computed on bed only (Bbar)*/
		if (!IsOnBed()) return;

		/*Depth Average B*/
		this->InputDepthAverageAtBase(RheologyBEnum,RheologyBbarEnum,MaterialsEnum);

		/*B is a 2d field, use MacAyeal(2d) gradient even if it is Stokes or Pattyn*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		tria->GradjB(gradient);
		delete tria->matice; delete tria;

		/*delete B average*/
		this->matice->inputs->DeleteInput(RheologyBbarEnum);
	}

}
/*}}}*/
/*FUNCTION Penta::GradjDrag {{{1*/
void  Penta::GradjDrag(Vec gradient){

	int i;
	Tria* tria=NULL;
	TriaVertexInput* triavertexinput=NULL;
	double temp_gradient[6]={0,0,0,0,0,0};

	/*retrieve inputs :*/
	int approximation;
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, skip: */
	if(IsOnWater())return;

	/*If on shelf, skip: */
	if(IsOnShelf())return;

	/*Bail out if this element does not touch the bedrock: */
	if (!IsOnBed()) return;

	if (approximation==MacAyealApproximationEnum || approximation==PattynApproximationEnum){

		/*MacAyeal or Pattyn*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->GradjDrag(gradient);
		delete tria->matice; delete tria;
	}
	else if (approximation==StokesApproximationEnum){

		/*Stokes*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->GradjDragStokes(gradient);
		delete tria->matice; delete tria;
	}
	else if (approximation==NoneApproximationEnum){
		return;
	}
	else ISSMERROR("approximation %s not supported yet",EnumToString(approximation));


}
/*}}}*/
/*FUNCTION Penta::InputExtrude {{{1*/
void  Penta::InputExtrude(int enum_type,int object_type){

	Penta *penta          = NULL;
	Input *original_input = NULL;

	/*Are we on the base, not on the surface?:*/
	if(IsOnBed()){

		/*OK, we are on bed. we will follow the steps:
		 * 1: find input and extrude it.
		 * 2: follow the upper element until we reach the surface
		 * 3: for each element, we will add a copy of the extruded input*/

		/*Step1: Extrude the original input: */
		if (object_type==ElementsEnum)
		 original_input=(Input*)this->inputs->GetInput(enum_type);
		else if (object_type==MaterialsEnum)
		 original_input=(Input*)matice->inputs->GetInput(enum_type);
		else
		 ISSMERROR("object of type %s not supported yet",EnumToString(object_type));
		if(!original_input) ISSMERROR("%s%s"," could not find input with enum:",EnumToString(enum_type));
		original_input->Extrude();

		/*Stop if there is only one layer of element*/
		if (this->IsOnSurface()) return;

		/*Step 2: this input has been extruded for this element, now follow the upper element*/
		penta=this;
		for(;;){

			/* get upper Penta*/
			penta=penta->GetUpperElement();
			ISSMASSERT(penta->Id()!=this->id);

			/*Add input of the basal element to penta->inputs*/
			Input* copy=NULL;
			copy=(Input*)original_input->copy();
			if (object_type==ElementsEnum)
			 penta->inputs->AddInput((Input*)copy);
			else if (object_type==MaterialsEnum)
			 penta->matice->inputs->AddInput((Input*)copy);
			else
			 ISSMERROR("object of type %s not supported yet",EnumToString(object_type));

			/*Stop if we have reached the surface*/
			if (penta->IsOnSurface()) break;

		}
	}

	return;
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticHoriz {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticHoriz(double* solution){

	/*Inputs*/
	int  approximation;

	/*Recover inputs*/
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*MacAyeal, everything is done by the element on bed*/
	if (approximation==MacAyealApproximationEnum){
		if (!IsOnBed()){
			/*Do nothing. Element on bed will take care of it*/
			return;
		}
		else{
			InputUpdateFromSolutionDiagnosticMacAyeal(solution);
			return;
		}
	}
	else if (approximation==PattynApproximationEnum){
		InputUpdateFromSolutionDiagnosticPattyn(solution);
	}
	else if (approximation==PattynStokesApproximationEnum){
		InputUpdateFromSolutionDiagnosticPattynStokes(solution);
	}
	else if (approximation==StokesApproximationEnum || approximation==NoneApproximationEnum){
		InputUpdateFromSolutionDiagnosticStokes(solution);
	}
	else if (approximation==MacAyealPattynApproximationEnum){
		InputUpdateFromSolutionDiagnosticMacAyealPattyn(solution);
	}
}

/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticMacAyeal {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticMacAyeal(double* solution){

	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vel[NUMVERTICES];
	int          dummy;
	double       pressure[NUMVERTICES];
	double       surface[NUMVERTICES];
	double       rho_ice,g;
	double       xyz_list[NUMVERTICES][3];

	double *vz_ptr          = NULL;
	Penta  *penta          = NULL;

	/*OK, we are on bed. Now recover results*/

	/*Get dof list: */
	GetDofList(&doflist,MacAyealApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays and extrude */
	for(i=0;i<3;i++){
		vx[i]  =values[i*numdofpervertex+0];
		vy[i]  =values[i*numdofpervertex+1];
		vx[i+3]=vx[i];
		vy[i+3]=vy[i];
	}

	/*Get parameters fro pressure computation*/
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();

	/*Start looping over all elements above current element and update all inputs*/
	penta=this;
	for(;;){

		/*Get node data: */
		GetVerticesCoordinates(&xyz_list[0][0],penta->nodes,NUMVERTICES);

		/*Now Compute vel*/
		Input* vz_input=inputs->GetInput(VzEnum);
		if (vz_input){
			if (vz_input->Enum()!=PentaVertexInputEnum) ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumToString(vz_input->Enum()));
			vz_input->GetValuesPtr(&vz_ptr,&dummy);
			for(i=0;i<NUMVERTICES;i++) vz[i]=vz_ptr[i];
		}
		else{for(i=0;i<NUMVERTICES;i++) vz[i]=0.0;}
		for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

		/*Now compute pressure*/
		GetParameterListOnVertices(&surface[0],SurfaceEnum);
		for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);

		/*Now, we have to move the previous Vx and Vy inputs  to old 
		 * status, otherwise, we'll wipe them off: */
		penta->inputs->ChangeEnum(VxEnum,VxOldEnum);
		penta->inputs->ChangeEnum(VyEnum,VyOldEnum);
		penta->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

		/*Add vx and vy as inputs to the tria element: */
		penta->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
		penta->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
		penta->inputs->AddInput(new PentaVertexInput(VelEnum,vel));
		penta->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

		/*Stop if we have reached the surface*/
		if (penta->IsOnSurface()) break;

		/* get upper Penta*/
		penta=penta->GetUpperElement(); ISSMASSERT(penta->Id()!=this->id);
	}
	
	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticMacAyealPattyn {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticMacAyealPattyn(double* solution){

	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	const int    numdof2d=numdofpervertex*NUMVERTICES2D;
	int*         doflistp=NULL;
	int*         doflistm=NULL;
	double       macayeal_values[numdof];
	double       pattyn_values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vel[NUMVERTICES];
	int          dummy;
	double       pressure[NUMVERTICES];
	double       surface[NUMVERTICES];
	double       rho_ice,g;
	double       xyz_list[NUMVERTICES][3];

	double *vz_ptr         = NULL;
	Penta  *penta          = NULL;

	/*OK, we have to add results of this element for pattyn 
	 * and results from the penta at base for macayeal. Now recover results*/
	penta=GetBasalElement();

	/*Get dof listof this element (pattyn dofs) and of the penta at base (macayeal dofs): */
	GetDofList(&doflistp,PattynApproximationEnum,GsetEnum);
	penta->GetDofList(&doflistm,MacAyealApproximationEnum,GsetEnum);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof2d;i++){
		pattyn_values[i]=solution[doflistp[i]];
		macayeal_values[i]=solution[doflistm[i]];
	}
	for(i=numdof2d;i<numdof;i++){
		pattyn_values[i]=solution[doflistp[i]];
		macayeal_values[i]=macayeal_values[i-numdof2d];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		vx[i]=macayeal_values[i*numdofpervertex+0]+pattyn_values[i*numdofpervertex+0];
		vy[i]=macayeal_values[i*numdofpervertex+1]+pattyn_values[i*numdofpervertex+1];
	}

	/*Get Vz*/
	Input* vz_input=inputs->GetInput(VzEnum);
	if (vz_input){
		if (vz_input->Enum()!=PentaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumToString(vz_input->Enum()));
		}
		vz_input->GetValuesPtr(&vz_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vz[i]=vz_ptr[i];
	}
	else{
		for(i=0;i<NUMVERTICES;i++) vz[i]=0.0;
	}

	/*Now Compute vel*/
	for(i=0;i<NUMVERTICES;i++) {
		vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);
	}

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 3D, 
	 *so the pressure is just the pressure at the z elevation: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	GetParameterListOnVertices(&surface[0],SurfaceEnum);
	for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new PentaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflistp);
	xfree((void**)&doflistm);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticPattyn {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticPattyn(double* solution){
	
	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vel[NUMVERTICES];
	int          dummy;
	double       pressure[NUMVERTICES];
	double       surface[NUMVERTICES];
	double       rho_ice,g;
	double       xyz_list[NUMVERTICES][3];
	double *vz_ptr          = NULL;

	/*Get dof list: */
	GetDofList(&doflist,PattynApproximationEnum,GsetEnum);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		vx[i]=values[i*numdofpervertex+0];
		vy[i]=values[i*numdofpervertex+1];
	}

	/*Get Vz*/
	Input* vz_input=inputs->GetInput(VzEnum);
	if (vz_input){
		if (vz_input->Enum()!=PentaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumToString(vz_input->Enum()));
		}
		vz_input->GetValuesPtr(&vz_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vz[i]=vz_ptr[i];
	}
	else{
		for(i=0;i<NUMVERTICES;i++) vz[i]=0.0;
	}

	/*Now Compute vel*/
	for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 3D, 
	 *so the pressure is just the pressure at the z elevation: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	GetParameterListOnVertices(&surface[0],SurfaceEnum);
	for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new PentaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflist);
}

/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticPattynStokes {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticPattynStokes(double* solution){

	int i;

	const int    numdofpervertexp=2;
	const int    numdofpervertexs=4;
	const int    numdofp=numdofpervertexp*NUMVERTICES;
	const int    numdofs=numdofpervertexs*NUMVERTICES;
	int*         doflistp=NULL;
	int*         doflists=NULL;
	double       pattyn_values[numdofp];
	double       stokes_values[numdofs];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vzpattyn[NUMVERTICES];
	double       vzstokes[NUMVERTICES];
	double       vel[NUMVERTICES];
	int          dummy;
	double       pressure[NUMVERTICES];
	double       surface[NUMVERTICES];
	double       rho_ice,g;
	double       xyz_list[NUMVERTICES][3];

	double *vzpattyn_ptr         = NULL;
	Penta  *penta          = NULL;

	/*OK, we have to add results of this element for pattyn 
	 * and results from the penta at base for macayeal. Now recover results*/
	penta=GetBasalElement();

	/*Get dof listof this element (pattyn dofs) and of the penta at base (macayeal dofs): */
	GetDofList(&doflistp,PattynApproximationEnum,GsetEnum);
	GetDofList(&doflists,StokesApproximationEnum,GsetEnum);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdofp;i++){
		pattyn_values[i]=solution[doflistp[i]];
	}
	for(i=0;i<numdofs;i++){
		stokes_values[i]=solution[doflists[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		vx[i]=stokes_values[i*numdofpervertexs+0]+pattyn_values[i*numdofpervertexp+0];
		vy[i]=stokes_values[i*numdofpervertexs+1]+pattyn_values[i*numdofpervertexp+1];
		vzstokes[i]=stokes_values[i*numdofpervertexs+2];
		pressure[i]=stokes_values[i*numdofpervertexs+3];
	}

	/*Get Vz*/
	Input* vzpattyn_input=inputs->GetInput(VzPattynEnum);
	if (vzpattyn_input){
		if (vzpattyn_input->Enum()!=PentaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as VzPattyn is of type %s",EnumToString(vzpattyn_input->Enum()));
		}
		vzpattyn_input->GetValuesPtr(&vzpattyn_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vzpattyn[i]=vzpattyn_ptr[i];
	}
	else{
		ISSMERROR("Cannot update solution as VzPattyn is not present");
	}

	/*Now Compute vel*/
	for(i=0;i<NUMVERTICES;i++) {
		vz[i]=vzpattyn[i]+vzstokes[i];
		vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);
	}

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(VzEnum,VzOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new PentaVertexInput(VzEnum,vz));
	this->inputs->AddInput(new PentaVertexInput(VzStokesEnum,vzstokes));
	this->inputs->AddInput(new PentaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflistp);
	xfree((void**)&doflists);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticHutter {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticHutter(double* solution){
	
	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vel[NUMVERTICES];
	int          dummy;
	double       pressure[NUMVERTICES];
	double       surface[NUMVERTICES];
	double       rho_ice,g;
	double       xyz_list[NUMVERTICES][3];
	double*      vz_ptr=NULL;

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		vx[i]=values[i*numdofpervertex+0];
		vy[i]=values[i*numdofpervertex+1];
	}

	/*Get Vz*/
	Input* vz_input=inputs->GetInput(VzEnum);
	if (vz_input){
		if (vz_input->Enum()!=PentaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumToString(vz_input->Enum()));
		}
		vz_input->GetValuesPtr(&vz_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vz[i]=vz_ptr[i];
	}
	else{
		for(i=0;i<NUMVERTICES;i++) vz[i]=0.0;
	}

	/*Now Compute vel*/
	for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 3D, 
	 *so the pressure is just the pressure at the z elevation: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	GetParameterListOnVertices(&surface[0],SurfaceEnum);
	for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new TriaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflist);
}

/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticVert {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticVert(double* solution){

	int i;

	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vzpattyn[NUMVERTICES];
	double       vzstokes[NUMVERTICES];
	double       vel[NUMVERTICES];
	int          dummy;
	double       pressure[NUMVERTICES];
	double       surface[NUMVERTICES];
	double       rho_ice,g;
	double       xyz_list[NUMVERTICES][3];

	double*      vx_ptr=NULL;
	double*      vy_ptr=NULL;
	double*      vzstokes_ptr=NULL;

	int         approximation;

	/*Get the approximation and do nothing if the element in Stokes or None*/
	inputs->GetParameterValue(&approximation,ApproximationEnum);
	if(approximation==StokesApproximationEnum || approximation==NoneApproximationEnum){
		return;
	}

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vz in values, fill in vz array: */
	for(i=0;i<NUMVERTICES;i++){
		vz[i]=values[i*numdofpervertex+0];
	}

	/*Get Vx and Vy*/
	Input* vx_input=inputs->GetInput(VxEnum);
	if (vx_input){
		if (vx_input->Enum()!=PentaVertexInputEnum) ISSMERROR("Cannot compute Vel as Vx is of type %s",EnumToString(vx_input->Enum()));
		vx_input->GetValuesPtr(&vx_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vx[i]=vx_ptr[i];
	}
	else for(i=0;i<NUMVERTICES;i++) vx[i]=0.0;

	Input* vy_input=inputs->GetInput(VyEnum);
	if (vy_input){
		if (vy_input->Enum()!=PentaVertexInputEnum) ISSMERROR("Cannot compute Vel as Vy is of type %s",EnumToString(vy_input->Enum()));
		vy_input->GetValuesPtr(&vy_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vy[i]=vy_ptr[i];
	}
	else for(i=0;i<NUMVERTICES;i++) vy[i]=0.0;

	/*Do some modifications if we actually have a PattynStokes element*/
	if(approximation==PattynStokesApproximationEnum){
		Input* vzstokes_input=inputs->GetInput(VzStokesEnum);
		if (vzstokes_input){
			if (vzstokes_input->Enum()!=PentaVertexInputEnum) ISSMERROR("Cannot compute Vel as VzStokes is of type %s",EnumToString(vy_input->Enum()));
			vzstokes_input->GetValuesPtr(&vzstokes_ptr,&dummy);
			for(i=0;i<NUMVERTICES;i++) vzstokes[i]=vzstokes_ptr[i];
		}
		else ISSMERROR("Cannot compute Vz as VzStokes in not present in PattynStokes element");
		for(i=0;i<NUMVERTICES;i++){
			vzpattyn[i]=vz[i];
			vz[i]=vzpattyn[i]+vzstokes[i];
		}
	}

	/*Now Compute vel*/

	for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 3D, 
	 *so the pressure is just the pressure at the z elevation: except it this is a PattynStokes element */
	if(approximation!=PattynStokesApproximationEnum){
		rho_ice=matpar->GetRhoIce();
		g=matpar->GetG();
		GetParameterListOnVertices(&surface[0],SurfaceEnum);
		for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);
	}

	/*Now, we have to move the previous Vz inputs to old 
	 * status, otherwise, we'll wipe them off and add the new inputs: */
	this->inputs->ChangeEnum(VzEnum,VzOldEnum);

	if(approximation!=PattynStokesApproximationEnum){
		this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);
		this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));
	}
	else{
		this->inputs->AddInput(new PentaVertexInput(VzPattynEnum,vzpattyn));
	}

	this->inputs->AddInput(new PentaVertexInput(VzEnum,vz));
	this->inputs->AddInput(new PentaVertexInput(VelEnum,vel));

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticStokes {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticStokes(double* solution){
	
	const int    numdof=NDOF4*NUMVERTICES;
	int i;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vel[NUMVERTICES];
	double       pressure[NUMVERTICES];
	double       stokesreconditioning;

	/*Get dof list: */
	GetDofList(&doflist,StokesApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in all arrays: */
	for(i=0;i<NUMVERTICES;i++){
		vx[i]=values[i*NDOF4+0];
		vy[i]=values[i*NDOF4+1];
		vz[i]=values[i*NDOF4+2];
		pressure[i]=values[i*NDOF4+3];
	}

	/*Recondition pressure: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	for(i=0;i<NUMVERTICES;i++){
		pressure[i]=pressure[i]*stokesreconditioning;
	}

	/*Now Compute vel*/
	for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);
	
	/*Now, we have to move the previous inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(VzEnum,VzOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new PentaVertexInput(VzEnum,vz));
	this->inputs->AddInput(new PentaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflist);
}

/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionAdjointStokes {{{1*/
void  Penta::InputUpdateFromSolutionAdjointStokes(double* solution){

	int i;

	const int    numdofpervertex=4;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       lambdax[NUMVERTICES];
	double       lambday[NUMVERTICES];
	double       lambdaz[NUMVERTICES];
	double       lambdap[NUMVERTICES];

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		lambdax[i]=values[i*numdofpervertex+0];
		lambday[i]=values[i*numdofpervertex+1];
		lambdaz[i]=values[i*numdofpervertex+2];
		lambdap[i]=values[i*numdofpervertex+3];
	}

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(AdjointxEnum,lambdax));
	this->inputs->AddInput(new PentaVertexInput(AdjointyEnum,lambday));
	this->inputs->AddInput(new PentaVertexInput(AdjointzEnum,lambdaz));
	this->inputs->AddInput(new PentaVertexInput(AdjointpEnum,lambdap));

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionAdjointHoriz {{{1*/
void  Penta::InputUpdateFromSolutionAdjointHoriz(double* solution){

	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       lambdax[NUMVERTICES];
	double       lambday[NUMVERTICES];

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		lambdax[i]=values[i*numdofpervertex+0];
		lambday[i]=values[i*numdofpervertex+1];
	}

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(AdjointxEnum,lambdax));
	this->inputs->AddInput(new PentaVertexInput(AdjointyEnum,lambday));

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionThermal {{{1*/
void  Penta::InputUpdateFromSolutionThermal(double* solution){

	int i;

	const int    numdof=NDOF1*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       B[numdof];
	double       B_average;
	bool         converged;

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	this->inputs->GetParameterValue(&converged,ConvergedEnum);
	if(converged){
		this->inputs->AddInput(new PentaVertexInput(TemperatureEnum,values));

		/*Update Rheology only if convreged (we must make sure that the temperature is below melting point
		 * otherwise the rheology could be negative*/
		B_average=Paterson((values[0]+values[1]+values[2]+values[3]+values[4]+values[5])/6.0);
		for(i=0;i<numdof;i++) B[i]=B_average;
		this->matice->inputs->AddInput(new PentaVertexInput(RheologyBEnum,B));
	}
	else{
		this->inputs->AddInput(new PentaVertexInput(TemporaryTemperatureEnum,values));
	}

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionOneDof{{{1*/
void  Penta::InputUpdateFromSolutionOneDof(double* solution,int enum_type){

	const int numdofpervertex = 1;
	const int numdof          = numdofpervertex *NUMVERTICES;
	int*         doflist=NULL;
	double    values[numdof];

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector: */
	for(int i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Add input to the element: */
	this->inputs->AddInput(new PentaVertexInput(enum_type,values));
	
	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionOneDofCollpased{{{1*/
void  Penta::InputUpdateFromSolutionOneDofCollapsed(double* solution,int enum_type){

	const int  numdofpervertex = 1;
	const int  numdof          = numdofpervertex *NUMVERTICES;
	const int  numdof2d        = numdof/2;
	int*         doflist=NULL;
	double     values[numdof];
	Penta     *penta           = NULL;

	/*If not on bed, return*/
	if (!IsOnBed()) return;

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector and extrude it */
	for(int i=0;i<numdof2d;i++){
		values[i]         =solution[doflist[i]];
		values[i+numdof2d]=values[i];
	}

	/*Start looping over all elements above current element and update all inputs*/
	penta=this;
	for(;;){

		/*Add input to the element: */
		penta->inputs->AddInput(new PentaVertexInput(enum_type,values));

		/*Stop if we have reached the surface*/
		if (penta->IsOnSurface()) break;

		/* get upper Penta*/
		penta=penta->GetUpperElement(); ISSMASSERT(penta->Id()!=this->id);
	}
	
	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::IsInput{{{1*/
bool Penta::IsInput(int name){
	if (
				name==ThicknessEnum ||
				name==SurfaceEnum ||
				name==BedEnum ||
				name==SurfaceSlopeXEnum ||
				name==SurfaceSlopeYEnum ||
				name==MeltingRateEnum ||
				name==AccumulationRateEnum ||
				name==GeothermalFluxEnum ||
				name==SurfaceAreaEnum||
				name==PressureEnum ||
				name==VxEnum ||
				name==VyEnum ||
				name==VzEnum ||
				name==VxObsEnum ||
				name==VyObsEnum ||
				name==VzObsEnum ||
				name==TemperatureEnum ||
				name==ControlParameterEnum ||
				name==CmResponseEnum ||
				name==DragCoefficientEnum ||
				name==GradientEnum ||
				name==OldGradientEnum  ||
				name==ConvergedEnum
				) {
		return true;
	}
	else return false;
}
/*}}}*/
/*FUNCTION Penta::IsOnShelf {{{1*/
bool   Penta::IsOnShelf(){

	bool onshelf;
	inputs->GetParameterValue(&onshelf,ElementOnIceShelfEnum);
	return onshelf;

}
/*}}}*/
/*FUNCTION Penta::IsOnWater {{{1*/
bool   Penta::IsOnWater(){

	bool onwater;
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	return onwater;

}
/*}}}*/
/*FUNCTION Penta::IsOnSurface{{{1*/
bool Penta::IsOnSurface(void){

	bool onsurface;
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);
	return onsurface;

}
/*}}}*/
/*FUNCTION Penta::IsOnBed{{{1*/
bool Penta::IsOnBed(void){

	bool onbed;
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	return onbed;

}
/*}}}*/
/*FUNCTION Penta::ReduceMatrixStokes {{{1*/
void Penta::ReduceMatrixStokes(double* Ke_reduced, double* Ke_temp){

	int i,j;

	double Kii[24][24];
	double Kib[24][3];
	double Kbb[3][3];
	double Kbi[3][24];
	double Kbbinv[3][3];
	double KibKbbinv[24][3];
	double Kright[24][24];

	/*Create the four matrices used for reduction */
	for(i=0;i<24;i++){
		for(j=0;j<24;j++){
			Kii[i][j]=*(Ke_temp+27*i+j);
		}
		for(j=0;j<3;j++){
			Kib[i][j]=*(Ke_temp+27*i+j+24);
		}
	}
	for(i=0;i<3;i++){
		for(j=0;j<24;j++){
			Kbi[i][j]=*(Ke_temp+27*(i+24)+j);
		}
		for(j=0;j<3;j++){
			Kbb[i][j]=*(Ke_temp+27*(i+24)+j+24);
		}
	}

	/*Inverse the matrix corresponding to bubble part Kbb */
	Matrix3x3Invert(&Kbbinv[0][0], &Kbb[0][0]);

	/*Multiply matrices to create the reduce matrix Ke_reduced */
	MatrixMultiply(&Kib[0][0],24,3,0,&Kbbinv[0][0],3,3,0,&KibKbbinv[0][0],0);
	MatrixMultiply(&KibKbbinv[0][0],24,3,0,&Kbi[0][0],3,24,0,&Kright[0][0],0);

	/*Affect value to the reduced matrix */
	for(i=0;i<24;i++){
		for(j=0;j<24;j++){
			*(Ke_reduced+24*i+j)=Kii[i][j]-Kright[i][j];
		}
	}
}
/*}}}*/
/*FUNCTION Penta::ReduceVectorStokes {{{1*/
void Penta::ReduceVectorStokes(double* Pe_reduced, double* Ke_temp, double* Pe_temp){

	int i,j;

	double Pi[24];
	double Pb[3];
	double Kbb[3][3];
	double Kib[24][3];
	double Kbbinv[3][3];
	double KibKbbinv[24][3];
	double Pright[24];

	/*Create the four matrices used for reduction */
	for(i=0;i<24;i++){
		Pi[i]=*(Pe_temp+i);
	}
	for(i=0;i<3;i++){
		Pb[i]=*(Pe_temp+i+24);
	}
	for(j=0;j<3;j++){
		for(i=0;i<24;i++){
			Kib[i][j]=*(Ke_temp+3*i+j);
		}
		for(i=0;i<3;i++){
			Kbb[i][j]=*(Ke_temp+3*(i+24)+j);
		}
	}

	/*Inverse the matrix corresponding to bubble part Kbb */
	Matrix3x3Invert(&Kbbinv[0][0], &Kbb[0][0]);

	/*Multiply matrices to create the reduce matrix Ke_reduced */
	MatrixMultiply(&Kib[0][0],24,3,0,&Kbbinv[0][0],3,3,0,&KibKbbinv[0][0],0);
	MatrixMultiply(&KibKbbinv[0][0],24,3,0,&Pb[0],3,1,0,&Pright[0],0);

	/*Affect value to the reduced matrix */
	for(i=0;i<24;i++){
		*(Pe_reduced+i)=Pi[i]-Pright[i];
	}
}
/*}}}*/
/*FUNCTION Penta::SetClone {{{1*/
void  Penta::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
/*FUNCTION Penta::SpawnTria {{{1*/
Tria*  Penta::SpawnTria(int g0, int g1, int g2){

	int i;
	int analysis_counter;
	
	/*go into parameters and get the analysis_counter: */
	this->parameters->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*out of grids g0,g1 and g2 from Penta, build a tria element: */
	Tria* tria=NULL;
	int indices[3];
	int zero=0;
	Parameters* tria_parameters=NULL;
	Inputs* tria_inputs=NULL;
	Results* tria_results=NULL;

	indices[0]=g0;
	indices[1]=g1;
	indices[2]=g2;

	tria_parameters=this->parameters;
	tria_inputs=(Inputs*)this->inputs->SpawnTriaInputs(indices);
	tria_results=(Results*)this->results->SpawnTriaResults(indices);
	
	tria=new Tria();
	tria->id=this->id;
	tria->inputs=tria_inputs;
	tria->results=tria_results;
	tria->parameters=tria_parameters;
	tria->element_type=P1Enum; //Only P1 CG for now (TO BE CHANGED)
	this->SpawnTriaHook(dynamic_cast<TriaHook*>(tria),&indices[0]);

	/*Spawn matice*/
	tria->matice=NULL;
	tria->matice=(Matice*)this->matice->copy();
	delete tria->matice->inputs;
	tria->matice->inputs=(Inputs*)this->matice->inputs->SpawnTriaInputs(indices);

	/*recover nodes, matice and matpar: */
	tria->nodes=(Node**)tria->hnodes[analysis_counter]->deliverp();
	tria->matpar=(Matpar*)tria->hmatpar->delivers();
	
	return tria;
}
/*}}}*/
/*FUNCTION Penta::SurfaceNormal {{{1*/
void Penta::SurfaceNormal(double* surface_normal, double xyz_list[3][3]){

	int i;
	double v13[3];
	double v23[3];
	double normal[3];
	double normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	normal_norm=sqrt( pow(normal[0],2)+pow(normal[1],2)+pow(normal[2],2) );

	*(surface_normal)=normal[0]/normal_norm;
	*(surface_normal+1)=normal[1]/normal_norm;
	*(surface_normal+2)=normal[2]/normal_norm;

}
/*}}}*/
