function plot_overlay(md,data,options,plotlines,plotcols,i)
%PLOT_OVERLAY - superimpose radar image to a given field
%
%   Usage:
%      plot_overlay(md,options,plotlines,plotcols,i)
%
%   See also: PLOTMODEL

%process mesh and data
[x y z elements is2d]=processmesh(md,[],options);
[data datatype]=processdata(md,data,options);

%check is2d
if ~is2d, 
	error('buildoverlay error message: overlay not supported for 3d meshes, project on a layer');
end
if datatype==3,
	error('buildoverlay error message: overlay not supported for quiver plots');
end

% radar power {{{1
if ~any(isnan(md.sarxm)) & ~any(isnan(md.sarym)) & ~any(isnan(md.sarpwr)),
	disp('plot_overlay info: the radar image held by the model is being used');
else
	t1=clock; fprintf('%s','Extracting radar image...');
	xlim=getfieldvalue(options,'xlim',[min(x) max(x)])/getfieldvalue(options,'unit',1);
	ylim=getfieldvalue(options,'ylim',[min(y) max(y)])/getfieldvalue(options,'unit',1);
	md=radarpower(md,getfieldvalue(options,'hem',md.hemisphere),xlim,ylim,getfieldvalue(options,'highres',0));
	t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);
end%}}}

% InterpFromMeshToGrid -> data_grid {{{1
redo=1;
if (ischar(data) & isstruct(md.mesh2grid_parameters) & ismember(data,mesh2grid_parameters)),
	choice=input(['mesh2grid has already been called for the parameter ''' data '''. Do you want to call it again (y/n)?'],'s');
	if strcmp(choice,'y')
		disp('use previous mesh2grid result');
		x_m=md.mesh2grid_x_m;
		y_m=md.mesh2grid_y_m;
		data_grid=md.mesh2grid_data;
		redo=0;
	end
end
if redo,
	%apply caxis if required
	if exist(options,'caxis'),
		caxis_opt=getfieldvalue(options,'caxis');
		data(find(data<caxis_opt(1)))=caxis_opt(1);
		data(find(data>caxis_opt(2)))=caxis_opt(2);
	end

	%use InterpFromMeshToGrid to get an gridded data to display using imagesc
	xlim=getfieldvalue(options,'xlim',[min(x) max(x)]);
	ylim=getfieldvalue(options,'ylim',[min(y) max(y)]);
	cornereast =min(xlim);
	cornernorth=max(ylim);
	xspacing=(max(xlim)-min(xlim))/(length(md.sarxm));
	yspacing=(max(ylim)-min(ylim))/(length(md.sarym));
	nlines=length(md.sarym);
	ncols =length(md.sarxm);
	t1=clock; fprintf('%s','Interpolating data on grid...');
	[x_m y_m data_grid]=InterpFromMeshToGrid(elements,x,y,data,cornereast,cornernorth,xspacing,yspacing,nlines,ncols,getfieldvalue(options,'cutoff',1.5));
	t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);
end%}}}

%Generate RGB image{{{1

%Build hsv color image from radar and results
radar=md.sarpwr;
transparency=getfieldvalue(options,'alpha',1.5);  %Rignot's setting: 1.5
cutoff=getfieldvalue(options,'cutoff',1.5);       %Rignot's setting: 1.5

%intensity
v_data=radar/max(radar(:)); %For the principla image, use radar power as intensity
v_coba=ones(256,1);         %For the colorbar: maximal intensity

%hue
data_grid(find(data_grid<cutoff))=cutoff;   %cut all values below cutoff
data_coba=linspace(max(min(data(:)),cutoff),max(max(data(:)),cutoff),256);%prepare colorbar (256 values between min and max)
h_data=bytscl(data_grid)/(255+1);           %scale between 0 and 1 (log applied in processdata)
h_coba=bytscl(data_coba)/(255+1);           %scale between 0 and 1

%saturation
s_data=(0.5+10.^data_grid/125)/transparency;s_data(find(s_data>1))=1;s_data(find(s_data<0))=0;
s_coba=(0.5+10.^data_coba/125)/transparency;s_coba(find(s_coba>1))=1;s_coba(find(s_coba<0))=0;
s_data(find(data_grid==cutoff))=0;
s_coba(find(data_coba==cutoff))=0;

%Transform hsv to rgb
image_hsv=zeros(size(data_grid,1),size(data_grid,2),3);
image_hsv(:,:,1)=h_data;
image_hsv(:,:,2)=s_data;
image_hsv(:,:,3)=v_data;
image_rgb=hsv2rgb(image_hsv);
colorbar_hsv=zeros(size(data_coba,2),size(data_coba,1),3);
colorbar_hsv(:,:,1)=h_coba;
colorbar_hsv(:,:,2)=s_coba;
colorbar_hsv(:,:,3)=v_coba;
colorbar_rgb=hsv2rgb(colorbar_hsv);
%}}}

%Select plot area 
subplot(plotlines,plotcols,i);
%uncomment following lines to have more space
%P=get(gca,'pos');
%P(3)=P(3)+0.05;
%P(2)=P(2)+0.08;
%P(1)=P(1)-0.02;
%set(gca,'pos',P);

%Plot: 
imagesc(md.sarxm*getfieldvalue(options,'unit',1),md.sarym*getfieldvalue(options,'unit',1),image_rgb);set(gca,'YDir','normal');

%last step: mesh overlay?
if exist(options,'edgecolor')
	A=elements(:,1); B=elements(:,2); C=elements(:,3); 
	patch( 'Faces', [A B C], 'Vertices', [x y z],'FaceVertexCData', [1 1 1],'FaceColor','none','EdgeColor',getfieldvalue(options,'edgecolor'));
end

%Apply options {{{1
%Apply options, without colorbar and without grid
iscolorbar=getfieldvalue(options,'colorbar',1);
options=changefieldvalue(options,'colorbar',0);
options=addfielddefault(options,'axis','equal off');
options=addfielddefault(options,'colorbarpos',[0.80 0.70 0.02 0.15]);
applyoptions(md,data,options);

%colorbar
if iscolorbar,

	%create colorbar with correct colors and position
	%colorbar_rgb=buildoverlaycolorbar(md,data,getfieldvalue(options,'aplha',1.5));
	colorbar_handle=colorbar; 
	colorbar_image_handle=get(colorbar_handle,'Children'); 
	set(colorbar_image_handle,'CData',colorbar_rgb);
	set(colorbar_handle,'Position',getfieldvalue(options,'colorbarpos'));
	set(colorbar_handle,'FontSize',getfieldvalue(options,'colorbarfontsize',14));

	%modify ticks.
	scalestring=get(colorbar_handle,'YTickLabel');
	numvalues=length(get(colorbar_handle,'YTick'));

	scalestring=[];
	scaleminmax=[max(min(data),cutoff) max(data)];
	for i=1:numvalues,
		fraction=(i-1)/(numvalues-1);
		scalevalues(i)=scaleminmax(1)+(scaleminmax(2)-scaleminmax(1))*fraction;
		if exist(options,'log'),
			logvalue=getfieldvalue(options,'log');
			scalestring=[scalestring; sprintf('%-8.4g',round_ice(logvalue^scalevalues(i),2) )];
		else
			scalestring=[scalestring; sprintf('%-8.4g',round_ice(scalevalues(i),2) )];
		end
	end
	set(colorbar_handle,'YTickLabel',scalestring);
	set(colorbar_handle,'YColor',getfieldvalue(options,'FontColor','k'));
	if exist(options,'colorbartitle'),
		backup=gca;
		axes(colorbar_handle);lab=title(getfieldvalue(options,'colorbartitle'),'Color',getfieldvalue(options,'FontColor','k'));
		set(lab,'Rotation',getfieldvalue(options,'colorbartitlerotation',0));
		set(lab,'VerticalAlignment','bottom');
		axes(backup);
	end
end%}}}
