/*!\file Solverx
 * \brief solver
 */

#include "./Solverx.h"
#include "../../shared/shared.h"
#include "../../include/include.h"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

void	Solverx(Vec* puf, Mat Kff, Vec pf, Vec uf0,Parameters* parameters){

	/*output: */
	Vec uf=NULL;

	/*intermediary: */
	int local_m,local_n;
	
	/*Solver*/
	KSP ksp=NULL; 
	PC  pc=NULL;
	int iteration_number;
	PetscTruth flag;
	int solver_type;
	char* solver_string=NULL;

	/*Display message*/
	int verbose; parameters->FindParam(&verbose,VerboseEnum);
	if (verbose) _printf_("   Solving\n");

	/*First, check that f-set is not NULL, ie model is fully constrained: */
	if(!Kff){
		*puf=NULL; return;
	}

	/*If initial guess for solution exists, use it to create uf, otherwise, 
	 * duplicate right hand side so that solution vector has same structure*/
	if(uf0){
		VecDuplicate(uf0,&uf); VecCopy(uf0,uf);
	}
	else{
		MatGetLocalSize(Kff,&local_m,&local_n);uf=NewVec(local_n,true);
	}

	/*Before preparing the solver, add options to the options database*/
	parameters->FindParam(&solver_string,SolverStringEnum);
	PetscOptionsInsertMultipleString(solver_string);

	/*Process solver_string to see if we are not using special types of external solvers: */
	PetscOptionsDetermineSolverType(&solver_type,solver_string);

	#if _PETSC_VERSION_ == 2 
	if (solver_type==MUMPSPACKAGE_LU){
		/*Convert Kff to MATTAIJMUMPS: */
		MatConvert(Kff,MATAIJMUMPS,MAT_REUSE_MATRIX,&Kff);
	}
	if (solver_type==MUMPSPACKAGE_CHOL){
		/*Convert Kff to MATTSBAIJMUMPS: */
		MatConvert(Kff,MATSBAIJMUMPS,MAT_REUSE_MATRIX,&Kff);
	}
	if (solver_type==SPOOLESPACKAGE_LU){
		/*Convert Kff to MATTSBAIJMUMPS: */
		MatConvert(Kff,MATAIJSPOOLES,MAT_REUSE_MATRIX,&Kff);
	}
	if (solver_type==SPOOLESPACKAGE_CHOL){
		/*Convert Kff to MATTSBAIJMUMPS: */
		MatConvert(Kff,MATSBAIJSPOOLES,MAT_REUSE_MATRIX,&Kff);
	}
	if (solver_type==SUPERLUDISTPACKAGE){
		/*Convert Kff to MATTSBAIJMUMPS: */
		MatConvert(Kff,MATSUPERLU_DIST,MAT_REUSE_MATRIX,&Kff);
	}
	#endif

	/*Prepare solver*/
	KSPCreate(MPI_COMM_WORLD,&ksp);
	KSPSetOperators(ksp,Kff,Kff,DIFFERENT_NONZERO_PATTERN);
	KSPSetFromOptions(ksp);

	#if _PETSC_VERSION_ == 3 
	/*specific solver?: */
	KSPGetPC(ksp,&pc);
	if (solver_type==MUMPSPACKAGE_LU){
		PCFactorSetMatSolverPackage(pc,MAT_SOLVER_MUMPS);
	}
	#endif


	/*If initial guess for solution, use it, except if we are using the MUMPS direct solver, where any initial 
	 * guess will crash Petsc: */
	if (uf0){
		if( (solver_type!=MUMPSPACKAGE_LU) && (solver_type!=MUMPSPACKAGE_CHOL) && (solver_type!=SPOOLESPACKAGE_LU)&& (solver_type!=SPOOLESPACKAGE_CHOL)&& (solver_type!=SUPERLUDISTPACKAGE)){
			KSPSetInitialGuessNonzero(ksp,PETSC_TRUE);
		}
	}
	KSPSolve(ksp,pf,uf);
	
	/*Check convergence*/
	KSPGetIterationNumber(ksp,&iteration_number);
	if (iteration_number<0) ISSMERROR("%s%i"," Solver diverged at iteration number: ",-iteration_number);

	/*Free ressources:*/
	KSPFree(&ksp);
	xfree((void**)&solver_string);
	
	/*Assign output pointers:*/
	*puf=uf;
}
