function md=qmu(md,varargin)
%QMU - apply Quantification of Margins and Uncertainties techniques 
%      to a solution sequence (like diagnostic.m, progonstic.m, etc ...), 
%      using the Dakota software from Sandia.
%   Usage:
%      md=qmu(md,varargin)
%      where varargin is a lit of paired arguments. 
%      arguments can be: 'analysis_type': 'diagnostic','thermal','prognostic','transient'
%      arguments can be: 'sub_analysis_type': 'transient','steady','' (default if empty = 'steady')
%      arguments can be: 'package': 'macayeal','ice','cielo' (default if not specified = 'cielo')
%
%
%   Examples:
%      md=qmu(md,'analysis_type','diagnostic','package','cielo');
%      md=qmu(md,'analysis_type','control','package','ice');
%      md=qmu(md,'analysis_type','thermal','sub_analysis_type','transient','package','ice');
%      md=qmu(md,'analysis_type','thermal','sub_analysis_type','steady','package','cielo');
%      md=qmu(md,'analysis_type','thermal','package','cielo');
%
%   On top of these solution arguments (found also in solve.m), user can specify QMU 
%   specific arguments: 
%
%       qmudir:  any directory where to run the qmu analysis
%       qmufile: input file for Dakota
%       ivar: selection number for variables input (if several are specified in variables)
%       iresp: same thing for response functions
%       imethod: same thing for methods
%       iparams: same thing for params
%       overwrite: overwrite qmudir
%       outfiles: (John?)
%       rstfile: backup file name
%       rundakota: (John?)
%       runmpi: (John?)

%some checks on list of arguments
global ISSM_DIR;

%recover options
options=recover_solve_options(md,varargin{:});

%add default options
options=process_solve_options(options);

%recover some fields
md.analysis_type=options.analysis_type;
md.sub_analysis_type=options.sub_analysis_type;
package=options.package;

%Use package to set solution namespace
usenamespace(package);

if ~ismodelselfconsistent(md,package),
	error(' '); %previous error messages should explain what is going on.
end

displaystring(md.debug,'\n%s\n','launching solution sequence');

%Ok, are we running serially or in parallel?
if ~strcmpi(md.cluster,'none'),
	%ok, we are running parallel. 
	md=qmuparallel(md,varargin{:});
else
	%we are running serially.
	md=qmuserial(md,varargin{:});
end
