/*!\file ElementMatrix.cpp
 * \brief: implementation of the ElementMatrix object, used to plug values from element into global stiffness matrix
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
/*}}}*/

/*ElementMatrix constructors and destructor*/
/*FUNCTION ElementMatrix::ElementMatrix(){{{1*/
ElementMatrix::ElementMatrix(){

	this->nrows=0;
	this->ncols=0;
	this->values=NULL;
	this->square=false;
	this->kff=false;

	this->row_fsize=0;
	this->row_flocaldoflist=NULL;
	this->row_fglobaldoflist=NULL;
	this->row_ssize=0;
	this->row_slocaldoflist=NULL;
	this->row_sglobaldoflist=NULL;

	this->col_fsize=0;
	this->col_flocaldoflist=NULL;
	this->col_fglobaldoflist=NULL;
	this->col_ssize=0;
	this->col_slocaldoflist=NULL;
	this->col_sglobaldoflist=NULL;

}
/*}}}*/
/*FUNCTION ElementMatrix::ElementMatrix(int gsize,bool square,int* in_gglobaldoflist){{{1*/
ElementMatrix::ElementMatrix(int gsize,bool square,int* in_gglobaldoflist){

	if(square=false)ISSMERROR(" calling square constructor with false square flag!");

	this->nrows=gsize;
	this->ncols=gsize;
	this->square=true;
	this->kff=false;

	/*fill values with 0: */
	this->values=(double*)xcalloc(this->nrows*this->ncols,sizeof(double));
	
	if(this->nrows){
		this->gglobaldoflist=(int*)xmalloc(this->nrows*sizeof(int));
		memcpy(this->gglobaldoflist,in_gglobaldoflist,this->nrows*sizeof(int));
	}
	else{
		this->gglobaldoflist=NULL;
	}

	/*don't do rows and cols, because kff is false:*/
	this->row_fsize=0;
	this->row_flocaldoflist=NULL;
	this->row_fglobaldoflist=NULL;
	this->row_ssize=0;
	this->row_slocaldoflist=NULL;
	this->row_sglobaldoflist=NULL;
	
	this->col_fsize=0;
	this->col_flocaldoflist=NULL;
	this->col_fglobaldoflist=NULL;
	this->col_ssize=0;
	this->col_slocaldoflist=NULL;
	this->col_sglobaldoflist=NULL;

}
/*}}}*/
/*FUNCTION ElementMatrix::ElementMatrix(int gsize,bool square,int* flocaldoflist,int* fglobaldoflist,int fsize,int* slocaldoflist,int* sglobaldoflist,int ssize){{{1*/
ElementMatrix::ElementMatrix(int gsize,bool square,int* flocaldoflist,int* fglobaldoflist,int fsize,int* slocaldoflist,int* sglobaldoflist,int ssize){

	if(square=false)ISSMERROR(" calling square constructor with false square flag!");

	this->nrows=gsize;
	this->ncols=gsize;
	this->square=true;
	this->kff=true;

	/*fill values with 0: */
	this->values=(double*)xcalloc(this->nrows*this->ncols,sizeof(double));
	
	/*row dofs: */
	this->row_fsize=fsize;
	if(fsize){
		this->row_flocaldoflist=(int*)xmalloc(fsize*sizeof(int));
		this->row_fglobaldoflist=(int*)xmalloc(fsize*sizeof(int));
		memcpy(this->row_flocaldoflist,flocaldoflist,fsize*sizeof(int));
		memcpy(this->row_fglobaldoflist,fglobaldoflist,fsize*sizeof(int));
	}
	else{
		this->row_flocaldoflist=NULL;
		this->row_fglobaldoflist=NULL;
	}

	this->row_ssize=ssize;
	if(ssize){
		this->row_slocaldoflist=(int*)xmalloc(ssize*sizeof(int));
		this->row_sglobaldoflist=(int*)xmalloc(ssize*sizeof(int));
		memcpy(this->row_slocaldoflist,slocaldoflist,ssize*sizeof(int));
		memcpy(this->row_sglobaldoflist,sglobaldoflist,ssize*sizeof(int));
	}
	else{
		this->row_slocaldoflist=NULL;
		this->row_sglobaldoflist=NULL;
	}

	/*don't do cols, we can't pick them up from the rows: */
	this->col_fsize=0;
	this->col_flocaldoflist=NULL;
	this->col_fglobaldoflist=NULL;
	this->col_ssize=0;
	this->col_slocaldoflist=NULL;
	this->col_sglobaldoflist=NULL;

	/*don't do g-set: */
	this->gglobaldoflist=NULL;

}
/*}}}*/
/*FUNCTION ElementMatrix::~ElementMatrix(){{{1*/
ElementMatrix::~ElementMatrix(){
	
	xfree((void**)&this->values);
	xfree((void**)&this->gglobaldoflist);
	xfree((void**)&this->row_flocaldoflist);
	xfree((void**)&this->row_fglobaldoflist);
	xfree((void**)&this->row_slocaldoflist);
	xfree((void**)&this->row_sglobaldoflist);
	xfree((void**)&this->col_flocaldoflist);
	xfree((void**)&this->col_fglobaldoflist);
	xfree((void**)&this->col_slocaldoflist);
	xfree((void**)&this->col_sglobaldoflist);
}
/*}}}*/

/*ElementMatrix specific routines: */
/*FUNCTION ElementMatrix::AddValues(double* Ke_gg){{{1*/
void ElementMatrix::AddValues(double* Ke_gg){

	if(Ke_gg){
		for (int i=0;i<this->nrows;i++){
			for(int j=0;j<this->ncols;j++){
				*(this->values+this->ncols*i+j)+=*(Ke_gg+this->ncols*i+j);
			}
		}
	}
}
/*}}}*/
/*FUNCTION ElementMatrix::AddToGlobal(Mat Kgg, Mat Kff, Mat Kfs){{{1*/
void ElementMatrix::AddToGlobal(Mat Kgg, Mat Kff, Mat Kfs){

	int i,j;
	double* localvalues=NULL;

	if(this->square){
		/*only use row dofs to add values into global matrices: */
		
		if(!this->kff){
			/*add local values into global  matrix, using the fglobaldoflist: */
			MatSetValues(Kgg,this->nrows,this->gglobaldoflist,this->nrows,this->gglobaldoflist,(const double*)values,ADD_VALUES);
		}
		else{
			if(this->row_fsize){
				/*first, retrieve values that are in the f-set from the g-set values matrix: */
				localvalues=(double*)xmalloc(this->row_fsize*this->row_fsize*sizeof(double));
				for(i=0;i<this->row_fsize;i++){
					for(j=0;j<this->row_fsize;j++){
						*(localvalues+this->row_fsize*i+j)=*(this->values+this->ncols*this->row_flocaldoflist[i]+this->row_flocaldoflist[j]);
					}
				}
				/*add local values into global  matrix, using the fglobaldoflist: */
				MatSetValues(Kff,this->row_fsize,this->row_fglobaldoflist,this->row_fsize,this->row_fglobaldoflist,(const double*)localvalues,ADD_VALUES);

				/*Free ressources:*/
				xfree((void**)&localvalues);
			}


			if((this->row_ssize!=0) && (this->row_fsize!=0)){
				/*first, retrieve values that are in the f and s-set from the g-set values matrix: */
				localvalues=(double*)xmalloc(this->row_fsize*this->row_ssize*sizeof(double));
				for(i=0;i<this->row_fsize;i++){
					for(j=0;j<this->row_ssize;j++){
						*(localvalues+this->row_ssize*i+j)=*(this->values+this->ncols*this->row_flocaldoflist[i]+this->row_slocaldoflist[j]);
					}
				}
				/*add local values into global  matrix, using the fglobaldoflist: */
				MatSetValues(Kfs,this->row_fsize,this->row_fglobaldoflist,this->row_ssize,this->row_sglobaldoflist,(const double*)localvalues,ADD_VALUES);

				/*Free ressources:*/
				xfree((void**)&localvalues);
			}
		}
	}
	else{
		ISSMERROR(" non square matrix AddToGlobal routine not support yet!");
	}

}
/*}}}*/
/*FUNCTION ElementMatrix::Echo(void){{{1*/
void ElementMatrix::Echo(void){

	int i,j;
	printf("Element Matrix echo: \n");
	printf("   nrows: %i\n",nrows);
	printf("   ncols: %i\n",ncols);
	printf("   square: %s\n",square?"true":"false");
	printf("   kff: %s\n",kff?"true":"false");

	printf("   values: \n");
	for(i=0;i<nrows;i++){
		printf("      %i: ",i);
		for(j=0;j<ncols;j++) printf("%g ",*(values+ncols*i+j));
		printf("\n");
	}

	printf("   gglobaldoflist (%p): ",gglobaldoflist);
    if(gglobaldoflist) for(i=0;i<nrows;i++)printf("%i ",gglobaldoflist[i]); printf("\n");

	printf("   row_fsize: %i\n",row_fsize);
	printf("   row_flocaldoflist (%p): ",row_flocaldoflist);
    if(row_flocaldoflist) for(i=0;i<row_fsize;i++)printf("%i ",row_flocaldoflist[i]); printf("\n");
	printf("   row_fglobaldoflist (%p): ",row_fglobaldoflist);
	if(row_fglobaldoflist)for(i=0;i<row_fsize;i++)printf("%i ",row_fglobaldoflist[i]); printf("\n");

	printf("   row_ssize: %i\n",row_ssize);
	printf("   row_slocaldoflist (%p): ",row_slocaldoflist);
    if(row_slocaldoflist)for(i=0;i<row_ssize;i++)printf("%i ",row_slocaldoflist[i]); printf("\n");
	printf("   row_sglobaldoflist (%p): ",row_sglobaldoflist);
	if(row_sglobaldoflist)for(i=0;i<row_ssize;i++)printf("%i ",row_sglobaldoflist[i]); printf("\n");

	if(!square){
		printf("   col_fsize: %i\n",col_fsize);
		printf("   col_flocaldoflist (%p): ",col_flocaldoflist);
		if(col_flocaldoflist)for(i=0;i<col_fsize;i++)printf("%i ",col_flocaldoflist[i]); printf("\n");
		printf("   col_fglobaldoflist (%p): ",col_fglobaldoflist);
		if(col_fglobaldoflist)for(i=0;i<col_fsize;i++)printf("%i ",col_fglobaldoflist[i]); printf("\n");

		printf("   col_ssize: %i\n",col_ssize);
		printf("   col_slocaldoflist (%p): ",col_slocaldoflist);
		if(col_slocaldoflist)for(i=0;i<col_ssize;i++)printf("%i ",col_slocaldoflist[i]); printf("\n");
		printf("   col_sglobaldoflist (%p): ",col_sglobaldoflist);
		if(col_sglobaldoflist)for(i=0;i<col_ssize;i++)printf("%i ",col_sglobaldoflist[i]); printf("\n");
	}
}
/*}}}*/
