/*!\file:  thermal.cpp
 * \brief: thermal solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "thermal"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

int main(int argc,char* *argv){

	int i,n;
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   numberofnodes;

	/*Fem models : */
	FemModel femmodels[2];

	/*initial velocity and pressure: */
	double* u_g=NULL;
	double* p_g=NULL;
	double* time=NULL;
	double  dt;
	double  ndt;
	int     nsteps;
	int     debug=0;

	/*solution vectors: */
	Vec* t_g=NULL;
	Vec* m_g=NULL;

	ParameterInputs* inputs=NULL;
	Param*           param=NULL;

	int    waitonlock=0;
	int    sub_analysis_type;
	double melting_offset;
	
	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	_printf_("read and create thermal finite element model:\n");
	CreateFemModel(&femmodels[0],fid,"thermal","");
	_printf_("read and create melting finite element model:\n");
	CreateFemModel(&femmodels[1],fid,"melting","");

	_printf_("initialize inputs:\n");
	femmodels[0].parameters->FindParam((void*)&u_g,"u_g");
	femmodels[0].parameters->FindParam((void*)&p_g,"p_g");
	femmodels[0].parameters->FindParam((void*)&numberofnodes,"numberofnodes");
	femmodels[0].parameters->FindParam((void*)&dt,"dt");
	femmodels[0].parameters->FindParam((void*)&ndt,"ndt");
	femmodels[0].parameters->FindParam((void*)&waitonlock,"waitonlock");
	femmodels[0].parameters->FindParam((void*)&sub_analysis_type,"sub_analysis_type");
	femmodels[0].parameters->FindParam((void*)&debug,"debug");

	inputs=new ParameterInputs;

	inputs->Add("velocity",u_g,3,numberofnodes);
	inputs->Add("pressure",p_g,1,numberofnodes);
	inputs->Add("dt",dt);

	//erase velocities and pressure embedded in parameters
	param=(Param*)femmodels[0].parameters->FindParamObject("u_g");
	femmodels[0].parameters->DeleteObject((Object*)param);
	param=(Param*)femmodels[0].parameters->FindParamObject("p_g");
	femmodels[0].parameters->DeleteObject((Object*)param);
	param=(Param*)femmodels[1].parameters->FindParamObject("u_g");
	femmodels[1].parameters->DeleteObject((Object*)param);
	param=(Param*)femmodels[1].parameters->FindParamObject("p_g");
	femmodels[1].parameters->DeleteObject((Object*)param);

	if(sub_analysis_type==SteadyAnalysisEnum()){

		time=(double*)xmalloc(sizeof(double));
		time[0]=0;

		/*allocate t_g and m_g arrays: */
		t_g=(Vec*)xmalloc(sizeof(Vec));
		m_g=(Vec*)xmalloc(sizeof(Vec));

		if(debug)_printf_("computing temperatures:\n");
		thermal_core(&t_g[0],&melting_offset,&femmodels[0],inputs,ThermalAnalysisEnum(),SteadyAnalysisEnum());
		inputs->Add("temperature",t_g[0],1,numberofnodes);
		inputs->Add("melting_offset",melting_offset);
		
		if(debug)_printf_("computing melting:\n");
		diagnostic_core_linear(&m_g[0],&femmodels[1],inputs,MeltingAnalysisEnum(),SteadyAnalysisEnum());
	}
	else{
		
		nsteps=(int)(ndt/dt);
		time=(double*)xmalloc((nsteps+1)*sizeof(double));
		time[0]=0;

		/*allocate t_g and m_g arrays: */
		t_g=(Vec*)xmalloc((nsteps+1)*sizeof(Vec));
		m_g=(Vec*)xmalloc((nsteps+1)*sizeof(Vec));

		//initialize temperature and melting
		femmodels[0].parameters->FindParam((void*)&t_g[0],"t_g");
		femmodels[1].parameters->FindParam((void*)&m_g[0],"m_g");

		//erase temperature and melting embedded in parameters
		param=(Param*)femmodels[0].parameters->FindParamObject("t_g");
		femmodels[0].parameters->DeleteObject((Object*)param);
		param=(Param*)femmodels[1].parameters->FindParamObject("m_g");
		femmodels[1].parameters->DeleteObject((Object*)param);

		for(i=0;i<nsteps;i++){
			if(debug)_printf_("time step: %i/%i\n",i+1,nsteps);
			time[i+1]=(i+1)*dt;
			
			if(debug)_printf_("computing temperatures:\n");
			inputs->Add("temperature",t_g[i],1,numberofnodes);
			thermal_core(&t_g[i+1],&melting_offset,&femmodels[0],inputs,ThermalAnalysisEnum(),TransientAnalysisEnum());
			
			if(debug)_printf_("computing melting:\n");
			inputs->Add("temperature",t_g[i+1],1,numberofnodes);
			inputs->Add("melting_offset",melting_offset);
			diagnostic_core_linear(&m_g[i+1],&femmodels[1],inputs,MeltingAnalysisEnum(),TransientAnalysisEnum());
		}
	}

	_printf_("write results to disk:\n");
	OutputThermal(t_g,m_g,time,&femmodels[0],outputfilename);

	_printf_("write lock file:\n");
	femmodels[0].parameters->FindParam((void*)&waitonlock,"waitonlock");

	if (waitonlock){
		WriteLockFile(lockname);
	}
		
	_printf_("closing MPI and Petsc\n");
	MPI_Barrier(MPI_COMM_WORLD);

	/*Close MPI libraries: */
	PetscFinalize(); 

	/*end module: */
	MODULEEND();
	
	return 0; //unix success return;
}
