/*!\file ElementMatrix.cpp
 * \brief: implementation of the ElementMatrix object, used to plug values from element into global stiffness matrix
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
/*}}}*/

/*ElementMatrix constructors and destructor*/
/*FUNCTION ElementMatrix::ElementMatrix(){{{1*/
ElementMatrix::ElementMatrix(){

	this->nrows=0;
	this->ncols=0;
	this->symmetric=false;
	this->row_fsize=0;
	this->values=NULL;
	this->row_finternaldoflist=NULL;
	this->row_fexternaldoflist=NULL;
	this->row_ssize=0;
	this->row_sinternaldoflist=NULL;
	this->row_sexternaldoflist=NULL;
	this->col_fsize=0;
	this->col_finternaldoflist=NULL;
	this->col_fexternaldoflist=NULL;
	this->col_ssize=0;
	this->col_sinternaldoflist=NULL;
	this->col_sexternaldoflist=NULL;

}
/*}}}*/
/*FUNCTION ElementMatrix::ElementMatrix(int gsize,bool symmetric,int* in_gexternaldoflist){{{1*/
ElementMatrix::ElementMatrix(int gsize,bool symmetric,int* in_gexternaldoflist){

	if(symmetric=false)ISSMERROR(" calling symmetric constructor with false symmetric flag!");

	this->nrows=gsize;
	this->ncols=gsize;
	this->symmetric=true;
	this->kff=false;

	/*fill values with 0: */
	this->values=(double*)xcalloc(this->nrows*this->ncols,sizeof(double));
	
	if(this->nrows){
		this->gexternaldoflist=(int*)xmalloc(this->nrows*sizeof(int));
		memcpy(this->gexternaldoflist,in_gexternaldoflist,this->nrows*sizeof(int));
	}
	else this->gexternaldoflist=NULL;

	/*don't do rows and cols, because kff is false:*/
	this->row_fsize=0;
	this->row_finternaldoflist=NULL;
	this->row_fexternaldoflist=NULL;
	this->row_ssize=0;
	this->row_sinternaldoflist=NULL;
	this->row_sexternaldoflist=NULL;
	
	this->col_fsize=0;
	this->col_finternaldoflist=NULL;
	this->col_fexternaldoflist=NULL;
	this->col_ssize=0;
	this->col_sinternaldoflist=NULL;
	this->col_sexternaldoflist=NULL;

}
/*}}}*/
/*FUNCTION ElementMatrix::ElementMatrix(int gsize,bool symmetric,int* finternaldoflist,int* fexternaldoflist,int fsize,int* sinternaldoflist,int* sexternaldoflist,int ssize){{{1*/
ElementMatrix::ElementMatrix(int gsize,bool symmetric,int* finternaldoflist,int* fexternaldoflist,int fsize,int* sinternaldoflist,int* sexternaldoflist,int ssize){

	if(symmetric=false)ISSMERROR(" calling symmetric constructor with false symmetric flag!");

	this->nrows=gsize;
	this->ncols=gsize;
	this->symmetric=true;
	this->kff=true;

	/*fill values with 0: */
	this->values=(double*)xcalloc(this->nrows*this->ncols,sizeof(double));
	
	/*row dofs: */
	this->row_fsize=fsize;
	if(fsize){
		this->row_finternaldoflist=(int*)xmalloc(fsize*sizeof(int));
		this->row_fexternaldoflist=(int*)xmalloc(fsize*sizeof(int));
		memcpy(this->row_finternaldoflist,finternaldoflist,fsize*sizeof(int));
		memcpy(this->row_fexternaldoflist,fexternaldoflist,fsize*sizeof(int));
	}
	else{
		this->row_finternaldoflist=NULL;
		this->row_fexternaldoflist=NULL;
	}

	this->row_ssize=ssize;
	if(ssize){
		this->row_sinternaldoflist=(int*)xmalloc(ssize*sizeof(int));
		this->row_sexternaldoflist=(int*)xmalloc(ssize*sizeof(int));
		memcpy(this->row_sinternaldoflist,sinternaldoflist,ssize*sizeof(int));
		memcpy(this->row_sexternaldoflist,sexternaldoflist,ssize*sizeof(int));
	}
	else{
		this->row_sinternaldoflist=NULL;
		this->row_sexternaldoflist=NULL;
	}

	/*don't do cols, we can't pick them up from the rows: */
	this->col_fsize=0;
	this->col_finternaldoflist=NULL;
	this->col_fexternaldoflist=NULL;
	this->col_ssize=0;
	this->col_sinternaldoflist=NULL;
	this->col_sexternaldoflist=NULL;

	/*don't do g-set: */
	this->gexternaldoflist=NULL;

}
/*}}}*/
/*FUNCTION ElementMatrix::~ElementMatrix(){{{1*/
ElementMatrix::~ElementMatrix(){
	
	xfree((void**)&this->values);
	xfree((void**)&this->gexternaldoflist);
	xfree((void**)&this->row_finternaldoflist);
	xfree((void**)&this->row_fexternaldoflist);
	xfree((void**)&this->row_sinternaldoflist);
	xfree((void**)&this->row_sexternaldoflist);
	xfree((void**)&this->col_finternaldoflist);
	xfree((void**)&this->col_fexternaldoflist);
	xfree((void**)&this->col_sinternaldoflist);
	xfree((void**)&this->col_sexternaldoflist);
}
/*}}}*/

/*ElementMatrix specific routines: */
/*FUNCTION ElementMatrix::AddValues(double* Ke_gg){{{1*/
void ElementMatrix::AddValues(double* Ke_gg){

	int i,j;

	if(Ke_gg){
		for (i=0;i<this->nrows;i++){
			for(j=0;j<this->ncols;j++){
				*(this->values+this->ncols*i+j)+=*(Ke_gg+this->ncols*i+j);
			}
		}
	}
}
/*}}}*/
/*FUNCTION ElementMatrix::AddToGlobal(Mat Kgg, Mat Kff, Mat Kfs){{{1*/
void ElementMatrix::AddToGlobal(Mat Kgg, Mat Kff, Mat Kfs){

	int i,j;
	double* internalvalues=NULL;

	if(this->symmetric){
		/*only use row dofs to add values into global matrices: */
		
		if(!this->kff){
			/*add internal values into global  matrix, using the fexternaldoflist: */
			MatSetValues(Kgg,this->nrows,this->gexternaldoflist,this->nrows,this->gexternaldoflist,(const double*)values,ADD_VALUES);
		}
		else{
			if(this->row_fsize){
				/*first, retrieve values that are in the f-set from the g-set values matrix: */
				internalvalues=(double*)xmalloc(this->row_fsize*this->row_fsize*sizeof(double));
				for(i=0;i<this->row_fsize;i++){
					for(j=0;j<this->row_fsize;j++){
						*(internalvalues+this->row_fsize*i+j)=*(this->values+this->ncols*this->row_finternaldoflist[i]+this->row_finternaldoflist[j]);
					}
				}
				/*add internal values into global  matrix, using the fexternaldoflist: */
				MatSetValues(Kff,this->row_fsize,this->row_fexternaldoflist,this->row_fsize,this->row_fexternaldoflist,(const double*)internalvalues,ADD_VALUES);

				/*Free ressources:*/
				xfree((void**)&internalvalues);
			}


			if((this->row_ssize!=0) && (this->row_fsize!=0)){
				/*first, retrieve values that are in the f and s-set from the g-set values matrix: */
				internalvalues=(double*)xmalloc(this->row_fsize*this->row_ssize*sizeof(double));
				for(i=0;i<this->row_fsize;i++){
					for(j=0;j<this->row_ssize;j++){
						*(internalvalues+this->row_ssize*i+j)=*(this->values+this->ncols*this->row_finternaldoflist[i]+this->row_sinternaldoflist[j]);
					}
				}
				/*add internal values into global  matrix, using the fexternaldoflist: */
				MatSetValues(Kfs,this->row_fsize,this->row_fexternaldoflist,this->row_ssize,this->row_sexternaldoflist,(const double*)internalvalues,ADD_VALUES);

				/*Free ressources:*/
				xfree((void**)&internalvalues);
			}
		}
	}
	else{
		ISSMERROR(" unsymmetric AddToGlobal routine not support yet!");
	}

}
/*}}}*/
/*FUNCTION ElementMatrix::Echo(void){{{1*/
void ElementMatrix::Echo(void){

	int i,j;
	printf("Element Matrix echo: \n");
	printf("   nrows: %i\n",nrows);
	printf("   ncols: %i\n",ncols);
	printf("   symmetric: %s\n",symmetric?"true":"false");
	printf("   kff: %s\n",kff?"true":"false");

	printf("   values: \n");
	for(i=0;i<nrows;i++){
		printf("      %i: ",i);
		for(j=0;j<ncols;j++){
			printf("%g ",*(values+ncols*i+j));
		}
		printf("\n");
	}

	printf("   gexternaldoflist (%p): ",gexternaldoflist);
    if(gexternaldoflist) for(i=0;i<nrows;i++)printf("%i ",gexternaldoflist[i]); printf("\n");

	printf("   row_fsize: %i\n",row_fsize);
	printf("   row_finternaldoflist (%p): ",row_finternaldoflist);
    if(row_finternaldoflist) for(i=0;i<row_fsize;i++)printf("%i ",row_finternaldoflist[i]); printf("\n");
	printf("   row_fexternaldoflist (%p): ",row_fexternaldoflist);
	if(row_fexternaldoflist)for(i=0;i<row_fsize;i++)printf("%i ",row_fexternaldoflist[i]); printf("\n");

	printf("   row_ssize: %i\n",row_ssize);
	printf("   row_sinternaldoflist (%p): ",row_sinternaldoflist);
    if(row_sinternaldoflist)for(i=0;i<row_ssize;i++)printf("%i ",row_sinternaldoflist[i]); printf("\n");
	printf("   row_sexternaldoflist (%p): ",row_sexternaldoflist);
	if(row_sexternaldoflist)for(i=0;i<row_ssize;i++)printf("%i ",row_sexternaldoflist[i]); printf("\n");

	if(!symmetric){

		printf("   col_fsize: %i\n",col_fsize);
		printf("   col_finternaldoflist (%p): ",col_finternaldoflist);
		if(col_finternaldoflist)for(i=0;i<col_fsize;i++)printf("%i ",col_finternaldoflist[i]); printf("\n");
		printf("   col_fexternaldoflist (%p): ",col_fexternaldoflist);
		if(col_fexternaldoflist)for(i=0;i<col_fsize;i++)printf("%i ",col_fexternaldoflist[i]); printf("\n");

		printf("   col_ssize: %i\n",col_ssize);
		printf("   col_sinternaldoflist (%p): ",col_sinternaldoflist);
		if(col_sinternaldoflist)for(i=0;i<col_ssize;i++)printf("%i ",col_sinternaldoflist[i]); printf("\n");
		printf("   col_sexternaldoflist (%p): ",col_sexternaldoflist);
		if(col_sexternaldoflist)for(i=0;i<col_ssize;i++)printf("%i ",col_sexternaldoflist[i]); printf("\n");
	}
}
/*}}}*/
