/*!\file DofIndexing.c
 * \brief: implementation of the DofIndexing object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "./objects.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../Container/Container.h"
#include "../include/include.h"

/*DofIndexing constructors and destructor*/
/*FUNCTION DofIndexing::DofIndexing(){{{1*/
DofIndexing::DofIndexing(){

	this->gsize=UNDEF;
	this->fsize=UNDEF;
	this->ssize=UNDEF;
	this->clone=0;
	this->f_set=NULL;
	this->s_set=NULL;
	this->svalues=NULL;
	this->doftype=NULL;
	this->gdoflist=NULL;
	this->fdoflist=NULL;
	this->sdoflist=NULL;

}
/*}}}*/
/*FUNCTION DofIndexing::DofIndexing(int gsize){{{1*/
DofIndexing::DofIndexing(int in_gsize){
	this->Init(in_gsize,NULL);
}
/*}}}*/
/*FUNCTION DofIndexing::DofIndexing(DofIndexing* in)  -> copy{{{1*/
DofIndexing::DofIndexing(DofIndexing* in){ //copy constructor

	int i;
	this->gsize=in->gsize;
	this->fsize=in->fsize;
	this->ssize=in->ssize;
	
	this->clone=in->clone;

	if(this->gsize){
		this->f_set=(bool*)xmalloc(this->gsize*sizeof(bool));
		this->s_set=(bool*)xmalloc(this->gsize*sizeof(bool));
		this->svalues=(double*)xmalloc(this->gsize*sizeof(int));
		if(in->doftype)this->doftype=(int*)xmalloc(this->gsize*sizeof(int)); 
		this->gdoflist=(int*)xmalloc(this->gsize*sizeof(int)); 
	}
	else{
		this->f_set=NULL;
		this->s_set=NULL;
		this->svalues=NULL;
		this->doftype=NULL;
		this->gdoflist=NULL;
	}
	if(this->fsize)this->fdoflist=(int*)xmalloc(this->fsize*sizeof(int)); else this->fdoflist=NULL;
	if(this->ssize)this->sdoflist=(int*)xmalloc(this->ssize*sizeof(int)); else this->sdoflist=NULL;

	if(this->gsize){
		memcpy(this->f_set,in->f_set,this->gsize*sizeof(bool));
		memcpy(this->s_set,in->s_set,this->gsize*sizeof(bool));
		memcpy(this->svalues,in->svalues,this->gsize*sizeof(double));
		if(this->doftype)memcpy(this->doftype,in->doftype,this->gsize*sizeof(int));
		memcpy(this->gdoflist,in->gdoflist,this->gsize*sizeof(int));
	}
	if(this->fsize)memcpy(this->fdoflist,in->fdoflist,this->fsize*sizeof(int));
	if(this->ssize)memcpy(this->sdoflist,in->sdoflist,this->ssize*sizeof(int));

}
/*}}}*/
/*FUNCTION DofIndexing::~DofIndexing() {{{1*/
DofIndexing::~DofIndexing(){ //destructor

	xfree((void**)&f_set); 
	xfree((void**)&s_set); 
	xfree((void**)&svalues);
	xfree((void**)&doftype); 
	xfree((void**)&gdoflist);
	xfree((void**)&fdoflist);
	xfree((void**)&sdoflist);

}
/*}}}*/
/*FUNCTION DofIndexing::Init{{{1*/
void DofIndexing::Init(int in_gsize,int* in_doftype){

	int i;
	this->gsize=in_gsize;
	
	this->clone=0;

	/*allocate: */
	if(this->gsize){
		this->f_set=(bool*)xmalloc(this->gsize*sizeof(bool));
		this->s_set=(bool*)xmalloc(this->gsize*sizeof(bool));
		this->svalues=(double*)xmalloc(this->gsize*sizeof(double));
		if(in_doftype)this->doftype=(int*)xmalloc(this->gsize*sizeof(int));
		this->gdoflist=(int*)xmalloc(this->gsize*sizeof(int));
	}

	for (i=0;i<this->gsize;i++){
		/*assume dof is free, no constraints, no rigid body constraint: */
		this->f_set[i]=true;
		this->s_set[i]=false;
		if(this->doftype)this->doftype[i]=in_doftype[i];
		this->svalues[i]=0; //0 constraint is the default value
	}
}
/*}}}*/
/*FUNCTION DofIndexing::InitSet{{{1*/
void DofIndexing::InitSet(int setenum){

	int i;
	int size=0;

	/*go through sets, and figure out how many dofs belong to this set, except for g-set, 
	 * which has already been initialized: */
	if(setenum==FsetEnum){
		size=0;
		for(i=0;i<this->gsize;i++) if(f_set[i])size++;
		this->fsize=size;
		if(this->fsize)this->fdoflist=(int*)xmalloc(size*sizeof(int));
		else this->fdoflist=NULL;
	}
	else if(setenum==SsetEnum){
		size=0;
		for(i=0;i<this->gsize;i++) if(s_set[i])size++;
		this->ssize=size;
		if(this->ssize)this->sdoflist=(int*)xmalloc(size*sizeof(int));
		else this->sdoflist=NULL;
	}
	else ISSMERROR("%s%s%s"," set of enum type ",EnumToString(setenum)," not supported yet!");


}
/*}}}*/

/*Some of the Object functionality: */
/*FUNCTION DofIndexing::Echo{{{1*/
void DofIndexing::Echo(void){

	int i;

	printf("DofIndexing:\n");
	printf("   gsize: %i\n",gsize);
	printf("   clone: %i\n",clone);
}
/*}}}*/
/*FUNCTION DofIndexing::DeepEcho{{{1*/
void DofIndexing::DeepEcho(void){

	int i;

	printf("DofIndexing:\n");
	printf("   gsize: %i\n",gsize);
	printf("   fsize: %i\n",fsize);
	printf("   ssize: %i\n",ssize);
	printf("   clone: %i\n",clone);
	
	printf("   set membership: f,s sets \n");
	for(i=0;i<gsize;i++){
		printf("      dof %i: %s %s\n",i,f_set[i]?"true":"false",s_set[i]?"true":"false");
	}

	printf("   svalues (%i): |",this->ssize);
	for(i=0;i<this->gsize;i++){
		if(this->s_set[i])printf(" %g |",svalues[i]);
	}
	printf("\n");

	if(doftype){
		printf("   doftype: |");
		for(i=0;i<gsize;i++){
			printf(" %i |",doftype[i]);
		}
		printf("\n");
	}
	else printf("   doftype: NULL\n");

	printf("   g_doflist (%i): |",this->gsize);
	for(i=0;i<this->gsize;i++){
		printf(" %i |",gdoflist[i]);
	}
	printf("\n");

	printf("   f_doflist (%i): |",this->fsize);
	for(i=0;i<this->fsize;i++){
		printf(" %i |",fdoflist[i]);
	}
	printf("\n");

	printf("   s_doflist (%i): |",this->ssize);
	for(i=0;i<this->ssize;i++){
		printf(" %i |",sdoflist[i]);
	}
	printf("\n");
}		
/*}}}*/
/*FUNCTION DofIndexing::Demarshall{{{1*/
void  DofIndexing::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type;
	int   flagdoftype;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of object since DofIndexing is not directly called by DataSet: */
	memcpy(&enum_type,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);

	/*easy part: */
	memcpy(&gsize,marshalled_dataset,sizeof(gsize));marshalled_dataset+=sizeof(gsize);
	memcpy(&fsize,marshalled_dataset,sizeof(fsize));marshalled_dataset+=sizeof(fsize);
	memcpy(&ssize,marshalled_dataset,sizeof(ssize));marshalled_dataset+=sizeof(ssize);
	memcpy(&flagdoftype,marshalled_dataset,sizeof(flagdoftype));marshalled_dataset+=sizeof(flagdoftype);
	memcpy(&clone,marshalled_dataset,sizeof(clone));marshalled_dataset+=sizeof(clone);
	
	/*Allocate: */
	if(this->gsize){
		this->f_set=(bool*)xmalloc(this->gsize*sizeof(bool));
		this->s_set=(bool*)xmalloc(this->gsize*sizeof(bool));
		this->svalues=(double*)xmalloc(this->ssize*sizeof(double));
		if(flagdoftype)this->doftype=(int*)xmalloc(this->gsize*sizeof(int));
		else           this->doftype=NULL;
		this->gdoflist=(int*)xmalloc(this->gsize*sizeof(int));
	}
	else{
		this->f_set=NULL;
		this->s_set=NULL;
		this->svalues=NULL;
		this->doftype=NULL;
		this->gdoflist=NULL;
	}
	if(this->fsize)this->fdoflist=(int*)xmalloc(this->fsize*sizeof(int));
	else           this->fdoflist=NULL;
	if(this->ssize)this->sdoflist=(int*)xmalloc(this->ssize*sizeof(int));
	else           this->sdoflist=NULL;

	/*Copy arrays: */
	if(this->gsize){
		memcpy(f_set,marshalled_dataset,gsize*sizeof(bool));marshalled_dataset+=gsize*sizeof(bool);
		memcpy(s_set,marshalled_dataset,gsize*sizeof(bool));marshalled_dataset+=gsize*sizeof(bool);
		memcpy(this->svalues,marshalled_dataset,this->gsize*sizeof(double));marshalled_dataset+=this->gsize*sizeof(double);
		if(flagdoftype){ memcpy(doftype,marshalled_dataset,gsize*sizeof(int));marshalled_dataset+=gsize*sizeof(int); }
		memcpy(this->gdoflist,marshalled_dataset,this->gsize*sizeof(int));marshalled_dataset+=this->gsize*sizeof(int);
	}
	
	if(this->fsize){ memcpy(this->fdoflist,marshalled_dataset,this->fsize*sizeof(int));marshalled_dataset+=this->fsize*sizeof(int); }
	if(this->ssize){ memcpy(this->sdoflist,marshalled_dataset,this->ssize*sizeof(int));marshalled_dataset+=this->ssize*sizeof(int); }

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION DofIndexing::Marshall{{{1*/
void  DofIndexing::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	bool  flagdoftype; //to indicate if there are some doftype or if NULL

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*preliminary: */
	if(this->doftype)flagdoftype=true;
	else             flagdoftype=false;

	/*get enum type of DofIndexing: */
	enum_type=DofIndexingEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall DofIndexing data: */
	memcpy(marshalled_dataset,&gsize,sizeof(gsize));marshalled_dataset+=sizeof(gsize);
	memcpy(marshalled_dataset,&fsize,sizeof(fsize));marshalled_dataset+=sizeof(fsize);
	memcpy(marshalled_dataset,&ssize,sizeof(ssize));marshalled_dataset+=sizeof(ssize);
	memcpy(marshalled_dataset,&flagdoftype,sizeof(flagdoftype));marshalled_dataset+=sizeof(flagdoftype);
	memcpy(marshalled_dataset,&clone,sizeof(clone));marshalled_dataset+=sizeof(clone);
	
	
	if(this->gsize){
		memcpy(marshalled_dataset,f_set,gsize*sizeof(bool));marshalled_dataset+=gsize*sizeof(bool);
		memcpy(marshalled_dataset,s_set,gsize*sizeof(bool));marshalled_dataset+=gsize*sizeof(bool);
		memcpy(marshalled_dataset,svalues,gsize*sizeof(double)); marshalled_dataset+=gsize*sizeof(double);
		if(flagdoftype){ memcpy(marshalled_dataset,doftype,gsize*sizeof(int)); marshalled_dataset+=gsize*sizeof(int); }
		memcpy(marshalled_dataset,gdoflist,gsize*sizeof(int)); marshalled_dataset+=gsize*sizeof(int);
	}
	if(this->fsize){ memcpy(marshalled_dataset,fdoflist,fsize*sizeof(int)); marshalled_dataset+=fsize*sizeof(int);}
	if(this->ssize){ memcpy(marshalled_dataset,sdoflist,ssize*sizeof(int)); marshalled_dataset+=ssize*sizeof(int);}

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION DofIndexing::MarshallSize{{{1*/
int   DofIndexing::MarshallSize(){

	int size=0;

	size+=4*sizeof(int)+sizeof(bool);
	if(this->gsize){
		size+= 2*this->gsize*sizeof(bool)+
			   this->gsize*sizeof(double)+
			   this->gsize*sizeof(int);
		if(this->doftype)size+=this->gsize*sizeof(int);
	}
	if(this->fsize)size+=this->fsize*sizeof(int);
	if(this->ssize)size+=this->ssize*sizeof(int);

	size+=sizeof(int); //sizeof(int) for enum type

	return size;
}
/*}}}*/
