/*!\file Icefront.c
 * \brief: implementation of the Icefront object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
/*}}}*/	

/*Load macros*/
#define NUMVERTICESSEG 2
#define NUMVERTICESQUA 4
#define NDOF1 1
#define NDOF2 2
#define NDOF3 3
#define NDOF4 4

/*Icefront constructors and destructor*/
/*FUNCTION Icefront::Icefront() {{{1*/
Icefront::Icefront(){

	this->inputs=NULL;
	this->parameters=NULL;

	this->hnodes=NULL;
	this->nodes= NULL;
	this->helement=NULL;
	this->element= NULL;
	this->hmatpar=NULL;
	this->matpar= NULL;
}
/*}}}*/
/*FUNCTION Icefront::Icefront(int id, int i, IoModel* iomodel,int analysis_type) {{{1*/
Icefront::Icefront(int icefront_id,int i, IoModel* iomodel,int in_icefront_type, int in_analysis_type){

	int segment_width;
	int element;
	int num_nodes;

	/*icefront constructor data: */
	int  icefront_eid;
	int  icefront_mparid;
	int  icefront_node_ids[NUMVERTICESQUA]; //initialize with largest size
	int  icefront_fill;

	/*First, retrieve element index and element type: */
	if (iomodel->dim==2){
		segment_width=4;
	}
	else{
		segment_width=6;
	}
	element=(int)(*(iomodel->pressureload+segment_width*i+segment_width-2)-1); //element is in the penultimate column (grid1 grid2 ... elem fill)

	/*Build ids for hook constructors: */
	icefront_eid=(int) *(iomodel->pressureload+segment_width*i+segment_width-2); //matlab indexing
	icefront_mparid=iomodel->numberofelements+1; //matlab indexing

	if (in_icefront_type==MacAyeal2dIceFrontEnum || in_icefront_type==MacAyeal3dIceFrontEnum){
		icefront_node_ids[0]=iomodel->nodecounter+(int)*(iomodel->pressureload+segment_width*i+0);
		icefront_node_ids[1]=iomodel->nodecounter+(int)*(iomodel->pressureload+segment_width*i+1);
	}
	else if (in_icefront_type==PattynIceFrontEnum || in_icefront_type==StokesIceFrontEnum){
		icefront_node_ids[0]=iomodel->nodecounter+(int)*(iomodel->pressureload+segment_width*i+0);
		icefront_node_ids[1]=iomodel->nodecounter+(int)*(iomodel->pressureload+segment_width*i+1);
		icefront_node_ids[2]=iomodel->nodecounter+(int)*(iomodel->pressureload+segment_width*i+2);
		icefront_node_ids[3]=iomodel->nodecounter+(int)*(iomodel->pressureload+segment_width*i+3);
	}
	else ISSMERROR("in_icefront_type %s not supported yet!",EnumToString(in_icefront_type));

	if (in_icefront_type==PattynIceFrontEnum || in_icefront_type==StokesIceFrontEnum) num_nodes=4;
	else num_nodes=2;
	icefront_fill=(int)*(iomodel->pressureload+segment_width*i+segment_width-1); 
	
	/*Ok, we have everything to build the object: */
	this->id=icefront_id;
	this->analysis_type=in_analysis_type;

	/*Hooks: */
	this->hnodes=new Hook(icefront_node_ids,num_nodes);
	this->helement=new Hook(&icefront_eid,1);
	this->hmatpar=new Hook(&icefront_mparid,1);

	//intialize  and add as many inputs per element as requested: 
	this->inputs=new Inputs();
	this->inputs->AddInput(new IntInput(FillEnum,icefront_fill));
	this->inputs->AddInput(new IntInput(TypeEnum,in_icefront_type));
	
	//parameters and hooked fields: we still can't point to them, they may not even exist. Configure will handle this.
	this->parameters=NULL;
	this->nodes= NULL;
	this->element= NULL;
	this->matpar= NULL;
}


/*}}}*/
/*FUNCTION Icefront::~Icefront() {{{1*/
Icefront::~Icefront(){
	delete inputs;
	this->parameters=NULL;
	delete hnodes;
	delete helement;
	delete hmatpar;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION Icefront::Echo {{{1*/
void Icefront::Echo(void){
	printf("Icefront:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumToString(analysis_type));
	hnodes->Echo();
	helement->Echo();
	hmatpar->Echo();
	printf("   parameters: %p\n",parameters);
	if(parameters)parameters->Echo();
	printf("   inputs: %p\n",inputs);
	if(inputs)inputs->Echo();
}
/*}}}*/
/*FUNCTION Icefront::DeepEcho{{{1*/
void Icefront::DeepEcho(void){

	printf("Icefront:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumToString(analysis_type));
	hnodes->DeepEcho();
	helement->DeepEcho();
	hmatpar->DeepEcho();
	printf("   parameters: %p\n",parameters);
	if(parameters)parameters->DeepEcho();
	printf("   inputs: %p\n",inputs);
	if(inputs)inputs->DeepEcho();
}
/*}}}*/
/*FUNCTION Icefront::Id {{{1*/
int    Icefront::Id(void){ return id; }
/*}}}*/
/*FUNCTION Icefront::MyRank {{{1*/
int    Icefront::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Icefront::Marshall {{{1*/
void  Icefront::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Icefront: */
	enum_type=IcefrontEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Icefront data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&analysis_type,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);

	/*Marshall hooks: */
	hnodes->Marshall(&marshalled_dataset);
	helement->Marshall(&marshalled_dataset);
	hmatpar->Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);

	*pmarshalled_dataset=marshalled_dataset;
}
/*}}}*/
/*FUNCTION Icefront::MarshallSize {{{1*/
int   Icefront::MarshallSize(){
	
	return sizeof(id)
		+sizeof(analysis_type)
		+hnodes->MarshallSize()
		+helement->MarshallSize()
		+hmatpar->MarshallSize()
		+inputs->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Icefront::Demarshall {{{1*/
void  Icefront::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&analysis_type,marshalled_dataset,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);

	/*demarshall hooks: */
	hnodes=new Hook(); hnodes->Demarshall(&marshalled_dataset);
	helement=new Hook(); helement->Demarshall(&marshalled_dataset);
	hmatpar=new Hook(); hmatpar->Demarshall(&marshalled_dataset);
	
	/*pointers are garbabe, until configuration is carried out: */
	nodes=NULL;
	element=NULL;
	matpar=NULL;

	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Icefront::Enum {{{1*/
int Icefront::Enum(void){

	return IcefrontEnum;

}
/*}}}*/
/*FUNCTION Icefront::copy {{{1*/
Object* Icefront::copy() {
	
	Icefront* icefront=NULL;

	icefront=new Icefront();

	/*copy fields: */
	icefront->id=this->id;
	icefront->analysis_type=this->analysis_type;
	if(this->inputs){
		icefront->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		icefront->inputs=new Inputs();
	}
	/*point parameters: */
	icefront->parameters=this->parameters;

	/*now deal with hooks and objects: */
	icefront->hnodes=(Hook*)this->hnodes->copy();
	icefront->helement=(Hook*)this->helement->copy();
	icefront->hmatpar=(Hook*)this->hmatpar->copy();

	/*corresponding fields*/
	icefront->nodes  =(Node**)icefront->hnodes->deliverp();
	icefront->element=(Element*)icefront->helement->delivers();
	icefront->matpar =(Matpar*)icefront->hmatpar->delivers();

	return icefront;

}
/*}}}*/

/*Load virtual functions definitions:*/
/*FUNCTION Icefront::Configure {{{1*/
void  Icefront::Configure(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes->configure(nodesin);
	helement->configure(elementsin);
	hmatpar->configure(materialsin);

	/*Initialize hooked fields*/
	this->nodes  =(Node**)hnodes->deliverp();
	this->element=(Element*)helement->delivers();
	this->matpar =(Matpar*)hmatpar->delivers();

	/*point parameters to real dataset: */
	this->parameters=parametersin;
}
/*}}}*/
/*FUNCTION Icefront::SetCurrentConfiguration {{{1*/
void  Icefront::SetCurrentConfiguration(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){
}
/*}}}*/
/*FUNCTION Icefront::CreateKMatrix {{{1*/
void  Icefront::CreateKMatrix(Mat Kgg){

	/*No stiffness loads applied, do nothing: */
	return;

}
/*}}}*/
/*FUNCTION Icefront::CreatePVector {{{1*/
void  Icefront::CreatePVector(Vec pg){

	/*Checks in debugging mode*/
	ISSMASSERT(nodes);
	ISSMASSERT(element);
	ISSMASSERT(matpar);

	/*Retrieve parameters: */
	int analysis_type;
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Just branch to the correct element icefront vector generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		int type;
		inputs->GetParameterValue(&type,TypeEnum);
		if (type==MacAyeal2dIceFrontEnum){
			CreatePVectorDiagnosticMacAyeal2d( pg);
		}
		else if (type==MacAyeal3dIceFrontEnum){
			CreatePVectorDiagnosticMacAyeal3d( pg);
		}
		else if (type==PattynIceFrontEnum){
			CreatePVectorDiagnosticPattyn( pg);
		}
		else if (type==StokesIceFrontEnum){
			CreatePVectorDiagnosticStokes( pg);
		}
		else ISSMERROR("Icefront type %s not supported yet",EnumToString(type));
	}
	else if (analysis_type==AdjointHorizAnalysisEnum){
		return;
	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}
}
/*}}}*/
/*FUNCTION Icefront::PenaltyCreateKMatrix {{{1*/
void  Icefront::PenaltyCreateKMatrix(Mat Kgg,double kmax){
	/*do nothing: */
}
/*}}}*/
/*FUNCTION Icefront::PenaltyCreatePVector{{{1*/
void  Icefront::PenaltyCreatePVector(Vec pg,double kmax){
	/*do nothing: */
}
/*}}}*/
/*FUNCTION Icefront::InAnalysis{{{1*/
bool Icefront::InAnalysis(int in_analysis_type){
	if (in_analysis_type==this->analysis_type)return true;
	else return false;
}
/*}}}*/

/*Update virtual functions definitions:*/
/*FUNCTION Icefront::InputUpdateFromVector(double* vector, int name, int type) {{{1*/
void  Icefront::InputUpdateFromVector(double* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Icefront::InputUpdateFromVector(int* vector, int name, int type) {{{1*/
void  Icefront::InputUpdateFromVector(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Icefront::InputUpdateFromVector(bool* vector, int name, int type) {{{1*/
void  Icefront::InputUpdateFromVector(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Icefront::InputUpdateFromVectorDakota(double* vector, int name, int type) {{{1*/
void  Icefront::InputUpdateFromVectorDakota(double* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Icefront::InputUpdateFromVectorDakota(int* vector, int name, int type) {{{1*/
void  Icefront::InputUpdateFromVectorDakota(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Icefront::InputUpdateFromVectorDakota(bool* vector, int name, int type) {{{1*/
void  Icefront::InputUpdateFromVectorDakota(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Icefront::InputUpdateFromConstant(double constant, int name) {{{1*/
void  Icefront::InputUpdateFromConstant(double constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Icefront::InputUpdateFromConstant(int constant, int name) {{{1*/
void  Icefront::InputUpdateFromConstant(int constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Icefront::InputUpdateFromConstant(bool constant, int name) {{{1*/
void  Icefront::InputUpdateFromConstant(bool constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Icefront::InputUpdateFromSolution{{{1*/
void  Icefront::InputUpdateFromSolution(double* solution){
	/*Nothing updated yet*/
}
/*}}}*/

/*Icefront numerics: */
/*FUNCTION Icefront::CreatePVectorDiagnosticMacAyeal2d{{{1*/
void Icefront::CreatePVectorDiagnosticMacAyeal2d( Vec pg){

	/*Constants*/
	const int numnodes= NUMVERTICESSEG;
	const int numdofs = numnodes *NDOF2;

	/*Intermediary*/
	int        ig,index1,index2,fill;
	double     Jdet;
	double     thickness,bed,pressure,ice_pressure,rho_water,rho_ice,gravity;
	double     water_pressure,air_pressure,surface_under_water,base_under_water;
	int       *doflist = NULL;
	double     xyz_list[numnodes][3];
	double     pe_g[numdofs] = {0.0};
	double     normal[2];
	double     L[2];
	GaussTria *gauss;

	Tria* tria=((Tria*)element);

	/* Get dof list and node coordinates: */
	GetDofList(&doflist,MacAyealApproximationEnum);
	GetVerticesCoordinates(&xyz_list[0][0],nodes,numnodes);

	/*Recover inputs and parameters */
	Input* thickness_input=tria->inputs->GetInput(ThicknessEnum); ISSMASSERT(thickness_input);
	Input* bed_input      =tria->inputs->GetInput(BedEnum);       ISSMASSERT(bed_input);
	inputs->GetParameterValue(&fill,FillEnum);

	rho_water=matpar->GetRhoWater();
	rho_ice  =matpar->GetRhoIce();
	gravity  =matpar->GetG();

	GetNormal(&normal[0],xyz_list);

	index1=tria->GetNodeIndex(nodes[0]);
	index2=tria->GetNodeIndex(nodes[1]);
	gauss=new GaussTria(index1,index2,3);

	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		thickness_input->GetParameterValue(&thickness,gauss);
		bed_input->GetParameterValue(&bed,gauss);

		switch(fill){
			case WaterEnum:
				surface_under_water=min(0,thickness+bed); // 0 if the top of the glacier is above water level
				base_under_water=min(0,bed);              // 0 if the bottom of the glacier is above water level
				water_pressure=1.0/2.0*gravity*rho_water*(pow(surface_under_water,2) - pow(base_under_water,2));
				break;
			case AirEnum:
				water_pressure=0;
				break;
			default:
				ISSMERROR("fill type %s not supported yet",EnumToString(fill));
		}
		ice_pressure=1.0/2.0*gravity*rho_ice*pow(thickness,2);
		air_pressure=0;
		pressure = ice_pressure + water_pressure + air_pressure;

		tria->GetSegmentJacobianDeterminant(&Jdet,&xyz_list[0][0],gauss);
		tria->GetSegmentNodalFunctions(&L[0],gauss,index1,index2);

		for (int i=0;i<numnodes;i++){
			pe_g[2*i+0]+= pressure*Jdet*gauss->weight*normal[0]*L[i];
			pe_g[2*i+1]+= pressure*Jdet*gauss->weight*normal[1]*L[i];
		}
	}
	
	VecSetValues(pg,numdofs,doflist,pe_g,ADD_VALUES);

	/*Free ressources:*/
	delete gauss;
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Icefront::CreatePVectorDiagnosticMacAyeal3d{{{1*/
void Icefront::CreatePVectorDiagnosticMacAyeal3d( Vec pg){

	Icefront* icefront=NULL;
	Penta*    penta=NULL;
	Tria*     tria=NULL;
	bool      onbed;

	/*Cast element onto Penta*/
	penta   =(Penta*)this->element;

	/*Return if not on bed*/
	penta->inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	if(!onbed) return;

	/*Spawn Tria and call MacAyeal2d*/
	tria    =(Tria*)penta->SpawnTria(0,1,2);
	icefront=(Icefront*)this->copy();
	icefront->element=tria;
	icefront->inputs->AddInput(new IntInput(TypeEnum,MacAyeal2dIceFrontEnum));
	icefront->CreatePVectorDiagnosticMacAyeal2d(pg);

	/*clean-up*/
	delete tria->matice;
	delete tria;
	delete icefront;
}
/*}}}*/
/*FUNCTION Icefront::CreatePVectorDiagnosticPattyn{{{1*/
void Icefront::CreatePVectorDiagnosticPattyn( Vec pg){

	int i,j;

	const int numnodes=NUMVERTICESQUA;
	const int numdofs=numnodes*NDOF2;
	int*      doflist=NULL;

	double xyz_list[numnodes][3];
	double xyz_list_quad[numnodes+1][3]; //center node

	double thickness_list[6]; //from penta element
	double thickness_list_quad[6]; //selection of 4 thickness from penta elements

	double bed_list[6]; //from penta element
	double bed_list_quad[6]; //selection of 4 beds from penta elements

	/*Objects: */
	double  pe_g[numdofs]={0.0};
	int    element_type;
		
	/*quad grids: */
	int grid1,grid2,grid3,grid4;

	/*normals: */
	double normal1[3];
	double normal2[3];
	double normal3[3];
	double normal4[3];
	double normal_norm;
	double v15[3];
	double v25[3];
	double v35[3];
	double v45[3];


	element_type=element->Enum();

	/*check icefront is associated to a pentaelem: */
	if(element_type!=PentaEnum){
		ISSMERROR(" quad icefront is associated to a TriaElem!");
	}

	/* Get dof list and node coordinates: */
	GetDofList(&doflist,PattynApproximationEnum);
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numnodes);
	
	//Identify which grids are comprised in the quad: 
	grid1=element->GetNodeIndex(nodes[0]);
	grid2=element->GetNodeIndex(nodes[1]);
	grid3=element->GetNodeIndex(nodes[2]);
	grid4=element->GetNodeIndex(nodes[3]);

	/*Build new xyz, bed and thickness lists for grids 1 to 4: */
	for(i=0;i<4;i++){
		for(j=0;j<3;j++){
			xyz_list_quad[i][j]=xyz_list[i][j];
		}
	}

	element->GetParameterListOnVertices(&thickness_list[0],ThicknessEnum);
	element->GetParameterListOnVertices(&bed_list[0],BedEnum);

	thickness_list_quad[0]=thickness_list[grid1];
	thickness_list_quad[1]=thickness_list[grid2];
	thickness_list_quad[2]=thickness_list[grid3];
	thickness_list_quad[3]=thickness_list[grid4];

	bed_list_quad[0]=bed_list[grid1];
	bed_list_quad[1]=bed_list[grid2];
	bed_list_quad[2]=bed_list[grid3];
	bed_list_quad[3]=bed_list[grid4];

	//Create a new grid in the midle of the quad and add it at the end of the list
	xyz_list_quad[4][0] = (xyz_list_quad[0][0]+xyz_list_quad[1][0]+xyz_list_quad[2][0]+xyz_list_quad[3][0])/4.0;
	xyz_list_quad[4][1] = (xyz_list_quad[0][1]+xyz_list_quad[1][1]+xyz_list_quad[2][1]+xyz_list_quad[3][1])/4.0;
	xyz_list_quad[4][2] = (xyz_list_quad[0][2]+xyz_list_quad[1][2]+xyz_list_quad[2][2]+xyz_list_quad[3][2])/4.0;

	//Compute four normals (the quad is divided into four triangles)
	v15[0]=xyz_list_quad[0][0]-xyz_list_quad[4][0];
	v15[1]=xyz_list_quad[0][1]-xyz_list_quad[4][1];
	v15[2]=xyz_list_quad[0][2]-xyz_list_quad[4][2];
	
	v25[0]=xyz_list_quad[1][0]-xyz_list_quad[4][0];
	v25[1]=xyz_list_quad[1][1]-xyz_list_quad[4][1];
	v25[2]=xyz_list_quad[1][2]-xyz_list_quad[4][2];
	
	v35[0]=xyz_list_quad[2][0]-xyz_list_quad[4][0];
	v35[1]=xyz_list_quad[2][1]-xyz_list_quad[4][1];
	v35[2]=xyz_list_quad[2][2]-xyz_list_quad[4][2];
	
	v45[0]=xyz_list_quad[3][0]-xyz_list_quad[4][0];
	v45[1]=xyz_list_quad[3][1]-xyz_list_quad[4][1];
	v45[2]=xyz_list_quad[3][2]-xyz_list_quad[4][2];

	cross(normal1,v15,v25); 
	cross(normal2,v25,v35);
	cross(normal3,v35,v45);
	cross(normal4,v45,v15);

	normal_norm=norm(normal1);normal1[0]=normal1[0]/normal_norm;normal1[1]=normal1[1]/normal_norm;normal1[2]=normal1[2]/normal_norm;
	normal_norm=norm(normal2);normal2[0]=normal2[0]/normal_norm;normal2[1]=normal2[1]/normal_norm;normal2[2]=normal2[2]/normal_norm;
	normal_norm=norm(normal3);normal3[0]=normal3[0]/normal_norm;normal3[1]=normal3[1]/normal_norm;normal3[2]=normal3[2]/normal_norm;
	normal_norm=norm(normal4);normal4[0]=normal4[0]/normal_norm;normal4[1]=normal4[1]/normal_norm;normal4[2]=normal4[2]/normal_norm;

	//Compute load contribution for this quad:
	QuadPressureLoad(pe_g,matpar->GetRhoWater(),matpar->GetRhoIce(),matpar->GetG(),thickness_list_quad,bed_list_quad,normal1,normal2,normal3,normal4,&xyz_list_quad[0][0]);

	/*Plug pe_g into vector: */
	VecSetValues(pg,numdofs,doflist,pe_g,ADD_VALUES);

	/*Free ressources:*/
	xfree((void**)&doflist);

}
/*}}}*/
/*FUNCTION Icefront::CreatePVectorDiagnosticStokes {{{1*/
void Icefront::CreatePVectorDiagnosticStokes( Vec pg){

	int i,j;

	const int numnodes=4;
	const int numdofs=numnodes*NDOF4;
	int*      doflist=NULL;

	double xyz_list[numnodes][3];
	double xyz_list_quad[numnodes+1][3]; //center node

	double thickness_list[6]; //from penta element
	double thickness_list_quad[6]; //selection of 4 thickness from penta elements

	double bed_list[6]; //from penta element
	double bed_list_quad[6]; //selection of 4 beds from penta elements

	/*Objects: */
	double  pe_g[numdofs]={0.0};
	Penta*   penta=NULL;

	/*quad grids: */
	int grid1,grid2,grid3,grid4;

	/*normals: */
	double normal1[3];
	double normal2[3];
	double normal3[3];
	double normal4[3];
	double normal_norm;
	double v15[3];
	double v25[3];
	double v35[3];
	double v45[3];
	int approximation;

	/*check icefront is associated to a pentaelem: */
	if(element->Enum()!=PentaEnum) ISSMERROR("Only Penta supported yet");
	penta=(Penta*)element;

	/*If not Stokes, return*/
	penta->inputs->GetParameterValue(&approximation,ApproximationEnum);
	if(approximation!=StokesApproximationEnum) return;

	/* Get dof list and node coordinates: */
	GetDofList(&doflist,StokesApproximationEnum);
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numnodes);
	
	//Identify which grids are comprised in the quad: 
	grid1=element->GetNodeIndex(nodes[0]);
	grid2=element->GetNodeIndex(nodes[1]);
	grid3=element->GetNodeIndex(nodes[2]);
	grid4=element->GetNodeIndex(nodes[3]);

	/*Build new xyz, bed and thickness lists for grids 1 to 4: */
	for(i=0;i<4;i++){
		for(j=0;j<3;j++){
			xyz_list_quad[i][j]=xyz_list[i][j];
		}
	}

	element->GetParameterListOnVertices(&thickness_list[0],ThicknessEnum);
	element->GetParameterListOnVertices(&bed_list[0],BedEnum);

	thickness_list_quad[0]=thickness_list[grid1];
	thickness_list_quad[1]=thickness_list[grid2];
	thickness_list_quad[2]=thickness_list[grid3];
	thickness_list_quad[3]=thickness_list[grid4];

	bed_list_quad[0]=bed_list[grid1];
	bed_list_quad[1]=bed_list[grid2];
	bed_list_quad[2]=bed_list[grid3];
	bed_list_quad[3]=bed_list[grid4];

	//Create a new grid in the midle of the quad and add it at the end of the list
	xyz_list_quad[4][0] = (xyz_list_quad[0][0]+xyz_list_quad[1][0]+xyz_list_quad[2][0]+xyz_list_quad[3][0])/4.0;
	xyz_list_quad[4][1] = (xyz_list_quad[0][1]+xyz_list_quad[1][1]+xyz_list_quad[2][1]+xyz_list_quad[3][1])/4.0;
	xyz_list_quad[4][2] = (xyz_list_quad[0][2]+xyz_list_quad[1][2]+xyz_list_quad[2][2]+xyz_list_quad[3][2])/4.0;

	//Compute four normals (the quad is divided into four triangles)
	v15[0]=xyz_list_quad[0][0]-xyz_list_quad[4][0];
	v15[1]=xyz_list_quad[0][1]-xyz_list_quad[4][1];
	v15[2]=xyz_list_quad[0][2]-xyz_list_quad[4][2];
	
	v25[0]=xyz_list_quad[1][0]-xyz_list_quad[4][0];
	v25[1]=xyz_list_quad[1][1]-xyz_list_quad[4][1];
	v25[2]=xyz_list_quad[1][2]-xyz_list_quad[4][2];
	
	v35[0]=xyz_list_quad[2][0]-xyz_list_quad[4][0];
	v35[1]=xyz_list_quad[2][1]-xyz_list_quad[4][1];
	v35[2]=xyz_list_quad[2][2]-xyz_list_quad[4][2];
	
	v45[0]=xyz_list_quad[3][0]-xyz_list_quad[4][0];
	v45[1]=xyz_list_quad[3][1]-xyz_list_quad[4][1];
	v45[2]=xyz_list_quad[3][2]-xyz_list_quad[4][2];

	cross(normal1,v15,v25); 
	cross(normal2,v25,v35);
	cross(normal3,v35,v45);
	cross(normal4,v45,v15);

	normal_norm=norm(normal1);normal1[0]=normal1[0]/normal_norm;normal1[1]=normal1[1]/normal_norm;normal1[2]=normal1[2]/normal_norm;
	normal_norm=norm(normal2);normal2[0]=normal2[0]/normal_norm;normal2[1]=normal2[1]/normal_norm;normal2[2]=normal2[2]/normal_norm;
	normal_norm=norm(normal3);normal3[0]=normal3[0]/normal_norm;normal3[1]=normal3[1]/normal_norm;normal3[2]=normal3[2]/normal_norm;
	normal_norm=norm(normal4);normal4[0]=normal4[0]/normal_norm;normal4[1]=normal4[1]/normal_norm;normal4[2]=normal4[2]/normal_norm;


	//Compute load contribution for this quad:
	QuadPressureLoadStokes(pe_g,matpar->GetRhoWater(),matpar->GetRhoIce(),matpar->GetG(),thickness_list_quad,bed_list_quad,normal1,normal2,normal3,normal4,&xyz_list_quad[0][0]);

	/*Plug pe_g into vector: */
	VecSetValues(pg,numdofs,doflist,pe_g,ADD_VALUES);

	/*Free ressources:*/
	xfree((void**)&doflist);

}
/*}}}*/
/*FUNCTION Icefront::GetDofList {{{1*/
void  Icefront::GetDofList(int** pdoflist,int approximation_enum){

	int i,j;
	int numberofdofs=0;
	int count=0;
	int type;
	int numberofnodes=2;

	/*output: */
	int* doflist=NULL;

	
	/*recover type: */
	inputs->GetParameterValue(&type,TypeEnum);

	/*Some checks for debugging*/
	ISSMASSERT(nodes);
		
	/*How many nodes? :*/
	if(type==MacAyeal2dIceFrontEnum || type==MacAyeal3dIceFrontEnum)
	 numberofnodes=2;
	else 
	 numberofnodes=4;
	
	/*Figure out size of doflist: */
	for(i=0;i<numberofnodes;i++){
		numberofdofs+=nodes[i]->GetNumberOfDofs(approximation_enum);
	}

	/*Allocate: */
	doflist=(int*)xmalloc(numberofdofs*sizeof(int));

	/*Populate: */
	count=0;
	for(i=0;i<numberofnodes;i++){
		nodes[i]->GetDofList(doflist+count,approximation_enum);
		count+=nodes[i]->GetNumberOfDofs(approximation_enum);
	}

	/*Assign output pointers:*/
	*pdoflist=doflist;
}
/*}}}*/
/*FUNCTION Icefront::QuadPressureLoad {{{1*/
void Icefront::QuadPressureLoad(double* pe_g,double rho_water,double rho_ice,double gravity, double* thickness_list, double* bed_list, 
		                              double* normal1,double* normal2,double* normal3,double* normal4,double* xyz_list){
	
	
	//The quad is divided into four triangles tria1 tria2 tria3 and tria4 as follows
	//
	//   grid 4 +-----------------+ grid 3
	//          |\2            1 /|
	//          |1\    tria3    /2|
	//          |  \           /  |
	//          |   \         /   |
	//          |    \       /    |
	//          |     \     /     |
	//          |      \ 3 /      |
	//          |tria4  \ / 3     |
	//          |      3 \grid5   |
	//          |       / \       | 
	//          |      / 3 \ tria2|
	//          |     /     \     |
	//          |    /       \    |
	//          |   /         \   |
	//          |  /   tria1   \  |
	//          |2/1           2\1|
	//   grid1  +-----------------+ grid 2
	//
	//

	int      i,j;

	double nx[4];
	double ny[4];

	/* gaussian points: */
	int     ig;
	GaussTria *gauss=NULL;

	double  surface_list[5];
	double  x[5];
	double  y[5];
	double  z[5];
	double l12,l14,l15,l23,l25,l34,l35,l45;
	double cos_theta_triangle1,cos_theta_triangle2,cos_theta_triangle3,cos_theta_triangle4;

	double xyz_tria[12][2];
	double l1l2l3_tria[3];
	double r_tria,s_tria;
	double r_quad[4];
	double s_quad[4];
	double l1l4_tria[4][4];

	double J[4];
	double z_g[4];
	double ice_pressure_tria[4];
	double air_pressure_tria;
	double water_level_above_g_tria;
	double water_pressure_tria;
	double pressure_tria[4];
	int    fill;


	/*To use tria functionalities: */
	Tria* tria=NULL;
	
	/*Recover inputs: */
	inputs->GetParameterValue(&fill,FillEnum);

	/*Initialize fake tria: */
	tria=new Tria();

	//Build the four normal vectors 
	nx[0]=normal1[0]; nx[1]=normal2[0]; nx[2]=normal3[0]; nx[3]=normal4[0];
	ny[0]=normal1[1]; ny[1]=normal2[1]; ny[2]=normal3[1]; ny[3]=normal4[1];
	
	//Recover the surface of the four nodes
	for(i=0;i<4;i++){
		surface_list[i]=thickness_list[i]+bed_list[i];
	}
	//Add surface sor the fifth point (average of the surfaces)
	surface_list[4]=(surface_list[0]+surface_list[1]+surface_list[2]+surface_list[3])/4.0;

	//Recover grid coordinates
	for(i=0;i<5;i++){
		x[i]=*(xyz_list+3*i+0);
		y[i]=*(xyz_list+3*i+1);
		z[i]=*(xyz_list+3*i+2);
	}
	
	//Build triangles in a 2D plan before using reference elements
	l12=pow((x[1]-x[0]),2)+pow((y[1]-y[0]),2)+pow((z[1]-z[0]),2);
	l14=pow((x[3]-x[0]),2)+pow((y[3]-y[0]),2)+pow((z[3]-z[0]),2);
	l15=pow((x[4]-x[0]),2)+pow((y[4]-y[0]),2)+pow((z[4]-z[0]),2);
	l23=pow((x[2]-x[1]),2)+pow((y[2]-y[1]),2)+pow((z[2]-z[1]),2);
	l25=pow((x[4]-x[1]),2)+pow((y[4]-y[1]),2)+pow((z[4]-z[1]),2);
	l34=pow((x[3]-x[2]),2)+pow((y[3]-y[2]),2)+pow((z[3]-z[2]),2);
	l35=pow((x[4]-x[2]),2)+pow((y[4]-y[2]),2)+pow((z[4]-z[2]),2);
	l45=pow((x[4]-x[3]),2)+pow((y[4]-y[3]),2)+pow((z[4]-z[3]),2);

	cos_theta_triangle1=(l15+l12-l25)/(2*sqrt(l12*l15));
	cos_theta_triangle2=(l25+l23-l35)/(2*sqrt(l23*l25));
	cos_theta_triangle3=(l35+l34-l45)/(2*sqrt(l34*l35));
	cos_theta_triangle4=(l45+l14-l15)/(2*sqrt(l14*l45));

	//First triangle : nodes 1, 2 and 5
	xyz_tria[0][0]=0;
	xyz_tria[0][1]=0;
	xyz_tria[1][0]=sqrt(l12);
	xyz_tria[1][1]=0;
	xyz_tria[2][0]=cos_theta_triangle1*sqrt(l15);
	xyz_tria[2][1]=sqrt(1-pow(cos_theta_triangle1,2))*sqrt(l15);

	//Second triangle : nodes 2, 3 and 5
	xyz_tria[3][0]=0;
	xyz_tria[3][1]=0;
	xyz_tria[4][0]=sqrt(l23);
	xyz_tria[4][1]=0;
	xyz_tria[5][0]=cos_theta_triangle2*sqrt(l25);
	xyz_tria[5][1]=sqrt(1-pow(cos_theta_triangle2,2))*sqrt(l25);
	
	//Third triangle : nodes 3, 4 and 5
	xyz_tria[6][0]=0;
	xyz_tria[6][1]=0;
	xyz_tria[7][0]=sqrt(l34);
	xyz_tria[7][1]=0;
	xyz_tria[8][0]=cos_theta_triangle3*sqrt(l35);
	xyz_tria[8][1]=sqrt(1-pow(cos_theta_triangle3,2))*sqrt(l35);

	//Fourth triangle : nodes 4, 1 and 5
	xyz_tria[9][0]=0;
	xyz_tria[9][1]=0;
	xyz_tria[10][0]=sqrt(l14);
	xyz_tria[10][1]=0;
	xyz_tria[11][0]=cos_theta_triangle4*sqrt(l45);
	xyz_tria[11][1]=sqrt(1-pow(cos_theta_triangle4,2))*sqrt(l45);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		tria->GetNodalFunctions(l1l2l3_tria,gauss);

		//Get the coordinates of gauss point for each triangle in penta/quad
		r_tria=gauss->coord2-gauss->coord1;
		s_tria=-3/sqrt((double)3)*(gauss->coord1+gauss->coord2-2/3);

		//Coordinates of gauss points in the reference triangle
		r_quad[0]=r_tria;
		s_quad[0]=1/sqrt((double)3)*s_tria-2/3;
		r_quad[1]=-1/sqrt((double)3)*s_tria+2/3;
		s_quad[1]=r_tria;
		r_quad[2]=-r_tria;
		s_quad[2]=-1/sqrt((double)3)*s_tria+2/3;
		r_quad[3]=1/sqrt((double)3)*s_tria-2/3;
		s_quad[3]=-r_tria;

		//Get the nodal function of the quad for the gauss points of each triangle
		for(i=0;i<4;i++){
			l1l4_tria[i][0]=1.0/4.0*(
					(r_quad[i]-1.0)*(s_quad[i]-1.0) 
					); 
			
			l1l4_tria[i][1]=1.0/4.0*(
					 -(r_quad[i]+1.0)*(s_quad[i]-1.0)
					); 

			l1l4_tria[i][2]=1.0/4.0*(
					(r_quad[i]+1.0)*(s_quad[i]+1.0) 
					);
			
			l1l4_tria[i][3]=1.0/4.0*(
					 -(r_quad[i]-1.0)*(s_quad[i]+1.0)
					);

		} //for(i=0;i<4;i++)
		
		
		//Compute jacobian of each triangle
		for (i=0;i<4;i++){
			double complete_list[3][3]; //a third coordinate is needed for the jacobian determinant calculation, here it is zero
			for(j=0;j<3;j++){
				complete_list[j][0]=xyz_tria[3*i+j][0];
				complete_list[j][1]=xyz_tria[3*i+j][1];
				complete_list[j][2]=0;
			}
			tria->GetJacobianDeterminant2d(&J[i],&complete_list[0][0],gauss);
		}

		//Calculation of the z coordinate for the gaussian point ig for each triangle
		z_g[0]=z[0]*l1l2l3_tria[0]+z[1]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		z_g[1]=z[1]*l1l2l3_tria[0]+z[2]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		z_g[2]=z[2]*l1l2l3_tria[0]+z[3]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		z_g[3]=z[3]*l1l2l3_tria[0]+z[0]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		
		//Loop on the triangles
		for(i=0;i<4;i++){

			//Loop on the grids of the quad
			//Calculate the ice pressure
			for(j=0;j<4;j++){
				ice_pressure_tria[j]=gravity*rho_ice*(surface_list[j]-z_g[i]);
			}
			air_pressure_tria=0;

			//Now deal with water pressure: 
			if(fill==WaterEnum){ //icefront ends in water
				water_level_above_g_tria=min(0,z_g[i]);//0 if the gaussian point is above water level
				water_pressure_tria=rho_water*gravity*water_level_above_g_tria;
			}
			else if(fill==AirEnum){
				water_pressure_tria=0;
			}
			else{
				ISSMERROR("QuadPressureLoad error message: unknow fill type for icefront boundary condition");
			}

			//Add pressure from triangle i
			//Loop on the grids of the quad
			for(j=0;j<4;j++){
				pressure_tria[j] = ice_pressure_tria[j] + water_pressure_tria + air_pressure_tria;
			}


			pe_g[0]+= J[i]*gauss->weight* pressure_tria[0]*l1l4_tria[i][0]*nx[i];
			pe_g[1]+= J[i]*gauss->weight* pressure_tria[0]*l1l4_tria[i][0]*ny[i];
			pe_g[2]+= J[i]*gauss->weight* pressure_tria[1]*l1l4_tria[i][1]*nx[i];
			pe_g[3]+= J[i]*gauss->weight* pressure_tria[1]*l1l4_tria[i][1]*ny[i];
			pe_g[4]+= J[i]*gauss->weight* pressure_tria[2]*l1l4_tria[i][2]*nx[i];
			pe_g[5]+= J[i]*gauss->weight* pressure_tria[2]*l1l4_tria[i][2]*ny[i];
			pe_g[6]+= J[i]*gauss->weight* pressure_tria[3]*l1l4_tria[i][3]*nx[i];
			pe_g[7]+= J[i]*gauss->weight* pressure_tria[3]*l1l4_tria[i][3]*ny[i];

			

		} //for(i=0;i<4;i++)
	}
	
	delete tria;
	delete gauss;
}
/*}}}*/
/*FUNCTION Icefront::QuadPressureLoadStokes {{{1*/
void Icefront::QuadPressureLoadStokes(double* pe_g,double rho_water,double rho_ice,double gravity, double* thickness_list, double* bed_list, 
		                              double* normal1,double* normal2,double* normal3,double* normal4,double* xyz_list){
	
	
	//The quad is divided into four triangles tria1 tria2 tria3 and tria4 as follows
	//
	//   grid 4 +-----------------+ grid 3
	//          |\2            1 /|
	//          |1\    tria3    /2|
	//          |  \           /  |
	//          |   \         /   |
	//          |    \       /    |
	//          |     \     /     |
	//          |      \ 3 /      |
	//          |tria4  \ / 3     |
	//          |      3 \grid5   |
	//          |       / \       | 
	//          |      / 3 \ tria2|
	//          |     /     \     |
	//          |    /       \    |
	//          |   /         \   |
	//          |  /   tria1   \  |
	//          |2/1           2\1|
	//   grid1  +-----------------+ grid 2
	//
	//

	int      i,j;

	double nx[4];
	double ny[4];
	double nz[4];

	/* gaussian points: */
	int     ig;
	GaussTria* gauss=NULL;

	double  surface_list[5];
	double  x[5];
	double  y[5];
	double  z[5];
	double l12,l14,l15,l23,l25,l34,l35,l45;
	double cos_theta_triangle1,cos_theta_triangle2,cos_theta_triangle3,cos_theta_triangle4;

	double xyz_tria[12][2];
	double l1l2l3_tria[3];
	double r_tria,s_tria;
	double r_quad[4];
	double s_quad[4];
	double l1l4_tria[4][4];

	double J[4];
	double z_g[4];
	double air_pressure_tria;
	double water_level_above_g_tria;
	double water_pressure_tria;
	double pressure_tria;
	int    fill;

	/*To use tria functionalities: */
	Tria* tria=NULL;

	/*Recover inputs: */
	inputs->GetParameterValue(&fill,FillEnum);

	/*Initialize fake tria: */
	tria=new Tria();

	//Build the four normal vectors 
	nx[0]=normal1[0]; nx[1]=normal2[0]; nx[2]=normal3[0]; nx[3]=normal4[0];
	ny[0]=normal1[1]; ny[1]=normal2[1]; ny[2]=normal3[1]; ny[3]=normal4[1];
	nz[0]=normal1[2]; nz[1]=normal2[2]; nz[2]=normal3[2]; nz[3]=normal4[2];
	
	//Recover the surface of the four nodes
	for(i=0;i<4;i++){
		surface_list[i]=thickness_list[i]+bed_list[i];
	}
	//Add surface sor the fifth point (average of the surfaces)
	surface_list[4]=(surface_list[0]+surface_list[1]+surface_list[2]+surface_list[3])/4.0;

	//Recover grid coordinates
	for(i=0;i<5;i++){
		x[i]=*(xyz_list+3*i+0);
		y[i]=*(xyz_list+3*i+1);
		z[i]=*(xyz_list+3*i+2);
	}
	
	//Build triangles in a 2D plan before using reference elements
	l12=pow((x[1]-x[0]),2)+pow((y[1]-y[0]),2)+pow((z[1]-z[0]),2);
	l14=pow((x[3]-x[0]),2)+pow((y[3]-y[0]),2)+pow((z[3]-z[0]),2);
	l15=pow((x[4]-x[0]),2)+pow((y[4]-y[0]),2)+pow((z[4]-z[0]),2);
	l23=pow((x[2]-x[1]),2)+pow((y[2]-y[1]),2)+pow((z[2]-z[1]),2);
	l25=pow((x[4]-x[1]),2)+pow((y[4]-y[1]),2)+pow((z[4]-z[1]),2);
	l34=pow((x[3]-x[2]),2)+pow((y[3]-y[2]),2)+pow((z[3]-z[2]),2);
	l35=pow((x[4]-x[2]),2)+pow((y[4]-y[2]),2)+pow((z[4]-z[2]),2);
	l45=pow((x[4]-x[3]),2)+pow((y[4]-y[3]),2)+pow((z[4]-z[3]),2);

	cos_theta_triangle1=(l15+l12-l25)/(2*sqrt(l12*l15));
	cos_theta_triangle2=(l25+l23-l35)/(2*sqrt(l23*l25));
	cos_theta_triangle3=(l35+l34-l45)/(2*sqrt(l34*l35));
	cos_theta_triangle4=(l45+l14-l15)/(2*sqrt(l14*l45));

	//First triangle : nodes 1, 2 and 5
	xyz_tria[0][0]=0;
	xyz_tria[0][1]=0;
	xyz_tria[1][0]=sqrt(l12);
	xyz_tria[1][1]=0;
	xyz_tria[2][0]=cos_theta_triangle1*sqrt(l15);
	xyz_tria[2][1]=sqrt(1-pow(cos_theta_triangle1,2))*sqrt(l15);

	//Second triangle : nodes 2, 3 and 5
	xyz_tria[3][0]=0;
	xyz_tria[3][1]=0;
	xyz_tria[4][0]=sqrt(l23);
	xyz_tria[4][1]=0;
	xyz_tria[5][0]=cos_theta_triangle2*sqrt(l25);
	xyz_tria[5][1]=sqrt(1-pow(cos_theta_triangle2,2))*sqrt(l25);
	
	//Third triangle : nodes 3, 4 and 5
	xyz_tria[6][0]=0;
	xyz_tria[6][1]=0;
	xyz_tria[7][0]=sqrt(l34);
	xyz_tria[7][1]=0;
	xyz_tria[8][0]=cos_theta_triangle3*sqrt(l35);
	xyz_tria[8][1]=sqrt(1-pow(cos_theta_triangle3,2))*sqrt(l35);

	//Fourth triangle : nodes 4, 1 and 5
	xyz_tria[9][0]=0;
	xyz_tria[9][1]=0;
	xyz_tria[10][0]=sqrt(l14);
	xyz_tria[10][1]=0;
	xyz_tria[11][0]=cos_theta_triangle4*sqrt(l45);
	xyz_tria[11][1]=sqrt(1-pow(cos_theta_triangle4,2))*sqrt(l45);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		tria->GetNodalFunctions(l1l2l3_tria,gauss);

		//Get the coordinates of gauss point for each triangle in penta/quad
		r_tria=gauss->coord2-gauss->coord1;
		s_tria=-3/sqrt((double)3)*(gauss->coord1+gauss->coord2-2/3);

		//Coordinates of gauss points in the reference triangle
		r_quad[0]=r_tria;
		s_quad[0]=1/sqrt((double)3)*s_tria-2/3;
		r_quad[1]=-1/sqrt((double)3)*s_tria+2/3;
		s_quad[1]=r_tria;
		r_quad[2]=-r_tria;
		s_quad[2]=-1/sqrt((double)3)*s_tria+2/3;
		r_quad[3]=1/sqrt((double)3)*s_tria-2/3;
		s_quad[3]=-r_tria;

		//Get the nodal function of the quad for the gauss points of each triangle
		for(i=0;i<4;i++){
			l1l4_tria[i][0]=1.0/4.0*(
					(r_quad[i]-1.0)*(s_quad[i]-1.0) 
					); 
			
			l1l4_tria[i][1]=1.0/4.0*(
					 -(r_quad[i]+1.0)*(s_quad[i]-1.0)
					); 

			l1l4_tria[i][2]=1.0/4.0*(
					(r_quad[i]+1.0)*(s_quad[i]+1.0) 
					);
			
			l1l4_tria[i][3]=1.0/4.0*(
					 -(r_quad[i]-1.0)*(s_quad[i]+1.0)
					);

		} //for(i=0;i<4;i++)
		
		
		//Compute jacobian of each triangle
		for (i=0;i<4;i++){
			double complete_list[3][3]; //a third coordinate is needed for the jacobian determinant calculation, here it is zero
			for(j=0;j<3;j++){
				complete_list[j][0]=xyz_tria[3*i+j][0];
				complete_list[j][1]=xyz_tria[3*i+j][1];
				complete_list[j][2]=0;
			}
			tria->GetJacobianDeterminant2d(&J[i],&complete_list[0][0],gauss);
		}

		//Calculation of the z coordinate for the gaussian point ig for each triangle
		z_g[0]=z[0]*l1l2l3_tria[0]+z[1]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		z_g[1]=z[1]*l1l2l3_tria[0]+z[2]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		z_g[2]=z[2]*l1l2l3_tria[0]+z[3]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		z_g[3]=z[3]*l1l2l3_tria[0]+z[0]*l1l2l3_tria[1]+z[4]*l1l2l3_tria[2];
		
		//Loop on the triangles
		for(i=0;i<4;i++){

			//Loop on the grids of the quad
			//Calculate the ice pressure
			air_pressure_tria=0;

			//Now deal with water pressure: 
			if(fill==WaterEnum){ //icefront ends in water
				water_level_above_g_tria=min(0,z_g[i]);//0 if the gaussian point is above water level
				water_pressure_tria=rho_water*gravity*water_level_above_g_tria;
			}
			else if(fill==AirEnum){
				water_pressure_tria=0;
			}
			else{
				ISSMERROR("QuadPressureLoadStokes error message: unknow fill type for icefront boundary condition");
			}

			//Add pressure 
			pressure_tria = water_pressure_tria + air_pressure_tria;

			pe_g[0]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][0]*nx[i];
			pe_g[1]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][0]*ny[i];
			pe_g[2]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][0]*nz[i];
			pe_g[3]+= 0;
			pe_g[4]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][1]*nx[i];
			pe_g[5]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][1]*ny[i];
			pe_g[6]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][1]*nz[i];
			pe_g[7]+= 0;
			pe_g[8]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][2]*nx[i];
			pe_g[9]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][2]*ny[i];
			pe_g[10]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][2]*nz[i];
			pe_g[11]+= 0;
			pe_g[12]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][3]*nx[i];
			pe_g[13]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][3]*ny[i];
			pe_g[14]+= J[i]*gauss->weight* pressure_tria*l1l4_tria[i][3]*nz[i];
			pe_g[15]+= 0;

		} //for(i=0;i<4;i++)
	} 

	delete tria;
	delete gauss;
}
/*}}}*/
/*FUNCTION Icefront::GetNormal {{{1*/
void Icefront:: GetNormal(double* normal,double xyz_list[2][3]){

	/*Build unit outward pointing vector*/
	const int numnodes=NUMVERTICESSEG;
	double vector[2];
	double norm;

	vector[0]=xyz_list[1][0] - xyz_list[0][0];
	vector[1]=xyz_list[1][1] - xyz_list[0][1];

	norm=sqrt(pow(vector[0],2.0)+pow(vector[1],2.0));

	normal[0]= + vector[1]/norm;
	normal[1]= - vector[0]/norm;
}
/*}}}*/
