/*!\file Penta.cpp
 * \brief: implementation of the Penta object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../Container/Container.h"
/*}}}*/

/*Element macros*/
#define NUMVERTICES   6
#define NUMVERTICES2D 3
#define NDOF1 1
#define NDOF2 2
#define NDOF3 3
#define NDOF4 4

/*Penta constructors and destructor*/
/*FUNCTION Penta::Penta(){{{1*/
Penta::Penta(){
	this->nodes=NULL;
	this->matice=NULL;
	this->matpar=NULL;
	this->neighbors=NULL;
	this->inputs=NULL;
	this->parameters=NULL;
	this->results=NULL;
}
/*}}}*/
/*FUNCTION Penta::~Penta(){{{1*/
Penta::~Penta(){
	delete inputs;
	delete results;
	this->parameters=NULL;
}
/*}}}*/
/*FUNCTION Penta::Penta(int id, int index, IoModel* iomodel,int nummodels) {{{1*/
Penta::Penta(int penta_id, int index, IoModel* iomodel,int nummodels)
	:PentaRef(nummodels)
	,PentaHook(nummodels,index+1,iomodel->numberofelements+1) //index+1: matice id, iomodel->numberofelements+1: matpar id
                                                                      { //i is the element index

	int i;
	int penta_elements_ids[2];

	/*Checks in debugging mode*/
	/*{{{2*/
	ISSMASSERT(iomodel->upperelements);
	ISSMASSERT(iomodel->lowerelements);
	/*}}}*/

	/*id: */
	this->id=penta_id;

	/*Build neighbors list*/
	if isnan(iomodel->upperelements[index]) penta_elements_ids[1]=this->id; //upper penta is the same penta
	else                                    penta_elements_ids[1]=(int)(iomodel->upperelements[index]);
	if isnan(iomodel->lowerelements[index]) penta_elements_ids[0]=this->id; //lower penta is the same penta
	else                                    penta_elements_ids[0]=(int)(iomodel->lowerelements[index]);
	this->InitHookNeighbors(penta_elements_ids);

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;

	/*intialize inputs and results: */
	this->inputs=new Inputs();
	this->results=new Results();
	
	/*initialize pointers:*/
	this->nodes=NULL;
	this->matice=NULL;
	this->matpar=NULL;
	this->neighbors=NULL;
}
/*}}}*/

/*Object virtual functions definitions: */
/*FUNCTION Penta::copy {{{1*/
Object* Penta::copy() {

	int i;

	Penta* penta=NULL;

	penta=new Penta();

	//deal with PentaRef mother class
	penta->element_type_list=(int*)xmalloc(this->numanalyses*sizeof(int));
	for(i=0;i<this->numanalyses;i++) penta->element_type_list[i]=this->element_type_list[i];

	//deal with PentaHook mother class
	penta->numanalyses=this->numanalyses;
	penta->hnodes=new Hook*[penta->numanalyses];
	for(i=0;i<penta->numanalyses;i++)penta->hnodes[i]=(Hook*)this->hnodes[i]->copy();
	penta->hmatice=(Hook*)this->hmatice->copy();
	penta->hmatpar=(Hook*)this->hmatpar->copy();
	penta->hneighbors=(Hook*)this->hneighbors->copy();

	/*deal with Penta  copy fields: */
	penta->id=this->id;
	if(this->inputs){
		penta->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		penta->inputs=new Inputs();
	}
	if(this->results){
		penta->results=(Results*)this->results->Copy();
	}
	else{
		penta->results=new Results();
	}
	/*point parameters: */
	penta->parameters=this->parameters;

	/*recover objects: */
	penta->nodes=(Node**)xmalloc(6*sizeof(Node*)); //we cannot rely on an analysis_counter to tell us which analysis_type we are running, so we just copy the nodes.
	for(i=0;i<6;i++)penta->nodes[i]=this->nodes[i];
	penta->matice=(Matice*)penta->hmatice->delivers();
	penta->matpar=(Matpar*)penta->hmatpar->delivers();
	penta->neighbors=(Penta**)penta->hneighbors->deliverp();

	return penta;
}
/*}}}*/
/*FUNCTION Penta::DeepEcho{{{1*/
void Penta::DeepEcho(void){

	int i;
	
	printf("Penta:\n");
	printf("   id: %i\n",id);
	nodes[0]->DeepEcho();
	nodes[1]->DeepEcho();
	nodes[2]->DeepEcho();
	nodes[3]->DeepEcho();
	nodes[4]->DeepEcho();
	nodes[5]->DeepEcho();
	matice->DeepEcho();
	matpar->DeepEcho();
	printf("   neighbor ids: %i-%i\n",neighbors[0]->Id(),neighbors[1]->Id());
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();
	printf("   results\n");
	results->DeepEcho();
	return;
}
/*}}}*/
/*FUNCTION Penta::Demarshall {{{1*/
void  Penta::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;
	int flaghook;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&numanalyses,marshalled_dataset,sizeof(numanalyses));marshalled_dataset+=sizeof(numanalyses);

	/*demarshall Ref: */
	this->element_type_list=(int*)xmalloc(this->numanalyses*sizeof(int));
	for(i=0;i<numanalyses;i++){ memcpy(&element_type_list[i],marshalled_dataset,sizeof(int));marshalled_dataset+=sizeof(int);}

	/*allocate dynamic memory: */
	this->hnodes=new Hook*[this->numanalyses];
	/*demarshall hooks: */
	for(i=0;i<numanalyses;i++){
		memcpy(&flaghook,marshalled_dataset,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
		if(flaghook){ // there is a hook so demarshall it
			hnodes[i]=new Hook();
			hnodes[i]->Demarshall(&marshalled_dataset);
		}
		else hnodes[i]=NULL; //There is no hook so it is NULL
	}
	hmatice=new Hook(); hmatice->Demarshall(&marshalled_dataset);
	hmatpar=new Hook(); hmatpar->Demarshall(&marshalled_dataset);
	hneighbors=new Hook(); hneighbors->Demarshall(&marshalled_dataset);

	/*pointers are garbage, until configuration is carried out: */
	nodes=NULL;
	matice=NULL;
	matpar=NULL;
	neighbors=NULL;
	
	/*demarshall inputs and results: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 
	results=(Results*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Penta::Echo{{{1*/

void Penta::Echo(void){
	this->DeepEcho();
}
/*}}}*/
/*FUNCTION Penta::Enum {{{1*/
int Penta::Enum(void){

	return PentaEnum;

}
/*}}}*/
/*FUNCTION Penta::Id {{{1*/
int    Penta::Id(void){
	return id; 
}
/*}}}*/
/*FUNCTION Penta::Marshall {{{1*/
void  Penta::Marshall(char** pmarshalled_dataset){

	int   i;
	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;
	char* marshalled_results=NULL;
	int   marshalled_results_size;
	int   flaghook; //to indicate if hook is NULL or exists

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Penta: */
	enum_type=PentaEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Penta data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&numanalyses,sizeof(numanalyses));marshalled_dataset+=sizeof(numanalyses);

	/*Mershall Ref: */
	for(i=0;i<numanalyses;i++){
		memcpy(marshalled_dataset,&element_type_list[i],sizeof(element_type_list[i]));marshalled_dataset+=sizeof(element_type_list[i]);
	}

	/*Marshall hooks: */
	for(i=0;i<numanalyses;i++){
		if(hnodes[i]){
			/*Set flag to 1 as there is a hook */
			flaghook=1;
			memcpy(marshalled_dataset,&flaghook,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
			hnodes[i]->Marshall(&marshalled_dataset);
		}
		else{
			/*Set flag to 0 and do not marshall flag as there is no Hook */
			flaghook=0;
			memcpy(marshalled_dataset,&flaghook,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
		}
	}
	hmatice->Marshall(&marshalled_dataset);
	hmatpar->Marshall(&marshalled_dataset);
	hneighbors->Marshall(&marshalled_dataset);

	/*Marshall inputs and results: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	marshalled_results_size=results->MarshallSize();
	marshalled_results=results->Marshall();
	memcpy(marshalled_dataset,marshalled_results,marshalled_results_size*sizeof(char));
	marshalled_dataset+=marshalled_results_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);
	xfree((void**)&marshalled_results);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Penta::MarshallSize {{{1*/
int   Penta::MarshallSize(){

	int i;
	int hnodes_size=0;;

	for(i=0;i<numanalyses;i++){
		hnodes_size+=sizeof(int); //Flag 0 or 1
		if (hnodes[i]) hnodes_size+=hnodes[i]->MarshallSize();
	}

	return sizeof(id)
		+hnodes_size
		+sizeof(numanalyses)
		+numanalyses*sizeof(int) //element_type_lists
		+hmatice->MarshallSize()
		+hmatpar->MarshallSize()
		+hneighbors->MarshallSize()
		+inputs->MarshallSize()
		+results->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Penta::MyRank {{{1*/
int    Penta::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/

/*Update virtual functions definitions: */
/*FUNCTION Penta::InputUpdateFromConstant(bool value, int name);{{{1*/
void  Penta::InputUpdateFromConstant(bool constant, int name){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new BoolInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromConstant(double value, int name);{{{1*/
void  Penta::InputUpdateFromConstant(double constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new DoubleInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromConstant(int value, int name);{{{1*/
void  Penta::InputUpdateFromConstant(int constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new IntInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolution {{{1*/
void  Penta::InputUpdateFromSolution(double* solution){

	int analysis_type;

	/*retreive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Just branch to the correct InputUpdateFromSolution generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		int approximation;
		inputs->GetParameterValue(&approximation,ApproximationEnum);
		if(approximation==StokesApproximationEnum || approximation==NoneApproximationEnum){
			InputUpdateFromSolutionDiagnosticStokes( solution);
		}
		else{
			InputUpdateFromSolutionDiagnosticHoriz( solution);
		}
	}
	else if (analysis_type==DiagnosticHutterAnalysisEnum){
		InputUpdateFromSolutionDiagnosticHutter( solution);
	}
	else if (analysis_type==DiagnosticVertAnalysisEnum){
		InputUpdateFromSolutionDiagnosticVert( solution);
	}
	else if (analysis_type==AdjointHorizAnalysisEnum){
		int approximation;
		inputs->GetParameterValue(&approximation,ApproximationEnum);
		if(approximation==StokesApproximationEnum || approximation==NoneApproximationEnum){
			InputUpdateFromSolutionAdjointStokes( solution);
		}
		else{
			InputUpdateFromSolutionAdjointHoriz( solution);
		}
	}
	else if (analysis_type==BedSlopeXAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,BedSlopeXEnum);
	}
	else if (analysis_type==BedSlopeYAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,BedSlopeYEnum);
	}
	else if (analysis_type==SurfaceSlopeXAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,SurfaceSlopeXEnum);
	}
	else if (analysis_type==SurfaceSlopeYAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,SurfaceSlopeYEnum);
	}
	else if (analysis_type==PrognosticAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,ThicknessEnum);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,ThicknessEnum);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){
		InputUpdateFromSolutionOneDofCollapsed(solution,VelEnum);
	}
	else if (analysis_type==ThermalAnalysisEnum){
		InputUpdateFromSolutionThermal( solution);
	}
	else if (analysis_type==MeltingAnalysisEnum){
		InputUpdateFromSolutionOneDof(solution,MeltingRateEnum);
	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVector(double* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVector(double* vector, int name, int type){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*Penta update B in InputUpdateFromSolutionThermal, so don't look for B update here.*/

	switch(type){

		case VertexEnum:

			/*New PentaVertexInpu*/
			double values[6];

			/*Get values on the 6 vertices*/
			for (int i=0;i<6;i++){
				values[i]=vector[this->nodes[i]->GetVertexDof()];
			}

			/*update input*/
			this->inputs->AddInput(new PentaVertexInput(name,values));
			return;

		default:

			ISSMERROR("type %i (%s) not implemented yet",type,EnumToString(type));
	}
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVector(int* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVector(int* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVector(bool* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVector(bool* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVectorDakota(double* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVectorDakota(double* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVectorDakota(int* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVectorDakota(int* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVectorDakota(bool* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVectorDakota(bool* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/

/*Element virtual functions definitions: */
/*FUNCTION Penta::AverageOntoPartition {{{1*/
void  Penta::AverageOntoPartition(Vec partition_contributions,Vec partition_areas,double* vertex_response,double* qmu_part){
	ISSMERROR("Not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::ComputeBasalStress {{{1*/
void  Penta::ComputeBasalStress(Vec sigma_b){

	int i,j;
	int    doflist[NUMVERTICES];
	double xyz_list[NUMVERTICES][3];
	double xyz_list_tria[3][3];

	/*Parameters*/
	double rho_ice,gravity;
	double bed_normal[3];
	double bed;
	double basalforce[3];
	double epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double devstresstensor[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double stresstensor[6]={0.0};
	double viscosity;
	int analysis_type;

	int  dofv[3]={0,1,2};
	int  dofp[1]={3};
	double Jdet2d;
	Tria* tria=NULL;

	/*Gauss*/
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_coord[4];

	/*Output*/
	double pressure;
	double sigma_xx,sigma_yy,sigma_zz;
	double sigma_xy,sigma_xz,sigma_yz;
	double surface=0;
	double value=0;
	
	/*flags: */
	bool onbed;
	int  approximation;
	Input* pressure_input=NULL;
	Input* vx_input=NULL;
	Input* vy_input=NULL;
	Input* vz_input=NULL;

	/*parameters: */
	double stokesreconditioning;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*Check analysis_types*/
	if (analysis_type!=DiagnosticHorizAnalysisEnum) ISSMERROR("Not supported yet!");
	if (approximation!=StokesApproximationEnum) ISSMERROR("Not supported yet!");

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	
	if(!onbed){
		//put zero
		VecSetValue(sigma_b,id-1,0.0,INSERT_VALUES);
		return;
	}

	/*recovre material parameters: */
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	for(i=0;i<3;i++){
		for(j=0;j<3;j++){
			xyz_list_tria[i][j]=xyz_list[i][j];
		}
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussLegendreTria(&num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights,2);

	/*Retrieve all inputs we will be needing: */
	pressure_input=inputs->GetInput(PressureEnum);
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);
	vz_input=inputs->GetInput(VzEnum);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

			/*Pick up the gaussian point: */
			gauss_weight=*(gauss_weights+ig);
			gauss_coord[0]=*(first_gauss_area_coord+ig); 
			gauss_coord[1]=*(second_gauss_area_coord+ig);
			gauss_coord[2]=*(third_gauss_area_coord+ig);
			gauss_coord[3]=-1.0; //take the base

			/*Compute strain rate viscosity and pressure: */
			this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss_coord,vx_input,vy_input,vz_input);
			matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);
			pressure_input->GetParameterValue(&pressure, &gauss_coord[0]);

			/*Compute Stress*/
			sigma_xx=2*viscosity*epsilon[0]-pressure*stokesreconditioning; // sigma = nu eps - pressure
			sigma_yy=2*viscosity*epsilon[1]-pressure*stokesreconditioning;
			sigma_zz=2*viscosity*epsilon[2]-pressure*stokesreconditioning;
			sigma_xy=2*viscosity*epsilon[3];
			sigma_xz=2*viscosity*epsilon[4];
			sigma_yz=2*viscosity*epsilon[5];

			/*Get normal vector to the bed */
			BedNormal(&bed_normal[0],xyz_list_tria);

			/*basalforce*/
			basalforce[0] += sigma_xx*bed_normal[0] + sigma_xy*bed_normal[1] + sigma_xz*bed_normal[2];
			basalforce[1] += sigma_xy*bed_normal[0] + sigma_yy*bed_normal[1] + sigma_yz*bed_normal[2];
			basalforce[2] += sigma_xz*bed_normal[0] + sigma_yz*bed_normal[1] + sigma_zz*bed_normal[2];

			/*Get the Jacobian determinant */
			tria->GetJacobianDeterminant3d(&Jdet2d, &xyz_list_tria[0][0],gauss_coord);
			value+=sigma_zz*Jdet2d*gauss_weight;
			surface+=Jdet2d*gauss_weight;
	}
	value=value/surface;

	/*Add value to output*/
	VecSetValue(sigma_b,id-1,(const double)value,INSERT_VALUES);
}
/*}}}*/
/*FUNCTION Penta::ComputeStrainRate {{{1*/
void  Penta::ComputeStrainRate(Vec eps){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
		/*FUNCTION Penta::Configure {{{1*/
void  Penta::Configure(Elements* elementsin, Loads* loadsin, DataSet* nodesin, Materials* materialsin, Parameters* parametersin){

	int analysis_counter;
	
	/*go into parameters and get the analysis_counter: */
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Get Element type*/
	this->element_type=this->element_type_list[analysis_counter];

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	if (this->hnodes[analysis_counter]) this->hnodes[analysis_counter]->configure(nodesin);
	this->hmatice->configure(materialsin);
	this->hmatpar->configure(materialsin);
	this->hneighbors->configure(elementsin);

	/*Now, go pick up the objects inside the hooks: */
	if (this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;
	this->matice=(Matice*)this->hmatice->delivers();
	this->matpar=(Matpar*)this->hmatpar->delivers();
	this->neighbors=(Penta**)this->hneighbors->deliverp();

	/*point parameters to real dataset: */
	this->parameters=parametersin;
}
/*}}}*/
/*FUNCTION Penta::RegularizeInversion {{{1*/
double Penta::RegularizeInversion(void){

	double J;
	Tria* tria=NULL;

	/*flags: */
	bool onbed;
	bool onwater;
	bool onsurface;
	int  approximation;

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*Bail out if this element if:
	 * -> Not MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !onsurface) || (approximation==MacAyealApproximationEnum && !onbed)){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute RegularizeInversion*/

		/*Depth Average B*/
		this->InputDepthAverageAtBase(RheologyBEnum,RheologyBbarEnum,MaterialsEnum);

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->RegularizeInversion();
		delete tria->matice; delete tria;

		/*delete B average*/
		this->matice->inputs->DeleteInput(RheologyBbarEnum);

		return J;
	}
	else{

		/*Depth Average B and put it in inputs*/
		Penta* penta=GetBasalElement();
		penta->InputDepthAverageAtBase(RheologyBEnum,RheologyBbarEnum,MaterialsEnum);
		Input* B_input=penta->matice->inputs->GetInput(RheologyBbarEnum);
		Input* B_copy=(Input*)B_input->copy();
		this->matice->inputs->AddInput((Input*)B_copy);

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->RegularizeInversion();
		delete tria->matice; delete tria;

		/*delete B average*/
		this->matice->inputs->DeleteInput(RheologyBbarEnum);
		penta->matice->inputs->DeleteInput(RheologyBbarEnum);

		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrix {{{1*/
void  Penta::CreateKMatrix(Mat Kgg){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*if debugging mode, check that all pointers exist*/
	ISSMASSERT(this->nodes && this->matice && this->matpar && this->neighbors && this->parameters && this->inputs);

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		int approximation;
		inputs->GetParameterValue(&approximation,ApproximationEnum);
		if(approximation==MacAyealApproximationEnum){
			CreateKMatrixDiagnosticMacAyeal( Kgg);
		}
		else if(approximation==PattynApproximationEnum){
			CreateKMatrixDiagnosticPattyn( Kgg);
		}
		else if(approximation==StokesApproximationEnum){
			CreateKMatrixDiagnosticStokes( Kgg);
		}
		else if(approximation==HutterApproximationEnum){
			return;
		}
		else if(approximation==NoneApproximationEnum){
			return;
		}
		else if(approximation==MacAyealPattynApproximationEnum){
			CreateKMatrixDiagnosticMacAyeal( Kgg);
			CreateKMatrixDiagnosticPattyn( Kgg);
			CreateKMatrixCouplingMacAyealPattyn( Kgg);
		}
		else if(approximation==PattynStokesApproximationEnum){
			CreateKMatrixDiagnosticPattyn( Kgg);
			CreateKMatrixDiagnosticStokes( Kgg);
			//CreateKMatrixCouplingPattynStokes( Kgg);
		}
		else ISSMERROR("Approximation %s not supported yet",EnumToString(approximation));
	}
	else if (analysis_type==AdjointHorizAnalysisEnum){
		/*Same as diagnostic*/
		int approximation;
		inputs->GetParameterValue(&approximation,ApproximationEnum);
		if(approximation==MacAyealApproximationEnum){
			CreateKMatrixDiagnosticMacAyeal( Kgg);
		}
		else if(approximation==PattynApproximationEnum){
			CreateKMatrixDiagnosticPattyn( Kgg);
		}
		else if(approximation==StokesApproximationEnum){
			CreateKMatrixDiagnosticStokes( Kgg);
		}
		else if(approximation==NoneApproximationEnum){
			return;
		}
		else ISSMERROR("Approximation %s not supported yet",EnumToString(approximation));
	}
	else if (analysis_type==DiagnosticHutterAnalysisEnum){
		CreateKMatrixDiagnosticHutter( Kgg);
	}
	else if (analysis_type==DiagnosticVertAnalysisEnum){
		CreateKMatrixDiagnosticVert( Kgg);
	}
	else if (analysis_type==BedSlopeXAnalysisEnum || analysis_type==SurfaceSlopeXAnalysisEnum || analysis_type==BedSlopeYAnalysisEnum || analysis_type==SurfaceSlopeYAnalysisEnum){
		CreateKMatrixSlope( Kgg);
	}
	else if (analysis_type==PrognosticAnalysisEnum){
		CreateKMatrixPrognostic( Kgg);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){
		CreateKMatrixBalancedthickness( Kgg);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){
		CreateKMatrixBalancedvelocities( Kgg);
	}
	else if (analysis_type==ThermalAnalysisEnum){
		CreateKMatrixThermal( Kgg);
	}
	else if (analysis_type==MeltingAnalysisEnum){
		CreateKMatrixMelting( Kgg);
	}
	else ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));

}
/*}}}*/
/*FUNCTION Penta::CreatePVector {{{1*/
void  Penta::CreatePVector(Vec pg){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*if debugging mode, check that all pointers exist*/
	ISSMASSERT(this->nodes && this->matice && this->matpar && this->neighbors && this->parameters && this->inputs);

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		int approximation;
		inputs->GetParameterValue(&approximation,ApproximationEnum);
		if(approximation==MacAyealApproximationEnum){
			CreatePVectorDiagnosticMacAyeal( pg);
		}
		else if(approximation==PattynApproximationEnum){
			CreatePVectorDiagnosticPattyn( pg);
		}
		else if(approximation==HutterApproximationEnum){
			return;
		}
		else if(approximation==NoneApproximationEnum){
			return;
		}
		else if(approximation==StokesApproximationEnum){
			CreatePVectorDiagnosticStokes( pg);
		}
		else if(approximation==MacAyealPattynApproximationEnum){
			CreatePVectorDiagnosticMacAyeal( pg);
			CreatePVectorDiagnosticPattyn( pg);
		}
		else if(approximation==PattynStokesApproximationEnum){
			CreatePVectorDiagnosticPattyn( pg);
			CreatePVectorDiagnosticStokes( pg);
			//CreatePVectorDiagnosticPattynStokes( pg);
		}
		else ISSMERROR("Approximation %s not supported yet",EnumToString(approximation));
	}
	else if (analysis_type==AdjointHorizAnalysisEnum){
		int approximation;
		inputs->GetParameterValue(&approximation,ApproximationEnum);
		if(approximation==MacAyealApproximationEnum){
			CreatePVectorAdjointMacAyeal( pg);
		}
		else if(approximation==PattynApproximationEnum){
			CreatePVectorAdjointPattyn( pg);
		}
		else if(approximation==StokesApproximationEnum){
			CreatePVectorAdjointStokes( pg);
		}
		else if(approximation==NoneApproximationEnum){
			return;
		}
		else ISSMERROR("Approximation %s not supported yet",EnumToString(approximation));
	}
	else if (analysis_type==DiagnosticHutterAnalysisEnum){
		CreatePVectorDiagnosticHutter( pg);
	}
	else if (analysis_type==DiagnosticVertAnalysisEnum){
		CreatePVectorDiagnosticVert( pg);
	}
	else if (analysis_type==BedSlopeXAnalysisEnum || analysis_type==SurfaceSlopeXAnalysisEnum || analysis_type==BedSlopeYAnalysisEnum || analysis_type==SurfaceSlopeYAnalysisEnum){
		CreatePVectorSlope( pg);
	}
	else if (analysis_type==PrognosticAnalysisEnum){
		CreatePVectorPrognostic( pg);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){
		CreatePVectorBalancedthickness( pg);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){
		CreatePVectorBalancedvelocities( pg);
	}
	else if (analysis_type==ThermalAnalysisEnum){
		CreatePVectorThermal( pg);
	}
	else if (analysis_type==MeltingAnalysisEnum){
		CreatePVectorMelting( pg);
	}
	else ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));

}
/*}}}*/
/*FUNCTION Penta::DeleteResults {{{1*/
void  Penta::DeleteResults(void){

	/*Delete and reinitialize results*/
	delete this->results;
	this->results=new Results();

}
/*}}}*/
/*FUNCTION Penta::GetNodes {{{1*/
void  Penta::GetNodes(void** vpnodes){

	int i;
	Node** pnodes=NULL;
	
	/*virtual object: */
	pnodes=(Node**)vpnodes;

	for(i=0;i<6;i++){
		pnodes[i]=nodes[i];
	}
}
/*}}}*/
/*FUNCTION Penta::GetNodeIndex {{{1*/
int Penta::GetNodeIndex(Node* node){

	int i;

	for(i=0;i<NUMVERTICES;i++){
		if(node==nodes[i])
		 return i;
	}
	ISSMERROR("Node provided not found among element nodes");

}
/*}}}*/
/*FUNCTION Penta::GetOnBed {{{1*/
bool Penta::GetOnBed(){
	
	bool onbed;

	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	return onbed;
}
/*}}}*/
/*FUNCTION Penta::GetShelf {{{1*/
bool   Penta::GetShelf(){
	bool onshelf;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onshelf,ElementOnIceShelfEnum);

	return onshelf;
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputs{{{1*/
void  Penta::GetSolutionFromInputs(Vec solution){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Just branch to the correct InputUpdateFromSolution generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		int approximation;
		inputs->GetParameterValue(&approximation,ApproximationEnum);
		if(approximation==StokesApproximationEnum || approximation==NoneApproximationEnum){
			GetSolutionFromInputsDiagnosticStokes(solution);
		}
		else{
			GetSolutionFromInputsDiagnosticHoriz(solution);
		}
	}
	else if(analysis_type==DiagnosticHutterAnalysisEnum){
		GetSolutionFromInputsDiagnosticHutter(solution);
	}
	else if(analysis_type==DiagnosticVertAnalysisEnum){
		GetSolutionFromInputsDiagnosticVert(solution);
	}
	else if(analysis_type==ThermalAnalysisEnum){
		GetSolutionFromInputsThermal(solution);
	}
	else{
		ISSMERROR("analysis: %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}
}
/*}}}*/
/*FUNCTION Penta::GetVectorFromInputs{{{1*/
void  Penta::GetVectorFromInputs(Vec vector,int NameEnum){

	int i;
	int doflist1[NUMVERTICES];

	/*Find NameEnum input in the inputs dataset, and get it to fill in the vector: */
	for(i=0;i<this->inputs->Size();i++){
		Input* input=(Input*)this->inputs->GetObjectByOffset(i);
		if(input->EnumType()==NameEnum){
			/*We found the enum.  Use its values to fill into the vector, using the vertices ids: */
			this->GetDofList1(&doflist1[0]);
			input->GetVectorFromInputs(vector,&doflist1[0]);
			break;
		}
	}
}
/*}}}*/
/*FUNCTION Penta::Gradj {{{1*/
void  Penta::Gradj(Vec gradient,int control_type){

	/*inputs: */
	bool onwater;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip grad (=0): */
	if(onwater)return;

	if (control_type==DragCoefficientEnum){
		GradjDrag(gradient);
	}
	else if (control_type==RheologyBbarEnum){
		GradjB(gradient);
	}
	else ISSMERROR("control type %s not supported yet: ",EnumToString(control_type));
}
/*}}}*/
/*FUNCTION Penta::InputConvergence{{{1*/
bool Penta::InputConvergence(double* eps, int* enums,int num_enums,int* criterionenums,double* criterionvalues,int num_criterionenums){

	int i;
	Input** new_inputs=NULL;
	Input** old_inputs=NULL;
	bool    converged=true;

	new_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the new inputs
	old_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the old inputs

	for(i=0;i<num_enums/2;i++){
		new_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+0]);
		old_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+1]);
		if(!new_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumToString(enums[2*i+0]));
		if(!old_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumToString(enums[2*i+0]));
	}

	/*ok, we've got the inputs (new and old), now loop throught the number of criterions and fill the eps array:*/
	for(i=0;i<num_criterionenums;i++){
		IsInputConverged(eps+i,new_inputs,old_inputs,num_enums/2,criterionenums[i]);
		if(eps[i]>criterionvalues[i]) converged=false; 
	}

	/*clean up*/
	xfree((void**)&new_inputs);
	xfree((void**)&old_inputs);

	/*Return output*/
	return converged;

}
/*}}}*/
/*FUNCTION Penta::InputControlUpdate{{{1*/
void  Penta::InputControlUpdate(double scalar,bool save_parameter){

	/*Intermediary*/
	Input* input=NULL;
	double cm_min,cm_max;
	int    control_type;
	bool   onbed;

	/*retrieve some parameters: */
	this->parameters->FindParam(&cm_min,CmMinEnum);
	this->parameters->FindParam(&cm_max,CmMaxEnum);
	this->parameters->FindParam(&control_type,ControlTypeEnum);

	/*Rheology*/
	if(control_type==RheologyBbarEnum){

		/*The update is done by the element on bed only*/
		inputs->GetParameterValue(&onbed,ElementOnBedEnum);
		if (!onbed) return;

		/*First, get revert to previous parameter value (kept in ControlParameter input)*/
		matice->inputs->DuplicateInput(ControlParameterEnum,RheologyBEnum);

		/*Now, update using the gradient new = old + scalar * gradient*/
		//matice->inputs->AXPY(RheologyBbarEnum,scalar,GradientEnum);
		// For now: Gradient is in element (TO BE CHANGED) and parameter in matice
		Input* input_B   =(Input*)matice->inputs->GetInput(RheologyBEnum); ISSMASSERT(input_B);
		Input* input_Grad=(Input*)this->inputs->GetInput(GradientEnum); ISSMASSERT(input_Grad);
		input_B->AXPY(input_Grad,scalar);

		/*Constrain constrain input*/
		input=(Input*)matice->inputs->GetInput(RheologyBEnum); ISSMASSERT(input);
		input->Constrain(cm_min,cm_max);

		/*OK, now we can propagate the new parameter up*/
		this->InputExtrude(RheologyBEnum,MaterialsEnum);

		/*Finally: save input if requested*/
		if (save_parameter){
			matice->inputs->DuplicateInput(RheologyBEnum,ControlParameterEnum);
			this->InputExtrude(ControlParameterEnum,MaterialsEnum);
		}

	}
	else if(control_type==DragCoefficientEnum){

		/*First, get revert to previous parameter value (kept in ControlParameter input)*/
		this->inputs->DuplicateInput(ControlParameterEnum,DragCoefficientEnum);

		/*Now, update using the gradient new = old + scalar * gradient*/
		this->inputs->AXPY(DragCoefficientEnum,scalar,GradientEnum);

		/*Constrain input*/
		input=(Input*)this->inputs->GetInput(DragCoefficientEnum); ISSMASSERT(input);
		input->Constrain(cm_min,cm_max);

		/*Finally: save input if requested*/
		if (save_parameter) inputs->DuplicateInput(DragCoefficientEnum,ControlParameterEnum);

	}
	else{
		ISSMERROR("control type %s not implemented yet",EnumToString(control_type));
	}
}
/*}}}*/
/*FUNCTION Penta::InputDepthAverageAtBase{{{1*/
void  Penta::InputDepthAverageAtBase(int enum_type,int average_enum_type,int object_enum){

	/*Intermediaries*/
	bool onbed;
	int  step,i;

	Penta* penta=NULL;
	Input* original_input=NULL;
	Input* element_integrated_input=NULL;
	Input* total_integrated_input=NULL;
	Input* element_thickness_input=NULL;
	Input* total_thickness_input=NULL;
	Input* depth_averaged_input=NULL;

	double  xyz_list[NUMVERTICES][3];
	double  Helem_list[NUMVERTICES];
	double  zeros_list[NUMVERTICES]={0.0};

	/*recover parameters: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*Are we on the base? If not, return*/
	if(!onbed) return;

	/*OK, we are on bed. Initialize global inputs as 0*/
	total_thickness_input =new PentaVertexInput(ThicknessEnum,zeros_list);

	/*Now follow all the upper element from the base to the surface to integrate the input*/
	penta=this;
	step =0;
	for(;;){

		/*Step1: Get original input (to be depth avegaged): */
		if (object_enum==ElementsEnum)
		 original_input=(Input*)penta->inputs->GetInput(enum_type);
		else if (object_enum==MaterialsEnum)
		 original_input=(Input*)penta->matice->inputs->GetInput(enum_type);
		else
		 ISSMERROR("object %s not supported yet",EnumToString(object_enum));
		if(!original_input) ISSMERROR("could not find input with enum %s",EnumToString(enum_type));

		/*If first time, initialize total_integrated_input*/
		if (step==0){
			if (original_input->Enum()==PentaVertexInputEnum)
			 total_integrated_input=new PentaVertexInput(average_enum_type,zeros_list);
			else if (original_input->Enum()==DoubleInputEnum)
			 total_integrated_input=new DoubleInput(average_enum_type,0.0);
			else
			 ISSMERROR("object %s not supported yet",EnumToString(original_input->Enum()));
		}

		/*Step2: Create element thickness input*/
		GetVerticesCoordinates(&xyz_list[0][0],penta->nodes,NUMVERTICES);
		for(i=0;i<3;i++){
			Helem_list[i]=xyz_list[i+3][2]-xyz_list[i][2];
			Helem_list[i+3]=Helem_list[i];
		}
		element_thickness_input=new PentaVertexInput(ThicknessEnum,Helem_list);

		/*Step3: Vertically integrate A COPY of the original*/
		element_integrated_input=(Input*)original_input->copy();
		element_integrated_input->VerticallyIntegrate(element_thickness_input);

		/*Add contributions to global inputs*/
		total_integrated_input->AXPY(element_integrated_input,1.0);
		total_thickness_input ->AXPY(element_thickness_input,1.0);

		/*Clean up*/
		delete element_thickness_input;
		delete element_integrated_input;

		/*Stop if we have reached the surface, else, take upper penta*/
		if (penta->IsOnSurface()) break;

		/* get upper Penta*/
		penta=penta->GetUpperElement();
		ISSMASSERT(penta->Id()!=this->id);

		/*increase couter*/
		step++;
	}

	/*OK, now we only need to divide the depth integrated input by the total thickness!*/
	depth_averaged_input=total_integrated_input->PointwiseDivide(total_thickness_input);
	depth_averaged_input->ChangeEnum(average_enum_type);

	/*Clean up*/
	delete total_thickness_input;
	delete total_integrated_input;

	/*Finally, add to inputs*/
	if (object_enum==ElementsEnum)
	 this->inputs->AddInput((Input*)depth_averaged_input);
	else if (object_enum==MaterialsEnum)
	 this->matice->inputs->AddInput((Input*)depth_averaged_input);
	else
	 ISSMERROR("object %s not supported yet",EnumToString(object_enum));
}
/*}}}*/
/*FUNCTION Penta::InputDuplicate{{{1*/
void  Penta::InputDuplicate(int original_enum,int new_enum){

	/*Call inputs method*/
	if (IsInput(original_enum)) inputs->DuplicateInput(original_enum,new_enum);

}
/*}}}*/
/*FUNCTION Penta::InputArtificialNoise{{{1*/
void  Penta::InputArtificialNoise(int enum_type,double min,double max){

	Input* input=NULL;

	/*Make a copy of the original input: */
	input=(Input*)this->inputs->GetInput(enum_type);
	if(!input)ISSMERROR(" could not find old input with enum: %s",EnumToString(enum_type));

	/*ArtificialNoise: */
	input->ArtificialNoise(min,max);
}
/*}}}*/
/*FUNCTION Penta::InputScale{{{1*/
void  Penta::InputScale(int enum_type,double scale_factor){

	Input* input=NULL;

	/*Make a copy of the original input: */
	input=(Input*)this->inputs->GetInput(enum_type);
	if(!input)ISSMERROR(" could not find old input with enum: %s",EnumToString(enum_type));

	/*Scale: */
	input->Scale(scale_factor);
}
/*}}}*/
/*FUNCTION Penta::InputToResult{{{1*/
void  Penta::InputToResult(int enum_type,int step,double time){

	int    i;
	bool   found = false;
	Input *input = NULL;

	/*Go through all the input objects, and find the one corresponding to enum_type, if it exists: */
	if (enum_type==RheologyBbarEnum){
		input=this->matice->inputs->GetInput(RheologyBEnum);
	}
	else{
		input=this->inputs->GetInput(enum_type);
	}
	if (!input) ISSMERROR("Input %s not found in penta->inputs",EnumToString(enum_type));

	/*If we don't find it, no big deal, just don't do the transfer. Otherwise, build a new Result 
	 * object out of the input, with the additional step and time information: */
	this->results->AddObject((Object*)input->SpawnResult(step,time));

}
/*}}}*/
/*FUNCTION Penta::MassFlux {{{1*/
double Penta::MassFlux( double* segment,bool process_units){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::MaxAbsVx{{{1*/
void  Penta::MaxAbsVx(double* pmaxabsvx, bool process_units){

	/*Get maximum:*/
	double maxabsvx=this->inputs->MaxAbs(VxEnum);

	/*process units if requested: */
	if(process_units) maxabsvx=UnitConversion(maxabsvx,IuToExtEnum,VxEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxabsvx=maxabsvx;
}
/*}}}*/
/*FUNCTION Penta::MaxAbsVy{{{1*/
void  Penta::MaxAbsVy(double* pmaxabsvy, bool process_units){

	/*Get maximum:*/
	double maxabsvy=this->inputs->MaxAbs(VyEnum);

	/*process units if requested: */
	if(process_units) maxabsvy=UnitConversion(maxabsvy,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxabsvy=maxabsvy;
}
/*}}}*/
/*FUNCTION Penta::MaxAbsVz{{{1*/
void  Penta::MaxAbsVz(double* pmaxabsvz, bool process_units){

	/*Get maximum:*/
	double maxabsvz=this->inputs->MaxAbs(VzEnum);

	/*process units if requested: */
	if(process_units) maxabsvz=UnitConversion(maxabsvz,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxabsvz=maxabsvz;
}
/*}}}*/
/*FUNCTION Penta::MaxVel{{{1*/
void  Penta::MaxVel(double* pmaxvel, bool process_units){

	/*Get maximum:*/
	double maxvel=this->inputs->Max(VelEnum);

	/*process units if requested: */
	if(process_units) maxvel=UnitConversion(maxvel,IuToExtEnum,VelEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvel=maxvel;

}
/*}}}*/
/*FUNCTION Penta::MaxVx{{{1*/
void  Penta::MaxVx(double* pmaxvx, bool process_units){

	/*Get maximum:*/
	double maxvx=this->inputs->Max(VxEnum);

	/*process units if requested: */
	if(process_units) maxvx=UnitConversion(maxvx,IuToExtEnum,VxEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvx=maxvx;

}
/*}}}*/
/*FUNCTION Penta::MaxVy{{{1*/
void  Penta::MaxVy(double* pmaxvy, bool process_units){

	/*Get maximum:*/
	double maxvy=this->inputs->Max(VyEnum);

	/*process units if requested: */
	if(process_units) maxvy=UnitConversion(maxvy,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvy=maxvy;

}
/*}}}*/
/*FUNCTION Penta::MaxVz{{{1*/
void  Penta::MaxVz(double* pmaxvz, bool process_units){

	/*Get maximum:*/
	double maxvz=this->inputs->Max(VzEnum);

	/*process units if requested: */
	if(process_units) maxvz=UnitConversion(maxvz,IuToExtEnum,VzEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvz=maxvz;

}
/*}}}*/
/*FUNCTION Penta::MinVel{{{1*/
void  Penta::MinVel(double* pminvel, bool process_units){

	/*Get minimum:*/
	double minvel=this->inputs->Min(VelEnum);

	/*process units if requested: */
	if(process_units) minvel=UnitConversion(minvel,IuToExtEnum,VelEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvel=minvel;

}
/*}}}*/
/*FUNCTION Penta::MinVx{{{1*/
void  Penta::MinVx(double* pminvx, bool process_units){

	/*Get minimum:*/
	double minvx=this->inputs->Min(VxEnum);

	/*process units if requested: */
	if(process_units) minvx=UnitConversion(minvx,IuToExtEnum,VxEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvx=minvx;

}
/*}}}*/
/*FUNCTION Penta::MinVy{{{1*/
void  Penta::MinVy(double* pminvy, bool process_units){

	/*Get minimum:*/
	double minvy=this->inputs->Min(VyEnum);

	/*process units if requested: */
	if(process_units) minvy=UnitConversion(minvy,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvy=minvy;

}
/*}}}*/
/*FUNCTION Penta::MinVz{{{1*/
void  Penta::MinVz(double* pminvz, bool process_units){

	/*Get minimum:*/
	double minvz=this->inputs->Min(VzEnum);

	/*process units if requested: */
	if(process_units) minvz=UnitConversion(minvz,IuToExtEnum,VzEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvz=minvz;

}
/*}}}*/
/*FUNCTION Penta::ThicknessAbsMisfit {{{1*/
double Penta::ThicknessAbsMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onsurface;
	bool onbed;
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(onwater)return 0;
	ISSMERROR("Not implemented yet");

	tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
	J=tria->ThicknessAbsMisfit(process_units);
	delete tria->matice; delete tria;
	return J;
}
/*}}}*/
/*FUNCTION Penta::SurfaceAbsVelMisfit {{{1*/
double Penta::SurfaceAbsVelMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onsurface;
	bool onbed;
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !onsurface) || (approximation==MacAyealApproximationEnum && !onbed)){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceAbsVelMisfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->SurfaceAbsVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->SurfaceAbsVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceRelVelMisfit {{{1*/
double Penta::SurfaceRelVelMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onsurface;
	bool onbed;
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !onsurface) || (approximation==MacAyealApproximationEnum && !onbed)){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceRelVelMisfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->SurfaceRelVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->SurfaceRelVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceLogVelMisfit {{{1*/
double Penta::SurfaceLogVelMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onsurface;
	bool onbed;
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !onsurface) || (approximation==MacAyealApproximationEnum && !onbed)){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceLogVelMisfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->SurfaceLogVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->SurfaceLogVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceLogVxVyMisfit {{{1*/
double Penta::SurfaceLogVxVyMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onsurface;
	bool onbed;
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !onsurface) || (approximation==MacAyealApproximationEnum && !onbed)){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceLogVxVyMisfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->SurfaceLogVxVyMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->SurfaceLogVxVyMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceAverageVelMisfit {{{1*/
double Penta::SurfaceAverageVelMisfit(bool process_units){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onsurface;
	bool onbed;
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !onsurface) || (approximation==MacAyealApproximationEnum && !onbed)){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceAverageVelMisfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->SurfaceAverageVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->SurfaceAverageVelMisfit(process_units);
		delete tria->matice; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::PatchFill{{{1*/
void  Penta::PatchFill(int* pcount, Patch* patch){

	int i;
	int count;
	int vertices_ids[6];


	/*recover pointer: */
	count=*pcount;
		
	/*will be needed later: */
	for(i=0;i<6;i++) vertices_ids[i]=nodes[i]->GetVertexId(); //vertices id start at column 3 of the patch.

	for(i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);

		/*For this result,fill the information in the Patch object (element id + vertices ids), and then hand 
		 *it to the result object, to fill the rest: */
		patch->fillelementinfo(count,this->id,vertices_ids,6);
		elementresult->PatchFill(count,patch);

		/*increment counter: */
		count++;
	}

	/*Assign output pointers:*/
	*pcount=count;
}/*}}}*/
/*FUNCTION Penta::PatchSize{{{1*/
void  Penta::PatchSize(int* pnumrows, int* pnumvertices,int* pnumnodes){

	int     i;
	
	/*output: */
	int     numrows     = 0;
	int     numnodes    = 0;

	/*Go through all the results objects, and update the counters: */
	for (i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);
		/*first, we have one more result: */
		numrows++;
		/*now, how many vertices and how many nodal values for this result? :*/
		numnodes=elementresult->NumberOfNodalValues(); //ask result object.
	}

	/*Assign output pointers:*/
	*pnumrows=numrows;
	*pnumvertices=NUMVERTICES;
	*pnumnodes=numnodes;
	
}
/*}}}*/
/*FUNCTION Penta::ProcessResultsUnits{{{1*/
void  Penta::ProcessResultsUnits(void){

	int i;

	for(i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);
		elementresult->ProcessUnits(this->parameters);
	}

}
/*}}}*/
/*FUNCTION Penta::SetCurrentConfiguration {{{1*/
void  Penta::SetCurrentConfiguration(Elements* elementsin, Loads* loadsin, DataSet* nodesin, Materials* materialsin, Parameters* parametersin){

	int analysis_counter;

	/*go into parameters and get the analysis_counter: */
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Get Element type*/
	this->element_type=this->element_type_list[analysis_counter];

	/*Pick up nodes */
	if (this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;
}
/*}}}*/
/*FUNCTION Penta::SurfaceArea {{{1*/
double Penta::SurfaceArea(void){

	double S;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onsurface;
	bool onbed;
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*Bail out if this element if:
	 * -> Non MacAyeal not on the surface
	 * -> MacAyeal (2d model) and not on bed) */
	if ((approximation!=MacAyealApproximationEnum && !onsurface) || (approximation==MacAyealApproximationEnum && !onbed)){
		return 0;
	}
	else if (approximation==MacAyealApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceArea*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		S=tria->SurfaceArea();
		delete tria->matice; delete tria;
		return S;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		S=tria->SurfaceArea();
		delete tria->matice; delete tria;
		return S;
	}
}
/*}}}*/
/*FUNCTION Penta::Update {{{1*/
void Penta::Update(int index,IoModel* iomodel,int analysis_counter,int analysis_type){ 

	/*Intermediaries*/
	IssmInt i;
	int     penta_type;
	int     penta_node_ids[6];
	int     penta_vertex_ids[6];
	double  nodeinputs[6];

	/*Checks if debuging*/
	/*{{{2*/
	ISSMASSERT(iomodel->elements);
	/*}}}*/

	/*Recover element type*/
	if ((analysis_type==PrognosticAnalysisEnum || analysis_type==BalancedthicknessAnalysisEnum) && iomodel->prognostic_DG){
		/*P1 Discontinuous Galerkin*/
		penta_type=P1DGEnum;
	}
	else if (analysis_type==StokesAnalysisEnum){
		/*MINI Continuous Galerkin*/
		penta_type=MiniEnum;
	}
	else{
		/*P1 Continuous Galerkin*/
		penta_type=P1Enum;
	}
	this->SetElementType(penta_type,analysis_counter);

	/*Recover vertices ids needed to initialize inputs*/
	for(i=0;i<6;i++){ 
		penta_vertex_ids[i]=(int)iomodel->elements[6*index+i]; //ids for vertices are in the elements array from Matlab
	}

	/*Recover nodes ids needed to initialize the node hook.*/
	for(i=0;i<6;i++){ 
		//go recover node ids, needed to initialize the node hook.
		//WARNING: We assume P1 elements here!!!!!
		penta_node_ids[i]=iomodel->nodecounter+(int)iomodel->elements[6*index+i]; //ids for vertices are in the elements array from Matlab
	}

	/*hooks: */
	this->SetHookNodes(penta_node_ids,analysis_counter); this->nodes=NULL; //set hook to nodes, for this analysis type

	//add as many inputs per element as requested: 
	if (iomodel->thickness) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->thickness[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(ThicknessEnum,nodeinputs));
	}
	if (iomodel->surface) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->surface[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(SurfaceEnum,nodeinputs));
	}
	if (iomodel->bed) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->bed[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(BedEnum,nodeinputs));
	}
	if (iomodel->drag_coefficient) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->drag_coefficient[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(DragCoefficientEnum,nodeinputs));

		if (iomodel->drag_p) this->inputs->AddInput(new DoubleInput(DragPEnum,iomodel->drag_p[index]));
		if (iomodel->drag_q) this->inputs->AddInput(new DoubleInput(DragQEnum,iomodel->drag_q[index]));
		this->inputs->AddInput(new IntInput(DragTypeEnum,iomodel->drag_type));

	}
	if (iomodel->control_parameter) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->control_parameter[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(ControlParameterEnum,nodeinputs));
	}
	if (iomodel->melting_rate) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->melting_rate[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(MeltingRateEnum,nodeinputs));
	}
	if (iomodel->accumulation_rate) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->accumulation_rate[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(AccumulationRateEnum,nodeinputs));
	}
	if (iomodel->geothermalflux) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->geothermalflux[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(GeothermalFluxEnum,nodeinputs));
	}	
	if (iomodel->pressure) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->pressure[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(PressureEnum,nodeinputs));
	}
	if (iomodel->temperature) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->temperature[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(TemperatureEnum,nodeinputs));
	}
	if (iomodel->dhdt) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->dhdt[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(DhDtEnum,nodeinputs));
	}
	/*vx,vy and vz: */
	if (iomodel->vx) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vx[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VxEnum,nodeinputs));
		this->inputs->AddInput(new PentaVertexInput(VxOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new PentaVertexInput(QmuVxEnum,nodeinputs));
	}
	if (iomodel->vy) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vy[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VyEnum,nodeinputs));
		this->inputs->AddInput(new PentaVertexInput(VyOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new PentaVertexInput(QmuVyEnum,nodeinputs));
	}
	if (iomodel->vz) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vz[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VzEnum,nodeinputs));
		this->inputs->AddInput(new PentaVertexInput(VzOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new PentaVertexInput(QmuVzEnum,nodeinputs));
	}
	if (iomodel->vx_obs) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vx_obs[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VxObsEnum,nodeinputs));
	}
	if (iomodel->vy_obs) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vy_obs[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VyObsEnum,nodeinputs));
	}
	if (iomodel->vz_obs) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vz_obs[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VzObsEnum,nodeinputs));
	}
	if (iomodel->weights) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->weights[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(WeightsEnum,nodeinputs));
	}
	if (iomodel->elementoniceshelf) this->inputs->AddInput(new BoolInput(ElementOnIceShelfEnum,(IssmBool)iomodel->elementoniceshelf[index]));
	if (iomodel->elementonbed) this->inputs->AddInput(new BoolInput(ElementOnBedEnum,(IssmBool)iomodel->elementonbed[index]));
	if (iomodel->elementonwater) this->inputs->AddInput(new BoolInput(ElementOnWaterEnum,(IssmBool)iomodel->elementonwater[index]));
	if (iomodel->elementonsurface) this->inputs->AddInput(new BoolInput(ElementOnSurfaceEnum,(IssmBool)iomodel->elementonsurface[index]));

	/*Defaults if not provided in iomodel*/
	switch(analysis_type){

		case DiagnosticHorizAnalysisEnum:

			/*default vx,vy and vz: either observation or 0 */
			if(!iomodel->vx){
				if (iomodel->vx_obs) for(i=0;i<6;i++)nodeinputs[i]=iomodel->vx_obs[penta_vertex_ids[i]-1]/iomodel->yts;
				else                 for(i=0;i<6;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new PentaVertexInput(VxEnum,nodeinputs));
				this->inputs->AddInput(new PentaVertexInput(VxOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new PentaVertexInput(QmuVxEnum,nodeinputs));
			}
			if(!iomodel->vy){
				if (iomodel->vy_obs) for(i=0;i<6;i++)nodeinputs[i]=iomodel->vy_obs[penta_vertex_ids[i]-1]/iomodel->yts;
				else                 for(i=0;i<6;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new PentaVertexInput(VyEnum,nodeinputs));
				this->inputs->AddInput(new PentaVertexInput(VyOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new PentaVertexInput(QmuVyEnum,nodeinputs));
			}
			if(!iomodel->vz){
				if (iomodel->vz_obs) for(i=0;i<6;i++)nodeinputs[i]=iomodel->vz_obs[penta_vertex_ids[i]-1]/iomodel->yts;
				else                 for(i=0;i<6;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new PentaVertexInput(VzEnum,nodeinputs));
				this->inputs->AddInput(new PentaVertexInput(VzOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new PentaVertexInput(QmuVzEnum,nodeinputs));
			}
			if(!iomodel->pressure){
				for(i=0;i<6;i++)nodeinputs[i]=0;
				if(iomodel->qmu_analysis){
					this->inputs->AddInput(new PentaVertexInput(PressureEnum,nodeinputs));
					this->inputs->AddInput(new PentaVertexInput(QmuPressureEnum,nodeinputs));
				}
				if(iomodel->isstokes){
					this->inputs->AddInput(new PentaVertexInput(PressureEnum,nodeinputs));
					this->inputs->AddInput(new PentaVertexInput(PressureOldEnum,nodeinputs));
				}
			}
			break;

		default:
			/*No update for other solution types*/
			break;

	}

	//Need to know the type of approximation for this element
	if(iomodel->elements_type){
		if (*(iomodel->elements_type+index)==MacAyealApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,MacAyealApproximationEnum));
		}
		else if (*(iomodel->elements_type+index)==PattynApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,PattynApproximationEnum));
		}
		else if (*(iomodel->elements_type+index)==MacAyealPattynApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,MacAyealPattynApproximationEnum));
		}
		else if (*(iomodel->elements_type+index)==HutterApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,HutterApproximationEnum));
		}
		else if (*(iomodel->elements_type+index)==StokesApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,StokesApproximationEnum));
		}
		else if (*(iomodel->elements_type+index)==NoneApproximationEnum){
			this->inputs->AddInput(new IntInput(ApproximationEnum,NoneApproximationEnum));
		}
		else{
			ISSMERROR("Approximation type %s not supported yet",EnumToString((int)*(iomodel->elements_type+index)));
		}
	}

}
/*}}}*/
/*FUNCTION Penta::UpdateGeometry{{{1*/
void  Penta::UpdateGeometry(void){

	/*Intermediaries*/
	int i;
	double rho_ice,rho_water;

	/*Correct thickness*/
	this->inputs->ConstrainMin(ThicknessEnum,1.0);

	/*If shelf: hydrostatic equilibrium*/
	if (this->GetShelf()){

		/*recover material parameters: */
		rho_ice=matpar->GetRhoIce();
		rho_water=matpar->GetRhoWater();

		/*Create New Surface: s = (1-rho_ice/rho_water) h*/
		this->inputs->DuplicateInput(ThicknessEnum,SurfaceEnum);     //1: copy thickness into surface
		InputScale(SurfaceEnum,(1-rho_ice/rho_water)); //2: surface = surface * (1-di)

		/*Create New Bed b = -rho_ice/rho_water h*/
		this->inputs->DuplicateInput(ThicknessEnum,BedEnum);         //1: copy thickness into bed
		InputScale(BedEnum, -rho_ice/rho_water);       //2: bed = bed * (-di)
	}

	/*If sheet: surface = bed + thickness*/
	else{

		/*The bed does not change, update surface only s = b + h*/
		this->inputs->DuplicateInput(BedEnum,SurfaceEnum);          //1: copy bed into surface
		this->inputs->AXPY(SurfaceEnum,1.0,ThicknessEnum);     //2: surface = surface + 1 * thickness
	}

}
/*}}}*/

/*Penta specific routines: */
/*FUNCTION Penta::BedNormal {{{1*/
void Penta::BedNormal(double* bed_normal, double xyz_list[3][3]){

	int i;
	double v13[3];
	double v23[3];
	double normal[3];
	double normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	normal_norm=sqrt( pow(normal[0],2)+pow(normal[1],2)+pow(normal[2],2) );

	/*Bed normal is opposite to surface normal*/
	*(bed_normal)=-normal[0]/normal_norm;
	*(bed_normal+1)=-normal[1]/normal_norm;
	*(bed_normal+2)=-normal[2]/normal_norm;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixBalancedthickness {{{1*/
void  Penta::CreateKMatrixBalancedthickness(Mat Kgg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*flags: */
	bool onwater;
	bool onbed;

	/*recover some inputs: */
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg);
	delete tria->matice; delete tria;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	return;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixBalancedvelocities {{{1*/

void  Penta::CreateKMatrixBalancedvelocities(Mat Kgg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*flags: */
	bool onbed;
	bool onwater;

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg);
	delete tria->matice; delete tria;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	return;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixCouplingMacAyealPattyn{{{1*/
void Penta::CreateKMatrixCouplingMacAyealPattyn( Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    NUMVERTICESm=3;  //MacAyealNUMVERTICES
	const int    numdofm=2*NUMVERTICESm;
	const int    NUMVERTICESp=6; //Pattyn NUMVERTICES
	const int    numdofp=2*NUMVERTICESp;
	double       xyz_list[NUMVERTICESp][3];
	int*         doflistm=NULL;
	int*         doflistp=NULL;

	/* 3d gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;
	GaussTria  *gauss_tria=NULL;

	/* material data: */
	double viscosity; //viscosity
	double oldviscosity; //viscosity
	double newviscosity; //viscosity

	/* strain rate: */
	double epsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double oldepsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/

	/* matrices: */
	double B[3][numdofp];
	double Bprime[3][numdofm];
	double L[2][numdofp];
	double D[3][3]={0.0};            // material matrix, simple scalar matrix.
	double D_scalar;
	double DL[2][2]={0.0}; //for basal drag
	double DL_scalar;

	/* local element matrices: */
	double Ke_gg[numdofp][numdofm]={0.0}; //local element stiffness matrix 
	double Ke_gg_transp[numdofm][numdofp]={0.0}; //local element stiffness matrix 
	double Ke_gg_gaussian[numdofp][numdofm]; //stiffness matrix evaluated at the gaussian point.
	double Jdet;

	/*friction: */
	double  alpha2_list[3];
	double  alpha2;

	/*parameters: */
	double viscosity_overshoot;

	/*Collapsed formulation: */
	Tria*  tria     =NULL;
	Penta* pentabase=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;
	Input* vx_input=NULL;
	Input* vy_input=NULL;
	Input* vxold_input=NULL;
	Input* vyold_input=NULL;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&viscosity_overshoot,ViscosityOvershootEnum);

	/*If on water, skip stiffness: */
	if(onwater)return;

	/*Find penta on bed as pattyn must be coupled to the dofs on the bed: */
	pentabase=GetBasalElement();
	tria=pentabase->SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICESp);
	tria->GetDofList(&doflistm,MacAyealApproximationEnum);  //Pattyn dof list
	GetDofList(&doflistp,PattynApproximationEnum); //MacAyeal dof list

	/*Retrieve all inputs we will be needing: */
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);
	vxold_input=inputs->GetInput(VxOldEnum);
	vyold_input=inputs->GetInput(VyOldEnum);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(5,5);
	gauss_tria=new GaussTria();
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);
		gauss->SynchronizeGaussTria(gauss_tria);

		/*Get strain rate from velocity: */
		this->GetStrainRate3dPattyn(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input);
		this->GetStrainRate3dPattyn(&oldepsilon[0],&xyz_list[0][0],gauss,vxold_input,vyold_input);

		/*Get viscosity: */
		matice->GetViscosity3d(&viscosity, &epsilon[0]);
		matice->GetViscosity3d(&oldviscosity, &oldepsilon[0]);

		/*Get B and Bprime matrices: */
		GetBMacAyealPattyn(&B[0][0], &xyz_list[0][0], gauss);
		tria->GetBprimeMacAyeal(&Bprime[0][0], &xyz_list[0][0], gauss_tria);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/*Build the D matrix: we plug the gaussian weight, the viscosity, and the jacobian determinant 
		  onto this scalar matrix, so that we win some computational time: */

		newviscosity=viscosity+viscosity_overshoot*(viscosity-oldviscosity);
		D_scalar=2*newviscosity*gauss->weight*Jdet;
		for (i=0;i<3;i++) D[i][i]=D_scalar;

		/*  Do the triple product tB*D*Bprime: */
		TripleMultiply( &B[0][0],3,numdofp,1,
					&D[0][0],3,3,0,
					&Bprime[0][0],3,numdofm,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_gaussian onto Ke_gg: */
		for( i=0; i<numdofp; i++) for(j=0;j<numdofm; j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
	} 

	/*Add Ke_gg and its transpose to global matrix Kgg: */
	MatrixTranspose(&Ke_gg_transp[0][0],&Ke_gg[0][0],12,6);
	MatSetValues(Kgg,numdofp,doflistp,numdofm,doflistm,(const double*)Ke_gg,ADD_VALUES);
	MatSetValues(Kgg,numdofm,doflistm,numdofp,doflistp,(const double*)Ke_gg_transp,ADD_VALUES);

	//Deal with 2d friction at the bedrock interface
	if((onbed && !shelf)){

		/*Build a tria element using the 3 grids of the base of the penta. Then use 
		 * the tria functionality to build a friction stiffness matrix on these 3
		 * grids: */

		tria->CreateKMatrixCouplingMacAyealPattynFriction(Kgg);
	}

	delete tria->matice; delete tria;

	xfree((void**)&doflistm);
	xfree((void**)&doflistp);
	delete gauss;
	delete gauss_tria;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticHutter{{{1*/
void  Penta::CreateKMatrixDiagnosticHutter(Mat Kgg){

	/*Collapsed formulation: */
	int       i;
	const int numdofs=NDOF2*NUMVERTICES;
	int*      doflist=NULL;
	double    Ke_gg[numdofs][numdofs]={0.0};
	bool      onbed;
	bool      onsurface;
	int       connectivity[2];
	double    one0,one1;
	int       i0,i1,j0,j1;
	
	/*flags: */
	bool onwater;
	
	/*recover some inputs: */
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	
	/*If on water, skip: */
	if(onwater)return;

	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	GetDofList(&doflist);

	/*Spawn 3 beam elements: */
	for(i=0;i<3;i++){
		/*2 dofs of first node*/
		i0=2*i;
		i1=2*i+1;
		/*2 dofs of second node*/
		j0=2*(i+3);
		j1=2*(i+3)+1;

		/*Find connectivity for the two nodes*/
		connectivity[0]=nodes[i]->GetConnectivity();
		connectivity[1]=nodes[i+3]->GetConnectivity();
		one0=1/(double)connectivity[0];
		one1=1/(double)connectivity[1];

		/*Create matrix for these two nodes*/
		if (onbed && onsurface){
			Ke_gg[i0][i0]=one0;
			Ke_gg[i1][i1]=one0;
			Ke_gg[j0][i0]=-one1;
			Ke_gg[j0][j0]=one1;
			Ke_gg[j1][i1]=-one1;
			Ke_gg[j1][j1]=one1;
		}
		else if (onbed){
			Ke_gg[i0][i0]=one0;
			Ke_gg[i1][i1]=one0;
			Ke_gg[j0][i0]=-2*one1;
			Ke_gg[j0][j0]=2*one1;
			Ke_gg[j1][i1]=-2*one1;
			Ke_gg[j1][j1]=2*one1;
		}
		else if (onsurface){
			Ke_gg[j0][i0]=-one1;
			Ke_gg[j0][j0]=one1;
			Ke_gg[j1][i1]=-one1;
			Ke_gg[j1][j1]=one1;
		}
		else{ //node is on two horizontal layers and beams include the values only once, so the have to use half of the connectivity
			Ke_gg[j0][i0]=-2*one1;
			Ke_gg[j0][j0]=2*one1;
			Ke_gg[j1][i1]=-2*one1;
			Ke_gg[j1][j1]=2*one1;
		}
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdofs,doflist,numdofs,doflist,(const double*)Ke_gg,ADD_VALUES);

	/*Free ressources:*/
	xfree((void**)&doflist);

}
/*FUNCTION Penta::CreateKMatrixDiagnosticMacAyeal{{{1*/
void Penta::CreateKMatrixDiagnosticMacAyeal( Mat Kgg){
	
	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numdof2d=2*NUMVERTICES2D;
	double       xyz_list[NUMVERTICES][3];
	int*         doflist=NULL;

	/* 3d gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;
	GaussTria  *gauss_tria=NULL;

	/* material data: */
	double viscosity; //viscosity
	double oldviscosity; //viscosity
	double newviscosity; //viscosity

	/* strain rate: */
	double epsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double oldepsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/

	/* matrices: */
	double B[3][numdof2d];
	double Bprime[3][numdof2d];
	double L[2][numdof2d];
	double D[3][3]={0.0};            // material matrix, simple scalar matrix.
	double D_scalar;
	double DL[2][2]={0.0}; //for basal drag
	double DL_scalar;

	/* local element matrices: */
	double Ke_gg[numdof2d][numdof2d]={0.0}; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof2d][numdof2d]; //stiffness matrix evaluated at the gaussian point.
	double Jdet;

	/*slope: */
	double  slope[2]={0.0};
	double  slope_magnitude;

	/*friction: */
	double  alpha2_list[3];
	double  alpha2;

	double MAXSLOPE=.06; // 6 %
	double MOUNTAINKEXPONENT=10;

	/*parameters: */
	double viscosity_overshoot;

	/*Collapsed formulation: */
	Tria*  tria=NULL;
	Penta* pentabase=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;
	int  approximation;
	Input* vx_input=NULL;
	Input* vy_input=NULL;
	Input* vxold_input=NULL;
	Input* vyold_input=NULL;

	inputs->GetParameterValue(&approximation,ApproximationEnum);
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*If on water, skip stiffness: */
	if(approximation==MacAyealApproximationEnum){
		if(onwater)return;

		/*Figure out if this pentaelem is collapsed. If so, then bailout, except if it is at the 
		  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
		  the stiffness matrix. */

		if (onbed==0){
			/*This element should be collapsed, but this element is not on the bedrock, therefore all its 
			 * dofs have already been frozen! Do nothing: */
			return;
		}
		else if (onbed==1){

			/*This element should be collapsed into a tria element at its base. Create this tria element, 
			 *and use its CreateKMatrix functionality to fill the global stiffness matrix: */

			/*Depth Averaging B*/
			this->InputDepthAverageAtBase(RheologyBEnum,RheologyBbarEnum,MaterialsEnum);

			/*Call Tria function*/
			tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
			tria->CreateKMatrix(Kgg);
			delete tria->matice; delete tria;

			/*Delete B averaged*/
			this->matice->inputs->DeleteInput(RheologyBbarEnum);

			return;
		}
	}
	else if(approximation==MacAyealPattynApproximationEnum){
		/*retrieve some parameters: */
		this->parameters->FindParam(&viscosity_overshoot,ViscosityOvershootEnum);

		/*If on water, skip stiffness: */
		if(onwater)return;

		/*Find penta on bed as this is a macayeal elements: */
		pentabase=GetBasalElement();
		tria=pentabase->SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.

		/* Get node coordinates and dof list: */
		GetVerticesCoordinates(&xyz_list[0][0], nodes,NUMVERTICES);
		tria->GetDofList(&doflist,MacAyealApproximationEnum);  //Pattyn dof list

		/*Retrieve all inputs we will be needing: */
		vx_input=inputs->GetInput(VxEnum);
		vy_input=inputs->GetInput(VyEnum);
		vxold_input=inputs->GetInput(VxOldEnum);
		vyold_input=inputs->GetInput(VyOldEnum);

		/* Start  looping on the number of gaussian points: */
		gauss=new GaussPenta(5,5);
		gauss_tria=new GaussTria();
		for (ig=gauss->begin();ig<gauss->end();ig++){

			gauss->GaussPoint(ig);
			gauss->SynchronizeGaussTria(gauss_tria);

			/*Get strain rate from velocity: */
			this->GetStrainRate3dPattyn(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input);
			this->GetStrainRate3dPattyn(&oldepsilon[0],&xyz_list[0][0],gauss,vxold_input,vyold_input);

			/*Get viscosity: */
			matice->GetViscosity3d(&viscosity, &epsilon[0]);
			matice->GetViscosity3d(&oldviscosity, &oldepsilon[0]);

			/*Get B and Bprime matrices: */
			tria->GetBMacAyeal(&B[0][0], &xyz_list[0][0], gauss_tria);
			tria->GetBprimeMacAyeal(&Bprime[0][0], &xyz_list[0][0], gauss_tria);

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

			/*Build the D matrix: we plug the gaussian weight, the viscosity, and the jacobian determinant 
			  onto this scalar matrix, so that we win some computational time: */
			newviscosity=viscosity+viscosity_overshoot*(viscosity-oldviscosity);
			D_scalar=2*newviscosity*gauss->weight*Jdet;
			for (i=0;i<3;i++) D[i][i]=D_scalar;

			/*  Do the triple product tB*D*Bprime: */
			TripleMultiply( &B[0][0],3,numdof2d,1,
						&D[0][0],3,3,0,
						&Bprime[0][0],3,numdof2d,0,
						&Ke_gg_gaussian[0][0],0);

			/* Add the Ke_gg_gaussian, and optionally Ke_gg_gaussian onto Ke_gg: */
			for(i=0;i<numdof2d;i++) for(j=0;j<numdof2d;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
		}

		/*Add Ke_gg to global matrix Kgg: */
		MatSetValues(Kgg,numdof2d,doflist,numdof2d,doflist,(const double*)Ke_gg,ADD_VALUES);

		//Deal with 2d friction at the bedrock interface
		if((onbed && !shelf)){
			/*Build a tria element using the 3 grids of the base of the penta. Then use 
			 * the tria functionality to build a friction stiffness matrix on these 3
			 * grids: */

			tria->CreateKMatrixDiagnosticMacAyealFriction(Kgg);
		}

		/*Clean up and return*/
		delete tria->matice;
		delete tria;
		delete gauss_tria;
		delete gauss;
		xfree((void**)&doflist);
	}
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticPattyn{{{1*/
void Penta::CreateKMatrixDiagnosticPattyn( Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numdof=2*NUMVERTICES;
	double       xyz_list[NUMVERTICES][3];
	int*         doflist=NULL;

	/* 3d gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;

	/* material data: */
	double viscosity; //viscosity
	double oldviscosity; //viscosity
	double newviscosity; //viscosity

	/* strain rate: */
	double epsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double oldepsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/

	/* matrices: */
	double B[5][numdof];
	double Bprime[5][numdof];
	double L[2][numdof];
	double D[5][5]={0.0};            // material matrix, simple scalar matrix.
	double D_scalar;
	double DL[2][2]={0.0}; //for basal drag
	double DL_scalar;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]={0.0}; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.
	double Jdet;

	/*slope: */
	double  slope[2]={0.0};
	double  slope_magnitude;

	/*friction: */
	double  alpha2_list[3];
	double  alpha2;

	double MAXSLOPE=.06; // 6 %
	double MOUNTAINKEXPONENT=10;

	/*parameters: */
	double viscosity_overshoot;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;
	Input* vx_input=NULL;
	Input* vy_input=NULL;
	Input* vxold_input=NULL;
	Input* vyold_input=NULL;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&viscosity_overshoot,ViscosityOvershootEnum);

	/*If on water, skip stiffness: */
	if(onwater)return;

	/*Implement standard penta element: */

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList(&doflist,PattynApproximationEnum);

	/*Retrieve all inputs we will be needing: */
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);
	vxold_input=inputs->GetInput(VxOldEnum);
	vyold_input=inputs->GetInput(VyOldEnum);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(5,5);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Get strain rate from velocity: */
		this->GetStrainRate3dPattyn(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input);
		this->GetStrainRate3dPattyn(&oldepsilon[0],&xyz_list[0][0],gauss,vxold_input,vyold_input);

		/*Get viscosity: */
		matice->GetViscosity3d(&viscosity, &epsilon[0]);
		matice->GetViscosity3d(&oldviscosity, &oldepsilon[0]);

		/*Get B and Bprime matrices: */
		GetBPattyn(&B[0][0], &xyz_list[0][0], gauss);
		GetBprimePattyn(&Bprime[0][0], &xyz_list[0][0], gauss);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/*Build the D matrix: we plug the gaussian weight, the viscosity, and the jacobian determinant 
		  onto this scalar matrix, so that we win some computational time: */

		newviscosity=viscosity+viscosity_overshoot*(viscosity-oldviscosity);
		D_scalar=2*newviscosity*gauss->weight*Jdet;
		for (i=0;i<5;i++) D[i][i]=D_scalar;

		/*  Do the triple product tB*D*Bprime: */
		TripleMultiply( &B[0][0],5,numdof,1,
					&D[0][0],5,5,0,
					&Bprime[0][0],5,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_gaussian onto Ke_gg: */
		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
	}


	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	//Deal with 2d friction at the bedrock interface
	if((onbed && !shelf)){
		/*Build a tria element using the 3 grids of the base of the penta. Then use 
		 * the tria functionality to build a friction stiffness matrix on these 3
		 * grids: */

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreateKMatrixDiagnosticPattynFriction(Kgg);
		delete tria->matice; delete tria;
	}

	/*Clean up and return*/
	delete gauss;
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticStokes {{{1*/
void Penta::CreateKMatrixDiagnosticStokes( Mat Kgg){

	int i,j;
	const int numdof=NUMVERTICES*NDOF4;
	int*      doflist=NULL;

	const int numdof2d=NUMVERTICES2D*NDOF4;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*Grid data: */
	double     xyz_list[NUMVERTICES][3];
	double	  xyz_list_tria[NUMVERTICES2D][3];
	double	  bed_normal[3];

	/*matrices: */
	double     Ke_temp[27][27]={0.0}; //for the six nodes and the bubble 
	double     Ke_reduced[numdof][numdof]; //for the six nodes only
	double     Ke_gaussian[27][27];
	double     Ke_drag_gaussian[numdof2d][numdof2d];
	double     B[8][27];
	double     B_prime[8][27];
	double     LStokes[14][numdof2d];
	double     LprimeStokes[14][numdof2d];
	double     Jdet;
	double     Jdet2d;
	double     D[8][8]={0.0};
	double     D_scalar;
	double     DLStokes[14][14]={0.0};

	/* gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;
	GaussTria  *gauss_tria=NULL;


	/* specific gaussian point: */
	double  gauss_weight,area_gauss_weight,vert_gauss_weight;
	double  gauss_coord[4];
	double  gauss_coord_tria[3];
	int     area_order=5;
	int	  num_vert_gauss=5;

	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity;
	double  alpha2_gauss;
	Friction* friction=NULL;

	/*parameters: */
	double stokesreconditioning;
	int analysis_type;
	int approximation;

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;

	/*inputs: */
	Input* vx_input=NULL;
	Input* vy_input=NULL;
	Input* vz_input=NULL;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water or not Stokes, skip stiffness: */
	if(onwater || approximation!=StokesApproximationEnum) return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList(&doflist,StokesApproximationEnum);

	/* Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		get tria gaussian points as well as segment gaussian points. For tria gaussian 
		points, order of integration is 2, because we need to integrate the product tB*D*B' 
		which is a polynomial of degree 3 (see GaussLegendreTria for more details). For segment gaussian 
		points, same deal, which yields 3 gaussian points.*/

	/*Retrieve all inputs we will be needing: */
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);
	vz_input=inputs->GetInput(VzEnum);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(5,5);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Compute strain rate: */
		this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);

		/*Get viscosity: */
		matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

		/*Get B and Bprime matrices: */
		GetBStokes(&B[0][0],&xyz_list[0][0],gauss); 
		GetBprimeStokes(&B_prime[0][0],&xyz_list[0][0],gauss); 

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
		 * onto this scalar matrix, so that we win some computational time: */
		D_scalar=gauss->weight*Jdet;
		for (i=0;i<6;i++) D[i][i]=D_scalar*2*viscosity;
		for (i=6;i<8;i++) D[i][i]=-D_scalar*stokesreconditioning;

		/*  Do the triple product tB*D*Bprime: */
		TripleMultiply( &B[0][0],8,27,1,
					&D[0][0],8,8,0,
					&B_prime[0][0],8,27,0,
					&Ke_gaussian[0][0],0);

		/*Add Ke_gaussian and Ke_gaussian to terms in pKe. Watch out for column orientation from matlab: */
		for(i=0;i<27;i++) for(j=0;j<27;j++) Ke_temp[i][j]+=Ke_gaussian[i][j];
	}

	if((onbed==1) && (shelf==0)){

		/*build friction object, used later on: */
		friction=new Friction("3d",inputs,matpar,analysis_type);

		for(i=0;i<NUMVERTICES2D;i++){
			for(j=0;j<3;j++){
				xyz_list_tria[i][j]=xyz_list[i][j];
			}
		}

		/* Start  looping on the number of gaussian points: */
		delete gauss; //gauss of previous loop
		gauss=new GaussPenta();
		gauss->GaussFaceTria(0,1,2,2);
		gauss_tria=new GaussTria();
		for (ig=gauss->begin();ig<gauss->end();ig++){

			gauss->GaussPoint(ig);
			gauss->SynchronizeGaussTria(gauss_tria);

			/*Get the Jacobian determinant */
			tria->GetJacobianDeterminant3d(&Jdet2d, &xyz_list_tria[0][0], gauss_tria);

			/*Get L matrix if viscous basal drag present: */
			GetLStokes(&LStokes[0][0], gauss);
			GetLprimeStokes(&LprimeStokes[0][0], &xyz_list[0][0], gauss);

			/*Compute strain rate: */
			this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);

			/*Get viscosity at last iteration: */
			matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

			/*Get normal vecyor to the bed */
			BedNormal(&bed_normal[0],xyz_list_tria);

			/*Calculate DL on gauss point */
			friction->GetAlpha2(&alpha2_gauss, gauss,VxEnum,VyEnum,VzEnum);

			DLStokes[0][0]=alpha2_gauss*gauss->weight*Jdet2d;
			DLStokes[1][1]=alpha2_gauss*gauss->weight*Jdet2d;
			DLStokes[2][2]=-alpha2_gauss*gauss->weight*Jdet2d*bed_normal[0]*bed_normal[2];
			DLStokes[3][3]=-alpha2_gauss*gauss->weight*Jdet2d*bed_normal[1]*bed_normal[2];
			DLStokes[4][4]=-alpha2_gauss*gauss->weight*Jdet2d*bed_normal[0]*bed_normal[2];
			DLStokes[5][5]=-alpha2_gauss*gauss->weight*Jdet2d*bed_normal[1]*bed_normal[2];
			DLStokes[6][6]=-2*viscosity*gauss->weight*Jdet2d*bed_normal[0];
			DLStokes[7][7]=-2*viscosity*gauss->weight*Jdet2d*bed_normal[1];
			DLStokes[8][8]=-2*viscosity*gauss->weight*Jdet2d*bed_normal[2];
			DLStokes[9][8]=-2*viscosity*gauss->weight*Jdet2d*bed_normal[0]/2.0;
			DLStokes[10][10]=-2*viscosity*gauss->weight*Jdet2d*bed_normal[1]/2.0;
			DLStokes[11][11]=stokesreconditioning*gauss->weight*Jdet2d*bed_normal[0];
			DLStokes[12][12]=stokesreconditioning*gauss->weight*Jdet2d*bed_normal[1];
			DLStokes[13][13]=stokesreconditioning*gauss->weight*Jdet2d*bed_normal[2];

			/*  Do the triple product tL*D*L: */
			TripleMultiply( &LStokes[0][0],14,numdof2d,1,
						&DLStokes[0][0],14,14,0,
						&LprimeStokes[0][0],14,numdof2d,0,
						&Ke_drag_gaussian[0][0],0);

			for(i=0;i<numdof2d;i++) for(j=0;j<numdof2d;j++) Ke_temp[i][j]+=Ke_drag_gaussian[i][j];
		}
	
		/*Free ressources:*/
		delete friction;

	} //if ( (onbed==1) && (shelf==0))

	/*Reduce the matrix */
	ReduceMatrixStokes(&Ke_reduced[0][0], &Ke_temp[0][0]);

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_reduced,ADD_VALUES);

	/*Free ressources:*/
	delete gauss;
	delete gauss_tria;
	xfree((void**)&doflist);

	return;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticVert {{{1*/
void Penta::CreateKMatrixDiagnosticVert( Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numdof=NDOF1*NUMVERTICES;
	double       xyz_list[NUMVERTICES][3];
	int*         doflist=NULL;

	/* 3d gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;

	/* matrices: */
	double  Ke_gg[numdof][numdof]={0.0};
	double  Ke_gg_gaussian[numdof][numdof];
	double  Jdet;
	double  B[NDOF1][NUMVERTICES];
	double  Bprime[NDOF1][NUMVERTICES];
	double  DL_scalar;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater; 
	bool onsurface;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	/*If on water, skip stiffness: */
	if(onwater)return;

	/*If this element  is on the surface, we have a dynamic boundary condition that applies, as a stiffness 
	 * matrix: */
	if(onsurface){
		tria=(Tria*)SpawnTria(3,4,5); //nodes 3,4 and 5 are on the surface
		tria->CreateKMatrixDiagnosticSurfaceVert(Kgg);
		delete tria->matice; delete tria;
	}

	/*Now, onto the formulation for the vertical velocity: */

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList(&doflist);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(2,2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Get B and Bprime matrices: */
		GetBVert(&B[0][0], &xyz_list[0][0], gauss);
		GetBprimeVert(&Bprime[0][0], &xyz_list[0][0], gauss);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		DL_scalar=gauss->weight*Jdet;

		/*  Do the triple product tB*D*Bprime: */
		TripleMultiply( &B[0][0],1,NUMVERTICES,1,
					&DL_scalar,1,1,0,
					&Bprime[0][0],1,NUMVERTICES,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
	} 

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	xfree((void**)&doflist);
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixMelting {{{1*/
void  Penta::CreateKMatrixMelting(Mat Kgg){

	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater)return;

	if (!onbed){
		return;
	}
	else{

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreateKMatrixMelting(Kgg);
		delete tria->matice; delete tria;
		return;
	}
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixPrognostic {{{1*/
void  Penta::CreateKMatrixPrognostic(Mat Kgg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg);
	delete tria->matice; delete tria;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	return;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixSlope {{{1*/
void  Penta::CreateKMatrixSlope(Mat Kgg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);


	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg);
	delete tria->matice; delete tria;
	return;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixThermal {{{1*/
void  Penta::CreateKMatrixThermal(Mat Kgg){

	/* local declarations */
	int i,j;
	int found=0;

	/* node data: */
	const int    numdof=NDOF1*NUMVERTICES;
	double       xyz_list[NUMVERTICES][3];
	int*         doflist=NULL;

	/* gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;

	double  K[2][2]={0.0};

	double  u,v,w;

	/*matrices: */
	double     K_terms[numdof][numdof]={0.0};
	double     Ke_gaussian_conduct[numdof][numdof];
	double     Ke_gaussian_advec[numdof][numdof];
	double     Ke_gaussian_artdiff[numdof][numdof];
	double     Ke_gaussian_transient[numdof][numdof];
	double     B[3][numdof];
	double     Bprime[3][numdof];
	double     B_conduct[3][numdof];
	double     B_advec[3][numdof];
	double     B_artdiff[2][numdof];
	double     Bprime_advec[3][numdof];
	double     L[numdof];
	double     D_scalar;
	double     D[3][3];
	double     l1l2l3[3];
	double     tl1l2l3D[3];
	double     tBD[3][numdof];
	double     tBD_conduct[3][numdof];
	double     tBD_advec[3][numdof];
	double     tBD_artdiff[3][numdof];
	double     tLD[numdof];

	double     Jdet;

	/*Material properties: */
	double     gravity,rho_ice,rho_water;
	double     heatcapacity,thermalconductivity;
	double     mixed_layer_capacity,thermal_exchange_velocity;

	/*parameters: */
	double dt,epsvel;
	bool   artdiff;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;
	Input* vx_input=NULL;
	Input* vy_input=NULL;
	Input* vz_input=NULL;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*If on water, skip: */
	if(onwater)return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList(&doflist);

	// /*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	heatcapacity=matpar->GetHeatCapacity();
	thermalconductivity=matpar->GetThermalConductivity();

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);
	this->parameters->FindParam(&artdiff,ArtDiffEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);

	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);
	vz_input=inputs->GetInput(VzEnum);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(2,2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/*Conduction: */

		/*Get B_conduct matrix: */
		GetBConduct(&B_conduct[0][0],&xyz_list[0][0],gauss); 

		/*Build D: */
		D_scalar=gauss->weight*Jdet*(thermalconductivity/(rho_ice*heatcapacity));

		if(dt) D_scalar=D_scalar*dt;

		D[0][0]=D_scalar; D[0][1]=0; D[0][2]=0;
		D[1][0]=0; D[1][1]=D_scalar; D[1][2]=0;
		D[2][0]=0; D[2][1]=0; D[2][2]=D_scalar;

		/*  Do the triple product B'*D*B: */
		MatrixMultiply(&B_conduct[0][0],3,numdof,1,&D[0][0],3,3,0,&tBD_conduct[0][0],0);
		MatrixMultiply(&tBD_conduct[0][0],numdof,3,0,&B_conduct[0][0],3,numdof,0,&Ke_gaussian_conduct[0][0],0);

		/*Advection: */

		/*Get B_advec and Bprime_advec matrices: */
		GetBAdvec(&B_advec[0][0],&xyz_list[0][0],gauss); 
		GetBprimeAdvec(&Bprime_advec[0][0],&xyz_list[0][0],gauss); 

		//Build the D matrix
		vx_input->GetParameterValue(&u, gauss);
		vy_input->GetParameterValue(&v, gauss);
		vz_input->GetParameterValue(&w, gauss);

		D_scalar=gauss->weight*Jdet;

		if(dt) D_scalar=D_scalar*dt;

		D[0][0]=D_scalar*u;D[0][1]=0;         D[0][2]=0;
		D[1][0]=0;         D[1][1]=D_scalar*v;D[1][2]=0;
		D[2][0]=0;         D[2][1]=0;         D[2][2]=D_scalar*w;

		/*  Do the triple product B'*D*Bprime: */
		MatrixMultiply(&B_advec[0][0],3,numdof,1,&D[0][0],3,3,0,&tBD_advec[0][0],0);
		MatrixMultiply(&tBD_advec[0][0],numdof,3,0,&Bprime_advec[0][0],3,numdof,0,&Ke_gaussian_advec[0][0],0);

		/*Transient: */
		if(dt){
			GetNodalFunctionsP1(&L[0], gauss);
			D_scalar=gauss->weight*Jdet;
			D_scalar=D_scalar;

			/*  Do the triple product L'*D*L: */
			MatrixMultiply(&L[0],numdof,1,0,&D_scalar,1,1,0,&tLD[0],0);
			MatrixMultiply(&tLD[0],numdof,1,0,&L[0],1,numdof,0,&Ke_gaussian_transient[0][0],0);
		}
		else{
			for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke_gaussian_transient[i][j]=0;
		}

		/*Artifficial diffusivity*/
		if(artdiff){
			/*Build K: */
			D_scalar=gauss->weight*Jdet/(pow(u,2)+pow(v,2)+epsvel);
			if(dt) D_scalar=D_scalar*dt;
			K[0][0]=D_scalar*pow(u,2);       K[0][1]=D_scalar*fabs(u)*fabs(v);
			K[1][0]=D_scalar*fabs(u)*fabs(v);K[1][1]=D_scalar*pow(v,2);

			/*Get B_artdiff: */
			GetBArtdiff(&B_artdiff[0][0],&xyz_list[0][0],gauss); 

			/*  Do the triple product B'*K*B: */
			MatrixMultiply(&B_artdiff[0][0],2,numdof,1,&K[0][0],2,2,0,&tBD_artdiff[0][0],0);
			MatrixMultiply(&tBD_artdiff[0][0],numdof,2,0,&B_artdiff[0][0],2,numdof,0,&Ke_gaussian_artdiff[0][0],0);
		}
		else{
			for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke_gaussian_artdiff[i][j]=0;
		}

		/*Add Ke_gaussian to pKe: */
		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) K_terms[i][j]+=Ke_gaussian_conduct[i][j]+Ke_gaussian_advec[i][j]+Ke_gaussian_transient[i][j]+Ke_gaussian_artdiff[i][j];
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)K_terms,ADD_VALUES);

	//Ice/ocean heat exchange flux on ice shelf base 
	if(onbed && shelf){

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreateKMatrixThermal(Kgg);
		delete tria->matice; delete tria;
	}
	
	/*Free ressources:*/
	delete gauss;
	xfree((void**)&doflist);

}
/*}}}*/
/*FUNCTION Penta::CreatePVectorAdjointMacAyeal{{{1*/
void  Penta::CreatePVectorAdjointMacAyeal(Vec p_g){

	int i;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater) return;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal(2d model) and not on bed) */
	if (!onbed){
		return;
	}
	else{ 

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute pe_g*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		tria->CreatePVectorAdjointHoriz(p_g);
		delete tria->matice; delete tria;
		return;
	}
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorAdjointPattyn{{{1*/
void  Penta::CreatePVectorAdjointPattyn(Vec p_g){

	int i;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onsurface;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	/*If on water, skip: */
	if(onwater) return;

	/*Bail out if this element if:
	 * -> Non MacAyeal and not on the surface
	 * -> MacAyeal(2d model) and not on bed) */
	if (!onsurface){
		return;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		tria->CreatePVectorAdjointHoriz(p_g);
		delete tria->matice; delete tria;
		return;
	}
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorBalancedthickness {{{1*/
void Penta::CreatePVectorBalancedthickness( Vec pg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*flags: */
	bool onbed;
	bool onwater;

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg);
	delete tria->matice; delete tria;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorBalancedvelocities {{{1*/
void Penta::CreatePVectorBalancedvelocities( Vec pg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*flags: */
	bool onbed;
	bool onwater;

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg);
	delete tria->matice; delete tria;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticHutter{{{1*/
void  Penta::CreatePVectorDiagnosticHutter(Vec pg){

	int i,j,k;
	
	const int numdofs=NDOF2*NUMVERTICES;
	int*      doflist=NULL;
	double    pe_g[numdofs]={0.0};
	double    xyz_list[NUMVERTICES][3];
	double    z_list[NUMVERTICES];
	double    z_segment[2];
	double    slope2,constant_part;
	double    gauss[numdofs][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	int       node0,node1;
	BeamRef*  beam=NULL;

	/*gaussian points: */
	int      num_gauss;
	double*  segment_gauss_coord=NULL;
	double   gauss_coord;
	double*  gauss_weights=NULL;
	double   gauss_weight;
	int      ig;
	double   slope[2];

	double   z_g;
	double   rho_ice,gravity,n,B;
	double   Jdet;
	double   ub,vb;
	double   surface,thickness;

	/*flags: */
	bool onwater;
	bool onbed;
	bool onsurface;
	int  connectivity[2];

	/*Inputs*/
	Input* thickness_input;
	Input* surface_input;
	Input* slopex_input;
	Input* slopey_input;

	/*recover some inputs: */
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*recover doflist: */
	GetDofList(&doflist);

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	/*recover parameters: */
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	n=matice->GetN();
	B=matice->GetB();

	//Initilaize beam
	beam=new BeamRef();

	//recover extra inputs
	thickness_input=inputs->GetInput(ThicknessEnum);
	surface_input=inputs->GetInput(SurfaceEnum);
	slopex_input=inputs->GetInput(SurfaceSlopeXEnum);
	slopey_input=inputs->GetInput(SurfaceSlopeYEnum);

	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	for(i=0;i<NUMVERTICES;i++)z_list[i]=xyz_list[i][2];

	//Get gaussian points and weights. order 2 since we have a product of 2 nodal functions
	num_gauss=3;
	gaussSegment(&segment_gauss_coord, &gauss_weights, num_gauss);

	/*Loop on the three segments*/
	for(i=0;i<3;i++){
		slopex_input->GetParameterValue(&slope[0], &gauss[i][0]);
		slopey_input->GetParameterValue(&slope[1], &gauss[i][0]);
		surface_input->GetParameterValue(&surface, &gauss[i][0]);
		thickness_input->GetParameterValue(&thickness, &gauss[i][0]);

		//compute slope2 slopex and slopey
		slope2=pow(slope[0],2)+pow(slope[1],2);

		//%compute constant_part
		constant_part=-2*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2));

		z_segment[0]=z_list[i];
		z_segment[1]=z_list[i+3];

		node0=i;
		node1=i+3;

		connectivity[0]=nodes[node0]->GetConnectivity();
		connectivity[1]=nodes[node1]->GetConnectivity();

		/*Loop on the Gauss points: */
		for(ig=0;ig<num_gauss;ig++){

			//Pick up the gaussian point and its weight:
			gauss_weight=gauss_weights[ig];
			gauss_coord=segment_gauss_coord[ig];

			beam->GetParameterValue(&z_g, &z_segment[0],gauss_coord);

			//Get Jacobian determinant:
			beam->GetJacobianDeterminant(&Jdet, &z_segment[0],gauss_coord);

			if (onsurface){
				for(j=0;j<NDOF2;j++){
					pe_g[2*node1+j]+=constant_part*pow((surface-z_g)/B,n)*slope[j]*Jdet*gauss_weight/(double)connectivity[1];
				}
			}
			else{//connectivity is too large, should take only half on it
				for(j=0;j<NDOF2;j++){
					pe_g[2*node1+j]+=constant_part*pow((surface-z_g)/B,n)*slope[j]*Jdet*gauss_weight*2/(double)connectivity[1];
				}
			}
		}

		//Deal with lower surface
		if (onbed){

			//compute ub
			constant_part=-1.58*pow((double)10.0,-(double)10.0)*rho_ice*gravity*thickness;
			ub=constant_part*slope[0];
			vb=constant_part*slope[1];

			//Add to pe: 
			pe_g[2*node0]+=ub/(double)connectivity[0];
			pe_g[2*node0+1]+=vb/(double)connectivity[0];
		}
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdofs,doflist,(const double*)pe_g,ADD_VALUES);

	/*Clean up */
	delete beam;
	xfree((void**)&gauss_weights);
	xfree((void**)&segment_gauss_coord);
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticMacAyeal{{{1*/
void  Penta::CreatePVectorDiagnosticMacAyeal(Vec pg){

	/*Spawning: */
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip load: */
	if(onwater)return;

	/*Figure out if this pentaelem is Macayeal. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the load vector. */

	if (onbed==0){
		/*This element should be collapsed, but this element is not on the bedrock, therefore all its 
		 * dofs have already been frozen! Do nothing: */
		return;
	}
	else if (onbed==1){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 *and use its CreatePVector functionality to return an elementary load vector: */
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreatePVector(pg);
		delete tria->matice; delete tria;
		return;
	}
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticPattyn{{{1*/
void Penta::CreatePVectorDiagnosticPattyn( Vec pg){

	int i,j;

	/* node data: */
	const int    numdof=NDOF2*NUMVERTICES;
	double       xyz_list[NUMVERTICES][3];
	int*         doflist=NULL;

	/* parameters: */
	double  slope[3]; //do not put 2! this goes into GetParameterDerivativeValue, which addresses slope[3] also!
	double  driving_stress_baseline;
	double  thickness;

	/* gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l6[6];

	/*element vector at the gaussian points: */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];

	/*inputs: */
	bool onwater;
	Input* surface_input=NULL;
	Input* thickness_input=NULL;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip load: */
	if(onwater)return;

	/*Implement standard penta element: */

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList(&doflist,PattynApproximationEnum);

	/*Retrieve all inputs we will be needing: */
	thickness_input=inputs->GetInput(ThicknessEnum);
	surface_input=inputs->GetInput(SurfaceEnum);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(2,3);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Compute thickness at gaussian point: */
		thickness_input->GetParameterValue(&thickness, gauss);

		/*Compute slope at gaussian point: */
		surface_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],gauss);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/*Get nodal functions: */
		GetNodalFunctionsP1(l1l6, gauss);

		/*Compute driving stress: */
		driving_stress_baseline=matpar->GetRhoIce()*matpar->GetG();

		/*Build pe_g_gaussian vector: */
		for (i=0;i<NUMVERTICES;i++){
			for (j=0;j<NDOF2;j++){
				pe_g_gaussian[i*NDOF2+j]=-driving_stress_baseline*slope[j]*Jdet*gauss->weight*l1l6[i];
			}
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	xfree((void**)&doflist);
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticStokes {{{1*/
void Penta::CreatePVectorDiagnosticStokes( Vec pg){

	/*indexing: */
	int i,j;
	const int numdof=NUMVERTICES*NDOF4;
	int       numdof2d=NUMVERTICES2D*NDOF4;
	int*      doflist=NULL;

	/*Material properties: */
	double         gravity,rho_ice,rho_water;

	/*parameters: */
	double		   xyz_list_tria[NUMVERTICES2D][3];
	double         xyz_list[NUMVERTICES][3];
	double		   bed_normal[3];
	double         bed;

	/* gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;
	GaussTria  *gauss_tria=NULL;

	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity;
	double  water_pressure;

	/*matrices: */
	double     Pe_temp[27]={0.0}; //for the six nodes and the bubble 
	double     Pe_gaussian[27]={0.0}; //for the six nodes and the bubble 
	double     Ke_temp[27][3]={0.0}; //for the six nodes and the bubble 
	double     Pe_reduced[numdof]; //for the six nodes only
	double     Ke_gaussian[27][3];
	double     l1l6[6]; //for the six nodes of the penta
	double     l1l7[7]; //for the six nodes and the bubble 
	double     B[8][27];
	double     B_prime[8][27];
	double     B_prime_bubble[8][3];
	double     Jdet;
	double     Jdet2d;
	double     D[8][8]={0.0};
	double     D_scalar;
	double     tBD[27][8];

	Tria*            tria=NULL;

	/*parameters: */
	double stokesreconditioning;

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;
	int  approximation;
	Input* vx_input=NULL;
	Input* vy_input=NULL;
	Input* vz_input=NULL;
	Input* bed_input=NULL;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);

	/*If on water or not Stokes, skip load: */
	if(onwater || approximation!=StokesApproximationEnum) return;

	/*If on water, skip load: */
	if(onwater)return;

	/*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList(&doflist,StokesApproximationEnum);

	/*Retrieve all inputs we will be needing: */
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);
	vz_input=inputs->GetInput(VzEnum);
	bed_input=inputs->GetInput(BedEnum);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(5,5);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Compute strain rate and viscosity: */
		this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);
		matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/* Get nodal functions */
		GetNodalFunctionsMINI(&l1l7[0], gauss);

		/* Build gaussian vector */
		for(i=0;i<NUMVERTICES+1;i++){
			Pe_gaussian[i*NDOF4+2]=-rho_ice*gravity*Jdet*gauss->weight*l1l7[i];
		}

		/*Add Pe_gaussian to terms in Pe_temp. Watch out for column orientation from matlab: */
		for(i=0;i<27;i++) Pe_temp[i]+=Pe_gaussian[i];

		/*Get B and Bprime matrices: */
		GetBStokes(&B[0][0],&xyz_list[0][0],gauss); 
		GetBprimeStokes(&B_prime[0][0],&xyz_list[0][0], gauss); 

		/*Get bubble part of Bprime */
		for(i=0;i<8;i++) for(j=0;j<3;j++) B_prime_bubble[i][j]=B_prime[i][j+24];

		/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
		 * onto this scalar matrix, so that we win some computational time: */
		D_scalar=gauss->weight*Jdet;
		for (i=0;i<6;i++) D[i][i]=D_scalar*2*viscosity;
		for (i=6;i<8;i++) D[i][i]=-D_scalar*stokesreconditioning;

		/*  Do the triple product tB*D*Bprime: */
		MatrixMultiply(&B[0][0],8,27,1,&D[0][0],8,8,0,&tBD[0][0],0);
		MatrixMultiply(&tBD[0][0],27,8,0,&B_prime_bubble[0][0],8,3,0,&Ke_gaussian[0][0],0);

		/*Add Ke_gaussian and Ke_gaussian to terms in pKe. Watch out for column orientation from matlab: */
		for(i=0;i<27;i++) for(j=0;j<3;j++) Ke_temp[i][j]+=Ke_gaussian[i][j];
	}

	/*Deal with 2d friction at the bedrock interface: */
	if ( (onbed==1) && (shelf==1)){

		for(i=0;i<NUMVERTICES2D;i++){
			for(j=0;j<3;j++){
				xyz_list_tria[i][j]=xyz_list[i][j];
			}
		}

		/* Start looping on the number of gauss 2d (nodes on the bedrock) */
		delete gauss; //gauss of previous loop
		gauss=new GaussPenta();
		gauss->GaussFaceTria(0,1,2,2);
		gauss_tria=new GaussTria();
		for (ig=gauss->begin();ig<gauss->end();ig++){

			gauss->GaussPoint(ig);
			gauss->SynchronizeGaussTria(gauss_tria);

			/*Get the Jacobian determinant */
			tria->GetJacobianDeterminant3d(&Jdet2d, &xyz_list_tria[0][0], gauss_tria);

			/* Get bed at gaussian point */
			bed_input->GetParameterValue(&bed, gauss);

			/*Get L matrix : */
			GetNodalFunctionsP1(l1l6, gauss);

			/*Get water_pressure at gaussian point */
			water_pressure=gravity*rho_water*bed;

			/*Get normal vecyor to the bed */
			BedNormal(&bed_normal[0],xyz_list_tria);

			for(i=0;i<NUMVERTICES2D;i++){
				for(j=0;j<3;j++){
					Pe_temp[i*NDOF4+j]+=water_pressure*gauss->weight*Jdet2d*l1l6[i]*bed_normal[j];
				}
			}
		}
	} //if ( (onbed==1) && (shelf==1))

	/*Reduce the matrix */
	ReduceVectorStokes(&Pe_reduced[0], &Ke_temp[0][0], &Pe_temp[0]);

	/*Add Pe_reduced to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)Pe_reduced,ADD_VALUES);

	/*Free ressources:*/
	delete gauss;
	delete gauss_tria;
	xfree((void**)&doflist);

}
/*}}}*/
/*FUNCTION Penta::CreatePVectorAdjointStokes{{{1*/
void  Penta::CreatePVectorAdjointStokes(Vec p_g){

	int i;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onsurface;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	/*If on water, skip: */
	if(onwater || !onsurface)return;

	/*Call Tria's function*/
	tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
	tria->CreatePVectorAdjointStokes(p_g);
	delete tria->matice; delete tria;
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticVert {{{1*/
void  Penta::CreatePVectorDiagnosticVert( Vec pg){

	int i;

	/* node data: */
	const int    numdof=NDOF1*NUMVERTICES;
	double       xyz_list[NUMVERTICES][3];
	int*         doflist=NULL;

	/* gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;

	/* Jacobian: */
	double Jdet;

	/*element vector at the gaussian points: */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];
	double l1l6[6];

	/*Spawning: */
	Tria* tria=NULL;

	/*input parameters for structural analysis (diagnostic): */
	double du[3];
	double dv[3];
	double dudx,dvdy;

	/*inputs: */
	bool onwater;
	bool onbed;
	Input* vx_input=NULL;
	Input* vy_input=NULL;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*If we are on the bedrock, spawn a tria on the bedrock, and use it to build the 
	 *diagnostic base vertical stifness: */
	if(onbed){
		tria=(Tria*)SpawnTria(0,1,2); //nodes 0, 1 and 2 are on the bedrock
		tria->CreatePVectorDiagnosticBaseVert(pg);
		delete tria->matice; delete tria;
	}

	/*Now, handle the standard penta element: */
	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList(&doflist);

	/*Retrieve all inputs we will be needing: */
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(2,2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Get velocity derivative, with respect to x and y: */
		vx_input->GetParameterDerivativeValue(&du[0],&xyz_list[0][0],gauss);
		vy_input->GetParameterDerivativeValue(&dv[0],&xyz_list[0][0],gauss);
		dudx=du[0];
		dvdy=dv[1];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/*Get nodal functions: */
		GetNodalFunctionsP1(l1l6, gauss);

		/*Build pe_g_gaussian vector: */
		for (i=0;i<NUMVERTICES;i++){
			pe_g_gaussian[i]=(dudx+dvdy)*Jdet*gauss->weight*l1l6[i];
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	xfree((void**)&doflist);
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorMelting {{{1*/
void Penta::CreatePVectorMelting( Vec pg){
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorPrognostic {{{1*/
void Penta::CreatePVectorPrognostic( Vec pg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	
	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg);
	delete tria->matice; delete tria;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorSlope {{{1*/
void Penta::CreatePVectorSlope( Vec pg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg);
	delete tria->matice; delete tria;
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorThermal {{{1*/
void Penta::CreatePVectorThermal( Vec pg){

	/*indexing: */
	int i,j;
	int found=0;

	const int  numdof=NUMVERTICES*NDOF1;
	int*       doflist=NULL;

	/*Grid data: */
	double        xyz_list[NUMVERTICES][3];

	/* gaussian points: */
	int     ig;
	GaussPenta *gauss=NULL;

	double temperature_list[NUMVERTICES];
	double temperature;

	/*Material properties: */
	double gravity,rho_ice,rho_water;
	double mixed_layer_capacity,heatcapacity;
	double beta,meltingpoint,thermal_exchange_velocity;

	/* element parameters: */
	int    friction_type;

	/*matrices: */
	double P_terms[numdof]={0.0};
	double L[numdof];
	double l1l2l3[3];
	double basalfriction;
	double epsilon[6];
	double epsilon_sqr[3][3];
	double epsilon_matrix[3][3];

	double Jdet;
	double viscosity;
	double epsilon_eff;
	double phi;
	double t_pmp;
	double scalar;
	double scalar_def;
	double scalar_ocean;
	double scalar_transient;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*parameters: */
	double dt;

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;
	Input* vx_input=NULL;
	Input* vy_input=NULL;
	Input* vz_input=NULL;
	Input* temperature_input=NULL;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/*If on water, skip: */
	if(onwater)return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList(&doflist);

	/*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	heatcapacity=matpar->GetHeatCapacity();
	beta=matpar->GetBeta();
	meltingpoint=matpar->GetMeltingPoint();

	/*Retrieve all inputs we will be needing: */
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);
	vz_input=inputs->GetInput(VzEnum);
	if (dt) temperature_input=inputs->GetInput(TemperatureEnum);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(2,3);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Compute strain rate and viscosity: */
		this->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);
		matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);

		/* Get nodal functions */
		GetNodalFunctionsP1(&L[0], gauss);

		/*Build deformational heating: */
		GetPhi(&phi, &epsilon[0], viscosity);

		/*Build pe_gaussian */
		scalar_def=phi/(rho_ice*heatcapacity)*Jdet*gauss->weight;
		if(dt) scalar_def=scalar_def*dt;

		for(i=0;i<NUMVERTICES;i++) P_terms[i]+=scalar_def*L[i];

		/* Build transient now */
		if(dt){
			temperature_input->GetParameterValue(&temperature, gauss);
			scalar_transient=temperature*Jdet*gauss->weight;
			for(i=0;i<NUMVERTICES;i++) P_terms[i]+=scalar_transient*L[i];
		}
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	/* Ice/ocean heat exchange flux on ice shelf base */
	if(onbed && shelf){

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreatePVectorThermalShelf(pg);
		delete tria->matice; delete tria;
	}

	/* Geothermal flux on ice sheet base and basal friction */
	if(onbed && !shelf){

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreatePVectorThermalSheet(pg);
		delete tria->matice; delete tria;
	}

	xfree((void**)&doflist);
	delete gauss;

}
/*}}}*/
/*FUNCTION Penta::GaussFromNode {{{1*/
double* Penta::GaussFromNode(Node* node){

	/*variable declaration*/
	int i,pos;
	double*  gauss=NULL;

	/*Allocate gauss*/
	gauss=(double*)xmalloc(4*sizeof(double));

	for(i=0;i<6;i++){
		if (node==nodes[i]){
			switch(i){
				case 0:
					gauss[0]=1.0; gauss[1]=0.0; gauss[2]=0.0; gauss[3]= -1.0;
					return gauss;
				case 1:
					gauss[0]=0.0; gauss[1]=1.0; gauss[2]=0.0; gauss[3]= -1.0;
					return gauss;
				case 2:
					gauss[0]=0.0; gauss[1]=0.0; gauss[2]=1.0; gauss[3]= -1.0;
					return gauss;
				case 3:
					gauss[0]=1.0; gauss[1]=0.0; gauss[2]=0.0; gauss[3]=1.0;
					return gauss;
				case 4:
					gauss[0]=0.0; gauss[1]=1.0; gauss[2]=0.0; gauss[3]=1.0;
					return gauss;
				case 5:
					gauss[0]=0.0; gauss[1]=0.0; gauss[2]=1.0; gauss[3]=1.0;
					return gauss;
			}
		}
	}

	/*output error if not found*/
	ISSMERROR("Node not found in Penta");
}
/*}}}*/
/*FUNCTION Penta::GetDofList {{{1*/
void  Penta::GetDofList(int** pdoflist,int approximation_enum){

	int i,j;
	int numberofdofs=0;
	int count=0;

	/*output: */
	int* doflist=NULL;

	/*First, figure out size of doflist: */
	for(i=0;i<6;i++){
		numberofdofs+=nodes[i]->GetNumberOfDofs(approximation_enum);
	}

	/*Allocate: */
	doflist=(int*)xmalloc(numberofdofs*sizeof(int));

	/*Populate: */
	count=0;
	for(i=0;i<6;i++){
		nodes[i]->GetDofList(doflist+count,approximation_enum);
		count+=nodes[i]->GetNumberOfDofs(approximation_enum);
	}

	/*Assign output pointers:*/
	*pdoflist=doflist;

}
/*}}}*/
/*FUNCTION Penta::GetDofList1 {{{1*/
void  Penta::GetDofList1(int* doflist){
	
	int i;

	for(i=0;i<6;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}
/*}}}*/
/*FUNCTION Penta::GetElementType {{{1*/
int Penta::GetElementType(){

	/*return PentaRef field*/
	return this->element_type;

}
/*}}}*/
/*FUNCTION Penta::GetParameterListOnVertices(double* pvalue,int enumtype) {{{1*/
void Penta::GetParameterListOnVertices(double* pvalue,int enumtype){

	/*Intermediaries*/
	double     value[NUMVERTICES];
	GaussPenta *gauss              = NULL;

	/*Recover input*/
	Input* input=inputs->GetInput(enumtype);
	if (!input) ISSMERROR("Input %s not found in element",EnumToString(enumtype));

	/*Checks in debugging mode*/
	ISSMASSERT(pvalue);

	/* Start looping on the number of vertices: */
	gauss=new GaussPenta();
	for (int iv=0;iv<NUMVERTICES;iv++){
		gauss->GaussVertex(iv);
		input->GetParameterValue(&pvalue[iv],gauss);
	}

	/*clean-up*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::GetParameterListOnVertices(double* pvalue,int enumtype,double defaultvalue) {{{1*/
void Penta::GetParameterListOnVertices(double* pvalue,int enumtype,double defaultvalue){

	/*Intermediaries*/
	double     value[NUMVERTICES];
	GaussPenta *gauss              = NULL;

	/*Recover input*/
	Input* input=inputs->GetInput(enumtype);

	/*Checks in debugging mode*/
	ISSMASSERT(pvalue);

	/* Start looping on the number of vertices: */
	if (input){
		gauss=new GaussPenta();
		for (int iv=0;iv<NUMVERTICES;iv++){
			gauss->GaussVertex(iv);
			input->GetParameterValue(&pvalue[iv],gauss);
		}
	}
	else{
		for (int iv=0;iv<NUMVERTICES;iv++) pvalue[iv]=defaultvalue;
	}

	/*clean-up*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::GetParameterValue(double* pvalue,Node* node,int enumtype) {{{1*/
void Penta::GetParameterValue(double* pvalue,Node* node,int enumtype){

	/*Output*/
	double value;

	/*Intermediaries*/
	const int numnodes=6;
	int       grid=-1;
	int       i;
	double gauss[numnodes][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	/*go through 3 nodes (all nodes for tria) and identify 1st and 2nd nodes: */
	ISSMASSERT(nodes);
	for(i=0;i<numnodes;i++){
		if (node==nodes[i]){
			grid=i;
			break;
		}
	}

	/*Check that the node has been found*/
	if (grid==-1) ISSMERROR("Node pointer not found in Penta's nodes");

	/*Get Parameter value on node*/
	inputs->GetParameterValue(pvalue,&gauss[grid][0],enumtype);
	return;

}
/*}}}*/
/*FUNCTION Penta::GetParameterValue(double* pvalue,Node* node1,Node* node2,double gauss_seg,Input* input_in) {{{1*/
void Penta::GetParameterValue(double* pvalue,Node* node1,Node* node2,double gauss_seg,Input* input_in){

	/*Output*/
	double value;

	/*Intermediaries*/
	const int numnodes=6;
	int       grid1=-1,grid2=-1;
	int       grid3;
	int       i;
	double    gauss_penta[numnodes];

	/*go through 6 nodes (all nodes for penta) and identify 1st and 2nd nodes: */
	ISSMASSERT(nodes);
	for(i=0;i<numnodes;i++){
		if (node1==nodes[i]) grid1=i;
		if (node2==nodes[i]) grid2=i;
	}

	/*Reverse grid1 and 2 if necessary*/
	if (grid1>grid2){

		/*Reverse grid1 and grid2*/
		grid3=grid1; grid1=grid2; grid2=grid3;

		/*Change segment gauss point (between -1 and +1)*/
		gauss_seg=-gauss_seg;
	}

	/*Build Penta Gauss point*/
	if (grid2<3){

		/*We are on the basal triangle*/
		gauss_penta[3]=-1.;

		if (grid1==0 && grid2==1){
			/*gauss_seg is between 0 and 1*/
			gauss_penta[0]=0.5*(1.-gauss_seg);
			gauss_penta[1]=1.-0.5*(1.-gauss_seg);
			gauss_penta[2]=0.;
		}
		else if (grid1==0 && grid2==2){
			/*gauss_seg is between 0 and 2*/
			gauss_penta[0]=0.5*(1.-gauss_seg);
			gauss_penta[1]=0.;
			gauss_penta[2]=1.-0.5*(1.-gauss_seg);
		}
		else if (grid1==1 && grid2==2){
			/*gauss_seg is between 1 and 2*/
			gauss_penta[0]=0.;
			gauss_penta[1]=0.5*(1.-gauss_seg);
			gauss_penta[2]=1.-0.5*(1.-gauss_seg);
		}
		else{
			ISSMERROR("The 2 nodes provided are either the same or did not match current Penta nodes");
		}
	}
	else if(grid1>2){

		/*We are on the surface triangle*/
		gauss_penta[3]=+1.;

		if (grid1==3 && grid2==4){
			/*gauss_seg is between 0 and 1*/
			gauss_penta[0]=0.5*(1.-gauss_seg);
			gauss_penta[1]=1.-0.5*(1.-gauss_seg);
			gauss_penta[2]=0.;
		}
		else if (grid1==3 && grid2==5){
			/*gauss_seg is between 0 and 2*/
			gauss_penta[0]=0.5*(1.-gauss_seg);
			gauss_penta[1]=0.;
			gauss_penta[2]=1.-0.5*(1.-gauss_seg);
		}
		else if (grid1==4 && grid2==5){
			/*gauss_seg is between 1 and 2*/
			gauss_penta[0]=0.;
			gauss_penta[1]=0.5*(1.-gauss_seg);
			gauss_penta[2]=1.-0.5*(1.-gauss_seg);
		}
		else{
			ISSMERROR("The 2 nodes provided are either the same or did not match current Penta nodes");
		}
	}
	else{
		ISSMERROR("vertical segments not implemented yet");
	}

	/*OK, now we can call input method*/
	input_in->GetParameterValue(&value, &gauss_penta[0]);

	/*Assign output pointers:*/
	*pvalue=value;
}
/*}}}*/
/*FUNCTION Penta::GetPhi {{{1*/
void Penta::GetPhi(double* phi, double*  epsilon, double viscosity){
	/*Compute deformational heating from epsilon and viscosity */

	double epsilon_matrix[3][3];
	double epsilon_eff;
	double epsilon_sqr[3][3];

	/* Build epsilon matrix */
	epsilon_matrix[0][0]=*(epsilon+0);
	epsilon_matrix[1][0]=*(epsilon+3);
	epsilon_matrix[2][0]=*(epsilon+4);
	epsilon_matrix[0][1]=*(epsilon+3);
	epsilon_matrix[1][1]=*(epsilon+1);
	epsilon_matrix[2][1]=*(epsilon+5);
	epsilon_matrix[0][2]=*(epsilon+4);
	epsilon_matrix[1][2]=*(epsilon+5);
	epsilon_matrix[2][2]=*(epsilon+2);

	/* Effective value of epsilon_matrix */
	epsilon_sqr[0][0]=pow(epsilon_matrix[0][0],2);
	epsilon_sqr[1][0]=pow(epsilon_matrix[1][0],2);
	epsilon_sqr[2][0]=pow(epsilon_matrix[2][0],2);
	epsilon_sqr[0][1]=pow(epsilon_matrix[0][1],2);
	epsilon_sqr[1][1]=pow(epsilon_matrix[1][1],2);
	epsilon_sqr[2][1]=pow(epsilon_matrix[2][1],2);
	epsilon_sqr[0][2]=pow(epsilon_matrix[0][2],2);
	epsilon_sqr[1][2]=pow(epsilon_matrix[1][2],2);
	epsilon_sqr[2][2]=pow(epsilon_matrix[2][2],2);
	epsilon_eff=1/pow(2,0.5)*pow((epsilon_sqr[0][0]+epsilon_sqr[0][1]+ epsilon_sqr[0][2]+ epsilon_sqr[1][0]+ epsilon_sqr[1][1]+ epsilon_sqr[1][2]+ epsilon_sqr[2][0]+ epsilon_sqr[2][1]+ epsilon_sqr[2][2]),0.5);

	/*Phi = Tr(sigma * eps) 
	 *    = Tr(sigma'* eps)
	 *    = 2 * eps_eff * sigma'_eff
	 *    = 4 * eps_eff ^2*/
	*phi=4*pow(epsilon_eff,2.0)*viscosity;
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticHoriz{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticHoriz(Vec solution){

	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	double       gauss[NUMVERTICES][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	int*         doflist=NULL;
	double       values[numdof];
	double       vx;
	double       vy;
	int          approximation;

	/*Get approximation enum and dof list: */
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If the element is a coupling, do nothing: every grid is also on an other elements 
	 * (as coupling is between MacAyeal and Pattyn) so the other element will take care of it*/
	if(approximation==MacAyealPattynApproximationEnum){
	 return;
	}
	else{
		GetDofList(&doflist,approximation);
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	for(i=0;i<NUMVERTICES;i++){

		/*Recover vx and vy*/
		inputs->GetParameterValue(&vx,&gauss[i][0],VxEnum);
		inputs->GetParameterValue(&vy,&gauss[i][0],VyEnum);
		values[i*numdofpervertex+0]=vx;
		values[i*numdofpervertex+1]=vy;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticHutter{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticHutter(Vec solution){

	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	double       gauss[NUMVERTICES][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	int*         doflist=NULL;
	double       values[numdof];
	double       vx;
	double       vy;

	/*Get dof list: */
	GetDofList(&doflist);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	for(i=0;i<NUMVERTICES;i++){

		/*Recover vx and vy*/
		inputs->GetParameterValue(&vx,&gauss[i][0],VxEnum);
		inputs->GetParameterValue(&vy,&gauss[i][0],VyEnum);
		values[i*numdofpervertex+0]=vx;
		values[i*numdofpervertex+1]=vy;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticVert{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticVert(Vec solution){

	int i;

	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*NUMVERTICES;
	double       gauss[NUMVERTICES][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	int*         doflist=NULL;
	double       values[numdof];
	double       vz;

	/*Get dof list: */
	GetDofList(&doflist);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	for(i=0;i<NUMVERTICES;i++){

		/*Recover vz */
		inputs->GetParameterValue(&vz,&gauss[i][0],VxEnum);
		values[i]=vz;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticStokes{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticStokes(Vec solution){

	int i;

	const int    numdofpervertex=4;
	const int    numdof=numdofpervertex*NUMVERTICES;
	double       gauss[NUMVERTICES][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	int*         doflist=NULL;
	double       values[numdof];
	double       vx,vy,vz,p;
	double       stokesreconditioning;

	/*Get dof list: */
	GetDofList(&doflist);

	/*Recondition pressure: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);

	/*Ok, we have vx vy vz and P in values, fill in vx vy vz P arrays: */
	/*P1 element only for now*/
	for(i=0;i<NUMVERTICES;i++){

		/*Recover vx and vy*/
		inputs->GetParameterValue(&vx,&gauss[i][0],VxEnum);
		inputs->GetParameterValue(&vy,&gauss[i][0],VyEnum);
		inputs->GetParameterValue(&vz,&gauss[i][0],VzEnum);
		inputs->GetParameterValue(&p ,&gauss[i][0],PressureEnum);
		values[i*numdofpervertex+0]=vx;
		values[i*numdofpervertex+1]=vy;
		values[i*numdofpervertex+2]=vz;
		values[i*numdofpervertex+3]=p/stokesreconditioning;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsThermal{{{1*/
void  Penta::GetSolutionFromInputsThermal(Vec solution){

	int i;

	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*NUMVERTICES;
	double       gauss[NUMVERTICES][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	int*         doflist=NULL;
	double       values[numdof];
	double       vz;


	/*Get dof list: */
	GetDofList(&doflist);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	for(i=0;i<NUMVERTICES;i++){

		/*Recover vz */
		inputs->GetParameterValue(&vz,&gauss[i][0],TemperatureEnum);
		values[i]=vz;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);
	
	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::GetStrainRate3dPattyn{{{1*/
void Penta::GetStrainRate3dPattyn(double* epsilon,double* xyz_list, double* gauss, Input* vx_input, Input* vy_input){
	/*Compute the 3d Blatter/PattynStrain Rate (5 components):
	 *
	 * epsilon=[exx eyy exy exz eyz]
	 *
	 * with exz=1/2 du/dz
	 *      eyz=1/2 dv/dz
	 *
	 * the contribution of vz is neglected
	 */

	int i;

	double epsilonvx[5];
	double epsilonvy[5];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input){
		ISSMERROR("Input missing. Here are the input pointers we have for vx: %p, vy: %p\n",vx_input,vy_input);
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vx_input->GetVxStrainRate3dPattyn(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate3dPattyn(epsilonvy,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<5;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i];

}
/*}}}*/
/*FUNCTION Penta::GetStrainRate3dPattyn{{{1*/
void Penta::GetStrainRate3dPattyn(double* epsilon,double* xyz_list, GaussPenta* gauss, Input* vx_input, Input* vy_input){
	/*Compute the 3d Blatter/PattynStrain Rate (5 components):
	 *
	 * epsilon=[exx eyy exy exz eyz]
	 *
	 * with exz=1/2 du/dz
	 *      eyz=1/2 dv/dz
	 *
	 * the contribution of vz is neglected
	 */

	int i;

	double epsilonvx[5];
	double epsilonvy[5];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input){
		ISSMERROR("Input missing. Here are the input pointers we have for vx: %p, vy: %p\n",vx_input,vy_input);
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vx_input->GetVxStrainRate3dPattyn(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate3dPattyn(epsilonvy,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<5;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i];

}
/*}}}*/
/*FUNCTION Penta::GetStrainRate3d{{{1*/
void Penta::GetStrainRate3d(double* epsilon,double* xyz_list, double* gauss, Input* vx_input, Input* vy_input, Input* vz_input){
	/*Compute the 3d Strain Rate (6 components):
	 *
	 * epsilon=[exx eyy ezz exy exz eyz]
	 */

	int i;

	double epsilonvx[6];
	double epsilonvy[6];
	double epsilonvz[6];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input || !vz_input){
		ISSMERROR("Input missing. Here are the input pointers we have for vx: %p, vy: %p, vz: %p\n",vx_input,vy_input,vz_input);
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vx_input->GetVxStrainRate3d(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate3d(epsilonvy,xyz_list,gauss);
	vz_input->GetVzStrainRate3d(epsilonvz,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<6;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i]+epsilonvz[i];

}
/*}}}*/
/*FUNCTION Penta::GetStrainRate3d{{{1*/
void Penta::GetStrainRate3d(double* epsilon,double* xyz_list, GaussPenta* gauss, Input* vx_input, Input* vy_input, Input* vz_input){
	/*Compute the 3d Strain Rate (6 components):
	 *
	 * epsilon=[exx eyy ezz exy exz eyz]
	 */

	int i;

	double epsilonvx[6];
	double epsilonvy[6];
	double epsilonvz[6];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input || !vz_input){
		ISSMERROR("Input missing. Here are the input pointers we have for vx: %p, vy: %p, vz: %p\n",vx_input,vy_input,vz_input);
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vx_input->GetVxStrainRate3d(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate3d(epsilonvy,xyz_list,gauss);
	vz_input->GetVzStrainRate3d(epsilonvz,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<6;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i]+epsilonvz[i];

}
/*}}}*/
/*FUNCTION Penta::GetBasalElement{{{1*/
Penta* Penta::GetBasalElement(void){

	/*Output*/
	Penta* penta=NULL;

	/*Go through all elements till the bed is reached*/
	penta=this;
	for(;;){

		/*Stop if we have reached the surface, else, take lower penta*/
		if (penta->IsOnBed()) break;

		/* get lower Penta*/
		penta=penta->GetLowerElement();
		ISSMASSERT(penta->Id()!=this->id);
	}

	/*return output*/
	return penta;
}
/*}}}*/
/*FUNCTION Penta::GetLowerElement{{{1*/
Penta* Penta::GetLowerElement(void){

	Penta* upper_penta=NULL;
	upper_penta=(Penta*)neighbors[0]; //first one (0) under, second one (1) above
	return upper_penta;

}
/*}}}*/
/*FUNCTION Penta::GetUpperElement{{{1*/
Penta* Penta::GetUpperElement(void){

	Penta* upper_penta=NULL;
	upper_penta=(Penta*)neighbors[1]; //first one under, second one above
	return upper_penta;

}
/*}}}*/
/*FUNCTION Penta::GradjB {{{1*/
void  Penta::GradjB(Vec gradient){

	int i;
	Tria* tria=NULL;
	TriaVertexInput* triavertexinput=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, skip: */
	if(onwater)return;

	if (approximation==MacAyealApproximationEnum){
		/*Bail out element if MacAyeal (2d) and not on bed*/
		if (!onbed) return;

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute gardj*/

		/*Depth Average B*/
		this->InputDepthAverageAtBase(RheologyBEnum,RheologyBbarEnum,MaterialsEnum);

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		tria->GradjB(gradient);
		delete tria->matice; delete tria;

		/*delete B average*/
		this->matice->inputs->DeleteInput(RheologyBbarEnum);
	}
	else{
		/*Gradient is computed on bed only (Bbar)*/
		if (!onbed) return;

		/*Depth Average B*/
		this->InputDepthAverageAtBase(RheologyBEnum,RheologyBbarEnum,MaterialsEnum);

		/*B is a 2d field, use MacAyeal(2d) gradient even if it is Stokes or Pattyn*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		tria->GradjB(gradient);
		delete tria->matice; delete tria;

		/*delete B average*/
		this->matice->inputs->DeleteInput(RheologyBbarEnum);
	}

}
/*}}}*/
/*FUNCTION Penta::GradjDrag {{{1*/
void  Penta::GradjDrag(Vec gradient){

	int i;
	Tria* tria=NULL;
	TriaVertexInput* triavertexinput=NULL;
	double temp_gradient[6]={0,0,0,0,0,0};

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;
	int approximation;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*If on shelf, skip: */
	if(shelf)return;

	/*Bail out if this element does not touch the bedrock: */
	if (!onbed) return;

	if (approximation==MacAyealApproximationEnum || approximation==PattynApproximationEnum){

		/*MacAyeal or Pattyn*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->GradjDrag(gradient);
		delete tria->matice; delete tria;
	}
	else if (approximation==StokesApproximationEnum){

		/*Stokes*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->GradjDragStokes(gradient);
		delete tria->matice; delete tria;
	}
	else if (approximation==NoneApproximationEnum){
		return;
	}
	else ISSMERROR("approximation %s not supported yet",EnumToString(approximation));


}
/*}}}*/
/*FUNCTION Penta::InputExtrude {{{1*/
void  Penta::InputExtrude(int enum_type,int object_type){

	bool   onbed;
	Penta *penta          = NULL;
	Input *original_input = NULL;

	/*recover parameters: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*Are we on the base, not on the surface?:*/
	if(onbed){

		/*OK, we are on bed. we will follow the steps:
		 * 1: find input and extrude it.
		 * 2: follow the upper element until we reach the surface
		 * 3: for each element, we will add a copy of the extruded input*/

		/*Step1: Extrude the original input: */
		if (object_type==ElementsEnum)
		 original_input=(Input*)this->inputs->GetInput(enum_type);
		else if (object_type==MaterialsEnum)
		 original_input=(Input*)matice->inputs->GetInput(enum_type);
		else
		 ISSMERROR("object of type %s not supported yet",EnumToString(object_type));
		if(!original_input) ISSMERROR("%s%s"," could not find input with enum:",EnumToString(enum_type));
		original_input->Extrude();

		/*Stop if there is only one layer of element*/
		if (this->IsOnSurface()) return;

		/*Step 2: this input has been extruded for this element, now follow the upper element*/
		penta=this;
		for(;;){

			/* get upper Penta*/
			penta=penta->GetUpperElement();
			ISSMASSERT(penta->Id()!=this->id);

			/*Add input of the basal element to penta->inputs*/
			Input* copy=NULL;
			copy=(Input*)original_input->copy();
			if (object_type==ElementsEnum)
			 penta->inputs->AddInput((Input*)copy);
			else if (object_type==MaterialsEnum)
			 penta->matice->inputs->AddInput((Input*)copy);
			else
			 ISSMERROR("object of type %s not supported yet",EnumToString(object_type));

			/*Stop if we have reached the surface*/
			if (penta->IsOnSurface()) break;

		}
	}

	return;
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticHoriz {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticHoriz(double* solution){

	/*Inputs*/
	bool onbed;
	int  approximation;

	/*Recover inputs*/
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&approximation,ApproximationEnum);

	/*MacAyeal, everything is done by the element on bed*/
	if (approximation==MacAyealApproximationEnum){
		if (!onbed){
			/*Do nothing. Element on bed will take care of it*/
			return;
		}
		else{
			InputUpdateFromSolutionDiagnosticMacAyeal(solution);
			return;
		}
	}
	else if (approximation==PattynApproximationEnum){
		InputUpdateFromSolutionDiagnosticPattyn(solution);
	}
	else if (approximation==MacAyealPattynApproximationEnum){
		InputUpdateFromSolutionDiagnosticMacAyealPattyn(solution);
	}
}

/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticMacAyeal {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticMacAyeal(double* solution){

	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vel[NUMVERTICES];
	int          dummy;
	double       pressure[NUMVERTICES];
	double       surface[NUMVERTICES];
	double       rho_ice,g;
	double       xyz_list[NUMVERTICES][3];
	double       gauss[NUMVERTICES][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	Input  *vz_input        = NULL;
	double *vz_ptr          = NULL;
	Penta  *penta          = NULL;

	/*OK, we are on bed. Now recover results*/

	/*Get dof list: */
	GetDofList(&doflist,MacAyealApproximationEnum);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays and extrude */
	for(i=0;i<3;i++){
		vx[i]  =values[i*numdofpervertex+0];
		vy[i]  =values[i*numdofpervertex+1];
		vx[i+3]=vx[i];
		vy[i+3]=vy[i];
	}

	/*Get parameters fro pressure computation*/
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();

	/*Start looping over all elements above current element and update all inputs*/
	penta=this;
	for(;;){

		/*Get node data: */
		GetVerticesCoordinates(&xyz_list[0][0],penta->nodes,NUMVERTICES);

		/*Now Compute vel*/
		vz_input=inputs->GetInput(VzEnum);
		if (vz_input){
			if (vz_input->Enum()!=PentaVertexInputEnum) ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumToString(vz_input->Enum()));
			vz_input->GetValuesPtr(&vz_ptr,&dummy);
			for(i=0;i<NUMVERTICES;i++) vz[i]=vz_ptr[i];
		}
		else{for(i=0;i<NUMVERTICES;i++) vz[i]=0.0;}
		for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

		/*Now compute pressure*/
		GetParameterListOnVertices(&surface[0],SurfaceEnum);
		for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);

		/*Now, we have to move the previous Vx and Vy inputs  to old 
		 * status, otherwise, we'll wipe them off: */
		penta->inputs->ChangeEnum(VxEnum,VxOldEnum);
		penta->inputs->ChangeEnum(VyEnum,VyOldEnum);
		penta->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

		/*Add vx and vy as inputs to the tria element: */
		penta->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
		penta->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
		penta->inputs->AddInput(new PentaVertexInput(VelEnum,vel));
		penta->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

		/*Stop if we have reached the surface*/
		if (penta->IsOnSurface()) break;

		/* get upper Penta*/
		penta=penta->GetUpperElement(); ISSMASSERT(penta->Id()!=this->id);
	}
	
	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticMacAyealPattyn {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticMacAyealPattyn(double* solution){

	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	const int    numdof2d=numdofpervertex*NUMVERTICES2D;
	int*         doflistp=NULL;
	int*         doflistm=NULL;
	double       macayeal_values[numdof];
	double       pattyn_values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vel[NUMVERTICES];
	int          dummy;
	double       pressure[NUMVERTICES];
	double       surface[NUMVERTICES];
	double       rho_ice,g;
	double       xyz_list[NUMVERTICES][3];
	double       gauss[NUMVERTICES][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	Input  *vz_input       = NULL;
	double *vz_ptr         = NULL;
	Penta  *penta          = NULL;

	/*OK, we have to add results of this element for pattyn 
	 * and results from the penta at base for macayeal. Now recover results*/
	penta=GetBasalElement();

	/*Get dof listof this element (pattyn dofs) and of the penta at base (macayeal dofs): */
	GetDofList(&doflistp,PattynApproximationEnum);
	penta->GetDofList(&doflistm,MacAyealApproximationEnum);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof2d;i++){
		pattyn_values[i]=solution[doflistp[i]];
		macayeal_values[i]=solution[doflistm[i]];
	}
	for(i=numdof2d;i<numdof;i++){
		pattyn_values[i]=solution[doflistp[i]];
		macayeal_values[i]=macayeal_values[i-numdof2d];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		vx[i]=macayeal_values[i*numdofpervertex+0]+pattyn_values[i*numdofpervertex+0];
		vy[i]=macayeal_values[i*numdofpervertex+1]+pattyn_values[i*numdofpervertex+1];
	}

	/*Get Vz*/
	vz_input=inputs->GetInput(VzEnum);
	if (vz_input){
		if (vz_input->Enum()!=PentaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumToString(vz_input->Enum()));
		}
		vz_input->GetValuesPtr(&vz_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vz[i]=vz_ptr[i];
	}
	else{
		for(i=0;i<NUMVERTICES;i++) vz[i]=0.0;
	}

	/*Now Compute vel*/
	for(i=0;i<NUMVERTICES;i++) {
		vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);
	}

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 3D, 
	 *so the pressure is just the pressure at the z elevation: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	GetParameterListOnVertices(&surface[0],SurfaceEnum);
	for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new PentaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflistp);
	xfree((void**)&doflistm);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticPattyn {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticPattyn(double* solution){
	
	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vel[NUMVERTICES];
	int          dummy;
	double       pressure[NUMVERTICES];
	double       surface[NUMVERTICES];
	double       rho_ice,g;
	double       xyz_list[NUMVERTICES][3];
	double       gauss[NUMVERTICES][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	
	Input  *vz_input        = NULL;
	double *vz_ptr          = NULL;

	/*Get dof list: */
	GetDofList(&doflist,PattynApproximationEnum);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		vx[i]=values[i*numdofpervertex+0];
		vy[i]=values[i*numdofpervertex+1];
	}

	/*Get Vz*/
	vz_input=inputs->GetInput(VzEnum);
	if (vz_input){
		if (vz_input->Enum()!=PentaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumToString(vz_input->Enum()));
		}
		vz_input->GetValuesPtr(&vz_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vz[i]=vz_ptr[i];
	}
	else{
		for(i=0;i<NUMVERTICES;i++) vz[i]=0.0;
	}

	/*Now Compute vel*/
	for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 3D, 
	 *so the pressure is just the pressure at the z elevation: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	GetParameterListOnVertices(&surface[0],SurfaceEnum);
	for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new PentaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflist);
}

/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticHutter {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticHutter(double* solution){
	
	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vel[NUMVERTICES];
	int          dummy;
	double       pressure[NUMVERTICES];
	double       surface[NUMVERTICES];
	double       rho_ice,g;
	double       xyz_list[NUMVERTICES][3];
	double       gauss[NUMVERTICES][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	
	Input*       vz_input=NULL;
	double*      vz_ptr=NULL;

	/*Get dof list: */
	GetDofList(&doflist);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		vx[i]=values[i*numdofpervertex+0];
		vy[i]=values[i*numdofpervertex+1];
	}

	/*Get Vz*/
	vz_input=inputs->GetInput(VzEnum);
	if (vz_input){
		if (vz_input->Enum()!=PentaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumToString(vz_input->Enum()));
		}
		vz_input->GetValuesPtr(&vz_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vz[i]=vz_ptr[i];
	}
	else{
		for(i=0;i<NUMVERTICES;i++) vz[i]=0.0;
	}

	/*Now Compute vel*/
	for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 3D, 
	 *so the pressure is just the pressure at the z elevation: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	GetParameterListOnVertices(&surface[0],SurfaceEnum);
	for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new TriaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflist);
}

/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticVert {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticVert(double* solution){

	int i;

	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vel[NUMVERTICES];
	int          dummy;
	double       pressure[NUMVERTICES];
	double       surface[NUMVERTICES];
	double       rho_ice,g;
	double       xyz_list[NUMVERTICES][3];
	double       gauss[NUMVERTICES][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	Input*       vx_input=NULL;
	double*      vx_ptr=NULL;
	Input*       vy_input=NULL;
	double*      vy_ptr=NULL;

	/*Get dof list: */
	GetDofList(&doflist);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vz in values, fill in vz array: */
	for(i=0;i<NUMVERTICES;i++){
		vz[i]=values[i*numdofpervertex+0];
	}

	/*Get Vx and Vy*/
	vx_input=inputs->GetInput(VxEnum);
	if (vx_input){
		if (vx_input->Enum()!=PentaVertexInputEnum) ISSMERROR("Cannot compute Vel as Vx is of type %s",EnumToString(vx_input->Enum()));
		vx_input->GetValuesPtr(&vx_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vx[i]=vx_ptr[i];
	}
	else for(i=0;i<NUMVERTICES;i++) vx[i]=0.0;

	vy_input=inputs->GetInput(VyEnum);
	if (vy_input){
		if (vy_input->Enum()!=PentaVertexInputEnum) ISSMERROR("Cannot compute Vel as Vy is of type %s",EnumToString(vy_input->Enum()));
		vy_input->GetValuesPtr(&vy_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vy[i]=vy_ptr[i];
	}
	else for(i=0;i<NUMVERTICES;i++) vy[i]=0.0;

	/*Now Compute vel*/
	for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 3D, 
	 *so the pressure is just the pressure at the z elevation: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	GetParameterListOnVertices(&surface[0],SurfaceEnum);
	for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);

	/*Now, we have to move the previous Vz inputs to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VzEnum,VzOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vz and vel as inputs to the penta element: */
	this->inputs->AddInput(new PentaVertexInput(VzEnum,vz));
	this->inputs->AddInput(new PentaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticStokes {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticStokes(double* solution){
	
	int i;

	const int    numdofpervertex=4;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       vx[NUMVERTICES];
	double       vy[NUMVERTICES];
	double       vz[NUMVERTICES];
	double       vel[NUMVERTICES];
	double       pressure[NUMVERTICES];
	double       stokesreconditioning;

	/*Get dof list: */
	GetDofList(&doflist);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in all arrays: */
	for(i=0;i<NUMVERTICES;i++){
		vx[i]=values[i*numdofpervertex+0];
		vy[i]=values[i*numdofpervertex+1];
		vz[i]=values[i*numdofpervertex+2];
		pressure[i]=values[i*numdofpervertex+3];
	}

	/*Recondition pressure: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	for(i=0;i<NUMVERTICES;i++){
		pressure[i]=pressure[i]*stokesreconditioning;
	}

	/*Now Compute vel*/
	for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);
	
	/*Now, we have to move the previous inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(VzEnum,VzOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new PentaVertexInput(VzEnum,vz));
	this->inputs->AddInput(new PentaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflist);
}

/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionAdjointStokes {{{1*/
void  Penta::InputUpdateFromSolutionAdjointStokes(double* solution){

	int i;

	const int    numdofpervertex=4;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       lambdax[NUMVERTICES];
	double       lambday[NUMVERTICES];
	double       lambdaz[NUMVERTICES];
	double       lambdap[NUMVERTICES];

	/*Get dof list: */
	GetDofList(&doflist);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		lambdax[i]=values[i*numdofpervertex+0];
		lambday[i]=values[i*numdofpervertex+1];
		lambdaz[i]=values[i*numdofpervertex+2];
		lambdap[i]=values[i*numdofpervertex+3];
	}

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(AdjointxEnum,lambdax));
	this->inputs->AddInput(new PentaVertexInput(AdjointyEnum,lambday));
	this->inputs->AddInput(new PentaVertexInput(AdjointzEnum,lambdaz));
	this->inputs->AddInput(new PentaVertexInput(AdjointpEnum,lambdap));

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionAdjointHoriz {{{1*/
void  Penta::InputUpdateFromSolutionAdjointHoriz(double* solution){

	int i;

	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       lambdax[NUMVERTICES];
	double       lambday[NUMVERTICES];

	/*Get dof list: */
	GetDofList(&doflist);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		lambdax[i]=values[i*numdofpervertex+0];
		lambday[i]=values[i*numdofpervertex+1];
	}

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(AdjointxEnum,lambdax));
	this->inputs->AddInput(new PentaVertexInput(AdjointyEnum,lambday));

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionThermal {{{1*/
void  Penta::InputUpdateFromSolutionThermal(double* solution){

	int i;

	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*NUMVERTICES;
	int*         doflist=NULL;
	double       values[numdof];
	double       B[numdof];
	double       B_average;
	bool         converged;

	/*Get dof list: */
	GetDofList(&doflist);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	this->inputs->GetParameterValue(&converged,ConvergedEnum);
	if(converged){
		this->inputs->AddInput(new PentaVertexInput(TemperatureEnum,values));

		/*Update Rheology only if convreged (we must make sure that the temperature is below melting point
		 * otherwise the rheology could be negative*/
		B_average=Paterson((values[0]+values[1]+values[2]+values[3]+values[4]+values[5])/6.0);
		for(i=0;i<numdof;i++) B[i]=B_average;
		this->matice->inputs->AddInput(new PentaVertexInput(RheologyBEnum,B));
	}
	else{
		this->inputs->AddInput(new PentaVertexInput(TemporaryTemperatureEnum,values));
	}

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionOneDof{{{1*/
void  Penta::InputUpdateFromSolutionOneDof(double* solution,int enum_type){

	const int numdofpervertex = 1;
	const int numdof          = numdofpervertex *NUMVERTICES;
	int*         doflist=NULL;
	double    values[numdof];

	/*Get dof list: */
	GetDofList(&doflist);

	/*Use the dof list to index into the solution vector: */
	for(int i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Add input to the element: */
	this->inputs->AddInput(new PentaVertexInput(enum_type,values));
	
	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionOneDofCollpased{{{1*/
void  Penta::InputUpdateFromSolutionOneDofCollapsed(double* solution,int enum_type){

	const int  numdofpervertex = 1;
	const int  numdof          = numdofpervertex *NUMVERTICES;
	const int  numdof2d        = numdof/2;
	int*         doflist=NULL;
	double     values[numdof];
	Penta     *penta           = NULL;
	bool       onbed;

	/*Recover inputs*/
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If not on bed, return*/
	if (!onbed) return;

	/*Get dof list: */
	GetDofList(&doflist);

	/*Use the dof list to index into the solution vector and extrude it */
	for(int i=0;i<numdof2d;i++){
		values[i]         =solution[doflist[i]];
		values[i+numdof2d]=values[i];
	}

	/*Start looping over all elements above current element and update all inputs*/
	penta=this;
	for(;;){

		/*Add input to the element: */
		penta->inputs->AddInput(new PentaVertexInput(enum_type,values));

		/*Stop if we have reached the surface*/
		if (penta->IsOnSurface()) break;

		/* get upper Penta*/
		penta=penta->GetUpperElement(); ISSMASSERT(penta->Id()!=this->id);
	}
	
	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Penta::IsInput{{{1*/
bool Penta::IsInput(int name){
	if (
				name==ThicknessEnum ||
				name==SurfaceEnum ||
				name==BedEnum ||
				name==SurfaceSlopeXEnum ||
				name==SurfaceSlopeYEnum ||
				name==MeltingRateEnum ||
				name==AccumulationRateEnum ||
				name==GeothermalFluxEnum ||
				name==SurfaceAreaEnum||
				name==PressureEnum ||
				name==VxEnum ||
				name==VyEnum ||
				name==VzEnum ||
				name==VxObsEnum ||
				name==VyObsEnum ||
				name==VzObsEnum ||
				name==TemperatureEnum ||
				name==ControlParameterEnum ||
				name==CmResponseEnum ||
				name==DragCoefficientEnum ||
				name==GradientEnum ||
				name==OldGradientEnum  ||
				name==ConvergedEnum
				) {
		return true;
	}
	else return false;
}
/*}}}*/
/*FUNCTION Penta::IsOnSurface{{{1*/
bool Penta::IsOnSurface(void){

	bool onsurface;
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);
	return onsurface;

}
/*}}}*/
/*FUNCTION Penta::IsOnBed{{{1*/
bool Penta::IsOnBed(void){

	bool onbed;
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	return onbed;

}
/*}}}*/
/*FUNCTION Penta::ReduceMatrixStokes {{{1*/
void Penta::ReduceMatrixStokes(double* Ke_reduced, double* Ke_temp){

	int i,j;

	double Kii[24][24];
	double Kib[24][3];
	double Kbb[3][3];
	double Kbi[3][24];
	double Kbbinv[3][3];
	double KibKbbinv[24][3];
	double Kright[24][24];

	/*Create the four matrices used for reduction */
	for(i=0;i<24;i++){
		for(j=0;j<24;j++){
			Kii[i][j]=*(Ke_temp+27*i+j);
		}
		for(j=0;j<3;j++){
			Kib[i][j]=*(Ke_temp+27*i+j+24);
		}
	}
	for(i=0;i<3;i++){
		for(j=0;j<24;j++){
			Kbi[i][j]=*(Ke_temp+27*(i+24)+j);
		}
		for(j=0;j<3;j++){
			Kbb[i][j]=*(Ke_temp+27*(i+24)+j+24);
		}
	}

	/*Inverse the matrix corresponding to bubble part Kbb */
	Matrix3x3Invert(&Kbbinv[0][0], &Kbb[0][0]);

	/*Multiply matrices to create the reduce matrix Ke_reduced */
	MatrixMultiply(&Kib[0][0],24,3,0,&Kbbinv[0][0],3,3,0,&KibKbbinv[0][0],0);
	MatrixMultiply(&KibKbbinv[0][0],24,3,0,&Kbi[0][0],3,24,0,&Kright[0][0],0);

	/*Affect value to the reduced matrix */
	for(i=0;i<24;i++){
		for(j=0;j<24;j++){
			*(Ke_reduced+24*i+j)=Kii[i][j]-Kright[i][j];
		}
	}
}
/*}}}*/
/*FUNCTION Penta::ReduceVectorStokes {{{1*/
void Penta::ReduceVectorStokes(double* Pe_reduced, double* Ke_temp, double* Pe_temp){

	int i,j;

	double Pi[24];
	double Pb[3];
	double Kbb[3][3];
	double Kib[24][3];
	double Kbbinv[3][3];
	double KibKbbinv[24][3];
	double Pright[24];

	/*Create the four matrices used for reduction */
	for(i=0;i<24;i++){
		Pi[i]=*(Pe_temp+i);
	}
	for(i=0;i<3;i++){
		Pb[i]=*(Pe_temp+i+24);
	}
	for(j=0;j<3;j++){
		for(i=0;i<24;i++){
			Kib[i][j]=*(Ke_temp+3*i+j);
		}
		for(i=0;i<3;i++){
			Kbb[i][j]=*(Ke_temp+3*(i+24)+j);
		}
	}

	/*Inverse the matrix corresponding to bubble part Kbb */
	Matrix3x3Invert(&Kbbinv[0][0], &Kbb[0][0]);

	/*Multiply matrices to create the reduce matrix Ke_reduced */
	MatrixMultiply(&Kib[0][0],24,3,0,&Kbbinv[0][0],3,3,0,&KibKbbinv[0][0],0);
	MatrixMultiply(&KibKbbinv[0][0],24,3,0,&Pb[0],3,1,0,&Pright[0],0);

	/*Affect value to the reduced matrix */
	for(i=0;i<24;i++){
		*(Pe_reduced+i)=Pi[i]-Pright[i];
	}
}
/*}}}*/
/*FUNCTION Penta::SetClone {{{1*/
void  Penta::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
/*FUNCTION Penta::SpawnTria {{{1*/
Tria*  Penta::SpawnTria(int g0, int g1, int g2){

	int i;
	int analysis_counter;
	
	/*go into parameters and get the analysis_counter: */
	this->parameters->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*out of grids g0,g1 and g2 from Penta, build a tria element: */
	Tria* tria=NULL;
	int indices[3];
	int zero=0;
	Parameters* tria_parameters=NULL;
	Inputs* tria_inputs=NULL;
	Results* tria_results=NULL;

	indices[0]=g0;
	indices[1]=g1;
	indices[2]=g2;

	tria_parameters=this->parameters;
	tria_inputs=(Inputs*)this->inputs->SpawnTriaInputs(indices);
	tria_results=(Results*)this->results->SpawnTriaResults(indices);
	
	tria=new Tria();
	tria->id=this->id;
	tria->inputs=tria_inputs;
	tria->results=tria_results;
	tria->parameters=tria_parameters;
	tria->element_type=P1Enum; //Only P1 CG for now (TO BE CHANGED)
	this->SpawnTriaHook(dynamic_cast<TriaHook*>(tria),&indices[0]);

	/*Spawn matice*/
	tria->matice=NULL;
	tria->matice=(Matice*)this->matice->copy();
	delete tria->matice->inputs;
	tria->matice->inputs=(Inputs*)this->matice->inputs->SpawnTriaInputs(indices);

	/*recover nodes, matice and matpar: */
	tria->nodes=(Node**)tria->hnodes[analysis_counter]->deliverp();
	tria->matpar=(Matpar*)tria->hmatpar->delivers();
	
	return tria;
}
/*}}}*/
/*FUNCTION Penta::SurfaceNormal {{{1*/
void Penta::SurfaceNormal(double* surface_normal, double xyz_list[3][3]){

	int i;
	double v13[3];
	double v23[3];
	double normal[3];
	double normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	normal_norm=sqrt( pow(normal[0],2)+pow(normal[1],2)+pow(normal[2],2) );

	*(surface_normal)=normal[0]/normal_norm;
	*(surface_normal+1)=normal[1]/normal_norm;
	*(surface_normal+2)=normal[2]/normal_norm;

}
/*}}}*/
