/*!\file SystemMatricesx
 * \brief: create system matrices (stiffness matrix, loads vector)
 */

#include "./SystemMatricesx.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "../../EnumDefinitions/EnumDefinitions.h"

void SystemMatricesx(Mat* pKgg, Vec* ppg,double* pkmax,Elements* elements,Nodes* nodes, Vertices* vertices,Loads* loads,Materials* materials, Parameters* parameters,bool kflag,bool pflag,bool penalty_kflag,bool penalty_pflag){
	
	/*intermediary: */
	int      i,j,gsize;
	int      connectivity, numberofdofspernode;
	int      analysis_type, configuration_type;
	Element *element = NULL;
	Load    *load    = NULL;
	
	/*output: */
	Mat    Kgg  = NULL;
	Vec    pg   = NULL;
	double kmax = 0;

	/*Display message*/
	int verbose; parameters->FindParam(&verbose,VerboseEnum);
	if (verbose) _printf_("   Generating matrices\n");

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	parameters->FindParam(&connectivity,ConnectivityEnum);

	/*Get size of matrix: */
	gsize=nodes->NumberOfDofs(configuration_type);
	numberofdofspernode=nodes->MaxNumDofs(configuration_type);

	/*Checks in debugging mode {{{1*/
	ISSMASSERT(*pKgg==NULL);
	ISSMASSERT(*ppg==NULL);
	if(penalty_kflag)ISSMASSERT(kflag);
	if(penalty_pflag)ISSMASSERT(pflag);
	/*}}}*/

	/*Compute penalty free mstiffness matrix and load vector*/
	if(kflag){

		Kgg=NewMat(gsize,gsize,NULL,&connectivity,&numberofdofspernode);

		/*Fill stiffness matrix from elements: */
		for (i=0;i<elements->Size();i++){
			element=(Element*)elements->GetObjectByOffset(i);
			element->CreateKMatrix(Kgg);
		}

		/*Fill stiffness matrix from loads if loads have the current configuration_type: */
		for (i=0;i<loads->Size();i++){
			load=(Load*)loads->GetObjectByOffset(i);
			if (load->InAnalysis(configuration_type)) load->CreateKMatrix(Kgg);
		}

		/*Assemble matrix and compress matrix to save memory: */
		MatAssemblyBegin(Kgg,MAT_FINAL_ASSEMBLY);
		MatAssemblyEnd(Kgg,MAT_FINAL_ASSEMBLY);
		MatCompress(Kgg);
	}
	if(pflag){

		pg=NewVec(gsize);

		/*Fill right hand side vector, from elements: */
		for (i=0;i<elements->Size();i++){
			element=(Element*)elements->GetObjectByOffset(i);
			element->CreatePVector(pg);
		}

		/*Fill right hand side from loads if loads have the current configuration_type: */
		for (i=0;i<loads->Size();i++){
			load=(Load*)loads->GetObjectByOffset(i);
			if (load->InAnalysis(configuration_type)) load->CreatePVector(pg);
		}

		VecAssemblyBegin(pg);
		VecAssemblyEnd(pg);
	}

	/*Now, deal with penalties*/
	if(penalty_kflag){

		/*Now, figure out maximum value of K_gg, so that we can penalize it correctly: */
		MatNorm(Kgg,NORM_INFINITY,&kmax);

		/*Fill stiffness matrix from loads: */
		for (i=0;i<loads->Size();i++){
			load=(Load*)loads->GetObjectByOffset(i);
			if (load->InAnalysis(configuration_type)) load->PenaltyCreateKMatrix(Kgg,kmax);
		}

		/*Assemble matrix and compress matrix to save memory: */
		MatAssemblyBegin(Kgg,MAT_FINAL_ASSEMBLY);
		MatAssemblyEnd(Kgg,MAT_FINAL_ASSEMBLY);
		MatCompress(Kgg);
	}
	if(penalty_pflag){

		/*Fill right hand side vector, from loads: */
		for (i=0;i<loads->Size();i++){
			load=(Load*)loads->GetObjectByOffset(i);
			if (load->InAnalysis(configuration_type)) load->PenaltyCreatePVector(pg,kmax);
		}

		VecAssemblyBegin(pg);
		VecAssemblyEnd(pg);
	}
	
	/*Assign output pointers: */
	*pKgg=Kgg;
	*ppg=pg;
	if(pkmax) *pkmax=kmax;
}
