function femmodel=solver_diagnostic_nonlinear(femmodel,conserve_loads)
%SOLVER_DIAGNOSTIC_NONLINEAR - core solver of diagnostic run
%
%   Usage:
%      [femmodel]=solver_diagnostic_nonlinear(femmodel,conserve_loads)

	%Recover parameters
	min_mechanical_constraints=femmodel.parameters.MinMechanicalConstraints;

	%keep a copy of loads for now
	loads=femmodel.loads;

	%initialize solution vector
	converged=0; count=1;

	%Start non-linear iteration using input velocity: 
	ug=GetSolutionFromInputs(femmodel.elements, femmodel.nodes, femmodel.vertices, loads, femmodel.materials, femmodel.parameters);
	uf=Reducevectorgtof( ug, femmodel.nodesets);

	%Update the solution to make sure that vx and vxold are similar
	[femmodel.elements,femmodel.materials]=InputUpdateFromSolution(femmodel.elements,femmodel.nodes,femmodel.vertices,loads,femmodel.materials,femmodel.parameters,ug);

	while(~converged),
		
		%save pointer to old velocity
		old_ug=ug;
		old_uf=uf;
		
		[K_gg, p_g, kmax]=SystemMatrices(femmodel.elements,femmodel.nodes,femmodel.vertices,loads,femmodel.materials,femmodel.parameters);
		
		[K_ff, K_fs] = Reducematrixfromgtof( K_gg, femmodel.nodesets,femmodel.parameters); 
		p_f = Reduceloadfromgtof( p_g, K_fs, femmodel.ys, femmodel.nodesets,femmodel.parameters);

		uf=Solver(K_ff,p_f,old_uf,femmodel.parameters);

		ug= Mergesolutionfromftog( uf, femmodel.ys, femmodel.nodesets,femmodel.parameters); 

		[femmodel.elements,femmodel.materials]=InputUpdateFromSolution(femmodel.elements,femmodel.nodes,femmodel.vertices,loads,femmodel.materials,femmodel.parameters,ug);
		[loads,constraints_converged,num_unstable_constraints] =PenaltyConstraints( femmodel.elements,femmodel.nodes,femmodel.vertices,loads, femmodel.materials,femmodel.parameters);
		
		displaystring(femmodel.parameters.Verbose,'%s%i','      number of unstable constraints: ',num_unstable_constraints);

		%Figure out if convergence have been reached
		converged=convergence(K_ff,p_f,uf,old_uf,femmodel.parameters);
			
		%add convergence status into  status
		[femmodel.elements loads]=InputUpdateFromConstant(femmodel.elements,femmodel.nodes,femmodel.vertices,loads,femmodel.materials,femmodel.parameters,double(converged),ConvergedEnum);

		%rift convergence
		if ~constraints_converged,
			if converged,
				if num_unstable_constraints <= min_mechanical_constraints,
					converged=1;
				else 
					converged=0;
				end
			end
		end

		%increase count
		count=count+1;
		if(converged==1)break;
			if(count>max_nonlinear_iterations),
				displaystring(femmodel.parameters.Verbose,'%s%i%s','      maximum number of iterations ',max_nonlinear_iterations,' exceeded');
			end
		end
		
	end

	%deal with loads:
	if conserve_loads==false,
		femmodel.loads=loads;
	end
end
