/*!\file: solver_thermal_nonlinear.cpp
 * \brief: core of the thermal solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"

void solver_thermal_nonlinear(FemModel* femmodel){

	/*solution : */
	Vec tg=NULL; 
	Vec tf=NULL; 
	Vec tf_old=NULL; 
	double melting_offset;

	/*intermediary: */
	Mat Kgg=NULL;
	Mat Kgg_nopenalty=NULL;
	Mat Kff=NULL;
	Mat Kfs=NULL;
	Vec pg=NULL;
	Vec pg_nopenalty=NULL;
	Vec pf=NULL;

	int converged;
	int constraints_converged;
	int num_unstable_constraints;
	int count;
	int min_thermal_constraints;
	bool reset_penalties;

	/*parameters:*/
	int kflag,pflag;
	int verbose=0;
	bool lowmem=0;

	/*Recover parameters: */
	kflag=1; pflag=1;

	femmodel->parameters->FindParam(&verbose,VerboseEnum);
	femmodel->parameters->FindParam(&lowmem,LowmemEnum);
	femmodel->parameters->FindParam(&min_thermal_constraints,MinThermalConstraintsEnum);

	count=1;
	converged=0;

	for(;;){

		if(verbose)_printf_("%s\n","starting direct shooting method");

		if(count==1) reset_penalties=1; else reset_penalties=0;
		InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,reset_penalties,ResetPenaltiesEnum);

		//*Generate system matrices
		if (!lowmem){

			/*Compute Kgg_nopenalty and pg_nopenalty once for all: */
			if (count==1){
				SystemMatricesx(&Kgg_nopenalty, &pg_nopenalty,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,kflag,pflag);
			}

			/*Copy K_gg_nopenalty into Kgg, same for pg: */
			MatDuplicate(Kgg_nopenalty,MAT_COPY_VALUES,&Kgg);
			VecDuplicatePatch(&pg,pg_nopenalty);

			//apply penalties each time
			PenaltySystemMatricesx(Kgg, pg,&melting_offset,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,kflag,pflag);
		}
		else{
			SystemMatricesx(&Kgg, &pg,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,kflag,pflag);
			//apply penalties
			PenaltySystemMatricesx(Kgg, pg,&melting_offset,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,kflag,pflag);
		}

		Reducematrixfromgtofx(&Kff,&Kfs,Kgg,femmodel->nodesets,femmodel->parameters);

		/*Free ressources: */
		MatFree(&Kgg);
	
		Reduceloadfromgtofx(&pf, pg, Kfs, femmodel->ys, femmodel->nodesets,femmodel->parameters);

		//no need for pg and Kfs anymore 
		VecFree(&pg); 
		MatFree(&Kfs);

		/*Solve: */
		VecFree(&tf);
		Solverx(&tf, Kff, pf,tf_old, femmodel->parameters);
		VecFree(&tf_old); VecDuplicatePatch(&tf_old,tf);
	
		//no need for Kff and pf anymore
		MatFree(&Kff);VecFree(&pf);VecFree(&tg);

		Mergesolutionfromftogx(&tg, tf,femmodel->ys,femmodel->nodesets,femmodel->parameters);

		InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,tg);

		//Deal with penalty loads
		PenaltyConstraintsx(&constraints_converged, &num_unstable_constraints, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

		if (!converged){
			if(verbose)_printf_("%s%i\n","   #unstable constraints = ",num_unstable_constraints);
			if (num_unstable_constraints <= min_thermal_constraints)converged=1;
		}
		count++;
		
		if(converged==1)break;
	}

	//add melting_offset to inputs:
	InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,melting_offset,MeltingOffsetEnum);

	/*Free ressources: */
	MatFree(&Kgg_nopenalty);
	VecFree(&pg_nopenalty);
	VecFree(&tg);
	VecFree(&tf);
	VecFree(&tf_old);
}
