/*!\file: solver_linear.cpp
 * \brief: numerical core of linear solutions
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"

void solver_linear(FemModel* femmodel){

	/*parameters:*/
	int kflag,pflag;
	int verbose=0;

	/*output: */
	Vec ug=NULL;
	Vec uf=NULL; 
	
	/*intermediary: */
	Mat Kgg=NULL;
	Mat Kff=NULL;
	Mat Kfs=NULL;
	Vec pg=NULL;
	Vec pf=NULL;

	/*Recover parameters: */
	kflag=1; pflag=1;
	femmodel->parameters->FindParam(&verbose,VerboseEnum);

	//*Generate system matrices
	if(verbose)_printf_("   Generating matrices\n");
	SystemMatricesx(&Kgg, &pg,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,kflag,pflag);

	if(verbose)_printf_("   Generating penalty matrices\n");
	//*Generate penalty system matrices
	PenaltySystemMatricesx(Kgg, pg,NULL,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,kflag,pflag);

	/*!Reduce matrix from g to f size:*/
	if(verbose)_printf_("   reducing matrix from g to f set\n");
	Reducematrixfromgtofx(&Kff,&Kfs,Kgg,femmodel->nodesets); MatFree(&Kgg);

	/*!Reduce load from g to f size: */
	if(verbose)_printf_("   reducing load from g to f set\n");
	Reduceloadfromgtofx(&pf, pg, Kfs, femmodel->ys, femmodel->nodesets);VecFree(&pg); MatFree(&Kfs);

	/*Solve: */
	if(verbose)_printf_("   solving\n");
	Solverx(&uf, Kff, pf, NULL, femmodel->parameters); MatFree(&Kff); VecFree(&pf);

	//Merge back to g set
	if(verbose)_printf_("   merging solution from f to g set\n");
	Mergesolutionfromftogx(&ug, uf,femmodel->ys,femmodel->nodesets);VecFree(&uf);

	//Update inputs using new solution:
	InputUpdateFromSolutionx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug);

	/*free ressources: */
	VecFree(&ug);
	VecFree(&uf);
}
