/*!\file AverageOntoPartitionx
 * \brief: average vertex vector values onto a sub-partition of the vertices
 * used by scaled responses in Qmu analysis. See DakotaResponses module.
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./AverageOntoPartitionx.h"
#include "../../Container/Container.h"    
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../toolkits/toolkits.h"
#include "../modules.h"

			
void AverageOntoPartitionx(double** paverage, Elements* elements, Nodes* nodes, Vertices* vertices, Loads* loads, Materials* materials, Parameters* parameters,double* vertex_response,Vec vertex_cluster_partition){

	int     i,j,k;
	int     dummy;

	int     qmu_npart;
	double *qmu_part_nocluster = NULL;
	double *qmu_part  = NULL;
	double* partition=NULL; //serial version of vertex_cluster_partition
	int     numberofvertices;

	/*output: */
	Vec partition_contributions=NULL;
	Vec partition_areas=NULL;
	Vec vec_average=NULL;
	double* average=NULL;

	/*First, recover qmu partition of vertices. Careful, do not confuse with vertex_cluster_partition, which is just used to distribute vertices 
	 * onto cpus in a cluter: */
	if(!parameters->FindParam(&qmu_part_nocluster,&dummy,QmuPartEnum))ISSMERROR(" could not find qmu partition vector");

	/*Some parameters: */
	numberofvertices=vertices->NumberOfVertices();
	parameters->FindParam(&qmu_npart,QmuNPartEnum);

	/*serialize vertex_cluster_partition: */
	VecToMPISerial(&partition,vertex_cluster_partition);

	/*Use partition vector to repartition qmu_part_nocluster, which is ordered for use on a serial machine, not a cluster: */
	qmu_part=(double*)xmalloc(numberofvertices*sizeof(double));
	for(k=0;k<numberofvertices;k++) qmu_part[(int)(partition[k])]=qmu_part_nocluster[k];

	/*Ok, now we have a qmu partition (into separate areas) that takes into account the parallelism of the cluster. 
	 *We want to use this vector, and the vector of responses, to average onto the separate areas. The result will 
	 be a npart sized vector. */

	/*allocate: */
	partition_contributions=NewVec(qmu_npart);
	partition_areas=NewVec(qmu_npart);
	vec_average=NewVec(qmu_npart);

	/*loop on each element, and add contribution of the element to the partition (surface weighted average): */
	for(i=0;i<elements->Size();i++){
		Element* element=(Element*)elements->GetObjectByOffset(i);
		element->AverageOntoPartition(partition_contributions,partition_areas,vertex_response,qmu_part);
	}

	/*Assemble: */
	VecAssemblyBegin(partition_contributions);
	VecAssemblyEnd(partition_contributions);

	VecAssemblyBegin(partition_areas);
	VecAssemblyEnd(partition_areas);

	/*We have the partition_areas and the partition_contributions for each partition -> compute the surfae weighted average: */
	VecPointwiseDivide(vec_average,partition_contributions,partition_areas);

	/*serialize:*/
	VecToMPISerial(&average,vec_average);

	/*Free ressources:*/
	xfree((void**)&partition);
	xfree((void**)&qmu_part_nocluster);
	xfree((void**)&qmu_part);
	VecFree(&partition_contributions);
	VecFree(&partition_areas);
	VecFree(&vec_average);

	/*Assign output pointers:*/
	*paverage=average;
}
