function md=thermal(md)
%THERMAL - thermal solution sequence: steady state and transient
%
%   Usage:
%      md=thermal(md)

	%timing
	t1=clock;

	%Build all models requested for diagnostic simulation
	displaystring(md.debug,'%s',['reading thermal model data']);
	md.analysis_type='thermal'; m_t=CreateFemModel(md);

	displaystring(md.debug,'%s',['reading melting model data']);
	md.analysis_type='melting'; m_m=CreateFemModel(md);

	% figure out number of dof: just for information purposes.
	md.dof=m_t.nodesets.fsize;

	%initialize inputs
	displaystring(md.debug,'\n%s',['setup inputs...']);
	inputs=inputlist;
	inputs=add(inputs,'velocity',m_t.parameters.u_g,'doublevec',3,m_t.parameters.numberofnodes);
	inputs=add(inputs,'pressure',m_t.parameters.p_g,'doublevec',1,m_t.parameters.numberofnodes);
	inputs=add(inputs,'dt',m_t.parameters.dt,'double');
	
	if strcmpi(md.sub_analysis_type,'steady'),
	
		displaystring(md.debug,'\n%s',['computing temperatures...']);
		[t_g m_t.loads melting_offset]=thermal_core(m_t,inputs,'thermal','steady');
		
		displaystring(md.debug,'\n%s',['computing melting...']);
		inputs=add(inputs,'melting_offset',melting_offset,'double');
		inputs=add(inputs,'temperature',t_g,'doublevec',1,m_t.parameters.numberofnodes);
		melting_g=diagnostic_core_linear(m_m,inputs,'melting','steady');
		
		displaystring(md.debug,'\n%s',['load results...']);
		md.temperature=t_g;
		md.melting=melting_g*md.yts; %from m/s to m/a

	else

		%initialize temperature and melting
		t_g=m_t.parameters.t_g;
		melting_g=m_t.parameters.melting_g;
		nsteps=m_t.parameters.ndt/m_t.parameters.dt;

		%initialize temperature and melting
		soln.t_g=t_g;
		soln.melting_g=melting_g;
		soln.time=0;

		for n=1:nsteps, 

			displaystring(md.debug,'\n%s%i/%i\n','time step: ',n,nsteps);
			soln(n+1).time=n*m_t.parameters.dt;
			
			displaystring(md.debug,'\n%s',['    computing temperatures...']);
			inputs=add(inputs,'temperature',soln(n).t_g,'doublevec',1,m_t.parameters.numberofnodes);
			[soln(n+1).t_g m_t.loads melting_offset]=thermal_core(m_t,inputs,'thermal','transient');
			
			disp('   computing melting...');
			inputs=add(inputs,'temperature',soln(n).t_g,'doublevec',1,m_t.parameters.numberofnodes);
			inputs=add(inputs,'melting_offset',melting_offset,'double');
			soln(n+1).melting_g=diagnostic_core_linear(m_m,inputs,'melting','transient');
			
		end
		
		%Wrap up
		solution=struct('time',{},'temperature',{},'melting',{});
		for n=1:nsteps+1,
			solution(n).temperature=soln(n).t_g;
			solution(n).melting=soln(n).melting_g*md.yts; %in m/year
			solution(n).time=soln(n).time/md.yts;         %in years
		end
		md.thermaltransient_results=solution;
	end

	%stop timing
	t2=clock;
	displaystring(md.debug,'\n%s\n',['solution converged in ' num2str(etime(t2,t1)) ' seconds']);	
