#include <cstdio>
#include <cstring>
#include <cmath>
#include <ctime>

#include "Metric.h"

namespace bamg {

	/*Constructor*/
	/*FUNCTION MatVVP2x2::MatVVP2x2(const Metric M){{{1*/
	MatVVP2x2::MatVVP2x2(const Metric M){
		/*From a metric (a11,a21,a22), get eigen values lambda1 and lambda2 and one eigen vector v*/

		/*Intermediaries*/
		double a11=M.a11,a21=M.a21,a22=M.a22;
		double norm1,norm2,normM;
		double delta,b;

		/*To get the eigen values, we must solve the following equation:
		 *     | a11 - lambda    a21        |
		 * det |                            | = 0
		 *     | a21             a22-lambda |
		 *
		 * We have to solve the following polynom:
		 *  lamda^2 + ( -a11 -a22)*lambda + (a11*a22-a21*a21) = 0*/

		/*Compute polynom determinant*/
		b=-a11-a22;
		delta=b*b - 4*(a11*a22-a21*a21);


		/*Compute norm of M to avoid round off errors*/
		normM=a11*a11 + a22*a22 + a21*a21;

		/*1: normM too small: eigen values = 0*/
		if(normM<1.e-30){
			lambda1=0;
			lambda2=0;
			v.x=1;
			v.y=0;
		}
		/*2: delta is small -> double root*/
		else if (delta < 1.e-5*normM){
			lambda1=-b/2;
			lambda2=-b/2;
			v.x=1;
			v.y=0;
		}
		/*3: general case -> two roots*/
		else{
			delta     = sqrt(delta);
			lambda1   = (-b-delta)/2.0;
			lambda2   = (-b+delta)/2.0;

			/*Now, one must find the eigen vectors. For that we use the following property of the inner product
			 *    <Ax,y> = <x,tAy>
			 * Here, M'(M-lambda*Id) is symmetrical, which gives:
			 *    ∀(x,y)∈R²xR² <M'x,y> = <M'y,x>
			 * And we have the following:
			 *    if y∈Ker(M'), ∀x∈R² <M'x,y> = <x,M'y> = 0
			 * We have shown that
			 *    Im(M') ⊥ Ker(M')
			 *
			 * To find the eigen vectors of M, we only have to find two vectors
			 * of the image of M' and take their perpendicular as long as they are
			 * not 0.
			 * To do that, we take the images (1,0) and (0,1):
			 *  x1 = (a11 - lambda)      x2 = a21
			 *  y1 = a21                 y2 = (a22-lambda)
			 *
			 * We take the vector that has the larger norm and take its perpendicular.*/

			double norm1 = (a11-lambda1)*(a11-lambda1) + a21*a21; 
			double norm2 = a21*a21 + (a22-lambda1)*(a22-lambda1);

			if (norm2<norm1){
				norm1=sqrt(norm1);
				v.x = - a21/norm1;
				v.y = (a11-lambda1)/norm1;
			}
			else{
				norm2=sqrt(norm2);
				v.x = - (a22-lambda1)/norm2;
				v.y = a21/norm2;
			}
		}

	}
	/*}}}1*/
	/*FUNCTION MatVVP2x2::MatVVP2x2(double r1,double r2,const D2 vp1){{{1*/
	MatVVP2x2::MatVVP2x2(double r1,double r2,const D2 vp1): lambda1(r1),lambda2(r2),v(vp1){

	}/*}}}*/

	/*Methods*/
	/*FUNCTION MatVVP2x2::Abs{{{1*/
	void   MatVVP2x2::Abs(){
		lambda1=bamg::Abs(lambda1),lambda2=bamg::Abs(lambda2);
	}/*}}}*/
	/*FUNCTION MatVVP2x2::Aniso{{{1*/
	double MatVVP2x2::Aniso() const  { 
		return sqrt( Aniso2());
	}/*}}}*/
	/*FUNCTION MatVVP2x2::Aniso2{{{1*/
	double MatVVP2x2::Aniso2() const  { 
		return lmax()/lmin();
	}/*}}}*/
	/*FUNCTION MatVVP2x2::BoundAniso{{{1*/
	void   MatVVP2x2::BoundAniso(const double c){ 
		BoundAniso2(1/(c*c));
	}/*}}}*/
	/*FUNCTION MatVVP2x2::Echo {{{1*/
	void MatVVP2x2::Echo(void){

		printf("MatVVP2x2:\n");
		printf("   lambda1: %g\n",lambda1);
		printf("   lambda2: %g\n",lambda2);
		printf("   v.x: %g\n",v.x);
		printf("   v.y: %g\n",v.y);

		return;
	}
	/*}}}*/
	/*FUNCTION MatVVP2x2::hmin{{{1*/
	double MatVVP2x2::hmin() const {
		return sqrt(1/bamg::Max3(lambda1,lambda2,1e-30));
	}/*}}}*/
	/*FUNCTION MatVVP2x2::hmax{{{1*/
	double MatVVP2x2::hmax() const {
		return sqrt(1/bamg::Max(bamg::Min(lambda1,lambda2),1e-30));
	}/*}}}*/
	/*FUNCTION MatVVP2x2::Isotrope{{{1*/
	void   MatVVP2x2::Isotrope() {
		lambda1=lambda2=bamg::Max(lambda1,lambda2);
	}/*}}}*/
	/*FUNCTION MatVVP2x2::lmax{{{1*/
	double MatVVP2x2::lmax() const {
		return bamg::Max3(lambda1,lambda2,1e-30);
	}/*}}}*/
	/*FUNCTION MatVVP2x2::lmin{{{1*/
	double MatVVP2x2::lmin() const {
		return bamg::Max(bamg::Min(lambda1,lambda2),1e-30);
	}/*}}}*/
	/*FUNCTION MatVVP2x2::Min{{{1*/
	void   MatVVP2x2::Min(double a) { 
		lambda1=bamg::Min(a,lambda1); lambda2=bamg::Min(a,lambda2) ;
	}/*}}}*/
	/*FUNCTION MatVVP2x2::Max{{{1*/
	void   MatVVP2x2::Max(double a) { 
		lambda1=bamg::Max(a,lambda1); lambda2=bamg::Max(a,lambda2) ;
	}/*}}}*/
	/*FUNCTION MatVVP2x2::Minh{{{1*/
	void   MatVVP2x2::Minh(double h) {
		Min(1.0/(h*h));
	}/*}}}*/
	/*FUNCTION MatVVP2x2::Maxh{{{1*/
	void   MatVVP2x2::Maxh(double h) {
		Max(1.0/(h*h));
	}/*}}}*/
	/*FUNCTION MatVVP2x2::pow{{{1*/
	void   MatVVP2x2::pow(double p){
		lambda1=::pow(lambda1,p);lambda2=::pow(lambda2,p);
	}/*}}}*/

} 
