function md=setelementstype(md,varargin)
%SETELEMENTSTYPE - associate a solution type to each element
%
%   This routine works like plotmodel: it works with an even number of inputs
%   'hutter','macayeal','pattyn','stokes' and 'fill' are the possible options
%   that must be followed by the corresponding exp file or flags list
%   It can either be a domain file (argus type, .exp extension), or an array of element flags. 
%   If user wants every element outside the domain to be 
%   setelementstyped, add '~' to the name of the domain file (ex: '~Pattyn.exp');
%   an empty string '' will be considered as an empty domain
%   a string 'all' will be considered as the entire domain
%   You can specify the type of coupling, 'penalties' or 'tiling', to use with the input 'coupling'
%
%   Usage:
%      md=setelementstype(md,varargin)
%
%   Example:
%      md=setelementstype(md,'pattyn','Pattyn.exp','macayeal',md.elementoniceshelf,'fill','hutter');
%      md=setelementstype(md,'pattyn','Pattyn.exp',fill','hutter','coupling','tiling');

%some checks on list of arguments
if ((nargin<2) | (nargout~=1)),
	error('setelementstype error message');
end

if md.counter<3,
	error('only fully parameterized 2d models can be setelementstyped');
end

%Find_out what kind of coupling to use
options=pairoptions(varargin{:});
coupling_method=getfieldvalue(options,'coupling','tiling');
if (~strcmpi(coupling_method,'tiling') & ~strcmpi(coupling_method,'penalties')),
	error('coupling type can only be: tiling or penalties');
end

[hutterflag macayealflag pattynflag stokesflag filltype]=recover_areas(md,varargin{:});

%Flag the elements that has not been flagged as filltype
if strcmpi(filltype,'hutter'),
	hutterflag(find(~macayealflag & ~pattynflag))=1;
elseif strcmpi(filltype,'macayeal'),
	macayealflag(find(~hutterflag & ~pattynflag))=1;
elseif strcmpi(filltype,'pattyn'),
	pattynflag(find(~hutterflag & ~macayealflag))=1;
end

%check that each element has at least one flag
if any(hutterflag+ macayealflag+pattynflag==0),
	error('setelementstype error message: elements type not assigned, must be specified')
end

%check that each element has only one flag
if any(hutterflag+ macayealflag+pattynflag>1),
	disp('setelementstype warning message: some elements have several types, higher order type is used for them')
	hutterflag(find(hutterflag & macayealflag))=0;
	macayealflag(find(macayealflag & pattynflag))=0;
	pattynflag(find(pattynflag & macayealflag))=0;
end

%Check that no pattyn or stokes for 2d mesh
if (md.dim==2),
	if any(stokesflag | pattynflag)
		error('setelementstype error message: stokes and pattyn elements no allowed in 2d mesh, extrude it first')
	end
end

%Modify stokesflag to get rid of elements contrained everywhere
nonstokesflag=find(~stokesflag);                   %non stokes elements
gridonstokes=ones(md.numberofgrids,1);
gridonstokes(md.elements(nonstokesflag,:))=0;      %non stokes grids
pos=find(sum(md.spcvelocity(:,1:2),2)==2);         %find all the grids on the boundary of the domain without icefront
gridonstokes(pos)=0;                               %we have to constrain all the boundary of the domain without icefront

%Rule out elements where all grids are on the border of the stokes grids (because those elements are completely 
%constrained 
pos=find(stokesflag);
constrained_list=pos(find(sum(gridonstokes(md.elements(pos,:)),2)==0));
stokesflag(constrained_list)=0;

%add in model who is who
md.elements_type=zeros(md.numberofelements,2);

%1: Hutter elements
gridonhutter=zeros(md.numberofgrids,1);
gridonhutter(md.elements(find(hutterflag),:))=1;
md.gridonhutter=gridonhutter;
md.elements_type(find(hutterflag),1)=HutterApproximationEnum();

%2: MacAyeal elements
gridonmacayeal=zeros(md.numberofgrids,1);
gridonmacayeal(md.elements(find(macayealflag),:))=1;
md.gridonmacayeal=gridonmacayeal;
md.elements_type(find(macayealflag),1)=MacAyealApproximationEnum();

%3: Pattyn elements
gridonpattyn=zeros(md.numberofgrids,1);
gridonpattyn(md.elements(find(pattynflag),:))=1;
md.gridonpattyn=gridonpattyn;
md.elements_type(find(pattynflag),1)=PattynApproximationEnum();

%4: Stokes elements
md.gridonstokes=gridonstokes;
md.elements_type(find(stokesflag),2)=StokesApproximationEnum();

%5: None elements (non Stokes)
md.elements_type(find(~stokesflag),2)=NoneApproximationEnum();

%Now take care of the coupling between MacAyeal and Pattyn
md.penalties=[];
if strcmpi(coupling_method,'penalties'),
	gridonmacayealpattyn=zeros(md.numberofgrids,1);
	%Create the border grids between Pattyn and MacAyeal and extrude them
	numgrids2d=md.numberofgrids2d;
	numlayers=md.numlayers;
	bordergrids2d=find(gridonpattyn(1:numgrids2d) & gridonmacayeal(1:numgrids2d)); %Grids connected to two different types of elements

	%initialize and fill in penalties structure
	if ~isnan(bordergrids2d),
		penalties=[];
		for	i=1:numlayers-1,
			penalties=[penalties; [bordergrids2d bordergrids2d+md.numberofgrids2d*(i)]];
		end
		md.penalties=penalties;
	end
elseif strcmpi(coupling_method,'tiling'),
	%Find grid at the border
	gridonmacayealpattyn=zeros(md.numberofgrids,1);
	gridonmacayealpattyn(find(gridonmacayeal & gridonpattyn))=1;
	%Macayeal elements in contact with this layer become MacAyealPattyn elements
	matrixelements=ismember(md.elements,find(gridonmacayealpattyn));
	commonelements=sum(matrixelements,2)~=0;
	commonelements(find(pattynflag))=0; %only one layer: the elements previously in macayeal
	macayealflag(find(commonelements))=1; %these elements are now macayealpattynelements
	macayealpattynflag=zeros(md.numberofelements,1);
	macayealpattynflag(find(commonelements))=1;
	gridonmacayeal=zeros(md.numberofgrids,1);
	gridonmacayeal(md.elements(find(macayealflag),:))=1;
	md.gridonmacayeal=gridonmacayeal;

	%Create MacaAyealPattynApproximation where needed
	md.elements_type(find(macayealpattynflag),1)=MacAyealPattynApproximationEnum();

	%Now recreate gridonmacayeal and gridonmacayealpattyn
	gridonmacayeal(md.elements(find(macayealflag),:))=1;
	gridonmacayealpattyn(md.elements(find(macayealpattynflag),:))=1;
end

%Create vertices_type
md.vertices_type=zeros(md.numberofgrids,2);
pos=find(gridonhutter);
md.vertices_type(pos,1)=HutterApproximationEnum();
pos=find(gridonmacayeal);
md.vertices_type(pos,1)=MacAyealApproximationEnum();
pos=find(gridonpattyn);
md.vertices_type(pos,1)=PattynApproximationEnum();
pos=find(gridonhutter);
md.vertices_type(pos,1)=HutterApproximationEnum();
pos=find(gridonpattyn & gridonmacayeal);
md.vertices_type(pos,1)=PattynApproximationEnum();
pos=find(gridonmacayealpattyn);
md.vertices_type(pos,1)=MacAyealPattynApproximationEnum();
pos=find(gridonstokes);
md.vertices_type(pos,2)=StokesApproximationEnum();
pos=find(~gridonstokes);
md.vertices_type(pos,2)=NoneApproximationEnum();

%flag dead grids (strictly in MacAyeal and not on bed -> not used in diagnostic horiz)
nonmacayeal_el=find(~macayealflag);                   %non macayeal elements
deadgrids=ones(md.numberofgrids,1);
deadgrids(md.elements(nonmacayeal_el,:))=0;          %non macayeal grids are not dead
deadgrids(find(md.gridonbed))=0;                      %grids from elements on bed are not dead
md.deadgrids=deadgrids;

%figure out the border stokes grids
stokes_elements=find(md.elements_type(:,2)==StokesApproximationEnum()); %find the elements on the stokes domain
borderflags=zeros(md.numberofgrids,1); 
borderflags(md.elements(stokes_elements,:))=1; %find all the grids of the elements on stokes domain, ie stokes grids and borderstokes
md.borderstokes=borderflags-md.gridonstokes; %remove stokes grids from this list

%figure out solution types
md.ishutter=double(any(md.elements_type(:,1)==HutterApproximationEnum));
md.ismacayealpattyn=double(any(md.elements_type(:,1)==MacAyealApproximationEnum | md.elements_type(:,1)==PattynApproximationEnum));
md.isstokes=double(any(md.elements_type(:,2)==StokesApproximationEnum));

end
