function md=parametercontrolB(md,varargin),
%PARAMETERCONTROLB - parameterization for control method on B
%
%   It is possible to specify the number of steps, values for the
%   minimum and maximum values of B, specify a noise dampening, the 
%   kind of fit to use or the the optscal.
%   
%   Usage:
%       md=parametercontrolB(md,varargin)
%
%   Example:
%      md=parametercontrolB(md)
%      md=parametercontrolB(md,'nsteps',20,'cm_noisedmp',10^-15,'fit',0)
%      md=parametercontrolB(md,'cm_min',10,'cm_max',10^8,'cm_jump',0.99,'maxiter',20)
%      md=parametercontrolB(md,'cm_maxdmp_value',5*10^8,'cm_maxdmp_slope',10^-20,,'cm_mindmp_value',10^7,'cm_mindmp_slope',10^-17)
%      md=parametercontrolB(md,eps_cm',10^-4,'optscal',[10^7 10^8])
%
%   See also  PARAMETERCONTROLDRAG

%process options
options=pairoptions(varargin{:});

%control type
md.control_type=RheologyBbarEnum;

%weights
weights=getfieldvalue(options,'weights',ones(md.numberofgrids,1));
if (length(weights)~=md.numberofgrids)
	md.weights=ones(md.numberofgrids,1);
else
	md.weights=weights;
end

%nsteps
nsteps=getfieldvalue(options,'nsteps',100);
if (length(nsteps)~=1 | nsteps<=0 | floor(nsteps)~=nsteps)
	md.nsteps=100;
else
	md.nsteps=nsteps;
end

%cm_min
cm_min=getfieldvalue(options,'cm_min',1);
if (length(cm_min)~=1)
	md.cm_min=1;
else
	md.cm_min=cm_min;
end

%cm_max
cm_max=getfieldvalue(options,'cm_max',10^10);
if (length(cm_max)~=1)
	md.cm_max=10^10;
else
	md.cm_max=cm_max;
end

%cm_noisedmp
cm_noisedmp=getfieldvalue(options,'cm_noisedmp',1*10^-21);
if (length(cm_noisedmp)~=1)
	md.cm_noisedmp=1*10^-21;
else
	md.cm_noisedmp=cm_noisedmp;
end

%cm_maxdmp
cm_maxdmp_value=getfieldvalue(options,'cm_maxdmp_value',6.5*10^8);%=paterson(223)
if (length(cm_maxdmp_value)~=1)
	md.cm_maxdmp_value=6.5*10^8;
else
	md.cm_maxdmp_value=cm_maxdmp_value;
end
cm_maxdmp_slope=getfieldvalue(options,'cm_maxdmp_slope',5*10^-20);
if (length(cm_maxdmp_slope)~=1)
	md.cm_maxdmp_slope=5*10^-20;
else
	md.cm_maxdmp_slope=cm_maxdmp_slope;
end

%cm_mindmp
cm_mindmp_value=getfieldvalue(options,'cm_mindmp_value',5*10^7); %=paterson(273.15)
if (length(cm_mindmp_value)~=1)
	md.cm_mindmp_value=5*10^7;
else
	md.cm_mindmp_value=cm_mindmp_value;
end
cm_mindmp_slope=getfieldvalue(options,'cm_mindmp_slope',10^-18);
if (length(cm_mindmp_slope)~=1)
	md.cm_mindmp_slope=10^-18;
else
	md.cm_mindmp_slope=cm_mindmp_slope;
end

%eps_cm
eps_cm=getfieldvalue(options,'eps_cm',NaN);
if (length(eps_cm)~=1 | eps_cm<0 )
	md.eps_cm=NaN;
else
	md.eps_cm=eps_cm;
end

%maxiter
maxiter=getfieldvalue(options,'maxiter',10*ones(md.nsteps,1));
if (any(maxiter<0) | any(floor(maxiter)~=maxiter))
	md.maxiter=10*ones(md.nsteps,1);
else
	md.maxiter=repmat(maxiter(:),md.nsteps,1);
	md.maxiter(md.nsteps+1:end)=[];
end

%cm_jump
cm_jump=getfieldvalue(options,'cm_jump',0.9*ones(md.nsteps,1));
if ~isreal(cm_jump)
	md.cm_jump=0.9*ones(md.nsteps,1);
else
	md.cm_jump=repmat(cm_jump(:),md.nsteps,1);
	md.cm_jump(md.nsteps+1:end)=[];
end

%fit
found=0;
if exist(options,'fit'),
	fit=getfieldvalue(options,'fit');
	if ~any(fit~=0 & fit~=1 & fit~=2),
		md.fit=repmat(fit(:),md.nsteps,1);
		md.fit(md.nsteps+1:end)=[];
		found=1;
	end
end
if ~found
	third=ceil(md.nsteps/3);
	md.fit=[2*ones(third,1);0*ones(third,1);repmat([0;0;2;0],third,1)];
	md.fit(md.nsteps+1:end)=[];
end

%optscal
found=0;
if exist(options,'optscal'),
	optscal=getfieldvalue(options,'optscal');
	if ~any(optscal<0),
		md.optscal=repmat(optscal(:),md.nsteps,1);
		md.optscal(md.nsteps+1:end)=[];
		found=1;
	end
end
if ~found
	third=ceil(md.nsteps/3);
	md.optscal=[2*10^8*ones(3,1);10^8*ones(third-3,1);10^7*ones(2*third,1);];
	md.optscal(md.nsteps+1:end)=[];
end
