/*!\file Matice.c
 * \brief: implementation of the Matice object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
		
/*Matice constructors and destructor*/
/*FUNCTION Matice::Matice(){{{1*/
Matice::Matice(){
	this->inputs=NULL;
	this->helement=NULL;
	return;
}
/*}}}*/
/*FUNCTION Matice::Matice(int id, int index, IoModel* iomodel, int num_vertices){{{1*/
Matice::Matice(int matice_mid,int index, IoModel* iomodel){

	/*Intermediaries:*/
	int    i;
	int    matice_eid;

	/*Initialize id*/
	this->mid=matice_mid;

	/*Initialize inputs*/
	this->inputs=new Inputs();

	/*if 2d*/
	if(iomodel->dim==2){

		/*Intermediaries*/
		const int num_vertices = 3; //Tria has 3 vertices
		double    nodeinputs[num_vertices];

		/*Get B*/
		if (iomodel->rheology_B) {
			for(i=0;i<num_vertices;i++) nodeinputs[i]=iomodel->rheology_B[int(iomodel->elements[num_vertices*index+i]-1)];
			this->inputs->AddInput(new TriaVertexInput(RheologyB2dEnum,nodeinputs));
		}

		/*Get n*/
		if (iomodel->rheology_n) {
			for(i=0;i<num_vertices;i++) nodeinputs[i]=iomodel->rheology_n[index];
			this->inputs->AddInput(new TriaVertexInput(RheologyNEnum,nodeinputs));
		}

		/*Get control_parameter*/
		if (iomodel->control_parameter) {
			for(i=0;i<num_vertices;i++)nodeinputs[i]=iomodel->control_parameter[int(iomodel->elements[num_vertices*index+i]-1)];
			this->inputs->AddInput(new TriaVertexInput(ControlParameterEnum,nodeinputs));
		}
	}

	/*if 3d*/
	else if(iomodel->dim==3){

		/*Intermediaries*/
		const int num_vertices = 6; //Penta has 6 vertices
		double    nodeinputs[num_vertices];

		/*Get B*/
		if (iomodel->rheology_B) {
			for(i=0;i<num_vertices;i++) nodeinputs[i]=iomodel->rheology_B[int(iomodel->elements[num_vertices*index+i]-1)];
			this->inputs->AddInput(new PentaVertexInput(RheologyBEnum,nodeinputs));
		}

		/*Get n*/
		if (iomodel->rheology_n) {
			for(i=0;i<num_vertices;i++) nodeinputs[i]=iomodel->rheology_n[index];
			this->inputs->AddInput(new PentaVertexInput(RheologyNEnum,nodeinputs));
		}

		/*Get control_parameter*/
		if (iomodel->control_parameter) {
			for(i=0;i<num_vertices;i++)nodeinputs[i]=iomodel->control_parameter[int(iomodel->elements[num_vertices*index+i]-1)];
			this->inputs->AddInput(new PentaVertexInput(ControlParameterEnum,nodeinputs));
		}
	}
	/*Else*/
	else ISSMERROR(" Mesh type not supported yet!");

	/*Hooks: */
	matice_eid=index+1;
	this->helement=new Hook(&matice_eid,1);

	return;

}
/*}}}*/
/*FUNCTION Matice::~Matice(){{{1*/
Matice::~Matice(){
	delete helement;
	delete inputs;
	return;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION Matice::Echo {{{1*/
void Matice::Echo(void){

	printf("Matice:\n");
	printf("   mid: %i\n",mid);
	printf("   inputs:\n");
	inputs->Echo();
	printf("   element:\n");
	helement->Echo();
}
/*}}}*/
/*FUNCTION Matice::DeepEcho {{{1*/
void Matice::DeepEcho(void){

	printf("Matice:\n");
	printf("   mid: %i\n",mid);
	printf("   inputs:\n");
	inputs->DeepEcho();
	printf("   element:\n");
	helement->Echo();
}		
/*}}}*/
/*FUNCTION Matice::Id {{{1*/
int    Matice::Id(void){ return mid; }
/*}}}*/
/*FUNCTION Matice::MyRank {{{1*/
int    Matice::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Matice::Marshall {{{1*/
void  Matice::Marshall(char** pmarshalled_dataset){

	/*Intermediaries*/
	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Matice: */
	enum_type=MaticeEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Matice data: */
	memcpy(marshalled_dataset,&mid,sizeof(mid));marshalled_dataset+=sizeof(mid);

	/*Marshall hooks: */
	helement->Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	*pmarshalled_dataset=marshalled_dataset;

	/*clean up and return*/
	xfree((void**)&marshalled_inputs);
}
/*}}}*/
/*FUNCTION Matice::MarshallSize{{{1*/
int   Matice::MarshallSize(){

	return sizeof(mid)
	  +helement->MarshallSize()
	  +inputs->MarshallSize()
	  +sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Matice::Demarshall {{{1*/
void  Matice::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&mid,marshalled_dataset,sizeof(mid));marshalled_dataset+=sizeof(mid);

	/*demarshall hooks: */
	helement=new Hook(); helement->Demarshall(&marshalled_dataset);

	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Matice::Enum {{{1*/
int Matice::Enum(void){

	return MaticeEnum;

}
/*}}}*/
/*FUNCTION Matice::copy {{{1*/
Object* Matice::copy() {

	/*Output*/
	Matice* matice=NULL;

	/*Initialize output*/
	matice=new Matice();

	/*copy fields: */
	matice->mid=this->mid;
	matice->helement=(Hook*)this->helement->copy();
	if(this->inputs) matice->inputs=(Inputs*)this->inputs->Copy();
	else  matice->inputs=new Inputs();

	return matice;
}
/*}}}*/

/*Matice management*/
/*FUNCTION Matice::Configure {{{1*/
void  Matice::Configure(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	helement->configure(elementsin);
}
/*}}}*/
/*FUNCTION Matice::SetCurrentConfiguration {{{1*/
void  Matice::SetCurrentConfiguration(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

}
/*}}}*/
/*FUNCTION Matice::GetB {{{1*/
double Matice::GetB(){

	/*Output*/
	double B;

	inputs->GetParameterAverage(&B,RheologyBEnum);
	return B;
}
/*}}}*/
/*FUNCTION Matice::GetB2d {{{1*/
double Matice::GetB2d(){

	/*Output*/
	double B2d;

	inputs->GetParameterAverage(&B2d,RheologyB2dEnum);
	return B2d;
}
/*}}}*/
/*FUNCTION Matice::GetN {{{1*/
double Matice::GetN(){

	/*Output*/
	double n;

	inputs->GetParameterAverage(&n,RheologyNEnum);
	return n;
}
/*}}}*/
/*FUNCTION Matice::GetViscosity2d {{{1*/
void  Matice::GetViscosity2d(double* pviscosity, double* epsilon){
	/*From a string tensor and a material object, return viscosity, using Glen's flow law.
												    B
	  viscosity= -------------------------------------------------------------------
						  2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]

	  where viscosity is the viscotiy, B the flow law parameter , (u,v) the velocity 
	  vector, and n the flow law exponent.

	  If epsilon is NULL, it means this is the first time SystemMatrices is being run, and we 
	  return 10^14, initial viscosity.
	  */

	/*output: */
	double viscosity;

	/*input strain rate: */
	double exx,eyy,exy;

	/*Intermediary: */
	double A,e;
	double B,n;

	/*Get B and n*/
	B=GetB2d();
	n=GetN();

	if (n==1){
		/*Viscous behaviour! viscosity=B: */
		viscosity=B;
	}
	else{
		if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0)){
			viscosity=0.5*pow((double)10,(double)14);
		}
		else{
			/*Retrive strain rate components: */
			exx=*(epsilon+0);
			eyy=*(epsilon+1);
			exy=*(epsilon+2);

			/*Build viscosity: viscosity=B/(2*A^e) */
			A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+exx*eyy;
			if(A==0){
				/*Maxiviscositym viscosity for 0 shear areas: */
				viscosity=2.5*pow((double)10,(double)17);
			}
			else{
				e=(n-1)/(2*n);
				viscosity=B/(2*pow(A,e));
			}
		}
	}

	/*Checks in debugging mode*/
	ISSMASSERT(B>0);
	ISSMASSERT(n>0);
	ISSMASSERT(viscosity>0);

	/*Return: */
	*pviscosity=viscosity;
}
/*}}}*/
/*FUNCTION Matice::GetViscosity3d {{{1*/
void  Matice::GetViscosity3d(double* pviscosity3d, double* epsilon){

	/*Return viscosity accounting for steady state power law creep [Thomas and MacAyeal, 1982]: 
	 *
	 *                                               B
	 * viscosity3d= -------------------------------------------------------------------
	 *                      2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 *
	 *     where mu is the viscotiy, B the flow law parameter , (u,v) the velocity 
	 *     vector, and n the flow law exponent.
	 *
	 * If epsilon is NULL, it means this is the first time Emg is being run, and we 
	 * return g, initial viscosity.
	 */
	
	/*output: */
	double viscosity3d;

	/*input strain rate: */
	double exx,eyy,exy,exz,eyz;

	/*Intermediaries: */
	double A,e;
	double B,n;

	/*Get B and n*/
	B=GetB();
	n=GetN();

	if (n==1){
		/*Viscous behaviour! viscosity3d=B: */
		viscosity3d=B;
	}
	else{
		if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0) && 
				(epsilon[3]==0) && (epsilon[4]==0)){
			viscosity3d=pow((double)10,(double)14);
		}
		else{

			/*Retrive strain rate components: */
			exx=*(epsilon+0);
			eyy=*(epsilon+1);
			exy=*(epsilon+2);
			exz=*(epsilon+3);
			eyz=*(epsilon+4);

			/*Build viscosity: viscosity3d=2*B/(2*A^e) */
			A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+pow(exz,2)+pow(eyz,2)+exx*eyy;
			if(A==0){
				/*Maxiviscosity3dm viscosity for 0 shear areas: */
				viscosity3d=2.25*pow((double)10,(double)17);
			}
			else{
				e=(n-1)/2/n;
			
				viscosity3d=B/(2*pow(A,e));
			}
		}
	}

	/*Checks in debugging mode*/
	ISSMASSERT(B>0);
	ISSMASSERT(n>0);
	ISSMASSERT(viscosity3d>0);

	/*Assign output pointers:*/
	*pviscosity3d=viscosity3d;
}
/*}}}*/
/*FUNCTION Matice::GetViscosity3dStokes {{{1*/
void  Matice::GetViscosity3dStokes(double* pviscosity3d, double* epsilon){
	/*Return viscosity accounting for steady state power law creep [Thomas and MacAyeal, 1982]: 
	 *
	 *                                          B
	 * viscosity3d= -------------------------------------------------------------------
	 *                   2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 *
	 *     where mu is the viscotiy, B the flow law parameter , (u,v) the velocity 
	 *     vector, and n the flow law exponent.
	 *
	 * If epsilon is NULL, it means this is the first time Emg is being run, and we 
	 * return g, initial viscosity.
	 */
	
	/*output: */
	double viscosity3d;

	/*input strain rate: */
	double exx,eyy,exy,exz,eyz,ezz;

	/*Intermediaries: */
	double A,e;
	double B,n;
	double eps0;

	/*Get B and n*/
	eps0=pow((double)10,(double)-27);
	B=GetB();
	n=GetN();
	
	if (n==1){
		/*Viscous behaviour! viscosity3d=B: */
		viscosity3d=B;
	}
	else{
		if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0) && 
				(epsilon[3]==0) && (epsilon[4]==0) && (epsilon[5]==0)){
			viscosity3d=pow((double)10,(double)14);
		}
		else{

			/*Retrive strain rate components: */
			exx=*(epsilon+0);
			eyy=*(epsilon+1);
			ezz=*(epsilon+2); //not used
			exy=*(epsilon+3);
			exz=*(epsilon+4);
			eyz=*(epsilon+5);

			/*Build viscosity: viscosity3d=B/(2*A^e) */
			A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+pow(exz,2)+pow(eyz,2)+exx*eyy+pow(eps0,2);
			if(A==0){
				/*Maxiviscosity3dm viscosity for 0 shear areas: */
				viscosity3d=2.25*pow((double)10,(double)17);
			}
			else{
				e=(n-1)/2/n;
				viscosity3d=B/(2*pow(A,e));
			}
		}
	}

	/*Checks in debugging mode*/
	ISSMASSERT(B>0);
	ISSMASSERT(n>0);
	ISSMASSERT(viscosity3d>0);

	/*Assign output pointers:*/
	*pviscosity3d=viscosity3d;
}
/*}}}*/
/*FUNCTION Matice::GetViscosityComplement {{{1*/
void  Matice::GetViscosityComplement(double* pviscosity_complement, double* epsilon){
	/*Return viscosity accounting for steady state power law creep [Thomas and MacAyeal, 1982]: 
	 *
	 *  										                1
	 * viscosity= -------------------------------------------------------------------
	 *  				  2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 *
	 * If epsilon is NULL, it means this is the first time Gradjb is being run, and we 
	 * return mu20, initial viscosity.
	 */
	
	/*output: */
	double viscosity_complement;

	/*input strain rate: */
	double exx,eyy,exy;

	/*Intermediary value A and exponent e: */
	double A,e;
	double B,n;

	/*Get B and n*/
	B=GetB2d();
	n=GetN();

	if(epsilon){
		exx=*(epsilon+0);
		eyy=*(epsilon+1);
		exy=*(epsilon+2);

		/*Build viscosity: mu2=B/(2*A^e) */
		A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+exx*eyy;
		if(A==0){
			/*Maximum viscosity_complement for 0 shear areas: */
			viscosity_complement=2.25*pow((double)10,(double)17);
		}
		else{
			e=(n-1)/(2*n);
		
			viscosity_complement=1/(2*pow(A,e));
		}
	}
	else{
		viscosity_complement=4.5*pow((double)10,(double)17);
	}

	/*Checks in debugging mode*/
	ISSMASSERT(B>0);
	ISSMASSERT(n>0);
	ISSMASSERT(viscosity_complement>0);
		
	/*Return: */
	*pviscosity_complement=viscosity_complement;
}
/*}}}*/
/*FUNCTION Matice::InputDuplicate{{{1*/
void  Matice::InputDuplicate(int original_enum,int new_enum){

	/*Call inputs method*/
	if (IsInput(original_enum)) inputs->DuplicateInput(original_enum,new_enum);

}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVector(double* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVector(double* vector, int name, int type){

	/*Intermediaries*/
	Element *element      = NULL;

	/*Recover element*/
	element=(Element*)helement->delivers();

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	switch(type){

		case VertexEnum:

			switch(element->Enum()){

				case TriaEnum:
					double values[3];
					for (int i=0;i<3;i++) values[i]=vector[((Tria*)element)->nodes[i]->GetVertexDof()];
					this->inputs->AddInput(new TriaVertexInput(name,values));
					return;

				default: ISSMERROR("element %s not implemented yet",EnumToString(element->Enum()));
			}
		default: ISSMERROR("type %i (%s) not implemented yet",type,EnumToString(type));
	}
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVector(int* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVector(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVector(bool* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVector(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVectorDakota(double* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVectorDakota(double* vector, int name, int type){

	/*Intermediaries*/
	Element *element      = NULL;

	/*Recover element*/
	element=(Element*)helement->delivers();

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	switch(type){

		case VertexEnum:

			switch(element->Enum()){

				case TriaEnum:
					double values[3];
					for (int i=0;i<3;i++) values[i]=vector[((Tria*)element)->nodes[i]->GetVertexDof()];
					this->inputs->AddInput(new TriaVertexInput(name,values));
					return;

				default: ISSMERROR("element %s not implemented yet",EnumToString(element->Enum()));
			}
		default: ISSMERROR("type %i (%s) not implemented yet",type,EnumToString(type));
	}
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVectorDakota(int* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVectorDakota(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVectorDakota(bool* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVectorDakota(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromConstant(double constant, int name) {{{1*/
void  Matice::InputUpdateFromConstant(double constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromConstant(int constant, int name) {{{1*/
void  Matice::InputUpdateFromConstant(int constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromConstant(bool constant, int name) {{{1*/
void  Matice::InputUpdateFromConstant(bool constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromSolution{{{1*/
void  Matice::InputUpdateFromSolution(double* solution){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::IsInput{{{1*/
bool Matice::IsInput(int name){
	if (
				name==RheologyBEnum ||
				name==RheologyB2dEnum ||
				name==RheologyNEnum ||
				name==ControlParameterEnum
		){
		return true;
	}
	else return false;
}
/*}}}*/
