/*!\file InputUpdateFromDakotax
 * \brief: update datasets using  parameter inputs
 */

#include "./InputUpdateFromDakotax.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../modules.h"

void  InputUpdateFromDakotax(Elements* elements,Nodes* nodes,Vertices* vertices,Loads* loads,Materials*  materials,Parameters* parameters,Vec vec_partition,double* variables,char* *variables_descriptors,int numvariables){

	int     i,j,k;
	int     dummy;
	
	int     verbose;
	int     numberofvertices;
	int     qmu_npart;
	double *qmu_part_serial = NULL;
	double *qmu_part  = NULL;

	double* distributed_values=NULL;
	double* parameter=NULL;
	double* parameter_serial=NULL;
	char*   descriptor=NULL;
	char*   root=NULL; //root name of distributed variable, ex: thickness, drag, etc ...
	double* partition=NULL; //serial version of vec_partition

	/*retrieve parameters: */
	parameters->FindParam(&verbose,VerboseEnum);
	parameters->FindParam(&qmu_npart,QmuNPartEnum);
	parameters->FindParam(&qmu_part_serial,&dummy,QmuPartEnum);
	numberofvertices=vertices->NumberOfVertices();

	/*serialize partition vector: */
	VecToMPISerial(&partition,vec_partition);

	/*Use partition vector to repartition qmu_part, which is ordered in a serial way: */
	qmu_part=(double*)xmalloc(numberofvertices*sizeof(double));
	for(k=0;k<numberofvertices;k++) qmu_part[(int)(partition[k])]=qmu_part_serial[k];

	/*Go through all dakota descriptors, ex: "rho_ice","thermal_conductivity","thickness1","thickness2", etc ..., and 
	 * for each descriptor, take the variable value and plug it into the inputs: */

	for(i=0;i<numvariables;i++){
	
		descriptor=variables_descriptors[i];

		/*From descriptor, figure out if the variable is distributed (distributed implies there is a numeric value at the 
		 * end of the descriptor, for ex: thickness1, thickness10, etc .... If it is distributed, the next qmu_npart (number 
		 * of partitions in the distributed variable) variable are the values for each partition of the distributed variable: */
		if (!isdistributed(&root,descriptor)){
			
			/*Ok, variable is not distributed, just update inputs using the variable: */
			InputUpdateFromConstantx( elements,nodes, vertices,loads, materials,  parameters, variables[i],StringToEnum(descriptor));

		}
		else{

			/*Ok, variable is distributed. Root name of variable is also known. Now, allocate distributed_values and fill the 
			 * distributed_values with the next qmu_npart variables: */

			distributed_values=(double*)xmalloc(qmu_npart*sizeof(double));
			for(j=0;j<qmu_npart;j++){
				distributed_values[j]=variables[i+j];
			}

		
			/*Now, pick up the parameter corresponding to root: */
			if(!parameters->FindParam(&parameter_serial,NULL,StringToEnum(root))){
				ISSMERROR("%s%s"," could not find Qmu parameter: ",root);
			}

			/*repartition parameter: */
			parameter=(double*)xmalloc(numberofvertices*sizeof(double));
			for(k=0;k<numberofvertices;k++) parameter[(int)(partition[k])]=parameter_serial[k];
			
			/*We've got the parameter, we need to update it using qmu_part (a partitioning vector), and the distributed_values: */
			for(k=0;k<numberofvertices;k++){
				parameter[k]=parameter[k]*distributed_values[(int)qmu_part[k]];
			}

			#ifdef _ISSM_DEBUG_
				PetscSynchronizedPrintf(MPI_COMM_WORLD,"Parameter vector:");
				PetscSynchronizedFlush(MPI_COMM_WORLD);
				for(k=0;k<numberofvertices;k++){
					PetscSynchronizedPrintf(MPI_COMM_WORLD," node %i value %g\n",k+1,parameter[k]);
					PetscSynchronizedFlush(MPI_COMM_WORLD);
				}
				PetscSynchronizedPrintf(MPI_COMM_WORLD," enum: %i\n",StringToEnum(root));
				PetscSynchronizedFlush(MPI_COMM_WORLD);
			#endif
			  

			/*Update inputs using the parameter vector: */
			InputUpdateFromVectorDakotax( elements,nodes, vertices,loads, materials,  parameters, parameter, StringToEnum(root), VertexEnum);

			/*increment i to skip the distributed values just collected: */
			i+=qmu_npart-1; //careful, the for loop will add 1.

			/*Free allocations: */
			xfree((void**)&parameter);
			xfree((void**)&parameter_serial);
			xfree((void**)&distributed_values);

		}
		xfree((void**)&root);
	}

	/*Free ressources:*/
	xfree((void**)&partition);
	xfree((void**)&qmu_part);
	xfree((void**)&qmu_part_serial);

}
