/*!\file:  control.cpp
 * \brief: control solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "control"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

int main(int argc,char* *argv){

	int n;
	int i;

	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	char* analysis_type="control";

	/*Intermediary: */
	FemModel femmodel;
	Vec u_g=NULL;
	ParameterInputs* inputs=NULL;
	int waitonlock=0;
	double search_scalar;
	char* control_type=NULL;
	int   gsize;
	double* fit=NULL;
	double* optscal=NULL;
	double* u_g_obs=NULL;
	int*    maxiter=NULL;
	double  tolx;
	double*   p_g=NULL;
	Vec       grad_g=NULL;
	Vec       new_grad_g=NULL;
	Vec       grad_g_old=NULL;
	double*   grad_g_double=NULL;
	double    mincontrolconstraint;
	double    maxcontrolconstraint;
	int       nsteps;
	double*   J=NULL;
	OptArgs   optargs;
	OptPars   optpars;


	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Open handle to data on disk: */
	fid=fopen(inputfilename,"rb");
	if(fid==NULL) throw ErrorException(__FUNCT__,exprintf("%s%s%s","could not open file ",inputfilename," for binary reading")); 
	
	_printf_("read and create finite element model:\n");
	CreateFemModel(&femmodel,fid,analysis_type);

	/*Recover parameters used throughout the solution:*/
	femmodel.parameters->FindParam((void*)&nsteps,"nsteps");
	femmodel.parameters->FindParam((void*)&control_type,"control_type");
	femmodel.parameters->FindParam((void*)&fit,"fit");
	femmodel.parameters->FindParam((void*)&optscal,"optscal");
	femmodel.parameters->FindParam((void*)&maxiter,"maxiter");
	femmodel.parameters->FindParam((void*)&tolx,"tolx");
	femmodel.parameters->FindParam((void*)&waitonlock,"waitonlock");
	femmodel.parameters->FindParam((void*)&mincontrolconstraint,"mincontrolconstraint");
	femmodel.parameters->FindParam((void*)&maxcontrolconstraint,"maxcontrolconstraint");
	gsize=femmodel.nodes->NumberOfDofs();
	
	femmodel.parameters->FindParam((void*)&p_g,"p_g");
	femmodel.parameters->FindParam((void*)&u_g_obs,"u_g_obs");
		
	/*Initialize inputs:*/
	inputs=NewParameterInputs();

	/*Start looping: */
	for(n=0;n<nsteps;n++){
			
		_printf_("\n%s%i%s%i\n","   control method step ",n+1,"/",nsteps);
		ParameterInputsAddFromMat(inputs,p_g,gsize,control_type);
		ParameterInputsAddFromDouble(inputs,fit[n],"fit");

		_printf_("%s\n","      computing gradJ...");
		grad_g=GradJCompute(inputs,&femmodel,u_g_obs);
		_printf_("%s\n","      done.");
			
		
		_printf_("%s\n","      normalizing directions...");
		Orthx(&new_grad_g,grad_g,grad_g_old);
		VecFree(&grad_g); grad_g=new_grad_g;
		VecFree(&grad_g_old); grad_g_old=grad_g;
		VecToMPISerial(&grad_g_double,grad_g);
		_printf_("%s\n","      done.");

	
		_printf_("%s\n","      optimizing along gradient direction...");
		optargs.femmodel=&femmodel; optargs.p_g=p_g; optargs.u_g_obs=u_g_obs; optargs.grad_g=grad_g_double; optargs.inputs=inputs;optargs.n=n;
		optpars.xmin=-1; optpars.xmax=1; optpars.tolerance=tolx; optpars.maxiter=maxiter[n];
		BrentSearch(&search_scalar,J+n,&optpars,&objectivefunctionC,&optargs);
		_printf_("%s\n","      done.");
	
	
		
		_printf_("%s\n","      updating parameter using optimized search scalar...");
		for(i=0;i<gsize;i++)p_g[i]=p_g[i]+search_scalar*optscal[n]*grad_g_double[i];
		_printf_("%s\n","      done.");

		_printf_("%s\n","      constraining the new distribution...");    
		ControlConstrainx( p_g,gsize,mincontrolconstraint,maxcontrolconstraint,control_type);
		_printf_("%s\n","      done.");
	
		//some temporary saving 
		if ((n%5)==0){
			_printf_("%s\n","      saving temporary results...");
			OutputControl(u_g,p_g,J,nsteps,femmodel.partition,outputfilename,femmodel.nodesets);
			_printf_("%s\n","      done.");
		}

		_printf_("%s\n","      value of misfit J after optimization #",n,": ",J[n]);
		
		/*some freeing:*/
		xfree((void**)&grad_g_double);
	}

	/*Write results to disk: */
	_printf_("%s\n","       preparing final velocity solution...");
	ParameterInputsAddFromMat(inputs,p_g,gsize,control_type);
	ParameterInputsAddFromDouble(inputs,fit[n],"fit");
	
	diagnostic_core_nonlinear(&u_g,NULL,NULL,inputs,&femmodel);
	
	_printf_("%s\n","      saving final results...");
	OutputControl(u_g,p_g,J,nsteps,femmodel.partition,outputfilename,femmodel.nodesets);
	_printf_("%s\n","      done.");

	/*Write lock file if requested: */
	if (waitonlock){
		WriteLockFile(lockname);
	}
			
	_printf_("closing MPI and Petsc\n");
	MPI_Barrier(MPI_COMM_WORLD);

	/*Close MPI libraries: */
	PetscFinalize(); 


	/*end module: */
	MODULEEND();
	
	return 0; //unix success return;
}
