/*!\file: control_core.cpp
 * \brief: core of the control solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"

void control_core(FemModel* femmodel){

	int     i,n;
	
	/*parameters: */
	int     verbose=0;
	int     control_type;
	int     nsteps;
	double  eps_cm;
	double  tolx;
	bool    cm_gradient;
	int     dim;
	int     solution_type;

	double* fit=NULL;
	double* optscal=NULL;
	double* maxiter=NULL;
	double* cm_jump=NULL;

		
	/*intermediary: */
	double  search_scalar;
	OptArgs optargs;
	OptPars optpars;

	/*Solution core pointer*/
	void (*solutioncore)(FemModel*)=NULL;

	/*output: */
	double* J=NULL;

	/*Recover parameters used throughout the solution:{{{1*/
	femmodel->parameters->FindParam(&nsteps,NStepsEnum);
	femmodel->parameters->FindParam(&control_type,ControlTypeEnum);
	femmodel->parameters->FindParam(&fit,NULL,FitEnum);
	femmodel->parameters->FindParam(&optscal,NULL,OptScalEnum);
	femmodel->parameters->FindParam(&maxiter,NULL,MaxIterEnum);
	femmodel->parameters->FindParam(&cm_jump,NULL,CmJumpEnum);
	femmodel->parameters->FindParam(&eps_cm,EpsCmEnum);
	femmodel->parameters->FindParam(&tolx,TolXEnum);
	femmodel->parameters->FindParam(&cm_gradient,CmGradientEnum);
	femmodel->parameters->FindParam(&dim,DimEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	/*}}}*/

	/*out of solution_type, figure out solution core function pointer*/
	CorePointerFromSolutionEnum(&solutioncore,solution_type);

	/*some preliminary work to be done if running full-Stokes analysis: */
	if (solution_type==SteadystateSolutionEnum || solution_type==DiagnosticSolutionEnum){
		stokescontrolinit(femmodel);
	}

	/*Initialize misfit: */
	J=(double*)xmalloc(nsteps*sizeof(double));
		
	/*Initialize some of the BrentSearch arguments: */
	optargs.femmodel=femmodel;
	optpars.xmin=0; optpars.xmax=1; optpars.tolerance=tolx ;
	
	/*Start looping: */
	for(n=0;n<nsteps;n++){

		_printf_("\n%s%i%s%i\n","   control method step ",n+1,"/",nsteps);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,(int)fit[n],FitEnum);
		
		/*In case we are running a steady state control method, compute new temperature field using new parameter distribution: */
		if (solution_type==SteadystateSolutionEnum) solutioncore(femmodel);
	
		_printf_("%s\n","      computing gradJ...");
		gradient_core(femmodel);

		/*Return gradient if asked: */
		if (cm_gradient){
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,GradientEnum);
			goto cleanup_and_return;
		}

		_printf_("%s\n","      optimizing along gradient direction");
		optargs.n=n; optpars.maxiter=(int)maxiter[n]; optpars.cm_jump=cm_jump[n];
		BrentSearch(&search_scalar,J+n,&optpars,&objectivefunctionC,&optargs);

		_printf_("%s\n","      updating parameter using optimized search scalar..."); //true means update parameter and copy it onto ControlParameter input
		InputControlUpdatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,search_scalar*optscal[n],true);
		
		if(controlconvergence(J,fit,eps_cm,n)) break;

		/*Temporary saving every 5 control steps: */
		if (((n+1)%5)==0){
			_printf_("%s\n","      saving temporary results...");
			controlrestart(femmodel,J);
		}
	}

	_printf_("%s\n","      preparing final velocity solution");
	solutioncore(femmodel);

	/*some results not computed by steadystate_core or diagnostic_core: */
	InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,control_type); //the parameter itself!
	InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,GradientEnum); 
	femmodel->results->AddObject(new DoubleVecExternalResult(femmodel->results->Size()+1,JEnum,J,nsteps,1,0));
	femmodel->results->AddObject(new StringExternalResult(femmodel->results->Size()+1,ControlTypeEnum,EnumToString(control_type),1,0));

	cleanup_and_return:
	/*Free ressources: */
	xfree((void**)&fit);
	xfree((void**)&optscal);
	xfree((void**)&maxiter);
	xfree((void**)&cm_jump);
	xfree((void**)&J);
}
