/* \file WriteData.c:
 * \brief: general interface for writing data
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../include/include.h"
#include "./shared/shared.h"
#include "./io.h"

#ifdef _SERIAL_
#include <mex.h>
/*Several prototypes for WriteData, according to type: */
/*FUNCTION WriteData(mxArray** pdataref,DataSet* dataset){{{1*/
void WriteData(mxArray** pdataref,DataSet* dataset){

	mxArray* dataref=NULL;
	char* marshalled_dataset=NULL;
	int   marshalled_dataset_size;

	/*Write a dataset: */
	if(dataset){
			/* marshall the dataset: */
			marshalled_dataset=dataset->Marshall();
			marshalled_dataset_size=dataset->MarshallSize();
			
			dataref = mxCreateDoubleMatrix(0,0,mxREAL);
			mxSetM(dataref,(mwSize)(marshalled_dataset_size/sizeof(double)));
			mxSetN(dataref,(mwSize)1);
			mxSetPr(dataref,(double*)marshalled_dataset);	
	}
	else{
		/* return empty matrix: */
		dataref=mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
	
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,Mat matrix){{{1*/
void WriteData(mxArray** pdataref,Mat matrix){
		
	mxArray* dataref=NULL;
	
	if(matrix){
		
		/*call toolkit routine: */
		PetscMatrixToMatlabMatrix(&dataref,matrix);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}

	*pdataref=dataref;
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,double* matrix, int M,int N){{{1*/
void WriteData(mxArray** pdataref,double* matrix, int M,int N){
	
	mxArray* dataref=NULL;
	mxArray* tdataref=NULL;
		
	if(matrix){
		
		/*data is a double* pointer. Copy into a matrix: */
		tdataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(tdataref,(mwSize)N);
		mxSetN(tdataref,(mwSize)M);
		mxSetPr(tdataref,(double*)matrix);

		//transpose
		mexCallMATLAB(1,&dataref,1,&tdataref, "transpose");
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,Vec vector){{{1*/
void WriteData(mxArray** pdataref,Vec vector){
	
	mxArray* dataref=NULL;
	
	if(vector){
		
		/*call toolkit routine: */
		PetscVectorToMatlabVector(&dataref,vector);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;

}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,double* vector, int M){{{1*/
void WriteData(mxArray** pdataref,double* vector, int M){
	
	mxArray* dataref=NULL;

	if(vector){

		/*data is a double* pointer. Copy into a vector: */
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(dataref,(mwSize)M);
		mxSetN(dataref,(mwSize)1);
		mxSetPr(dataref,vector);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}

	*pdataref=dataref;
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,double scalar){{{1*/
void WriteData(mxArray** pdataref,double scalar){

	*pdataref=mxCreateDoubleScalar(scalar);
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,int integer){{{1*/
void WriteData(mxArray** pdataref,int integer){

		*pdataref=mxCreateDoubleScalar((double)integer);

}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,int boolean){{{1*/
void WriteData(mxArray** pdataref,bool boolean){

	*pdataref=mxCreateDoubleScalar((double)boolean);

}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,char* string){{{1*/
void WriteData(mxArray** pdataref,char* string){

		*pdataref=mxCreateString(string);
}
/*}}}*/

#else
/*FUNCTION WriteData(int* pdummy,void* data,char* data_type){{{1*/
void WriteData(int* pdummy,void* data,char* data_type){
	
	/*In parallel mode, WriteData is not used, instead we access the data directly through pointers in the solution sequences. */
}
/*}}}*/
#endif
