#include <limits.h>
#include <string.h>
#include <stdlib.h>

#include "../objects.h"

namespace bamg {

	/*MACROS {{{1*/
#define INTER_SEG(a,b,x,y) (((y) > (a)) && ((x) <(b)))
#define ABS(i) ((i)<0 ?-(i) :(i))
#define MAX1(i,j) ((i)>(j) ?(i) :(j))
#define NORM(i1,j1,i2,j2) MAX1(ABS((i1)-(j1)),ABS((i2)-(j2)))
	//IJ(i,j,l) returns the box number of i and j with respect to l
	//if !j&l and !i$l -> 0 (box zero: lower left )
	//if !j&l and  i$l -> 1 (box one:  lower right)
	//if  j&l and !i$l -> 2 (box two:  upper left )
	//if  j&l and  i$l -> 3 (box three:upper right)
#define IJ(i,j,l)  ((j&l) ? ((i&l) ? 3:2 ) :((i&l) ? 1:0 ))
	//I_IJ(k,l) returns l if first  bit of k is 1, else 0
#define I_IJ(k,l)  ((k&1) ? l:0)
	//J_IJ(k,l) returns l if second bit of k is 1, else 0
#define J_IJ(k,l)  ((k&2) ? l:0)
	/*}}}*/
	/*DOCUMENTATION What is a QuadTree? {{{1
	 * A Quadtree is a very simple way to group vertices according
	 * to their locations. A square that holds all the points of the mesh
	 * (or the geometry) is divided into 4 boxes. As soon as one box
	 * hold more than 4 vertices, it is divided into 4 new boxes, etc...
	 * There cannot be more than MAXDEEP (=30) subdivision.
	 * This process is like a Dichotomy in dimension 2
	 *
	 *  + - -  -    - -    -    - - + -   - + - + - + - -     - - +
	 *  |                           |       |   | X |             |
	 *                                      + - + - +
	 *  |                           |       |   |   |             |
	 *                              + -   - + - + - +             +
	 *  |                           |       |       |             |
	 *                         
	 *  |                           |       |       |             |
	 *  + - -  -    - -    -    - - + -   - + -   - + - -     - - +
	 *  |                           |               |             |
	 *                         
	 *  |                           |               |             |
	 *                         
	 *  |                           |               |             |
	 *  |                           |               |             |
	 *  + - -  -    - -    -    - - + -   -   -   - + - -     - - +
	 *  |                           |                             |
	 *                         
	 *  |                           |                             |
	 *                         
	 *  |                           |                             |
	 *                         
	 *  |                           |                             |
	 *  |                           |                             |
	 *  |                           |                             |
	 *  |                           |                             |
	 *  |                           |                             |
	 *  + - -  -    - -    -    - - + -   -   -   -   - -     - - +
	 *
	 * The coordinate system used in a quadtree are integers to avoid
	 * round-off errors. The vertex in the lower left box has the coordinates
	 * (0 0) 
	 * The upper right vertex has the follwing coordinates:
	 * 2^30 -1           2^30 -1        in decimal
	 * 0 1 1 1 .... 1    0 1 1 1 .... 1 in binary
	 *  \--   29  --/     \--   29  --/
	 * Using binaries is therefore very easy to locate a vertex in a box:
	 * we just need to look at the bits from the left to the right (See ::Add)
	 }}}1*/

	/*Constructors/Destructors*/
	/*FUNCTION QuadTree::QuadTree(Mesh * t,long nbv){{{1*/
	QuadTree::QuadTree(Mesh * t,long nbv) : 
		/*Original code from Frederic Hecht <hecht@ann.jussieu.fr> (BAMG v1.01, QuadTree.cpp/QuadTree)*/

		lenStorageQuadTreeBox(t->maxnbv/8+10),
		th(t),
		NbQuadTreeBox(0),
		NbVertices(0),
		NbQuadTreeBoxSearch(0),
		NbVerticesSearch(0)
	{ 
	 if (nbv == -1) nbv = t->nbv;
	 sb =new StorageQuadTreeBox(lenStorageQuadTreeBox);
	 root=NewQuadTreeBox();
	 if ( MaxISize <= MaxICoor){
		 ISSMERROR("MaxISize <= MaxICoor");
	 }
	 for (int i=0;i<nbv;i++) 
	  Add(t->vertices[i]);
	}
	/*}}}1*/
	/*FUNCTION QuadTree::QuadTree(){{{1*/
	QuadTree::QuadTree() : 
		/*Original code from Frederic Hecht <hecht@ann.jussieu.fr> (BAMG v1.01, QuadTree.cpp/QuadTree)*/

		lenStorageQuadTreeBox(100), // by default 100 vertices by box
		th(0),                      // initial mesh = NULL
		NbQuadTreeBox(0),           // initial number of quadtree boxes = 0
		NbVertices(0),              // initial number of vertices = 0
		NbQuadTreeBoxSearch(0),     // initial ?? = 0
		NbVerticesSearch(0){        // initial ?? = 0
			//create lenStorageQuadTreeBox (100) StorageQuadTreeBox elements
			sb  =new StorageQuadTreeBox(lenStorageQuadTreeBox); 
			//root=QuadTreeBox* pointer roward ??
			root=NewQuadTreeBox();
		}
	/*}}}1*/
	/*FUNCTION QuadTree::~QuadTree(){{{1*/
	QuadTree::~QuadTree() {
		/*Original code from Frederic Hecht <hecht@ann.jussieu.fr> (BAMG v1.01, QuadTree.cpp/~QuadTree)*/

		delete sb; 
		root=0;
	}
	/*}}}1*/

	/*Methods*/
	/*FUNCTION QuadTree::Add{{{1*/
	void  QuadTree::Add(BamgVertex &w){
		/*Original code from Frederic Hecht <hecht@ann.jussieu.fr> (BAMG v1.01, QuadTree.cpp/Add)*/

		QuadTreeBox** pb;
		QuadTreeBox*  b;
		register long i=w.i.x, j=w.i.y;
		register long level=MaxISize;

		//Get inital box (the largest)
		pb = &root;

		//Find the smallest box where w is located
		while((b=*pb) && (b->n<0)){ 

			//shift b->n by -1
			b->n--;

			//shifted righ by one bit: level=00000010 -> 00000001
			level >>= 1;

			//Get next subbox according to the bit value (level)
			pb = &b->b[IJ(i,j,level)];
		}

		//OK, we have found b, a Subbox holding vertices (might be full)
		//check that the vertex is not already in the box
		if  (b) {      
			if (b->n > 3 &&  b->v[3] == &w) return;
			if (b->n > 2 &&  b->v[2] == &w) return;
			if (b->n > 1 &&  b->v[1] == &w) return;
			if (b->n > 0 &&  b->v[0] == &w) return;
		}

		//check that l is not 0 (this should not happen as MaxDeep = 30)
		if (level==0){
			ISSMERROR("level==0 cannot be true as it has been initialized as MaxISize = %i",MaxISize);
		}

		//Now, try to add the vertex, if the subbox is full (n=4), we have to divide it
		//in 4 new subboxes
		while ((b= *pb) && (b->n == 4)){ // the QuadTreeBox is full

			//Copy the 4 vertices in the current QuadTreebox
			BamgVertex* v4[4];
			v4[0]= b->v[0];
			v4[1]= b->v[1];
			v4[2]= b->v[2];
			v4[3]= b->v[3];

			//set n as negative (box full -> holds 4 pointers toward subboxes and not 4 vertices)
			b->n = -b->n;

			//Initialize the 4 pointers toward the 4 subboxes
			b->b[0]=b->b[1]=b->b[2]=b->b[3]=NULL;

			// div the size by 2
			level >>= 1;

			//Put the four vertices in the new boxes
			for (register int k=0;k<4;k++){
				register int ij;
				register QuadTreeBox* bb =  b->b[ij=IJ(v4[k]->i.x,v4[k]->i.y,level)];

				// alloc the QuadTreeBox 
				if (!bb) bb=b->b[ij]=NewQuadTreeBox(); 

				//Copy the 4 vertices
				bb->v[bb->n++] = v4[k];
			}

			//Get the subbox where w (i,j) is located
			pb = &b->b[IJ(i,j,level)];
		}

		//  alloc the QuadTreeBox 
		if (!(b = *pb)) b=*pb= NewQuadTreeBox();

		//Add w
		b->v[b->n++]=&w;

		//Increase NbVertices by one (we have one new vertex)
		NbVertices++;    
	}
	/*}}}1*/
	/*FUNCTION QuadTree::NearestVertex{{{1*/
	BamgVertex*  QuadTree::NearestVertex(Icoor1 i,Icoor1 j) {
		/*Original code from Frederic Hecht <hecht@ann.jussieu.fr> (BAMG v1.01, QuadTree.cpp/NearestVertex)*/

		/*Build QuadTree*/
		QuadTreeBox* pb[MaxDeep];
		int      pi[MaxDeep];
		Icoor1   ii[MaxDeep];
		Icoor1   jj[MaxDeep];
		register int level=0; // levelevelevel
		register long n0;
		register QuadTreeBox* b;
		long     h=MaxISize,h0;
		long     hb=MaxISize;
		Icoor1   i0=0,j0=0;
		//iplus= i projected in [0,MaxISize-1] (example: if i<0, i=0)
		Icoor1   iplus( i<MaxISize?(i<0?0:i):MaxISize-1);
		//jplus= j projected in [0,MaxISize-1] (example: if j>=MaxISize, j=MaxISize-1)
		Icoor1   jplus( j<MaxISize?(j<0?0:j):MaxISize-1);
		//initial nearest vertex pointer
		BamgVertex*  vn=NULL;

		//Get initial Quadtree box (largest)
		b = root;

		//if the tree is empty, return NULL pointer
		if (!root->n) return vn; 

		//else, find the non empty QuadTreeBox containing  the point (i,j)
		while((n0=b->n)<0){

			//shifted righ by one bit: hb2=01000000 -> 00100000
			register Icoor1 hb2 = hb >> 1;
			//Get QuadTreeBox number of size hb2 containing i;j (Macro)
			register int      k = IJ(iplus,jplus,hb2);
			//Get the corresponding box b0
			register QuadTreeBox* b0=b->b[k];

			// break if NULL box or empty
			if (( b0 == NULL) || (b0->n == 0)) break;

			//Get next Qudtree box
			NbQuadTreeBoxSearch++;
			b=b0;	
			i0 += I_IJ(k,hb2); // i orign of QuadTreeBox (macro)
			j0 += J_IJ(k,hb2); // j orign of QuadTreeBox 
			hb = hb2; 
		}

		// if the current subbox is holding vertices, we are almost done
		if ( n0>0 ){  
			//loop over the vertices of the box and find the closest vertex
			for(register int k=0;k<n0;k++){
				I2 i2=b->v[k]->i;
				h0=NORM(iplus,i2.x,jplus,i2.y);
				if (h0<h){
					h = h0;
					vn = b->v[k];
				}
				NbVerticesSearch++;
			}
			return vn;
		}

		/* general case: the current box is empty, we have to get backwards
			and find the closest not-empty box and find the closest vertex*/

		//initialize pb pi ii and jj
		pb[0]=b;                  //pointer toward the box b
		pi[0]=b->n>0? (int)b->n:4;//number of vertices in the box
		ii[0]=i0;                 // i coordinate of the box
		jj[0]=j0;                 // j coordinate of the box

		//initialize h as hb
		h=hb;

		//loop, until level=0
		do {
			//get current box
			b= pb[level];

			//Loop over the vertices in current box (if not empty!)
			while (pi[level]--){

				//k = number of vertices in the box if there are vertices
				//k = 4 if the current box is pointing toward 4 other boxes
				register int k=pi[level];

				//if the current subbox is holding vertices,
				if (b->n>0){ // BamgVertex QuadTreeBox not empty
					NbVerticesSearch++;
					I2 i2 =  b->v[k]->i;
					h0 = NORM(iplus,i2.x,jplus,i2.y);
					if (h0 <h){
						h = h0;
						vn = b->v[k];
					}
				}

				else{
					register QuadTreeBox* b0=b;
					NbQuadTreeBoxSearch++;

					//if the next box exists:
					if ((b=b->b[k])){
						//shifted righ by one bit: hb2=01000000 -> 00100000
						hb>>=1;
						register Icoor1 iii = ii[level]+I_IJ(k,hb);
						register Icoor1 jjj = jj[level]+J_IJ(k,hb);

						//if the current point is in b,go to next box
						if (INTER_SEG(iii,iii+hb,iplus-h,iplus+h) && INTER_SEG(jjj,jjj+hb,jplus-h,jplus+h)){
							pb[++level]=  b;
							pi[level]= b->n>0 ?(int)  b->n : 4  ;
							ii[level]= iii;
							jj[level]= jjj;
						}

						//else go backwards
						else{
							//shifted righ by one bit: hb=001000000 -> 01000000
							b=b0;
							hb<<=1;
						}
					}
					//Go backwards
					else b=b0;
				}
			}

			//else go backwards
			//shifted righ by one bit: hb=001000000 -> 01000000
			hb <<= 1;
		} while (level--);

		//return vn, nearest vertex
		return vn;
	}
	/*}}}1*/
	/*FUNCTION QuadTree::NearestVertexWithNormal{{{1*/
	BamgVertex*  QuadTree::NearestVertexWithNormal(Icoor1 i,Icoor1 j) {
		/*Original code from Frederic Hecht <hecht@ann.jussieu.fr> (BAMG v1.01, QuadTree.cpp/NearestVertexWithNormal)*/

		QuadTreeBox * pb[ MaxDeep ];
		int  pi[ MaxDeep  ];
		Icoor1 ii[  MaxDeep ], jj [ MaxDeep];
		int l; // level
		QuadTreeBox * b;
		long     h =MaxISize,h0;
		long     hb=MaxISize;
		Icoor1  i0=0,j0=0;
		Icoor1  iplus( i<MaxISize?(i<0?0:i):MaxISize-1);
		Icoor1  jplus( j<MaxISize?(j<0?0:j):MaxISize-1);

		BamgVertex *vn=0;

		// init for optimisation ---
		b = root;
		register long  n0;
		if (!root->n)
		 return vn; // empty tree 

		while( (n0 = b->n) < 0) 
		  {
			// search the non empty 
			// QuadTreeBox containing  the point (i,j)
			register Icoor1 hb2 = hb >> 1 ;
			register  int k = IJ(iplus,jplus,hb2);// QuadTreeBox number of size hb2 contening i;j
			register QuadTreeBox * b0= b->b[k];
			if ( ( b0 == 0) || (b0->n == 0) ) 
			 break; // null box or empty   => break 	    
			NbQuadTreeBoxSearch++;
			b=b0;	
			i0 += I_IJ(k,hb2); // i orign of QuadTreeBox
			j0 += J_IJ(k,hb2); // j orign of QuadTreeBox 
			hb = hb2; 
		  }


		if ( n0 > 0) 
		  {  
			for(register int k=0;k<n0;k++)
			  {
				I2 i2 =  b->v[k]->i;
				//   try if is in the right direction -- 
				h0 = NORM(iplus,i2.x,jplus,i2.y);
				if (h0 <h) {
					h = h0;
					vn = b->v[k];}
					NbVerticesSearch++;
			  }
			if (vn) return vn; 
		  }
		// general case -----
		// INITIALISATION OF THE HEAP 
		l =0; // level 
		pb[0]= b;
		pi[0]=b->n>0 ?(int)  b->n : 4  ;
		ii[0]=i0;
		jj[0]=j0;
		h=hb;
		do {   // walk on the tree  
			b= pb[l];
			while (pi[l]--) // loop on 4 element of the box
			  { 	      
				int k = pi[l];

				if (b->n>0) // BamgVertex QuadTreeBox none empty
				  { 
					NbVerticesSearch++;
					I2 i2 =  b->v[k]->i;
					// if good direction when try -- 

					h0 = NORM(iplus,i2.x,jplus,i2.y);
					if (h0 <h) 
					  {
						h = h0;
						vn = b->v[k];
					  }
				  }
				else // Pointer QuadTreeBox 
				  { 
					register QuadTreeBox *b0=b;
					NbQuadTreeBoxSearch++;
					if ((b=b->b[k])) 
					  {
						hb >>=1 ; // div by 2
						register Icoor1 iii = ii[l]+I_IJ(k,hb);
						register Icoor1 jjj = jj[l]+J_IJ(k,hb);

						if  (INTER_SEG(iii,iii+hb,iplus-h,iplus+h) && INTER_SEG(jjj,jjj+hb,jplus-h,jplus+h)) 
						  {
							pb[++l]=  b;
							pi[l]= b->n>0 ?(int)  b->n : 4  ;
							ii[l]= iii;
							jj[l]= jjj;

						  }
						else
						 b=b0, hb <<=1 ;
					  }
					else
					 b=b0;
				  }
			  }
			hb <<= 1; // mul by 2 
		} while (l--);

		return vn;
	}
	/*}}}1*/
	/*FUNCTION QuadTree::SizeOf{{{1*/
	long QuadTree::SizeOf() const {
		return sizeof(QuadTree)+sb->SizeOf();
	}
	/*}}}1*/
	/*FUNCTION QuadTree::StorageQuadTreeBox::StorageQuadTreeBox{{{1*/
	QuadTree::StorageQuadTreeBox::StorageQuadTreeBox(long ll,StorageQuadTreeBox *nn) {
		/*Original code from Frederic Hecht <hecht@ann.jussieu.fr> (BAMG v1.01, QuadTree.cpp/StorageQuadTreeBox)*/

		len = ll;
		n = nn;
		b = new QuadTreeBox[ll];
		for (int i = 0; i <ll;i++)
		 b[i].n =0,b[i].b[0]=b[i].b[1]=b[i].b[2]=b[i].b[3]=0;
		bc =b;
		be = b +ll;
		if (!b){
			ISSMERROR("!b");
		}
	}
	/*}}}1*/
	/*FUNCTION QuadTree::ToClose {{{1*/
	BamgVertex*   QuadTree::ToClose(BamgVertex & v,double seuil,Icoor1 hx,Icoor1 hy){
		/*Original code from Frederic Hecht <hecht@ann.jussieu.fr> (BAMG v1.01, QuadTree.cpp/ToClose)*/

		const Icoor1 i=v.i.x;
		const Icoor1 j=v.i.y;
		const R2 X(v.r);
		const Metric  Mx(v.m);

		QuadTreeBox * pb[ MaxDeep ];
		int  pi[ MaxDeep  ];
		Icoor1 ii[  MaxDeep ], jj [ MaxDeep];
		register int l=0; // level
		register QuadTreeBox * b;
		Icoor1 h=MaxISize;
		Icoor1 hb =  MaxISize;
		Icoor1 i0=0,j0=0;

		//  BamgVertex *vn=0;

		if (!root->n)
		 return 0; // empty tree 

		// general case -----
		pb[0]=root;
		pi[0]=root->n>0 ?(int)  root->n : 4  ;
		ii[0]=i0;
		jj[0]=j0;
		h=hb;
		do {    
			b= pb[l];
			while (pi[l]--)
			  { 	      
				register int k = pi[l];

				if (b->n>0) // BamgVertex QuadTreeBox none empty
				  { 
					NbVerticesSearch++;
					I2 i2 =  b->v[k]->i;
					if ( ABS(i-i2.x) <hx && ABS(j-i2.y) <hy )
					  {
						R2 XY(X,b->v[k]->r);
						double dd;
						if( (dd= LengthInterpole(Mx(XY), b->v[k]->m(XY)))  < seuil ){
							return b->v[k]; 
						  }
					  }
				  }
				else // Pointer QuadTreeBox 
				  { 
					register QuadTreeBox *b0=b;
					NbQuadTreeBoxSearch++;
					if ((b=b->b[k]))
					  {
						hb >>=1 ; // div by 2
						register long iii = ii[l]+I_IJ(k,hb);
						register long jjj = jj[l]+J_IJ(k,hb);

						if  (INTER_SEG(iii,iii+hb,i-hx,i+hx) && INTER_SEG(jjj,jjj+hb,j-hy,j+hy)) 
						  {
							pb[++l]=  b;
							pi[l]= b->n>0 ?(int)  b->n : 4  ;
							ii[l]= iii;
							jj[l]= jjj;

						  }
						else
						 b=b0, hb <<=1 ;
					  }
					else
					 b=b0;
				  }
			  }
			hb <<= 1; // mul by 2 
		} while (l--);

		return 0;
	}
	/*}}}1*/

}
