function varargout=runme(varargin)
%RUNME - test deck for ISSM nightly runs
%
%   In a test deck directory (tests/Vertification/test01_IceShelfIceFrontM2d for example)
%   The following command will launch all the existing tests for test01
%   >> runme
%   To run the tests in serial only
%   >> runme('parallel',0)
%   etc...
%
%   Available options:
%      'analysis_type' followed by the desired SolutionEnum
%      'parallel'      0: serial only
%                      1: parallel only
%      'procedure'     'check' : run the test (default)
%                      'update': update the archive
%                      'model' : prepare the model but no test is run
%
%   Usage:
%      md=runme(varargin);
%
%   Examples:
%      runme;
%      runme('analysis_type',DiagnosticSolutionEnum);
%      runme('analysis_type',DiagnosticSolutionEnum,'parallel',0);
%      runme('procedure','update','analysis_type',DiagnosticSolutionEnum,'parallel',1);

% This file can be run to check that the current version of issm is giving 
% coherent results.
% is square. Just run this file in Matlab, with a properly setup ISSM code. 
% The results of this test will indicate if there is a difference between current computations 
% and archived results.

% Errors  between archived results and the current version will get flagged if they are not within 
% a certain tolerance.

%use ISSM_DIR generated by startup.m
global ISSM_DIR

%Check inputs
% {{{1
if nargout>1
	help runme
	error('runme error message: bad usage');
end

%recover options
options=pairoptions(varargin{:});
% }}}

%Process options
%GET benchmark {{{1
benchmark=getfieldvalue(options,'benchmark','nightly');
if ~ismember(benchmark,{'all','nightly'})
	disp('runme warnig: benchmark not supported, defaulting to test ''nightly''')
	benchmark='nighlty';
end
% }}}
%GET procedure {{{1
procedure=getfieldvalue(options,'procedure','check');
if ~ismember(procedure,{'check','update'})
	disp('runme warning: procedure not supported, defaulting to test ''check''')
	procedure='check';
end
% }}}
%GET output {{{1
output=getfieldvalue(options,'output','none');
if ~ismember(output,{'nightly','daily','none'})
	disp('runme warning: output not supported, defaulting to test ''none''')
	output='none';
end
% }}}
%GET RANK and NUMPROCS for mutlithreaded runs  {{{1
rank=getfieldvalue(options,'rank',1);
numprocs=getfieldvalue(options,'numprocs',1);
if (numprocs<rank), numprocs=1; end
% }}}
%GET ids  {{{1
list=dir;%use dir, as it seems to act OS independent
list_ids=[];
for i=1:numel(list),
	if ( strncmpi(list(i).name,'test',4) &...                         %File name must start with 'test'
			strncmpi(fliplr(list(i).name),fliplr('.m'),2)&...           %File name must end by '.m'
			~strncmpi(fliplr(list(i).name),fliplr('_nightly.m'),10)),   %File name end should be different that '_nightly.m'
		list_ids(end+1)=eval(list(i).name(5:end-2));                   %Keep test id only (skip 'test' and '.m')
	end
end
[i1,i2]=parallelrange(rank,numprocs,length(list_ids));               %Get tests for this cpu only
list_ids=list_ids(i1:i2);

test_ids=getfieldvalue(options,'id',list_ids);
test_ids=intersect(test_ids,list_ids);
% }}}
%Process Ids according to benchmarks{{{1
if strcmpi(benchmark,'nightly'),
	test_ids=intersect(test_ids,[1:999]);
end
% }}}

%Loop over tests and launch sequence
for id=test_ids,
	try,

		%Execute test
		eval(['test' num2str(id)]);
		eval(['test' num2str(id) '_nightly']);

		%UPDATE ARCHIVE?
		archive_name=['Archive' num2str(id) ];
		if strcmpi(procedure,'update'),

			for k=1:length(field_names),
				field=field_values{k};
				eval([ archive_name '_field' num2str(k) ' =  field ;']);
			end
			eval(['save ./../Archives/' archive_name ' ' archive_name '_field*']);
			disp(sprintf(['File ./../Archives/' archive_name ' saved\n']));

			%ELSE: CHECK TEST
		else,

			%load archive
			load(['./../Archives/' archive_name ]);

			for k=1:length(field_names),

				try,
					%Get field and tolerance
					field=field_values{k};
					fieldname=field_names{k};
					tolerance=field_tolerances{k};

					%compare to archive
					eval(['archive=' archive_name '_field' num2str(k) ';']);
					error_diff=full(max(abs(archive-field))/(max(abs(archive))+eps));

					%disp test result
					if (error_diff>tolerance);
						disp(sprintf(['ERROR   difference: %-7.2g > %7.2g test id: %i test name: %s field: %s'],...
							error_diff,tolerance,id,Id2Name(id),fieldname));
					else
						disp(sprintf(['SUCCESS difference: %-7.2g < %7.2g test id: %i test name: %s field: %s'],...
							error_diff,tolerance,id,Id2Name(id),fieldname));
					end

				catch me2

					%something went wrong, print failure message:
					directory=strsplit(pwd,'/');
					message=getReport(me2)
					if strcmpi(output,'nightly')
						fid=fopen([ISSM_DIR '/nightlylog/matlaberror.log'], 'at');
						fprintf(fid,'%s',message);
						fprintf(fid,'\n------------------------------------------------------------------\n');
						fclose(fid);
						disp(sprintf(['FAILURE difference: N/A test id: %i test name: %s field: %s'],id,Id2Name(id),fieldname));
					elseif strcmpi(output,'daily');
						fid=fopen([ISSM_DIR '/dailylog/matlaberror.log'], 'at');
						fprintf(fid,'%s',message);
						fprintf(fid,'\n------------------------------------------------------------------\n');
						fclose(fid);
						disp(sprintf(['FAILURE difference: N/A test id: %i test name: %s field: %s'],id,Id2Name(id),fieldname));
					end
				end
			end
		end

	catch me,

		%something went wrong, print failure message:
		directory=strsplit(pwd,'/');
		message=getReport(me)
		if strcmpi(output,'nightly')
			fid=fopen([ISSM_DIR '/nightlylog/matlaberror.log'], 'at');
			fprintf(fid,'%s',message);
			fprintf(fid,'\n------------------------------------------------------------------\n');
			fclose(fid);
			disp(sprintf(['FAILURE difference: N/A test id: %i test name: %s field: %s'],id,Id2Name(id),fieldname));
		elseif strcmpi(output,'daily');
			fid=fopen([ISSM_DIR '/dailylog/matlaberror.log'], 'at');
			fprintf(fid,'%s',message);
			fprintf(fid,'\n------------------------------------------------------------------\n');
			fclose(fid);
			disp(sprintf(['FAILURE difference: N/A test id: %i test name: %s field: %s'],id,Id2Name(id),fieldname));
		end
	end
end

%output md if requested
if nargout==1
	varargout{1}=md;
end
