#!/bin/bash
#This bash script calls the nightlyrun.m matlab file to run our nightly test decks. 
#It then processes the results and sends an email to the Ice developpers.

#some functions
function timer() #{{{1
{
	if [[ $# -eq 0 ]]; then
		echo $(date '+%s')
	else
		local  stime=$1
		etime=$(date '+%s')

		if [[ -z "$stime" ]]; then stime=$etime; fi

		dt=$((etime - stime))
		ds=$((dt % 60))
		dm=$(((dt / 60) % 60))
		dh=$((dt / 3600))
		printf '%d:%02d:%02d' $dh $dm $ds
	fi
} #}}}
function todaydate() #{{{1
{
	suffix=`date | awk '{printf("%s-%s-%s  %s",$2,$3,$6,$4);}'`			 
	echo $suffix;			 
} #}}}
function host_name() #{{{1
{
	#return host name depending on the OS
	if [ "$1" = "winxp32" ] 
	then
		HOST_NAME=`hostname`;
	else
		HOST_NAME=`hostname -s`;
	fi
	echo $HOST_NAME;
} #}}}

#Get configuration
#Source config file{{{1
if [ $# -ne 1 ];
then
	#no config file specified: exit
	echo "no config file specified. Exiting..." >&2 # Error message to stderr.
	exit 1
fi
if [ ! -f "$1" ]
then
	echo "File $1 not found!" >&2   # Error message to stderr.
	exit 1
fi 
source $1;
#}}}
#Export ISSM_* variables{{{1
export ISSM_DIR
export ISSM_ARCH
#}}}
#Initialize variables {{{1
TODAY=$(todaydate);
HOST_NAME=$(host_name $OS);
START_TIME=$(timer);
ISSM_RELEASE=$(basename $(echo $REPOSITORY));
USER=$(whoami);
INIT_PATH=$(pwd);
#}}}

#Lauch installation
#Checkout/update/none ISSM if requested (ISSM_INSTALLATION){{{1
if [ "$ISSM_INSTALLATION" == "checkout" ]
then

	#Erase previous code and Fetch the new one
	rm -rf $ISSM_RELEASE
	svn checkout $REPOSITORY

elif [ "$ISSM_INSTALLATION" == "update" ]
then

	#only update ISSM
	cd $ISSM_DIR
	svn update

elif [ "$ISSM_INSTALLATION" == "none" ]
then

	#do nothing
	echo "Skipping ISSM installation"

else

	#Error message
	echo "ISSM_INSTALLATION supported values are: checkout, update and none. Exiting..." >&2 # Error message to stderr.
	exit 1

fi
#}}}
#Create cluster.rc {{{1
#create simpler  cluster.rc file, with only the cluster we are interested in.
cd $ISSM_DIR/etc
cat << END > cluster.rc
begin
cluster_name=$HOST_NAME
cluster_codepath=$ISSM_DIR/bin
cluster_executionpath=$EXECUTION_PATH
cluster_login=$USER
cluster_port=0

end
END
#}}}
#Source environment variables with new matlab path {{{1
cd $ISSM_DIR/etc
source environment.sh MATLAB_DIR=$MATLAB_PATH
#}}}
#install/copy/none external packages    (ISSM_EXTERNALPACKAGES){{{1
if [ "$ISSM_EXTERNALPACKAGES" == "install" ]
then
	cd $ISSM_DIR/externalpackages
	for ep in $EXTERNALPACKAGES
	do 
		cd $ep
		if [ "$ep" == "petsc" ]
		then
			#For Petsc (2.3 for now)
			cp configs/2.3.2-p3/$OS/* . 
			#cp configs/3.0.0-p11/$OS/* . 
			#cat install.sh | sed -e "s/version='2.3.2-p3'/version='3.0.0-p11'/g" > temp.sh
			#mv temp.sh install.sh
		else
			cp configs/$OS/* .
		fi
		./install.sh $NUMCPUS_INSTALL
		cd ..
	done

	#3: install automake and autoconf
	cd $ISSM_DIR/devpackages
	make
	cd ..

elif [ "$ISSM_EXTERNALPACKAGES" == "copy" ]
then

	#erase externapackages, and link with externalpackages_dir
	cd $ISSM_DIR
	rm -rf externalpackages
	cp -Rf $EXTERNALPACKAGESDIR ./

	rm -rf devpackages
	cp -Rf $DEVPACKAGESDIR ./

elif [ "$ISSM_EXTERNALPACKAGES" == "none" ]
then

	#Do nothing
	echo "Skipping external packages installation"

else

	#Error message
	echo "ISSM_EXTERNALPACKAGES supported values are: install, copy and none. Exiting..." >&2 # Error message to stderr.
	exit 1

fi
#}}}
#ISSM compilation yes/no                (ISSM_COMPILATION) {{{1
if [ "$ISSM_COMPILATION" == "yes" ]
then

	cd $ISSM_DIR
	make distclean
	sh scripts/automakererun.sh
	sh configs/$OS/$OS.sh.petsc2 #use petsc2 for now
	#sh configs/$OS/$OS.sh.petsc3 #switch to petsc3 to debug

	#4: compile and install ISSM
	if [ "$OS" = "winxp32" ] 
	then
		cd $ISSM_DIR/src/c
		./intel-compile.sh
		cd $ISSM_DIR/src/mex
		make install
		cd $ISSM_DIR
	else
		make -j $NUMCPUS_INSTALL
		make -j $NUMCPUS_INSTALL install
	fi

elif [ "$ISSM_COMPILATION" == "no" ]
then

	#do nothing
	echo "Skipping ISSM compilation"

else

	#Error message
	echo "ISSM_COMPILATION supported values are: yes and no. Exiting..." >&2 # Error message to stderr.
	exit 1

fi
#}}}

#Prepare run
#Windows hack for startup.m {{{1
#windows environments: ISSM_DIR_WIN variable not correctly picked up when using 
#the cron job. just get startup to take the ISSM_DIR variable as the pwd:
if [ "$OS" = "winxp32" ]
then
	cat startup.m | sed 's/clear status/clear status; ISSM_DIR=pwd;/g' > startup.m.bak
	mv startup.m.bak startup.m
fi
#}}}
#create softlink to startup {{{1
cd $ISSM_DIR/test/NightlyRun/
ln -s $ISSM_DIR/startup.m .
#}}}
#Create nightlylog directory and info.log {{{1
#put installation elapsed time in info.log
INSTALL_TIME=$(timer)
ELAPSED_INSTALL=$(timer $START_TIME)
rm -rf $ISSM_DIR/nightlylog
mkdir  $ISSM_DIR/nightlylog
cat << END > $ISSM_DIR/nightlylog/info.log
today:     $(echo $TODAY)
user:      $(echo $USER)
host:      $(echo $HOST_NAME)
OS:        $(echo $OS)
release:   $(echo $ISSM_RELEASE)
init_path: $(echo $INIT_PATH)
elapsed_install: $(echo $ELAPSED_INSTALL)
END
#}}}
#check NUMCPUS_RUN options {{{1
if [ "$NUMCPUS_RUN" = "" ]
then
	echo "NUMCPUS_RUN option not found, defaulting to NUMCPUS_RUN = 1"
	NUMCPUS_RUN=1
fi
#}}}

#Run tests
#Launch all tests on different cpus {{{1
for (( i=1;i<=$NUMCPUS_RUN;i++ ))
do
	#Launch matlab and the nightly run script
	cat > $ISSM_DIR/nightlylog/matlab_run$i.m << EOF
	warning off %necessary to avoid a info.log of several Go for parallel runs
	try,
	cd $ISSM_DIR/test/NightlyRun
	startup;
	$(if [ "$NROPTIONS" = ""  ]
	then
		echo "runme('output','nightly','rank',$i,'numprocs',$NUMCPUS_RUN);"
	else
		echo "runme($NROPTIONS,'output','nightly','rank',$i,'numprocs',$NUMCPUS_RUN);"
	fi
	)
	catch me,
		%An error occured, get report and exit
		directory=strsplit(pwd,'/');
		message=getReport(me)
		fid=fopen([ISSM_DIR '/nightlylog/matlaberror.log'], 'at');
		fprintf(fid,'\nMatlab error occured in: %s\n\n',directory{end});
		fprintf(fid,'%s',message);
		fclose(fid);
	end
	exit
EOF

	#Start run from nightlylog directory
	cd $ISSM_DIR/nightlylog/

	#Start test
	MATLAB_VERSION="7.6" #7.2,7.4,7.6 and 7.8
	/usr/local/pkgs/matlab-$MATLAB_VERSION/bin/matlab  -nojvm -nosplash  -r matlab_run$i -logfile matlab_log$i.log &
done

#wait until matlab closes
wait
#}}}
#concatenate all reports {{{1
cd $ISSM_DIR/nightlylog/
mv matlab_log1.log  matlab_log.log
for (( i=2;i<=$NUMCPUS_RUN;i++ ))
do
	cat matlab_log.log matlab_log$i.log > matlab_log.log.bak
	mv matlab_log.log.bak matlab_log.log
	rm matlab_log$i.log

done
#}}}
#Complete info.log {{{1
ELAPSED_RUN=$(timer $INSTALL_TIME)
ELAPSED_TOTAL=$(timer $START_TIME)
cat << END >>  $ISSM_DIR/nightlylog/info.log
elapsed_run:   $(echo $ELAPSED_RUN)
elapsed_total: $(echo $ELAPSED_TOTAL)
END
#}}}

#Send Report
#Build html report {{{1
cd $ISSM_DIR/nightlylog/
sh ../scripts/report.sh
echo "html report located in $ISSM_DIR/nightlylog/report.html"
#}}}
#send mail if requested                  (SKIPMAIL) {{{1
if [ "$SKIPMAIL" != "yes" ]
then
	echo "sending report..."
	source $ISSM_DIR/cron/mailinglist
	for i in `echo $MAILINGLIST`; do

		if [ "$OS" = "winxp32" ]
		then
			email -html -f "ISSM Nightly run $EMAIL_ADRESS" -s "Nightly runs of $ISSM_RELEASE , operating system: $OS, host: $HOST_NAME, user: $USER. " $i < $ISSM_DIR/test/Verification/NightlyRun/report.html
		else 
			if [ "$OS" = "linux64" ]
			then
cat - $ISSM_DIR/nightlylog/report.html <<HERE | /usr/lib/sendmail  -oi -t
From: "ISSM Nightly run" <$EMAIL_ADRESS>
To: $i
Subject: Nightly runs of $ISSM_RELEASE, operating system: $OS, host: $HOST_NAME, user: $USER.
Mime-Version: 1.0
Content-Type: text/html
HERE
		else
cat - $ISSM_DIR/nightlylog/report.html <<HERE | /usr/sbin/sendmail  -oi -t
From: "ISSM Nightly run" <$EMAIL_ADRESS>
To: $i
Subject: Nightly runs of $ISSM_RELEASE, operating system: $OS, host: $HOST_NAME, user: $USER.
Mime-Version: 1.0
Content-Type: text/html
HERE
			fi
		fi
	done
fi
#}}}
