/*!\file: solver_diagnostic_nonlinear.cpp
 * \brief: core of the diagnostic solution for non linear materials
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"
#include "../solutions/solutions.h"
#include "./solvers.h"

void solver_diagnostic_nonlinear(FemModel* femmodel,bool conserve_loads){


	/*solution : */
	Vec ug=NULL; 
	Vec uf=NULL; 
	Vec old_ug=NULL; 
	Vec old_uf=NULL; 
	Loads* loads=NULL;

	/*intermediary: */
	Mat Kgg=NULL;
	Mat Kff=NULL;
	Mat Kfs=NULL;
	Vec pg=NULL;
	Vec pf=NULL;
	int converged;
	int constraints_converged;
	int num_unstable_constraints;
	int count;
	int min_mechanical_constraints;
	int max_nonlinear_iterations;

	/*parameters:*/
	int kflag,pflag;
	char* solver_string=NULL;
	int verbose=0;
	int dim;

	/*Recover parameters: */
	kflag=1; pflag=1;
	femmodel->parameters->FindParam(&solver_string,SolverStringEnum);
	femmodel->parameters->FindParam(&dim,DimEnum);
	femmodel->parameters->FindParam(&verbose,VerboseEnum);
	femmodel->parameters->FindParam(&min_mechanical_constraints,MinMechanicalConstraintsEnum);
	femmodel->parameters->FindParam(&max_nonlinear_iterations,MaxNonlinearIterationsEnum);
	
	/*Were loads requested as output? : */
	if(conserve_loads){
		loads=(Loads*)femmodel->loads->Copy(); //protect loads from being modified by the solution
	}
	else{
		loads=(Loads*)femmodel->loads; //modify loads  in this solution
	}

	count=1;
	converged=0;

	/*Start non-linear iteration using input velocity: */
	GetSolutionFromInputsx(&ug, femmodel->elements, femmodel->nodes, femmodel->vertices, loads, femmodel->materials, femmodel->parameters);
	Reducevectorgtofx(&uf, ug, femmodel->nodesets);

	//Update once again the solution to make sure that vx and vxold are similar (for next step in transient or steadystate)
	InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug);

	for(;;){

		//save pointer to old velocity
		VecFree(&old_ug);old_ug=ug;
		VecFree(&old_uf);old_uf=uf;

		if(verbose)_printf_("   Generating matrices\n");
		SystemMatricesx(&Kgg, &pg,femmodel->elements,femmodel->nodes,femmodel->vertices,loads,femmodel->materials,femmodel->parameters,kflag,pflag);

		if(verbose)_printf_("   Generating penalty matrices\n");
		PenaltySystemMatricesx(Kgg, pg,NULL,femmodel->elements,femmodel->nodes,femmodel->vertices,loads,femmodel->materials,femmodel->parameters,kflag,pflag);
		
		if(verbose)_printf_("   reducing matrix from g to f set\n");
		Reducematrixfromgtofx(&Kff,&Kfs,Kgg,femmodel->nodesets);

		/*Free ressources: */
		MatFree(&Kgg);
	
		if(verbose)_printf_("   reducing load from g to f set\n");
		Reduceloadfromgtofx(&pf, pg, Kfs, femmodel->ys, femmodel->nodesets);

		//no need for pg and Kfs anymore 
		VecFree(&pg); 
		MatFree(&Kfs);

		if(verbose)_printf_("   solving\n");
		Solverx(&uf, Kff, pf, old_uf, solver_string);

		if(verbose)_printf_("   merging solution from f to g set\n");
		Mergesolutionfromftogx(&ug, uf,femmodel->ys,femmodel->nodesets);

		//Update inputs using new solution:
		InputUpdateFromSolutionx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug);

		if(verbose)_printf_("   penalty constraints\n");
		PenaltyConstraintsx(&constraints_converged, &num_unstable_constraints, femmodel->elements,femmodel->nodes,femmodel->vertices,loads,femmodel->materials,femmodel->parameters);

		if(verbose)_printf_("   number of unstable constraints: %i\n",num_unstable_constraints);

		/*Figure out if convergence is reached.*/
		convergence(&converged,Kff,pf,uf,old_uf,femmodel->parameters);
		MatFree(&Kff);VecFree(&pf);
		
		/*add converged to inputs: */
		InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,converged,ConvergedEnum);

		//rift convergence
		if (!constraints_converged) {
			if (converged){
				if (num_unstable_constraints <= min_mechanical_constraints) converged=1;
				else converged=0;
			}
		}

		/*Increase count: */
		count++;
		if(converged==1)break;
		if(count>=max_nonlinear_iterations){
			_printf_("   maximum number of iterations (%i) exceeded\n",max_nonlinear_iterations); 
			break;
		}
	}

	/*Delete loads only if no ouput was requested: */
	if(conserve_loads) delete loads;

	/*clean up*/
	VecFree(&uf);
	VecFree(&ug);
	VecFree(&old_uf);
	VecFree(&old_ug);
	xfree((void**)&solver_string);
	
}
