/*! \file  ContourtoNodes
    \brief: takes a  contour file, and figures out which nodes  (x,y list)
    are inside this contour. 

	usage:
	[flags]=ContourToNodes(x,y,contours,interptype,edgevalue);
	
	where:

	input:

		x,y: node cooredinates
		
		contours: structure holding sets of vertices making open contours. 
		
		interptype: string definining type of interpolation ('element', or 'node', or 'element and node');

		edgevalue: integer (0, 1 or 2) defining the value associated to the nodes on the edges of the polygons

	output:
		flags: vector of flags (0 or 1), of size nods 
*/
	
#include "./ContourToNodes.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]) {

	int i,j;

	/* required input: */
	double* x=NULL;
	double* y=NULL;
	int     edgevalue;
	char*   interptype=NULL;

	/* output: */
	Vec  flags=NULL;
	int  nods;

	//contours
	mxArray*  matlabstructure=NULL;
	int numcontours;
	Contour** contours=NULL;
	Contour*  contouri=NULL;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	CheckNumMatlabArguments(nlhs,NLHS,nrhs,NRHS,__FUNCT__,&ContourToNodesUsage);

	
	/*Fetch inputs: */
	FetchData(&x,&nods,NULL,XHANDLE);
	FetchData(&y,NULL,NULL,YHANDLE);
	FetchData(&edgevalue,EDGEVALUEHANDLE);

	//Fetch contours

	if(mxIsChar(FILENAME)){
		/*Call expread on filename to build a contour array in the matlab workspace: */
		mexCallMATLAB( 1, &matlabstructure, 1, (mxArray**)&FILENAME, "expread");
	}
	else{
		/*FILENAME is actually a structure, coming directly from expread: */
		matlabstructure=(mxArray*)FILENAME;
	}

	numcontours=mxGetNumberOfElements(matlabstructure);
	contours=(Contour**)xmalloc(numcontours*sizeof(Contour*));
	for(i=0;i<numcontours;i++){
		//allocate
		contouri=(Contour*)xmalloc(sizeof(Contour));
		//retrieve dimension of this contour.
		contouri->nods=(int)mxGetScalar(mxGetField(matlabstructure,i,"nods"));
		//set pointers.
		contouri->x=mxGetPr(mxGetField(matlabstructure,i,"x"));
		contouri->y=mxGetPr(mxGetField(matlabstructure,i,"y"));
		*(contours+i)=contouri;
	}

	/* Debugging of contours :{{{1*/
	/*for(i=0;i<numcontours;i++){
		printf("\nContour echo: contour number  %i / %i\n",i+1,numcontours);
		contouri=*(contours+i);
		printf("   Number of grids %i\n",contouri->nods);
		for (j=0;j<contouri->nods;j++){
			printf("   %lf %lf\n",*(contouri->x+j),*(contouri->y+j));
		}
	}*/
	/*}}}*/

	/*Run interpolation routine: */
	ContourToNodesx(&flags,x,y,nods,contours,numcontours,edgevalue);

	/* output: */
	WriteData(FLAGS,flags);

	/*end module: */
	MODULEEND();

}

void ContourToNodesUsage(void){
	printf("   usage:\n");
	printf("   [flags]=ContourToNodes(x,y,contourname,edgevalue);\n\n");
	printf("   where:\n");
	printf("      x,y: list of nodes.\n");
	printf("      contourname: name of .exp file containing the contours, or resulting structure from call to expread.\n");
	printf("      interptype: string definining type of interpolation ('element', or 'node').\n");
	printf("      edgevalue: integer (0, 1 or 2) defining the value associated to the nodes on the edges of the polygons.\n");
	printf("      flags: vector of flags (0 or 1), of size nods.\n");
	printf("\n");
}
