/*!\file:  issm.cpp
 * \brief: ISSM main parallel program
 */ 

#include "./issm.h"
#include "config.h"

#ifdef _HAVE_DAKOTA_ 
#else
#error "No Dakota"
#endif
int main(int argc,char* *argv){

	/*MPI: */
	extern int my_rank;
	extern int num_procs;

	/*I/O: */
	FILE     *input_fid        = NULL;
	FILE     *output_fid       = NULL;
	char     *inputfilename    = NULL;
	char     *outputfilename   = NULL;
	char     *lockname         = NULL;
	bool      qmu_analysis     = false;
	bool      control_analysis = false;
	bool      waitonlock       = false;

	/*FemModel: */
	FemModel *femmodel         = NULL;

	/*configuration: */
	void (*solutioncore)(FemModel*)=NULL; //core solution function pointer
	int* analyses=NULL;
	int  numanalyses;
	int  solution_type;

	/*time*/
	double   start, finish;
	double   start_core, finish_core;
	double   start_init, finish_init;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	ISSMERROR(" parallel executable was compiled without support of parallel libraries!");
	#endif

	/*Initialize Petsc and get start time*/
	PetscInitialize(&argc,&argv,(char *)0,"");  
	MPI_Barrier(MPI_COMM_WORLD); start=MPI_Wtime();

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover solution and file names:\n");
	solution_type=StringAsEnum(argv[1]);
	inputfilename=argv[3];
	outputfilename=argv[4];
	lockname=argv[5];

	MPI_Barrier(MPI_COMM_WORLD); start_init=MPI_Wtime();

	/*out of solution_type, figure out solution core function pointer, 
	 *and types of analyses needed: */
	SolutionConfiguration(&analyses,&numanalyses,&solutioncore,solution_type);

	/*Open input file to process model
	 * and ouput file to start unload results*/
	input_fid =pfopen(inputfilename ,"rb");
	output_fid=pfopen(outputfilename,"wb");

	_printf_("create finite element model:\n");
	femmodel=new FemModel(input_fid,solution_type,analyses,numanalyses);

	/*add output_fid to parameters: */
	femmodel->parameters->SetParam(output_fid,OutputFilePointerEnum);
	
	/*get parameters: */
	femmodel->parameters->FindParam(&qmu_analysis,QmuAnalysisEnum);
	femmodel->parameters->FindParam(&control_analysis,ControlAnalysisEnum);
	femmodel->parameters->FindParam(&waitonlock,WaitOnLockEnum);

	MPI_Barrier(MPI_COMM_WORLD); finish_init=MPI_Wtime();

	/*are we running the solution sequence, or a qmu wrapper around it? : */
	if(!qmu_analysis){
		if(!control_analysis){
			
			_printf_("call computational core:\n");
			MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
			solutioncore(femmodel);
			MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );

		}
		else{
			/*run control analysis: */
			_printf_("call computational core:\n");
			MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
			control_core(femmodel);
			MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );

		}

		_printf_("write results to disk:\n");
		OutputResultsx(femmodel->elements, femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,&femmodel->results);
	}
	else{
		/*run qmu analysis: */
		_printf_("calling qmu analysis on diagnostic core:\n");

		#ifdef _HAVE_DAKOTA_ 
		MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
		Qmux(femmodel,femmodel->parameters);
		MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );
	 	#else
		ISSMERROR(" Dakota not present, cannot do qmu!");
		#endif
	}

	/*Close output file and write lock file if requested*/
	pfclose(output_fid,outputfilename);
	if (waitonlock>0){
		_printf_("write lock file:\n");
		WriteLockFile(lockname);
	}

	/*Free ressources */
	xfree((void**)&analyses);
	delete femmodel;

	/*Get finish time and close*/
	MPI_Barrier(MPI_COMM_WORLD); finish = MPI_Wtime( );
	_printf_("\n   %-34s %f seconds  \n","FemModel initialization elapsed time:",finish_init-start_init);
	_printf_("   %-34s %f seconds  \n","Core solution elapsed time:",finish_core-start_core);
	_printf_("\n   %s %i hrs %i min %i sec\n\n","Total elapsed time:",int((finish-start)/3600),int(int(finish-start)%3600/60),int(finish-start)%60);
	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 
	
	/*end module: */
	MODULEEND();

	return 0; //unix success return;
}
