/*
 * \file Inputs.c
 * \brief: implementation of the Inputs class, derived from DataSet class
 */

/*Headers: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./DataSet.h"
#include "../shared/shared.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"

using namespace std;
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION Inputs::Inputs(){{{1*/
Inputs::Inputs(){
	return;
}
/*}}}*/
/*FUNCTION Inputs::Inputs(int in_enum){{{1*/
Inputs::Inputs(int in_enum): DataSet(in_enum) {
	//do nothing;
	return;
}
/*}}}*/
/*FUNCTION Inputs::~Inputs(){{{1*/
Inputs::~Inputs(){
	return;
}
/*}}}*/

/*Object management*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,double* gauss,int enum_type){{{1*/
void Inputs::GetParameterValue(double* pvalue,double* gauss, int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValue(pvalue,gauss);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,double* gauss,int enum_type,double defaultvalue){{{1*/
void Inputs::GetParameterValue(double* pvalue,double* gauss, int enum_type,double defaultvalue){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. Return the default value: */
		*pvalue=defaultvalue;
	}
	else{
		input->GetParameterValue(pvalue,gauss);
	}
}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(bool* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(bool* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(int* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(int* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(double* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type){{{1*/
void Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValues(values,gauss_pointers,numgauss);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type,double* defaultvalues){{{1*/
void Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type,double* defaultvalues){

	int i;
	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. Return the default values: */
		for(i=0;i<numgauss;i++) values[i]=defaultvalues[i];
	}
	else{
		input->GetParameterValues(values,gauss_pointers,numgauss);
	}

}
/*}}}*/
/*FUNCTION Inputs::GetParameterAverage{{{1*/
void Inputs::GetParameterAverage(double* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterAverage(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterDerivativeValue{{{1*/
void Inputs::GetParameterDerivativeValue(double* derivativevalues, double* xyz_list, double* gauss,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterDerivativeValue(derivativevalues,xyz_list,gauss);
}
/*}}}*/
/*FUNCTION Inputs::AddInput{{{1*/
int  Inputs::AddInput(Input* in_input){

	/*First, go through dataset of inputs and check whether any input 
	 * with the same name is already in. If so, erase the corresponding 
	 * object before adding this new one: */
	vector<Object*>::iterator object;
	Input* input=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==in_input->EnumType()){
			this->DeleteObject(input);
			break;
		}
	}
	this->AddObject(in_input);

	return 1;
}
/*}}}*/
/*FUNCTION Inputs::GetInput{{{1*/
Input* Inputs::GetInput(int enum_name){

	vector<Object*>::iterator object;
	Input* input=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==enum_name){
			return input;
		}
	}
	return NULL;
}
/*}}}*/
/*FUNCTION Inputs::DeleteInput{{{1*/
int  Inputs::DeleteInput(int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==enum_type){
			this->DeleteObject(input);
			break;
		}
	}

	return 1;

}
/*}}}*/
/*FUNCTION Inputs::ChangeEnum{{{1*/
void  Inputs::ChangeEnum(int oldenumtype,int newenumtype){

	/*Go through dataset of inputs and look for input with 
	 * same enum as input enum, once found, just change its name */
	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Delete existing input of newenumtype if it exists*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		input=(Input*)(*object); 

		if (input->EnumType()==newenumtype){
			this->DeleteObject(input);
			break;
		}
	}

	/*Change enum_type of input of oldenumtype*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==oldenumtype){
			input->ChangeEnum(newenumtype);
			break;
		}
	}
}
/*}}}*/
/*FUNCTION Inputs::SpawnTriaInputs{{{1*/
Inputs* Inputs::SpawnTriaInputs(int* indices){

	/*Intermediary*/
	vector<Object*>::iterator object;
	Input* inputin=NULL;
	Input* inputout=NULL;

	/*Output*/
	Inputs* newinputs=new Inputs();

	/*Go through inputs and call Spawn function*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Create new input*/
		inputin=(Input*)(*object); 
		inputout=inputin->SpawnTriaInput(indices);

		/*Add input to new inputs*/
		newinputs->AddObject(inputout);
	}

	/*Assign output pointer*/
	return newinputs;
}
/*}}}*/
