/*!\file PentaRef.c
 * \brief: implementation of the PentaRef object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION PentaRef::PentaRef(){{{1*/
PentaRef::PentaRef(){
	this->element_type_list=NULL;
}
/*}}}*/
/*FUNCTION PentaRef::PentaRef(int* types,int nummodels){{{1*/
PentaRef::PentaRef(const int nummodels){

	/*Only allocate pointer*/
	element_type_list=(int*)xmalloc(nummodels*sizeof(int));

}
/*}}}*/
/*FUNCTION PentaRef::~PentaRef(){{{1*/
PentaRef::~PentaRef(){
	xfree((void**)&element_type_list);
}
/*}}}*/

/*Management*/
/*FUNCTION PentaRef::SetElementType{{{1*/
void PentaRef::SetElementType(int type,int type_counter){

	ISSMASSERT(type==P1Enum || type==P1DGEnum);

	/*initialize element type*/
	this->element_type_list[type_counter]=type;
}
/*}}}*/

/*Reference Element numerics*/
/*FUNCTION PentaRef::GetBPattyn {{{1*/
void PentaRef::GetBPattyn(double* B, double* xyz_list, double* gauss){
	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*NDOF2. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ dh/dx          0      ]
	 *          [   0           dh/dy   ]
	 *          [ 1/2*dh/dy  1/2*dh/dx  ]
	 *          [ 1/2*dh/dz      0      ]
	 *          [  0         1/2*dh/dz  ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 5x(NDOF2*numgrids)
	 */

	int i;
	const int numgrids=6;
	const int NDOF3=3;
	const int NDOF2=2;

	double dh1dh6[NDOF3][numgrids];

	/*Get dh1dh6 in actual coordinate system: */
	GetNodalFunctionsP1Derivatives(&dh1dh6[0][0],xyz_list, gauss);

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=dh1dh6[0][i]; 
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=0.0;

		*(B+NDOF2*numgrids*1+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=dh1dh6[1][i];

		*(B+NDOF2*numgrids*2+NDOF2*i)=(float).5*dh1dh6[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=(float).5*dh1dh6[0][i]; 

		*(B+NDOF2*numgrids*3+NDOF2*i)=(float).5*dh1dh6[2][i]; 
		*(B+NDOF2*numgrids*3+NDOF2*i+1)=0.0;

		*(B+NDOF2*numgrids*4+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*4+NDOF2*i+1)=(float).5*dh1dh6[2][i]; 
	}

}
/*}}}*/
/*FUNCTION PentaRef::GetBprimePattyn {{{1*/
void PentaRef::GetBprimePattyn(double* B, double* xyz_list, double* gauss_coord){
	/*Compute B  prime matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*NDOF2. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ 2*dh/dx     dh/dy   ]
	 *                [   dh/dx    2*dh/dy  ]
	 *                [ dh/dy      dh/dx    ]
	 *                [ dh/dz         0     ]
	 *                [  0         dh/dz    ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 5x(NDOF2*numgrids)
	 */

	int i;
	const int NDOF3=3;
	const int NDOF2=2;
	const int numgrids=6;

	double dh1dh6[NDOF3][numgrids];

	/*Get dh1dh6 in actual coordinate system: */
	GetNodalFunctionsP1Derivatives(&dh1dh6[0][0],xyz_list, gauss_coord);

	/*Build BPrime: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=2.0*dh1dh6[0][i]; 
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=dh1dh6[1][i];

		*(B+NDOF2*numgrids*1+NDOF2*i)=dh1dh6[0][i];
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=2.0*dh1dh6[1][i];

		*(B+NDOF2*numgrids*2+NDOF2*i)=dh1dh6[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=dh1dh6[0][i]; 

		*(B+NDOF2*numgrids*3+NDOF2*i)=dh1dh6[2][i]; 
		*(B+NDOF2*numgrids*3+NDOF2*i+1)=0.0;

		*(B+NDOF2*numgrids*4+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*4+NDOF2*i+1)=dh1dh6[2][i]; 
	}
}
/*}}}*/
/*FUNCTION PentaRef::GetBStokes {{{1*/
void PentaRef::GetBStokes(double* B, double* xyz_list, double* gauss){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 3*DOFPERGRID. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 		Bi=[ dh/dx          0              0       0  ]
	 *					[   0           dh/dy           0       0  ]
	 *					[   0             0           dh/dy     0  ]
	 *					[ 1/2*dh/dy    1/2*dh/dx        0       0  ]
	 *					[ 1/2*dh/dz       0         1/2*dh/dx   0  ]
	 *					[   0          1/2*dh/dz    1/2*dh/dy   0  ]
	 *					[   0             0             0       h  ]
	 *					[ dh/dx         dh/dy         dh/dz     0  ]
	 *	where h is the interpolation function for grid i.
	 *	Same thing for Bb except the last column that does not exist.
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=4;

	double dh1dh7[calculationdof][numgrids+1];
	double l1l6[numgrids];


	/*Get dh1dh7 in actual coordinate system: */
	GetNodalFunctionsMINIDerivatives(&dh1dh7[0][0],xyz_list, gauss);
	GetNodalFunctionsP1(l1l6, gauss);

	/*Build B: */
	for (i=0;i<numgrids+1;i++){
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i)=dh1dh7[0][i]; //B[0][DOFPERGRID*i]=dh1dh6[0][i];
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+2)=dh1dh7[2][i];
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i)=(float).5*dh1dh7[1][i]; 
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+1)=(float).5*dh1dh7[0][i]; 
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i)=(float).5*dh1dh7[2][i];
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+2)=(float).5*dh1dh7[0][i];
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+1)=(float).5*dh1dh7[2][i];
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+2)=(float).5*dh1dh7[1][i];
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i)=dh1dh7[0][i];
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+2)=dh1dh7[2][i];
	}

	for (i=0;i<numgrids;i++){ //last column not for the bubble function
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+3)=l1l6[i];
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+3)=0;
	}

}
/*}}}*/
/*FUNCTION PentaRef::GetBprimeStokes {{{1*/
void PentaRef::GetBprimeStokes(double* B_prime, double* xyz_list, double* gauss){
	/*	Compute B'  matrix. B'=[B1' B2' B3' B4' B5' B6' Bb'] where Bi' is of size 3*NDOF2. 
	 *	For grid i, Bi' can be expressed in the actual coordinate system
	 *	by: 
	 *				Bi'=[  dh/dx   0          0       0]
	 *					 [   0      dh/dy      0       0]
	 *					 [   0      0         dh/dz    0]
	 *					 [  dh/dy   dh/dx      0       0]
	 *					 [  dh/dz   0        dh/dx     0]
	 *					 [   0      dh/dz    dh/dy     0]
	 *					 [  dh/dx   dh/dy    dh/dz     0]
	 *					 [   0      0          0       h]
	 *	where h is the interpolation function for grid i.
	 *
	 * 	Same thing for the bubble fonction except that there is no fourth column
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=4;

	double dh1dh7[calculationdof][numgrids+1];
	double l1l6[numgrids];

	/*Get dh1dh7 in actual coordinate system: */
	GetNodalFunctionsMINIDerivatives(&dh1dh7[0][0],xyz_list, gauss);

	GetNodalFunctionsP1(l1l6, gauss);

	/*B_primeuild B_prime: */
	for (i=0;i<numgrids+1;i++){
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i)=dh1dh7[0][i]; //B_prime[0][DOFPERGRID*i]=dh1dh6[0][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+2)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+2)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+2)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i)=dh1dh7[1][i]; 
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+1)=dh1dh7[0][i]; 
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+2)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+2)=dh1dh7[0][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+1)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+2)=dh1dh7[1][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i)=dh1dh7[0][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+2)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+2)=0;
	}

	for (i=0;i<numgrids;i++){ //last column not for the bubble function
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+3)=l1l6[i];
	}

}
/*}}}*/
/*FUNCTION PentaRef::GetBArtdiff {{{1*/
void PentaRef::GetBArtdiff(double* B_artdiff, double* xyz_list, double* gauss){
	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_artdiff=[ dh/dx ]
	 *                 [ dh/dy ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 2x(DOFPERGRID*numgrids)
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double dh1dh6[calculationdof][numgrids];

	/*Get dh1dh6 in actual coordinates system : */
	GetNodalFunctionsP1Derivatives(&dh1dh6[0][0],xyz_list,gauss);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(B_artdiff+DOFPERGRID*numgrids*0+DOFPERGRID*i)=dh1dh6[0][i]; 
		*(B_artdiff+DOFPERGRID*numgrids*1+DOFPERGRID*i)=dh1dh6[1][i]; 
	}
}
/*}}}*/
/*FUNCTION PentaRef::GetBAdvec{{{1*/
void PentaRef::GetBAdvec(double* B_advec, double* xyz_list, double* gauss){
	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_advec =[ h ]
	 *                 [ h ]
	 *                 [ h ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(DOFPERGRID*numgrids)
	 */

	int i;
	int calculationdof=3;
	int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double l1l6[6];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsP1(l1l6, gauss);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(B_advec+DOFPERGRID*numgrids*0+DOFPERGRID*i)=l1l6[i]; 
		*(B_advec+DOFPERGRID*numgrids*1+DOFPERGRID*i)=l1l6[i]; 
		*(B_advec+DOFPERGRID*numgrids*2+DOFPERGRID*i)=l1l6[i]; 
	}
}
/*}}}*/
/*FUNCTION PentaRef::GetBConduct{{{1*/
void PentaRef::GetBConduct(double* B_conduct, double* xyz_list, double* gauss){
	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_conduct=[ dh/dx ]
	 *                  [ dh/dy ]
	 *                  [ dh/dz ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(DOFPERGRID*numgrids)
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double dh1dh6[calculationdof][numgrids];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsP1Derivatives(&dh1dh6[0][0],xyz_list,gauss);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(B_conduct+DOFPERGRID*numgrids*0+DOFPERGRID*i)=dh1dh6[0][i]; 
		*(B_conduct+DOFPERGRID*numgrids*1+DOFPERGRID*i)=dh1dh6[1][i]; 
		*(B_conduct+DOFPERGRID*numgrids*2+DOFPERGRID*i)=dh1dh6[2][i]; 
	}
}
/*}}}*/
/*FUNCTION PentaRef::GetBVert{{{1*/
void PentaRef::GetBVert(double* B, double* xyz_list, double* gauss){
	/*	Compute B  matrix. B=[dh1/dz dh2/dz dh3/dz dh4/dz dh5/dz dh6/dz];
		where hi is the interpolation function for grid i.*/

	int i;
	const int NDOF3=3;
	const int numgrids=6;
	double dh1dh6[NDOF3][numgrids];

	/*Get dh1dh6 in actual coordinate system: */
	GetNodalFunctionsP1Derivatives(&dh1dh6[0][0],xyz_list, gauss);

	/*Build B: */
	for (i=0;i<numgrids;i++){
		B[i]=dh1dh6[2][i];  
	}

}
/*}}}*/
/*FUNCTION PentaRef::GetBprimeAdvec{{{1*/
void PentaRef::GetBprimeAdvec(double* Bprime_advec, double* xyz_list, double* gauss){
	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Biprime_advec=[ dh/dx ]
	 *                     [ dh/dy ]
	 *                     [ dh/dz ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(DOFPERGRID*numgrids)
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double dh1dh6[calculationdof][numgrids];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsP1Derivatives(&dh1dh6[0][0],xyz_list,gauss);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(Bprime_advec+DOFPERGRID*numgrids*0+DOFPERGRID*i)=dh1dh6[0][i]; 
		*(Bprime_advec+DOFPERGRID*numgrids*1+DOFPERGRID*i)=dh1dh6[1][i]; 
		*(Bprime_advec+DOFPERGRID*numgrids*2+DOFPERGRID*i)=dh1dh6[2][i]; 
	}
}
/*}}}*/
/*FUNCTION PentaRef::GetBprimeVert{{{1*/
void PentaRef::GetBprimeVert(double* B, double* xyz_list, double* gauss){
	/* Compute Bprime  matrix. Bprime=[L1 L2 L3 L4 L5 L6] where Li is the nodal function for grid i*/

	GetNodalFunctionsP1(B, gauss);

}
/*}}}*/
/*FUNCTION PentaRef::GetLStokes {{{1*/
void PentaRef::GetLStokes(double* LStokes, double* gauss_tria){
	/*
	 * Compute L  matrix. L=[L1 L2 L3] where Li is square and of size numdof. 
	 * For grid i, Li can be expressed in the actual coordinate system
	 * by: 
	 *       Li=[ h    0    0   0]
	 *	 	      [ 0    h    0   0]
	 *		      [ 0    0    h   0]
	 *		      [ 0    0    h   0]
	 *	 	      [ h    0    0   0]
	 *	 	      [ 0    h    0   0]
	 *	 	      [ h    0    0   0]
	 *	 	      [ 0    h    0   0]
	 *		      [ 0    0    h   0]
	 *		      [ 0    0    h   0]
	 *		      [ 0    0    h   0]
	 *	 	      [ h    0    0   0]
	 *	 	      [ 0    h    0   0]
	 *		      [ 0    0    h   0]
	 * where h is the interpolation function for grid i.
	 */

	int i;
	const int numgrids2d=3;
	int num_dof=4;

	double l1l2l3[numgrids2d];


	/*Get l1l2l3 in actual coordinate system: */
	l1l2l3[0]=gauss_tria[0];
	l1l2l3[1]=gauss_tria[1];
	l1l2l3[2]=gauss_tria[2];

	/*Build LStokes: */
	for (i=0;i<3;i++){
		*(LStokes+num_dof*numgrids2d*0+num_dof*i)=l1l2l3[i]; //LStokes[0][NDOF2*i]=dh1dh3[0][i];
		*(LStokes+num_dof*numgrids2d*0+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*0+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*0+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*1+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*1+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*1+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*1+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*2+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*2+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*2+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*2+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*3+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*3+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*3+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*3+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*4+num_dof*i)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*4+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*4+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*4+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*5+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*5+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*5+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*5+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*6+num_dof*i)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*6+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*6+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*6+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*7+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*7+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*7+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*7+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*8+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*8+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*8+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*8+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*9+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*9+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*9+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*9+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*10+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*10+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*10+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*10+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*11+num_dof*i)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*11+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*11+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*11+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*12+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*12+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*12+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*12+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*13+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*13+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*13+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*13+num_dof*i+3)=0;

	}
}
/*}}}*/
/*FUNCTION PentaRef::GetLprimeStokes {{{1*/
void PentaRef::GetLprimeStokes(double* LprimeStokes, double* xyz_list, double* gauss_tria, double* gauss){

	/*
	 * Compute Lprime  matrix. Lprime=[Lp1 Lp2 Lp3] where Lpi is square and of size numdof. 
	 * For grid i, Lpi can be expressed in the actual coordinate system
	 * by: 
	 *       Lpi=[ h    0    0   0]
	 *		       [ 0    h    0   0]
	 *		       [ h    0    0   0]
	 *		       [ 0    h    0   0]
	 *		       [ 0    0    h   0]
	 *		       [ 0    0    h   0]
	 *		       [ 0    0  dh/dz 0]
	 *		       [ 0    0  dh/dz 0]
	 *		       [ 0    0  dh/dz 0]
	 *		       [dh/dz 0  dh/dx 0]
	 *		       [ 0 dh/dz dh/dy 0]
	 *           [ 0    0    0   h]
	 *           [ 0    0    0   h]
	 *           [ 0    0    0   h]
	 * where h is the interpolation function for grid i.
	 */
	int i;
	const int numgrids2d=3;
	int num_dof=4;

	double l1l2l3[numgrids2d];
	double dh1dh6[3][6];

	/*Get l1l2l3 in actual coordinate system: */
	l1l2l3[0]=gauss_tria[0];
	l1l2l3[1]=gauss_tria[1];
	l1l2l3[2]=gauss_tria[2];

	GetNodalFunctionsP1Derivatives(&dh1dh6[0][0],xyz_list,gauss);

	/*Build LprimeStokes: */
	for (i=0;i<3;i++){
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i)=l1l2l3[i]; //LprimeStokes[0][NDOF2*i]=dh1dh3[0][i];
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i+1)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i+1)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i+2)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i+2)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i+2)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i+2)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i+2)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i+2)=dh1dh6[0][i];
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i+1)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i+2)=dh1dh6[1][i];
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i+3)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i+3)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i+3)=l1l2l3[i];

	}
}
/*}}}*/
/*FUNCTION PentaRef::GetJacobian {{{1*/
void PentaRef::GetJacobian(double* J, double* xyz_list,double* gauss){

	const int NDOF3=3;
	int i,j;

	/*The Jacobian is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double A1,A2,A3; //area coordinates
	double xi,eta,zi; //parametric coordinates

	double x1,x2,x3,x4,x5,x6;
	double y1,y2,y3,y4,y5,y6;
	double z1,z2,z3,z4,z5,z6;

	/*Figure out xi,eta and zi (parametric coordinates), for this gaussian point: */
	A1=gauss[0];
	A2=gauss[1];
	A3=gauss[2];

	xi=A2-A1;
	eta=SQRT3*A3;
	zi=gauss[3];

	x1=*(xyz_list+3*0+0);
	x2=*(xyz_list+3*1+0);
	x3=*(xyz_list+3*2+0);
	x4=*(xyz_list+3*3+0);
	x5=*(xyz_list+3*4+0);
	x6=*(xyz_list+3*5+0);

	y1=*(xyz_list+3*0+1);
	y2=*(xyz_list+3*1+1);
	y3=*(xyz_list+3*2+1);
	y4=*(xyz_list+3*3+1);
	y5=*(xyz_list+3*4+1);
	y6=*(xyz_list+3*5+1);

	z1=*(xyz_list+3*0+2);
	z2=*(xyz_list+3*1+2);
	z3=*(xyz_list+3*2+2);
	z4=*(xyz_list+3*3+2);
	z5=*(xyz_list+3*4+2);
	z6=*(xyz_list+3*5+2);

	*(J+NDOF3*0+0)=0.25*(x1-x2-x4+x5)*zi+0.25*(-x1+x2-x4+x5);
	*(J+NDOF3*1+0)=SQRT3/12.0*(x1+x2-2*x3-x4-x5+2*x6)*zi+SQRT3/12.0*(-x1-x2+2*x3-x4-x5+2*x6);
	*(J+NDOF3*2+0)=SQRT3/12.0*(x1+x2-2*x3-x4-x5+2*x6)*eta+1/4*(x1-x2-x4+x5)*xi +0.25*(-x1+x5-x2+x4);

	*(J+NDOF3*0+1)=0.25*(y1-y2-y4+y5)*zi+0.25*(-y1+y2-y4+y5);
	*(J+NDOF3*1+1)=SQRT3/12.0*(y1+y2-2*y3-y4-y5+2*y6)*zi+SQRT3/12.0*(-y1-y2+2*y3-y4-y5+2*y6);
	*(J+NDOF3*2+1)=SQRT3/12.0*(y1+y2-2*y3-y4-y5+2*y6)*eta+0.25*(y1-y2-y4+y5)*xi+0.25*(y4-y1+y5-y2);

	*(J+NDOF3*0+2)=0.25*(z1-z2-z4+z5)*zi+0.25*(-z1+z2-z4+z5);
	*(J+NDOF3*1+2)=SQRT3/12.0*(z1+z2-2*z3-z4-z5+2*z6)*zi+SQRT3/12.0*(-z1-z2+2*z3-z4-z5+2*z6);
	*(J+NDOF3*2+2)=SQRT3/12.0*(z1+z2-2*z3-z4-z5+2*z6)*eta+0.25*(z1-z2-z4+z5)*xi+0.25*(-z1+z5-z2+z4);

}
/*}}}*/
/*FUNCTION PentaRef::GetJacobianDeterminant {{{1*/
void PentaRef::GetJacobianDeterminant(double*  Jdet, double* xyz_list,double* gauss){
	/*On a penta, Jacobian varies according to coordinates. We need to get the Jacobian, and take 
	 * the determinant of it: */
	double J[3][3];

	/*Get Jacobian*/
	GetJacobian(&J[0][0],xyz_list,gauss);

	/*Get Determinant*/
	Matrix3x3Determinant(Jdet,&J[0][0]);
	if(*Jdet<0) ISSMERROR("negative jacobian determinant!");

}
/*}}}*/
/*FUNCTION PentaRef::GetJacobianInvert {{{1*/
void PentaRef::GetJacobianInvert(double* Jinv, double* xyz_list,double* gauss){

	/*Jacobian*/
	double J[3][3];

	/*Call Jacobian routine to get the jacobian:*/
	GetJacobian(&J[0][0], xyz_list, gauss);

	/*Invert Jacobian matrix: */
	Matrix3x3Invert(Jinv,&J[0][0]);
}
/*}}}*/
/*FUNCTION PentaRef::GetNodalFunctionsMINI{{{1*/
void PentaRef::GetNodalFunctionsMINI(double* l1l7, double* gauss){
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	l1l7[0]=gauss[0]*(1.0-gauss[3])/2.0;
	l1l7[1]=gauss[1]*(1.0-gauss[3])/2.0;
	l1l7[2]=gauss[2]*(1.0-gauss[3])/2.0;
	l1l7[3]=gauss[0]*(1.0+gauss[3])/2.0;
	l1l7[4]=gauss[1]*(1.0+gauss[3])/2.0;
	l1l7[5]=gauss[2]*(1.0+gauss[3])/2.0;
	l1l7[6]=27*gauss[0]*gauss[1]*gauss[2]*(1.0+gauss[3])*(1.0-gauss[3]);

}
/*}}}*/
/*FUNCTION PentaRef::GetNodalFunctionsMINIDerivatives{{{1*/
void PentaRef::GetNodalFunctionsMINIDerivatives(double* dh1dh7,double* xyz_list, double* gauss){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * actual coordinate system): */

	int       i;
	const int numgrids = 7;
	double    dh1dh7_ref[3][numgrids];
	double    Jinv[3][3];

	/*Get derivative values with respect to parametric coordinate system: */
	GetNodalFunctionsMINIDerivativesReference(&dh1dh7_ref[0][0], gauss); 

	/*Get Jacobian invert: */
	GetJacobianInvert(&Jinv[0][0], xyz_list, gauss);

	/*Build dh1dh6: 
	 *
	 * [dhi/dx]= Jinv'*[dhi/dr]
	 * [dhi/dy]        [dhi/ds]
	 * [dhi/dz]        [dhi/dzeta]
	 */

	for (i=0;i<numgrids;i++){
		*(dh1dh7+numgrids*0+i)=Jinv[0][0]*dh1dh7_ref[0][i]+Jinv[0][1]*dh1dh7_ref[1][i]+Jinv[0][2]*dh1dh7_ref[2][i];
		*(dh1dh7+numgrids*1+i)=Jinv[1][0]*dh1dh7_ref[0][i]+Jinv[1][1]*dh1dh7_ref[1][i]+Jinv[1][2]*dh1dh7_ref[2][i];
		*(dh1dh7+numgrids*2+i)=Jinv[2][0]*dh1dh7_ref[0][i]+Jinv[2][1]*dh1dh7_ref[1][i]+Jinv[2][2]*dh1dh7_ref[2][i];
	}

}
/*}}}*/
/*FUNCTION PentaRef::GetNodalFunctionsMINIDerivativesReference{{{1*/
void PentaRef::GetNodalFunctionsMINIDerivativesReference(double* dl1dl7,double* gauss){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. */

	int    numgrids=7; //six plus bubble grids

	double r=gauss[1]-gauss[0];
	double s=-3.0/SQRT3*(gauss[0]+gauss[1]-2.0/3.0);
	double zeta=gauss[3];

	/*First nodal function: */
	*(dl1dl7+numgrids*0+0)=-0.5*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*1+0)=-SQRT3/6.0*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*2+0)=-0.5*(-0.5*r-SQRT3/6.0*s+ONETHIRD);

	/*Second nodal function: */
	*(dl1dl7+numgrids*0+1)=0.5*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*1+1)=-SQRT3/6.0*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*2+1)=-0.5*(0.5*r-SQRT3/6.0*s+ONETHIRD);

	/*Third nodal function: */
	*(dl1dl7+numgrids*0+2)=0;
	*(dl1dl7+numgrids*1+2)=SQRT3/3.0*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*2+2)=-0.5*(SQRT3/3.0*s+ONETHIRD);

	/*Fourth nodal function: */
	*(dl1dl7+numgrids*0+3)=-0.5*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*1+3)=-SQRT3/6.0*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*2+3)=0.5*(-0.5*r-SQRT3/6.0*s+ONETHIRD);

	/*Fith nodal function: */
	*(dl1dl7+numgrids*0+4)=0.5*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*1+4)=-SQRT3/6.0*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*2+4)=0.5*(0.5*r-SQRT3/6.0*s+ONETHIRD);

	/*Sixth nodal function: */
	*(dl1dl7+numgrids*0+5)=0;
	*(dl1dl7+numgrids*1+5)=SQRT3/3.0*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*2+5)=0.5*(SQRT3/3.0*s+ONETHIRD);

	/*Seventh nodal function: */
	*(dl1dl7+numgrids*0+6)=9.0/2.0*r*(1.0+zeta)*(zeta-1.0)*(SQRT3*s+1.0);
	*(dl1dl7+numgrids*1+6)=9.0/4.0*(1+zeta)*(1-zeta)*(SQRT3*pow(s,2.0)-2.0*s-SQRT3*pow(r,2.0));
	*(dl1dl7+numgrids*2+6)=27*gauss[0]*gauss[1]*gauss[2]*(-2.0*zeta);

}
/*}}}*/
/*FUNCTION PentaRef::GetNodalFunctionsP1 {{{1*/
void PentaRef::GetNodalFunctionsP1(double* l1l6, double* gauss){
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	l1l6[0]=gauss[0]*(1-gauss[3])/2.0;
	l1l6[1]=gauss[1]*(1-gauss[3])/2.0;
	l1l6[2]=gauss[2]*(1-gauss[3])/2.0;
	l1l6[3]=gauss[0]*(1+gauss[3])/2.0;
	l1l6[4]=gauss[1]*(1+gauss[3])/2.0;
	l1l6[5]=gauss[2]*(1+gauss[3])/2.0;

}
/*}}}*/
/*FUNCTION PentaRef::GetNodalFunctionsP1Derivatives {{{1*/
void PentaRef::GetNodalFunctionsP1Derivatives(double* dh1dh6,double* xyz_list, double* gauss){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * actual coordinate system): */
	int       i;
	const int NDOF3    = 3;
	const int numgrids = 6;
	double    dh1dh6_ref[NDOF3][numgrids];
	double    Jinv[NDOF3][NDOF3];

	/*Get derivative values with respect to parametric coordinate system: */
	GetNodalFunctionsP1DerivativesReference(&dh1dh6_ref[0][0], gauss); 

	/*Get Jacobian invert: */
	GetJacobianInvert(&Jinv[0][0], xyz_list, gauss);

	/*Build dh1dh3: 
	 *
	 * [dhi/dx]= Jinv*[dhi/dr]
	 * [dhi/dy]       [dhi/ds]
	 * [dhi/dz]       [dhi/dn]
	 */

	for (i=0;i<numgrids;i++){
		*(dh1dh6+numgrids*0+i)=Jinv[0][0]*dh1dh6_ref[0][i]+Jinv[0][1]*dh1dh6_ref[1][i]+Jinv[0][2]*dh1dh6_ref[2][i];
		*(dh1dh6+numgrids*1+i)=Jinv[1][0]*dh1dh6_ref[0][i]+Jinv[1][1]*dh1dh6_ref[1][i]+Jinv[1][2]*dh1dh6_ref[2][i];
		*(dh1dh6+numgrids*2+i)=Jinv[2][0]*dh1dh6_ref[0][i]+Jinv[2][1]*dh1dh6_ref[1][i]+Jinv[2][2]*dh1dh6_ref[2][i];
	}

}
/*}}}*/
/*FUNCTION PentaRef::GetNodalFunctionsP1DerivativesReference {{{1*/
void PentaRef::GetNodalFunctionsP1DerivativesReference(double* dl1dl6,double* gauss){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. Those values vary along xi,eta,z */

	const int numgrids=6;
	double A1,A2,A3,z;

	A1=gauss[0]; //first area coordinate value. In term of xi and eta: A1=(1-xi)/2-eta/(2*SQRT3);
	A2=gauss[1]; //second area coordinate value In term of xi and eta: A2=(1+xi)/2-eta/(2*SQRT3);
	A3=gauss[2]; //third area coordinate value  In term of xi and eta: A3=y/SQRT3;
	z=gauss[3]; //fourth vertical coordinate value. Corresponding nodal function: (1-z)/2 and (1+z)/2


	/*First nodal function derivatives. The corresponding nodal function is N=A1*(1-z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+0)=-0.5*(1.0-z)/2.0;
	*(dl1dl6+numgrids*1+0)=-0.5/SQRT3*(1.0-z)/2.0;
	*(dl1dl6+numgrids*2+0)=-0.5*A1;

	/*Second nodal function: The corresponding nodal function is N=A2*(1-z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+1)=0.5*(1.0-z)/2.0;
	*(dl1dl6+numgrids*1+1)=-0.5/SQRT3*(1.0-z)/2.0;
	*(dl1dl6+numgrids*2+1)=-0.5*A2;

	/*Third nodal function: The corresponding nodal function is N=A3*(1-z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+2)=0.0;
	*(dl1dl6+numgrids*1+2)=1.0/SQRT3*(1.0-z)/2.0;
	*(dl1dl6+numgrids*2+2)=-0.5*A3;

	/*Fourth nodal function: The corresponding nodal function is N=A1*(1+z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+3)=-0.5*(1.0+z)/2.0;
	*(dl1dl6+numgrids*1+3)=-0.5/SQRT3*(1.0+z)/2.0;
	*(dl1dl6+numgrids*2+3)=0.5*A1;

	/*Fifth nodal function: The corresponding nodal function is N=A2*(1+z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+4)=0.5*(1.0+z)/2.0;
	*(dl1dl6+numgrids*1+4)=-0.5/SQRT3*(1.0+z)/2.0;
	*(dl1dl6+numgrids*2+4)=0.5*A2;

	/*Sixth nodal function: The corresponding nodal function is N=A3*(1+z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+5)=0.0;
	*(dl1dl6+numgrids*1+5)=1.0/SQRT3*(1.0+z)/2.0;
	*(dl1dl6+numgrids*2+5)=0.5*A3;
}
/*}}}*/
/*FUNCTION PentaRef::GetParameterValue{{{1*/
void PentaRef::GetParameterValue(double* pvalue,double* plist,double* gauss){
	/*P1 interpolation on Gauss point*/

	/*intermediary*/
	double l1l6[6];

	/*nodal functions: */
	GetNodalFunctionsP1(&l1l6[0],gauss);

	/*Assign output pointers:*/
	*pvalue=l1l6[0]*plist[0]+l1l6[1]*plist[1]+l1l6[2]*plist[2]+l1l6[3]*plist[3]+l1l6[4]*plist[4]+l1l6[5]*plist[5];

}
/*}}}*/
/*FUNCTION PentaRef::GetParameterDerivativeValue{{{1*/
void PentaRef::GetParameterDerivativeValue(double* p, double* plist,double* xyz_list, double* gauss){
	/*From grid values of parameter p (p_list[0], p_list[1], p_list[2], p_list[3], p_list[4] and p_list[4]), return parameter derivative value at gaussian point specified by gauss_coord:
	 *   dp/dx=p_list[0]*dh1/dx+p_list[1]*dh2/dx+p_list[2]*dh3/dx+p_list[3]*dh4/dx+p_list[4]*dh5/dx+p_list[5]*dh6/dx;
	 *   dp/dy=p_list[0]*dh1/dy+p_list[1]*dh2/dy+p_list[2]*dh3/dy+p_list[3]*dh4/dy+p_list[4]*dh5/dy+p_list[5]*dh6/dy;
	 *   dp/dz=p_list[0]*dh1/dz+p_list[1]*dh2/dz+p_list[2]*dh3/dz+p_list[3]*dh4/dz+p_list[4]*dh5/dz+p_list[5]*dh6/dz;
	 *
	 *   p is a vector of size 3x1 already allocated.
	 */
	double dh1dh6[3][6];

	/*Get nodal funnctions derivatives in actual coordinate system: */
	GetNodalFunctionsP1Derivatives(&dh1dh6[0][0],xyz_list, gauss);

	/*Assign output*/
	p[0]=plist[0]*dh1dh6[0][0]+plist[1]*dh1dh6[0][1]+plist[2]*dh1dh6[0][2]+plist[3]*dh1dh6[0][3]+plist[4]*dh1dh6[0][4]+plist[5]*dh1dh6[0][5];
	p[1]=plist[0]*dh1dh6[1][0]+plist[1]*dh1dh6[1][1]+plist[2]*dh1dh6[1][2]+plist[3]*dh1dh6[1][3]+plist[4]*dh1dh6[1][4]+plist[5]*dh1dh6[1][5];
	p[2]=plist[0]*dh1dh6[2][0]+plist[1]*dh1dh6[2][1]+plist[2]*dh1dh6[2][2]+plist[3]*dh1dh6[2][3]+plist[4]*dh1dh6[2][4]+plist[5]*dh1dh6[2][5];

}
/*}}}*/
