/*!\file Penpair.c
 * \brief: implementation of the Penpair object
 */


#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../include/include.h"
#include "../../shared/shared.h"
		
/*Penpair constructors and destructor*/
/*FUNCTION Penpair::constructor {{{1*/
Penpair::Penpair(){

	this->hnodes=NULL;
	this->parameters=NULL;
	return;
}
/*}}}1*/
/*FUNCTION Penpair::creation {{{1*/
Penpair::Penpair(int penpair_id, int* penpair_node_ids,int in_analysis_type){
	
	this->id=penpair_id;
	this->analysis_type=in_analysis_type;
	this->hnodes=new Hook(penpair_node_ids,2);
	this->parameters=NULL;
	
	return;
}
/*}}}1*/
/*FUNCTION Penpair::destructor {{{1*/
Penpair::~Penpair(){
	delete hnodes;
	return;
}
/*}}}1*/

/*Object virtual functions definitions:*/
/*FUNCTION Penpair::Echo {{{1*/
void Penpair::Echo(void){

	int i;

	printf("Penpair:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumAsString(analysis_type));
	hnodes->Echo();
	
	return;
}
/*}}}1*/
/*FUNCTION Penpair::DeepEcho {{{1*/
void Penpair::DeepEcho(void){

	printf("Penpair:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumAsString(analysis_type));
	hnodes->DeepEcho();

	return;
}		
/*}}}1*/
/*FUNCTION Penpair::Id {{{1*/
int    Penpair::Id(void){ return id; }
/*}}}1*/
/*FUNCTION Penpair::MyRank {{{1*/
int    Penpair::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}1*/
/*FUNCTION Penpair::Marshall {{{1*/
void  Penpair::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Penpair: */
	enum_type=PenpairEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Penpair data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&analysis_type,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);

	/*Marshall hooks*/
	hnodes->Marshall(&marshalled_dataset);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}1*/
/*FUNCTION Penpair::MarshallSize {{{1*/
int   Penpair::MarshallSize(){

	return sizeof(id)+
		+sizeof(analysis_type)
		+hnodes->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}1*/
/*FUNCTION Penpair::Demarshall {{{1*/
void  Penpair::Demarshall(char** pmarshalled_dataset){

	int i;
	char* marshalled_dataset=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&analysis_type,marshalled_dataset,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);

	/*demarshall hooks: */
	hnodes=new Hook(); hnodes->Demarshall(&marshalled_dataset);

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}1*/
/*FUNCTION Penpair::Enum {{{1*/
int Penpair::Enum(void){

	return PenpairEnum;
}
/*}}}1*/
/*FUNCTION Penpair::copy {{{1*/
Object* Penpair::copy() {
	
	Penpair* penpair=NULL;

	penpair=new Penpair();

	/*copy fields: */
	penpair->id=this->id;
	penpair->analysis_type=this->analysis_type;

	/*now deal with hooks and objects: */
	penpair->hnodes=(Hook*)this->hnodes->copy();

	/*point parameters: */
	penpair->parameters=this->parameters;

	return penpair;

}
/*}}}*/
		
/*Load virtual functions definitions:*/
/*FUNCTION Penpair::Configure {{{1*/
void  Penpair::Configure(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes->configure(nodesin);
	/*point parameters to real dataset: */
	this->parameters=parametersin;

}
/*}}}1*/
/*FUNCTION Penpair::SetCurrentConfiguration {{{1*/
void  Penpair::SetCurrentConfiguration(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes->configure(nodesin);

	/*point parameters to real dataset: */
	this->parameters=parametersin;
}
/*}}}1*/
/*FUNCTION Penpair::CreateKMatrix {{{1*/
void  Penpair::CreateKMatrix(Mat Kgg){
	/*If you code this piece, don't forget that a penalty will be inactive if it is dealing with clone nodes*/
	/*No loads applied, do nothing: */
	return;

}
/*}}}1*/
/*FUNCTION Penpair::CreatePVector {{{1*/
void  Penpair::CreatePVector(Vec pg){

	/*No loads applied, do nothing: */
	return;

}
/*}}}1*/
/*FUNCTION Penpair::PenaltyCreateKMatrix {{{1*/
void  Penpair::PenaltyCreateKMatrix(Mat Kgg,double kmax){
	int analysis_type;

	/*Retrieve parameters: */
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	if (analysis_type==DiagnosticHorizAnalysisEnum){
		PenaltyCreateKMatrixDiagnosticHoriz(Kgg,kmax);
	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));
	}
}
/*}}}1*/
/*FUNCTION Penpair::PenaltyCreatePVector {{{1*/
void  Penpair::PenaltyCreatePVector(Vec pg,double kmax){
	/*No loads applied, do nothing: */
	return;
}
/*}}}1*/
/*FUNCTION Penpair::InAnalysis{{{1*/
bool Penpair::InAnalysis(int in_analysis_type){
	if (in_analysis_type==this->analysis_type)return true;
	else return false;
}
/*}}}*/

/*Update virtual functions definitions:*/
/*FUNCTION Penpair::InputUpdateFromConstant(double constant, int name) {{{1*/
void  Penpair::InputUpdateFromConstant(double constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Penpair::InputUpdateFromConstant(int constant, int name) {{{1*/
void  Penpair::InputUpdateFromConstant(int constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Penpair::InputUpdateFromConstant(bool constant, int name) {{{1*/
void  Penpair::InputUpdateFromConstant(bool constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/

/*Penpair management:*/
/*FUNCTION Penpair::PenaltyCreateKMatrixDiagnosticHoriz {{{1*/
void  Penpair::PenaltyCreateKMatrixDiagnosticHoriz(Mat Kgg,double kmax){
	
	const int numgrids=2;
	const int NDOF2=2;
	const int numdof=numgrids*NDOF2;
	int       doflist[numdof];
	int       numberofdofspernode;

	double Ke[4][4]={0.0};
	double penalty_offset;

	/*pointers: */
	Node** nodes=NULL;

	/*Get dof list: */
	GetDofList(&doflist[0],&numberofdofspernode);

	/*recover pointers: */
	nodes=(Node**)hnodes->deliverp();

	/*recover parameters: */
	parameters->FindParam(&penalty_offset,PenaltyOffsetEnum);

	//Create elementary matrix: add penalty to 
	Ke[0][0]=kmax*pow((double)10.0,penalty_offset);
	Ke[0][2]=-kmax*pow((double)10.0,penalty_offset);
	Ke[2][0]=-kmax*pow((double)10.0,penalty_offset);
	Ke[2][2]=kmax*pow((double)10.0,penalty_offset);

	Ke[1][1]=kmax*pow((double)10.0,penalty_offset);
	Ke[1][3]=-kmax*pow((double)10.0,penalty_offset);
	Ke[3][1]=-kmax*pow((double)10.0,penalty_offset);
	Ke[3][3]=kmax*pow((double)10.0,penalty_offset);
	
	/*Add Ke to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke,ADD_VALUES);
}
/*}}}1*/
/*FUNCTION Penpair::GetDofList {{{1*/
void  Penpair::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;

	/*pointers: */
	Node** nodes=NULL;

	/*recover pointers: */
	nodes=(Node**)hnodes->deliverp();

	/*Some checks for debugging*/
	ISSMASSERT(doflist);
	ISSMASSERT(pnumberofdofspernode);
	ISSMASSERT(nodes);

	/*Build doflist from nodes*/
	for(i=0;i<2;i++){
		nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
		for(j=0;j<numberofdofspernode;j++){
			doflist[i*numberofdofspernode+j]=doflist_per_node[j];
		}
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
