/*!\file TriaRef.c
 * \brief: implementation of the TriaRef object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION TriaRef::TriaRef(){{{1*/
TriaRef::TriaRef(){
	this->element_type_list=NULL;
}
/*}}}*/
/*FUNCTION TriaRef::TriaRef(int* types,int nummodels){{{1*/
TriaRef::TriaRef(const int nummodels){

	/*Only allocate pointer*/
	element_type_list=(int*)xmalloc(nummodels*sizeof(int));

}
/*}}}*/
/*FUNCTION TriaRef::~TriaRef(){{{1*/
TriaRef::~TriaRef(){
	xfree((void**)&element_type_list);
}
/*}}}*/

/*Management*/
/*FUNCTION TriaRef::SetElementType{{{1*/
void TriaRef::SetElementType(int type,int type_counter){

	ISSMASSERT(type==P1Enum || type==P1DGEnum);

	/*initialize element type*/
	this->element_type_list[type_counter]=type;
}
/*}}}*/

/*Reference Element numerics*/
/*FUNCTION TriaRef::GetBMacAyeal {{{1*/
void TriaRef::GetBMacAyeal(double* B, double* xyz_list, double* gauss){
	/*Compute B  matrix. B=[B1 B2 B3] where Bi is of size 3*NDOF2. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ dh/dx           0    ]
	 *          [   0           dh/dy  ]
	 *          [ 1/2*dh/dy  1/2*dh/dx ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(NDOF2*numgrids)
	 */

	int i;
	const int NDOF2=2;
	const int numgrids=3;

	double dh1dh3[NDOF2][numgrids];


	/*Get dh1dh2dh3 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh3[0][0],xyz_list,gauss);

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=dh1dh3[0][i]; //B[0][NDOF2*i]=dh1dh3[0][i];
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=0;
		*(B+NDOF2*numgrids*1+NDOF2*i)=0;
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=dh1dh3[1][i];
		*(B+NDOF2*numgrids*2+NDOF2*i)=(float).5*dh1dh3[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=(float).5*dh1dh3[0][i]; 
	}
}
/*}}}*/
/*FUNCTION TriaRef::GetBPrognostic{{{1*/
void TriaRef::GetBPrognostic(double* B_prog, double* xyz_list, double* gauss){
	/*Compute B  matrix. B=[B1 B2 B3] where Bi is of size 3*NDOF2. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ h ]
	 *          [ h ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B_prog has been allocated already, of size: 2x(NDOF1*numgrids)
	 */

	int i;
	const int NDOF1=1;
	const int numgrids=3;

	double l1l2l3[numgrids];


	/*Get dh1dh2dh3 in actual coordinate system: */
	GetNodalFunctions(&l1l2l3[0],gauss);

	/*Build B_prog: */
	for (i=0;i<numgrids;i++){
		*(B_prog+NDOF1*numgrids*0+NDOF1*i)=l1l2l3[i];
		*(B_prog+NDOF1*numgrids*1+NDOF1*i)=l1l2l3[i];
	}
}
/*}}}*/
/*FUNCTION TriaRef::GetBprimeMacAyeal {{{1*/
void TriaRef::GetBprimeMacAyeal(double* Bprime, double* xyz_list, double* gauss){

	/*Compute B'  matrix. B'=[B1' B2' B3'] where Bi' is of size 3*NDOF2. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_prime=[ 2*dh/dx    dh/dy ]
	 *                [   dh/dx  2*dh/dy ]
	 *                [   dh/dy    dh/dx ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B' has been allocated already, of size: 3x(NDOF2*numgrids)
	 */

	int i;
	const int NDOF2=2;
	const int numgrids=3;

	/*Same thing in the actual coordinate system: */
	double dh1dh3[NDOF2][numgrids];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsDerivatives(&dh1dh3[0][0],xyz_list,gauss);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(Bprime+NDOF2*numgrids*0+NDOF2*i)=2*dh1dh3[0][i]; 
		*(Bprime+NDOF2*numgrids*0+NDOF2*i+1)=dh1dh3[1][i]; 
		*(Bprime+NDOF2*numgrids*1+NDOF2*i)=dh1dh3[0][i]; 
		*(Bprime+NDOF2*numgrids*1+NDOF2*i+1)=2*dh1dh3[1][i]; 
		*(Bprime+NDOF2*numgrids*2+NDOF2*i)=dh1dh3[1][i]; 
		*(Bprime+NDOF2*numgrids*2+NDOF2*i+1)=dh1dh3[0][i]; 
	}
}
/*}}}*/
/*FUNCTION TriaRef::GetBprimePrognostic{{{1*/
void TriaRef::GetBprimePrognostic(double* Bprime_prog, double* xyz_list, double* gauss){
	/*Compute B'  matrix. B'=[B1' B2' B3'] where Bi' is of size 3*NDOF2. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_prime=[ dh/dx ]
	 *                [ dh/dy ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B' has been allocated already, of size: 3x(NDOF2*numgrids)
	 */

	int i;
	const int NDOF1=1;
	const int NDOF2=2;
	const int numgrids=3;

	/*Same thing in the actual coordinate system: */
	double dh1dh3[NDOF2][numgrids];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsDerivatives(&dh1dh3[0][0],xyz_list,gauss);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(Bprime_prog+NDOF1*numgrids*0+NDOF1*i)=dh1dh3[0][i]; 
		*(Bprime_prog+NDOF1*numgrids*1+NDOF1*i)=dh1dh3[1][i]; 
	}
}
/*}}}*/
/*FUNCTION TriaRef::GetL {{{1*/
void TriaRef::GetL(double* L, double* xyz_list, double* gauss,int numdof){
	/*Compute L  matrix. L=[L1 L2 L3] where Li is square and of size numdof. 
	 * For grid i, Li can be expressed in the actual coordinate system
	 * by: 
	 *       numdof=1: 
	 *                 Li=h;
	 *       numdof=2:
	 *                 Li=[ h   0 ]
	 *                    [ 0   h ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume L has been allocated already, of size: numgrids (numdof=1), or numdofx(numdof*numgrids) (numdof=2)
	 */

	int i;
	const int NDOF2=2;
	const int numgrids=3;

	double l1l2l3[3];


	/*Get l1l2l3 in actual coordinate system: */
	GetNodalFunctions(l1l2l3, gauss);

	/*Build L: */
	if(numdof==1){
		for (i=0;i<numgrids;i++){
			L[i]=l1l2l3[i]; 
		}
	}
	else{
		for (i=0;i<numgrids;i++){
			*(L+numdof*numgrids*0+numdof*i)=l1l2l3[i]; //L[0][NDOF2*i]=dh1dh3[0][i];
			*(L+numdof*numgrids*0+numdof*i+1)=0;
			*(L+numdof*numgrids*1+numdof*i)=0;
			*(L+numdof*numgrids*1+numdof*i+1)=l1l2l3[i];
		}
	}
}
/*}}}*/
/*FUNCTION TriaRef::GetJacobian {{{1*/
void TriaRef::GetJacobian(double* J, double* xyz_list,double* gauss){
	/*The Jacobian is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	const int NDOF2=2;
	const int numgrids=3;
	double x1,y1,x2,y2,x3,y3;

	x1=*(xyz_list+numgrids*0+0);
	y1=*(xyz_list+numgrids*0+1);
	x2=*(xyz_list+numgrids*1+0);
	y2=*(xyz_list+numgrids*1+1);
	x3=*(xyz_list+numgrids*2+0);
	y3=*(xyz_list+numgrids*2+1);


	*(J+NDOF2*0+0)=0.5*(x2-x1);
	*(J+NDOF2*1+0)=SQRT3/6.0*(2*x3-x1-x2);
	*(J+NDOF2*0+1)=0.5*(y2-y1);
	*(J+NDOF2*1+1)=SQRT3/6.0*(2*y3-y1-y2);
}
/*}}}*/
/*FUNCTION TriaRef::GetJacobianDeterminant2d {{{1*/
void TriaRef::GetJacobianDeterminant2d(double* Jdet, double* xyz_list,double* gauss){

	/*The Jacobian determinant is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double x1,x2,x3,y1,y2,y3;

	x1=*(xyz_list+3*0+0);
	y1=*(xyz_list+3*0+1);
	x2=*(xyz_list+3*1+0);
	y2=*(xyz_list+3*1+1);
	x3=*(xyz_list+3*2+0);
	y3=*(xyz_list+3*2+1);

	*Jdet=SQRT3/6.0*((x2-x1)*(y3-y1)-(y2-y1)*(x3-x1));

	if(Jdet<0){
		ISSMERROR("negative jacobian determinant!");
	}

}
/*}}}*/
/*FUNCTION TriaRef::GetJacobianDeterminant3d {{{1*/
void TriaRef::GetJacobianDeterminant3d(double*  Jdet, double* xyz_list,double* gauss){
	/*The Jacobian determinant is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double x1,x2,x3,y1,y2,y3,z1,z2,z3;

	x1=*(xyz_list+3*0+0);
	y1=*(xyz_list+3*0+1);
	z1=*(xyz_list+3*0+2);
	x2=*(xyz_list+3*1+0);
	y2=*(xyz_list+3*1+1);
	z2=*(xyz_list+3*1+2);
	x3=*(xyz_list+3*2+0);
	y3=*(xyz_list+3*2+1);
	z3=*(xyz_list+3*2+2);


	*Jdet=SQRT3/6.0*pow(pow(((y2-y1)*(z3-z1)-(z2-z1)*(y3-y1)),2.0)+pow(((z2-z1)*(x3-x1)-(x2-x1)*(z3-z1)),2.0)+pow(((x2-x1)*(y3-y1)-(y2-y1)*(x3-x1)),2.0),0.5);

	if(Jdet<0){
		ISSMERROR("negative jacobian determinant!");
	}

}
/*}}}*/
/*FUNCTION TriaRef::GetJacobianInvert {{{1*/
void TriaRef::GetJacobianInvert(double*  Jinv, double* xyz_list,double* gauss){

	double Jdet;
	const int NDOF2=2;
	const int numgrids=3;

	/*Call Jacobian routine to get the jacobian:*/
	GetJacobian(Jinv, xyz_list, gauss);

	/*Invert Jacobian matrix: */
	MatrixInverse(Jinv,NDOF2,NDOF2,NULL,0,&Jdet);

}
/*}}}*/
/*FUNCTION TriaRef::GetNodalFunctions {{{1*/
void TriaRef::GetNodalFunctions(double* l1l2l3, double* gauss){
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	l1l2l3[0]=gauss[0];
	l1l2l3[1]=gauss[1];
	l1l2l3[2]=gauss[2];

}
/*}}}*/
/*FUNCTION TriaRef::GetNodalFunctionsDerivatives {{{1*/
void TriaRef::GetNodalFunctionsDerivatives(double* dh1dh3,double* xyz_list, double* gauss){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * actual coordinate system): */
	int       i;
	const int NDOF2    = 2;
	const int numgrids = 3;
	double    dh1dh3_ref[NDOF2][numgrids];
	double    Jinv[NDOF2][NDOF2];

	/*Get derivative values with respect to parametric coordinate system: */
	GetNodalFunctionsDerivativesReference(&dh1dh3_ref[0][0], gauss); 

	/*Get Jacobian invert: */
	GetJacobianInvert(&Jinv[0][0], xyz_list, gauss);

	/*Build dh1dh3: 
	 *
	 * [dhi/dx]= Jinv*[dhi/dr]
	 * [dhi/dy]       [dhi/ds]
	 */
	for (i=0;i<numgrids;i++){
		dh1dh3[numgrids*0+i]=Jinv[0][0]*dh1dh3_ref[0][i]+Jinv[0][1]*dh1dh3_ref[1][i];
		dh1dh3[numgrids*1+i]=Jinv[1][0]*dh1dh3_ref[0][i]+Jinv[1][1]*dh1dh3_ref[1][i];
	}

}
/*}}}*/
/*FUNCTION TriaRef::GetNodalFunctionsDerivativesReference {{{1*/
void TriaRef::GetNodalFunctionsDerivativesReference(double* dl1dl3,double* gauss){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. */

	const int NDOF2=2;
	const int numgrids=3;

	/*First nodal function: */
	*(dl1dl3+numgrids*0+0)=-0.5; 
	*(dl1dl3+numgrids*1+0)=-1.0/(2.0*SQRT3);

	/*Second nodal function: */
	*(dl1dl3+numgrids*0+1)=0.5;
	*(dl1dl3+numgrids*1+1)=-1.0/(2.0*SQRT3);

	/*Third nodal function: */
	*(dl1dl3+numgrids*0+2)=0;
	*(dl1dl3+numgrids*1+2)=1.0/SQRT3;

}
/*}}}*/
