/*!\file:  ElementResultsToPatch
 * \brief: go through our finite elements, and see what results they have stored within. 
 * Then output them into serialized patch arrays, and add to results dataset
 */ 

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "../../Container/Container.h"
#include "../../io/io.h"
#include "../../objects/objects.h"
		
void ElementResultsToPatch(Elements* elements,  Nodes* nodes, Vertices* vertices, Loads* loads, Materials* materials, Parameters* parameters,DataSet* results,int step, double time){

	int i;

	/*output: */
	Patch* patch=NULL;

	/*intermediary: */
	int count;
	int numrows;
	int numvertices;
	int numnodes;
	int max_numvertices;
	int max_numnodes;
	int element_numvertices;
	int element_numrows;
	int element_numnodes;

	//Process results to be output in the correct units
	for(i=0;i<elements->Size();i++){
		Element* element=(Element*)elements->GetObjectByOffset(i);
		element->ProcessResultsUnits();
	}

	/*We are going to extract from the results within the elements, the desired results , and create a table 
	 * of patch information, that will hold, for each element that computed the result that 
	 * we desire, the enum_type of the result, the step and time, the id of the element, the interpolation type, the vertices ids, and the values 
	 * at the nodes (could be different from the vertices). This will be used for visualization purposes. 
	 * For example, we could build the following patch table, for velocities: 
	 * 
	 1. on a Beam element, Vx, at step 1, time .5, element id 1, interpolation type P0 (constant), vertices ids 1 and 2, one constant value 4.5
	 VxEnum 1  .5  1 P0  1 2       4.5 NaN  NaN
	 2. on a Tria element, Vz, at step 2, time .8, element id 2, interpolation type P1 (linear), vertices ids 1 3 and 4, with values at 3 nodes 4.5, 3.2, 2.5
	 VzEnum 2  .8  2 P1  1 3 4     4.5 3.2  2.5
	 * ... etc ...
	 *
	 * So what do we need to build the table: the maximum number of vertices included in the table, 
	 * and the maximum number of nodal values, as well as the number of rows. Once we have that, 
	 * we ask the elements to fill their own row in the table, by looping on the elememnts. 
	 *
	 * We will use the Patch object, which will store all of the information needed, and will be able 
	 * to output itself to disk on its own. See object/Patch.h for format of this object.*/
	
	/*First, determine maximum number of vertices, nodes, and number of results: */
	numrows=0;
	numvertices=0;
	numnodes=0;

	for(i=0;i<elements->Size();i++){
		Element* element=(Element*)elements->GetObjectByOffset(i);
		element->PatchSize(&element_numrows,&element_numvertices,&element_numnodes);

		numrows+=element_numrows;
		if(element_numvertices>numvertices)numvertices=element_numvertices;
		if(element_numnodes>numnodes)numnodes=element_numnodes;
	}

	#ifdef _PARALLEL_
	/*Synchronize across cluster, so as to not end up with different sizes for each patch on each cpu: */
	MPI_Reduce (&numvertices,&max_numvertices,1,MPI_INT,MPI_MAX,0,MPI_COMM_WORLD );
	MPI_Bcast(&max_numvertices,1,MPI_INT,0,MPI_COMM_WORLD);
	numvertices=max_numvertices;

	MPI_Reduce (&numnodes,&max_numnodes,1,MPI_INT,MPI_MAX,0,MPI_COMM_WORLD );
	MPI_Bcast(&max_numnodes,1,MPI_INT,0,MPI_COMM_WORLD);
	numnodes=max_numnodes;
	#endif

	/*Ok, initialize Patch object: */
	patch=new Patch(numrows,numvertices,numnodes);

	/*Now, go through elements, and fill the Patch object: */
	count=0;
	for(i=0;i<elements->Size();i++){
		Element* element=(Element*)elements->GetObjectByOffset(i);
		element->PatchFill(&count,patch);
	}

	/*Now, gather patch onto node 0, so that we do not dump to disk from every node: */
	patch->MPI_Gather();

	/*create result object and add to results dataset (if not empty): */
	if (patch->maxvertices && patch->maxnodes){
		results->AddObject(new       IntExternalResult(results->Size()+1,PatchVerticesEnum,patch->maxvertices,step,time));
		results->AddObject(new       IntExternalResult(results->Size()+1,PatchNodesEnum,   patch->maxnodes,step,time));
		results->AddObject(new DoubleMatExternalResult(results->Size()+1,PatchEnum,patch->values,patch->numrows,patch->numcols,step,time));
	}

	/*Free ressources:*/
	delete patch;

}
