/*!\file: diagnostic_core.cpp
 * \brief: core of the diagnostic solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"

void diagnostic_core(FemModel* femmodel){

	/*parameters: */
	int    verbose              = 0;
	bool   qmu_analysis         = false;
	int    dim                  = -1;
	bool   ishutter             = false;
	bool   ismacayealpattyn     = false;
	bool   isstokes             = false;
	double stokesreconditioning;
	bool   conserve_loads       = true;
	bool   modify_loads         = true;
	int    solution_type;

	/* recover parameters:*/
	femmodel->parameters->FindParam(&verbose,VerboseEnum);
	femmodel->parameters->FindParam(&dim,DimEnum);
	femmodel->parameters->FindParam(&ishutter,IsHutterEnum);
	femmodel->parameters->FindParam(&ismacayealpattyn,IsMacAyealPattynEnum);
	femmodel->parameters->FindParam(&isstokes,IsStokesEnum);
	femmodel->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	femmodel->parameters->FindParam(&qmu_analysis,QmuAnalysisEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);

	/*for qmu analysis, reinitialize velocity so that fake sensitivities do not show up as a result of a different restart of the convergence at each trial.*/
	if(qmu_analysis){
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuVxEnum,VxEnum);
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuVyEnum,VyEnum);
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuVzEnum,VzEnum);
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuPressureEnum,PressureEnum);
	}

	/*Compute slopes: */
	if(ishutter) surfaceslope_core(femmodel);
	if(isstokes) bedslope_core(femmodel);
		
	if(ishutter){
			
		if(verbose)_printf_("%s\n"," computing hutter velocities...");

		//Take the last velocity into account so that the velocity on the MacAyeal domain is not zero
		if (ismacayealpattyn) ResetBoundaryConditions(femmodel,DiagnosticHutterAnalysisEnum);

		femmodel->SetCurrentConfiguration(DiagnosticHutterAnalysisEnum);
		solver_linear(femmodel);
		
		if (ismacayealpattyn) ResetBoundaryConditions(femmodel,DiagnosticHorizAnalysisEnum);
	}

	if (ismacayealpattyn){
		
		if(verbose)_printf_("%s\n"," computing horizontal velocities...");
		femmodel->SetCurrentConfiguration(DiagnosticHorizAnalysisEnum);
		solver_diagnostic_nonlinear(femmodel,modify_loads); 
	}
	
	
	if (dim==3){

		if(verbose)_printf_("%s\n"," computing vertical velocities...");
		femmodel->SetCurrentConfiguration(DiagnosticVertAnalysisEnum);
		solver_linear(femmodel);

		if (isstokes){

			if(verbose)_printf_("%s\n"," update boundary conditions for stokes using velocities previously computed...");
			ResetBoundaryConditions(femmodel,DiagnosticStokesAnalysisEnum);

			if(verbose)_printf_("%s\n"," computing stokes velocities and pressure ...");
			femmodel->SetCurrentConfiguration(DiagnosticStokesAnalysisEnum);
			solver_diagnostic_nonlinear(femmodel,conserve_loads);
		}
	}

	if(verbose)_printf_("saving results:\n");
	if(solution_type==DiagnosticSolutionEnum){
		InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VxEnum);
		InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VyEnum);
		InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VelEnum);
		InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,PressureEnum);
		if(dim==3) InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VzEnum);
	}
}
