/*!\file Tria.cpp
 * \brief: implementation of the Tria object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
/*}}}*/

/*Tria constructors and destructor*/
/*FUNCTION Tria::Tria(){{{1*/
Tria::Tria(){

	this->nodes=NULL;
	this->matice=NULL;
	this->matpar=NULL;
	this->inputs=NULL;
	this->parameters=NULL;
	this->results=NULL;
}
/*}}}*/
/*FUNCTION Tria::Tria(int id, int index, IoModel* iomodel,int nummodels){{{1*/
Tria::Tria(int tria_id, int index, IoModel* iomodel,int nummodels)
	:TriaRef(nummodels)
	,TriaHook(nummodels,index+1,iomodel->numberofelements+1) //index+1: matice id, iomodel->numberofelements+1: matpar id
                                                                  { //i is the element index
	/*id: */
	this->id=tria_id;

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;

	/*intialize inputs and results: */
	this->inputs=new Inputs();
	this->results=new Results();

	/*initialize pointers:*/
	this->nodes=NULL;
	this->matice=NULL;
	this->matpar=NULL;

}
/*}}}*/
/*FUNCTION Tria::~Tria(){{{1*/
Tria::~Tria(){
	delete inputs;
	delete results;
	this->parameters=NULL;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION Tria::copy {{{1*/
Object* Tria::copy() {

	int i;
	Tria* tria=NULL;

	tria=new Tria();

	//deal with TriaRef mother class
	tria->element_type_list=(int*)xmalloc(this->numanalyses*sizeof(int));
	for(i=0;i<this->numanalyses;i++) tria->element_type_list[i]=this->element_type_list[i];

	//deal with TriaHook mother class
	tria->numanalyses=this->numanalyses;
	tria->hnodes=new Hook*[tria->numanalyses];
	for(i=0;i<tria->numanalyses;i++)tria->hnodes[i]=(Hook*)this->hnodes[i]->copy();
	tria->hmatice=(Hook*)this->hmatice->copy();
	tria->hmatpar=(Hook*)this->hmatpar->copy();

	/*deal with Tria fields: */
	tria->id=this->id;
	if(this->inputs){
		tria->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		tria->inputs=new Inputs();
	}
	if(this->results){
		tria->results=(Results*)this->results->Copy();
	}
	else{
		tria->results=new Results();
	}
	/*point parameters: */
	tria->parameters=this->parameters;
	
	/*recover objects: */
	tria->nodes=(Node**)xmalloc(3*sizeof(Node*)); //we cannot rely on an analysis_counter to tell us which analysis_type we are running, so we just copy the nodes.
	for(i=0;i<3;i++)tria->nodes[i]=this->nodes[i];
	tria->matice=(Matice*)tria->hmatice->delivers();
	tria->matpar=(Matpar*)tria->hmatpar->delivers();

	return tria;
}
/*}}}*/
/*FUNCTION Tria::DeepEcho{{{1*/
void Tria::DeepEcho(void){

	printf("Tria:\n");
	printf("   id: %i\n",id);
	if(nodes){
		nodes[0]->DeepEcho();
		nodes[1]->DeepEcho();
		nodes[2]->DeepEcho();
	}
	else printf("nodes = NULL\n");

	if (matice) matice->DeepEcho();
	else printf("matice = NULL\n");

	if (matpar) matpar->DeepEcho();
	else printf("matpar = NULL\n");

	printf("   parameters\n");
	if (parameters) parameters->DeepEcho();
	else printf("parameters = NULL\n");

	printf("   inputs\n");
	if (inputs) inputs->DeepEcho();
	else printf("inputs=NULL\n");

	if (results) results->DeepEcho();
	else printf("results=NULL\n");
	
	return;
}
/*}}}*/
/*FUNCTION Tria::Demarshall {{{1*/
void  Tria::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int i;
	int flaghook;
	int type;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&numanalyses,marshalled_dataset,sizeof(numanalyses));marshalled_dataset+=sizeof(numanalyses);

	/*demarshall Ref: */
	this->element_type_list=(int*)xmalloc(this->numanalyses*sizeof(int));
	for(i=0;i<numanalyses;i++){ memcpy(&element_type_list[i],marshalled_dataset,sizeof(type));marshalled_dataset+=sizeof(type);}

	/*allocate dynamic memory: */
	this->hnodes=new Hook*[this->numanalyses];
	/*demarshall hooks: */
	for(i=0;i<numanalyses;i++){
		memcpy(&flaghook,marshalled_dataset,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
		if(flaghook){ // there is a hook so demarshall it
			hnodes[i]=new Hook();
			hnodes[i]->Demarshall(&marshalled_dataset);
		}
		else hnodes[i]=NULL; //There is no hook so it is NULL
	}
	hmatice=new Hook(); hmatice->Demarshall(&marshalled_dataset);
	hmatpar=new Hook(); hmatpar->Demarshall(&marshalled_dataset);

	/*pointers are garbabe, until configuration is carried out: */
	nodes=NULL;
	matice=NULL;
	matpar=NULL;
	
	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 
	results=(Results*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Tria::Echo{{{1*/
void Tria::Echo(void){
	printf("Tria:\n");
	printf("   id: %i\n",id);
	if(nodes){
		nodes[0]->Echo();
		nodes[1]->Echo();
		nodes[2]->Echo();
	}
	else printf("nodes = NULL\n");

	if (matice) matice->Echo();
	else printf("matice = NULL\n");

	if (matpar) matpar->Echo();
	else printf("matpar = NULL\n");

	printf("   parameters\n");
	if (parameters) parameters->Echo();
	else printf("parameters = NULL\n");

	printf("   inputs\n");
	if (inputs) inputs->Echo();
	else printf("inputs=NULL\n");

	if (results) results->Echo();
	else printf("results=NULL\n");
}
/*}}}*/
/*FUNCTION Tria::Enum {{{1*/
int Tria::Enum(void){

	return TriaEnum;

}
/*}}}*/
/*FUNCTION Tria::Id {{{1*/
int    Tria::Id(){ return id; }
/*}}}*/
/*FUNCTION Tria::Marshall {{{1*/
void  Tria::Marshall(char** pmarshalled_dataset){

	int   i;
	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;
	char* marshalled_results=NULL;
	int   marshalled_results_size;
	int   flaghook; //to indicate if hook is NULL or exists

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Tria: */
	enum_type=TriaEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Tria data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&numanalyses,sizeof(numanalyses));marshalled_dataset+=sizeof(numanalyses);

	/*Mershall Ref: */
	for(i=0;i<numanalyses;i++){
		memcpy(marshalled_dataset,&element_type_list[i],sizeof(element_type_list[i]));marshalled_dataset+=sizeof(element_type_list[i]);
	}

 /*Marshall hooks: */
	for(i=0;i<numanalyses;i++){
		if(hnodes[i]){
			/*Set flag to 1 as there is a hook */
			flaghook=1;
			memcpy(marshalled_dataset,&flaghook,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
			hnodes[i]->Marshall(&marshalled_dataset);
		}
		else{
			/*Set flag to 0 and do not marshall flag as there is no Hook */
			flaghook=0;
			memcpy(marshalled_dataset,&flaghook,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
		}
	}
	hmatice->Marshall(&marshalled_dataset);
	hmatpar->Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*Marshall results: */
	marshalled_results_size=results->MarshallSize();
	marshalled_results=results->Marshall();
	memcpy(marshalled_dataset,marshalled_results,marshalled_results_size*sizeof(char));
	marshalled_dataset+=marshalled_results_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);
	xfree((void**)&marshalled_results);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Tria::MarshallSize {{{1*/
int   Tria::MarshallSize(){

	int i;
	int hnodes_size=0;;

	for(i=0;i<numanalyses;i++){
		hnodes_size+=sizeof(int); //Flag 0 or 1
		if (hnodes[i]) hnodes_size+=hnodes[i]->MarshallSize();
	}

	return sizeof(id)
		+hnodes_size
		+sizeof(numanalyses)
		+numanalyses*sizeof(int) //element_type_lists
		+hmatice->MarshallSize()
		+hmatpar->MarshallSize()
		+inputs->MarshallSize()
		+results->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Tria::MyRank {{{1*/
int    Tria::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/

/*Update virtual functions definitions: */
/*FUNCTION Tria::InputUpdateFromConstant(int value, int name);{{{1*/
void  Tria::InputUpdateFromConstant(int constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new IntInput(name,constant));
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromConstant(double value, int name);{{{1*/
void  Tria::InputUpdateFromConstant(double constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new DoubleInput(name,constant));
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromConstant(bool value, int name);{{{1*/
void  Tria::InputUpdateFromConstant(bool constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new BoolInput(name,constant));
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromSolution {{{1*/
void  Tria::InputUpdateFromSolution(double* solution){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Just branch to the correct InputUpdateFromSolution generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		InputUpdateFromSolutionDiagnosticHoriz( solution);
	}
	else if (analysis_type==DiagnosticHutterAnalysisEnum){
		InputUpdateFromSolutionDiagnosticHoriz( solution);
	}
	else if (analysis_type==AdjointHorizAnalysisEnum){
		InputUpdateFromSolutionAdjointHoriz( solution);
	}
	else if (analysis_type==BedSlopeXAnalysisEnum){
		InputUpdateFromSolutionOneDof(solution,BedSlopeXEnum);
	}
	else if (analysis_type==BedSlopeYAnalysisEnum){
		InputUpdateFromSolutionOneDof(solution,BedSlopeYEnum);
	}
	else if (analysis_type==SurfaceSlopeXAnalysisEnum){
		InputUpdateFromSolutionOneDof(solution,SurfaceSlopeXEnum);
	}
	else if (analysis_type==SurfaceSlopeYAnalysisEnum){
		InputUpdateFromSolutionOneDof(solution,SurfaceSlopeYEnum);
	}
	else if (analysis_type==PrognosticAnalysisEnum){
		InputUpdateFromSolutionOneDof(solution,ThicknessEnum);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){
		InputUpdateFromSolutionOneDof(solution,ThicknessEnum);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){
		InputUpdateFromSolutionOneDof(solution,VelEnum);
	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));
	}
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromVector(double* vector, int name, int type);{{{1*/
void  Tria::InputUpdateFromVector(double* vector, int name, int type){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	switch(type){

		case VertexEnum:

			/*New TriaVertexInput*/
			double values[3];

			/*Get values on the 6 vertices*/
			for (int i=0;i<3;i++){
				values[i]=vector[this->nodes[i]->GetVertexDof()];
			}

			/*update input*/
			this->inputs->AddInput(new TriaVertexInput(name,values));
			return;

		default:

			ISSMERROR("type %i (%s) not implemented yet",type,EnumAsString(type));
	}
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromVector(int* vector, int name, int type);{{{1*/
void  Tria::InputUpdateFromVector(int* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromVector(bool* vector, int name, int type);{{{1*/
void  Tria::InputUpdateFromVector(bool* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/

/*Element virtual functions definitions: */
/*FUNCTION Tria::ComputeBasalStress {{{1*/
void  Tria::ComputeBasalStress(Vec eps){

	int i;
	const int numgrids=3;
	int doflist[numgrids];
	double value;

	/*plug local pressure values into global pressure vector: */
	ISSMERROR("Not Implemented yet");
	//VecSetValues(eps,1,2,(const double*)&value,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Tria::ComputePressure {{{1*/
void  Tria::ComputePressure(Vec pg){

	int i;
	const int numvertices=3;
	int doflist[numvertices];
	double pressure[numvertices];
	double thickness[numvertices];
	double rho_ice,g;
	double gauss[numvertices][numvertices]={{1,0,0},{0,1,0},{0,0,1}};
	
	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();

	/*recover value of thickness at gauss points (0,0,1), (0,1,0),(1,0,0): */
	inputs->GetParameterValues(&thickness[0],&gauss[0][0],3,ThicknessEnum);

	for(i=0;i<numvertices;i++){
		pressure[i]=rho_ice*g*thickness[i];
	}

	/*plug local pressure values into global pressure vector: */
	VecSetValues(pg,numvertices,doflist,(const double*)pressure,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Tria::ComputeStrainRate {{{1*/
void  Tria::ComputeStrainRate(Vec eps){

	int i;
	const int numgrids=3;
	int doflist[numgrids];
	double value;

	/*plug local pressure values into global pressure vector: */
	ISSMERROR("Not Implemented yet");
	//VecSetValues(eps,1,2,(const double*)&value,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Tria::SetCurrentConfiguration {{{1*/
void  Tria::SetCurrentConfiguration(Elements* elementsin, Loads* loadsin, DataSet* nodesin, Materials* materialsin, Parameters* parametersin){

	int analysis_counter;
	
	/*go into parameters and get the analysis_counter: */
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Get Element type*/
	this->element_type=this->element_type_list[analysis_counter];

	/*Pick up the objects inside the hooks: */
	if(this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;
	this->matice=(Matice*)this->hmatice->delivers();
	this->matpar=(Matpar*)this->hmatpar->delivers();

	/*point parameters to real dataset: */
	this->parameters=parametersin;

}
/*}}}*/
/*FUNCTION Tria::Configure {{{1*/
void  Tria::Configure(Elements* elementsin, Loads* loadsin, DataSet* nodesin, Materials* materialsin, Parameters* parametersin){

	int analysis_counter;
	
	/*go into parameters and get the analysis_counter: */
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Get Element type*/
	this->element_type=this->element_type_list[analysis_counter];

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	if(this->hnodes[analysis_counter]) this->hnodes[analysis_counter]->configure(nodesin);
	this->hmatice->configure(materialsin);
	this->hmatpar->configure(materialsin);

	/*Now, go pick up the objects inside the hooks: */
	if(this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;
	this->matice=(Matice*)this->hmatice->delivers();
	this->matpar=(Matpar*)this->hmatpar->delivers();

	/*point parameters to real dataset: */
	this->parameters=parametersin;

}
/*}}}*/
/*FUNCTION Tria::CostFunction {{{1*/
double Tria::CostFunction(void){

	int i;

	/* output: */
	double Jelem;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	int          dofs1[1]={0};
	int          dofs2[2]={0,1};
	double       xyz_list[numgrids][3];

	/* grid data: */
	double B[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  k_gauss;
	double  B_gauss;

	/* parameters: */
	double  dk[NDOF2]; 
	double  dB[NDOF2]; 
	int    control_type;
	double  cm_noisedmp;

	/* Jacobian: */
	double Jdet;

	/*inputs: */
	bool onwater;
	bool shelf;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*retrieve some parameters: */
	this->parameters->FindParam(&control_type,ControlTypeEnum);
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);

	/*First, get Misfit*/
	Jelem=Misfit();

	  /* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	  GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/*Add Tikhonov regularization term to misfit*/
		if (control_type==DragCoefficientEnum){
			if (shelf){
				inputs->GetParameterDerivativeValue(&dk[0],&xyz_list[0][0],&gauss_l1l2l3[0],DragCoefficientEnum);
				Jelem+=cm_noisedmp*1/2*(pow(dk[0],2)+pow(dk[1],2))*Jdet*gauss_weight;

			}
		}
		else if (control_type==RheologyBEnum){
			inputs->GetParameterDerivativeValue(&dB[0], &xyz_list[0][0], &gauss_l1l2l3[0],RheologyBEnum);
			Jelem+=cm_noisedmp*1/2*(pow(dB[0],2)+pow(dB[1],2))*Jdet*gauss_weight;
		}
		else{
			ISSMERROR("%s%i","unsupported control type: ",control_type);
		}

	}

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

	/*Return: */
	return Jelem;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrix {{{1*/
void  Tria::CreateKMatrix(Mat Kgg){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*asserts: {{{2*/
	ISSMASSERT(this->nodes && this->matice && this->matpar && this->parameters && this->inputs);
	/*}}}*/

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		CreateKMatrixDiagnosticHoriz( Kgg);
	}
	else if (analysis_type==AdjointHorizAnalysisEnum){
		/*Same as diagnostic*/
		CreateKMatrixDiagnosticHoriz( Kgg);
	}
	else if (analysis_type==DiagnosticHutterAnalysisEnum){
		CreateKMatrixDiagnosticHutter( Kgg);
	}
	else if (analysis_type==BedSlopeXAnalysisEnum || analysis_type==SurfaceSlopeXAnalysisEnum || analysis_type==BedSlopeYAnalysisEnum || analysis_type==SurfaceSlopeYAnalysisEnum){
		CreateKMatrixSlope( Kgg);
	}
	else if (analysis_type==PrognosticAnalysisEnum){
		if (GetElementType()==P1Enum)
		 CreateKMatrixPrognostic_CG( Kgg);
		else if (GetElementType()==P1DGEnum)
		 CreateKMatrixPrognostic_DG( Kgg);
		else
		 ISSMERROR("Element type %s not supported yet",EnumAsString(GetElementType()));
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){
		if (GetElementType()==P1Enum)
		 CreateKMatrixBalancedthickness_CG( Kgg);
		else if (GetElementType()==P1DGEnum)
		 CreateKMatrixBalancedthickness_DG( Kgg);
		else
		 ISSMERROR("Element type %s not supported yet",EnumAsString(GetElementType()));
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){
		CreateKMatrixBalancedvelocities( Kgg);
	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));
	}

}
/*}}}*/
/*FUNCTION Tria::CreatePVector {{{1*/
void  Tria::CreatePVector(Vec pg){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*asserts: {{{*/
	/*if debugging mode, check that all pointers exist*/
	ISSMASSERT(this->nodes && this->matice && this->matpar && this->parameters && this->inputs);
	/*}}}*/

	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		CreatePVectorDiagnosticHoriz( pg);
	}
	else if (analysis_type==AdjointHorizAnalysisEnum){
		CreatePVectorAdjointHoriz( pg);
	}
	else if (analysis_type==DiagnosticHutterAnalysisEnum){
		CreatePVectorDiagnosticHutter( pg);
	}
	else if (analysis_type==BedSlopeXAnalysisEnum || analysis_type==SurfaceSlopeXAnalysisEnum || analysis_type==BedSlopeYAnalysisEnum || analysis_type==SurfaceSlopeYAnalysisEnum){
		CreatePVectorSlope( pg);
	}
	else if (analysis_type==PrognosticAnalysisEnum){
		if (GetElementType()==P1Enum)
		 CreatePVectorPrognostic_CG( pg);
		else if (GetElementType()==P1DGEnum)
		CreatePVectorPrognostic_DG( pg);
		else
		 ISSMERROR("Element type %s not supported yet",EnumAsString(GetElementType()));
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){
		if (GetElementType()==P1Enum)
		 CreatePVectorBalancedthickness_CG( pg);
		else if (GetElementType()==P1DGEnum)
		 CreatePVectorBalancedthickness_DG( pg);
		else
		 ISSMERROR("Element type %s not supported yet",EnumAsString(GetElementType()));
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){
		CreatePVectorBalancedvelocities( pg);
	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));
	}

}
/*}}}*/
/*FUNCTION Tria::GetBedList {{{1*/
void  Tria::GetBedList(double* bedlist){
	
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	
	inputs->GetParameterValues(bedlist,&gaussgrids[0][0],3,BedEnum);

}
/*}}}*/
/*FUNCTION Tria::GetMatPar {{{1*/
void* Tria::GetMatPar(){

	return matpar;
}
/*}}}*/
/*FUNCTION Tria::GetNodes {{{1*/
void  Tria::GetNodes(void** vpnodes){
	int i;
	Node** pnodes=NULL;

	/*recover nodes: */
	pnodes=(Node**)vpnodes;

	for(i=0;i<3;i++){
		pnodes[i]=nodes[i];
	}
}
/*}}}*/
/*FUNCTION Tria::GetOnBed {{{1*/
bool Tria::GetOnBed(){
	
	bool onbed;
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	return onbed;
}
/*}}}*/
/*FUNCTION Tria::GetShelf {{{1*/
bool   Tria::GetShelf(){
	/*inputs: */
	bool shelf;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	return shelf;
}
/*}}}*/
/*FUNCTION Tria::GetSolutionFromInputs{{{1*/
void  Tria::GetSolutionFromInputs(Vec solution){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	
	/*Just branch to the correct InputUpdateFromSolution generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum)
	 GetSolutionFromInputsDiagnosticHoriz(solution);
	else if (analysis_type==AdjointHorizAnalysisEnum)
	 GetSolutionFromInputsAdjointHoriz(solution);
	else if (analysis_type==DiagnosticHutterAnalysisEnum)
	 GetSolutionFromInputsDiagnosticHutter(solution);
	else
	 ISSMERROR("analysis: %s not supported yet",EnumAsString(analysis_type));

}
/*}}}*/
/*FUNCTION Tria::GetThicknessList {{{1*/
void Tria::GetThicknessList(double* thickness_list){

	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	inputs->GetParameterValues(thickness_list,&gaussgrids[0][0],3,ThicknessEnum);

}
/*}}}*/
/*FUNCTION Tria::GetVectorFromInputs{{{1*/
void  Tria::GetVectorFromInputs(Vec vector,int NameEnum){

	int i;
	const int numvertices=3;
	int doflist1[numvertices];

	/*Find NameEnum input in the inputs dataset, and get it to fill in the vector: */
	for(i=0;i<this->inputs->Size();i++){
		Input* input=(Input*)this->inputs->GetObjectByOffset(i);
		if(input->EnumType()==NameEnum){
			/*We found the enum.  Use its values to fill into the vector, using the vertices ids: */
			this->GetDofList1(&doflist1[0]);
			input->GetVectorFromInputs(vector,&doflist1[0]);
			break;
		}
	}
}
/*}}}*/
/*FUNCTION Tria::Gradj {{{1*/
void  Tria::Gradj(Vec gradient,int control_type){

	/*inputs: */
	bool onwater;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, grad = 0: */
	if(onwater)return;

	if (control_type==DragCoefficientEnum){
		GradjDrag(gradient);
	}
	else if (control_type==RheologyBEnum){
		GradjB(gradient);
	}
	else ISSMERROR("%s%i","control type not supported yet: ",control_type);
}
/*}}}*/
/*FUNCTION Tria::GradjB{{{1*/
void  Tria::GradjB(Vec gradient){

	int i;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    NDOF2=2;
	const int    numdof=NDOF2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist1[numgrids];
	double       dh1dh3[NDOF2][numgrids];

	/* grid data: */
	double B[numgrids];

	int    dofs1[1]={0};
	int    dofs2[2]={0,1};

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/*element vector at the gaussian points: */
	double  grade_g[numgrids]={0.0};
	double  grade_g_gaussian[numgrids];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/

	/* parameters: */
	double  viscosity_complement;
	double  dvx[NDOF2];
	double  dvy[NDOF2]; 
	double  dadjx[NDOF2];
	double  dadjy[NDOF2];
	double  vx,vy;
	double  lambda,mu;
	double  thickness;
	int     dofs[1]={0};
	double  dB[NDOF2]; 
	double  B_gauss;
	
	/*parameters: */
	double  cm_noisedmp;
	double  cm_mindmp_slope;
	double  cm_mindmp_value;
	double  cm_maxdmp_value;
	double  cm_maxdmp_slope;

	/*retrieve some parameters: */
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);
	this->parameters->FindParam(&cm_mindmp_value,CmMinDmpValueEnum);
	this->parameters->FindParam(&cm_mindmp_slope,CmMinDmpSlopeEnum);
	this->parameters->FindParam(&cm_maxdmp_value,CmMaxDmpValueEnum);
	this->parameters->FindParam(&cm_maxdmp_slope,CmMaxDmpSlopeEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList1(&doflist1[0]);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 4);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Get thickness: */
		inputs->GetParameterValue(&thickness, gauss_l1l2l3,ThicknessEnum);

		/*Get strain rate, if velocity has been supplied: */
		inputs->GetStrainRate2d(&epsilon[0],&xyz_list[0][0],gauss_l1l2l3,VxEnum,VyEnum);

		/*Get viscosity complement: */
		matice->GetViscosityComplement(&viscosity_complement, &epsilon[0]);

		/*Get dvx, dvy, dadjx and dadjx: */
		inputs->GetParameterDerivativeValue(&dvx[0],&xyz_list[0][0],&gauss_l1l2l3[0],VxEnum);
		inputs->GetParameterDerivativeValue(&dvy[0],&xyz_list[0][0],&gauss_l1l2l3[0],VyEnum);
		inputs->GetParameterDerivativeValue(&dadjx[0],&xyz_list[0][0],&gauss_l1l2l3[0],AdjointxEnum);
		inputs->GetParameterDerivativeValue(&dadjy[0],&xyz_list[0][0],&gauss_l1l2l3[0],AdjointyEnum);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Get nodal functions derivatives*/
		GetNodalFunctionsDerivatives(&dh1dh3[0][0],&xyz_list[0][0],gauss_l1l2l3);

		/*Get B derivative: dB/dx */
		inputs->GetParameterDerivativeValue(&dB[0],&xyz_list[0][0],&gauss_l1l2l3[0],RheologyBEnum);
		inputs->GetParameterValue(&B_gauss, gauss_l1l2l3,RheologyBEnum);

		/*Build gradje_g_gaussian vector (actually -dJ/dB): */
		for (i=0;i<numgrids;i++){
			//standard gradient dJ/dki
			grade_g_gaussian[i]=-viscosity_complement*thickness*( (2*dvx[0]+dvy[1])*2*dadjx[0]+(dvx[1]+dvy[0])*(dadjx[1]+dadjy[0])+(2*dvy[1]+dvx[0])*2*dadjy[1])*Jdet*gauss_weight*l1l2l3[i]; 

			//Add regularization term
			grade_g_gaussian[i]-=cm_noisedmp*Jdet*gauss_weight*(dh1dh3[0][i]*dB[0]+dh1dh3[1][i]*dB[1]);

			//min dampening
			if(B_gauss<cm_mindmp_value){ 
				grade_g_gaussian[i]+= cm_mindmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

			//max dampening
			if(B_gauss>cm_maxdmp_value){ 
				grade_g_gaussian[i]+= - cm_maxdmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

		}

		/*Add grade_g_gaussian to grade_g: */
		for( i=0; i<numgrids;i++) grade_g[i]+=grade_g_gaussian[i];
	}

	/*Add grade_g to global vector gradient: */
	VecSetValues(gradient,numgrids,doflist1,(const double*)grade_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Tria::GradjDrag {{{1*/
void  Tria::GradjDrag(Vec gradient){

	int i;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF2=2;
	const int    numdof=NDOF2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist1[numgrids];
	double       dh1dh3[NDOF2][numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* parameters: */
	double  dk[NDOF2]; 
	double  vx,vy;
	double  lambda,mu;
	double  bed,thickness,Neff;
	double  alpha_complement;
	int     drag_type;
	double  drag;
	Friction* friction=NULL;

	/*element vector at the gaussian points: */
	double  grade_g[numgrids]={0.0};
	double  grade_g_gaussian[numgrids];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];
	double    alpha2complement_list[numgrids];                          //TO BE DELETED
	double    gauss[numgrids][numgrids] = {{1,0,0},{0,1,0},{0,0,1}}; //TO BE DELETED

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/

	/*inputs: */
	bool shelf;

	/*parameters: */
	double  cm_noisedmp;
	double  cm_mindmp_slope;
	double  cm_mindmp_value;
	double  cm_maxdmp_value;
	double  cm_maxdmp_slope;

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*retrieve inputs :*/
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);
	this->parameters->FindParam(&cm_mindmp_value,CmMinDmpValueEnum);
	this->parameters->FindParam(&cm_mindmp_slope,CmMinDmpSlopeEnum);
	this->parameters->FindParam(&cm_maxdmp_value,CmMaxDmpValueEnum);
	this->parameters->FindParam(&cm_maxdmp_slope,CmMaxDmpSlopeEnum);

	/*Get out if shelf*/
	if(shelf)return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList1(&doflist1[0]);

	/*Build frictoin element, needed later: */
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	friction=new Friction("2d",inputs,matpar,analysis_type);

	/*COMPUT alpha2_list (TO BE DELETED)*/
	for(i=0;i<numgrids;i++){
		friction->GetAlphaComplement(&alpha2complement_list[i],&gauss[i][0],VxEnum,VyEnum);
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 4);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Build alpha_complement_list: */
		//if (drag_type==2) friction->GetAlphaComplement(&alpha_complement, gauss_l1l2l3,VxEnum,VyEnum); // TO BE UNCOMMENTED
		//else alpha_complement=0;
		GetParameterValue(&alpha_complement,&alpha2complement_list[0],gauss_l1l2l3); // TO BE DELETED
	
		/*Recover alpha_complement and k: */
		inputs->GetParameterValue(&drag, gauss_l1l2l3,DragCoefficientEnum);

		/*recover lambda and mu: */
		inputs->GetParameterValue(&lambda, gauss_l1l2l3,AdjointxEnum);
		inputs->GetParameterValue(&mu, gauss_l1l2l3,AdjointyEnum);
			
		/*recover vx and vy: */
		inputs->GetParameterValue(&vx, gauss_l1l2l3,VxEnum);
		inputs->GetParameterValue(&vy, gauss_l1l2l3,VyEnum);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Get nodal functions derivatives*/
		GetNodalFunctionsDerivatives(&dh1dh3[0][0],&xyz_list[0][0],gauss_l1l2l3);

		/*Get k derivative: dk/dx */
		inputs->GetParameterDerivativeValue(&dk[0],&xyz_list[0][0],&gauss_l1l2l3[0],DragCoefficientEnum);

		/*Build gradje_g_gaussian vector (actually -dJ/ddrag): */
		for (i=0;i<numgrids;i++){

			//standard term dJ/dki
			grade_g_gaussian[i]=-2*drag*alpha_complement*((lambda*vx+mu*vy))*Jdet*gauss_weight*l1l2l3[i];

			//noise dampening d/dki(1/2*(dk/dx)^2)
			grade_g_gaussian[i]+=-cm_noisedmp*Jdet*gauss_weight*(dh1dh3[0][i]*dk[0]+dh1dh3[1][i]*dk[1]);
			
			//min dampening
			if(drag<cm_mindmp_value){ 
				grade_g_gaussian[i]+=cm_mindmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

			//max dampening
			if(drag>cm_maxdmp_value){ 
				grade_g_gaussian[i]+= - cm_maxdmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}
		}
		
		/*Add gradje_g_gaussian vector to gradje_g: */
		for( i=0; i<numgrids; i++)grade_g[i]+=grade_g_gaussian[i];
	}


	/*Add grade_g to global vector gradient: */
	VecSetValues(gradient,numgrids,doflist1,(const double*)grade_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
	delete friction;

}
/*}}}*/
/*FUNCTION Tria::InputAXPY{{{1*/
void  Tria::InputAXPY(int YEnum, double scalar, int XEnum){

	Input* xinput=NULL;
	Input* yinput=NULL;

	/*Find x and y inputs: */
	xinput=(Input*)this->inputs->GetInput(XEnum);
	yinput=(Input*)this->inputs->GetInput(YEnum);

	/*some checks: */
	if(!xinput || !yinput) ISSMERROR("%s%s%s%s%s"," input ",EnumAsString(XEnum)," or input ",EnumAsString(YEnum)," could not be found!");

	/*Scale: */
	yinput->AXPY(xinput,scalar);

	/*Move input to Material if required (needed if control method) TO BE IMPROVED*/
	if (YEnum==RheologyBEnum){
		this->matice->inputs->AddInput((Input*)yinput->copy());
	}

}
/*}}}*/
/*FUNCTION Tria::InputControlConstrain{{{1*/
void  Tria::InputControlConstrain(int control_type, double cm_min, double cm_max){

	Input* input=NULL;

	/*Find input: */
	input=(Input*)this->inputs->GetInput(control_type);
	
	/*Do nothing if we  don't find it: */
	if(!input) return;

	/*Constrain input using cm_min and cm_max: */
	input->Constrain(cm_min,cm_max);

}
/*}}}*/
/*FUNCTION Tria::InputConvergence{{{1*/
void  Tria::InputConvergence(int* pconverged,double* eps, int* enums,int num_enums,int* criterionenums,double* criterionvalues,int num_criterionenums){

	int i;
	Input** new_inputs=NULL;
	Input** old_inputs=NULL;
	int     converged=1;

	new_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the new inputs
	old_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the old inputs
	
	for(i=0;i<num_enums/2;i++){
		new_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+0]);
		old_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+1]);
		if(!new_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumAsString(enums[2*i+0]));
		if(!old_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumAsString(enums[2*i+0]));
	}

	/*ok, we've got the inputs (new and old), now loop throught the number of criterions and fill the eps array:*/
	for(i=0;i<num_criterionenums;i++){
		IsInputConverged(eps+i,new_inputs,old_inputs,num_enums/2,criterionenums[i]);
		if(eps[i]>criterionvalues[i]) converged=0; 
	}

	/*clean up*/
	xfree((void**)&new_inputs);
	xfree((void**)&old_inputs);

	/*Assign output pointers:*/
	*pconverged=converged;

}
/*}}}*/
/*FUNCTION Tria::InputDepthAverageAtBase {{{1*/
void  Tria::InputDepthAverageAtBase(int enum_type,int average_enum_type){

	/*New input*/
	Input* oldinput=NULL;
	Input* newinput=NULL;

	/*copy input of enum_type*/
	oldinput=this->inputs->GetInput(enum_type);
	if(!oldinput)ISSMERROR("%s%s"," could not find old input with enum: ",EnumAsString(enum_type));
	newinput=(Input*)oldinput->copy();

	/*Assign new name (average)*/
	newinput->ChangeEnum(average_enum_type);

	/*Add new input to current element*/
	this->inputs->AddInput(newinput);

}
/*}}}*/
/*FUNCTION Tria::InputDuplicate{{{1*/
void  Tria::InputDuplicate(int original_enum,int new_enum){

	Input* original=NULL;
	Input* copy=NULL;

	/*Make a copy of the original input: */
	original=(Input*)this->inputs->GetInput(original_enum);
	if(!original)ISSMERROR(" could not find old input with enum: %s",EnumAsString(original_enum));
	copy=(Input*)original->copy();

	/*Change copy enum to reinitialized_enum: */
	copy->ChangeEnum(new_enum);

	/*Add copy into inputs, it will wipe off the one already there: */
	inputs->AddInput((Input*)copy);
}
/*}}}*/
/*FUNCTION Tria::InputScale{{{1*/
void  Tria::InputScale(int enum_type,double scale_factor){

	Input* input=NULL;

	/*Make a copy of the original input: */
	input=(Input*)this->inputs->GetInput(enum_type);
	if(!input)ISSMERROR(" could not find old input with enum: %s",EnumAsString(enum_type));

	/*Scale: */
	input->Scale(scale_factor);
}
/*}}}*/
/*FUNCTION Tria::InputToResult{{{1*/
void  Tria::InputToResult(int enum_type,int step,double time){

	int    i;
	bool   found = false;
	Input *input = NULL;

	/*Go through all the input objects, and find the one corresponding to enum_type, if it exists: */
	for (i=0;i<this->inputs->Size();i++){
		input=(Input*)this->inputs->GetObjectByOffset(i);
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}
	if (!found) ISSMERROR("Input %s not found in tria->inputs",EnumAsString(enum_type));

	/*If we don't find it, no big deal, just don't do the transfer. Otherwise, build a new Result 
	 * object out of the input, with the additional step and time information: */
	this->results->AddObject((Object*)input->SpawnResult(step,time));

}
/*}}}*/
/*FUNCTION Tria::MassFlux {{{1*/
double Tria::MassFlux( double* segment){

	int i;

	const int    numgrids=3;
	const int    numdofs=2;
	int          numberofdofspernode;
	double mass_flux=0;
	double xyz_list[numgrids][3];
	double gauss_1[3];
	double gauss_2[3];
	double normal[2];
	double length;
	double x1,y1,x2,y2;
	double h1,h2;
	double vx1,vx2,vy1,vy2;
	double rho_ice;

	/*Get material parameters :*/
	rho_ice=matpar->GetRhoIce();

	/*First off, check that this segment belongs to this element: */
	if ((int)*(segment+4)!=this->id)ISSMERROR("%s%i%s%i","error message: segment with id ",(int)*(segment+4)," does not belong to element with id:",this->id);

	/*Recover segment node locations: */
	x1=*(segment+0); y1=*(segment+1); x2=*(segment+2); y2=*(segment+3);

	/*Get xyz list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);

	/*get area coordinates of 0 and 1 locations: */
	for(i=0;i<3;i++){
		gauss_1[i]=this->GetAreaCoordinate(x1,y1,i+1);
		gauss_2[i]=this->GetAreaCoordinate(x2,y2,i+1);
	}

	/*get normal of segment: */
	normal[0]=cos(atan2(x1-x2,y2-y1));
	normal[1]=sin(atan2(x1-x2,y2-y1));

	/*get length of segment: */
	length=sqrt(pow(x2-x1,2.0)+pow(y2-y1,2));

	/*get thickness and velocity at two segment extremities: */
	inputs->GetParameterValue(&h1, &gauss_1[0],ThicknessEnum);
	inputs->GetParameterValue(&h2, &gauss_2[0],ThicknessEnum);
	inputs->GetParameterValue(&vx1, &gauss_1[0],VxEnum);
	inputs->GetParameterValue(&vx2, &gauss_2[0],VxEnum);
	inputs->GetParameterValue(&vy1, &gauss_1[0],VyEnum);
	inputs->GetParameterValue(&vy2, &gauss_2[0],VyEnum);

	mass_flux= rho_ice*length*(  
				(ONETHIRD*(h1-h2)*(vx1-vx2)+0.5*h2*(vx1-vx2)+0.5*(h1-h2)*vx2+h2*vx2)*normal[0]+
				(ONETHIRD*(h1-h2)*(vy1-vy2)+0.5*h2*(vy1-vy2)+0.5*(h1-h2)*vy2+h2*vy2)*normal[1]
				);
	return mass_flux;
}
/*}}}*/
/*FUNCTION Tria::MaxAbsVx{{{1*/
void  Tria::MaxAbsVx(double* pmaxabsvx, bool process_units){

	int i;
	int dim;
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	double  vx_values[numgrids];
	double  maxabsvx;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vx_values[0],&gaussgrids[0][0],numgrids,VxEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vx_values[0],numgrids,VxEnum,this->parameters);

	/*now, compute maximum:*/
	maxabsvx=fabs(vx_values[0]);
	for(i=1;i<numgrids;i++){
		if (fabs(vx_values[i])>maxabsvx)maxabsvx=fabs(vx_values[i]);
	}

	/*Assign output pointers:*/
	*pmaxabsvx=maxabsvx;
}
/*}}}*/
/*FUNCTION Tria::MaxAbsVy{{{1*/
void  Tria::MaxAbsVy(double* pmaxabsvy, bool process_units){

	int i;
	int dim;
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	double  vy_values[numgrids];
	double  maxabsvy;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vy_values[0],&gaussgrids[0][0],numgrids,VyEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vy_values[0],numgrids,VyEnum,this->parameters);

	/*now, compute maximum:*/
	maxabsvy=fabs(vy_values[0]);
	for(i=1;i<numgrids;i++){
		if (fabs(vy_values[i])>maxabsvy)maxabsvy=fabs(vy_values[i]);
	}

	/*Assign output pointers:*/
	*pmaxabsvy=maxabsvy;
}
/*}}}*/
/*FUNCTION Tria::MaxAbsVz{{{1*/
void  Tria::MaxAbsVz(double* pmaxabsvz, bool process_units){

	int i;
	int dim;
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	double  vz_values[numgrids];
	double  maxabsvz;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vz_values[0],&gaussgrids[0][0],numgrids,VzEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vz_values[0],numgrids,VzEnum,this->parameters);

	/*now, compute maximum:*/
	maxabsvz=fabs(vz_values[0]);
	for(i=1;i<numgrids;i++){
		if (fabs(vz_values[i])>maxabsvz)maxabsvz=fabs(vz_values[i]);
	}

	/*Assign output pointers:*/
	*pmaxabsvz=maxabsvz;
}
/*}}}*/
/*FUNCTION Tria::MaxVel{{{1*/
void  Tria::MaxVel(double* pmaxvel, bool process_units){

	int i;
	int dim;
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	double  vel_values[numgrids];
	double  maxvel;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vel_values[0],&gaussgrids[0][0],numgrids,VelEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vel_values[0],numgrids,VelEnum,this->parameters);

	/*now, compute maximum:*/
	maxvel=vel_values[0];
	for(i=1;i<numgrids;i++){
		if (vel_values[i]>maxvel)maxvel=vel_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvel=maxvel;

}
/*}}}*/
/*FUNCTION Tria::MaxVx{{{1*/
void  Tria::MaxVx(double* pmaxvx, bool process_units){

	int i;
	int dim;
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	double  vx_values[numgrids];
	double  maxvx;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vx_values[0],&gaussgrids[0][0],numgrids,VxEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vx_values[0],numgrids,VxEnum,this->parameters);

	/*now, compute maximum:*/
	maxvx=vx_values[0];
	for(i=1;i<numgrids;i++){
		if (vx_values[i]>maxvx)maxvx=vx_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvx=maxvx;

}
/*}}}*/
/*FUNCTION Tria::MaxVy{{{1*/
void  Tria::MaxVy(double* pmaxvy, bool process_units){

	int i;
	int dim;
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	double  vy_values[numgrids];
	double  maxvy;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vy_values[0],&gaussgrids[0][0],numgrids,VyEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vy_values[0],numgrids,VyEnum,this->parameters);

	/*now, compute maximum:*/
	maxvy=vy_values[0];
	for(i=1;i<numgrids;i++){
		if (vy_values[i]>maxvy)maxvy=vy_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvy=maxvy;

}
/*}}}*/
/*FUNCTION Tria::MaxVz{{{1*/
void  Tria::MaxVz(double* pmaxvz, bool process_units){

	int i;
	int dim;
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	double  vz_values[numgrids];
	double  maxvz;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vz_values[0],&gaussgrids[0][0],numgrids,VzEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vz_values[0],numgrids,VzEnum,this->parameters);

	/*now, compute maximum:*/
	maxvz=vz_values[0];
	for(i=1;i<numgrids;i++){
		if (vz_values[i]>maxvz)maxvz=vz_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvz=maxvz;

}
/*}}}*/
/*FUNCTION Tria::MinVel{{{1*/
void  Tria::MinVel(double* pminvel, bool process_units){

	int i;
	int dim;
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	double  vel_values[numgrids];
	double  minvel;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vel_values[0],&gaussgrids[0][0],numgrids,VelEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vel_values[0],numgrids,VelEnum,this->parameters);

	/*now, compute minimum:*/
	minvel=vel_values[0];
	for(i=1;i<numgrids;i++){
		if (vel_values[i]<minvel)minvel=vel_values[i];
	}

	/*Assign output pointers:*/
	*pminvel=minvel;

}
/*}}}*/
/*FUNCTION Tria::MinVx{{{1*/
void  Tria::MinVx(double* pminvx, bool process_units){

	int i;
	int dim;
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	double  vx_values[numgrids];
	double  minvx;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vx_values[0],&gaussgrids[0][0],numgrids,VxEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vx_values[0],numgrids,VxEnum,this->parameters);

	/*now, compute minimum:*/
	minvx=vx_values[0];
	for(i=1;i<numgrids;i++){
		if (vx_values[i]<minvx)minvx=vx_values[i];
	}

	/*Assign output pointers:*/
	*pminvx=minvx;

}
/*}}}*/
/*FUNCTION Tria::MinVy{{{1*/
void  Tria::MinVy(double* pminvy, bool process_units){

	int i;
	int dim;
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	double  vy_values[numgrids];
	double  minvy;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vy_values[0],&gaussgrids[0][0],numgrids,VyEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vy_values[0],numgrids,VyEnum,this->parameters);

	/*now, compute minimum:*/
	minvy=vy_values[0];
	for(i=1;i<numgrids;i++){
		if (vy_values[i]<minvy)minvy=vy_values[i];
	}

	/*Assign output pointers:*/
	*pminvy=minvy;

}
/*}}}*/
/*FUNCTION Tria::MinVz{{{1*/
void  Tria::MinVz(double* pminvz, bool process_units){

	int i;
	int dim;
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	double  vz_values[numgrids];
	double  minvz;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vz_values[0],&gaussgrids[0][0],numgrids,VzEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vz_values[0],numgrids,VzEnum,this->parameters);

	/*now, compute minimum:*/
	minvz=vz_values[0];
	for(i=1;i<numgrids;i++){
		if (vz_values[i]<minvz)minvz=vz_values[i];
	}

	/*Assign output pointers:*/
	*pminvz=minvz;

}
/*}}}*/
/*FUNCTION Tria::Misfit {{{1*/
double Tria::Misfit(void){

	int i;

	/* output: */
	double Jelem=0;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	int          dofs1[1]={0};
	int          dofs2[2]={0,1};
	double       xyz_list[numgrids][3];

	/* grid data: */
	double vx_list[numgrids];
	double vy_list[numgrids];
	double obs_vx_list[numgrids];
	double obs_vy_list[numgrids];
	double misfit_list[numgrids];
	double weights_list[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* parameters: */
	double  velocity_mag,obs_velocity_mag;
	double  misfit;

	/* Jacobian: */
	double Jdet;

	/*relative and logarithmic control method :*/
	double scalex=1;
	double scaley=1;
	double S=0;
	int    fit=-1;

	/*inputs: */
	bool onwater;
	
	double  meanvel, epsvel;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);

	/* Recover input data: */
	inputs->GetParameterValue(&fit,FitEnum);
	if(fit==3){
		inputs->GetParameterValue(&S,SurfaceAreaEnum);
	}
	inputs->GetParameterValues(&obs_vx_list[0],&gaussgrids[0][0],3,VxObsEnum);
	inputs->GetParameterValues(&obs_vy_list[0],&gaussgrids[0][0],3,VyObsEnum);
	inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxEnum);
	inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyEnum);
	inputs->GetParameterValues(&weights_list[0],&gaussgrids[0][0],3,WeightsEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);
	
	/* Compute Misfit at the 3 nodes
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */
	if(fit==0){
		/*We are using an absolute misfit:
		 *
		 *      1  [           2              2 ]
		 * J = --- | (u - u   )  +  (v - v   )  |
		 *      2  [       obs            obs   ]
		 *
		 */
		for (i=0;i<numgrids;i++){
			misfit_list[i]=0.5*(pow((vx_list[i]-obs_vx_list[i]),(double)2)+pow((vy_list[i]-obs_vy_list[i]),(double)2));
		}
	}
	else if(fit==1){
		/*We are using a relative misfit: 
		 *                        
		 *      1  [     \bar{v}^2             2   \bar{v}^2              2 ]
		 * J = --- | -------------  (u - u   ) + -------------  (v - v   )  |
		 *      2  [  (u   + eps)^2       obs    (v   + eps)^2       obs    ]
		 *              obs                        obs                      
		 */
		for (i=0;i<numgrids;i++){
			scalex=pow(meanvel/(obs_vx_list[i]+epsvel),(double)2);
			scaley=pow(meanvel/(obs_vy_list[i]+epsvel),(double)2);
			if(obs_vx_list[i]==0)scalex=0;
			if(obs_vy_list[i]==0)scaley=0;
			misfit_list[i]=0.5*(scalex*pow((vx_list[i]-obs_vx_list[i]),2)+scaley*pow((vy_list[i]-obs_vy_list[i]),2));
		}
	}
	else if(fit==2){
		/*We are using a logarithmic misfit:
		*                        
		*                 [        vel + eps     ] 2
		* J = 4 \bar{v}^2 | log ( -----------  ) |  
		*                 [       vel   + eps    ]
		*                            obs
		*/
		for (i=0;i<numgrids;i++){
			velocity_mag=sqrt(pow(vx_list[i],(double)2)+pow(vy_list[i],(double)2))+epsvel; //epsvel to avoid velocity being nil.
			obs_velocity_mag=sqrt(pow(obs_vx_list[i],(double)2)+pow(obs_vy_list[i],(double)2))+epsvel; //epsvel to avoid observed velocity being nil.
			misfit_list[i]=4*pow(meanvel,(double)2)*pow(log(velocity_mag/obs_velocity_mag),(double)2);
		}
	}
	else if(fit==3){
		/*We are using an spacially average absolute misfit:
		 *
		 *      1                    2              2
		 * J = ---  sqrt(  (u - u   )  +  (v - v   )  )
		 *      S                obs            obs
		 */
		for (i=0;i<numgrids;i++){
			misfit_list[i]=sqrt(pow(vx_list[i]-obs_vx_list[i],2)+pow(vy_list[i]-obs_vx_list[i],2))/S;
		}
	}
	else if(fit==4){
		/*We are using an logarithmic 2 misfit:
		 *
		 *      1            [        |u| + eps     2          |v| + eps     2  ]
		 * J = --- \bar{v}^2 | log ( -----------  )   +  log ( -----------  )   |  
		 *      2            [       |u    |+ eps              |v    |+ eps     ]
		 *                              obs                       obs
		 */
		for (i=0;i<numgrids;i++){
			misfit_list[i]=0.5*pow(meanvel,(double)2)*(
			  pow(log((fabs(vx_list[i])+epsvel)/(fabs(obs_vx_list[i])+epsvel)),(double)2) +
			  pow(log((fabs(vy_list[i])+epsvel)/(fabs(obs_vy_list[i])+epsvel)),(double)2) );
		}
	}
	else{
		/*Not supported yet! : */
		ISSMERROR("%s%g","unsupported type of fit: ",fit);
	}

	/*Apply weights to misfits*/
	for (i=0;i<numgrids;i++){
		misfit_list[i]=weights_list[i]*misfit_list[i];
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/*Compute misfit at gaussian point: */
		GetParameterValue(&misfit, &misfit_list[0],gauss_l1l2l3);

		/*compute Misfit*/
		Jelem+=misfit*Jdet*gauss_weight;
	}

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
		
	/*Return: */
	return Jelem;
}
/*}}}*/
/*FUNCTION Tria::PatchFill{{{1*/
void  Tria::PatchFill(int* prow, Patch* patch){

	int i;
	int row;
	int vertices_ids[3];


	/*recover pointer: */
	row=*prow;
		
	/*will be needed later: */
	for(i=0;i<3;i++) vertices_ids[i]=nodes[i]->GetVertexId(); //vertices id start at column 3 of the patch.

	for(i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);

		/*For this result,fill the information in the Patch object (element id + vertices ids), and then hand 
		 *it to the result object, to fill the rest: */
		patch->fillelementinfo(row,this->id,vertices_ids,3);
		elementresult->PatchFill(row,patch);

		/*increment rower: */
		row++;
	}

	/*Assign output pointers:*/
	*prow=row;
}
/*}}}*/
/*FUNCTION Tria::PatchSize{{{1*/
void  Tria::PatchSize(int* pnumrows, int* pnumvertices,int* pnumnodes){

	int     i;
	
	/*output: */
	int     numrows     = 0;
	int     numvertices = 0;
	int     numnodes    = 0;

	/*Go through all the results objects, and update the counters: */
	for (i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);
		/*first, we have one more result: */
		numrows++;
		/*now, how many vertices and how many nodal values for this result? :*/
		numvertices=3; //this is a tria element, with 3 vertices
		numnodes=elementresult->NumberOfNodalValues(); //ask result object.
	}

	/*Assign output pointers:*/
	*pnumrows=numrows;
	*pnumvertices=numvertices;
	*pnumnodes=numnodes;
	
}
/*}}}*/
/*FUNCTION Tria::ProcessResultsUnits{{{1*/
void  Tria::ProcessResultsUnits(void){

	int i;

	for(i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);
		elementresult->ProcessUnits(this->parameters);
	}

}
/*}}}*/
/*FUNCTION Tria::SurfaceArea {{{1*/
double Tria::SurfaceArea(void){

	int i;

	/* output: */
	double S;

	/* node data: */
	int numgrids=3;
	double xyz_list[numgrids][3];
	double v13[3];
	double v23[3];
	double normal[3];

	/*inputs: */
	bool onwater;
	int fit;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&fit,FitEnum);
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If fit!=3, do not compute surface: */
	if(fit!=3)return 0;

	/*If on water, return 0: */
	if(onwater)return 0;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	S = 0.5 * sqrt(pow(normal[0],(double)2)+pow(normal[1],(double)2)+pow(normal[2],(double)2));

	/*Return: */
	return S;
}
/*}}}*/
/*FUNCTION Tria::Update{{{1*/
void Tria::Update(int index, IoModel* iomodel,int analysis_counter,int analysis_type){ //i is the element index

	/*Intermediaries*/
	int    i;
	int    tria_node_ids[3];
	int    tria_vertex_ids[3];
	double nodeinputs[3];
	int    tria_type;

	/*Checks if debuging*/
	/*{{{2*/
	ISSMASSERT(iomodel->elements);
	/*}}}*/

	/*Recover element type*/
	if ((analysis_type==PrognosticAnalysisEnum || analysis_type==BalancedthicknessAnalysisEnum) && iomodel->prognostic_DG){

		/*P1 Discontinuous Galerkin*/
		tria_type=P1DGEnum;
	}
	else{
		/*P1 Continuous Galerkin*/
		tria_type=P1Enum;
	}

	/*Set TriaRef*/
	this->SetElementType(tria_type,analysis_counter);

	/*Recover vertices ids needed to initialize inputs*/
	for(i=0;i<3;i++){ 
		tria_vertex_ids[i]=(int)iomodel->elements[3*index+i]; //ids for vertices are in the elements array from Matlab
	}

	/*Recover nodes ids needed to initialize the node hook.*/
	if (tria_type==P1DGEnum){
		/*Discontinuous Galerkin*/
		tria_node_ids[0]=iomodel->nodecounter+3*index+1;
		tria_node_ids[1]=iomodel->nodecounter+3*index+2;
		tria_node_ids[2]=iomodel->nodecounter+3*index+3;
	}
	else{
		/*Continuous Galerkin*/
		for(i=0;i<3;i++){ 
			tria_node_ids[i]=iomodel->nodecounter+(int)*(iomodel->elements+3*index+i); //ids for vertices are in the elements array from Matlab
		}
	}

	/*hooks: */
	this->SetHookNodes(tria_node_ids,analysis_counter); this->nodes=NULL; //set hook to nodes, for this analysis type
	
	/*add as many inputs per element as requested:*/
	if (iomodel->thickness) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->thickness[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(ThicknessEnum,nodeinputs));
	}
	if (iomodel->surface) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->surface[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(SurfaceEnum,nodeinputs));
	}
	if (iomodel->bed) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->bed[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(BedEnum,nodeinputs));
	}
	if (iomodel->drag_coefficient) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->drag_coefficient[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(DragCoefficientEnum,nodeinputs));

		if (iomodel->drag_p) this->inputs->AddInput(new DoubleInput(DragPEnum,iomodel->drag_p[index]));
		if (iomodel->drag_q) this->inputs->AddInput(new DoubleInput(DragQEnum,iomodel->drag_q[index]));
		this->inputs->AddInput(new IntInput(DragTypeEnum,iomodel->drag_type));
	}
	if (iomodel->rheology_B) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->rheology_B[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(RheologyBEnum,nodeinputs));
	}
	if (iomodel->control_parameter) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->control_parameter[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(ControlParameterEnum,nodeinputs));
	}
	if (iomodel->melting_rate) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->melting_rate[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(MeltingRateEnum,nodeinputs));
	}
	if (iomodel->accumulation_rate) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->accumulation_rate[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(AccumulationRateEnum,nodeinputs));
	}
	if (iomodel->geothermalflux) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->geothermalflux[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(GeothermalFluxEnum,nodeinputs));
	}
	if (iomodel->dhdt) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->dhdt[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(DhDtEnum,nodeinputs));
	}
	if (iomodel->pressure) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->pressure[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(PressureEnum,nodeinputs));
	}
	if (iomodel->temperature) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->temperature[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(TemperatureEnum,nodeinputs));
	}
	/*vx,vy and vz: */
	if (iomodel->vx) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vx[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VxEnum,nodeinputs));
		this->inputs->AddInput(new TriaVertexInput(VxOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new TriaVertexInput(QmuVxEnum,nodeinputs));
	}
	if (iomodel->vy) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vy[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VyEnum,nodeinputs));
		this->inputs->AddInput(new TriaVertexInput(VyOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new TriaVertexInput(QmuVyEnum,nodeinputs));
	}
	if (iomodel->vz) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vz[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VzEnum,nodeinputs));
		this->inputs->AddInput(new TriaVertexInput(VzOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new TriaVertexInput(QmuVzEnum,nodeinputs));
	}
	if (iomodel->vx_obs) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vx_obs[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VxObsEnum,nodeinputs));
	}
	if (iomodel->vy_obs) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vy_obs[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VyObsEnum,nodeinputs));
	}
	if (iomodel->vz_obs) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vz_obs[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VzObsEnum,nodeinputs));
	}
	if (iomodel->weights) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->weights[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(WeightsEnum,nodeinputs));
	}
	if (iomodel->elementoniceshelf) this->inputs->AddInput(new BoolInput(ElementOnIceShelfEnum,(IssmBool)iomodel->elementoniceshelf[index]));
	if (iomodel->elementonbed) this->inputs->AddInput(new BoolInput(ElementOnBedEnum,(IssmBool)iomodel->elementonbed[index]));
	if (iomodel->elementonwater) this->inputs->AddInput(new BoolInput(ElementOnWaterEnum,(IssmBool)iomodel->elementonwater[index]));
	if (iomodel->elementonsurface) this->inputs->AddInput(new BoolInput(ElementOnSurfaceEnum,(IssmBool)iomodel->elementonsurface[index]));

	/*Defaults if not provided in iomodel*/
	switch(analysis_type){

		case DiagnosticHorizAnalysisEnum: case DiagnosticVertAnalysisEnum: case DiagnosticStokesAnalysisEnum:

			/*default vx,vy and vz: either observation or 0 */
			if(!iomodel->vx){
				if (false && iomodel->vx_obs) for(i=0;i<3;i++)nodeinputs[i]=iomodel->vx_obs[tria_vertex_ids[i]-1]/iomodel->yts; //false TO BE DELETED (for NR CM)
				else                 for(i=0;i<3;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new TriaVertexInput(VxEnum,nodeinputs));
				this->inputs->AddInput(new TriaVertexInput(VxOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new TriaVertexInput(QmuVxEnum,nodeinputs));
			}
			if(!iomodel->vy){
				if (false && iomodel->vy_obs) for(i=0;i<3;i++)nodeinputs[i]=iomodel->vy_obs[tria_vertex_ids[i]-1]/iomodel->yts; //false TO BE DELETED (for NR CM)
				else                 for(i=0;i<3;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new TriaVertexInput(VyEnum,nodeinputs));
				this->inputs->AddInput(new TriaVertexInput(VyOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new TriaVertexInput(QmuVyEnum,nodeinputs));
			}
			if(!iomodel->vz){
				if (iomodel->vz_obs) for(i=0;i<3;i++)nodeinputs[i]=iomodel->vz_obs[tria_vertex_ids[i]-1]/iomodel->yts;
				else                 for(i=0;i<3;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new TriaVertexInput(VzEnum,nodeinputs));
				this->inputs->AddInput(new TriaVertexInput(VzOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new TriaVertexInput(QmuVzEnum,nodeinputs));
			}
			if(!iomodel->pressure){
				for(i=0;i<3;i++)nodeinputs[i]=0;
				if(iomodel->qmu_analysis){
					this->inputs->AddInput(new TriaVertexInput(PressureEnum,nodeinputs));
					this->inputs->AddInput(new TriaVertexInput(QmuPressureEnum,nodeinputs));
				}
			}
			break;

		default:
			/*No update for other solution types*/
			break;

	}

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;

}
/*}}}*/
/*FUNCTION Tria::UpdateGeometry{{{1*/
void  Tria::UpdateGeometry(void){

	/*Intermediaries*/
	double rho_ice,rho_water;

	/*If shelf: hydrostatic equilibrium*/
	if (this->GetShelf()){

		/*recover material parameters: */
		rho_ice=matpar->GetRhoIce();
		rho_water=matpar->GetRhoWater();

		/*Create New Surface: s = (1-rho_ice/rho_water) h*/
		InputDuplicate(ThicknessEnum,SurfaceEnum);     //1: copy thickness into surface
		InputScale(SurfaceEnum,(1.0-rho_ice/rho_water)); //2: surface = surface * (1-di)

		/*Create New Bed b = -rho_ice/rho_water h*/
		InputDuplicate(ThicknessEnum,BedEnum);         //1: copy thickness into bed
		InputScale(BedEnum, -rho_ice/rho_water);       //2: bed = bed * (-di)
	}

	/*If sheet: surface = bed + thickness*/
	else{

		/*The bed does not change, update surface only s = b + h*/
		InputDuplicate(BedEnum,SurfaceEnum);          //1: copy bed into surface
		InputAXPY(SurfaceEnum,1.0,ThicknessEnum);     //2: surface = surface + 1 * thickness
	}

}
/*}}}*/

/*Tria specific routines: */
/*FUNCTION Tria::CreateKMatrixBalancedthickness_CG {{{1*/
void  Tria::CreateKMatrixBalancedthickness_CG(Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double L[numgrids];
	double B[2][numgrids];
	double Bprime[2][numgrids];
	double DL[2][2]={0.0};
	double DLprime[2][2]={0.0};
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0};//local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_thickness1[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_thickness2[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.

	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  dvx[2];
	double  dvy[2];
	double  vx,vy;
	double  dvxdx,dvydy;
	double  v_gauss[2]={0.0};


	double  K[2][2]={0.0};
	double  KDL[2][2]={0.0};
	int     dofs[2]={0,1};
	int     found=0;

	/*inputs: */
	Input* vxaverage_input=NULL;
	Input* vyaverage_input=NULL;
	bool artdiff;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/*retrieve some parameters: */
	this->parameters->FindParam(&artdiff,ArtDiffEnum);

	/*Retrieve all inputs we will be needed*/
	vxaverage_input=inputs->GetInput(VxAverageEnum);
	vyaverage_input=inputs->GetInput(VyAverageEnum);

	//Create Artificial diffusivity once for all if requested
	if(artdiff){
		//Get the Jacobian determinant
		gauss_l1l2l3[0]=ONETHIRD; gauss_l1l2l3[1]=ONETHIRD; gauss_l1l2l3[2]=ONETHIRD;
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		//Build K matrix (artificial diffusivity matrix)
		vxaverage_input->GetParameterAverage(&v_gauss[0]);
		vyaverage_input->GetParameterAverage(&v_gauss[1]);

		K[0][0]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[0]);
		K[1][1]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[1]);
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get B  and B prime matrix: */
		GetB_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime_prog(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);

		//Get vx, vy and their derivatives at gauss point
		vxaverage_input->GetParameterValue(&vx, &gauss_l1l2l3[0]);
		vyaverage_input->GetParameterValue(&vy, &gauss_l1l2l3[0]);

		vxaverage_input->GetParameterDerivativeValue(&dvx[0],&xyz_list[0][0],&gauss_l1l2l3[0]);
		vyaverage_input->GetParameterDerivativeValue(&dvy[0],&xyz_list[0][0],&gauss_l1l2l3[0]);

		dvxdx=dvx[0];
		dvydy=dvy[1];

		DL_scalar=gauss_weight*Jdettria;

		//Create DL and DLprime matrix
		DL[0][0]=DL_scalar*dvxdx;
		DL[1][1]=DL_scalar*dvydy;

		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		//Do the triple product tL*D*L. 
		//Ke_gg_thickness=B'*DLprime*Bprime;

		TripleMultiply( &B[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&B[0][0],2,numdof,0,
					&Ke_gg_thickness1[0][0],0);

		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg_thickness2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_thickness1[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_thickness2[i][j];

		if(artdiff){

			/* Compute artificial diffusivity */
			KDL[0][0]=DL_scalar*K[0][0];
			KDL[1][1]=DL_scalar*K[1][1];

			TripleMultiply( &Bprime[0][0],2,numdof,1,
						&KDL[0][0],2,2,0,
						&Bprime[0][0],2,numdof,0,
						&Ke_gg_gaussian[0][0],0);

			/* Add artificial diffusivity matrix */
			for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

		}
	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixBalancedthickness_DG {{{1*/
void  Tria::CreateKMatrixBalancedthickness_DG(Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double B[2][numgrids];
	double Bprime[2][numgrids];
	double DL[2][2]={0.0};
	double DLprime[2][2]={0.0};
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg2[numdof][numdof]={0.0};
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  vx,vy;
	int     dofs[1]={0};
	int     found;

	/*inputs: */
	Input* vx_input=NULL;
	Input* vy_input=NULL;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Retrieve all inputs we will be needed*/
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get B  and B prime matrix: */
		/*WARNING: B and Bprime are inverted compared to usual prognostic!!!!*/
		GetB_prog(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);

		//Get vx, vy and their derivatives at gauss point
		vx_input->GetParameterValue(&vx, &gauss_l1l2l3[0]);
		vy_input->GetParameterValue(&vy, &gauss_l1l2l3[0]);

		DL_scalar=-gauss_weight*Jdettria;

		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		//Do the triple product tL*D*L. 
		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg2[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixBalancedvelocities {{{1*/
void  Tria::CreateKMatrixBalancedvelocities(Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double L[numgrids];
	double B[2][numgrids];
	double Bprime[2][numgrids];
	double DL[2][2]={0.0};
	double DLprime[2][2]={0.0};
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0};//local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_velocities1[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_velocities2[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  surface_normal[3];
	double  surface_list[3];
	double  nx,ny,norm;
	double  dvx[2];
	double  dvy[2];
	double  vx,vy;
	double  dvxdx,dvydy;
	double  v_gauss[2]={0.0};
	double  K[2][2]={0.0};
	double  KDL[2][2]={0.0};
	int     dofs[2]={0,1};
	int     found=0;

	/*inputs: */
	Input* vxaverage_input=NULL;
	Input* vyaverage_input=NULL;
	bool artdiff;

	/*retrieve some parameters: */
	this->parameters->FindParam(&artdiff,ArtDiffEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/*Modify z so that it reflects the surface*/
	inputs->GetParameterValues(&surface_list[0],&gaussgrids[0][0],3,SurfaceEnum);
	for(i=0;i<numgrids;i++) xyz_list[i][2]=surface_list[i];

	/*Get normal vector to the surface*/
	inputs->GetParameterAverage(&nx,VxAverageEnum);
	inputs->GetParameterAverage(&ny,VyAverageEnum);
	if(nx==0 && ny==0){
		SurfaceNormal(&surface_normal[0],xyz_list);
		nx=surface_normal[0];
		ny=surface_normal[1];
	}
	if(nx==0 && ny==0){
		nx=0;
		ny=1;
	}
	norm=pow( pow(nx,2)+pow(ny,2) , (double).5);
	nx=nx/norm;
	ny=ny/norm;

	/*Retrieve all inputs we will be needed*/
	vxaverage_input=inputs->GetInput(VxAverageEnum);
	vyaverage_input=inputs->GetInput(VyAverageEnum);

	//Create Artificial diffusivity once for all if requested
	if(artdiff){
		//Get the Jacobian determinant
		gauss_l1l2l3[0]=ONETHIRD; gauss_l1l2l3[1]=ONETHIRD; gauss_l1l2l3[2]=ONETHIRD;
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		//Build K matrix (artificial diffusivity matrix)
		vxaverage_input->GetParameterAverage(&v_gauss[0]);
		vyaverage_input->GetParameterAverage(&v_gauss[1]);

		K[0][0]=pow(10,2)*pow(Jdettria,(double).5)/2.0*fabs(v_gauss[0]); //pow should be zero!!
		K[1][1]=pow(10,2)*pow(Jdettria,(double).5)/2.0*fabs(v_gauss[1]);
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get B  and B prime matrix: */
		GetB_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime_prog(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);

		//Get vx, vy and their derivatives at gauss point
		vxaverage_input->GetParameterValue(&vx,&gauss_l1l2l3[0]);
		vyaverage_input->GetParameterValue(&vy,&gauss_l1l2l3[0]);

		vxaverage_input->GetParameterDerivativeValue(&dvx[0],&xyz_list[0][0],&gauss_l1l2l3[0]);
		vyaverage_input->GetParameterDerivativeValue(&dvy[0],&xyz_list[0][0],&gauss_l1l2l3[0]);

		dvxdx=dvx[0];
		dvydy=dvy[1];

		DL_scalar=gauss_weight*Jdettria;

		DLprime[0][0]=DL_scalar*nx;
		DLprime[1][1]=DL_scalar*ny;

		//Do the triple product tL*D*L. 
		//Ke_gg_velocities=B'*DLprime*Bprime;
		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg_velocities2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_velocities2[i][j];

		if(artdiff){

			/* Compute artificial diffusivity */
			KDL[0][0]=DL_scalar*K[0][0];
			KDL[1][1]=DL_scalar*K[1][1];

			TripleMultiply( &Bprime[0][0],2,numdof,1,
						&KDL[0][0],2,2,0,
						&Bprime[0][0],2,numdof,0,
						&Ke_gg_gaussian[0][0],0);

			/* Add artificial diffusivity matrix */
			for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

		}

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);


cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticHoriz {{{1*/
void  Tria::CreateKMatrixDiagnosticHoriz(Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* material data: */
	double viscosity; //viscosity
	double newviscosity; //viscosity
	double oldviscosity; //viscosity

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/
	double oldepsilon[3]; /* oldepsilon=[exx,eyy,exy];*/

	/* matrices: */
	double B[3][numdof];
	double Bprime[3][numdof];
	double D[3][3]={0.0};  // material matrix, simple scalar matrix.
	double D_scalar;

	/*parameters: */
	double viscosity_overshoot;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.

	double Jdet;

	/*input parameters for structural analysis (diagnostic): */
	double  thickness;
	int     dofs[2]={0,1};

	/*inputs: */
	Input* thickness_input=NULL;
	Input* vx_input=NULL;
	Input* vy_input=NULL;
	Input* vxold_input=NULL;
	Input* vyold_input=NULL;
	bool onwater,shelf;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&viscosity_overshoot,ViscosityOvershootEnum);

	/*First, if we are on water, return empty matrix: */
	if(onwater) return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Retrieve all inputs we will be needing: */
	thickness_input=inputs->GetInput(ThicknessEnum);
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);
	vxold_input=inputs->GetInput(VxOldEnum);
	vyold_input=inputs->GetInput(VyOldEnum);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);


		/*Compute thickness at gaussian point: */
		thickness_input->GetParameterValue(&thickness, gauss_l1l2l3);

		/*Get strain rate from velocity: */
		this->GetStrainRate2d(&epsilon[0],&xyz_list[0][0],gauss_l1l2l3,vx_input,vy_input);
		this->GetStrainRate2d(&oldepsilon[0],&xyz_list[0][0],gauss_l1l2l3,vxold_input,vyold_input);

		/*Get viscosity: */
		matice->GetViscosity2d(&viscosity, &epsilon[0]);
		matice->GetViscosity2d(&oldviscosity, &oldepsilon[0]);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
			onto this scalar matrix, so that we win some computational time: */
		newviscosity=viscosity+viscosity_overshoot*(viscosity-oldviscosity);
		D_scalar=newviscosity*thickness*gauss_weight*Jdet;

		for (i=0;i<3;i++){
			D[i][i]=D_scalar;
		}

		/*Get B and Bprime matrices: */
		GetB(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);

		/*  Do the triple product tB*D*Bprime: */
		TripleMultiply( &B[0][0],3,numdof,1,
					&D[0][0],3,3,0,
					&Bprime[0][0],3,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	/*Do not forget to include friction: */
	if(!shelf){
		CreateKMatrixDiagnosticHorizFriction(Kgg);
	}

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticHorizFriction {{{1*/
void  Tria::CreateKMatrixDiagnosticHorizFriction(Mat Kgg){

	/* local declarations */
	int       i,j;
	int       analysis_type;

	/* node data: */
	const int numvertices = 3;
	const int numdof   = 2 *numvertices;
	double    xyz_list[numvertices][3];
	int       doflist[numdof];
	int       numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double *first_gauss_area_coord  = NULL;
	double *second_gauss_area_coord = NULL;
	double *third_gauss_area_coord  = NULL;
	double *gauss_weights           = NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double L[2][numdof];
	double DL[2][2]={{ 0,0 },{0,0}}; //for basal drag
	double DL_scalar;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix contribution from drag
	
	double Jdet;
	
	/*slope: */
	double  slope[2]={0.0,0.0};
	double  slope_magnitude;

	/*friction: */
	Friction *friction = NULL;
	double    alpha2;
	double    alpha2_list[numvertices];                                       //TO BE DELETED
	double    gauss[numvertices][numvertices] = {{1,0,0},{0,1,0},{0,0,1}}; //TO BE DELETED

	double MAXSLOPE=.06; // 6 %
	double MOUNTAINKEXPONENT=10;

	/*inputs: */
	bool shelf;
	int  drag_type;
	Input* surface_input=NULL;
	Input* vx_input=NULL;
	Input* vy_input=NULL;
	Input* vz_input=NULL;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*retrieve inputs :*/
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	surface_input=inputs->GetInput(SurfaceEnum);
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);
	vz_input=inputs->GetInput(VzEnum);
	
	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numvertices);
	GetDofList(&doflist[0],&numberofdofspernode);

	if (shelf){
		/*no friction, do nothing*/
		return;
	}

	/*build friction object, used later on: */
	if (drag_type!=2)ISSMERROR(" non-viscous friction not supported yet!");
	friction=new Friction("2d",inputs,matpar,analysis_type);

	/*COMPUT alpha2_list (TO BE DELETED)*/
	for(i=0;i<numvertices;i++){
		friction->GetAlpha2(&alpha2_list[i],&gauss[i][0],VxEnum,VyEnum,VzEnum);
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Friction: */
		// friction->GetAlpha2(&alpha2, gauss_l1l2l3,VxEnum,VyEnum,VzEnum); // TO UNCOMMENT
		GetParameterValue(&alpha2,&alpha2_list[0],gauss_l1l2l3); // TO BE DELETED

		// If we have a slope > 6% for this element,  it means  we are on a mountain. In this particular case, 
		//velocity should be = 0. To achieve this result, we set alpha2_list to a very high value: */
		surface_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],&gauss_l1l2l3[0]);
		slope_magnitude=sqrt(pow(slope[0],2)+pow(slope[1],2));

		if (slope_magnitude>MAXSLOPE){
			alpha2=pow((double)10,MOUNTAINKEXPONENT);
		}

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0][0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		
		DL_scalar=alpha2*gauss_weight*Jdet;
		for (i=0;i<2;i++){
			DL[i][i]=DL_scalar;
		}
		
		/*  Do the triple producte tL*D*L: */
		TripleMultiply( &L[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&L[0][0],2,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
	delete friction;

}	
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticHutter{{{1*/
void  Tria::CreateKMatrixDiagnosticHutter(Mat Kgg){

	/*Collapsed formulation: */
	Sing*  sing=NULL;
	int    i;

	/*flags: */
	bool onwater;

	/*recover some inputs: */
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Spawn 3 sing elements: */
	for(i=0;i<3;i++){

		/*Create Sing and call sing method*/
		sing=(Sing*)SpawnSing(i);
		sing->CreateKMatrix(Kgg);

		/*clean up*/
		delete sing;
	}

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticSurfaceVert {{{1*/
void  Tria::CreateKMatrixDiagnosticSurfaceVert(Mat Kgg){

	int i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];


	/* surface normal: */
	double x4,y4,z4;
	double x5,y5,z5;
	double x6,y6,z6;
	double v46[3];
	double v56[3];
	double normal[3];
	double norm_normal;
	double nz;

	/*Matrices: */
	double DL_scalar;
	double L[3];
	double Jdet;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]={0.0}; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Build normal vector to the surface:*/

	x4=xyz_list[0][0];
	y4=xyz_list[0][1];
	z4=xyz_list[0][2];

	x5=xyz_list[1][0];
	y5=xyz_list[1][1];
	z5=xyz_list[1][2];

	x6=xyz_list[2][0];
	y6=xyz_list[2][1];
	z6=xyz_list[2][2];

	v46[0]=x4-x6;
	v46[1]=y4-y6;
	v46[2]=z4-z6;

	v56[0]=x5-x6;
	v56[1]=y5-y6;
	v56[2]=z5-z6;

	normal[0]=(y4-y6)*(z5-z6)-(z4-z6)*(y5-y6);
	normal[1]=(z4-z6)*(x5-x6)-(x4-x6)*(z5-z6);
	normal[2]=(x4-x6)*(y5-y6)-(y4-y6)*(x5-x6);

	norm_normal=sqrt(pow(normal[0],(double)2)+pow(normal[1],(double)2)+pow(normal[2],(double)2));
	nz=1.0/norm_normal*normal[2];

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		//Get L matrix if viscous basal drag present:
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

		/**********************Do not forget the sign**********************************/
		DL_scalar=- gauss_weight*Jdet*nz; 
		/******************************************************************************/

		/*  Do the triple producte tL*D*L: */
		TripleMultiply( L,1,3,1,
					&DL_scalar,1,1,0,
					L,1,3,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];


	} //for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixMelting {{{1*/
void  Tria::CreateKMatrixMelting(Mat Kgg){

	/*indexing: */
	int i,j;

	const int  numgrids=3;
	const int  NDOF1=1;
	const int  numdof=numgrids*NDOF1;
	int        doflist[numdof];
	int        numberofdofspernode;

	/*Grid data: */
	double     xyz_list[numgrids][3];

	/*Material constants */
	double     heatcapacity,latentheat;

	/* gaussian points: */
	int     num_area_gauss,ig;
	double* gauss_weights  =  NULL;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];

	/*matrices: */
	double     Jdet;
	double     D_scalar;
	double     K_terms[numdof][numdof]={0.0};
	double     L[3];
	double     tLD[3];
	double     Ke_gaussian[numdof][numdof]={0.0};

	/*Recover constants of ice */
	latentheat=matpar->GetLatentHeat();
	heatcapacity=matpar->GetHeatCapacity();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights: */
	GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start looping on the number of gauss  (nodes on the bedrock) */
	for (ig=0; ig<num_area_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);

		//Get the Jacobian determinant
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0], gauss_coord);

		/*Get L matrix : */
		GetL(&L[0], &xyz_list[0][0], gauss_coord,NDOF1);

		/*Calculate DL on gauss point */
		D_scalar=latentheat/heatcapacity*gauss_weight*Jdet;

		/*  Do the triple product tL*D*L: */
		MatrixMultiply(&L[0],numdof,1,0,&D_scalar,1,1,0,&tLD[0],0);
		MatrixMultiply(&tLD[0],numdof,1,0,&L[0],1,numdof,0,&Ke_gaussian[0][0],0);

		for(i=0;i<numgrids;i++){
			for(j=0;j<numgrids;j++){
				K_terms[i][j]+=Ke_gaussian[i][j];
			}
		}
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)K_terms,ADD_VALUES);

cleanup_and_return:
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixPrognostic_CG {{{1*/
void  Tria::CreateKMatrixPrognostic_CG(Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double L[numgrids];
	double B[2][numgrids];
	double Bprime[2][numgrids];
	double DL[2][2]={0.0};
	double DLprime[2][2]={0.0};
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg_gaussian[numdof][numdof]={0.0};
	double Ke_gg_thickness1[numdof][numdof]={0.0};
	double Ke_gg_thickness2[numdof][numdof]={0.0};
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  dvx[2];
	double  dvy[2];
	double  vx,vy;
	double  dvxdx,dvydy;
	double  v_gauss[2]={0.0};
	double  K[2][2]={0.0};
	double  KDL[2][2]={0.0};
	int     dofs[2]={0,1};
	int     found;

	/*inputs: */
	Input* vxaverage_input=NULL;
	Input* vyaverage_input=NULL;
	double dt;
	bool artdiff;

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);
	this->parameters->FindParam(&artdiff,ArtDiffEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/*Retrieve all inputs we will be needing: */
	vxaverage_input=inputs->GetInput(VxAverageEnum);
	vyaverage_input=inputs->GetInput(VyAverageEnum);

	//Create Artificial diffusivity once for all if requested
	if(artdiff){
		//Get the Jacobian determinant
		gauss_l1l2l3[0]=ONETHIRD; gauss_l1l2l3[1]=ONETHIRD; gauss_l1l2l3[2]=ONETHIRD;
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		//Build K matrix (artificial diffusivity matrix)
		vxaverage_input->GetParameterAverage(&v_gauss[0]);
		vyaverage_input->GetParameterAverage(&v_gauss[1]);

		K[0][0]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[0]);
		K[1][1]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[1]);
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		DL_scalar=gauss_weight*Jdettria;

		/*  Do the triple product tL*D*L: */
		TripleMultiply( &L[0],1,numdof,1,
					&DL_scalar,1,1,0,
					&L[0],1,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		/*Get B  and B prime matrix: */
		GetB_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime_prog(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);

		//Get vx, vy and their derivatives at gauss point
		vxaverage_input->GetParameterValue(&vx,&gauss_l1l2l3[0]);
		vyaverage_input->GetParameterValue(&vy,&gauss_l1l2l3[0]);

		vxaverage_input->GetParameterDerivativeValue(&dvx[0],&xyz_list[0][0],&gauss_l1l2l3[0]);
		vyaverage_input->GetParameterDerivativeValue(&dvy[0],&xyz_list[0][0],&gauss_l1l2l3[0]);

		dvxdx=dvx[0];
		dvydy=dvy[1];

		DL_scalar=dt*gauss_weight*Jdettria;

		//Create DL and DLprime matrix
		DL[0][0]=DL_scalar*dvxdx;
		DL[1][1]=DL_scalar*dvydy;

		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		//Do the triple product tL*D*L. 
		//Ke_gg_thickness=B'*DL*B+B'*DLprime*Bprime;

		TripleMultiply( &B[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&B[0][0],2,numdof,0,
					&Ke_gg_thickness1[0][0],0);

		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg_thickness2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_thickness1[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_thickness2[i][j];

		if(artdiff){

			/* Compute artificial diffusivity */
			KDL[0][0]=DL_scalar*K[0][0];
			KDL[1][1]=DL_scalar*K[1][1];

			TripleMultiply( &Bprime[0][0],2,numdof,1,
						&KDL[0][0],2,2,0,
						&Bprime[0][0],2,numdof,0,
						&Ke_gg_gaussian[0][0],0);

			/* Add artificial diffusivity matrix */
			for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

		}

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);


cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixPrognostic_DG {{{1*/
void  Tria::CreateKMatrixPrognostic_DG(Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double L[numgrids];
	double B[2][numgrids];
	double Bprime[2][numgrids];
	double DL[2][2]={0.0};
	double DLprime[2][2]={0.0};
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg1[numdof][numdof]={0.0};
	double Ke_gg2[numdof][numdof]={0.0};
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  vx,vy;
	int     dofs[1]={0};
	int     found;

	/*inputs: */
	Input* vxaverage_input=NULL;
	Input* vyaverage_input=NULL;
	double dt;

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Retrieve all inputs we will be needed*/
	vxaverage_input=inputs->GetInput(VxAverageEnum);
	vyaverage_input=inputs->GetInput(VyAverageEnum);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		DL_scalar=gauss_weight*Jdettria;

		/*  Do the triple product tL*D*L: */
		TripleMultiply( &L[0],1,numdof,1,
					&DL_scalar,1,1,0,
					&L[0],1,numdof,0,
					&Ke_gg1[0][0],0);

		/*Get B  and B prime matrix: */
		/*WARNING: B and Bprime are inverted compared to usual prognostic!!!!*/
		GetB_prog(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);

		//Get vx, vy and their derivatives at gauss point
		vxaverage_input->GetParameterValue(&vx,&gauss_l1l2l3[0]);
		vyaverage_input->GetParameterValue(&vy,&gauss_l1l2l3[0]);

		DL_scalar=-dt*gauss_weight*Jdettria;

		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		//Do the triple product tL*D*L. 
		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg1[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg2[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);


cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixSlope {{{1*/
void  Tria::CreateKMatrixSlope(Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double L[1][3];
	double DL_scalar;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]={0.0}; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.
	
	double Jdet;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		
		/*Get L matrix: */
		GetL(&L[0][0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		DL_scalar=gauss_weight*Jdet;

		/*  Do the triple producte tL*D*L: */
		TripleMultiply( &L[0][0],1,3,1,
					&DL_scalar,1,1,0,
					&L[0][0],1,3,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
	} //for (ig=0; ig<num_gauss; ig++

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);
		
cleanup_and_return:
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixThermal {{{1*/
void  Tria::CreateKMatrixThermal(Mat Kgg){

	int i,j;
	int found=0;
	
	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	double mixed_layer_capacity;
	double thermal_exchange_velocity;
	double rho_water;
	double rho_ice;
	double heatcapacity;

	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];

	/*matrices: */
	double  Jdet;
	double  K_terms[numdof][numdof]={0.0};
	double  Ke_gaussian[numdof][numdof]={0.0};
	double  l1l2l3[numgrids];
	double     tl1l2l3D[3];
	double  D_scalar;

	/*parameters: */
	double dt;

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	//recover material parameters
	mixed_layer_capacity=matpar->GetMixedLayerCapacity();
	thermal_exchange_velocity=matpar->GetThermalExchangeVelocity();
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();

	GaussTria (&num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start looping on the number of gauss (nodes on the bedrock) */
	for (ig=0; ig<num_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);
		
		//Get the Jacobian determinant
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0], gauss_coord);
		
		/*Get nodal functions values: */
		GetNodalFunctions(&l1l2l3[0], gauss_coord);
				
		/*Calculate DL on gauss point */
		D_scalar=gauss_weight*Jdet*rho_water*mixed_layer_capacity*thermal_exchange_velocity/(heatcapacity*rho_ice);
		if(dt){
			D_scalar=dt*D_scalar;
		}

		/*  Do the triple product tL*D*L: */
		MatrixMultiply(&l1l2l3[0],numdof,1,0,&D_scalar,1,1,0,&tl1l2l3D[0],0);
		MatrixMultiply(&tl1l2l3D[0],numdof,1,0,&l1l2l3[0],1,numdof,0,&Ke_gaussian[0][0],0);

		for(i=0;i<3;i++){
			for(j=0;j<3;j++){
				K_terms[i][j]+=Ke_gaussian[i][j];
			}
		}
	}
	
	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)K_terms,ADD_VALUES);

	cleanup_and_return:
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorBalancedthickness_CG{{{1*/
void  Tria::CreatePVectorBalancedthickness_CG(Vec pg ){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrix */
	double pe_g[numgrids]={0.0};
	double L[numgrids];
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  accumulation_g;
	double  melting_g;

	/*inputs: */
	Input* accumulation_input=NULL;
	Input* melting_input=NULL;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*retrieve inputs :*/
	accumulation_input=inputs->GetInput(AccumulationRateEnum);
	melting_input=inputs->GetInput(MeltingRateEnum);
	
	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: *
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/* Get accumulation, melting at gauss point */
		accumulation_input->GetParameterValue(&accumulation_g, &gauss_l1l2l3[0]);
		melting_input->GetParameterValue(&melting_g, &gauss_l1l2l3[0]);

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i]+=Jdettria*gauss_weight*(accumulation_g-melting_g)*L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorBalancedthickness_DG {{{1*/
void  Tria::CreatePVectorBalancedthickness_DG(Vec pg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrix */
	double pe_g[numgrids]={0.0};
	double L[numgrids];
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  accumulation_g;
	double  melting_g;
	double  dhdt_g;

	/*inputs: */
	Input* accumulation_input=NULL;
	Input* melting_input=NULL;
	Input* dhdt_input=NULL;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Retrieve all inputs we will be needing: */
	accumulation_input=inputs->GetInput(AccumulationRateEnum);
	melting_input=inputs->GetInput(MeltingRateEnum);
	dhdt_input=inputs->GetInput(DhDtEnum);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/* Get accumulation, melting and thickness at gauss point */
		accumulation_input->GetParameterValue(&accumulation_g, &gauss_l1l2l3[0]);
		melting_input->GetParameterValue(&melting_g, &gauss_l1l2l3[0]);
		dhdt_input->GetParameterValue(&dhdt_g, &gauss_l1l2l3[0]);

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i]+=Jdettria*gauss_weight*(accumulation_g-melting_g+dhdt_g)*L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorBalancedvelocities {{{1*/
void  Tria::CreatePVectorBalancedvelocities(Vec pg){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrix */
	double pe_g[numgrids]={0.0};
	double L[numgrids];
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  accumulation_g;
	double  melting_g;

	/*inputs: */
	Input* accumulation_input=NULL;
	Input* melting_input=NULL;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Retrieve all inputs we will be needing: */
	accumulation_input=inputs->GetInput(AccumulationRateEnum);
	melting_input=inputs->GetInput(MeltingRateEnum);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/* Get accumulation, melting at gauss point */
		accumulation_input->GetParameterValue(&accumulation_g, &gauss_l1l2l3[0]);
		melting_input->GetParameterValue(&melting_g, &gauss_l1l2l3[0]);

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i]+=Jdettria*gauss_weight*(accumulation_g-melting_g)*L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorDiagnosticBaseVert {{{1*/
void  Tria::CreatePVectorDiagnosticBaseVert(Vec pg){

	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*element vector at the gaussian points: */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];

	/* matrices: */
	double L[numgrids];

	/*input parameters for structural analysis (diagnostic): */
	double  vx,vy;
	double  meltingvalue;
	double  slope[2];
	double  dbdx,dbdy;

	/*inputs: */
	Input* melting_input=NULL;
	Input* vx_input=NULL;
	Input* vy_input=NULL;
	Input* bed_input=NULL;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Retrieve all inputs we will be needing: */
	melting_input=inputs->GetInput(MeltingRateEnum);
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);
	bed_input=inputs->GetInput(BedEnum);

	/*For icesheets: */
	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Get melting at gaussian point: */
		melting_input->GetParameterValue(&meltingvalue, &gauss_l1l2l3[0]);

		/*Get velocity at gaussian point: */
		vx_input->GetParameterValue(&vx, &gauss_l1l2l3[0]);
		vy_input->GetParameterValue(&vy, &gauss_l1l2l3[0]);

		/*Get bed slope: */
		bed_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],&gauss_l1l2l3[0]);
		dbdx=slope[0];
		dbdy=slope[1];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		//Get L matrix if viscous basal drag present:
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

		/*Build gaussian vector: */
		for(i=0;i<numgrids;i++){
			pe_g_gaussian[i]=-Jdet*gauss_weight*(vx*dbdx+vy*dbdy-meltingvalue)*L[i];
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorDiagnosticHoriz {{{1*/
void Tria::CreatePVectorDiagnosticHoriz( Vec pg){

	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* parameters: */
	double  plastic_stress; 
	double  slope[NDOF2];
	double  driving_stress_baseline;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*element vector at the gaussian points: */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];

	/*input parameters for structural analysis (diagnostic): */
	double  thickness;

	/*inputs: */
	bool onwater;
	int  drag_type;
	Input* thickness_input=NULL;
	Input* surface_input=NULL;
	Input* drag_input=NULL;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	thickness_input=inputs->GetInput(ThicknessEnum);
	surface_input=inputs->GetInput(SurfaceEnum);
	drag_input=inputs->GetInput(DragCoefficientEnum);

	/*First, if we are on water, return empty vector: */
	if(onwater)return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);


	/* Get gaussian points and weights: */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2); /*We need higher order because our load is order 2*/

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Compute thickness at gaussian point: */
		thickness_input->GetParameterValue(&thickness, &gauss_l1l2l3[0]);
		surface_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],&gauss_l1l2l3[0]);
		
		/*In case we have plastic basal drag, compute plastic stress at gaussian point from k1, k2 and k3 fields in the 
		 * element itself: */
		if(drag_type==1){
			drag_input->GetParameterValue(&plastic_stress, &gauss_l1l2l3[0]);
		}

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		 /*Get nodal functions: */
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Compute driving stress: */
		driving_stress_baseline=matpar->GetRhoIce()*matpar->GetG()*thickness;

		/*Build pe_g_gaussian vector: */
		if(drag_type==1){
			for (i=0;i<numgrids;i++){
				for (j=0;j<NDOF2;j++){
					pe_g_gaussian[i*NDOF2+j]=(-driving_stress_baseline*slope[j]-plastic_stress)*Jdet*gauss_weight*l1l2l3[i]; 
				}
			}
		}
		else {
			for (i=0;i<numgrids;i++){
				for (j=0;j<NDOF2;j++){
					pe_g_gaussian[i*NDOF2+j]=-driving_stress_baseline*slope[j]*Jdet*gauss_weight*l1l2l3[i];
				}
			}
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

	} //for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorAdjointHoriz{{{1*/
void Tria::CreatePVectorAdjointHoriz(Vec p_g){

	int i;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* grid data: */
	double vx_list[numgrids];
	double vy_list[numgrids];
	double obs_vx_list[numgrids];
	double obs_vy_list[numgrids];
	double dux_list[numgrids];
	double duy_list[numgrids];
	double weights_list[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* parameters: */
	double  obs_velocity_mag,velocity_mag;
	double  dux,duy;
	double  meanvel, epsvel;

	/*element vector : */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*relative and algorithmic fitting: */
	double scalex=0;
	double scaley=0;
	double scale=0;
	double S=0;
	int    fit=-1;

	/*retrieve some parameters: */
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Recover input data: */
	inputs->GetParameterValues(&obs_vx_list[0],&gaussgrids[0][0],3,VxObsEnum);
	inputs->GetParameterValues(&obs_vy_list[0],&gaussgrids[0][0],3,VyObsEnum);

	inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxEnum);
	inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyEnum);

	inputs->GetParameterValues(&weights_list[0],&gaussgrids[0][0],3,WeightsEnum);

	inputs->GetParameterValue(&fit,FitEnum);
	if(fit==3){
		inputs->GetParameterValue(&S,SurfaceAreaEnum);
	}

	/*Get Du at the 3 nodes (integration of the linearized function)
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 *       d J                  dJ_i
	 * DU= - --- = sum_{i=1}^3  - ---  Phi_i = sum_{i=1}^3 DU_i Phi_i
	 *       d u                  du_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */
	if(fit==0){
		/*We are using an absolute misfit:
		 *
		 *      1  [           2              2 ]
		 * J = --- | (u - u   )  +  (v - v   )  |
		 *      2  [       obs            obs   ]
		 *
		 *        dJ             2
		 * DU = - -- = (u   - u )
		 *        du     obs
		 */
		for (i=0;i<numgrids;i++){
			dux_list[i]=obs_vx_list[i]-vx_list[i];
			duy_list[i]=obs_vy_list[i]-vy_list[i];
		}
	}
	else if(fit==1){
		/*We are using a relative misfit: 
		 *                        
		 *      1  [     \bar{v}^2             2   \bar{v}^2              2 ]
		 * J = --- | -------------  (u - u   ) + -------------  (v - v   )  |
		 *      2  [  (u   + eps)^2       obs    (v   + eps)^2       obs    ]
		 *              obs                        obs                      
		 *
		 *        dJ     \bar{v}^2
		 * DU = - -- = ------------- (u   - u )
		 *        du   (u   + eps)^2    obs
		 *               obs
		 */
		for (i=0;i<numgrids;i++){
			scalex=pow(meanvel/(obs_vx_list[i]+epsvel),2);
			scaley=pow(meanvel/(obs_vy_list[i]+epsvel),2);
			if(obs_vx_list[i]==0)scalex=0;
			if(obs_vy_list[i]==0)scaley=0;
			dux_list[i]=scalex*(obs_vx_list[i]-vx_list[i]);
			duy_list[i]=scaley*(obs_vy_list[i]-vy_list[i]);
		}
	}
	else if(fit==2){
		/*We are using a logarithmic misfit:
		 *                        
		 *                 [        vel + eps     ] 2
		 * J = 4 \bar{v}^2 | log ( -----------  ) |  
		 *                 [       vel   + eps    ]
		 *                            obs
		 *
		 *        dJ                 2 * log(...)
		 * DU = - -- = - 4 \bar{v}^2 -------------  u
		 *        du                 vel^2 + eps
		 *            
		 */
		for (i=0;i<numgrids;i++){
			velocity_mag=sqrt(pow(vx_list[i],2)+pow(vy_list[i],2))+epsvel; //epsvel to avoid velocity being nil.
			obs_velocity_mag=sqrt(pow(obs_vx_list[i],2)+pow(obs_vy_list[i],2))+epsvel; //epsvel to avoid observed velocity being nil.
			scale=-8*pow(meanvel,2)/pow(velocity_mag,2)*log(velocity_mag/obs_velocity_mag);
			dux_list[i]=scale*vx_list[i];
			duy_list[i]=scale*vy_list[i];
		}
	}
	else if(fit==3){
		/*We are using an spacially average absolute misfit:
		 *
		 *      1                    2              2
		 * J = ---  sqrt(  (u - u   )  +  (v - v   )  )
		 *      S                obs            obs
		 *
		 *        dJ      1       1 
		 * DU = - -- = - --- ----------- * 2 (u - u   )
		 *        du      S  2 sqrt(...)           obs
		 */
		for (i=0;i<numgrids;i++){
			scale=1.0/(S*sqrt(pow(vx_list[i]-obs_vx_list[i],2)+pow(vy_list[i]-obs_vx_list[i],2))+epsvel);
			dux_list[i]=scale*(obs_vx_list[i]-vx_list[i]);
			duy_list[i]=scale*(obs_vy_list[i]-vy_list[i]);
		}
	}
	else if(fit==4){
		/*We are using an logarithmic 2 misfit:
		 *
		 *      1            [        |u| + eps     2          |v| + eps     2  ]
		 * J = --- \bar{v}^2 | log ( -----------  )   +  log ( -----------  )   |  
		 *      2            [       |u    |+ eps              |v    |+ eps     ]
		 *                              obs                       obs
		 *        dJ                              1      u                             1
		 * DU = - -- = - \bar{v}^2 log(u...) --------- ----  ~ - \bar{v}^2 log(u...) ------
		 *        du                         |u| + eps  |u|                           u + eps
		 */
		for (i=0;i<numgrids;i++){
			dux_list[i] = - pow(meanvel,(double)2)*(
						log((fabs(vx_list[i])+epsvel)/(fabs(obs_vx_list[i])+epsvel)) * 1/(vx_list[i]+epsvel));
			duy_list[i] = - pow(meanvel,(double)2)*(
						log((fabs(vy_list[i])+epsvel)/(fabs(obs_vy_list[i])+epsvel)) * 1/(vy_list[i]+epsvel));
		}
	}
	else{
		/*Not supported yet! : */
		ISSMERROR("%s%g","unsupported type of fit: ",fit);
	}

	/*Apply weights to DU*/
	for (i=0;i<numgrids;i++){
		dux_list[i]=weights_list[i]*dux_list[i];
		duy_list[i]=weights_list[i]*duy_list[i];
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Build due_g_gaussian vector: we have three cases here, according to which type of misfit we are using. */

		/*Compute absolute(x/y) at gaussian point: */
		GetParameterValue(&dux, &dux_list[0],gauss_l1l2l3);
		GetParameterValue(&duy, &duy_list[0],gauss_l1l2l3);

		/*compute Du*/
		for (i=0;i<numgrids;i++){
			pe_g_gaussian[i*NDOF2+0]=dux*Jdet*gauss_weight*l1l2l3[i]; 
			pe_g_gaussian[i*NDOF2+1]=duy*Jdet*gauss_weight*l1l2l3[i]; 
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++){
			pe_g[i]+=pe_g_gaussian[i];
		}
	}

	/*Add pe_g to global vector p_g: */
	VecSetValues(p_g,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	/*Clean up*/
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorAdjointStokes{{{1*/
void Tria::CreatePVectorAdjointStokes(Vec p_g){

	int i;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* grid data: */
	double vx_list[numgrids];
	double vy_list[numgrids];
	double obs_vx_list[numgrids];
	double obs_vy_list[numgrids];
	double dux_list[numgrids];
	double duy_list[numgrids];
	double weights_list[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* parameters: */
	double  obs_velocity_mag,velocity_mag;
	double  dux,duy;
	double  meanvel, epsvel;

	/*element vector : */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*relative and algorithmic fitting: */
	double scalex=0;
	double scaley=0;
	double scale=0;
	double S=0;
	int    fit=-1;

	/*retrieve some parameters: */
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Recover input data: */
	inputs->GetParameterValues(&obs_vx_list[0],&gaussgrids[0][0],3,VxObsEnum);
	inputs->GetParameterValues(&obs_vy_list[0],&gaussgrids[0][0],3,VyObsEnum);

	inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxEnum);
	inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyEnum);

	inputs->GetParameterValues(&weights_list[0],&gaussgrids[0][0],3,WeightsEnum);

	inputs->GetParameterValue(&fit,FitEnum);
	if(fit==3){
		inputs->GetParameterValue(&S,SurfaceAreaEnum);
	}

	/*Get Du at the 3 nodes (integration of the linearized function)
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 *       d J                  dJ_i
	 * DU= - --- = sum_{i=1}^3  - ---  Phi_i = sum_{i=1}^3 DU_i Phi_i
	 *       d u                  du_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */
	if(fit==0){
		/*We are using an absolute misfit:
		 *
		 *      1  [           2              2 ]
		 * J = --- | (u - u   )  +  (v - v   )  |
		 *      2  [       obs            obs   ]
		 *
		 *        dJ             2
		 * DU = - -- = (u   - u )
		 *        du     obs
		 */
		for (i=0;i<numgrids;i++){
			dux_list[i]=obs_vx_list[i]-vx_list[i];
			duy_list[i]=obs_vy_list[i]-vy_list[i];
		}
	}
	else if(fit==1){
		/*We are using a relative misfit: 
		 *                        
		 *      1  [     \bar{v}^2             2   \bar{v}^2              2 ]
		 * J = --- | -------------  (u - u   ) + -------------  (v - v   )  |
		 *      2  [  (u   + eps)^2       obs    (v   + eps)^2       obs    ]
		 *              obs                        obs                      
		 *
		 *        dJ     \bar{v}^2
		 * DU = - -- = ------------- (u   - u )
		 *        du   (u   + eps)^2    obs
		 *               obs
		 */
		for (i=0;i<numgrids;i++){
			scalex=pow(meanvel/(obs_vx_list[i]+epsvel),2);
			scaley=pow(meanvel/(obs_vy_list[i]+epsvel),2);
			if(obs_vx_list[i]==0)scalex=0;
			if(obs_vy_list[i]==0)scaley=0;
			dux_list[i]=scalex*(obs_vx_list[i]-vx_list[i]);
			duy_list[i]=scaley*(obs_vy_list[i]-vy_list[i]);
		}
	}
	else if(fit==2){
		/*We are using a logarithmic misfit:
		 *                        
		 *                 [        vel + eps     ] 2
		 * J = 4 \bar{v}^2 | log ( -----------  ) |  
		 *                 [       vel   + eps    ]
		 *                            obs
		 *
		 *        dJ                 2 * log(...)
		 * DU = - -- = - 4 \bar{v}^2 -------------  u
		 *        du                 vel^2 + eps
		 *            
		 */
		for (i=0;i<numgrids;i++){
			velocity_mag=sqrt(pow(vx_list[i],2)+pow(vy_list[i],2))+epsvel; //epsvel to avoid velocity being nil.
			obs_velocity_mag=sqrt(pow(obs_vx_list[i],2)+pow(obs_vy_list[i],2))+epsvel; //epsvel to avoid observed velocity being nil.
			scale=-8*pow(meanvel,2)/pow(velocity_mag,2)*log(velocity_mag/obs_velocity_mag);
			dux_list[i]=scale*vx_list[i];
			duy_list[i]=scale*vy_list[i];
		}
	}
	else if(fit==3){
		/*We are using an spacially average absolute misfit:
		 *
		 *      1                    2              2
		 * J = ---  sqrt(  (u - u   )  +  (v - v   )  )
		 *      S                obs            obs
		 *
		 *        dJ      1       1 
		 * DU = - -- = - --- ----------- * 2 (u - u   )
		 *        du      S  2 sqrt(...)           obs
		 */
		for (i=0;i<numgrids;i++){
			scale=1.0/(S*sqrt(pow(vx_list[i]-obs_vx_list[i],2)+pow(vy_list[i]-obs_vx_list[i],2))+epsvel);
			dux_list[i]=scale*(obs_vx_list[i]-vx_list[i]);
			duy_list[i]=scale*(obs_vy_list[i]-vy_list[i]);
		}
	}
	else if(fit==4){
		/*We are using an logarithmic 2 misfit:
		 *
		 *      1            [        |u| + eps     2          |v| + eps     2  ]
		 * J = --- \bar{v}^2 | log ( -----------  )   +  log ( -----------  )   |  
		 *      2            [       |u    |+ eps              |v    |+ eps     ]
		 *                              obs                       obs
		 *        dJ                              1      u                             1
		 * DU = - -- = - \bar{v}^2 log(u...) --------- ----  ~ - \bar{v}^2 log(u...) ------
		 *        du                         |u| + eps  |u|                           u + eps
		 */
		for (i=0;i<numgrids;i++){
			dux_list[i] = - pow(meanvel,(double)2)*(
						log((fabs(vx_list[i])+epsvel)/(fabs(obs_vx_list[i])+epsvel)) * 1/(vx_list[i]+epsvel));
			duy_list[i] = - pow(meanvel,(double)2)*(
						log((fabs(vy_list[i])+epsvel)/(fabs(obs_vy_list[i])+epsvel)) * 1/(vy_list[i]+epsvel));
		}
	}
	else{
		/*Not supported yet! : */
		ISSMERROR("%s%g","unsupported type of fit: ",fit);
	}

	/*Apply weights to DU*/
	for (i=0;i<numgrids;i++){
		dux_list[i]=weights_list[i]*dux_list[i];
		duy_list[i]=weights_list[i]*duy_list[i];
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Build due_g_gaussian vector: we have three cases here, according to which type of misfit we are using. */

		/*Compute absolute(x/y) at gaussian point: */
		GetParameterValue(&dux, &dux_list[0],gauss_l1l2l3);
		GetParameterValue(&duy, &duy_list[0],gauss_l1l2l3);

		/*compute Du*/
		for (i=0;i<numgrids;i++){
			pe_g_gaussian[i*NDOF2+0]=dux*Jdet*gauss_weight*l1l2l3[i]; 
			pe_g_gaussian[i*NDOF2+1]=duy*Jdet*gauss_weight*l1l2l3[i]; 
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++){
			pe_g[i]+=pe_g_gaussian[i];
		}
	}

	/*Add pe_g to global vector p_g: */
	VecSetValues(p_g,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	/*Clean up*/
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorDiagnosticHutter{{{1*/
void  Tria::CreatePVectorDiagnosticHutter(Vec pg){

	/*Collapsed formulation: */
	Sing*  sing=NULL;
	int    i;

	/*flags: */
	bool onwater;

	/*recover some inputs: */
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Spawn 3 sing elements: */
	for(i=0;i<3;i++){

		/*Create Sing and call sing method*/
		sing=(Sing*)SpawnSing(i);
		sing->CreatePVector(pg);

		/*clean up*/
		delete sing;
	}

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorPrognostic_CG {{{1*/
void  Tria::CreatePVectorPrognostic_CG(Vec pg){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrix */
	double pe_g[numgrids]={0.0};
	double L[numgrids];
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  accumulation_g;
	double  melting_g;
	double  thickness_g;

	/*inputs: */
	double  dt;
	Input* accumulation_input=NULL;
	Input* melting_input=NULL;
	Input* thickness_input=NULL;

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Retrieve all inputs we will be needing: */
	accumulation_input=inputs->GetInput(AccumulationRateEnum);
	melting_input=inputs->GetInput(MeltingRateEnum);
	thickness_input=inputs->GetInput(ThicknessEnum);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/* Get accumulation, melting and thickness at gauss point */
		accumulation_input->GetParameterValue(&accumulation_g, &gauss_l1l2l3[0]);
		melting_input->GetParameterValue(&melting_g, &gauss_l1l2l3[0]);
		thickness_input->GetParameterValue(&thickness_g, &gauss_l1l2l3[0]);

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i]+=Jdettria*gauss_weight*(thickness_g+dt*(accumulation_g-melting_g))*L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorPrognostic_DG {{{1*/
void  Tria::CreatePVectorPrognostic_DG(Vec pg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrix */
	double pe_g[numgrids]={0.0};
	double L[numgrids];
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  accumulation_g;
	double  melting_g;
	double  thickness_g;
	double  dt;
	Input*  accumulation_input=NULL;
	Input*  melting_input=NULL;
	Input*  thickness_input=NULL;

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Retrieve all inputs we will be needing: */
	accumulation_input=inputs->GetInput(AccumulationRateEnum);
	melting_input=inputs->GetInput(MeltingRateEnum);
	thickness_input=inputs->GetInput(ThicknessEnum);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/* Get accumulation, melting and thickness at gauss point */
		accumulation_input->GetParameterValue(&accumulation_g, &gauss_l1l2l3[0]);
		melting_input->GetParameterValue(&melting_g, &gauss_l1l2l3[0]);
		thickness_input->GetParameterValue(&thickness_g, &gauss_l1l2l3[0]);

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i]+=Jdettria*gauss_weight*(thickness_g+dt*(accumulation_g-melting_g))*L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorSlope {{{1*/
void Tria::CreatePVectorSlope( Vec pg){

	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*element vector at the gaussian points: */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];
	double  slope[2];
	int     analysis_type;

	/*inputs :*/
	Input* slope_input=NULL;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights: */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2); /*We need higher order because our load is order 2*/

	/*Retrieve all inputs we will be needing: */
	if ( (analysis_type==SurfaceSlopeXAnalysisEnum) || (analysis_type==SurfaceSlopeYAnalysisEnum)){
		slope_input=inputs->GetInput(SurfaceEnum);
	}
	if ( (analysis_type==BedSlopeXAnalysisEnum) || (analysis_type==BedSlopeYAnalysisEnum)){
		slope_input=inputs->GetInput(BedEnum);
	}
		
	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		slope_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],&gauss_l1l2l3[0]);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		 /*Get nodal functions: */
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Build pe_g_gaussian vector: */
		if ( (analysis_type==SurfaceSlopeXAnalysisEnum) || (analysis_type==BedSlopeXAnalysisEnum)){
			for(i=0;i<numdof;i++) pe_g_gaussian[i]=Jdet*gauss_weight*slope[0]*l1l2l3[i];
		}
		if ( (analysis_type==SurfaceSlopeYAnalysisEnum) || (analysis_type==BedSlopeYAnalysisEnum)){
			for(i=0;i<numdof;i++) pe_g_gaussian[i]=Jdet*gauss_weight*slope[1]*l1l2l3[i];
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

	} //for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorThermalShelf {{{1*/
void Tria::CreatePVectorThermalShelf( Vec pg){

	int i,found;
	
	const int  numgrids=3;
	const int  NDOF1=1;
	const int  numdof=numgrids*NDOF1;
	int        doflist[numdof];
	int        numberofdofspernode;
	double       xyz_list[numgrids][3];

	double mixed_layer_capacity;
	double thermal_exchange_velocity;
	double rho_water;
	double rho_ice;
	double heatcapacity;
	double beta;
	double meltingpoint;

	/*inputs: */
	double dt;
	double pressure;

	/* gaussian points: */
	int     num_area_gauss,ig;
	double* gauss_weights  =  NULL;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];
	int     dofs1[1]={0};

	/*matrices: */
	double  Jdet;
	double  P_terms[numdof]={0.0};
	double  l1l2l3[numgrids];

	double  t_pmp;
	double  scalar_ocean;

	/*inputs: */
	Input* pressure_input=NULL;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	//recover material parameters
	mixed_layer_capacity=matpar->GetMixedLayerCapacity();
	thermal_exchange_velocity=matpar->GetThermalExchangeVelocity();
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();
	beta=matpar->GetBeta();
	meltingpoint=matpar->GetMeltingPoint();
	
	/*retrieve some solution parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/* Ice/ocean heat exchange flux on ice shelf base */

	GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Retrieve all inputs we will be needing: */
	pressure_input=inputs->GetInput(PressureEnum);

	/* Start looping on the number of gauss 2d (nodes on the bedrock) */
	for (ig=0; ig<num_area_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);

		//Get the Jacobian determinant
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0], gauss_coord);

		/*Get nodal functions values: */
		GetNodalFunctions(&l1l2l3[0], gauss_coord);

		/*Get geothermal flux and basal friction */
		pressure_input->GetParameterValue(&pressure, &gauss_coord[0]);
		t_pmp=meltingpoint-beta*pressure;

		/*Calculate scalar parameter*/
		scalar_ocean=gauss_weight*Jdet*rho_water*mixed_layer_capacity*thermal_exchange_velocity*(t_pmp)/(heatcapacity*rho_ice);
		if(dt){
			scalar_ocean=dt*scalar_ocean;
		}

		for(i=0;i<3;i++){
			P_terms[i]+=scalar_ocean*l1l2l3[i];
		}
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorThermalSheet {{{1*/
void Tria::CreatePVectorThermalSheet( Vec pg){

	int i,found;
	
	const int  numgrids=3;
	const int  NDOF1=1;
	const int  numdof=numgrids*NDOF1;
	int        doflist[numdof];
	int        numberofdofspernode;
	double       xyz_list[numgrids][3];

	double rho_ice;
	double heatcapacity;

	/*inputs: */
	double dt;
	double pressure_list[3];
	double pressure;
	int    drag_type;
	double basalfriction;
	Friction* friction=NULL;
	double alpha2,vx,vy;
	double geothermalflux_value;
	double    alpha2_list[numgrids];                                       //TO BE DELETED
	double    gauss[numgrids][numgrids] = {{1,0,0},{0,1,0},{0,0,1}}; //TO BE DELETED
	double vx_list[numgrids]; //TO BE DELETED
	double vy_list[numgrids]; //TO BE DELETED
	double basalfriction_list[numgrids]; //TO BE DELETED

	/* gaussian points: */
	int     num_area_gauss,ig;
	double* gauss_weights  =  NULL;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];

	/*matrices: */
	double  Jdet;
	double  P_terms[numdof]={0.0};
	double  l1l2l3[numgrids];
	double  scalar;

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*inputs: */
	Input* vx_input=NULL; //TO BE DELETED
	Input* vy_input=NULL; //TO BE DELETED
	Input* vz_input=NULL; //TO BE DELETED
	Input* geothermalflux_input=NULL;
	
	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	//recover material parameters
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/*Build frictoin element, needed later: */
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	if (drag_type!=2)ISSMERROR(" non-viscous friction not supported yet!");
	friction=new Friction("3d",inputs,matpar,analysis_type);

	/*Retrieve all inputs we will be needing: */
	vx_input=inputs->GetInput(VxEnum);
	vy_input=inputs->GetInput(VyEnum);
	vz_input=inputs->GetInput(VzEnum);
	geothermalflux_input=inputs->GetInput(GeothermalFluxEnum);

	/*COMPUT alpha2_list and basalfriction_list (TO BE DELETED)*/
	for(i=0;i<numgrids;i++){
		friction->GetAlpha2(&alpha2_list[i],&gauss[i][0],VxEnum,VyEnum,VzEnum); //TO BE DELETED
	}
	vx_input->GetParameterValues(&vx_list[0],&gauss[0][0],3); //TO BE DELETED
	vy_input->GetParameterValues(&vy_list[0],&gauss[0][0],3); //TO BE DELETED
	for(i=0;i<numgrids;i++){
		basalfriction_list[i]=alpha2_list[i]*(pow(vx_list[i],(double)2.0)+pow(vy_list[i],(double)2.0)); //TO BE DELETED
	}
	
	/* Ice/ocean heat exchange flux on ice shelf base */
	GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start looping on the number of gauss 2d (nodes on the bedrock) */
	for (ig=0; ig<num_area_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);

		//Get the Jacobian determinant
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0], gauss_coord);

		/*Get nodal functions values: */
		GetNodalFunctions(&l1l2l3[0], gauss_coord);

		/*Get geothermal flux and basal friction */
		geothermalflux_input->GetParameterValue(&geothermalflux_value, &gauss_coord[0]);
	
		/*Friction: */
		//friction->GetAlpha2(&alpha2,&gauss_coord[0],VxEnum,VyEnum,VzEnum);
		GetParameterValue(&basalfriction,&basalfriction_list[0],gauss_coord); // TO BE DELETED
		
		/*Calculate scalar parameter*/
		scalar=gauss_weight*Jdet*(basalfriction+geothermalflux_value)/(heatcapacity*rho_ice);
		if(dt){
			scalar=dt*scalar;
		}

		for(i=0;i<3;i++){
			P_terms[i]+=scalar*l1l2l3[i];
		}

	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
	delete friction;

}
/*}}}*/
/*FUNCTION Tria::GetArea {{{1*/
double Tria::GetArea(void){

	double area=0;
	const int    numgrids=3;
	double xyz_list[numgrids][3];
	double x1,y1,x2,y2,x3,y3;

	/*Get xyz list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	x1=xyz_list[0][0]; y1=xyz_list[0][1];
	x2=xyz_list[1][0]; y2=xyz_list[1][1];
	x3=xyz_list[2][0]; y3=xyz_list[2][1];
 
	return x2*y3 - y2*x3 + x1*y2 - y1*x2 + x3*y1 - y3*x1;
}
/*}}}*/
/*FUNCTION Tria::GetAreaCoordinate {{{1*/
double Tria::GetAreaCoordinate(double x, double y, int which_one){

	/*Intermediaries*/
	double    area = 0;
	const int numgrids = 3;
	double    xyz_list[numgrids][3];
	double    x1,y1,x2,y2,x3,y3;

	/*Get area: */
	area=this->GetArea();

	/*Get xyz list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	x1=xyz_list[0][0]; y1=xyz_list[0][1];
	x2=xyz_list[1][0]; y2=xyz_list[1][1];
	x3=xyz_list[2][0]; y3=xyz_list[2][1];

	switch(which_one){
		case 1:
			/*Get first area coordinate = det(x-x3  x2-x3 ; y-y3   y2-y3)/area*/
			return ((x-x3)*(y2-y3)-(x2-x3)*(y-y3))/area;
		case 2:
			/*Get second area coordinate = det(x1-x3  x-x3 ; y1-y3   y-y3)/area*/
			return ((x1-x3)*(y-y3)-(x-x3)*(y1-y3))/area;
		case 3:
			/*Get third  area coordinate 1-area1-area2: */
			return 1-((x-x3)*(y2-y3)-(x2-x3)*(y-y3))/area -((x1-x3)*(y-y3)-(x-x3)*(y1-y3))/area;
		default:
			ISSMERROR("%s%i%s\n"," error message: area coordinate ",which_one," done not exist!");
	}
}
/*}}}*/
/*FUNCTION Tria::GetElementType {{{1*/
int Tria::GetElementType(){

	/*return TriaRef field*/
	return this->element_type;

}
/*}}}*/
/*FUNCTION Tria::GetDofList {{{1*/
void  Tria::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;

	/*Some checks for debugging*/
	ISSMASSERT(doflist);
	ISSMASSERT(pnumberofdofspernode);
	ISSMASSERT(nodes);

	/*Build doflist from nodes*/
	for(i=0;i<3;i++){
		nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
		for(j=0;j<numberofdofspernode;j++){
			doflist[i*numberofdofspernode+j]=doflist_per_node[j];
		}
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION Tria::GetDofList1 {{{1*/
void  Tria::GetDofList1(int* doflist){

	int i;
	for(i=0;i<3;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}
/*}}}*/
/*FUNCTION Tria::GetParameterDerivativeValue {{{1*/
void Tria::GetParameterDerivativeValue(double* p, double* plist,double* xyz_list, double* gauss_l1l2l3){
	 
	const int NDOF2=2;
	const int numgrids=3;
	/*From node values of parameter p (plist[0],plist[1],plist[2]), return parameter derivative value at gaussian 
	 * point specified by gauss_l1l2l3:
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *
	 * p is a vector of size 2x1 already allocated.
	 */
	
	double dh1dh3[NDOF2][numgrids]; //nodal derivative functions in actual coordinate system.

	/*Get dh1dh2dh3 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh3[0][0],xyz_list, gauss_l1l2l3);

	*(p+0)=plist[0]*dh1dh3[0][0]+plist[1]*dh1dh3[0][1]+plist[2]*dh1dh3[0][2];
	*(p+1)=plist[0]*dh1dh3[1][0]+plist[1]*dh1dh3[1][1]+plist[2]*dh1dh3[1][2];

}
/*}}}*/
/*FUNCTION Tria::GetParameterValue(double* pp, double* plist, double* gauss_l1l2l3) {{{1*/
void Tria::GetParameterValue(double* pp, double* plist, double* gauss_l1l2l3){
	
	/*From node values of parameter p (plist[0],plist[1],plist[2]), return parameter value at gaussian 
	 * point specifie by gauss_l1l2l3: */
	
	/*nodal functions: */
	double l1l2l3[3];

	/*output: */
	double p;

	GetNodalFunctions(l1l2l3, gauss_l1l2l3);

	p=l1l2l3[0]*plist[0]+l1l2l3[1]*plist[1]+l1l2l3[2]*plist[2];

	/*Assign output pointers:*/
	*pp=p;
}
/*}}}*/
/*FUNCTION Tria::GetParameterValue(double* pvalue,Node* node,int enumtype) {{{1*/
void Tria::GetParameterValue(double* pvalue,Node* node,int enumtype){

	/*Output*/
	double value;

	/*Intermediaries*/
	const int numnodes=3;
	int       i;
	double    gauss_tria[numnodes]={0.0};

	/*go through 3 nodes (all nodes for tria) and identify node: */
	ISSMASSERT(nodes);
	for(i=0;i<numnodes;i++){
		if (node==nodes[i]){

			/*OK we have found the node, update the gauss point and get value*/
			gauss_tria[i]=1.0;

			/*Recover input*/
			this->inputs->GetParameterValue(&value, &gauss_tria[0],enumtype);

			/*Assign output pointers:*/
			*pvalue=value;
			return;

		}
	}

	/*If we reach tis point, the node provided has not been found;*/
	ISSMERROR("The node provided are either the same or did not match current Tria nodes");
}
/*}}}*/
/*FUNCTION Tria::GetParameterValue(double* pvalue,Node* node1,Node* node2,double gauss_seg,int enumtype) {{{1*/
void Tria::GetParameterValue(double* pvalue,Node* node1,Node* node2,double gauss_seg,int enumtype){

	/*Output*/
	double value;

	/*Intermediaries*/
	const int numnodes=3;
	int       grid1=-1,grid2=-1;
	int       grid3;
	int       i;
	double    gauss_tria[numnodes];

	/*go through 3 nodes (all nodes for tria) and identify 1st and 2nd nodes: */
	ISSMASSERT(nodes);
	for(i=0;i<numnodes;i++){
		if (node1==nodes[i]) grid1=i;
		if (node2==nodes[i]) grid2=i;
	}

	/*Reverse grid1 and 2 if necessary*/
	if (grid1>grid2){

		/*Reverse grid1 and grid2*/
		grid3=grid1; grid1=grid2; grid2=grid3;

		/*Change segment gauss point (between -1 and +1)*/
		gauss_seg=-gauss_seg;
	}

	/*Build Triangle Gauss point*/
	if (grid1==0 && grid2==1){
		/*gauss_seg is between 0 and 1*/
		gauss_tria[0]=0.5*(1.-gauss_seg);
		gauss_tria[1]=1.-0.5*(1.-gauss_seg);
		gauss_tria[2]=0.;
	}
	else if (grid1==0 && grid2==2){
		/*gauss_seg is between 0 and 2*/
		gauss_tria[0]=0.5*(1.-gauss_seg);
		gauss_tria[1]=0.;
		gauss_tria[2]=1.-0.5*(1.-gauss_seg);
	}
	else if (grid1==1 && grid2==2){
		/*gauss_seg is between 1 and 2*/
		gauss_tria[0]=0.;
		gauss_tria[1]=0.5*(1.-gauss_seg);
		gauss_tria[2]=1.-0.5*(1.-gauss_seg);
	}
	else{
		ISSMERROR("The 2 nodes provided are either the same or did not match current Tria nodes");
	}

	/*OK, now we can call input method*/
	this->inputs->GetParameterValue(&value, &gauss_tria[0],enumtype);

	/*Assign output pointers:*/
	*pvalue=value;
}
/*}}}*/
/*FUNCTION Tria::GetSolutionFromInputsDiagnosticHoriz{{{1*/
void  Tria::GetSolutionFromInputsDiagnosticHoriz(Vec solution){

	int i;

	const int    numvertices=3;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;
	double       gauss[numvertices][numvertices]={{1,0,0},{0,1,0},{0,0,1}};

	int          doflist[numdof];
	double       values[numdof];
	double       vx;
	double       vy;

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	for(i=0;i<numvertices;i++){

		/*Recover vx and vy*/
		inputs->GetParameterValue(&vx,&gauss[i][0],VxEnum);
		inputs->GetParameterValue(&vy,&gauss[i][0],VyEnum);
		values[i*numdofpervertex+0]=vx;
		values[i*numdofpervertex+1]=vy;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Tria::GetSolutionFromInputsAdjointHoriz{{{1*/
void  Tria::GetSolutionFromInputsAdjointHoriz(Vec solution){

	int i;

	const int    numvertices=3;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;
	double       gauss[numvertices][numvertices]={{1,0,0},{0,1,0},{0,0,1}};

	int          doflist[numdof];
	double       values[numdof];
	double       lambdax;
	double       lambday;

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Ok, we have lambdax and lambday in values, fill in lambdax and lambday arrays: */
	/*P1 element only for now*/
	for(i=0;i<numvertices;i++){

		/*Recover lambdax and lambday*/
		inputs->GetParameterValue(&lambdax,&gauss[i][0],VxEnum);
		inputs->GetParameterValue(&lambday,&gauss[i][0],VyEnum);
		values[i*numdofpervertex+0]=lambdax;
		values[i*numdofpervertex+1]=lambday;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Tria::GetSolutionFromInputsDiagnosticHutter{{{1*/
void  Tria::GetSolutionFromInputsDiagnosticHutter(Vec solution){

	int i;

	const int    numvertices=3;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;
	double       gauss[numvertices][numvertices]={{1,0,0},{0,1,0},{0,0,1}};

	int          doflist[numdof];
	double       values[numdof];
	double       vx;
	double       vy;

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	for(i=0;i<numvertices;i++){

		/*Recover vx and vy*/
		inputs->GetParameterValue(&vx,&gauss[i][0],VxEnum);
		inputs->GetParameterValue(&vy,&gauss[i][0],VyEnum);
		values[i*numdofpervertex+0]=vx;
		values[i*numdofpervertex+1]=vy;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Tria::GetStrainRate2d{{{1*/
void Tria::GetStrainRate2d(double* epsilon,double* xyz_list, double* gauss, Input* vx_input, Input* vy_input){
	/*Compute the 2d Strain Rate (3 components):
	 *
	 * epsilon=[exx eyy exy]
	 */

	int i;

	double epsilonvx[3];
	double epsilonvy[3];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input){
		ISSMERROR("Input missing. Here are the input pointers we have for vx: %p, vy: %p\n",vx_input,vy_input);
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vx_input->GetVxStrainRate2d(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate2d(epsilonvy,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<3;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i];

}
/*}}}*/
/*FUNCTION Tria::GradjDragStokes {{{1*/
void  Tria::GradjDragStokes(Vec gradient){

	int i;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF2=2;
	double       xyz_list[numgrids][3];
	int          doflist1[numgrids];
	double       dh1dh3[NDOF2][numgrids];

	/* grid data: */
	double drag;
	double alpha_complement;
	Friction* friction=NULL;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* parameters: */
	double  vx,vy,vz;
	double  lambda,mu,xi;
	double  bed,thickness,Neff;
	double  surface_normal[3];
	double  bed_normal[3];
	double  dk[NDOF2]; 

	/*element vector at the gaussian points: */
	double  grade_g[numgrids]={0.0};
	double  grade_g_gaussian[numgrids];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/

	/*inputs: */
	bool shelf;
	int  drag_type;

	/*parameters: */
	double  cm_noisedmp;
	double  cm_mindmp_slope;
	double  cm_mindmp_value;
	double  cm_maxdmp_value;
	double  cm_maxdmp_slope;

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*retrieve inputs :*/
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);
	inputs->GetParameterValue(&drag_type,DragTypeEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);
	this->parameters->FindParam(&cm_mindmp_value,CmMinDmpValueEnum);
	this->parameters->FindParam(&cm_mindmp_slope,CmMinDmpSlopeEnum);
	this->parameters->FindParam(&cm_maxdmp_value,CmMaxDmpValueEnum);
	this->parameters->FindParam(&cm_maxdmp_slope,CmMaxDmpSlopeEnum);

	/*Get out if shelf*/
	if(shelf)return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList1(&doflist1[0]);

	/*Build frictoin element, needed later: */
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	friction=new Friction("2d",inputs,matpar,analysis_type);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 4);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Recover alpha_complement and drag: */
		if (drag_type==2) friction->GetAlphaComplement(&alpha_complement, gauss_l1l2l3,VxEnum,VyEnum);
		else alpha_complement=0;
		inputs->GetParameterValue(&drag, &gauss_l1l2l3[0],DragCoefficientEnum);

		/*recover lambda mu and xi: */
		inputs->GetParameterValue(&lambda, &gauss_l1l2l3[0],AdjointxEnum);
		inputs->GetParameterValue(&mu, &gauss_l1l2l3[0],AdjointyEnum);
		inputs->GetParameterValue(&xi, &gauss_l1l2l3[0],AdjointzEnum);

		/*recover vx vy and vz: */
		inputs->GetParameterValue(&vx, &gauss_l1l2l3[0],VxEnum);
		inputs->GetParameterValue(&vy, &gauss_l1l2l3[0],VyEnum);
		inputs->GetParameterValue(&vz, &gauss_l1l2l3[0],VzEnum);

		/*Get normal vector to the bed */
		SurfaceNormal(&surface_normal[0],xyz_list);

		bed_normal[0]=-surface_normal[0]; //Program is for surface, so the normal to the bed is the opposite of the result
		bed_normal[1]=-surface_normal[1];
		bed_normal[2]=-surface_normal[2];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Get nodal functions derivatives*/
		GetNodalFunctionsDerivatives(&dh1dh3[0][0],&xyz_list[0][0],gauss_l1l2l3);

		/*Get k derivative: dk/dx */
		inputs->GetParameterDerivativeValue(&dk[0],&xyz_list[0][0],&gauss_l1l2l3[0],DragCoefficientEnum);

		/*Build gradje_g_gaussian vector (actually -dJ/ddrag): */
		for (i=0;i<numgrids;i++){
			//standard gradient dJ/dki
			grade_g_gaussian[i]=(
						-lambda*(2*drag*alpha_complement*(vx - vz*bed_normal[0]*bed_normal[2]))
						-mu    *(2*drag*alpha_complement*(vy - vz*bed_normal[1]*bed_normal[2]))
						-xi    *(2*drag*alpha_complement*(-vx*bed_normal[0]*bed_normal[2]-vy*bed_normal[1]*bed_normal[2]))
						)*Jdet*gauss_weight*l1l2l3[i]; 

			//Add regularization term
			grade_g_gaussian[i]+= - cm_noisedmp*Jdet*gauss_weight*(dh1dh3[0][i]*dk[0]+dh1dh3[1][i]*dk[1]);

			//min dampening
			if(drag<cm_mindmp_value){ 
				grade_g_gaussian[i]+= cm_mindmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

			//max dampening
			if(drag>cm_maxdmp_value){ 
				grade_g_gaussian[i]+= - cm_maxdmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}
		}

		/*Add gradje_g_gaussian vector to gradje_g: */
		for( i=0; i<numgrids; i++)grade_g[i]+=grade_g_gaussian[i];
	}

	/*Add grade_g to global vector gradient: */
	VecSetValues(gradient,numgrids,doflist1,(const double*)grade_g,ADD_VALUES);

	/*Add grade_g to the inputs of this element: */
	this->inputs->AddInput(new TriaVertexInput(GradientEnum,&grade_g[0]));

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
	delete friction;

}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromSolutionAdjointHoriz {{{1*/
void  Tria::InputUpdateFromSolutionAdjointHoriz(double* solution){

	int i;

	const int    numvertices=3;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];
	double       lambdax[numvertices];
	double       lambday[numvertices];

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<numvertices;i++){
		lambdax[i]=values[i*numdofpervertex+0];
		lambday[i]=values[i*numdofpervertex+1];
	}

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new TriaVertexInput(AdjointxEnum,lambdax));
	this->inputs->AddInput(new TriaVertexInput(AdjointyEnum,lambday));

}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromSolutionDiagnosticHoriz {{{1*/
void  Tria::InputUpdateFromSolutionDiagnosticHoriz(double* solution){
	
	int i;

	const int    numvertices=3;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;
	
	int          doflist[numdof];
	double       values[numdof];
	double       vx[numvertices];
	double       vy[numvertices];
	double       vz[numvertices];
	double       vel[numvertices];
	double       pressure[numvertices];
	double       thickness[numvertices];
	double       rho_ice,g;
	double       gauss[numvertices][numvertices]={{1,0,0},{0,1,0},{0,0,1}};

	int          dummy;
	Input*       VzInput=NULL;
	double*      VzPtr=NULL;
	
	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<numvertices;i++){
		vx[i]=values[i*numdofpervertex+0];
		vy[i]=values[i*numdofpervertex+1];
	}

	/*Get Vz*/
	VzInput=inputs->GetInput(VzEnum);
	if (VzInput){
		if (VzInput->Enum()!=TriaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumAsString(VzInput->Enum()));
		}
		VzInput->GetValuesPtr(&VzPtr,&dummy);
		for(i=0;i<numvertices;i++) vz[i]=VzPtr[i];
	}
	else{
		for(i=0;i<numvertices;i++) vz[i]=0.0;
	}

	/*Now Compute vel*/
	for(i=0;i<numvertices;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 2D, 
	 *so the pressure is just the pressure at the bedrock: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	inputs->GetParameterValues(&thickness[0],&gauss[0][0],3,ThicknessEnum);
	
	for(i=0;i<numvertices;i++){
		pressure[i]=rho_ice*g*thickness[i];
	}

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new TriaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new TriaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new TriaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new TriaVertexInput(PressureEnum,pressure));

}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromSolutionDiagnosticHutter {{{1*/
void  Tria::InputUpdateFromSolutionDiagnosticHutter(double* solution){
	
	int i;

	const int    numvertices=3;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;
	
	int          doflist[numdof];
	double       values[numdof];
	double       vx[numvertices];
	double       vy[numvertices];
	double       vz[numvertices];
	double       vel[numvertices];
	double       pressure[numvertices];
	double       thickness[numvertices];
	double       rho_ice,g;
	double       gauss[numvertices][numvertices]={{1,0,0},{0,1,0},{0,0,1}};

	int          dummy;
	Input*       VzInput=NULL;
	double*      VzPtr=NULL;
	
	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<numvertices;i++){
		vx[i]=values[i*numdofpervertex+0];
		vy[i]=values[i*numdofpervertex+1];
	}

	/*Get Vz*/
	VzInput=inputs->GetInput(VzEnum);
	if (VzInput){
		if (VzInput->Enum()!=TriaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumAsString(VzInput->Enum()));
		}
		VzInput->GetValuesPtr(&VzPtr,&dummy);
		for(i=0;i<numvertices;i++) vz[i]=VzPtr[i];
	}
	else{
		for(i=0;i<numvertices;i++) vz[i]=0.0;
	}

	/*Now Compute vel*/
	for(i=0;i<numvertices;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 2D, 
	 *so the pressure is just the pressure at the bedrock: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	inputs->GetParameterValues(&thickness[0],&gauss[0][0],3,ThicknessEnum);
	
	for(i=0;i<numvertices;i++){
		pressure[i]=rho_ice*g*thickness[i];
	}

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new TriaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new TriaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new TriaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new TriaVertexInput(PressureEnum,pressure));

}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromSolutionOneDof{{{1*/
void  Tria::InputUpdateFromSolutionOneDof(double* solution,int enum_type){

	const int numvertices     = 3;
	const int numdofpervertex = 1;
	const int numdof          = numdofpervertex *numvertices;
	int       doflist[numdof];
	double    values[numdof];
	int       dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(int i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Add input to the element: */
	this->inputs->AddInput(new TriaVertexInput(enum_type,values));
}
/*}}}*/
/*FUNCTION Tria::IsInput{{{1*/
bool Tria::IsInput(int name){
	if (
				name==ThicknessEnum ||
				name==SurfaceEnum ||
				name==BedEnum ||
				name==SurfaceSlopeXEnum ||
				name==SurfaceSlopeYEnum ||
				name==MeltingRateEnum ||
				name==AccumulationRateEnum ||
				name==VxEnum ||
				name==VyEnum ||
				name==RheologyBEnum ||
				name==RheologyNEnum ||
				name==FitEnum ||
				name==DragCoefficientEnum ||
				name==GradientEnum ||
				name==OldGradientEnum
		){
		return true;
	}
	else return false;
}
/*}}}*/
/*FUNCTION Tria::SetClone {{{1*/
void  Tria::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
/*FUNCTION Tria::SpawnBeam {{{1*/
void* Tria::SpawnBeam(int g0, int g1){

	int i;

	/*out of grids g0,g1 and g2 from Tria, build a beam element: */
	Beam* beam=NULL;
	int indices[2];
	int zero=0;
	Parameters *beam_parameters = NULL;
	Inputs     *beam_inputs     = NULL;

	indices[0]=g0;
	indices[1]=g1;

	beam_parameters=this->parameters;
	beam_inputs=(Inputs*)this->inputs->SpawnBeamInputs(indices);

	beam=new Beam();
	beam->id=this->id;
	beam->inputs=beam_inputs;
	beam->parameters=beam_parameters;

	/*now deal with nodes, matice and matpar: */
	beam->nodes=(Node**)xmalloc(2*sizeof(Node*));
	for(i=0;i<2;i++)beam->nodes[i]=this->nodes[indices[i]];
	beam->matice=this->matice;
	beam->matpar=this->matpar;


	return beam;
}
/*}}}*/
/*FUNCTION Tria::SpawnSing {{{1*/
void* Tria::SpawnSing(int index){

	Sing* sing=NULL;
	int zero=0;
	Parameters *sing_parameters = NULL;
	Inputs     *sing_inputs     = NULL;

	sing_parameters=this->parameters;
	sing_inputs=(Inputs*)this->inputs->SpawnSingInputs(index);

	sing=new Sing();
	sing->id=this->id;
	sing->inputs=sing_inputs;
	sing->parameters=sing_parameters;

	/*now deal with node,matice and matpar: */
	sing->node=this->nodes[index];
	sing->matice=this->matice;
	sing->matpar=this->matpar;
	
	return sing;
}
/*}}}*/
/*FUNCTION Tria::SurfaceNormal{{{1*/

void Tria::SurfaceNormal(double* surface_normal, double xyz_list[3][3]){

	int i;
	double v13[3];
	double v23[3];
	double normal[3];
	double normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	normal_norm=sqrt( pow(normal[0],(double)2)+pow(normal[1],(double)2)+pow(normal[2],(double)2) );

	*(surface_normal)=normal[0]/normal_norm;
	*(surface_normal+1)=normal[1]/normal_norm;
	*(surface_normal+2)=normal[2]/normal_norm;

}
/*}}}*/
