function [femmodel ug varargout]=solver_diagnostic_nonlinear(femmodel,conserve_loads)
%SOLVER_DIAGNOSTIC_NONLINEAR - core solver of diagnostic run
%
%   Usage:
%      [femmodel ug varargout]=solver_diagnostic_nonlinear(femmodel,conserve_loads)
	

	%Recover parameters
	femmodel.parameters.Kflag=1; femmodel.parameters.Pflag=1;
	dim=femmodel.parameters.Dim;

	%keep a copy of loads for now
	loads=femmodel.loads;

	%initialize solution vector
	converged=0; count=1;

	%Start non-linear iteration using input velocity: 
	ug=GetSolutionFromInputs(femmodel.elements, femmodel.nodes, femmodel.vertices, loads, femmodel.materials, femmodel.parameters);
	uf=Reducevectorgtof( ug, femmodel.nodesets);

	%Update the solution to make sure that vx and vxold are similar
	[femmodel.elements,femmodel.materials]=InputUpdateFromSolution(femmodel.elements,femmodel.nodes,femmodel.vertices,loads,femmodel.materials,femmodel.parameters,ug);

	while(~converged),
		
		%save pointer to old velocity
		old_ug=ug;
		old_uf=uf;
		
		displaystring(femmodel.parameters.Verbose,'%s','   Generating matrices:');
		[K_gg_nopenalty , p_g_nopenalty]=SystemMatrices(femmodel.elements,femmodel.nodes,femmodel.vertices,loads,femmodel.materials,femmodel.parameters);
	
		displaystring(femmodel.parameters.Verbose,'%s','   Generating penalty matrices:');
		[K_gg , p_g, kmax]=PenaltySystemMatrices(K_gg_nopenalty,p_g_nopenalty,femmodel.elements,femmodel.nodes,femmodel.vertices,loads,femmodel.materials,femmodel.parameters);

		displaystring(femmodel.parameters.Verbose,'%s','   reducing matrix from g to f set');
		[K_ff, K_fs] = Reducematrixfromgtof( K_gg, femmodel.Gmn, femmodel.nodesets); 

		displaystring(femmodel.parameters.Verbose,'%s','   reduce load from g size to f size');
		p_f = Reduceloadfromgtof( p_g, femmodel.Gmn, K_fs, femmodel.ys, femmodel.nodesets);
		
		displaystring(femmodel.parameters.Verbose,'%s','   solving');
		uf=Solver(K_ff,p_f,old_uf,femmodel.parameters);

		displaystring(femmodel.parameters.Verbose,'%s','   merge back to g set');
		ug= Mergesolutionfromftog( uf, femmodel.Gmn, femmodel.ys, femmodel.nodesets ); 

		%Update inputs using new solution
		[femmodel.elements,femmodel.materials]=InputUpdateFromSolution(femmodel.elements,femmodel.nodes,femmodel.vertices,loads,femmodel.materials,femmodel.parameters,ug);
		
		displaystring(femmodel.parameters.Verbose,'%s','   penalty constraints');
		[loads,constraints_converged,num_unstable_constraints] =PenaltyConstraints( femmodel.elements,femmodel.nodes,femmodel.vertices,loads, femmodel.materials,femmodel.parameters);
		
		displaystring(femmodel.parameters.Verbose,'%s%i','      number of unstable constraints: ',num_unstable_constraints);
		
		%Figure out if convergence have been reached
		converged=convergence(K_ff,p_f,uf,old_uf,femmodel.parameters);
			
		%add convergence status into  status
		[femmodel.elements loads]=InputUpdateFromConstant(femmodel.elements,femmodel.nodes,femmodel.vertices,loads,femmodel.materials,femmodel.parameters,double(converged),ConvergedEnum);

		%rift convergence
		if ~constraints_converged,
			if converged,
				if num_unstable_constraints <= min_mechanical_constraints,
					converged=1;
				else 
					converged=0;
				end
			end
		end

		%increase count
		count=count+1;
		if(converged==1)break;
			if(count>max_nonlinear_iterations),
				displaystring(femmodel.parameters.Verbose,'%s%i%s','      maximum number of iterations ',max_nonlinear_iterations,' exceeded');
			end
		end
		
	end

	%more output might be needed, when running in control_core.m
	nout=max(nargout,1)-2;
	if nout==2,
		femmodel.parameters.Kflag=1; femmodel.parameters.Pflag=0;
		[K_gg, p_g]=SystemMatrices(femmodel.elements,femmodel.nodes,femmodel.vertices,loads,femmodel.materials,femmodel.parameters);
		[K_ff, K_fs] = Reducematrixfromgtof( K_gg, femmodel.Gmn, femmodel.nodesets); 
		varargout(1)={K_ff};
		varargout(2)={K_fs};
	end


	%deal with loads:
	if conserve_loads==false,
		femmodel.loads=loads;
	end
end
