% This file can be run to check that the current version of issm is giving 
% coherent results. This test deals with Pine Island Glacierfor a 2d model.
% Just run this file in Matlab, with a properly setup ISSM code. 
% The results of this test will indicate if there is a difference between current computations 
% and archived results.

% Errors  between archived results and the current version will get flagged if they are not within 
% a certain tolerance. The current tolerance is 10^-12. If you have good reasons to believe this 
% tolerance should be lowered (for example, if you are running single precision compilers?), feel 
% free to tweak the tolerance variable.

%packages and misfits to be tested
packages={'macayeal','cielo_parallel','cielo_serial'};
misfits={'absolute','relative','logarithmic'};

%Initialize log message for nightly runs.
testname='PigControlMethodBM2d_21';
tolerance=10^-12;

%go through the solutions requested
for i=1:length(packages),
	package=packages{i};

	for j=1:length(misfits),
		misfit=misfits{j};

		%check package
		if ~(strcmpi(package,'macayeal') | strcmpi(package,'cielo_serial') | strcmpi(package,'cielo_parallel'));
			error(['package: ' package  ' in test: ' testname  ' not supported yet']);
			%check solution
		elseif ~(strcmpi(misfit,'absolute') | strcmpi(misfit,'relative') | strcmpi(misfit,'logarithmic')),
			error(['misfit: ' misfit  ' in test: ' testname  ' not supported yet']);
		end
		%check solution requested
		if strcmpi(package,'macayeal') & ~strcmpi(misfit,'absolute')
			disp(['package: ' package  ' and misfit: ' misfit  ' in test: ' testname  ' not supported yet']);
			continue
		end

		%initialize model
		md=model;
		md=mesh(md,'DomainOutline.exp',9000);
		md=geography(md,'Iceshelves.exp','Islands.exp');
		md=parameterize(md,'Pig.par');
		md=setelementstype(md,'macayeal','all');
		if strcmpi(misfit,'absolute'),
			md.fit(:)=0;
		elseif strcmpi(misfit,'relative'),
			md.fit(:)=1;
		elseif strcmpi(misfit,'logarithmic'),
			md.fit(:)=2;
		end
		md=modelextract(md,md.elementoniceshelf);
		if strcmpi(package,'cielo_parallel'), md.cluster='wilkes'; end

		%launch control method
		if (strcmpi(package,'cielo_parallel') | strcmpi(package,'cielo_serial')),
			md=solve(md,'analysis_type','control','package','cielo');
		else
			md=solve(md,'analysis_type','control','package',package);
		end

		%compute field to be checked
		fields={'cont_vel','cont_J','cont_parameter'};

		%load archive
		eval(['load Archive' package misfit ]);

		for k=1:length(fields),
			field=fields{k};

			%compare to archive
			eval(['Archive=Archive' package misfit '_field' num2str(k) ';']);
			eval(['error_diff=abs(norm((Archive(find(Archive))-md.' field  '(find(Archive)))./Archive(find(Archive)),2));']);

			%disp test result
			if (error_diff>tolerance);
				disp(sprintf(['\n\nERROR   (difference=%-7.2g > %g) --> test: %-25s misfit: %-16s package: %-14s field: ' field  '.\n\n'],error_diff,tolerance,testname,misfit,package));
			else
				disp(sprintf(['\n\nSUCCESS (difference=%-7.2g < %g) --> test: %-25s misfit: %-16s package: %-14s field: ' field  '.\n\n'],error_diff,tolerance,testname,misfit,package));
			end

		end
	end
end
