/*!\file Penta.cpp
 * \brief: implementation of the Penta object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../Container/Container.h"
/*}}}*/

/*Penta constructors and destructor*/
/*FUNCTION Penta::Penta(){{{1*/
Penta::Penta(){
	this->nodes=NULL;
	this->matice=NULL;
	this->matpar=NULL;
	this->neighbors=NULL;
	
	this->inputs=NULL;
	this->results=NULL;
	this->parameters=NULL;
}
/*}}}*/
/*FUNCTION Penta::~Penta(){{{1*/
Penta::~Penta(){
	delete inputs;
	delete results;
	this->parameters=NULL;
}
/*}}}*/
/*FUNCTION Penta::Penta(int id, int index, IoModel* iomodel,int nummodels) {{{1*/
Penta::Penta(int penta_id, int index, IoModel* iomodel,int nummodels):

	PentaHook(nummodels,index+1,iomodel->numberofelements+1) //index+1: matice id, iomodel->numberofelements+1: matpar id
                                                                      { //i is the element index

	int i;
	int penta_elements_ids[2];

	/*Checks in debugging mode*/
	/*{{{2*/
	ISSMASSERT(iomodel->upperelements);
	ISSMASSERT(iomodel->lowerelements);
	/*}}}*/

	/*id: */
	this->id=penta_id;

	/*penta_elements_ids: */
	if isnan(iomodel->upperelements[index]){
		penta_elements_ids[1]=this->id; //upper penta is the same penta
	}
	else{
		penta_elements_ids[1]=(int)(iomodel->upperelements[index]);
	}
	
	if isnan(iomodel->lowerelements[index]){
		penta_elements_ids[0]=this->id; //lower penta is the same penta
	}
	else{
		penta_elements_ids[0]=(int)(iomodel->lowerelements[index]);
	}
	this->InitHookNeighbors(penta_elements_ids);

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;

	/*intialize inputs and results: */
	this->inputs=new Inputs();
	this->results=new Results();
	
	/*initialize pointers:*/
	this->nodes=NULL;
	this->matice=NULL;
	this->matpar=NULL;
	this->neighbors=NULL;
}
/*}}}*/

/*Object virtual functions definitions: */
/*FUNCTION Penta::copy {{{1*/
Object* Penta::copy() {

	int i;

	Penta* penta=NULL;

	penta=new Penta();

	//deal with PentaHook mother class
	penta->numanalyses=this->numanalyses;
	penta->hnodes=new Hook*[penta->numanalyses];
	for(i=0;i<penta->numanalyses;i++)penta->hnodes[i]=(Hook*)this->hnodes[i]->copy();
	penta->hmatice=(Hook*)this->hmatice->copy();
	penta->hmatpar=(Hook*)this->hmatpar->copy();
	penta->hneighbors=(Hook*)this->hneighbors->copy();

	/*deal with Penta  copy fields: */
	penta->id=this->id;
	if(this->inputs){
		penta->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		penta->inputs=new Inputs();
	}
	if(this->results){
		penta->results=(Results*)this->results->Copy();
	}
	else{
		penta->results=new Results();
	}
	/*point parameters: */
	penta->parameters=this->parameters;

	/*recover objects: */
	penta->nodes=(Node**)xmalloc(6*sizeof(Node*)); //we cannot rely on an analysis_counter to tell us which analysis_type we are running, so we just copy the nodes.
	for(i=0;i<6;i++)penta->nodes[i]=this->nodes[i];
	penta->matice=(Matice*)penta->hmatice->delivers();
	penta->matpar=(Matpar*)penta->hmatpar->delivers();
	penta->neighbors=(Penta**)penta->hneighbors->deliverp();

	return penta;
}
/*}}}*/
/*FUNCTION Penta::DeepEcho{{{1*/
void Penta::DeepEcho(void){

	int i;
	
	printf("Penta:\n");
	printf("   id: %i\n",id);
	nodes[0]->DeepEcho();
	nodes[1]->DeepEcho();
	nodes[2]->DeepEcho();
	nodes[3]->DeepEcho();
	nodes[4]->DeepEcho();
	nodes[5]->DeepEcho();
	matice->DeepEcho();
	matpar->DeepEcho();
	printf("   neighbor ids: %i-%i\n",neighbors[0]->Id(),neighbors[1]->Id());
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();
	printf("   results\n");
	results->DeepEcho();
	return;
}
/*}}}*/
/*FUNCTION Penta::Demarshall {{{1*/
void  Penta::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;
	int flaghook;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&numanalyses,marshalled_dataset,sizeof(numanalyses));marshalled_dataset+=sizeof(numanalyses);

	/*allocate dynamic memory: */
	this->hnodes=new Hook*[this->numanalyses];
	/*demarshall hooks: */
	for(i=0;i<numanalyses;i++){
		memcpy(&flaghook,marshalled_dataset,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
		if(flaghook){ // there is a hook so demarshall it
			hnodes[i]=new Hook();
			hnodes[i]->Demarshall(&marshalled_dataset);
		}
		else hnodes[i]=NULL; //There is no hook so it is NULL
	}
	hmatice=new Hook(); hmatice->Demarshall(&marshalled_dataset);
	hmatpar=new Hook(); hmatpar->Demarshall(&marshalled_dataset);
	hneighbors=new Hook(); hneighbors->Demarshall(&marshalled_dataset);

	/*pointers are garbage, until configuration is carried out: */
	nodes=NULL;
	matice=NULL;
	matpar=NULL;
	neighbors=NULL;
	
	/*demarshall inputs and results: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 
	results=(Results*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Penta::Echo{{{1*/

void Penta::Echo(void){
	this->DeepEcho();
}
/*}}}*/
/*FUNCTION Penta::Enum {{{1*/
int Penta::Enum(void){

	return PentaEnum;

}
/*}}}*/
/*FUNCTION Penta::Id {{{1*/
int    Penta::Id(void){
	return id; 
}
/*}}}*/
/*FUNCTION Penta::Marshall {{{1*/
void  Penta::Marshall(char** pmarshalled_dataset){

	int   i;
	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;
	char* marshalled_results=NULL;
	int   marshalled_results_size;
	int   flaghook; //to indicate if hook is NULL or exists

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Penta: */
	enum_type=PentaEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Penta data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&numanalyses,sizeof(numanalyses));marshalled_dataset+=sizeof(numanalyses);

	/*Marshall hooks: */
	for(i=0;i<numanalyses;i++){
		if(hnodes[i]){
			/*Set flag to 1 as there is a hook */
			flaghook=1;
			memcpy(marshalled_dataset,&flaghook,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
			hnodes[i]->Marshall(&marshalled_dataset);
		}
		else{
			/*Set flag to 0 and do not marshall flag as there is no Hook */
			flaghook=0;
			memcpy(marshalled_dataset,&flaghook,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
		}
	}
	hmatice->Marshall(&marshalled_dataset);
	hmatpar->Marshall(&marshalled_dataset);
	hneighbors->Marshall(&marshalled_dataset);

	/*Marshall inputs and results: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	marshalled_results_size=results->MarshallSize();
	marshalled_results=results->Marshall();
	memcpy(marshalled_dataset,marshalled_results,marshalled_results_size*sizeof(char));
	marshalled_dataset+=marshalled_results_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);
	xfree((void**)&marshalled_results);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Penta::MarshallSize {{{1*/
int   Penta::MarshallSize(){

	int i;
	int hnodes_size=0;;

	for(i=0;i<numanalyses;i++){
		hnodes_size+=sizeof(int); //Flag 0 or 1
		if (hnodes[i]) hnodes_size+=hnodes[i]->MarshallSize();
	}

	return sizeof(id)
		+hnodes_size
		+sizeof(numanalyses)
		+hmatice->MarshallSize()
		+hmatpar->MarshallSize()
		+hneighbors->MarshallSize()
		+inputs->MarshallSize()
		+results->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Penta::MyRank {{{1*/
int    Penta::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/

/*Update virtual functions definitions: */
/*FUNCTION Penta::InputUpdateFromConstant(bool value, int name);{{{1*/
void  Penta::InputUpdateFromConstant(bool constant, int name){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new BoolInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromConstant(double value, int name);{{{1*/
void  Penta::InputUpdateFromConstant(double constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new DoubleInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromConstant(int value, int name);{{{1*/
void  Penta::InputUpdateFromConstant(int constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new IntInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolution {{{1*/
void  Penta::InputUpdateFromSolution(double* solution){

	int analysis_type;

	/*retreive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Just branch to the correct InputUpdateFromSolution generator, according to the type of analysis we are carrying out: */
	if (analysis_type==ControlAnalysisEnum){
		InputUpdateFromSolutionDiagnosticHoriz( solution);
	}
	else if (analysis_type==DiagnosticHorizAnalysisEnum){
		InputUpdateFromSolutionDiagnosticHoriz( solution);
	}
	else if (analysis_type==DiagnosticHutterAnalysisEnum){
		InputUpdateFromSolutionDiagnosticHutter( solution);
	}
	else if (analysis_type==DiagnosticVertAnalysisEnum){
		InputUpdateFromSolutionDiagnosticVert( solution);
	}
	else if (analysis_type==DiagnosticStokesAnalysisEnum){
		InputUpdateFromSolutionDiagnosticStokes( solution);
	}
	else if (analysis_type==AdjointHorizAnalysisEnum){
		InputUpdateFromSolutionAdjointHoriz( solution);
	}
	else if (analysis_type==AdjointStokesAnalysisEnum){
		InputUpdateFromSolutionAdjointStokes( solution);
	}
	else if (analysis_type==BedSlopeXAnalysisEnum){
		InputUpdateFromBedSlopeX( solution);
	}
	else if (analysis_type==BedSlopeYAnalysisEnum){
		InputUpdateFromBedSlopeY( solution);
	}
	else if (analysis_type==SurfaceSlopeXAnalysisEnum){
		InputUpdateFromSurfaceSlopeX( solution);
	}
	else if (analysis_type==SurfaceSlopeYAnalysisEnum){
		InputUpdateFromSurfaceSlopeY( solution);
	}
	else if (analysis_type==PrognosticAnalysisEnum){
		InputUpdateFromSolutionPrognostic( solution);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){
		InputUpdateFromSolutionBalancedthickness( solution);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){
		InputUpdateFromSolutionBalancedvelocities( solution);
	}
	else if (analysis_type==ThermalAnalysisEnum){
		InputUpdateFromSolutionThermal( solution);
	}
	else if (analysis_type==MeltingAnalysisEnum){
		InputUpdateFromSolutionMelting( solution);
	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));
	}
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVector(double* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVector(double* vector, int name, int type){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	switch(type){

		case VertexEnum:

			/*New PentaVertexInpu*/
			double values[6];

			/*Get values on the 6 vertices*/
			for (int i=0;i<6;i++){
				values[i]=vector[this->nodes[i]->GetVertexDof()];
			}

			/*update input*/
			this->inputs->AddInput(new PentaVertexInput(name,values));
			return;

		default:

			ISSMERROR("type %i (%s) not implemented yet",type,EnumAsString(type));
	}
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVector(int* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVector(int* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVector(bool* vector, int name, int type);{{{1*/
void  Penta::InputUpdateFromVector(bool* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/

/*Element virtual functions definitions: */
/*FUNCTION Penta::ComputeBasalStress {{{1*/
void  Penta::ComputeBasalStress(Vec sigma_b){

	int i,j;
	const int numgrids=6;
	int    doflist[numgrids];
	double xyz_list[numgrids][3];
	double xyz_list_tria[3][3];

	/*Parameters*/
	double rho_ice,gravity;
	double surface_normal[3];
	double bed_normal[3];
	double bed;
	double basalforce[3];
	double epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double devstresstensor[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double stresstensor[6]={0.0};
	double viscosity;
	int analysis_type;

	int  dofv[3]={0,1,2};
	int  dofp[1]={3};
	double Jdet2d;
	Tria* tria=NULL;

	/*Gauss*/
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_coord[4];

	/*Output*/
	double pressure;
	double sigma_xx,sigma_yy,sigma_zz;
	double sigma_xy,sigma_xz,sigma_yz;
	double surface=0;
	double value=0;
	
	/*flags: */
	bool onbed;

	/*parameters: */
	double stokesreconditioning;


	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Check analysis_types*/
	if (analysis_type!=DiagnosticStokesAnalysisEnum) ISSMERROR("Not supported yet!");

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	
	if(!onbed){
		//put zero
		VecSetValue(sigma_b,id-1,0.0,INSERT_VALUES);
		return;
	}

	/*recovre material parameters: */
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	for(i=0;i<3;i++){
		for(j=0;j<3;j++){
			xyz_list_tria[i][j]=xyz_list[i][j];
		}
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria(&num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights,2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

			/*Pick up the gaussian point: */
			gauss_weight=*(gauss_weights+ig);
			gauss_coord[0]=*(first_gauss_area_coord+ig); 
			gauss_coord[1]=*(second_gauss_area_coord+ig);
			gauss_coord[2]=*(third_gauss_area_coord+ig);
			gauss_coord[3]=-1.0; //take the base

			/*Compute strain rate viscosity and pressure: */
			inputs->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss_coord,VxEnum,VyEnum,VzEnum);
			matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);
			inputs->GetParameterValue(&pressure, &gauss_coord[0],PressureEnum);

			/*Compute Stress*/
			sigma_xx=viscosity*epsilon[0]-pressure*stokesreconditioning; // sigma = nu eps - pressure
			sigma_yy=viscosity*epsilon[1]-pressure*stokesreconditioning;
			sigma_zz=viscosity*epsilon[2]-pressure*stokesreconditioning;
			sigma_xy=viscosity*epsilon[3];
			sigma_xz=viscosity*epsilon[4];
			sigma_yz=viscosity*epsilon[5];

			/*Get normal vector to the bed */
			SurfaceNormal(&surface_normal[0],xyz_list_tria);
			bed_normal[0] = - surface_normal[0]; //Program is for surface, so the normal to the bed is the opposite of the result
			bed_normal[1] = - surface_normal[1];
			bed_normal[2] = - surface_normal[2];

			/*basalforce*/
			basalforce[0] += sigma_xx*bed_normal[0] + sigma_xy*bed_normal[1] + sigma_xz*bed_normal[2];
			basalforce[1] += sigma_xy*bed_normal[0] + sigma_yy*bed_normal[1] + sigma_yz*bed_normal[2];
			basalforce[2] += sigma_xz*bed_normal[0] + sigma_yz*bed_normal[1] + sigma_zz*bed_normal[2];

			/*Get the Jacobian determinant */
			tria->GetJacobianDeterminant3d(&Jdet2d, &xyz_list_tria[0][0],gauss_coord);
			value+=sigma_zz*Jdet2d*gauss_weight;
			surface+=Jdet2d*gauss_weight;
	}
	value=value/surface;

	/*Add value to output*/
	VecSetValue(sigma_b,id-1,(const double)value,INSERT_VALUES);
}
/*}}}*/
/*FUNCTION Penta::ComputePressure {{{1*/
void  Penta::ComputePressure(Vec pg){

	int i;
	const int numgrids=6;
	int    doflist[numgrids];
	double pressure[numgrids];
	double rho_ice,g;
	double surface[numgrids];
	double xyz_list[numgrids][3];
	double gauss[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	/*inputs: */
	bool onwater;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);

	/*pressure is lithostatic: */
	//md.pressure=md.rho_ice*md.g*(md.surface-md.z); a la matlab

	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*recover value of surface at grids: */
	inputs->GetParameterValues(&surface[0],&gauss[0][0],6,SurfaceEnum);

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	for(i=0;i<numgrids;i++){
		pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);
	}

	/*plug local pressure values into global pressure vector: */
	VecSetValues(pg,numgrids,doflist,(const double*)pressure,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Penta::ComputeStrainRate {{{1*/
void  Penta::ComputeStrainRate(Vec eps){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
		/*FUNCTION Penta::Configure {{{1*/
void  Penta::Configure(Elements* elementsin, Loads* loadsin, DataSet* nodesin, Materials* materialsin, Parameters* parametersin){

	int analysis_counter;
	
	/*go into parameters and get the analysis_counter: */
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	if (this->hnodes[analysis_counter]) this->hnodes[analysis_counter]->configure(nodesin);
	this->hmatice->configure(materialsin);
	this->hmatpar->configure(materialsin);
	this->hneighbors->configure(elementsin);

	/*Now, go pick up the objects inside the hooks: */
	if (this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;
	this->matice=(Matice*)this->hmatice->delivers();
	this->matpar=(Matpar*)this->hmatpar->delivers();
	this->neighbors=(Penta**)this->hneighbors->deliverp();

	/*point parameters to real dataset: */
	this->parameters=parametersin;
}
/*}}}*/
/*FUNCTION Penta::SetCurrentConfiguration {{{1*/
void  Penta::SetCurrentConfiguration(Elements* elementsin, Loads* loadsin, DataSet* nodesin, Materials* materialsin, Parameters* parametersin){

	int analysis_counter;
	
	/*go into parameters and get the analysis_counter: */
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Pick up the objects inside the hooks: */
	if (this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;
	this->matice=(Matice*)this->hmatice->delivers();
	this->matpar=(Matpar*)this->hmatpar->delivers();
	this->neighbors=(Penta**)this->hneighbors->deliverp();

	/*point parameters to real dataset: */
	this->parameters=parametersin;
}
/*}}}*/
/*FUNCTION Penta::CostFunction {{{1*/
double Penta::CostFunction(void){

	double J;
	Tria* tria=NULL;

	/*flags: */
	bool onbed;
	bool onwater;
	bool collapse;
	bool onsurface;

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&collapse,CollapseEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*Bail out if this element if:
	 * -> Non collapsed and not on the surface
	 * -> collapsed (2d model) and not on bed) */
	if ((!collapse && !onsurface) || (collapse && !onbed)){
		return 0;
	}
	else if (collapse){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute CostFunction*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->CostFunction();
		delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->CostFunction();
		delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrix {{{1*/
void  Penta::CreateKMatrix(Mat Kgg){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*if debugging mode, check that all pointers exist*/
	ISSMASSERT(this->nodes && this->matice && this->matpar && this->neighbors && this->parameters && this->inputs);

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==ControlAnalysisEnum){
		CreateKMatrixDiagnosticHoriz( Kgg);
	}
	else if (analysis_type==DiagnosticHorizAnalysisEnum){
		CreateKMatrixDiagnosticHoriz( Kgg);
	}
	else if (analysis_type==DiagnosticHutterAnalysisEnum){
		CreateKMatrixDiagnosticHutter( Kgg);
	}
	else if (analysis_type==DiagnosticVertAnalysisEnum){
		CreateKMatrixDiagnosticVert( Kgg);
	}
	else if (analysis_type==DiagnosticStokesAnalysisEnum){
		CreateKMatrixDiagnosticStokes( Kgg);
	}
	else if (analysis_type==BedSlopeXAnalysisEnum || analysis_type==SurfaceSlopeXAnalysisEnum || analysis_type==BedSlopeYAnalysisEnum || analysis_type==SurfaceSlopeYAnalysisEnum){
		CreateKMatrixSlope( Kgg);
	}
	else if (analysis_type==PrognosticAnalysisEnum){
		CreateKMatrixPrognostic( Kgg);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){
		CreateKMatrixBalancedthickness( Kgg);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){
		CreateKMatrixBalancedvelocities( Kgg);
	}
	else if (analysis_type==ThermalAnalysisEnum){
		CreateKMatrixThermal( Kgg);
	}
	else if (analysis_type==MeltingAnalysisEnum){
		CreateKMatrixMelting( Kgg);
	}
	else ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));

}
/*}}}*/
/*FUNCTION Penta::CreatePVector {{{1*/
void  Penta::CreatePVector(Vec pg){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*if debugging mode, check that all pointers exist*/
	ISSMASSERT(this->nodes && this->matice && this->matpar && this->neighbors && this->parameters && this->inputs);

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==ControlAnalysisEnum){
		CreatePVectorDiagnosticHoriz( pg);
	}
	else if (analysis_type==DiagnosticHorizAnalysisEnum){
		CreatePVectorDiagnosticHoriz( pg);
	}
	else if (analysis_type==DiagnosticHutterAnalysisEnum){
		CreatePVectorDiagnosticHutter( pg);
	}
	else if (analysis_type==DiagnosticVertAnalysisEnum){
		CreatePVectorDiagnosticVert( pg);
	}
	else if (analysis_type==DiagnosticStokesAnalysisEnum){
		CreatePVectorDiagnosticStokes( pg);
	}
	else if (analysis_type==BedSlopeXAnalysisEnum || analysis_type==SurfaceSlopeXAnalysisEnum || analysis_type==BedSlopeYAnalysisEnum || analysis_type==SurfaceSlopeYAnalysisEnum){
		CreatePVectorSlope( pg);
	}
	else if (analysis_type==PrognosticAnalysisEnum){
		CreatePVectorPrognostic( pg);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){
		CreatePVectorBalancedthickness( pg);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){
		CreatePVectorBalancedvelocities( pg);
	}
	else if (analysis_type==ThermalAnalysisEnum){
		CreatePVectorThermal( pg);
	}
	else if (analysis_type==MeltingAnalysisEnum){
		CreatePVectorMelting( pg);
	}
	else ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));

}
/*}}}*/
/*FUNCTION Penta::Du {{{1*/
void  Penta::Du(Vec du_g){

	int i;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool collapse;
	bool onsurface;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&collapse,CollapseEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Bail out if this element if:
	 * -> Non collapsed and not on the surface
	 * -> collapsed (2d model) and not on bed) */
	if ((!collapse && !onsurface) || (collapse && !onbed)){
		return;
	}
	else if (collapse){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute Du*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		tria->Du(du_g);
		delete tria;
		return;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		tria->Du(du_g);
		delete tria;
		return;
	}
}
/*}}}*/
/*FUNCTION Penta::GetBedList {{{1*/
void Penta::GetBedList(double* bedlist){

	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	
	inputs->GetParameterValues(bedlist,&gaussgrids[0][0],6,BedEnum);

}
/*}}}*/
/*FUNCTION Penta::GetMatPar {{{1*/
void* Penta::GetMatPar(){

	return matpar;
}
/*}}}*/
/*FUNCTION Penta::GetNodes {{{1*/
void  Penta::GetNodes(void** vpnodes){

	int i;
	Node** pnodes=NULL;
	
	/*virtual object: */
	pnodes=(Node**)vpnodes;

	for(i=0;i<6;i++){
		pnodes[i]=nodes[i];
	}
}
/*}}}*/
/*FUNCTION Penta::GetOnBed {{{1*/
bool Penta::GetOnBed(){
	
	bool onbed;

	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	return onbed;
}
/*}}}*/
/*FUNCTION Penta::GetShelf {{{1*/
bool   Penta::GetShelf(){
	bool onshelf;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onshelf,ElementOnIceShelfEnum);

	return onshelf;
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputs{{{1*/
void  Penta::GetSolutionFromInputs(Vec solution){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Just branch to the correct InputUpdateFromSolution generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		GetSolutionFromInputsDiagnosticHoriz(solution);
	}
	else if(analysis_type==DiagnosticHutterAnalysisEnum){
		GetSolutionFromInputsDiagnosticHutter(solution);
	}
	else if(analysis_type==DiagnosticVertAnalysisEnum){
		GetSolutionFromInputsDiagnosticVert(solution);
	}
	else if(analysis_type==DiagnosticStokesAnalysisEnum){
		GetSolutionFromInputsDiagnosticStokes(solution);
	}
	else if(analysis_type==ThermalAnalysisEnum){
		GetSolutionFromInputsThermal(solution);
	}
	else{
		ISSMERROR("analysis: %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));
	}
}
/*}}}*/
/*FUNCTION Penta::GetThicknessList {{{1*/
void Penta::GetThicknessList(double* thickness_list){

	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	inputs->GetParameterValues(thickness_list,&gaussgrids[0][0],6,ThicknessEnum);

}
/*}}}*/
/*FUNCTION Penta::GetVectorFromInputs{{{1*/
void  Penta::GetVectorFromInputs(Vec vector,int NameEnum){

	int i;
	const int numvertices=6;
	int doflist1[numvertices];

	/*Find NameEnum input in the inputs dataset, and get it to fill in the vector: */
	for(i=0;i<this->inputs->Size();i++){
		Input* input=(Input*)this->inputs->GetObjectByOffset(i);
		if(input->EnumType()==NameEnum){
			/*We found the enum.  Use its values to fill into the vector, using the vertices ids: */
			this->GetDofList1(&doflist1[0]);
			input->GetVectorFromInputs(vector,&doflist1[0]);
			break;
		}
	}
}
/*}}}*/
/*FUNCTION Penta::Gradj {{{1*/
void  Penta::Gradj(Vec gradient,int control_type){

	/*inputs: */
	bool onwater;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip grad (=0): */
	if(onwater)return;

	if (control_type==DragCoefficientEnum){
		GradjDrag(gradient);
	}
	else if (control_type=RheologyBEnum){
		GradjB(gradient);
	}
	else ISSMERROR("%s%i","control type not supported yet: ",control_type);
}
/*}}}*/
/*FUNCTION Penta::GradjDrag {{{1*/
void  Penta::GradjDrag(Vec gradient){

	int i;
	Tria* tria=NULL;
	TriaVertexInput* triavertexinput=NULL;
	double temp_gradient[6]={0,0,0,0,0,0};

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;
	int analysis_type;

	/*retrieve parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*If on shelf, skip: */
	if(shelf)return;

	/*Bail out if this element does not touch the bedrock: */
	if (!onbed) return;

	if (analysis_type==AdjointHorizAnalysisEnum){

		/*MacAyeal or Pattyn*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->GradjDrag(gradient);
		delete tria;
	}
	else if (analysis_type==AdjointStokesAnalysisEnum){

		/*Stokes*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->GradjDragStokes(gradient);
		delete tria;
	}
	else ISSMERROR("analysis %s not supported yet",EnumAsString(analysis_type));


}
/*}}}*/
/*FUNCTION Penta::InputAXPY{{{1*/
void  Penta::InputAXPY(int YEnum, double scalar, int XEnum){

	Input* xinput=NULL;
	Input* yinput=NULL;

	/*Find x and y inputs: */
	xinput=(Input*)this->inputs->GetInput(XEnum);
	yinput=(Input*)this->inputs->GetInput(YEnum);

	/*some checks: */
	if(!xinput || !yinput)ISSMERROR("%s%s%s%s%s"," input ",EnumAsString(XEnum)," or input ",EnumAsString(YEnum)," could not be found!");
	if(xinput->Enum()!=yinput->Enum())ISSMERROR("%s%s%s%s%s"," input ",EnumAsString(XEnum)," and input ",EnumAsString(YEnum)," are not of the same type!");

	/*Scale: */
	yinput->AXPY(xinput,scalar);
}
/*}}}*/
/*FUNCTION Penta::InputControlConstrain{{{1*/
void  Penta::InputControlConstrain(int control_type, double cm_min, double cm_max){

	Input* input=NULL;

	/*Find input: */
	input=(Input*)this->inputs->GetInput(control_type);
	
	/*Do nothing if we  don't find it: */
	if(!input) return;

	/*Constrain input using cm_min and cm_max: */
	input->Constrain(cm_min,cm_max);

}
/*}}}*/
/*FUNCTION Penta::InputConvergence{{{1*/
void  Penta::InputConvergence(int* pconverged,double* eps, int* enums,int num_enums,int* criterionenums,double* criterionvalues,int num_criterionenums){

	int     i;
	Input** new_inputs=NULL;
	Input** old_inputs=NULL;
	int     converged=1;

	new_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the new inputs
	old_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the old inputs
	
	for(i=0;i<num_enums/2;i++){
		new_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+0]);
		old_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+1]);
		if(!new_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumAsString(enums[2*i+0]));
		if(!old_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumAsString(enums[2*i+0]));
	}

	/*ok, we've got the inputs (new and old), now loop throught the number of criterions and fill the eps array:*/
	for(i=0;i<num_criterionenums;i++){
		IsInputConverged(eps+i,new_inputs,old_inputs,num_enums/2,criterionenums[i]);
		if(eps[i]>criterionvalues[i]) converged=0; 
	}

	/*Assign output pointers:*/
	*pconverged=converged;

}
/*}}}*/
/*FUNCTION Penta::InputDepthAverageAtBase{{{1*/
void  Penta::InputDepthAverageAtBase(int enum_type,int average_enum_type){

	/*Intermediaries*/
	const int numvertices=6;
	bool onbed;
	int  i;

	Penta* penta=NULL;
	Input* original_input=NULL;
	Input* element_integrated_input=NULL;
	Input* total_integrated_input=NULL;
	Input* element_thickness_input=NULL;
	Input* total_thickness_input=NULL;
	Input* depth_averaged_input=NULL;

	double  xyz_list[numvertices][3];
	double  Helem_list[numvertices];
	double  zeros_list[numvertices]={0.0};

	/*recover parameters: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*Are we on the base? If not, return*/
	if(!onbed) return;

	/*OK, we are on bed. Initialize global inputs as 0*/
	total_thickness_input =new PentaVertexInput(ThicknessEnum,zeros_list);
	total_integrated_input=new PentaVertexInput(average_enum_type,zeros_list);

	/*Now follow all the upper element from the base to the surface to integrate the input*/
	penta=this;
	for(;;){

		/*Step1: Get original input (to be depth avegaged): */
		original_input=(Input*)penta->inputs->GetInput(enum_type);
		if(!original_input) ISSMERROR("could not find input with enum %s",EnumAsString(enum_type));

		/*Step2: Create element thickness input*/
		GetVerticesCoordinates(&xyz_list[0][0],penta->nodes,numvertices);
		for(i=0;i<3;i++){
			Helem_list[i]=xyz_list[i+3][2]-xyz_list[i][2];
			Helem_list[i+3]=Helem_list[i];
		}
		element_thickness_input=new PentaVertexInput(ThicknessEnum,Helem_list);

		/*Step3: Vertically integrate A COPY of the original*/
		element_integrated_input=(Input*)original_input->copy();
		element_integrated_input->VerticallyIntegrate(element_thickness_input);

		/*Add contributions to global inputs*/
		total_integrated_input->AXPY(element_integrated_input,1.0);
		total_thickness_input ->AXPY(element_thickness_input,1.0);

		/*Clean up*/
		delete element_thickness_input;
		delete element_integrated_input;

		/*Stop if we have reached the surface, else, take upper penta*/
		if (penta->IsOnSurface()){
			break;
		}
		else{
			/* get upper Penta*/
			penta=penta->GetUpperElement();
			ISSMASSERT(penta->Id()!=this->id);
		}
	}

	/*OK, now we only need to divide the depth integrated input by the total thickness!*/
	depth_averaged_input=total_integrated_input->PointwiseDivide(total_thickness_input);
	depth_averaged_input->ChangeEnum(average_enum_type);

	/*Clean up*/
	delete total_thickness_input;
	delete total_integrated_input;

	/*Finally, add to inputs*/
	this->inputs->AddInput((Input*)depth_averaged_input);
	return;
}
/*}}}*/
/*FUNCTION Penta::InputDuplicate{{{1*/
void  Penta::InputDuplicate(int original_enum,int new_enum){

	Input* original=NULL;
	Input* copy=NULL;

	/*Make a copy of the original input: */
	original=(Input*)this->inputs->GetInput(original_enum);
	if(!original)ISSMERROR(" could not find old input with enum: %s",EnumAsString(original_enum)); 
	copy=(Input*)original->copy();

	/*Change copy enum to reinitialized_enum: */
	copy->ChangeEnum(new_enum);

	/*Add copy into inputs, it will wipe off the one already there: */
	inputs->AddInput((Input*)copy);
}
/*}}}*/
/*FUNCTION Penta::InputScale{{{1*/
void  Penta::InputScale(int enum_type,double scale_factor){

	Input* input=NULL;

	/*Make a copy of the original input: */
	input=(Input*)this->inputs->GetInput(enum_type);
	if(!input)ISSMERROR(" could not find old input with enum: %s",EnumAsString(enum_type));

	/*Scale: */
	input->Scale(scale_factor);
}
/*}}}*/
/*FUNCTION Penta::InputToResult{{{1*/
void  Penta::InputToResult(int enum_type,int step,double time){

	int    i;
	bool   found = false;
	Input *input = NULL;

	/*Go through all the input objects, and find the one corresponding to enum_type, if it exists: */
	for (i=0;i<this->inputs->Size();i++){
		input=(Input*)this->inputs->GetObjectByOffset(i);
		if (input->EnumType()==enum_type){
			found=true; break;
		}
	}
	if (!found) ISSMERROR("Input %s not found in penta->inputs",EnumAsString(enum_type));

	/*If we don't find it, no big deal, just don't do the transfer. Otherwise, build a new Result 
	 * object out of the input, with the additional step and time information: */
	this->results->AddObject((Object*)input->SpawnResult(step,time));

}
/*}}}*/
/*FUNCTION Penta::MassFlux {{{1*/
double Penta::MassFlux( double* segment){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::MaterialUpdateFromTemperature{{{1*/
void  Penta::MaterialUpdateFromTemperature(void){

	/*Intermediaries*/
	double Taverage;
	double B;

	/*inputs: */
	bool   collapse;
	Input *TempInput = NULL;
	Input *RheologyBInput = NULL;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&collapse,CollapseEnum);

	if (collapse){
		/*find temperature average*/
		this->inputs->GetParameterAverage(&Taverage,TemperatureAverageEnum);
	}
	else{
		/*find 3d temperature*/
		this->inputs->GetParameterAverage(&Taverage,TemperatureEnum);
	}

	/*Get B from T using Paterson*/
	B=Paterson(Taverage);

	/*Add input to Matice*/
	this->matice->inputs->AddInput(new DoubleInput(RheologyBEnum,B));

}
/*}}}*/
/*FUNCTION Penta::MaxAbsVx{{{1*/
void  Penta::MaxAbsVx(double* pmaxabsvx, bool process_units){

	int i;
	int dim;
	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	double  vx_values[numgrids];
	double  maxabsvx;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vx_values[0],&gaussgrids[0][0],numgrids,VxEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vx_values[0],numgrids,VxEnum,this->parameters);

	/*now, compute maximum:*/
	maxabsvx=fabs(vx_values[0]);
	for(i=1;i<numgrids;i++){
		if (fabs(vx_values[i])>maxabsvx)maxabsvx=fabs(vx_values[i]);
	}

	/*Assign output pointers:*/
	*pmaxabsvx=maxabsvx;
}
/*}}}*/
/*FUNCTION Penta::MaxAbsVy{{{1*/
void  Penta::MaxAbsVy(double* pmaxabsvy, bool process_units){

	int i;
	int dim;
	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	double  vy_values[numgrids];
	double  maxabsvy;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vy_values[0],&gaussgrids[0][0],numgrids,VyEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vy_values[0],numgrids,VyEnum,this->parameters);

	/*now, compute maximum:*/
	maxabsvy=fabs(vy_values[0]);
	for(i=1;i<numgrids;i++){
		if (fabs(vy_values[i])>maxabsvy)maxabsvy=fabs(vy_values[i]);
	}

	/*Assign output pointers:*/
	*pmaxabsvy=maxabsvy;
}
/*}}}*/
/*FUNCTION Penta::MaxAbsVz{{{1*/
void  Penta::MaxAbsVz(double* pmaxabsvz, bool process_units){

	int i;
	int dim;
	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	double  vz_values[numgrids];
	double  maxabsvz;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vz_values[0],&gaussgrids[0][0],numgrids,VzEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vz_values[0],numgrids,VzEnum,this->parameters);

	/*now, compute maximum:*/
	maxabsvz=fabs(vz_values[0]);
	for(i=1;i<numgrids;i++){
		if (fabs(vz_values[i])>maxabsvz)maxabsvz=fabs(vz_values[i]);
	}

	/*Assign output pointers:*/
	*pmaxabsvz=maxabsvz;
}
/*}}}*/
/*FUNCTION Penta::MaxVel{{{1*/
void  Penta::MaxVel(double* pmaxvel, bool process_units){

	int i;
	int dim;
	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	double  vel_values[numgrids];
	double  maxvel;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vel_values[0],&gaussgrids[0][0],numgrids,VelEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vel_values[0],numgrids,VelEnum,this->parameters);

	/*now, compute maximum:*/
	maxvel=vel_values[0];
	for(i=1;i<numgrids;i++){
		if (vel_values[i]>maxvel)maxvel=vel_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvel=maxvel;

}
/*}}}*/
/*FUNCTION Penta::MaxVx{{{1*/
void  Penta::MaxVx(double* pmaxvx, bool process_units){

	int i;
	int dim;
	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	double  vx_values[numgrids];
	double  maxvx;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vx_values[0],&gaussgrids[0][0],numgrids,VxEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vx_values[0],numgrids,VxEnum,this->parameters);

	/*now, compute maximum:*/
	maxvx=vx_values[0];
	for(i=1;i<numgrids;i++){
		if (vx_values[i]>maxvx)maxvx=vx_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvx=maxvx;

}
/*}}}*/
/*FUNCTION Penta::MaxVy{{{1*/
void  Penta::MaxVy(double* pmaxvy, bool process_units){

	int i;
	int dim;
	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	double  vy_values[numgrids];
	double  maxvy;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vy_values[0],&gaussgrids[0][0],numgrids,VyEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vy_values[0],numgrids,VyEnum,this->parameters);

	/*now, compute maximum:*/
	maxvy=vy_values[0];
	for(i=1;i<numgrids;i++){
		if (vy_values[i]>maxvy)maxvy=vy_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvy=maxvy;

}
/*}}}*/
/*FUNCTION Penta::MaxVz{{{1*/
void  Penta::MaxVz(double* pmaxvz, bool process_units){

	int i;
	int dim;
	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	double  vz_values[numgrids];
	double  maxvz;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vz_values[0],&gaussgrids[0][0],numgrids,VzEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vz_values[0],numgrids,VzEnum,this->parameters);

	/*now, compute maximum:*/
	maxvz=vz_values[0];
	for(i=1;i<numgrids;i++){
		if (vz_values[i]>maxvz)maxvz=vz_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvz=maxvz;

}
/*}}}*/
/*FUNCTION Penta::MinVel{{{1*/
void  Penta::MinVel(double* pminvel, bool process_units){

	int i;
	int dim;
	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	double  vel_values[numgrids];
	double  minvel;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vel_values[0],&gaussgrids[0][0],numgrids,VelEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vel_values[0],numgrids,VelEnum,this->parameters);

	/*now, compute minimum:*/
	minvel=vel_values[0];
	for(i=1;i<numgrids;i++){
		if (vel_values[i]<minvel)minvel=vel_values[i];
	}

	/*Assign output pointers:*/
	*pminvel=minvel;

}
/*}}}*/
/*FUNCTION Penta::MinVx{{{1*/
void  Penta::MinVx(double* pminvx, bool process_units){

	int i;
	int dim;
	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	double  vx_values[numgrids];
	double  minvx;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vx_values[0],&gaussgrids[0][0],numgrids,VxEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vx_values[0],numgrids,VxEnum,this->parameters);

	/*now, compute minimum:*/
	minvx=vx_values[0];
	for(i=1;i<numgrids;i++){
		if (vx_values[i]<minvx)minvx=vx_values[i];
	}

	/*Assign output pointers:*/
	*pminvx=minvx;

}
/*}}}*/
/*FUNCTION Penta::MinVy{{{1*/
void  Penta::MinVy(double* pminvy, bool process_units){

	int i;
	int dim;
	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	double  vy_values[numgrids];
	double  minvy;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vy_values[0],&gaussgrids[0][0],numgrids,VyEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vy_values[0],numgrids,VyEnum,this->parameters);

	/*now, compute minimum:*/
	minvy=vy_values[0];
	for(i=1;i<numgrids;i++){
		if (vy_values[i]<minvy)minvy=vy_values[i];
	}

	/*Assign output pointers:*/
	*pminvy=minvy;

}
/*}}}*/
/*FUNCTION Penta::MinVz{{{1*/
void  Penta::MinVz(double* pminvz, bool process_units){

	int i;
	int dim;
	const int numgrids=6;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	double  vz_values[numgrids];
	double  minvz;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vz_values[0],&gaussgrids[0][0],numgrids,VzEnum);

	/*process units if requested: */
	if(process_units)NodalValuesUnitConversion(&vz_values[0],numgrids,VzEnum,this->parameters);

	/*now, compute minimum:*/
	minvz=vz_values[0];
	for(i=1;i<numgrids;i++){
		if (vz_values[i]<minvz)minvz=vz_values[i];
	}

	/*Assign output pointers:*/
	*pminvz=minvz;

}
/*}}}*/
/*FUNCTION Penta::Misfit {{{1*/
double Penta::Misfit(void){

	double J;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool collapse;
	bool onsurface;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&collapse,CollapseEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*Bail out if this element if:
	 * -> Non collapsed and not on the surface
	 * -> collapsed (2d model) and not on bed) */
	if ((!collapse && !onsurface) || (collapse && !onbed)){
		return 0;
	}
	else if (collapse){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute Misfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->Misfit();
		delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->Misfit();
		delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::PatchFill{{{1*/
void  Penta::PatchFill(int* pcount, Patch* patch){

	int i;
	int count;
	int vertices_ids[6];


	/*recover pointer: */
	count=*pcount;
		
	/*will be needed later: */
	for(i=0;i<6;i++) vertices_ids[i]=nodes[i]->GetVertexId(); //vertices id start at column 3 of the patch.

	for(i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);

		/*For this result,fill the information in the Patch object (element id + vertices ids), and then hand 
		 *it to the result object, to fill the rest: */
		patch->fillelementinfo(count,this->id,vertices_ids,6);
		elementresult->PatchFill(count,patch);

		/*increment counter: */
		count++;
	}

	/*Assign output pointers:*/
	*pcount=count;
}/*}}}*/
/*FUNCTION Penta::PatchSize{{{1*/
void  Penta::PatchSize(int* pnumrows, int* pnumvertices,int* pnumnodes){

	int     i;
	
	/*output: */
	int     numrows     = 0;
	int     numvertices = 0;
	int     numnodes    = 0;

	/*Go through all the results objects, and update the counters: */
	for (i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);
		/*first, we have one more result: */
		numrows++;
		/*now, how many vertices and how many nodal values for this result? :*/
		numvertices=6; //this is a penta element, with 6 vertices
		numnodes=elementresult->NumberOfNodalValues(); //ask result object.
	}

	/*Assign output pointers:*/
	*pnumrows=numrows;
	*pnumvertices=numvertices;
	*pnumnodes=numnodes;
	
}
/*}}}*/
/*FUNCTION Penta::ProcessResultsUnits{{{1*/
void  Penta::ProcessResultsUnits(void){

	int i;

	for(i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);
		elementresult->ProcessUnits(this->parameters);
	}

}
/*}}}*/
/*FUNCTION Penta::SurfaceArea {{{1*/
double Penta::SurfaceArea(void){

	double S;
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool collapse;
	bool onsurface;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&collapse,CollapseEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*Bail out if this element if:
	 * -> Non collapsed and not on the surface
	 * -> collapsed (2d model) and not on bed) */
	if ((!collapse && !onsurface) || (collapse && !onbed)){
		return 0;
	}
	else if (collapse){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceArea*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		S=tria->SurfaceArea();
		delete tria;
		return S;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		S=tria->SurfaceArea();
		delete tria;
		return S;
	}
}
/*}}}*/
/*FUNCTION Penta::Update {{{1*/
void Penta::Update(int index,IoModel* iomodel,int analysis_counter,int analysis_type){ 

	/*Intermediaries*/
	IssmInt i;
	int penta_node_ids[6];
	int penta_vertex_ids[6];
	double nodeinputs[6];

	/*Checks if debuging*/
	/*{{{2*/
	ISSMASSERT(iomodel->elements);
	/*}}}*/

	/*Recover vertices ids needed to initialize inputs*/
	for(i=0;i<6;i++){ 
		penta_vertex_ids[i]=(int)iomodel->elements[6*index+i]; //ids for vertices are in the elements array from Matlab
	}

	/*Recover nodes ids needed to initialize the node hook.*/
	for(i=0;i<6;i++){ 
		//go recover node ids, needed to initialize the node hook.
		//WARNING: We assume P1 elements here!!!!!
		penta_node_ids[i]=iomodel->nodecounter+(int)iomodel->elements[6*index+i]; //ids for vertices are in the elements array from Matlab
	}

	/*hooks: */
	this->SetHookNodes(penta_node_ids,analysis_counter); this->nodes=NULL; //set hook to nodes, for this analysis type

	//add as many inputs per element as requested: 
	if (iomodel->thickness) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->thickness[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(ThicknessEnum,nodeinputs));
	}
	if (iomodel->surface) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->surface[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(SurfaceEnum,nodeinputs));
	}
	if (iomodel->bed) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->bed[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(BedEnum,nodeinputs));
	}
	if (iomodel->drag_coefficient) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->drag_coefficient[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(DragCoefficientEnum,nodeinputs));

		if (iomodel->drag_p) this->inputs->AddInput(new DoubleInput(DragPEnum,iomodel->drag_p[index]));
		if (iomodel->drag_q) this->inputs->AddInput(new DoubleInput(DragQEnum,iomodel->drag_q[index]));
		this->inputs->AddInput(new IntInput(DragTypeEnum,iomodel->drag_type));

	}
	if (iomodel->rheology_B) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->rheology_B[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(RheologyBEnum,nodeinputs));
	}
	if (iomodel->control_parameter) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->control_parameter[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(ControlParameterEnum,nodeinputs));
	}
	if (iomodel->melting_rate) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->melting_rate[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(MeltingRateEnum,nodeinputs));
	}
	if (iomodel->accumulation_rate) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->accumulation_rate[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(AccumulationRateEnum,nodeinputs));
	}
	if (iomodel->geothermalflux) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->geothermalflux[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(GeothermalFluxEnum,nodeinputs));
	}	
	if (iomodel->pressure) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->pressure[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(PressureEnum,nodeinputs));
	}
	if (iomodel->temperature) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->temperature[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(TemperatureEnum,nodeinputs));
	}
	if (iomodel->dhdt) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->dhdt[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(DhDtEnum,nodeinputs));
	}
	/*vx,vy and vz: */
	if (iomodel->vx) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vx[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VxEnum,nodeinputs));
		this->inputs->AddInput(new PentaVertexInput(VxOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new PentaVertexInput(QmuVxEnum,nodeinputs));
	}
	if (iomodel->vy) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vy[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VyEnum,nodeinputs));
		this->inputs->AddInput(new PentaVertexInput(VyOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new PentaVertexInput(QmuVyEnum,nodeinputs));
	}
	if (iomodel->vz) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vz[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VzEnum,nodeinputs));
		this->inputs->AddInput(new PentaVertexInput(VzOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new PentaVertexInput(QmuVzEnum,nodeinputs));
	}
	if (iomodel->vx_obs) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vx_obs[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VxObsEnum,nodeinputs));
	}
	if (iomodel->vy_obs) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vy_obs[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VyObsEnum,nodeinputs));
	}
	if (iomodel->vz_obs) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->vz_obs[penta_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new PentaVertexInput(VzObsEnum,nodeinputs));
	}
	if (iomodel->weights) {
		for(i=0;i<6;i++)nodeinputs[i]=iomodel->weights[penta_vertex_ids[i]-1];
		this->inputs->AddInput(new PentaVertexInput(WeightsEnum,nodeinputs));
	}
	if (iomodel->elementoniceshelf) this->inputs->AddInput(new BoolInput(ElementOnIceShelfEnum,(IssmBool)iomodel->elementoniceshelf[index]));
	if (iomodel->elementonbed) this->inputs->AddInput(new BoolInput(ElementOnBedEnum,(IssmBool)iomodel->elementonbed[index]));
	if (iomodel->elementonwater) this->inputs->AddInput(new BoolInput(ElementOnWaterEnum,(IssmBool)iomodel->elementonwater[index]));
	if (iomodel->elementonsurface) this->inputs->AddInput(new BoolInput(ElementOnSurfaceEnum,(IssmBool)iomodel->elementonsurface[index]));

	/*Defaults if not provided in iomodel*/
	switch(analysis_type){

		case DiagnosticHorizAnalysisEnum: case DiagnosticVertAnalysisEnum: case DiagnosticStokesAnalysisEnum:

			/*default vx,vy and vz: either observation or 0 */
			if(!iomodel->vx){
				if (false && iomodel->vx_obs) for(i=0;i<6;i++)nodeinputs[i]=iomodel->vx_obs[penta_vertex_ids[i]-1]/iomodel->yts; // false TO BE DELETED (for NR CM)
				else                 for(i=0;i<6;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new PentaVertexInput(VxEnum,nodeinputs));
				this->inputs->AddInput(new PentaVertexInput(VxOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new PentaVertexInput(QmuVxEnum,nodeinputs));
			}
			if(!iomodel->vy){
				if (false && iomodel->vy_obs) for(i=0;i<6;i++)nodeinputs[i]=iomodel->vy_obs[penta_vertex_ids[i]-1]/iomodel->yts; // false TO BE DELETED (for NR CM)
				else                 for(i=0;i<6;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new PentaVertexInput(VyEnum,nodeinputs));
				this->inputs->AddInput(new PentaVertexInput(VyOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new PentaVertexInput(QmuVyEnum,nodeinputs));
			}
			if(!iomodel->vz){
				if (iomodel->vz_obs) for(i=0;i<6;i++)nodeinputs[i]=iomodel->vz_obs[penta_vertex_ids[i]-1]/iomodel->yts;
				else                 for(i=0;i<6;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new PentaVertexInput(VzEnum,nodeinputs));
				this->inputs->AddInput(new PentaVertexInput(VzOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new PentaVertexInput(QmuVzEnum,nodeinputs));
			}
			if(!iomodel->pressure){
				for(i=0;i<6;i++)nodeinputs[i]=0;
				if(iomodel->qmu_analysis){
					this->inputs->AddInput(new PentaVertexInput(PressureEnum,nodeinputs));
					this->inputs->AddInput(new PentaVertexInput(QmuPressureEnum,nodeinputs));
				}
			}
			break;

		default:
			/*No update for other solution types*/
			break;

	}

	//elements of type 3 are macayeal type penta. we collapse the formulation on their base.
	if(iomodel->elements_type){
		if (*(iomodel->elements_type+2*index+0)==MacAyealFormulationEnum) 
		 this->inputs->AddInput(new BoolInput(CollapseEnum,true));
		else
		 this->inputs->AddInput(new BoolInput(CollapseEnum,false));
	}

	//if diagnostic Full-Stokes: check that the element is Stokes
	if(analysis_type==DiagnosticStokesAnalysisEnum){
		ISSMASSERT(iomodel->elements_type);
		if(iomodel->elements_type[2*index+1]==StokesFormulationEnum)
		 this->inputs->AddInput(new BoolInput(IsStokesEnum,true));
		else
		 this->inputs->AddInput(new BoolInput(IsStokesEnum,false));
	}

}
/*}}}*/
/*FUNCTION Penta::UpdateGeometry{{{1*/
void  Penta::UpdateGeometry(void){

	/*Intermediaries*/
	int i;
	double rho_ice,rho_water;

	/*If shelf: hydrostatic equilibrium*/
	if (this->GetShelf()){

		/*recover material parameters: */
		rho_ice=matpar->GetRhoIce();
		rho_water=matpar->GetRhoWater();

		/*Create New Surface: s = (1-rho_ice/rho_water) h*/
		InputDuplicate(ThicknessEnum,SurfaceEnum);     //1: copy thickness into surface
		InputScale(SurfaceEnum,(1-rho_ice/rho_water)); //2: surface = surface * (1-di)

		/*Create New Bed b = -rho_ice/rho_water h*/
		InputDuplicate(ThicknessEnum,BedEnum);         //1: copy thickness into bed
		InputScale(BedEnum, -rho_ice/rho_water);       //2: bed = bed * (-di)
	}

	/*If sheet: surface = bed + thickness*/
	else{

		/*The bed does not change, update surface only s = b + h*/
		InputDuplicate(BedEnum,SurfaceEnum);          //1: copy bed into surface
		InputAXPY(SurfaceEnum,1.0,ThicknessEnum);     //2: surface = surface + 1 * thickness
	}

}
/*}}}*/

/*Penta specific routines: */
/*FUNCTION Penta::SpawnTria {{{1*/
Tria*  Penta::SpawnTria(int g0, int g1, int g2){

	int i;
	int analysis_counter;
	
	/*go into parameters and get the analysis_counter: */
	this->parameters->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*out of grids g0,g1 and g2 from Penta, build a tria element: */
	Tria* tria=NULL;
	int indices[3];
	int zero=0;
	Parameters* tria_parameters=NULL;
	Inputs* tria_inputs=NULL;
	Results* tria_results=NULL;

	indices[0]=g0;
	indices[1]=g1;
	indices[2]=g2;

	tria_parameters=this->parameters;
	tria_inputs=(Inputs*)this->inputs->SpawnTriaInputs(indices);
	tria_results=(Results*)this->results->SpawnTriaResults(indices);
	
	tria=new Tria();
	tria->id=this->id;
	tria->inputs=tria_inputs;
	tria->results=tria_results;
	tria->parameters=tria_parameters;
	tria->element_type=P1Enum; //Only P1 CG for now
	this->SpawnTriaHook(dynamic_cast<TriaHook*>(tria),&indices[0]);

	/*recover nodes, matice and matpar: */
	tria->nodes=(Node**)tria->hnodes[analysis_counter]->deliverp();
	tria->matice=(Matice*)tria->hmatice->delivers();
	tria->matpar=(Matpar*)tria->hmatpar->delivers();
	
	return tria;
}
/*}}}*/
/*FUNCTION Penta::SpawnBeam {{{1*/
void* Penta::SpawnBeam(int g0, int g1){

	int i;

	/*out of grids g0,g1 and g2 from Penta, build a beam element: */
	Beam* beam=NULL;
	int indices[2];
	int zero=0;
	Parameters *beam_parameters = NULL;
	Inputs     *beam_inputs     = NULL;

	indices[0]=g0;
	indices[1]=g1;

	beam_parameters=this->parameters;
	beam_inputs=(Inputs*)this->inputs->SpawnBeamInputs(indices);

	beam=new Beam();
	beam->id=this->id;
	beam->inputs=beam_inputs;
	beam->parameters=beam_parameters;

	/*now deal with ndoes,matice and matpar: */
	beam->nodes=(Node**)xmalloc(2*sizeof(Node*));
	for(i=0;i<2;i++)beam->nodes[i]=this->nodes[indices[i]];
	beam->matice=this->matice;
	beam->matpar=this->matpar;

	return beam;
}
/*}}}*/
/*FUNCTION Penta::SpawnSing {{{1*/
void* Penta::SpawnSing(int index){

	Sing* sing=NULL;
	int zero=0;
	Parameters *sing_parameters = NULL;
	Inputs     *sing_inputs     = NULL;

	sing_parameters=this->parameters;
	sing_inputs=(Inputs*)this->inputs->SpawnSingInputs(index);

	sing=new Sing();
	sing->id=this->id;
	sing->inputs=sing_inputs;
	sing->parameters=sing_parameters;

	/*now deal with nodes,matice and matpar: */
	sing->node=this->nodes[index];
	sing->matice=this->matice;
	sing->matpar=this->matpar;
	
	return sing;
}
/*}}}*/
/*FUNCTION Penta::GaussFromNode {{{1*/
double* Penta::GaussFromNode(Node* node){

	/*variable declaration*/
	int i,pos;
	double*  gauss=NULL;

	/*Allocate gauss*/
	gauss=(double*)xmalloc(4*sizeof(double));

	for(i=0;i<6;i++){
		if (node==nodes[i]){
			switch(i){
				case 0:
					gauss[0]=1.0; gauss[1]=0.0; gauss[2]=0.0; gauss[3]= -1.0;
					return gauss;
				case 1:
					gauss[0]=0.0; gauss[1]=1.0; gauss[2]=0.0; gauss[3]= -1.0;
					return gauss;
				case 2:
					gauss[0]=0.0; gauss[1]=0.0; gauss[2]=1.0; gauss[3]= -1.0;
					return gauss;
				case 3:
					gauss[0]=1.0; gauss[1]=0.0; gauss[2]=0.0; gauss[3]=1.0;
					return gauss;
				case 4:
					gauss[0]=0.0; gauss[1]=1.0; gauss[2]=0.0; gauss[3]=1.0;
					return gauss;
				case 5:
					gauss[0]=0.0; gauss[1]=0.0; gauss[2]=1.0; gauss[3]=1.0;
					return gauss;
			}
		}
	}

	/*output error if not found*/
	ISSMERROR("Node not found in Penta");
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticHoriz {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticHoriz(double* solution){
	
	int i;

	const int    numvertices=6;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;
	
	int          doflist[numdof];
	double       values[numdof];
	double       vx[numvertices];
	double       vy[numvertices];
	double       vz[numvertices];
	double       vel[numvertices];
	int          dummy;
	double       pressure[numvertices];
	double       surface[numvertices];
	double       rho_ice,g;
	double       xyz_list[numvertices][3];
	double       gauss[numvertices][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	
	Input*       VzInput=NULL;
	double*      VzPtr=NULL;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numvertices);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<numvertices;i++){
		vx[i]=values[i*numdofpervertex+0];
		vy[i]=values[i*numdofpervertex+1];
	}

	/*Get Vz*/
	VzInput=inputs->GetInput(VzEnum);
	if (VzInput){
		if (VzInput->Enum()!=PentaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumAsString(VzInput->Enum()));
		}
		VzInput->GetValuesPtr(&VzPtr,&dummy);
		for(i=0;i<numvertices;i++) vz[i]=VzPtr[i];
	}
	else{
		for(i=0;i<numvertices;i++) vz[i]=0.0;
	}

	/*Now Compute vel*/
	for(i=0;i<numvertices;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 3D, 
	 *so the pressure is just the pressure at the z elevation: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	inputs->GetParameterValues(&surface[0],&gauss[0][0],6,SurfaceEnum);
	
	for(i=0;i<numvertices;i++){
		pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);
	}
	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new TriaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));
}

/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticHutter {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticHutter(double* solution){
	
	int i;

	const int    numvertices=6;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;
	
	int          doflist[numdof];
	double       values[numdof];
	double       vx[numvertices];
	double       vy[numvertices];
	double       vz[numvertices];
	double       vel[numvertices];
	int          dummy;
	double       pressure[numvertices];
	double       surface[numvertices];
	double       rho_ice,g;
	double       xyz_list[numvertices][3];
	double       gauss[numvertices][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};
	
	Input*       VzInput=NULL;
	double*      VzPtr=NULL;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numvertices);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<numvertices;i++){
		vx[i]=values[i*numdofpervertex+0];
		vy[i]=values[i*numdofpervertex+1];
	}

	/*Get Vz*/
	VzInput=inputs->GetInput(VzEnum);
	if (VzInput){
		if (VzInput->Enum()!=PentaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumAsString(VzInput->Enum()));
		}
		VzInput->GetValuesPtr(&VzPtr,&dummy);
		for(i=0;i<numvertices;i++) vz[i]=VzPtr[i];
	}
	else{
		for(i=0;i<numvertices;i++) vz[i]=0.0;
	}

	/*Now Compute vel*/
	for(i=0;i<numvertices;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 3D, 
	 *so the pressure is just the pressure at the z elevation: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	inputs->GetParameterValues(&surface[0],&gauss[0][0],6,SurfaceEnum);
	
	for(i=0;i<numvertices;i++){
		pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);
	}
	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new TriaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));
}

/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticVert {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticVert(double* solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];
	double       vx[numvertices];
	double       vy[numvertices];
	double       vz[numvertices];
	double       vel[numvertices];
	int          dummy;
	double       pressure[numvertices];
	double       surface[numvertices];
	double       rho_ice,g;
	double       xyz_list[numvertices][3];
	double       gauss[numvertices][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	Input*       VxInput=NULL;
	double*      VxPtr=NULL;
	Input*       VyInput=NULL;
	double*      VyPtr=NULL;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numvertices);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vz in values, fill in vz array: */
	for(i=0;i<numvertices;i++){
		vz[i]=values[i*numdofpervertex+0];
	}

	/*Get Vx and Vy*/
	VxInput=inputs->GetInput(VxEnum);
	if (VxInput){
		if (VxInput->Enum()!=PentaVertexInputEnum) ISSMERROR("Cannot compute Vel as Vx is of type %s",EnumAsString(VxInput->Enum()));
		VxInput->GetValuesPtr(&VxPtr,&dummy);
		for(i=0;i<numvertices;i++) vx[i]=VxPtr[i];
	}
	else for(i=0;i<numvertices;i++) vx[i]=0.0;

	VyInput=inputs->GetInput(VyEnum);
	if (VyInput){
		if (VyInput->Enum()!=PentaVertexInputEnum) ISSMERROR("Cannot compute Vel as Vy is of type %s",EnumAsString(VyInput->Enum()));
		VyInput->GetValuesPtr(&VyPtr,&dummy);
		for(i=0;i<numvertices;i++) vy[i]=VyPtr[i];
	}
	else for(i=0;i<numvertices;i++) vy[i]=0.0;

	/*Now Compute vel*/
	for(i=0;i<numvertices;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 3D, 
	 *so the pressure is just the pressure at the z elevation: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	inputs->GetParameterValues(&surface[0],&gauss[0][0],6,SurfaceEnum);

	for(i=0;i<numvertices;i++){
		pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);
	}
	/*Now, we have to move the previous Vz inputs to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VzEnum,VzOldEnum);

	/*Add vz and vel as inputs to the penta element: */
	this->inputs->AddInput(new PentaVertexInput(VzEnum,vz));
	this->inputs->AddInput(new PentaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));
} /*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionDiagnosticStokes {{{1*/
void  Penta::InputUpdateFromSolutionDiagnosticStokes(double* solution){
	
	int i;

	const int    numvertices=6;
	const int    numdofpervertex=4;
	const int    numdof=numdofpervertex*numvertices;
	
	int          doflist[numdof];
	double       values[numdof];
	double       vx[numvertices];
	double       vy[numvertices];
	double       vz[numvertices];
	double       pressure[numvertices];
	int          dummy;
	double       stokesreconditioning;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<numvertices;i++){
		vx[i]=values[i*numdofpervertex+0];
		vy[i]=values[i*numdofpervertex+1];
		vz[i]=values[i*numdofpervertex+2];
		pressure[i]=values[i*numdofpervertex+3];
	}

	/*Recondition pressure: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	for(i=0;i<numvertices;i++){
		pressure[i]=pressure[i]*stokesreconditioning;
	}
	
	/*Now, we have to move the previous inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(VzEnum,VzOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new PentaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new PentaVertexInput(VzEnum,vz));
	this->inputs->AddInput(new PentaVertexInput(PressureEnum,pressure));
}

/*}}}*/
/*FUNCTION Penta::InputUpdateFromBedSlopeX{{{1*/
void  Penta::InputUpdateFromBedSlopeX(double* solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Add bedslopex as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(BedSlopeXEnum,values));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromBedSlopeY{{{1*/
void  Penta::InputUpdateFromBedSlopeY(double* solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Add bedslopey as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(BedSlopeYEnum,values));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSurfaceSlopeX{{{1*/
void  Penta::InputUpdateFromSurfaceSlopeX(double* solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Add surfaceslopex as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(SurfaceSlopeXEnum,values));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSurfaceSlopeY{{{1*/
void  Penta::InputUpdateFromSurfaceSlopeY(double* solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Add surfaceslopey as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(SurfaceSlopeYEnum,values));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionAdjointStokes {{{1*/
void  Penta::InputUpdateFromSolutionAdjointStokes(double* solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=4;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];
	double       lambdax[numvertices];
	double       lambday[numvertices];
	double       lambdaz[numvertices];
	double       lambdap[numvertices];

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<numvertices;i++){
		lambdax[i]=values[i*numdofpervertex+0];
		lambday[i]=values[i*numdofpervertex+1];
		lambdaz[i]=values[i*numdofpervertex+2];
		lambdap[i]=values[i*numdofpervertex+3];
	}

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(AdjointxEnum,lambdax));
	this->inputs->AddInput(new PentaVertexInput(AdjointyEnum,lambday));
	this->inputs->AddInput(new PentaVertexInput(AdjointzEnum,lambdaz));
	this->inputs->AddInput(new PentaVertexInput(AdjointpEnum,lambdap));

}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionAdjointHoriz {{{1*/
void  Penta::InputUpdateFromSolutionAdjointHoriz(double* solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];
	double       lambdax[numvertices];
	double       lambday[numvertices];

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<numvertices;i++){
		lambdax[i]=values[i*numdofpervertex+0];
		lambday[i]=values[i*numdofpervertex+1];
	}

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(AdjointxEnum,lambdax));
	this->inputs->AddInput(new PentaVertexInput(AdjointyEnum,lambday));

}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionPrognostic {{{1*/
void  Penta::InputUpdateFromSolutionPrognostic(double* solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Add thickness as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(ThicknessEnum,values));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionBalancedthickness {{{1*/
void  Penta::InputUpdateFromSolutionBalancedthickness(double* solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Add thickness as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(ThicknessEnum,values));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionBalancedvelocities {{{1*/
void  Penta::InputUpdateFromSolutionBalancedvelocities(double* solution){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionThermal {{{1*/
void  Penta::InputUpdateFromSolutionThermal(double* solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];
	double       B[numdof];
	double       B_average;

	bool         collapse;
	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
		//B[i]=Paterson(values[i]);
	}

	B_average=Paterson((values[0]+values[1]+values[2]+values[3]+values[4]+values[5])/6.0);
	for(i=0;i<numdof;i++){
		B[i]=B_average;
	}

	/*Add thickness as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(TemperatureEnum,values));

	/*Also update the rheology WARNING: ONLY FOR PATTYN and STOKES FOR NOW*/
	inputs->GetParameterValue(&collapse,CollapseEnum);
	if (!collapse) this->matice->inputs->AddInput(new PentaVertexInput(RheologyBEnum,B));

}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionMelting {{{1*/
void  Penta::InputUpdateFromSolutionMelting(double* solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*numvertices;

	int          doflist[numdof];
	double       values[numdof];

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Add thickness as inputs to the tria element: */
	this->inputs->AddInput(new PentaVertexInput(MeltingRateEnum,values));
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticHoriz{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticHoriz(Vec solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;
	double       gauss[numvertices][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	int          doflist[numdof];
	double       values[numdof];
	double       vx;
	double       vy;

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	for(i=0;i<numvertices;i++){

		/*Recover vx and vy*/
		inputs->GetParameterValue(&vx,&gauss[i][0],VxEnum);
		inputs->GetParameterValue(&vy,&gauss[i][0],VyEnum);
		values[i*numdofpervertex+0]=vx;
		values[i*numdofpervertex+1]=vy;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticHutter{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticHutter(Vec solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;
	double       gauss[numvertices][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	int          doflist[numdof];
	double       values[numdof];
	double       vx;
	double       vy;

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	for(i=0;i<numvertices;i++){

		/*Recover vx and vy*/
		inputs->GetParameterValue(&vx,&gauss[i][0],VxEnum);
		inputs->GetParameterValue(&vy,&gauss[i][0],VyEnum);
		values[i*numdofpervertex+0]=vx;
		values[i*numdofpervertex+1]=vy;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticVert{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticVert(Vec solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*numvertices;
	double       gauss[numvertices][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	int          doflist[numdof];
	double       values[numdof];
	double       vz;

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	for(i=0;i<numvertices;i++){

		/*Recover vz */
		inputs->GetParameterValue(&vz,&gauss[i][0],VxEnum);
		values[i]=vz;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsDiagnosticStokes{{{1*/
void  Penta::GetSolutionFromInputsDiagnosticStokes(Vec solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=4;
	const int    numdof=numdofpervertex*numvertices;
	double       gauss[numvertices][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	int          doflist[numdof];
	double       values[numdof];
	double       vx,vy,vz,p;

	int          dummy;
	double       stokesreconditioning;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Recondition pressure: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);

	/*Ok, we have vx vy vz and P in values, fill in vx vy vz P arrays: */
	/*P1 element only for now*/
	for(i=0;i<numvertices;i++){

		/*Recover vx and vy*/
		inputs->GetParameterValue(&vx,&gauss[i][0],VxEnum);
		inputs->GetParameterValue(&vy,&gauss[i][0],VyEnum);
		inputs->GetParameterValue(&vz,&gauss[i][0],VzEnum);
		inputs->GetParameterValue(&p ,&gauss[i][0],PressureEnum);
		values[i*numdofpervertex+0]=vx;
		values[i*numdofpervertex+1]=vy;
		values[i*numdofpervertex+2]=vz;
		values[i*numdofpervertex+3]=p/stokesreconditioning;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsThermal{{{1*/
void  Penta::GetSolutionFromInputsThermal(Vec solution){

	int i;

	const int    numvertices=6;
	const int    numdofpervertex=1;
	const int    numdof=numdofpervertex*numvertices;
	double       gauss[numvertices][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	int          doflist[numdof];
	double       values[numdof];
	double       vz;

	int          dummy;

	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	for(i=0;i<numvertices;i++){

		/*Recover vz */
		inputs->GetParameterValue(&vz,&gauss[i][0],TemperatureEnum);
		values[i]=vz;
	}

	/*Add value to global vector*/
	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);
}
/*}}}*/
/*FUNCTION Penta::IsInput{{{1*/
bool Penta::IsInput(int name){
	if (
				name==ThicknessEnum ||
				name==SurfaceEnum ||
				name==BedEnum ||
				name==SurfaceSlopeXEnum ||
				name==SurfaceSlopeYEnum ||
				name==MeltingRateEnum ||
				name==AccumulationRateEnum ||
				name==GeothermalFluxEnum ||
				name==PressureEnum ||
				name==VxEnum ||
				name==VyEnum ||
				name==VzEnum ||
				name==TemperatureEnum ||
				name==TemperatureAverageEnum ||
				name==RheologyBEnum ||
				name==RheologyNEnum ||
				name==FitEnum ||
				name==DragCoefficientEnum ||
				name==GradientEnum ||
				name==OldGradientEnum 
				) {
		return true;
	}
	else return false;
}
/*}}}*/
/*FUNCTION Penta::IsOnSurface{{{1*/
bool Penta::IsOnSurface(void){

	bool onsurface;
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);
	return onsurface;

}
/*}}}*/
/*FUNCTION Penta::GetUpperElement{{{1*/
Penta* Penta::GetUpperElement(void){

	Penta* upper_penta=NULL;
	upper_penta=(Penta*)neighbors[1]; //first one under, second one above
	return upper_penta;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixBalancedthickness {{{1*/

void  Penta::CreateKMatrixBalancedthickness(Mat Kgg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*flags: */
	bool onwater;
	bool onbed;

	/*recover some inputs: */
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg);
	delete tria;
	return;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixBalancedvelocities {{{1*/

void  Penta::CreateKMatrixBalancedvelocities(Mat Kgg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*flags: */
	bool onbed;
	bool onwater;

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg);
	delete tria;
	return;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticHoriz {{{1*/
void Penta::CreateKMatrixDiagnosticHoriz( Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=6;
	const int    numdof=2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;


	/* 3d gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights           =  NULL;
	double* vert_gauss_weights           =  NULL;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_coord[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	double  gauss_weight;

	/* 2d gaussian point: */
	int     num_gauss2d;
	double* first_gauss_area_coord2d  =  NULL;
	double* second_gauss_area_coord2d =  NULL;
	double* third_gauss_area_coord2d  =  NULL;
	double* gauss_weights2d=NULL;
	double  gauss_l1l2l3[3];

	/* material data: */
	double viscosity; //viscosity
	double oldviscosity; //viscosity
	double newviscosity; //viscosity

	/* strain rate: */
	double epsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double oldepsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/

	/* matrices: */
	double B[5][numdof];
	double Bprime[5][numdof];
	double L[2][numdof];
	double D[5][5]={0.0};            // material matrix, simple scalar matrix.
	double D_scalar;
	double DL[2][2]={0.0}; //for basal drag
	double DL_scalar;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]={0.0}; //local element stiffness matrix 

	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_drag_gaussian[numdof][numdof]; //stiffness matrix contribution from drag
	double Jdet;

	/*slope: */
	double  slope[2]={0.0};
	double  slope_magnitude;

	/*friction: */
	double  alpha2_list[3];
	double  alpha2;

	double MAXSLOPE=.06; // 6 %
	double MOUNTAINKEXPONENT=10;

	/*parameters: */
	double viscosity_overshoot;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater;
	bool collapse;
	bool onbed;
	bool shelf;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&collapse,CollapseEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&viscosity_overshoot,ViscosityOvershootEnum);

	/*If on water, skip stiffness: */
	if(onwater)return;

	/*Figure out if this pentaelem is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the stiffness matrix. */


	if ((collapse==1) && (onbed==0)){
		/*This element should be collapsed, but this element is not on the bedrock, therefore all its 
		 * dofs have already been frozen! Do nothing: */
		return;
	}
	else if ((collapse==1) && (onbed==1)){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 *and use its CreateKMatrix functionality to fill the global stiffness matrix: */
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreateKMatrix(Kgg);
		delete tria;
		return;
	}
	else{

		/*Implement standard penta element: */

		/* Get node coordinates and dof list: */
		GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
		GetDofList(&doflist[0],&numberofdofspernode);


		/*Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		  get tria gaussian points as well as segment gaussian points. For tria gaussian 
		  points, order of integration is 2, because we need to integrate the product tB*D*B' 
		  which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
		  points, same deal, which yields 3 gaussian points.*/

		order_area_gauss=5;
		num_vert_gauss=5;

		GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);

		/* Start  looping on the number of gaussian points: */
		for (ig1=0; ig1<num_area_gauss; ig1++){
			for (ig2=0; ig2<num_vert_gauss; ig2++){

				/*Pick up the gaussian point: */
				gauss_weight1=*(area_gauss_weights+ig1);
				gauss_weight2=*(vert_gauss_weights+ig2);
				gauss_weight=gauss_weight1*gauss_weight2;


				gauss_coord[0]=*(first_gauss_area_coord+ig1); 
				gauss_coord[1]=*(second_gauss_area_coord+ig1);
				gauss_coord[2]=*(third_gauss_area_coord+ig1);
				gauss_coord[3]=*(fourth_gauss_vert_coord+ig2);


				/*Get strain rate from velocity: */
				inputs->GetStrainRate3dPattyn(&epsilon[0],&xyz_list[0][0],gauss_coord,VxEnum,VyEnum);
				inputs->GetStrainRate3dPattyn(&oldepsilon[0],&xyz_list[0][0],gauss_coord,VxOldEnum,VyOldEnum);

				/*Get viscosity: */
				matice->GetViscosity3d(&viscosity, &epsilon[0]);
				matice->GetViscosity3d(&oldviscosity, &oldepsilon[0]);

				/*Get B and Bprime matrices: */
				GetB(&B[0][0], &xyz_list[0][0], gauss_coord);
				GetBPrime(&Bprime[0][0], &xyz_list[0][0], gauss_coord);

				/* Get Jacobian determinant: */
				GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);

				/*Build the D matrix: we plug the gaussian weight, the viscosity, and the jacobian determinant 
				  onto this scalar matrix, so that we win some computational time: */

				newviscosity=viscosity+viscosity_overshoot*(viscosity-oldviscosity);
				D_scalar=newviscosity*gauss_weight*Jdet;
				for (i=0;i<5;i++){
					D[i][i]=D_scalar;
				}

				/*  Do the triple product tB*D*Bprime: */
				TripleMultiply( &B[0][0],5,numdof,1,
							&D[0][0],5,5,0,
							&Bprime[0][0],5,numdof,0,
							&Ke_gg_gaussian[0][0],0);

				/* Add the Ke_gg_gaussian, and optionally Ke_gg_gaussian onto Ke_gg: */
				for( i=0; i<numdof; i++){
					for(j=0;j<numdof;j++){
						Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
					}
				}
			} //for (ig2=0; ig2<num_vert_gauss; ig2++)
		} //for (ig1=0; ig1<num_area_gauss; ig1++)


		/*Add Ke_gg to global matrix Kgg: */
		MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

		//Deal with 2d friction at the bedrock interface
		if((onbed && !shelf)){

			/*Build a tria element using the 3 grids of the base of the penta. Then use 
			 * the tria functionality to build a friction stiffness matrix on these 3
			 * grids: */

			tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
			tria->CreateKMatrixDiagnosticHorizFriction(Kgg);
			delete tria;
		}

	} 

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
	xfree((void**)&first_gauss_area_coord2d);
	xfree((void**)&second_gauss_area_coord2d);
	xfree((void**)&third_gauss_area_coord2d);
	xfree((void**)&gauss_weights2d);

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticHutter{{{1*/
void  Penta::CreateKMatrixDiagnosticHutter(Mat Kgg){

	/*Collapsed formulation: */
	Beam*  beam=NULL;
	int    i;

	/*flags: */
	bool onwater;

	/*recover some inputs: */
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Spawn 3 beam elements: */
	for(i=0;i<3;i++){
		beam=(Beam*)SpawnBeam(i,i+3); //[0 3], [1 4] and [2 5] are the four vertical edges of the Penta
		beam->CreateKMatrix(Kgg);
	}

	/*clean up*/
	delete beam;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticStokes {{{1*/
void Penta::CreateKMatrixDiagnosticStokes( Mat Kgg){

	int i,j;

	const int numgrids=6;
	const int DOFPERGRID=4;
	const int numdof=numgrids*DOFPERGRID;
	int doflist[numdof];
	int numberofdofspernode;

	const int numgrids2d=3;
	const int numdof2d=numgrids2d*DOFPERGRID;

	int   dofs[3]={0,1,2};

	double K_terms[numdof][numdof]={0.0};

	/*Material properties: */
	double         gravity,rho_ice,rho_water;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*Grid data: */
	double        xyz_list[numgrids][3];

	/*parameters: */
	double		   xyz_list_tria[numgrids2d][3];
	double		   surface_normal[3];
	double		   bed_normal[3];
	double         thickness;

	/*matrices: */
	double     Ke_temp[27][27]={0.0}; //for the six nodes and the bubble 
	double     Ke_reduced[numdof][numdof]; //for the six nodes only
	double     Ke_gaussian[27][27];
	double     Ke_drag_gaussian[numdof2d][numdof2d];
	double     B[8][27];
	double     B_prime[8][27];
	double     LStokes[14][numdof2d];
	double     LprimeStokes[14][numdof2d];
	double     Jdet;
	double     Jdet2d;
	double     D[8][8]={0.0};
	double     D_scalar;
	double     tBD[27][8];
	double     DLStokes[14][14]={0.0};
	double     tLDStokes[numdof2d][14];

	/* gaussian points: */
	int     num_area_gauss;
	int     igarea,igvert;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* vert_gauss_coord = NULL;
	double* area_gauss_weights  =  NULL;
	double* vert_gauss_weights  =  NULL;
	double  gaussgrids[numgrids][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	/* specific gaussian point: */
	double  gauss_weight,area_gauss_weight,vert_gauss_weight;
	double  gauss_coord[4];
	double  gauss_coord_tria[3];
	int area_order=5;
	int	num_vert_gauss=5;

	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity;
	double  alpha2_gauss;
	Friction* friction=NULL;

	/*parameters: */
	double stokesreconditioning;
	int analysis_type;

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;
	bool isstokes;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);
	inputs->GetParameterValue(&isstokes,IsStokesEnum);

	/*If on water or not Stokes, skip stiffness: */
	if(onwater || !isstokes) return;

	/*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();

	/*retrieve some parameters: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		get tria gaussian points as well as segment gaussian points. For tria gaussian 
		points, order of integration is 2, because we need to integrate the product tB*D*B' 
		which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
		points, same deal, which yields 3 gaussian points.*/

	area_order=5;
	num_vert_gauss=5;

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights,&vert_gauss_coord, &vert_gauss_weights, area_order, num_vert_gauss);

	/* Start  looping on the number of gaussian points: */
	for (igarea=0; igarea<num_area_gauss; igarea++){
		for (igvert=0; igvert<num_vert_gauss; igvert++){
			/*Pick up the gaussian point: */
			area_gauss_weight=*(area_gauss_weights+igarea);
			vert_gauss_weight=*(vert_gauss_weights+igvert);
			gauss_weight=area_gauss_weight*vert_gauss_weight;
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=*(vert_gauss_coord+igvert);

			/*Compute strain rate: */
			inputs->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss_coord,VxEnum,VyEnum,VzEnum);

			/*Get viscosity: */
			matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

			/*Get B and Bprime matrices: */
			GetBStokes(&B[0][0],&xyz_list[0][0],gauss_coord); 
			GetBprimeStokes(&B_prime[0][0],&xyz_list[0][0], gauss_coord); 

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],&gauss_coord[0]);

			/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
			 * onto this scalar matrix, so that we win some computational time: */
			D_scalar=gauss_weight*Jdet;
			for (i=0;i<6;i++){
				D[i][i]=D_scalar*viscosity;
			}
			for (i=6;i<8;i++){
				D[i][i]=-D_scalar*stokesreconditioning;
			}

			/*  Do the triple product tB*D*Bprime: */
			MatrixMultiply(&B[0][0],8,27,1,&D[0][0],8,8,0,&tBD[0][0],0);
			MatrixMultiply(&tBD[0][0],27,8,0,&B_prime[0][0],8,27,0,&Ke_gaussian[0][0],0);

			/*Add Ke_gaussian and Ke_gaussian to terms in pKe. Watch out for column orientation from matlab: */
			for(i=0;i<27;i++){
				for(j=0;j<27;j++){
					Ke_temp[i][j]+=Ke_gaussian[i][j];
				}
			}
		}
	}

	if((onbed==1) && (shelf==0)){

		/*build friction object, used later on: */
		friction=new Friction("3d",inputs,matpar,analysis_type);

		for(i=0;i<numgrids2d;i++){
			for(j=0;j<3;j++){
				xyz_list_tria[i][j]=xyz_list[i][j];
			}
		}

		xfree((void**)&first_gauss_area_coord); xfree((void**)&second_gauss_area_coord); xfree((void**)&third_gauss_area_coord); xfree((void**)&area_gauss_weights);
		GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, 2);

		/* Start looping on the number of gauss 2d (nodes on the bedrock) */
		for (igarea=0; igarea<num_area_gauss; igarea++){
			gauss_weight=*(area_gauss_weights+igarea);
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=-1;

			gauss_coord_tria[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord_tria[1]=*(second_gauss_area_coord+igarea);
			gauss_coord_tria[2]=*(third_gauss_area_coord+igarea);

			/*Get the Jacobian determinant */
			tria->GetJacobianDeterminant3d(&Jdet2d, &xyz_list_tria[0][0], gauss_coord_tria);

			/*Get L matrix if viscous basal drag present: */
			GetLStokes(&LStokes[0][0],  gauss_coord_tria);
			GetLprimeStokes(&LprimeStokes[0][0], &xyz_list[0][0], gauss_coord_tria, gauss_coord);

			/*Compute strain rate: */
			inputs->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss_coord,VxEnum,VyEnum,VzEnum);

			/*Get viscosity at last iteration: */
			matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

			/*Get normal vecyor to the bed */
			SurfaceNormal(&surface_normal[0],xyz_list_tria);

			bed_normal[0]=-surface_normal[0]; //Program is for surface, so the normal to the bed is the opposite of the result
			bed_normal[1]=-surface_normal[1];
			bed_normal[2]=-surface_normal[2];

			/*Calculate DL on gauss point */
			friction->GetAlpha2(&alpha2_gauss, gauss_coord,VxEnum,VyEnum,VzEnum);

			DLStokes[0][0]=alpha2_gauss*gauss_weight*Jdet2d;
			DLStokes[1][1]=alpha2_gauss*gauss_weight*Jdet2d;
			DLStokes[2][2]=-alpha2_gauss*gauss_weight*Jdet2d*bed_normal[0]*bed_normal[2];
			DLStokes[3][3]=-alpha2_gauss*gauss_weight*Jdet2d*bed_normal[1]*bed_normal[2];
			DLStokes[4][4]=-alpha2_gauss*gauss_weight*Jdet2d*bed_normal[0]*bed_normal[2];
			DLStokes[5][5]=-alpha2_gauss*gauss_weight*Jdet2d*bed_normal[1]*bed_normal[2];
			DLStokes[6][6]=-viscosity*gauss_weight*Jdet2d*bed_normal[0];
			DLStokes[7][7]=-viscosity*gauss_weight*Jdet2d*bed_normal[1];
			DLStokes[8][8]=-viscosity*gauss_weight*Jdet2d*bed_normal[2];
			DLStokes[9][8]=-viscosity*gauss_weight*Jdet2d*bed_normal[0]/2.0;
			DLStokes[10][10]=-viscosity*gauss_weight*Jdet2d*bed_normal[1]/2.0;
			DLStokes[11][11]=stokesreconditioning*gauss_weight*Jdet2d*bed_normal[0];
			DLStokes[12][12]=stokesreconditioning*gauss_weight*Jdet2d*bed_normal[1];
			DLStokes[13][13]=stokesreconditioning*gauss_weight*Jdet2d*bed_normal[2];

			/*  Do the triple product tL*D*L: */
			MatrixMultiply(&LStokes[0][0],14,numdof2d,1,&DLStokes[0][0],14,14,0,&tLDStokes[0][0],0);
			MatrixMultiply(&tLDStokes[0][0],numdof2d,14,0,&LprimeStokes[0][0],14,numdof2d,0,&Ke_drag_gaussian[0][0],0);

			for(i=0;i<numdof2d;i++){
				for(j=0;j<numdof2d;j++){
					Ke_temp[i][j]+=Ke_drag_gaussian[i][j];
				}
			}
		}
	
		/*Free ressources:*/
		delete friction;

	} //if ( (onbed==1) && (shelf==0))

	/*Reduce the matrix */
	ReduceMatrixStokes(&Ke_reduced[0][0], &Ke_temp[0][0]);

	for(i=0;i<numdof;i++){
		for(j=0;j<numdof;j++){
			K_terms[i][j]+=Ke_reduced[i][j];
		}
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)K_terms,ADD_VALUES);


	/*Free ressources:*/
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_coord);
	xfree((void**)&vert_gauss_weights);

	return;
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixDiagnosticVert {{{1*/
void Penta::CreateKMatrixDiagnosticVert( Mat Kgg){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=6;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* 3d gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights           =  NULL;
	double* vert_gauss_weights           =  NULL;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_coord[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	double  gauss_weight;

	/* matrices: */
	double  Ke_gg[numdof][numdof]={0.0};
	double  Ke_gg_gaussian[numdof][numdof];
	double  Jdet;
	double  B[NDOF1][numgrids];
	double  Bprime[NDOF1][numgrids];
	double  DL_scalar;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater; 
	bool onsurface;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	/*If on water, skip stiffness: */
	if(onwater)return;

	/*If this element  is on the surface, we have a dynamic boundary condition that applies, as a stiffness 
	 * matrix: */
	if(onsurface){
		tria=(Tria*)SpawnTria(3,4,5); //nodes 3,4 and 5 are on the surface
		tria->CreateKMatrixDiagnosticSurfaceVert(Kgg);
		delete tria;
	}

	/*Now, onto the formulation for the vertical velocity: */

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/*Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
	  get tria gaussian points as well as segment gaussian points. For tria gaussian 
	  points, order of integration is 2, because we need to integrate the product tB*D*B' 
	  which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
	  points, same deal, which yields 3 gaussian points.*/

	order_area_gauss=2;
	num_vert_gauss=2;

	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);

	/* Start  looping on the number of gaussian points: */
	for (ig1=0; ig1<num_area_gauss; ig1++){
		for (ig2=0; ig2<num_vert_gauss; ig2++){

			/*Pick up the gaussian point: */
			gauss_weight1=*(area_gauss_weights+ig1);
			gauss_weight2=*(vert_gauss_weights+ig2);
			gauss_weight=gauss_weight1*gauss_weight2;

			gauss_coord[0]=*(first_gauss_area_coord+ig1); 
			gauss_coord[1]=*(second_gauss_area_coord+ig1);
			gauss_coord[2]=*(third_gauss_area_coord+ig1);
			gauss_coord[3]=*(fourth_gauss_vert_coord+ig2);

			/*Get B and Bprime matrices: */
			GetB_vert(&B[0][0], &xyz_list[0][0], gauss_coord);
			GetBPrime_vert(&Bprime[0][0], &xyz_list[0][0], gauss_coord);

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);
			DL_scalar=gauss_weight*Jdet;

			/*  Do the triple product tB*D*Bprime: */
			TripleMultiply( &B[0][0],1,numgrids,1,
						&DL_scalar,1,1,0,
						&Bprime[0][0],1,numgrids,0,
						&Ke_gg_gaussian[0][0],0);

			/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
			for( i=0; i<numdof; i++){
				for(j=0;j<numdof;j++){
					Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
				}
			}	
		} //for (ig2=0; ig2<num_vert_gauss; ig2++)
	} //for (ig1=0; ig1<num_area_gauss; ig1++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixMelting {{{1*/
void  Penta::CreateKMatrixMelting(Mat Kgg){

	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater)return;

	if (!onbed){
		return;
	}
	else{

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreateKMatrixMelting(Kgg);
		delete tria;
		return;
	}
}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixPrognostic {{{1*/

void  Penta::CreateKMatrixPrognostic(Mat Kgg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg);
	delete tria;
	return;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixSlope {{{1*/

void  Penta::CreateKMatrixSlope(Mat Kgg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);


	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg);
	delete tria;
	return;

}
/*}}}*/
/*FUNCTION Penta::CreateKMatrixThermal {{{1*/
void  Penta::CreateKMatrixThermal(Mat Kgg){

	/* local declarations */
	int i,j;
	int found=0;

	/* node data: */
	const int    numgrids=6;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double xyz_list[numgrids][3];
	int    doflist[numdof];
	int    numberofdofspernode;

	/* gaussian points: */
	int     num_area_gauss,igarea,igvert;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* vert_gauss_coord = NULL;
	double* area_gauss_weights  =  NULL;
	double* vert_gauss_weights  =  NULL;
	double  gauss_weight,area_gauss_weight,vert_gauss_weight;
	double  gauss_coord[4];
	double  gauss_l1l2l3[3];

	int area_order=5;
	int num_vert_gauss=5;

	int     dofs[3]={0,1,2};
	double  K[2][2]={0.0};

	double  u,v,w;

	/*matrices: */
	double     K_terms[numdof][numdof]={0.0};
	double     Ke_gaussian_conduct[numdof][numdof];
	double     Ke_gaussian_advec[numdof][numdof];
	double     Ke_gaussian_artdiff[numdof][numdof];
	double     Ke_gaussian_transient[numdof][numdof];
	double     B[3][numdof];
	double     Bprime[3][numdof];
	double     B_conduct[3][numdof];
	double     B_advec[3][numdof];
	double     B_artdiff[2][numdof];
	double     Bprime_advec[3][numdof];
	double     L[numdof];
	double     D_scalar;
	double     D[3][3];
	double     l1l2l3[3];
	double     tl1l2l3D[3];
	double     tBD[3][numdof];
	double     tBD_conduct[3][numdof];
	double     tBD_advec[3][numdof];
	double     tBD_artdiff[3][numdof];
	double     tLD[numdof];

	double     Jdet;

	/*Material properties: */
	double     gravity,rho_ice,rho_water;
	double     heatcapacity,thermalconductivity;
	double     mixed_layer_capacity,thermal_exchange_velocity;

	/*parameters: */
	double dt,epsvel;
	bool   artdiff;

	/*Collapsed formulation: */
	Tria*  tria=NULL;


	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*If on water, skip: */
	if(onwater)return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	// /*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	heatcapacity=matpar->GetHeatCapacity();
	thermalconductivity=matpar->GetThermalConductivity();

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);
	this->parameters->FindParam(&artdiff,ArtDiffEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);

	/* Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		get tria gaussian points as well as segment gaussian points. For tria gaussian 
		points, order of integration is 2, because we need to integrate the product tB*D*B' 
		which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
		points, same deal, which yields 3 gaussian points.: */

	/*Get gaussian points: */
	area_order=2;
	num_vert_gauss=2;

	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights,&vert_gauss_coord, &vert_gauss_weights, area_order, num_vert_gauss);

	/* Start  looping on the number of gaussian points: */
	for (igarea=0; igarea<num_area_gauss; igarea++){
		for (igvert=0; igvert<num_vert_gauss; igvert++){
			/*Pick up the gaussian point: */
			area_gauss_weight=*(area_gauss_weights+igarea);
			vert_gauss_weight=*(vert_gauss_weights+igvert);
			gauss_weight=area_gauss_weight*vert_gauss_weight;
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=*(vert_gauss_coord+igvert);

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);

			/*Conduction: */

			/*Get B_conduct matrix: */
			GetB_conduct(&B_conduct[0][0],&xyz_list[0][0],gauss_coord); 

			/*Build D: */
			D_scalar=gauss_weight*Jdet*(thermalconductivity/(rho_ice*heatcapacity));

			if(dt){
				D_scalar=D_scalar*dt;
			}

			D[0][0]=D_scalar; D[0][1]=0; D[0][2]=0;
			D[1][0]=0; D[1][1]=D_scalar; D[1][2]=0;
			D[2][0]=0; D[2][1]=0; D[2][2]=D_scalar;

			/*  Do the triple product B'*D*B: */
			MatrixMultiply(&B_conduct[0][0],3,numdof,1,&D[0][0],3,3,0,&tBD_conduct[0][0],0);
			MatrixMultiply(&tBD_conduct[0][0],numdof,3,0,&B_conduct[0][0],3,numdof,0,&Ke_gaussian_conduct[0][0],0);

			/*Advection: */

			/*Get B_advec and Bprime_advec matrices: */
			GetB_advec(&B_advec[0][0],&xyz_list[0][0],gauss_coord); 
			GetBprime_advec(&Bprime_advec[0][0],&xyz_list[0][0],gauss_coord); 

			//Build the D matrix
			inputs->GetParameterValue(&u, gauss_coord,VxEnum);
			inputs->GetParameterValue(&v, gauss_coord,VyEnum);
			inputs->GetParameterValue(&w, gauss_coord,VzEnum);

			D_scalar=gauss_weight*Jdet;

			if(dt){
				D_scalar=D_scalar*dt;
			}

			D[0][0]=D_scalar*u;D[0][1]=0;         D[0][2]=0;
			D[1][0]=0;         D[1][1]=D_scalar*v;D[1][2]=0;
			D[2][0]=0;         D[2][1]=0;         D[2][2]=D_scalar*w;

			/*  Do the triple product B'*D*Bprime: */
			MatrixMultiply(&B_advec[0][0],3,numdof,1,&D[0][0],3,3,0,&tBD_advec[0][0],0);
			MatrixMultiply(&tBD_advec[0][0],numdof,3,0,&Bprime_advec[0][0],3,numdof,0,&Ke_gaussian_advec[0][0],0);

			/*Transient: */
			if(dt){
				GetNodalFunctions(&L[0], gauss_coord);
				D_scalar=gauss_weight*Jdet;
				D_scalar=D_scalar;

				/*  Do the triple product L'*D*L: */
				MatrixMultiply(&L[0],numdof,1,0,&D_scalar,1,1,0,&tLD[0],0);
				MatrixMultiply(&tLD[0],numdof,1,0,&L[0],1,numdof,0,&Ke_gaussian_transient[0][0],0);
			}
			else{
				for(i=0;i<numdof;i++){
					for(j=0;j<numdof;j++){
						Ke_gaussian_transient[i][j]=0;
					}
				}
			}

			/*Artifficial diffusivity*/
			if(artdiff){
				/*Build K: */
				D_scalar=gauss_weight*Jdet/(pow(u,2)+pow(v,2)+epsvel);
				if(dt){
					D_scalar=D_scalar*dt;
				}
				K[0][0]=D_scalar*pow(u,2);       K[0][1]=D_scalar*fabs(u)*fabs(v);
				K[1][0]=D_scalar*fabs(u)*fabs(v);K[1][1]=D_scalar*pow(v,2);

				/*Get B_artdiff: */
				GetB_artdiff(&B_artdiff[0][0],&xyz_list[0][0],gauss_coord); 

				/*  Do the triple product B'*K*B: */
				MatrixMultiply(&B_artdiff[0][0],2,numdof,1,&K[0][0],2,2,0,&tBD_artdiff[0][0],0);
				MatrixMultiply(&tBD_artdiff[0][0],numdof,2,0,&B_artdiff[0][0],2,numdof,0,&Ke_gaussian_artdiff[0][0],0);
			}
			else{
				for(i=0;i<numdof;i++){
					for(j=0;j<numdof;j++){
						Ke_gaussian_artdiff[i][j]=0;
					}
				}
			}

			/*Add Ke_gaussian to pKe: */
			for(i=0;i<numdof;i++){
				for(j=0;j<numdof;j++){
					K_terms[i][j]+=Ke_gaussian_conduct[i][j]+Ke_gaussian_advec[i][j]+Ke_gaussian_transient[i][j]+Ke_gaussian_artdiff[i][j];
				}
			}
		}
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)K_terms,ADD_VALUES);

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
	xfree((void**)&vert_gauss_coord);

	//Ice/ocean heat exchange flux on ice shelf base 
	if(onbed && shelf){

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreateKMatrixThermal(Kgg);
		delete tria;
	}
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorBalancedthickness {{{1*/
void Penta::CreatePVectorBalancedthickness( Vec pg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*flags: */
	bool onbed;
	bool onwater;

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg);
	delete tria;
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorBalancedvelocities {{{1*/
void Penta::CreatePVectorBalancedvelocities( Vec pg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*flags: */
	bool onbed;
	bool onwater;

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg);
	delete tria;
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticHoriz {{{1*/
void Penta::CreatePVectorDiagnosticHoriz( Vec pg){

	int i,j;

	/* node data: */
	const int    numgrids=6;
	const int    NDOF2=2;
	const int    numdof=NDOF2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* parameters: */
	double  slope[3]; //do not put 2! this goes into GetParameterDerivativeValue, which addresses slope[3] also!
	double  driving_stress_baseline;
	double  thickness;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights      =  NULL;
	double* vert_gauss_weights      =  NULL;
	double  gauss_coord[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_weight;

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l6[6];

	/*element vector at the gaussian points: */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];

	/*Spawning: */
	Tria* tria=NULL;

	/*inputs: */
	bool onwater;
	bool collapse;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&collapse,CollapseEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip load: */
	if(onwater)return;

	/*Figure out if this pentaelem is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the load vector. */

	if ((collapse==1) && (onbed==0)){
		/*This element should be collapsed, but this element is not on the bedrock, therefore all its 
		 * dofs have already been frozen! Do nothing: */
		return;
	}
	else if ((collapse==1) && (onbed==1)){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 *and use its CreatePVector functionality to return an elementary load vector: */
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreatePVector(pg);
		delete tria;
		return;
	}
	else{

		/*Implement standard penta element: */

		/* Get node coordinates and dof list: */
		GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
		GetDofList(&doflist[0],&numberofdofspernode);

		/*Get gaussian points and weights :*/
		order_area_gauss=2;
		num_vert_gauss=3;

		GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);

		/* Start  looping on the number of gaussian points: */
		for (ig1=0; ig1<num_area_gauss; ig1++){
			for (ig2=0; ig2<num_vert_gauss; ig2++){

				/*Pick up the gaussian point: */
				gauss_weight1=*(area_gauss_weights+ig1);
				gauss_weight2=*(vert_gauss_weights+ig2);
				gauss_weight=gauss_weight1*gauss_weight2;

				gauss_coord[0]=*(first_gauss_area_coord+ig1); 
				gauss_coord[1]=*(second_gauss_area_coord+ig1);
				gauss_coord[2]=*(third_gauss_area_coord+ig1);
				gauss_coord[3]=*(fourth_gauss_vert_coord+ig2);

				/*Compute thickness at gaussian point: */
				inputs->GetParameterValue(&thickness, gauss_coord,ThicknessEnum);

				/*Compute slope at gaussian point: */
				inputs->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],gauss_coord,SurfaceEnum);

				/* Get Jacobian determinant: */
				GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);

				/*Get nodal functions: */
				GetNodalFunctions(l1l6, gauss_coord);

				/*Compute driving stress: */
				driving_stress_baseline=matpar->GetRhoIce()*matpar->GetG();

				/*Build pe_g_gaussian vector: */
				for (i=0;i<numgrids;i++){
					for (j=0;j<NDOF2;j++){
						pe_g_gaussian[i*NDOF2+j]=-driving_stress_baseline*slope[j]*Jdet*gauss_weight*l1l6[i];
					}
				}

				/*Add pe_g_gaussian vector to pe_g: */
				for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

			} //for (ig2=0; ig2<num_vert_gauss; ig2++)
		} //for (ig1=0; ig1<num_area_gauss; ig1++)

	} //else if ((collapse==1) && (onbed==1))

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);

}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticHutter{{{1*/
void  Penta::CreatePVectorDiagnosticHutter(Vec pg){

	/*Collapsed formulation: */
	Beam*  beam=NULL;
	int    i;

	/*flags: */
	bool onwater;

	/*recover some inputs: */
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Spawn 3 beam elements: */
	for(i=0;i<3;i++){
		beam=(Beam*)SpawnBeam(i,i+3); //[0 3], [1 4] and [2 5] are the four vertical edges of the Penta
		beam->CreatePVector(pg);
	}

	/*clean up*/
	delete beam;

}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticStokes {{{1*/
void Penta::CreatePVectorDiagnosticStokes( Vec pg){

	/*indexing: */
	int i,j;

	const int numgrids=6;
	const int DOFPERGRID=4;
	const int numdof=numgrids*DOFPERGRID;
	const int numgrids2d=3;
	int numdof2d=numgrids2d*DOFPERGRID;
	int doflist[numdof];
	int numberofdofspernode;

	/*Material properties: */
	double         gravity,rho_ice,rho_water;

	/*parameters: */
	double		   xyz_list_tria[numgrids2d][3];
	double         xyz_list[numgrids][3];
	double		   surface_normal[3];
	double		   bed_normal[3];
	double         bed;

	/* gaussian points: */
	int     num_area_gauss;
	int     igarea,igvert;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* vert_gauss_coord = NULL;
	double* area_gauss_weights  =  NULL;
	double* vert_gauss_weights  =  NULL;

	/* specific gaussian point: */
	double  gauss_weight,area_gauss_weight,vert_gauss_weight;
	double  gauss_coord[4];
	double  gauss_coord_tria[3];

	int     area_order=5;
	int	  num_vert_gauss=5;

	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity;
	double  water_pressure;
	int     dofs[3]={0,1,2};

	/*matrices: */
	double     Pe_temp[27]={0.0}; //for the six nodes and the bubble 
	double     Pe_gaussian[27]={0.0}; //for the six nodes and the bubble 
	double     Ke_temp[27][3]={0.0}; //for the six nodes and the bubble 
	double     Pe_reduced[numdof]; //for the six nodes only
	double     Ke_gaussian[27][3];
	double     L[3]; //for the three nodes of the bed
	double     l1l7[7]; //for the six nodes and the bubble 
	double     B[8][27];
	double     B_prime[8][27];
	double     B_prime_bubble[8][3];
	double     Jdet;
	double     Jdet2d;
	double     D[8][8]={0.0};
	double     D_scalar;
	double     tBD[27][8];
	double     P_terms[numdof]={0.0};

	Tria*            tria=NULL;

	/*parameters: */
	double stokesreconditioning;

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;
	bool isstokes;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);
	inputs->GetParameterValue(&isstokes,IsStokesEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);

	/*If on water or not Stokes, skip load: */
	if(onwater || !isstokes) return;

	/*If on water, skip load: */
	if(onwater)return;

	/*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		get tria gaussian points as well as segment gaussian points. For tria gaussian 
		points, order of integration is 2, because we need to integrate the product tB*D*B' 
		which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
		points, same deal, which yields 3 gaussian points.*/

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights,&vert_gauss_coord, &vert_gauss_weights, area_order, num_vert_gauss);

	/* Start  looping on the number of gaussian points: */
	for (igarea=0; igarea<num_area_gauss; igarea++){
		for (igvert=0; igvert<num_vert_gauss; igvert++){
			/*Pick up the gaussian point: */
			area_gauss_weight=*(area_gauss_weights+igarea);
			vert_gauss_weight=*(vert_gauss_weights+igvert);
			gauss_weight=area_gauss_weight*vert_gauss_weight;
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=*(vert_gauss_coord+igvert);

			/*Compute strain rate and viscosity: */
			inputs->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss_coord,VxEnum,VyEnum,VzEnum);
			matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);

			/* Get nodal functions */
			GetNodalFunctionsStokes(&l1l7[0], gauss_coord);

			/* Build gaussian vector */
			for(i=0;i<numgrids+1;i++){
				Pe_gaussian[i*DOFPERGRID+2]=-rho_ice*gravity*Jdet*gauss_weight*l1l7[i];
			}

			/*Add Pe_gaussian to terms in Pe_temp. Watch out for column orientation from matlab: */
			for(i=0;i<27;i++){
				Pe_temp[i]+=Pe_gaussian[i];
			}

			/*Get B and Bprime matrices: */
			GetBStokes(&B[0][0],&xyz_list[0][0],gauss_coord); 
			GetBprimeStokes(&B_prime[0][0],&xyz_list[0][0], gauss_coord); 

			/*Get bubble part of Bprime */
			for(i=0;i<8;i++){
				for(j=0;j<3;j++){
					B_prime_bubble[i][j]=B_prime[i][j+24];
				}
			}

			/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
			 * onto this scalar matrix, so that we win some computational time: */
			D_scalar=gauss_weight*Jdet;
			for (i=0;i<6;i++){
				D[i][i]=D_scalar*viscosity;
			}
			for (i=6;i<8;i++){
				D[i][i]=-D_scalar*stokesreconditioning;
			}

			/*  Do the triple product tB*D*Bprime: */
			MatrixMultiply(&B[0][0],8,27,1,&D[0][0],8,8,0,&tBD[0][0],0);
			MatrixMultiply(&tBD[0][0],27,8,0,&B_prime_bubble[0][0],8,3,0,&Ke_gaussian[0][0],0);

			/*Add Ke_gaussian and Ke_gaussian to terms in pKe. Watch out for column orientation from matlab: */
			for(i=0;i<27;i++){
				for(j=0;j<3;j++){
					Ke_temp[i][j]+=Ke_gaussian[i][j];
				}
			}
		}
	}

	/*Deal with 2d friction at the bedrock interface: */
	if ( (onbed==1) && (shelf==1)){

		for(i=0;i<numgrids2d;i++){
			for(j=0;j<3;j++){
				xyz_list_tria[i][j]=xyz_list[i][j];
			}
		}

		xfree((void**)&first_gauss_area_coord); xfree((void**)&second_gauss_area_coord); xfree((void**)&third_gauss_area_coord); xfree((void**)&area_gauss_weights);
		GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, 2);

		/* Start looping on the number of gauss 2d (nodes on the bedrock) */
		for (igarea=0; igarea<num_area_gauss; igarea++){
			gauss_weight=*(area_gauss_weights+igarea);
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=-1;

			gauss_coord_tria[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord_tria[1]=*(second_gauss_area_coord+igarea);
			gauss_coord_tria[2]=*(third_gauss_area_coord+igarea);

			/*Get the Jacobian determinant */
			tria->GetJacobianDeterminant3d(&Jdet2d, &xyz_list_tria[0][0], gauss_coord_tria);

			/* Get bed at gaussian point */
			inputs->GetParameterValue(&bed, gauss_coord,BedEnum);

			/*Get L matrix : */
			tria->GetL(&L[0], &xyz_list[0][0], gauss_coord_tria,1);

			/*Get water_pressure at gaussian point */
			water_pressure=gravity*rho_water*bed;

			/*Get normal vecyor to the bed */
			SurfaceNormal(&surface_normal[0],xyz_list_tria);

			bed_normal[0]=-surface_normal[0]; //Program is for surface, so the normal to the bed is the opposite of the result
			bed_normal[1]=-surface_normal[1];
			bed_normal[2]=-surface_normal[2];

			for(i=0;i<numgrids2d;i++){
				for(j=0;j<3;j++){
					Pe_temp[i*DOFPERGRID+j]+=water_pressure*gauss_weight*Jdet2d*L[i]*bed_normal[j];
				}
			}
		}
	} //if ( (onbed==1) && (shelf==1))

	/*Reduce the matrix */
	ReduceVectorStokes(&Pe_reduced[0], &Ke_temp[0][0], &Pe_temp[0]);

	for(i=0;i<numdof;i++){
		P_terms[i]+=Pe_reduced[i];
	}

	/*Add P_terms to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	/*Free ressources:*/
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_coord);
	xfree((void**)&vert_gauss_weights);

}
/*}}}*/
/*FUNCTION Penta::CreatePVectorDiagnosticVert {{{1*/
void  Penta::CreatePVectorDiagnosticVert( Vec pg){

	int i;

	/* node data: */
	const int    numgrids=6;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights      =  NULL;
	double* vert_gauss_weights      =  NULL;
	double  gauss_coord[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_weight;

	/* Jacobian: */
	double Jdet;

	/*element vector at the gaussian points: */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];
	double l1l6[6];

	/*Spawning: */
	Tria* tria=NULL;

	/*input parameters for structural analysis (diagnostic): */
	double du[3];
	double dv[3];
	double dudx,dvdy;
	int     dofs1[1]={0};
	int     dofs2[1]={1};

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*If we are on the bedrock, spawn a tria on the bedrock, and use it to build the 
	 *diagnostic base vertical stifness: */
	if(onbed){
		tria=(Tria*)SpawnTria(0,1,2); //nodes 0, 1 and 2 are on the bedrock
		tria->CreatePVectorDiagnosticBaseVert(pg);
		delete tria;
	}

	/*Now, handle the standard penta element: */
	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/*Get gaussian points and weights :*/
	order_area_gauss=2;
	num_vert_gauss=2;

	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);

	/* Start  looping on the number of gaussian points: */
	for (ig1=0; ig1<num_area_gauss; ig1++){
		for (ig2=0; ig2<num_vert_gauss; ig2++){

			/*Pick up the gaussian point: */
			gauss_weight1=*(area_gauss_weights+ig1);
			gauss_weight2=*(vert_gauss_weights+ig2);
			gauss_weight=gauss_weight1*gauss_weight2;

			gauss_coord[0]=*(first_gauss_area_coord+ig1); 
			gauss_coord[1]=*(second_gauss_area_coord+ig1);
			gauss_coord[2]=*(third_gauss_area_coord+ig1);
			gauss_coord[3]=*(fourth_gauss_vert_coord+ig2);

			/*Get velocity derivative, with respect to x and y: */

			inputs->GetParameterDerivativeValue(&du[0],&xyz_list[0][0],gauss_coord,VxEnum);
			inputs->GetParameterDerivativeValue(&dv[0],&xyz_list[0][0],gauss_coord,VyEnum);
			dudx=du[0];
			dvdy=dv[1];

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);

			/*Get nodal functions: */
			GetNodalFunctions(l1l6, gauss_coord);

			/*Build pe_g_gaussian vector: */
			for (i=0;i<numgrids;i++){
				pe_g_gaussian[i]=(dudx+dvdy)*Jdet*gauss_weight*l1l6[i];
			}

			/*Add pe_g_gaussian vector to pe_g: */
			for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

		} //for (ig2=0; ig2<num_vert_gauss; ig2++)
	} //for (ig1=0; ig1<num_area_gauss; ig1++)

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorMelting {{{1*/
void Penta::CreatePVectorMelting( Vec pg){
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorPrognostic {{{1*/

void Penta::CreatePVectorPrognostic( Vec pg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	
	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg);
	delete tria;
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorSlope {{{1*/

void Penta::CreatePVectorSlope( Vec pg){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*inputs: */
	bool onwater;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg);
	delete tria;
	return;
}
/*}}}*/
/*FUNCTION Penta::CreatePVectorThermal {{{1*/
void Penta::CreatePVectorThermal( Vec pg){

	/*indexing: */
	int i,j;
	int found=0;

	const int  numgrids=6;
	const int  NDOF1=1;
	const int  numdof=numgrids*NDOF1;
	int        doflist[numdof];
	int        numberofdofspernode;

	/*Grid data: */
	double        xyz_list[numgrids][3];

	/* gaussian points: */
	int     num_area_gauss,igarea,igvert;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* vert_gauss_coord = NULL;
	double* area_gauss_weights  =  NULL;
	double* vert_gauss_weights  =  NULL;
	double  gauss_weight,area_gauss_weight,vert_gauss_weight;
	double  gauss_coord[4];
	int     area_order=2;
	int	  num_vert_gauss=3;

	double temperature_list[numgrids];
	double temperature;

	/*Material properties: */
	double gravity,rho_ice,rho_water;
	double mixed_layer_capacity,heatcapacity;
	double beta,meltingpoint,thermal_exchange_velocity;

	/* element parameters: */
	int    friction_type;

	int    dofs[3]={0,1,2};
	int    dofs1[1]={0};

	/*matrices: */
	double P_terms[numdof]={0.0};
	double L[numdof];
	double l1l2l3[3];
	double alpha2_list[3];
	double basalfriction_list[3]={0.0};
	double basalfriction;
	double epsilon[6];
	double epsilon_sqr[3][3];
	double epsilon_matrix[3][3];

	double Jdet;
	double viscosity;
	double epsilon_eff;
	double phi;
	double t_pmp;
	double scalar;
	double scalar_def;
	double scalar_ocean;
	double scalar_transient;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*parameters: */
	double dt;

	/*inputs: */
	bool onwater;
	bool onbed;
	bool shelf;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/*If on water, skip: */
	if(onwater)return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	heatcapacity=matpar->GetHeatCapacity();
	beta=matpar->GetBeta();
	meltingpoint=matpar->GetMeltingPoint();

	/* Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		get tria gaussian points as well as segment gaussian points. For tria gaussian 
		points, order of integration is 2, because we need to integrate the product tB*D*B' 
		which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
		points, same deal, which yields 3 gaussian points.: */

	/*Get gaussian points: */
	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights,&vert_gauss_coord, &vert_gauss_weights, area_order, num_vert_gauss);

	/* Start  looping on the number of gaussian points: */
	for (igarea=0; igarea<num_area_gauss; igarea++){
		for (igvert=0; igvert<num_vert_gauss; igvert++){
			/*Pick up the gaussian point: */
			area_gauss_weight=*(area_gauss_weights+igarea);
			vert_gauss_weight=*(vert_gauss_weights+igvert);
			gauss_weight=area_gauss_weight*vert_gauss_weight;
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=*(vert_gauss_coord+igvert);

			/*Compute strain rate and viscosity: */
			inputs->GetStrainRate3d(&epsilon[0],&xyz_list[0][0],gauss_coord,VxEnum,VyEnum,VzEnum);
			matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);

			/* Get nodal functions */
			GetNodalFunctions(&L[0], gauss_coord);

			/*Build deformational heating: */
			GetPhi(&phi, &epsilon[0], viscosity);

			/*Build pe_gaussian */
			scalar_def=phi/(rho_ice*heatcapacity)*Jdet*gauss_weight;
			if(dt){
				scalar_def=scalar_def*dt;
			}

			for(i=0;i<numgrids;i++){
				P_terms[i]+=scalar_def*L[i];
			}

			/* Build transient now */
			if(dt){
				inputs->GetParameterValue(&temperature, gauss_coord,TemperatureEnum);
				scalar_transient=temperature*Jdet*gauss_weight;
				for(i=0;i<numgrids;i++){
					P_terms[i]+=scalar_transient*L[i];
				}
			}
		}
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	/* Ice/ocean heat exchange flux on ice shelf base */
	if(onbed && shelf){

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreatePVectorThermalShelf(pg);
		delete tria;
	}

	/* Geothermal flux on ice sheet base and basal friction */
	if(onbed && !shelf){

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreatePVectorThermalSheet(pg);
		delete tria;
	}
	extern int my_rank;

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&vert_gauss_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);

}
/*}}}*/
/*FUNCTION Penta::VecExtrude {{{1*/
void  Penta::VecExtrude(Vec vector,double* vector_serial,int iscollapsed){

	/* node data: */
	int   i;
	Node* node=NULL;
	int   extrude=0;

	/*inputs: */
	bool collapse;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&collapse,CollapseEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*Figure out if we should extrude for this element: */
	if (iscollapsed){
		/*From higher level, we are told to extrude only elements that have the collapse flag on: */
		if (collapse)extrude=1;
		else extrude=0;
	}
	else{
		/*From higher level, we are told to extrude all elements: */
		extrude=1;
	}

	/*Now, extrusion starts from the bed on, so double check this element is on 
	 * the bedrock: */
	if(onbed==0)extrude=0;

	/*Go on and extrude vector: */
	if (extrude){

		/* node data: */
		int          dof1;
		double       vectorel;

		/*this penta is a collapsed macayeal. For each node on the base of this penta, 
		 * we grab the vector. Once we know the vector, we follow the upper nodes, 
		 * inserting the same vector value into vector, until we reach the surface: */
		for(i=0;i<3;i++){

			node=nodes[i]; //base nodes
			dof1=node->GetDofList1();

			/*get vector for this base node: */
			vectorel=vector_serial[dof1];

			//go throvectorn all nodes which sit on top of this node, until we reach the surface, 
			//and plvector  vector in vector
			for(;;){

				dof1=node->GetDofList1();
				VecSetValues(vector,1,&dof1,&vectorel,INSERT_VALUES);

				if (node->IsOnSurface())break;
				/*get next node: */
				node=node->GetUpperNode();
			}
		}
	} 
}
/*}}}*/
/*FUNCTION Penta::InputExtrude {{{1*/
void  Penta::InputExtrude(int enum_type,bool only_if_collapsed){

	bool   onbed,collapse=false;
	Penta *penta          = NULL;
	Input *original_input = NULL;

	/*recover parameters: */
	if (only_if_collapsed) inputs->GetParameterValue(&collapse,CollapseEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*First: if only_if_collapsed, check wether this penta is collapsed*/
	if (only_if_collapsed && !collapse) return;

	/*Are we on the base, not on the surface?:*/
	if(onbed==1){

		/*OK, we are on bed. we will follow the steps:
		 * 1: find input and extrude it.
		 * 2: follow the upper element until we reach the surface
		 * 3: for each element, we will add a copy of the extruded input*/

		/*Step1: Extrude the original input: */
		original_input=(Input*)this->inputs->GetInput(enum_type);
		if(!original_input) ISSMERROR("%s%s"," could not find input with enum:",EnumAsString(enum_type));
		original_input->Extrude();

		/*Stop if there is only one layer of element*/
		if (this->IsOnSurface()) return;

		/*Step 2: this input has been extruded for this element, now follow the upper element*/
		penta=this;
		for(;;){

			/* get upper Penta*/
			penta=penta->GetUpperElement();
			ISSMASSERT(penta->Id()!=this->id);

			/*Add input of the basal element to penta->inputs*/
			Input* copy=NULL;
			copy=(Input*)original_input->copy();
			penta->inputs->AddInput((Input*)copy);

			/*Stop if we have reached the surface*/
			if (penta->IsOnSurface()) break;

		}
	}

	return;
}
/*}}}*/
/*FUNCTION Penta::GetB {{{1*/
void Penta::GetB(double* B, double* xyz_list, double* gauss_coord){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*NDOF2. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ dh/dx          0      ]
	 *                [   0           dh/dy   ]
	 *                [ 1/2*dh/dy  1/2*dh/dx  ]
	 *                [ 1/2*dh/dz      0      ]
	 *                [  0         1/2*dh/dz  ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 5x(NDOF2*numgrids)
	 */

	int i;
	const int numgrids=6;
	const int NDOF3=3;
	const int NDOF2=2;

	double dh1dh6[NDOF3][numgrids];

	/*Get dh1dh6 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list, gauss_coord);

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=dh1dh6[0][i]; 
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=0.0;

		*(B+NDOF2*numgrids*1+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=dh1dh6[1][i];

		*(B+NDOF2*numgrids*2+NDOF2*i)=(float).5*dh1dh6[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=(float).5*dh1dh6[0][i]; 

		*(B+NDOF2*numgrids*3+NDOF2*i)=(float).5*dh1dh6[2][i]; 
		*(B+NDOF2*numgrids*3+NDOF2*i+1)=0.0;

		*(B+NDOF2*numgrids*4+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*4+NDOF2*i+1)=(float).5*dh1dh6[2][i]; 
	}

}
/*}}}*/
/*FUNCTION Penta::GetB_artdiff {{{1*/
void Penta::GetB_artdiff(double* B_artdiff, double* xyz_list, double* gauss_coord){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_artdiff=[ dh/dx ]
	 *                       [ dh/dy ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 2x(DOFPERGRID*numgrids)
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double dh1dh6[calculationdof][numgrids];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list,gauss_coord);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(B_artdiff+DOFPERGRID*numgrids*0+DOFPERGRID*i)=dh1dh6[0][i]; 
		*(B_artdiff+DOFPERGRID*numgrids*1+DOFPERGRID*i)=dh1dh6[1][i]; 
	}
}
/*}}}*/
/*FUNCTION Penta::GetB_advec {{{1*/
void Penta::GetB_advec(double* B_advec, double* xyz_list, double* gauss_coord){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_advec =[ h ]
	 *                       [ h ]
	 *                       [ h ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(DOFPERGRID*numgrids)
	 */

	int i;
	int calculationdof=3;
	int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double l1l6[6];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctions(l1l6, gauss_coord);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(B_advec+DOFPERGRID*numgrids*0+DOFPERGRID*i)=l1l6[i]; 
		*(B_advec+DOFPERGRID*numgrids*1+DOFPERGRID*i)=l1l6[i]; 
		*(B_advec+DOFPERGRID*numgrids*2+DOFPERGRID*i)=l1l6[i]; 
	}
}
/*}}}*/
/*FUNCTION Penta::GetB_conduct {{{1*/
void Penta::GetB_conduct(double* B_conduct, double* xyz_list, double* gauss_coord){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_conduct=[ dh/dx ]
	 *                       [ dh/dy ]
	 *                       [ dh/dz ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(DOFPERGRID*numgrids)
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double dh1dh6[calculationdof][numgrids];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list,gauss_coord);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(B_conduct+DOFPERGRID*numgrids*0+DOFPERGRID*i)=dh1dh6[0][i]; 
		*(B_conduct+DOFPERGRID*numgrids*1+DOFPERGRID*i)=dh1dh6[1][i]; 
		*(B_conduct+DOFPERGRID*numgrids*2+DOFPERGRID*i)=dh1dh6[2][i]; 
	}
}
/*}}}*/
/*FUNCTION Penta::GetB_vert {{{1*/
void Penta::GetB_vert(double* B, double* xyz_list, double* gauss_coord){


	/*	Compute B  matrix. B=[dh1/dz dh2/dz dh3/dz dh4/dz dh5/dz dh6/dz];
		where hi is the interpolation function for grid i.*/

	int i;
	const int NDOF3=3;
	const int numgrids=6;
	double dh1dh6[NDOF3][numgrids];

	/*Get dh1dh6 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list, gauss_coord);

	/*Build B: */
	for (i=0;i<numgrids;i++){
		B[i]=dh1dh6[2][i];  
	}

}
/*}}}*/
/*FUNCTION Penta::GetBPrime {{{1*/
void Penta::GetBPrime(double* B, double* xyz_list, double* gauss_coord){

	/*Compute B  prime matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*NDOF2. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ 2*dh/dx     dh/dy   ]
	 *                [   dh/dx    2*dh/dy  ]
	 *                [ dh/dy      dh/dx    ]
	 *                [ dh/dz         0     ]
	 *                [  0         dh/dz    ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 5x(NDOF2*numgrids)
	 */

	int i;
	const int NDOF3=3;
	const int NDOF2=2;
	const int numgrids=6;

	double dh1dh6[NDOF3][numgrids];

	/*Get dh1dh6 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list, gauss_coord);

	/*Build BPrime: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=2.0*dh1dh6[0][i]; 
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=dh1dh6[1][i];

		*(B+NDOF2*numgrids*1+NDOF2*i)=dh1dh6[0][i];
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=2.0*dh1dh6[1][i];

		*(B+NDOF2*numgrids*2+NDOF2*i)=dh1dh6[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=dh1dh6[0][i]; 

		*(B+NDOF2*numgrids*3+NDOF2*i)=dh1dh6[2][i]; 
		*(B+NDOF2*numgrids*3+NDOF2*i+1)=0.0;

		*(B+NDOF2*numgrids*4+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*4+NDOF2*i+1)=dh1dh6[2][i]; 
	}
}
/*}}}*/
/*FUNCTION Penta::GetBprime_advec {{{1*/
void Penta::GetBprime_advec(double* Bprime_advec, double* xyz_list, double* gauss_coord){


	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Biprime_advec=[ dh/dx ]
	 *                       [ dh/dy ]
	 *                       [ dh/dz ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(DOFPERGRID*numgrids)
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double dh1dh6[calculationdof][numgrids];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list,gauss_coord);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(Bprime_advec+DOFPERGRID*numgrids*0+DOFPERGRID*i)=dh1dh6[0][i]; 
		*(Bprime_advec+DOFPERGRID*numgrids*1+DOFPERGRID*i)=dh1dh6[1][i]; 
		*(Bprime_advec+DOFPERGRID*numgrids*2+DOFPERGRID*i)=dh1dh6[2][i]; 
	}
}
/*}}}*/
/*FUNCTION Penta::GetBPrime_vert {{{1*/
void Penta::GetBPrime_vert(double* B, double* xyz_list, double* gauss_coord){

	// Compute Bprime  matrix. Bprime=[L1 L2 L3 L4 L5 L6] where Li is the nodal function for grid i

	int i;

	GetNodalFunctions(B, gauss_coord);

}
/*}}}*/
/*FUNCTION Penta::GetBprimeStokes {{{1*/
void Penta::GetBprimeStokes(double* B_prime, double* xyz_list, double* gauss_coord){

	/*	Compute B'  matrix. B'=[B1' B2' B3' B4' B5' B6' Bb'] where Bi' is of size 3*NDOF2. 
	 *	For grid i, Bi' can be expressed in the actual coordinate system
	 *	by: 
	 *				Bi'=[ dh/dx   0          0       0]
	 *					 [   0      dh/dy      0       0]
	 *					 [   0      0         dh/dz    0]
	 *					 [  dh/dy   dh/dx      0       0]
	 *					 [  dh/dz   0        dh/dx     0]
	 *					 [   0      dh/dz    dh/dy     0]
	 *					 [  dh/dx   dh/dy    dh/dz     0]
	 *					 [   0      0          0       h]
	 *	where h is the interpolation function for grid i.
	 *
	 * 	Same thing for the bubble fonction except that there is no fourth column
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=4;

	double dh1dh7[calculationdof][numgrids+1];
	double l1l6[numgrids];

	/*Get dh1dh7 in actual coordinate system: */
	GetNodalFunctionsDerivativesStokes(&dh1dh7[0][0],xyz_list, gauss_coord);

	GetNodalFunctions(l1l6, gauss_coord);

	/*B_primeuild B_prime: */
	for (i=0;i<numgrids+1;i++){
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i)=dh1dh7[0][i]; //B_prime[0][DOFPERGRID*i]=dh1dh6[0][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+2)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+2)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+2)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i)=dh1dh7[1][i]; 
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+1)=dh1dh7[0][i]; 
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+2)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+2)=dh1dh7[0][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+1)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+2)=dh1dh7[1][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i)=dh1dh7[0][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+2)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+2)=0;
	}

	for (i=0;i<numgrids;i++){ //last column not for the bubble function
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+3)=l1l6[i];
	}

}
/*}}}*/
/*FUNCTION Penta::GetBStokes {{{1*/
void Penta::GetBStokes(double* B, double* xyz_list, double* gauss_coord){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 3*DOFPERGRID. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 				Bi=[ dh/dx          0             0       0  ]
	 *					[   0           dh/dy           0       0  ]
	 *					[   0             0           dh/dy     0  ]
	 *					[ 1/2*dh/dy    1/2*dh/dx        0       0  ]
	 *					[ 1/2*dh/dz       0         1/2*dh/dx   0  ]
	 *					[   0          1/2*dh/dz    1/2*dh/dy   0  ]
	 *					[   0             0             0       h  ]
	 *					[ dh/dx         dh/dy         dh/dz     0  ]
	 *	where h is the interpolation function for grid i.
	 *	Same thing for Bb except the last column that does not exist.
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=4;

	double dh1dh7[calculationdof][numgrids+1];
	double l1l6[numgrids];


	/*Get dh1dh7 in actual coordinate system: */
	GetNodalFunctionsDerivativesStokes(&dh1dh7[0][0],xyz_list, gauss_coord);

	GetNodalFunctions(l1l6, gauss_coord);

	/*Build B: */
	for (i=0;i<numgrids+1;i++){
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i)=dh1dh7[0][i]; //B[0][DOFPERGRID*i]=dh1dh6[0][i];
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+2)=dh1dh7[2][i];
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i)=(float).5*dh1dh7[1][i]; 
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+1)=(float).5*dh1dh7[0][i]; 
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i)=(float).5*dh1dh7[2][i];
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+2)=(float).5*dh1dh7[0][i];
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+1)=(float).5*dh1dh7[2][i];
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+2)=(float).5*dh1dh7[1][i];
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i)=dh1dh7[0][i];
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+2)=dh1dh7[2][i];
	}

	for (i=0;i<numgrids;i++){ //last column not for the bubble function
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+3)=l1l6[i];
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+3)=0;
	}

}
/*}}}*/
/*FUNCTION Penta::GetDofList {{{1*/
void  Penta::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;

	for(i=0;i<6;i++){
		nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
		for(j=0;j<numberofdofspernode;j++){
			doflist[i*numberofdofspernode+j]=doflist_per_node[j];
		}
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION Penta::GetDofList1 {{{1*/
void  Penta::GetDofList1(int* doflist){
	
	int i;

	for(i=0;i<6;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}
/*}}}*/
/*FUNCTION Penta::GetJacobian {{{1*/
void Penta::GetJacobian(double* J, double* xyz_list,double* gauss_coord){

	const int NDOF3=3;
	int i,j;

	/*The Jacobian is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double A1,A2,A3; //area coordinates
	double xi,eta,zi; //parametric coordinates

	double x1,x2,x3,x4,x5,x6;
	double y1,y2,y3,y4,y5,y6;
	double z1,z2,z3,z4,z5,z6;

	/*Figure out xi,eta and zi (parametric coordinates), for this gaussian point: */
	A1=gauss_coord[0];
	A2=gauss_coord[1];
	A3=gauss_coord[2];

	xi=A2-A1;
	eta=SQRT3*A3;
	zi=gauss_coord[3];

	x1=*(xyz_list+3*0+0);
	x2=*(xyz_list+3*1+0);
	x3=*(xyz_list+3*2+0);
	x4=*(xyz_list+3*3+0);
	x5=*(xyz_list+3*4+0);
	x6=*(xyz_list+3*5+0);

	y1=*(xyz_list+3*0+1);
	y2=*(xyz_list+3*1+1);
	y3=*(xyz_list+3*2+1);
	y4=*(xyz_list+3*3+1);
	y5=*(xyz_list+3*4+1);
	y6=*(xyz_list+3*5+1);

	z1=*(xyz_list+3*0+2);
	z2=*(xyz_list+3*1+2);
	z3=*(xyz_list+3*2+2);
	z4=*(xyz_list+3*3+2);
	z5=*(xyz_list+3*4+2);
	z6=*(xyz_list+3*5+2);


	*(J+NDOF3*0+0)=0.25*(x1-x2-x4+x5)*zi+0.25*(-x1+x2-x4+x5);
	*(J+NDOF3*1+0)=SQRT3/12.0*(x1+x2-2*x3-x4-x5+2*x6)*zi+SQRT3/12.0*(-x1-x2+2*x3-x4-x5+2*x6);
	*(J+NDOF3*2+0)=SQRT3/12.0*(x1+x2-2*x3-x4-x5+2*x6)*eta+1/4*(x1-x2-x4+x5)*xi +0.25*(-x1+x5-x2+x4);

	*(J+NDOF3*0+1)=0.25*(y1-y2-y4+y5)*zi+0.25*(-y1+y2-y4+y5);
	*(J+NDOF3*1+1)=SQRT3/12.0*(y1+y2-2*y3-y4-y5+2*y6)*zi+SQRT3/12.0*(-y1-y2+2*y3-y4-y5+2*y6);
	*(J+NDOF3*2+1)=SQRT3/12.0*(y1+y2-2*y3-y4-y5+2*y6)*eta+0.25*(y1-y2-y4+y5)*xi+0.25*(y4-y1+y5-y2);

	*(J+NDOF3*0+2)=0.25*(z1-z2-z4+z5)*zi+0.25*(-z1+z2-z4+z5);
	*(J+NDOF3*1+2)=SQRT3/12.0*(z1+z2-2*z3-z4-z5+2*z6)*zi+SQRT3/12.0*(-z1-z2+2*z3-z4-z5+2*z6);
	*(J+NDOF3*2+2)=SQRT3/12.0*(z1+z2-2*z3-z4-z5+2*z6)*eta+0.25*(z1-z2-z4+z5)*xi+0.25*(-z1+z5-z2+z4);

}
/*}}}*/
/*FUNCTION Penta::GetJacobianDeterminant {{{1*/
void Penta::GetJacobianDeterminant(double*  Jdet, double* xyz_list,double* gauss_coord){

	/*On a penta, Jacobian varies according to coordinates. We need to get the Jacobian, and take 
	 * the determinant of it: */
	const int NDOF3=3;

	double J[NDOF3][NDOF3];

	GetJacobian(&J[0][0],xyz_list,gauss_coord);

	*Jdet= J[0][0]*J[1][1]*J[2][2]-J[0][0]*J[1][2]*J[2][1]-J[1][0]*J[0][1]*J[2][2]+J[1][0]*J[0][2]*J[2][1]+J[2][0]*J[0][1]*J[1][2]-J[2][0]*J[0][2]*J[1][1];

	if(*Jdet<0){
		ISSMERROR("%s%i","negative jacobian determinant on element ",id); 
	}
}
/*}}}*/
/*FUNCTION Penta::GetJacobianInvert {{{1*/
void Penta::GetJacobianInvert(double*  Jinv, double* xyz_list,double* gauss_coord){

	double Jdet;
	const int NDOF3=3;

	/*Call Jacobian routine to get the jacobian:*/
	GetJacobian(Jinv, xyz_list, gauss_coord);

	/*Invert Jacobian matrix: */
	MatrixInverse(Jinv,NDOF3,NDOF3,NULL,0,&Jdet);
}
/*}}}*/
/*FUNCTION Penta::GetLStokes {{{1*/
void Penta::GetLStokes(double* LStokes, double* gauss_coord_tria){

	/*
	 * Compute L  matrix. L=[L1 L2 L3] where Li is square and of size numdof. 
	 * For grid i, Li can be expressed in the actual coordinate system
	 * by: 
	 *       Li=[ h    0    0   0]
	 *	 	 [ 0    h    0   0]
	 *		 [ 0    0    h   0]
	 *		 [ 0    0    h   0]
	 *	 	 [ h    0    0   0]
	 *	 	 [ 0    h    0   0]
	 *	 	 [ h    0    0   0]
	 *	 	 [ 0    h    0   0]
	 *		 [ 0    0    h   0]
	 *		 [ 0    0    h   0]
	 *		 [ 0    0    h   0]
	 *	 	 [ h    0    0   0]
	 *	 	 [ 0    h    0   0]
	 *		 [ 0    0    h   0]
	 * where h is the interpolation function for grid i.
	 */

	int i;
	const int numgrids2d=3;
	int num_dof=4;

	double l1l2l3[numgrids2d];


	/*Get l1l2l3 in actual coordinate system: */
	l1l2l3[0]=gauss_coord_tria[0];
	l1l2l3[1]=gauss_coord_tria[1];
	l1l2l3[2]=gauss_coord_tria[2];

	/*Build LStokes: */
	for (i=0;i<3;i++){
		*(LStokes+num_dof*numgrids2d*0+num_dof*i)=l1l2l3[i]; //LStokes[0][NDOF2*i]=dh1dh3[0][i];
		*(LStokes+num_dof*numgrids2d*0+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*0+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*0+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*1+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*1+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*1+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*1+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*2+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*2+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*2+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*2+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*3+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*3+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*3+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*3+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*4+num_dof*i)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*4+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*4+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*4+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*5+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*5+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*5+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*5+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*6+num_dof*i)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*6+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*6+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*6+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*7+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*7+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*7+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*7+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*8+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*8+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*8+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*8+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*9+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*9+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*9+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*9+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*10+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*10+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*10+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*10+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*11+num_dof*i)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*11+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*11+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*11+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*12+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*12+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*12+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*12+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*13+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*13+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*13+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*13+num_dof*i+3)=0;

	}
}
/*}}}*/
/*FUNCTION Penta::GetLprimeStokes {{{1*/
void Penta::GetLprimeStokes(double* LprimeStokes, double* xyz_list, double* gauss_coord_tria, double* gauss_coord){

	/*
	 * Compute Lprime  matrix. Lprime=[Lp1 Lp2 Lp3] where Lpi is square and of size numdof. 
	 * For grid i, Lpi can be expressed in the actual coordinate system
	 * by: 
	 *       Lpi=[ h    0    0   0]
	 *		 [ 0    h    0   0]
	 *		 [ h    0    0   0]
	 *		 [ 0    h    0   0]
	 *		 [ 0    0    h   0]
	 *		 [ 0    0    h   0]
	 *		 [ 0    0  dh/dz 0]
	 *		 [ 0    0  dh/dz 0]
	 *		 [ 0    0  dh/dz 0]
	 *		 [dh/dz 0  dh/dx 0]
	 *		 [ 0 dh/dz dh/dy 0]
	 * 		 [ 0    0    0   h]
	 * 		 [ 0    0    0   h]
	 * 		 [ 0    0    0   h]
	 * where h is the interpolation function for grid i.
	 */

	int i;
	const int numgrids2d=3;
	int num_dof=4;

	double l1l2l3[numgrids2d];
	double dh1dh6[3][6];


	/*Get l1l2l3 in actual coordinate system: */
	l1l2l3[0]=gauss_coord_tria[0];
	l1l2l3[1]=gauss_coord_tria[1];
	l1l2l3[2]=gauss_coord_tria[2];

	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list,gauss_coord);

	/*Build LprimeStokes: */
	for (i=0;i<3;i++){
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i)=l1l2l3[i]; //LprimeStokes[0][NDOF2*i]=dh1dh3[0][i];
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i+1)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i+1)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i+2)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i+2)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i+2)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i+2)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i+2)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i+2)=dh1dh6[0][i];
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i+1)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i+2)=dh1dh6[1][i];
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i+3)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i+3)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i+3)=l1l2l3[i];

	}
}
/*}}}*/
/*FUNCTION Penta::GetMatrixInvert {{{1*/
void Penta::GetMatrixInvert(double*  Ke_invert, double* Ke){
	/*Inverse a 3 by 3 matrix only */

	double a,b,c,d,e,f,g,h,i;
	double det;
	int calculationdof=3;

	/*Take the matrix components: */
	a=*(Ke+calculationdof*0+0);
	b=*(Ke+calculationdof*0+1);
	c=*(Ke+calculationdof*0+2);
	d=*(Ke+calculationdof*1+0);
	e=*(Ke+calculationdof*1+1);
	f=*(Ke+calculationdof*1+2);
	g=*(Ke+calculationdof*2+0);
	h=*(Ke+calculationdof*2+1);
	i=*(Ke+calculationdof*2+2);

	det=a*(e*i-f*h)-b*(d*i-f*g)+c*(d*h-e*g);

	*(Ke_invert+calculationdof*0+0)=(e*i-f*h)/det;
	*(Ke_invert+calculationdof*0+1)=(c*h-b*i)/det;
	*(Ke_invert+calculationdof*0+2)=(b*f-c*e)/det;
	*(Ke_invert+calculationdof*1+0)=(f*g-d*i)/det;
	*(Ke_invert+calculationdof*1+1)=(a*i-c*g)/det;
	*(Ke_invert+calculationdof*1+2)=(c*d-a*f)/det;
	*(Ke_invert+calculationdof*2+0)=(d*h-e*g)/det;
	*(Ke_invert+calculationdof*2+1)=(b*g-a*h)/det;
	*(Ke_invert+calculationdof*2+2)=(a*e-b*d)/det;

}
/*}}}*/
/*FUNCTION Penta::GetNodalFunctions {{{1*/
void Penta::GetNodalFunctions(double* l1l6, double* gauss_coord){

	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	l1l6[0]=gauss_coord[0]*(1-gauss_coord[3])/2.0;

	l1l6[1]=gauss_coord[1]*(1-gauss_coord[3])/2.0;

	l1l6[2]=gauss_coord[2]*(1-gauss_coord[3])/2.0;

	l1l6[3]=gauss_coord[0]*(1+gauss_coord[3])/2.0;

	l1l6[4]=gauss_coord[1]*(1+gauss_coord[3])/2.0;

	l1l6[5]=gauss_coord[2]*(1+gauss_coord[3])/2.0;

}
/*}}}*/
/*FUNCTION Penta::GetNodalFunctionsDerivatives {{{1*/
void Penta::GetNodalFunctionsDerivatives(double* dh1dh6,double* xyz_list, double* gauss_coord){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the actual coordinate system: */


	int i;
	const int NDOF3=3;
	const int numgrids=6;

	double dh1dh6_ref[NDOF3][numgrids];
	double Jinv[NDOF3][NDOF3];

	/*Get derivative values with respect to parametric coordinate system: */
	GetNodalFunctionsDerivativesReference(&dh1dh6_ref[0][0], gauss_coord); 

	/*Get Jacobian invert: */
	GetJacobianInvert(&Jinv[0][0], xyz_list, gauss_coord);

	/*Build dh1dh3: 
	 *
	 * [dhi/dx]= Jinv*[dhi/dr]
	 * [dhi/dy]       [dhi/ds]
	 * [dhi/dz]       [dhi/dn]
	 */

	for (i=0;i<numgrids;i++){
		*(dh1dh6+numgrids*0+i)=Jinv[0][0]*dh1dh6_ref[0][i]+Jinv[0][1]*dh1dh6_ref[1][i]+Jinv[0][2]*dh1dh6_ref[2][i];
		*(dh1dh6+numgrids*1+i)=Jinv[1][0]*dh1dh6_ref[0][i]+Jinv[1][1]*dh1dh6_ref[1][i]+Jinv[1][2]*dh1dh6_ref[2][i];
		*(dh1dh6+numgrids*2+i)=Jinv[2][0]*dh1dh6_ref[0][i]+Jinv[2][1]*dh1dh6_ref[1][i]+Jinv[2][2]*dh1dh6_ref[2][i];
	}

}
/*}}}*/
/*FUNCTION Penta::GetNodalFunctionsDerivativesStokes {{{1*/
void Penta::GetNodalFunctionsDerivativesStokes(double* dh1dh7,double* xyz_list, double* gauss_coord){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * actual coordinate system: */

	int i;

	const  int numgrids=7;
	double dh1dh7_ref[3][numgrids];
	double Jinv[3][3];


	/*Get derivative values with respect to parametric coordinate system: */
	GetNodalFunctionsDerivativesReferenceStokes(&dh1dh7_ref[0][0], gauss_coord); 

	/*Get Jacobian invert: */
	GetJacobianInvert(&Jinv[0][0], xyz_list, gauss_coord);

	/*Build dh1dh6: 
	 *
	 * [dhi/dx]= Jinv'*[dhi/dr]
	 * [dhi/dy]        [dhi/ds]
	 * [dhi/dz]        [dhi/dzeta]
	 */

	for (i=0;i<numgrids;i++){
		*(dh1dh7+numgrids*0+i)=Jinv[0][0]*dh1dh7_ref[0][i]+Jinv[0][1]*dh1dh7_ref[1][i]+Jinv[0][2]*dh1dh7_ref[2][i];
		*(dh1dh7+numgrids*1+i)=Jinv[1][0]*dh1dh7_ref[0][i]+Jinv[1][1]*dh1dh7_ref[1][i]+Jinv[1][2]*dh1dh7_ref[2][i];
		*(dh1dh7+numgrids*2+i)=Jinv[2][0]*dh1dh7_ref[0][i]+Jinv[2][1]*dh1dh7_ref[1][i]+Jinv[2][2]*dh1dh7_ref[2][i];
	}

}
/*}}}*/
/*FUNCTION Penta::GetNodalFunctionsDerivativesReference {{{1*/
void Penta::GetNodalFunctionsDerivativesReference(double* dl1dl6,double* gauss_coord){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. Those values vary along xi,eta,z */

	const int numgrids=6;
	double A1,A2,A3,z;

	A1=gauss_coord[0]; //first area coordinate value. In term of xi and eta: A1=(1-xi)/2-eta/(2*SQRT3);
	A2=gauss_coord[1]; //second area coordinate value In term of xi and eta: A2=(1+xi)/2-eta/(2*SQRT3);
	A3=gauss_coord[2]; //third area coordinate value  In term of xi and eta: A3=y/SQRT3;
	z=gauss_coord[3]; //fourth vertical coordinate value. Corresponding nodal function: (1-z)/2 and (1+z)/2


	/*First nodal function derivatives. The corresponding nodal function is N=A1*(1-z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+0)=-0.5*(1.0-z)/2.0;
	*(dl1dl6+numgrids*1+0)=-0.5/SQRT3*(1.0-z)/2.0;
	*(dl1dl6+numgrids*2+0)=-0.5*A1;

	/*Second nodal function: The corresponding nodal function is N=A2*(1-z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+1)=0.5*(1.0-z)/2.0;
	*(dl1dl6+numgrids*1+1)=-0.5/SQRT3*(1.0-z)/2.0;
	*(dl1dl6+numgrids*2+1)=-0.5*A2;

	/*Third nodal function: The corresponding nodal function is N=A3*(1-z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+2)=0.0;
	*(dl1dl6+numgrids*1+2)=1.0/SQRT3*(1.0-z)/2.0;
	*(dl1dl6+numgrids*2+2)=-0.5*A3;

	/*Fourth nodal function: The corresponding nodal function is N=A1*(1+z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+3)=-0.5*(1.0+z)/2.0;
	*(dl1dl6+numgrids*1+3)=-0.5/SQRT3*(1.0+z)/2.0;
	*(dl1dl6+numgrids*2+3)=0.5*A1;

	/*Fifth nodal function: The corresponding nodal function is N=A2*(1+z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+4)=0.5*(1.0+z)/2.0;
	*(dl1dl6+numgrids*1+4)=-0.5/SQRT3*(1.0+z)/2.0;
	*(dl1dl6+numgrids*2+4)=0.5*A2;

	/*Sixth nodal function: The corresponding nodal function is N=A3*(1+z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+5)=0.0;
	*(dl1dl6+numgrids*1+5)=1.0/SQRT3*(1.0+z)/2.0;
	*(dl1dl6+numgrids*2+5)=0.5*A3;
}
/*}}}*/
/*FUNCTION Penta::GetNodalFunctionsDerivativesReferenceStokes {{{1*/
void Penta::GetNodalFunctionsDerivativesReferenceStokes(double* dl1dl7,double* gauss_coord){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. */

	int    numgrids=7; //six plus bubble grids

	double r=gauss_coord[1]-gauss_coord[0];
	double s=-3.0/SQRT3*(gauss_coord[0]+gauss_coord[1]-2.0/3.0);
	double zeta=gauss_coord[3];

	/*First nodal function: */
	*(dl1dl7+numgrids*0+0)=-0.5*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*1+0)=-SQRT3/6.0*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*2+0)=-0.5*(-0.5*r-SQRT3/6.0*s+ONETHIRD);

	/*Second nodal function: */
	*(dl1dl7+numgrids*0+1)=0.5*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*1+1)=-SQRT3/6.0*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*2+1)=-0.5*(0.5*r-SQRT3/6.0*s+ONETHIRD);

	/*Third nodal function: */
	*(dl1dl7+numgrids*0+2)=0;
	*(dl1dl7+numgrids*1+2)=SQRT3/3.0*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*2+2)=-0.5*(SQRT3/3.0*s+ONETHIRD);

	/*Fourth nodal function: */
	*(dl1dl7+numgrids*0+3)=-0.5*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*1+3)=-SQRT3/6.0*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*2+3)=0.5*(-0.5*r-SQRT3/6.0*s+ONETHIRD);

	/*Fith nodal function: */
	*(dl1dl7+numgrids*0+4)=0.5*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*1+4)=-SQRT3/6.0*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*2+4)=0.5*(0.5*r-SQRT3/6.0*s+ONETHIRD);

	/*Sixth nodal function: */
	*(dl1dl7+numgrids*0+5)=0;
	*(dl1dl7+numgrids*1+5)=SQRT3/3.0*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*2+5)=0.5*(SQRT3/3.0*s+ONETHIRD);

	/*Seventh nodal function: */
	*(dl1dl7+numgrids*0+6)=9.0/2.0*r*(1.0+zeta)*(zeta-1.0)*(SQRT3*s+1.0);
	*(dl1dl7+numgrids*1+6)=9.0/4.0*(1+zeta)*(1-zeta)*(SQRT3*pow(s,2.0)-2.0*s-SQRT3*pow(r,2.0));
	*(dl1dl7+numgrids*2+6)=27*gauss_coord[0]*gauss_coord[1]*gauss_coord[2]*(-2.0*zeta);

}
/*}}}*/
/*FUNCTION Penta::GetNodalFunctionsStokes {{{1*/
void Penta::GetNodalFunctionsStokes(double* l1l7, double* gauss_coord){

	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	/*First nodal function: */
	l1l7[0]=gauss_coord[0]*(1.0-gauss_coord[3])/2.0;

	/*Second nodal function: */
	l1l7[1]=gauss_coord[1]*(1.0-gauss_coord[3])/2.0;

	/*Third nodal function: */
	l1l7[2]=gauss_coord[2]*(1.0-gauss_coord[3])/2.0;

	/*Fourth nodal function: */
	l1l7[3]=gauss_coord[0]*(1.0+gauss_coord[3])/2.0;

	/*Fifth nodal function: */
	l1l7[4]=gauss_coord[1]*(1.0+gauss_coord[3])/2.0;

	/*Sixth nodal function: */
	l1l7[5]=gauss_coord[2]*(1.0+gauss_coord[3])/2.0;

	/*Seventh nodal function: */
	l1l7[6]=27*gauss_coord[0]*gauss_coord[1]*gauss_coord[2]*(1.0+gauss_coord[3])*(1.0-gauss_coord[3]);

}
/*}}}*/
/*FUNCTION Penta::GetParameterDerivativeValue {{{1*/
void Penta::GetParameterDerivativeValue(double* p, double* p_list,double* xyz_list, double* gauss_coord){

	/*From grid values of parameter p (p_list[0], p_list[1], p_list[2], p_list[3], p_list[4] and p_list[4]), return parameter derivative value at gaussian point specified by gauss_coord:
	 *   dp/dx=p_list[0]*dh1/dx+p_list[1]*dh2/dx+p_list[2]*dh3/dx+p_list[3]*dh4/dx+p_list[4]*dh5/dx+p_list[5]*dh6/dx;
	 *   dp/dy=p_list[0]*dh1/dy+p_list[1]*dh2/dy+p_list[2]*dh3/dy+p_list[3]*dh4/dy+p_list[4]*dh5/dy+p_list[5]*dh6/dy;
	 *   dp/dz=p_list[0]*dh1/dz+p_list[1]*dh2/dz+p_list[2]*dh3/dz+p_list[3]*dh4/dz+p_list[4]*dh5/dz+p_list[5]*dh6/dz;
	 *
	 *   p is a vector of size 3x1 already allocated.
	 */

	const int NDOF3=3;
	const int numgrids=6;
	double dh1dh6[NDOF3][numgrids];

	/*Get dh1dh6 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list, gauss_coord);

	*(p+0)=p_list[0]*dh1dh6[0][0]+p_list[1]*dh1dh6[0][1]+p_list[2]*dh1dh6[0][2]+p_list[3]*dh1dh6[0][3]+p_list[4]*dh1dh6[0][4]+p_list[5]*dh1dh6[0][5];
	;
	*(p+1)=p_list[0]*dh1dh6[1][0]+p_list[1]*dh1dh6[1][1]+p_list[2]*dh1dh6[1][2]+p_list[3]*dh1dh6[1][3]+p_list[4]*dh1dh6[1][4]+p_list[5]*dh1dh6[1][5];

	*(p+2)=p_list[0]*dh1dh6[2][0]+p_list[1]*dh1dh6[2][1]+p_list[2]*dh1dh6[2][2]+p_list[3]*dh1dh6[2][3]+p_list[4]*dh1dh6[2][4]+p_list[5]*dh1dh6[2][5];

}
/*}}}*/
/*FUNCTION Penta::GetParameterValue(double* pvalue, double* v_list,double* gauss_coord) {{{1*/
void Penta::GetParameterValue(double* pvalue, double* v_list,double* gauss_coord){

	const int numgrids=6;
	double l1l6[numgrids];

	GetNodalFunctions(&l1l6[0], gauss_coord);

	*pvalue=l1l6[0]*v_list[0]+l1l6[1]*v_list[1]+l1l6[2]*v_list[2]+l1l6[3]*v_list[3]+l1l6[4]*v_list[4]+l1l6[5]*v_list[5];
}
/*}}}*/
/*FUNCTION Penta::GetParameterValue(double* pvalue,Node* node1,Node* node2,double gauss_seg,int enumtype) {{{1*/
void Penta::GetParameterValue(double* pvalue,Node* node,int enumtype){

	/*Output*/
	double value;

	/*Intermediaries*/
	const int numnodes=6;
	int       grid=-1;
	int       i;
	double gauss[numnodes][4]={{1,0,0,-1},{0,1,0,-1},{0,0,1,-1},{1,0,0,1},{0,1,0,1},{0,0,1,1}};

	/*go through 3 nodes (all nodes for tria) and identify 1st and 2nd nodes: */
	ISSMASSERT(nodes);
	for(i=0;i<numnodes;i++){
		if (node==nodes[i]){
			grid=i;
			break;
		}
	}

	/*Check that the node has been found*/
	if (grid==-1) ISSMERROR("Node pointer not found in Penta's nodes");

	/*Get Parameter value on node*/
	inputs->GetParameterValue(pvalue,&gauss[grid][0],enumtype);
	return;

}
/*}}}*/
/*FUNCTION Penta::GetPhi {{{1*/
void Penta::GetPhi(double* phi, double*  epsilon, double viscosity){
	/*Compute deformational heating from epsilon and viscosity */

	double epsilon_matrix[3][3];
	double epsilon_eff;
	double epsilon_sqr[3][3];

	/* Build epsilon matrix */
	epsilon_matrix[0][0]=*(epsilon+0);
	epsilon_matrix[1][0]=*(epsilon+3);
	epsilon_matrix[2][0]=*(epsilon+4);
	epsilon_matrix[0][1]=*(epsilon+3);
	epsilon_matrix[1][1]=*(epsilon+1);
	epsilon_matrix[2][1]=*(epsilon+5);
	epsilon_matrix[0][2]=*(epsilon+4);
	epsilon_matrix[1][2]=*(epsilon+5);
	epsilon_matrix[2][2]=*(epsilon+2);

	/* Effective value of epsilon_matrix */
	epsilon_sqr[0][0]=pow(epsilon_matrix[0][0],2);
	epsilon_sqr[1][0]=pow(epsilon_matrix[1][0],2);
	epsilon_sqr[2][0]=pow(epsilon_matrix[2][0],2);
	epsilon_sqr[0][1]=pow(epsilon_matrix[0][1],2);
	epsilon_sqr[1][1]=pow(epsilon_matrix[1][1],2);
	epsilon_sqr[2][1]=pow(epsilon_matrix[2][1],2);
	epsilon_sqr[0][2]=pow(epsilon_matrix[0][2],2);
	epsilon_sqr[1][2]=pow(epsilon_matrix[1][2],2);
	epsilon_sqr[2][2]=pow(epsilon_matrix[2][2],2);

	epsilon_eff=1/pow(2,0.5)*pow((epsilon_sqr[0][0]+epsilon_sqr[0][1]+ epsilon_sqr[0][2]+ epsilon_sqr[1][0]+ epsilon_sqr[1][1]+ epsilon_sqr[1][2]+ epsilon_sqr[2][0]+ epsilon_sqr[2][1]+ epsilon_sqr[2][2]),0.5);
	*phi=2*pow(epsilon_eff,2.0)*viscosity;
}
/*}}}*/
/*FUNCTION Penta::GradjB {{{1*/
void  Penta::GradjB(Vec gradient){

	int i;
	Tria* tria=NULL;
	TriaVertexInput* triavertexinput=NULL;

	/*inputs: */
	bool onwater;
	bool collapse;
	bool onbed;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&collapse,CollapseEnum);
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*If on water, skip: */
	if(onwater)return;

	if (collapse){
		/*Bail out element if collapsed (2d) and not on bed*/
		if (!onbed) return;

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute gardj*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		tria->GradjB(gradient);
		delete tria;
	}
	else{
		/*B is a 2d field, use MacAyeal(2d) gradient even if it is Stokes or Pattyn*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		tria->GradjB(gradient);
		delete tria;
	}
	

}
/*}}}*/
/*FUNCTION Penta::ReduceMatrixStokes {{{1*/
void Penta::ReduceMatrixStokes(double* Ke_reduced, double* Ke_temp){

	int i,j;

	double Kii[24][24];
	double Kib[24][3];
	double Kbb[3][3];
	double Kbi[3][24];
	double Kbbinv[3][3];
	double KibKbbinv[24][3];
	double Kright[24][24];

	/*Create the four matrices used for reduction */
	for(i=0;i<24;i++){
		for(j=0;j<24;j++){
			Kii[i][j]=*(Ke_temp+27*i+j);
		}
		for(j=0;j<3;j++){
			Kib[i][j]=*(Ke_temp+27*i+j+24);
		}
	}
	for(i=0;i<3;i++){
		for(j=0;j<24;j++){
			Kbi[i][j]=*(Ke_temp+27*(i+24)+j);
		}
		for(j=0;j<3;j++){
			Kbb[i][j]=*(Ke_temp+27*(i+24)+j+24);
		}
	}

	/*Inverse the matrix corresponding to bubble part Kbb */
	GetMatrixInvert(&Kbbinv[0][0], &Kbb[0][0]);

	/*Multiply matrices to create the reduce matrix Ke_reduced */
	MatrixMultiply(&Kib[0][0],24,3,0,&Kbbinv[0][0],3,3,0,&KibKbbinv[0][0],0);
	MatrixMultiply(&KibKbbinv[0][0],24,3,0,&Kbi[0][0],3,24,0,&Kright[0][0],0);

	/*Affect value to the reduced matrix */
	for(i=0;i<24;i++){
		for(j=0;j<24;j++){
			*(Ke_reduced+24*i+j)=Kii[i][j]-Kright[i][j];
		}
	}
}
/*}}}*/
/*FUNCTION Penta::ReduceVectorStokes {{{1*/
void Penta::ReduceVectorStokes(double* Pe_reduced, double* Ke_temp, double* Pe_temp){

	int i,j;

	double Pi[24];
	double Pb[3];
	double Kbb[3][3];
	double Kib[24][3];
	double Kbbinv[3][3];
	double KibKbbinv[24][3];
	double Pright[24];

	/*Create the four matrices used for reduction */
	for(i=0;i<24;i++){
		Pi[i]=*(Pe_temp+i);
	}
	for(i=0;i<3;i++){
		Pb[i]=*(Pe_temp+i+24);
	}
	for(j=0;j<3;j++){
		for(i=0;i<24;i++){
			Kib[i][j]=*(Ke_temp+3*i+j);
		}
		for(i=0;i<3;i++){
			Kbb[i][j]=*(Ke_temp+3*(i+24)+j);
		}
	}

	/*Inverse the matrix corresponding to bubble part Kbb */
	GetMatrixInvert(&Kbbinv[0][0], &Kbb[0][0]);

	/*Multiply matrices to create the reduce matrix Ke_reduced */
	MatrixMultiply(&Kib[0][0],24,3,0,&Kbbinv[0][0],3,3,0,&KibKbbinv[0][0],0);
	MatrixMultiply(&KibKbbinv[0][0],24,3,0,&Pb[0],3,1,0,&Pright[0],0);

	/*Affect value to the reduced matrix */
	for(i=0;i<24;i++){
		*(Pe_reduced+i)=Pi[i]-Pright[i];
	}
}
/*}}}*/
/*FUNCTION Penta::SetClone {{{1*/
void  Penta::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
/*FUNCTION Penta::SurfaceNormal {{{1*/
void Penta::SurfaceNormal(double* surface_normal, double xyz_list[3][3]){

	int i;
	double v13[3];
	double v23[3];
	double normal[3];
	double normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	normal_norm=sqrt( pow(normal[0],2)+pow(normal[1],2)+pow(normal[2],2) );

	*(surface_normal)=normal[0]/normal_norm;
	*(surface_normal+1)=normal[1]/normal_norm;
	*(surface_normal+2)=normal[2]/normal_norm;

}
/*}}}*/
