/*!\file TriaVertexInput.c
 * \brief: implementation of the TriaVertexInput object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"

/*TriaVertexInput constructors and destructor*/
/*FUNCTION TriaVertexInput::TriaVertexInput(){{{1*/
TriaVertexInput::TriaVertexInput(){
	return;
}
/*}}}*/
/*FUNCTION TriaVertexInput::TriaVertexInput(int in_enum_type,double* values){{{1*/
TriaVertexInput::TriaVertexInput(int in_enum_type,double* in_values)
	:TriaRef(1)
{

	/*Set TriaRef*/
	this->SetElementType(P1Enum,0);
	this->element_type=P1Enum;

	/*Set Enum*/
	enum_type=in_enum_type;

	/*Set values*/
	values[0]=in_values[0];
	values[1]=in_values[1];
	values[2]=in_values[2];
}
/*}}}*/
/*FUNCTION TriaVertexInput::~TriaVertexInput(){{{1*/
TriaVertexInput::~TriaVertexInput(){
	return;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION TriaVertexInput::Echo {{{1*/
void TriaVertexInput::Echo(void){
	this->DeepEcho();
}
/*}}}*/
/*FUNCTION TriaVertexInput::DeepEcho{{{1*/
void TriaVertexInput::DeepEcho(void){

	printf("TriaVertexInput:\n");
	printf("   enum: %i (%s)\n",this->enum_type,EnumAsString(this->enum_type));
	printf("   values: [%g %g %g]\n",this->values[0],this->values[1],this->values[2]);
}
/*}}}*/
/*FUNCTION TriaVertexInput::Id{{{1*/
int    TriaVertexInput::Id(void){ return -1; }
/*}}}*/
/*FUNCTION TriaVertexInput::MyRank{{{1*/
int    TriaVertexInput::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION TriaVertexInput::Marshall{{{1*/
void  TriaVertexInput::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_value=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum value of TriaVertexInput: */
	enum_value=TriaVertexInputEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_value,sizeof(enum_value));marshalled_dataset+=sizeof(enum_value);
	
	/*marshall TriaVertexInput data: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	memcpy(marshalled_dataset,&values,sizeof(values));marshalled_dataset+=sizeof(values);

	*pmarshalled_dataset=marshalled_dataset;
}
/*}}}*/
/*FUNCTION TriaVertexInput::MarshallSize{{{1*/
int   TriaVertexInput::MarshallSize(){
	
	return sizeof(values)+
		+sizeof(enum_type)+
		+sizeof(int); //sizeof(int) for enum value
}
/*}}}*/
/*FUNCTION TriaVertexInput::Demarshall{{{1*/
void  TriaVertexInput::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&enum_type,marshalled_dataset,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	memcpy(&values,marshalled_dataset,sizeof(values));marshalled_dataset+=sizeof(values);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION TriaVertexInput::Enum{{{1*/
int TriaVertexInput::Enum(void){

	return TriaVertexInputEnum;

}
/*}}}*/
/*FUNCTION TriaVertexInput::copy{{{1*/
Object* TriaVertexInput::copy() {
	
	return new TriaVertexInput(this->enum_type,this->values);

}
/*}}}*/
	
/*TriaVertexInput management*/
/*FUNCTION TriaVertexInput::EnumType{{{1*/
int TriaVertexInput::EnumType(void){

	return this->enum_type;

}
/*}}}*/
/*FUNCTION TriaVertexInput::SpawnSingInput{{{1*/
Input* TriaVertexInput::SpawnSingInput(int index){

	/*output*/
	SingVertexInput* outinput=NULL;

	/*Create new Sing input (copy of current input)*/
	ISSMASSERT(index<3 && index>=0);
	outinput=new SingVertexInput(this->enum_type,this->values[index]);

	/*Assign output*/
	return outinput;

}
/*}}}*/
/*FUNCTION BeamVertexInput::SpawnBeamInput{{{1*/
Input* TriaVertexInput::SpawnBeamInput(int* indices){

	/*output*/
	BeamVertexInput* outinput=NULL;
	double newvalues[2];

	/*Loop over the new indices*/
	for(int i=0;i<2;i++){

		/*Check index value*/
		ISSMASSERT(indices[i]>=0 && indices[i]<3);

		/*Assign value to new input*/
		newvalues[i]=this->values[indices[i]];
	}

	/*Create new Beam input*/
	outinput=new BeamVertexInput(this->enum_type,&newvalues[0]);

	/*Assign output*/
	return outinput;

}
/*}}}*/
/*FUNCTION TriaVertexInput::SpawnTriaInput{{{1*/
Input* TriaVertexInput::SpawnTriaInput(int* indices){

	/*output*/
	TriaVertexInput* outinput=NULL;

	/*Create new Tria input (copy of current input)*/
	outinput=new TriaVertexInput(this->enum_type,&this->values[0]);

	/*Assign output*/
	return outinput;

}
/*}}}*/
/*FUNCTION TriaVertexInput::SpawnResult{{{1*/
ElementResult* TriaVertexInput::SpawnResult(int step, double time){

	return new TriaVertexElementResult(this->enum_type,this->values,step,time);

}
/*}}}*/

/*Object functions*/
/*FUNCTION TriaVertexInput::GetParameterValue{{{1*/
void TriaVertexInput::GetParameterValue(double* pvalue,double* gauss){
	/*P1 interpolation on Gauss point*/

	/*intermediary*/
	double l1l2l3[3];

	/*nodal functions: */
	GetNodalFunctions(l1l2l3,gauss);

	/*Assign output pointers:*/
	*pvalue=l1l2l3[0]*values[0]+l1l2l3[1]*values[1]+l1l2l3[2]*values[2];

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetParameterValues{{{1*/
void TriaVertexInput::GetParameterValues(double* values,double* gauss_pointers, int numgauss){
	/*It is assumed that output values has been correctly allocated*/

	int i,j;
	double gauss[3];

	for (i=0;i<numgauss;i++){

		/*Get current Gauss point coordinates*/
		for (j=0;j<3;j++) gauss[j]=gauss_pointers[i*3+j];

		/*Assign parameter value*/
		GetParameterValue(&values[i],&gauss[0]);
	}

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetParameterDerivativeValue{{{1*/
void TriaVertexInput::GetParameterDerivativeValue(double* p, double* xyz_list, double* gauss){
	/*From node values of parameter p (plist[0],plist[1],plist[2]), return parameter derivative value at gaussian 
	 * point specified by gauss_l1l2l3:
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *
	 * p is a vector of size 2x1 already allocated.
	 */

	const int NDOF2=2;
	const int numgrids=3;
	double dh1dh3[NDOF2][numgrids]; //nodal derivative functions in actual coordinate system.

	/*Get nodal funnctions derivatives in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh3[0][0],xyz_list,gauss);

	p[0]=this->values[0]*dh1dh3[0][0]+this->values[1]*dh1dh3[0][1]+this->values[2]*dh1dh3[0][2];
	p[1]=this->values[0]*dh1dh3[1][0]+this->values[1]*dh1dh3[1][1]+this->values[2]*dh1dh3[1][2];

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetVxStrainRate2d{{{1*/
void TriaVertexInput::GetVxStrainRate2d(double* epsilonvx,double* xyz_list, double* gauss){

	/*Intermediary*/
	int       i;
	const int numgrids=3;
	const int NDOF2=2;
	double B[3][NDOF2*numgrids];
	double velocity[3][NDOF2];

	/*Get B matrix: */
	GetB(&B[0][0], xyz_list, gauss);

	/*Here, we are computing the strain rate of (vx,0)*/
	for(i=0;i<3;i++){
		velocity[i][0]=this->values[i];
		velocity[i][1]=0.0;
	}
	/*Get epsilon(vx) = B*velocity*/
	MatrixMultiply( &B[0][0],3,NDOF2*numgrids,0,
				&velocity[0][0],NDOF2*numgrids,1,0,
				epsilonvx,0);
}
/*}}}*/
/*FUNCTION TriaVertexInput::GetVyStrainRate2d{{{1*/
void TriaVertexInput::GetVyStrainRate2d(double* epsilonvy,double* xyz_list, double* gauss){

	/*Intermediary*/
	int       i;
	const int numgrids=3;
	const int NDOF2=2;
	double B[3][NDOF2*numgrids];
	double velocity[3][NDOF2];

	/*Get B matrix: */
	GetB(&B[0][0], xyz_list, gauss);

	/*Here, we are computing the strain rate of (0,vy)*/
	for(i=0;i<3;i++){
		velocity[i][0]=0.0;
		velocity[i][1]=this->values[i];
	}
	/*Get epsilon(vy) = B*velocity*/
	MatrixMultiply( &B[0][0],3,NDOF2*numgrids,0,
				&velocity[0][0],NDOF2*numgrids,1,0,
				epsilonvy,0);
}
/*}}}*/
/*FUNCTION TriaVertexInput::ChangeEnum{{{1*/
void TriaVertexInput::ChangeEnum(int newenumtype){
	this->enum_type=newenumtype;
}
/*}}}*/
/*FUNCTION TriaVertexInput::GetParameterAverage{{{1*/
void TriaVertexInput::GetParameterAverage(double* pvalue){
	*pvalue=1./3.*(values[0]+values[1]+values[2]);
}
/*}}}*/

/*Intermediary*/
/*FUNCTION TriaVertexInput::SquareMin{{{1*/
void TriaVertexInput::SquareMin(double* psquaremin, bool process_units,Parameters* parameters){

	int i;
	const int numnodes=3;
	double valuescopy[numnodes];
	double squaremin;

	/*First,  copy values, to process units if requested: */
	for(i=0;i<numnodes;i++)valuescopy[i]=this->values[i];

	/*Process units if requested: */
	if(process_units)NodalValuesUnitConversion(&valuescopy[0],numnodes,enum_type,parameters);

	/*Now, figure out minimum of valuescopy: */
	squaremin=pow(valuescopy[0],2);
	for(i=1;i<numnodes;i++){
		if(pow(valuescopy[i],2)<squaremin)squaremin=pow(valuescopy[i],2);
	}
	/*Assign output pointers:*/
	*psquaremin=squaremin;
}
/*}}}*/
/*FUNCTION TriaVertexInput::Scale{{{1*/
void TriaVertexInput::Scale(double scale_factor){
	
	int i;
	const int numgrids=3;

	for(i=0;i<numgrids;i++)values[i]=values[i]*scale_factor;
}
/*}}}*/
/*FUNCTION TriaVertexInput::AXPY{{{1*/
void TriaVertexInput::AXPY(Input* xinput,double scalar){

	int i;
	const int numgrids=3;
	TriaVertexInput*  xtriavertexinput=NULL;

	/*xinput is of the same type, so cast it: */
	xtriavertexinput=(TriaVertexInput*)xinput;

	/*Carry out the AXPY operation depending on type:*/
	switch(xinput->Enum()){

		case TriaVertexInputEnum :
			for(i=0;i<numgrids;i++)this->values[i]=this->values[i]+scalar*xtriavertexinput->values[i];
			return;

		default :
			ISSMERROR("not implemented yet");
	}

}
/*}}}*/
/*FUNCTION TriaVertexInput::Constrain{{{1*/
void TriaVertexInput::Constrain(double cm_min, double cm_max){

	int i;
	const int numgrids=3;
		
	if(!isnan(cm_min)) for(i=0;i<numgrids;i++)if (this->values[i]<cm_min)this->values[i]=cm_min;
	if(!isnan(cm_max)) for(i=0;i<numgrids;i++)if (this->values[i]>cm_max)this->values[i]=cm_max;

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetVectorFromInputs{{{1*/
void TriaVertexInput::GetVectorFromInputs(Vec vector,int* doflist){

	const int numvertices=3;
	VecSetValues(vector,numvertices,doflist,(const double*)this->values,INSERT_VALUES);

} /*}}}*/
/*FUNCTION TriaVertexInput::GetValuesPtr{{{1*/
void TriaVertexInput::GetValuesPtr(double** pvalues,int* pnum_values){

	*pvalues=this->values;
	*pnum_values=3;

}
/*}}}*/
