function md=solve(md,varargin)
%SOLVE - apply solution sequence for this model
%
%   Usage:
%       md=solve(md,varargin)
%       where varargin is a lit of paired arguments. 
%       arguments can be: 'analysis_type': 'diagnostic','thermal','prognostic','transient'
%       arguments can be: 'sub_analysis_type': 'transient',''
%       arguments can be: 'package': 'macayeal','ice','cielo'
%
%   Examples:
%      md=solve(md,'analysis_type','diagnostic','package','cielo');
%      md=solve(md,'analysis_type','control','package','ice');
%      md=solve(md,'analysis_type','thermal','sub_analysis_type','transient','package','ice');
%      md=solve(md,'analysis_type','thermal','sub_analysis_type','','package','cielo');

%some checks on list of arguments
global ISSM_DIR

%recover options
options=recover_solve_options(md,varargin{:});

%add default options
options=process_solve_options(options);

%recover some fields
md.analysis_type=options.analysis_type;
md.sub_analysis_type=options.sub_analysis_type;
package=options.package;

%Use package to set solution namespace
usenamespace(package);

if ~ismodelselfconsistent(md,package),
	error(' '); %previous error messages should explain what is going on.
end

displaystring(md.debug,'\n%s\n','launching solution sequence');


%If running in parallel, we have a different way of launching the solution
%sequences. qmu is the only solution sequence that cannot run in parallel
if ~strcmpi(md.analysis_type,'qmu'),
	if ~strcmpi(md.cluster,'none'),
		md=solveparallel(md)
		return;
	end
end

%Lauch correct solution sequence
if strcmpi(md.analysis_type,'diagnostic'),
	md=diagnostic(md);

elseif strcmpi(md.analysis_type,'mesh'),
	md=mesh(md);

elseif strcmpi(md.analysis_type,'transient'),
	md=transient(md);

elseif strcmpi(md.analysis_type,'qmu'),
	md=qmu(md,package);

elseif strcmpi(md.analysis_type,'mesh2grid'),
	md=mesh2grid(md);;

elseif strcmpi(md.analysis_type,'prognostic'),
	md=prognostic(md);;

elseif strcmpi(md.analysis_type,'control'),
	md=control(md);

elseif strcmpi(md.analysis_type,'thermal'),
	md=thermal(md);
else
	error('solution type not supported for this model configuration.');
end

%Check result is consistent
displaystring(md.debug,'%s\n','checking result consistency');
if ~isresultconsistent(md),
	disp('!! results not consistent correct the model !!') %it would be very cruel to put an error, it would kill the computed results (even if not consistent...)
end

%re-add the paths
addpath(genpath_ice([ISSM_DIR '/src/m/solutions/macayeal']));
addpath(genpath_ice([ISSM_DIR '/src/m/solutions/ice']));
addpath(genpath_ice([ISSM_DIR '/src/m/solutions/cielo']));
addpath(genpath_ice([ISSM_DIR '/bin']));

function solveusage();
disp(' ');
disp('   Solve usage: md=solve(md,md.analysis_type,package)');
disp('         md.analysis_type can be ''diagnostic'',''transient'', ''thermal'',''thermaltransient'',''parameters'',''mesh2grid''  or ''control''');
disp('         package is either ''cielo'' or ''ice''');
