function part=partitioner(md,varargin)
%PARTITIONER - partition mesh 
%
%   List of options to partitioner: 
%
%   package: 'chaco', 'metis' or 'scotch'
%   npart: number of partitions.
%   weighting: 'on' or 'off': default off
%   section:  1 by defaults(1=bisection, 2=quadrisection, 3=octasection)
%   
%   Usage:
%      md.part=partitioner(md,'package','chaco','npart',100,'weighting','on');
%


%get options: 
options=pairoptions(varargin{:});

%set defaults
options=addfielddefault(options,'package','chaco');
options=addfielddefault(options,'npart',10);
options=addfielddefault(options,'weighting','on');
options=addfielddefault(options,'section',1);

%get package: 
package=getfieldvalue(options,'package');
npart=getfieldvalue(options,'npart');

%adjacency matrix if needed:
if (strcmpi(package,'chaco') || strcmpi(package,'scotch')),
	md=adjacency(md);
end


if strcmpi(package,'chaco'),

	%  default method (from chaco.m)
	method=[1 1 0 0 1 1 50 0 .001 7654321]';
	method(1)=3;    %  global method (3=inertial (geometric))
	method(3)=0;    %  vertex weights (0=off, 1=on)
	
	%specify bisection
	method(6)=getfieldvalue(options,'section');%  ndims (1=bisection, 2=quadrisection, 3=octasection)

	%are we using weights? 
	if strcmpi(getfieldvalue(options,'weighting'),'on'),
		weights=floor(md.vwgt/min(md.vwgt));
		method(3)=1;
	else 
		weights=[];
	end
	
	%  partition into nparts
	part=Chaco(md.adjacency,weights,[],md.x, md.y ,md.z,method,npart,[])'+1; %index partitions from 1 up. like metis.

elseif strcmpi(package,'scotch'),

	%are we using weights? 
	if strcmpi(getfieldvalue(options,'weighting'),'on'),
		weights=floor(md.vwgt/min(md.vwgt));
	end
	maptab=Scotch(md.adjacency,[],weights,[],'cmplt',[npart]);
	
	part=maptab(:,2)+1;%index partitions from 1 up. like metis.


elseif strcmpi(package,'linear'),

	part=1:1:md.numberofgrids;

elseif strcmpi(package,'metis'),

	[element_partitioning,part]=MeshPartition(md,npart);

else

	error(['partitioner error message: could not find ' package ' partitioner']);
	help partitioner

end
