/*!\file:  objectivefunctionC
 * \brief  objective function that returns a misfit, for a certain parameter.
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "diagnostic"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

double objectivefunctionC(double search_scalar,OptArgs* optargs){

	int i;  
	
	/*output: */
	double J;
	
	/*parameters: */
	FemModel* femmodel=NULL;
	double* p_g=NULL;
	double* u_g_obs=NULL;
	double* grad_g=NULL;
	ParameterInputs* inputs=NULL;
	int n;

	/*intermediary:*/
	int gsize;
	double* optscal=NULL;
	double* fit=NULL;
	double  mincontrolconstraint;
	double  maxcontrolconstraint;
	char*   control_type=NULL;
	double* p_g_copy=NULL;
	int     analysis_type;
	int     sub_analysis_type;
	Vec     u_g=NULL;
	double* u_g_double=NULL;
	int     numberofnodes;


	/*Recover parameters: */
	femmodel=optargs->femmodel;
	p_g=optargs->p_g;
	u_g_obs=optargs->u_g_obs;
	grad_g=optargs->grad_g;
	inputs=optargs->inputs;
	n=optargs->n;

	gsize=femmodel->nodesets->GetGSize();
	femmodel->parameters->FindParam((void*)&optscal,"optscal");
	femmodel->parameters->FindParam((void*)&mincontrolconstraint,"mincontrolconstraint");
	femmodel->parameters->FindParam((void*)&maxcontrolconstraint,"maxcontrolconstraint");
	femmodel->parameters->FindParam((void*)&control_type,"control_type");
	femmodel->parameters->FindParam((void*)&fit,"fit");
	femmodel->parameters->FindParam((void*)&analysis_type,"analysis_type");
	femmodel->parameters->FindParam((void*)&sub_analysis_type,"sub_analysis_type");
	femmodel->parameters->FindParam((void*)&numberofnodes,"numberofnodes");

	/*First copy p_g so we don't modify it: */
	p_g_copy=(double*)xmalloc(gsize*sizeof(double));
	memcpy(p_g_copy,p_g,gsize*sizeof(double));

	/*First, update p_g using search_scalar: */
	for(i=0;i<gsize;i++)p_g_copy[i]=p_g_copy[i]+search_scalar*optscal[n]*grad_g[i];

	/*Constrain:*/
	ControlConstrainx( p_g_copy,gsize,mincontrolconstraint,maxcontrolconstraint,control_type);

	/*Add new parameter to inputs: */
	inputs->Add(control_type,p_g_copy,2,numberofnodes);

	//Run diagnostic with updated parameters.
	diagnostic_core_nonlinear(&u_g,NULL,NULL,femmodel,inputs,analysis_type,sub_analysis_type);
	VecToMPISerial(&u_g_double,u_g); VecFree(&u_g);

	//Compute misfit for this velocity field. 
	inputs->Add("fit",fit[n]);
	Misfitx( &J, femmodel->elements,femmodel->nodes, femmodel->loads, femmodel->materials, 
		u_g_double,u_g_obs, inputs,analysis_type,sub_analysis_type);


	/*Free ressources:*/
	xfree((void**)&optscal);
	xfree((void**)&control_type);
	xfree((void**)&p_g_copy);
	xfree((void**)&u_g_double);

	return J;
}
