/*!\file: diagnostic_core.cpp
 * \brief: core of the diagnostic solution 
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "cielodiagnostic_core"

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./parallel.h"
#include "../issm.h"

void diagnostic_core(Vec* pug, Vec* ppg,FemModel* fems, ParameterInputs* inputs){

	/*fem models: */
	FemModel* fem_dh=NULL;
	FemModel* fem_dv=NULL;
	FemModel* fem_dhu=NULL;
	FemModel* fem_ds=NULL;
	FemModel* fem_sl=NULL;

	/*solutions: */
	Vec ug=NULL;
	Vec ug_horiz=NULL;
	Vec ug_vert=NULL;
	Vec ug_stokes=NULL;
	Vec pg=NULL;
	Vec slopex=NULL;
	Vec slopey=NULL;

	/*flags: */
	int debug=0;
	int dim=-1;
	int ishutter=0;
	int ismacayealpattyn=0;
	int isstokes=0;
	int numberofdofspernode_sl;
	int numberofdofspernode_dh;
	int numberofdofspernode_ds;
	int numberofnodes;

	double stokesreconditioning;

	/*dof recovery: */
	int dof01[2]={0,1};
	int dof2[1]={2};
	int dof012[3]={0,1,2};
	int dof3[1]={3};

	/*recover fem models: */
	fem_dh=fems+0;
	fem_dv=fems+1;
	fem_ds=fems+2;
	fem_dhu=fems+3;
	fem_sl=fems+4;

	//first recover parameters common to all solutions
	fem_dh->parameters->FindParam((void*)&debug,"debug");
	fem_dh->parameters->FindParam((void*)&dim,"dim");
	fem_dhu->parameters->FindParam((void*)&ishutter,"ishutter");
	fem_dh->parameters->FindParam((void*)&ismacayealpattyn,"ismacayealpattyn");
	fem_dh->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
	fem_ds->parameters->FindParam((void*)&isstokes,"isstokes");
	fem_ds->parameters->FindParam((void*)&stokesreconditioning,"stokesreconditioning");

	//specific parameters for specific models
	fem_dh->parameters->FindParam((void*)&numberofdofspernode_dh,"numberofdofspernode");
	fem_sl->parameters->FindParam((void*)&numberofdofspernode_sl,"numberofdofspernode");
	fem_ds->parameters->FindParam((void*)&numberofdofspernode_ds,"numberofdofspernode");

	if(ishutter){
			
		if(debug)_printf_("%s\n","computing surface slope (x and y derivatives)...");
		diagnostic_core_linear(&slopex,fem_sl,inputs,SlopeComputeAnalysisEnum(),SurfaceXAnalysisEnum());
		diagnostic_core_linear(&slopey,fem_sl,inputs,SlopeComputeAnalysisEnum(),SurfaceYAnalysisEnum());

		if (dim==3){
		
			if(debug)_printf_("%s\n","extruding slopes in 3d...");
			SlopeExtrudex( slopex, fem_sl->elements,fem_sl->nodes,fem_sl->loads,fem_sl->materials);
			SlopeExtrudex( slopey, fem_sl->elements,fem_sl->nodes,fem_sl->loads,fem_sl->materials);
		}

		if(debug)_printf_("%s\n"," adding slopes in inputs...");
		inputs->Add("surfaceslopex",slopex,numberofdofspernode_sl,numberofnodes);
		inputs->Add("surfaceslopey",slopey,numberofdofspernode_sl,numberofnodes);

		if(debug)_printf_("%s\n"," computing hutter velocities...");
		diagnostic_core_linear(&ug,fem_dhu,inputs,DiagnosticAnalysisEnum(),HutterAnalysisEnum());

		if(debug)_printf_("%s\n"," computing pressure according to MacAyeal...");
		ComputePressurex( &pg,fem_dhu->elements,fem_dhu->nodes,fem_dhu->loads,fem_dhu->materials, numberofnodes);

		if(debug)_printf_("%s\n"," update boundary conditions for macyeal pattyn using hutter results...");
		if (ismacayealpattyn){
			VecFree(&fem_dh->yg); VecFree(&fem_dh->ys);VecFree(&fem_dh->ys0);
			VecDuplicatePatch(&fem_dh->yg,ug);
			Reducevectorgtosx(&fem_dh->ys,&fem_dh->ys0, fem_dh->yg,fem_dh->nodesets);
		}

	}

	if (ismacayealpattyn){
		
		if(debug)_printf_("%s\n"," computing horizontal velocities...");
		diagnostic_core_nonlinear(&ug,NULL,NULL,fem_dh,inputs,DiagnosticAnalysisEnum(),HorizAnalysisEnum());

		if(debug)_printf_("%s\n"," computing pressure according to MacAyeal...");
		ComputePressurex( &pg,fem_dh->elements, fem_dh->nodes, fem_dh->loads,  fem_dh->materials, numberofnodes);

	}
	
	
	if (dim==3){

		if(debug)_printf_("%s\n"," extruding horizontal velocities...");
		VecDuplicatePatch(&ug_horiz,ug); VelocityExtrudex( ug_horiz,fem_dh->elements,fem_dh->nodes, fem_dh->loads,fem_dh-> materials);

		if(debug)_printf_("%s\n"," computing vertical velocities...");
		inputs->Add("velocity",ug_horiz,numberofdofspernode_dh,numberofnodes);
		diagnostic_core_linear(&ug_vert,fem_dv,inputs,DiagnosticAnalysisEnum(),VertAnalysisEnum());

		if(debug)_printf_("%s\n"," combining horizontal and vertical velocities...");
		VecFree(&ug); ug=NewVec(numberofnodes*3);

		VecMerge(ug,ug_horiz,dofsetgen(2,&dof01[0],3,numberofnodes*3),numberofnodes*2);
		VecMerge(ug,ug_vert,dofsetgen(1,&dof2[0],3,numberofnodes*3),numberofnodes*1);

		if(debug)_printf_("%s\n"," computing pressure according to Pattyn...");
		ComputePressurex( &pg,fem_dh->elements, fem_dh->nodes, fem_dh->loads,  fem_dh->materials, numberofnodes);
		
		if (isstokes){

			//"recondition" pressure 
			VecScale(pg,1.0/stokesreconditioning);

			if(debug)_printf_("%s\n","computing bed slope (x and y derivatives)...");
			diagnostic_core_linear(&slopex,fem_sl,inputs,SlopeComputeAnalysisEnum(),BedXAnalysisEnum());
			diagnostic_core_linear(&slopey,fem_sl,inputs,SlopeComputeAnalysisEnum(),BedYAnalysisEnum());
			SlopeExtrudex( slopex, fem_sl->elements,fem_sl->nodes,fem_sl->loads,fem_sl->materials);
			SlopeExtrudex( slopey, fem_sl->elements,fem_sl->nodes,fem_sl->loads,fem_sl->materials);

			inputs->Add("bedslopex",slopex,numberofdofspernode_sl,numberofnodes);
			inputs->Add("bedslopey",slopey,numberofdofspernode_sl,numberofnodes);
			
			//recombine ug and pg: 
			ug_stokes=NewVec(fem_ds->nodesets->GetGSize());
			VecMerge(ug_stokes,ug,dofsetgen(3,dof012,4,numberofnodes*4),numberofnodes*3);
			VecMerge(ug_stokes,pg,dofsetgen(1,dof3,4,numberofnodes*4),numberofnodes);

			inputs->Add("velocity",ug_stokes,numberofdofspernode_ds,numberofnodes);

			if(debug)_printf_("%s\n"," update boundary conditions for stokes using velocities previously computed...");
			VecMerge(fem_ds->yg,ug,dofsetgen(3,dof012,4,numberofnodes*4),3*numberofnodes);
			VecFree(&fem_ds->ys); VecFree(&fem_ds->ys0);
			Reducevectorgtosx(&fem_ds->ys,&fem_ds->ys0, fem_ds->yg,fem_ds->nodesets);

			if(debug)_printf_("%s\n"," computing stokes velocities and pressure ...");
			VecFree(&ug);
			diagnostic_core_nonlinear(&ug,NULL,NULL,fem_ds,inputs,DiagnosticAnalysisEnum(),StokesAnalysisEnum());
		
			//decondition" pressure
			VecFree(&pg);
			VecPartition(&pg, ug, dofsetgen(1,dof3,4,numberofnodes*4), numberofnodes*1);
			VecScale(pg,stokesreconditioning);
		}
	}
	
	/*Assign output pointers: */
	*pug=ug;
	*ppg=pg;
}
