/*
	PentaElement.c
*/


#include <stdio.h>
#include <string.h>	
#include <math.h>

/* variadic functions: */
#include <stdarg.h>

/* environment: */
#include "../../include/cielo_types.h"
#include "../../include/cielo_macros.h"
#include "../../include/cielo_externals.h"

/* hosting environment: */
#include "../../include/matlab_includes.h"

/* object handlers: */
#include "../../DataSet/DataSet.h"
#include "../../EnumDefinitions/EnumDefinitions.h"

/* shared utilities (includes alloc.h memory management): */
#include "../../shared/shared.h"

/* "virtual function" access: */
#include "./../VirtualFunctions.h"

#include "./PentaElement.h"

#include "./GaussPoints.h"
#include "./GetElementGridData.h"
#include "./MatrixOperations.h"
#include "../Matice.h"

/*Objects: */
#include "../objects.h"

#include "./PentaElement.h"


/*For debugging purposes: */
#define ELID 141 //element for which to print debug statements
#define RANK 2 //rank of cpu for which to print debug statements.
//#define _DEBUGELEMENTS_
//#define _DEBUGGAUSS_


#define NDOF1 1
#define NDOF2 2
#define NDOF3 3
#define numgrids 6

/*--------------------------------------------------
	NewPentaElement
  --------------------------------------------------*/
PentaElement* NewPentaElement( void* vppenta)
{

	Penta* 		penta = (Penta*)vppenta;
	PentaElement*	this   = NULL;
	int	i,j;

	this=(PentaElement*)xcalloc(1,sizeof(PentaElement));	/* guaranteed NULL, 0 initialisation */

	/* include the "base class" in the "derived" object: */
	memcpy( &this->penta, penta, sizeof(Penta));

	/* since zero could be a valid index, set all DataSet index
	"pointers" to UNDEF: */

	for ( i=0; i<numgrids; i++){
		this->internalgrid_indices[i] = UNDEF;
		this->internalgrid_indicesb[i] = UNDEF;
		this->pg[i]=NULL;
	}
	
	this->matice = NULL;
	this->matice_index=0;
	this->matice_enum=0;

	this->matpar = NULL;
	this->matpar_index=0;
	this->matpar_enum=0;


	return this;
}


/*--------------------------------------------------
	DeletePentaElement
  --------------------------------------------------*/

void DeletePentaElement(void* *vpthis)
{
	/* vpthis for polymorphic function compatibility */

	PentaElement* *pthis = (PentaElement**)vpthis;
	PentaElement*   this = *pthis;

	#ifdef _DEBUG_
	_printf_("   DeletePentaElement message:\n");
	_printf_("   deleting PentaElement with eid = %d\n", this->penta.eid);
	#endif

	xfree(vpthis);
}

/*--------------------------------------------------
	PentaElementEcho
  --------------------------------------------------*/

void PentaElementEcho(void* vpthis)
{
	/* vpthis for polymorphic function compatibility */
	PentaElement* this = (PentaElement*)vpthis;

	int	i,j;
	;

	printf("\nPentaElement echo:\n");
	printf("-------------------\n");
	printf("   base Penta object:\n");
	PentaEcho( &this->penta);
	printf("\n");

	printf("   element points to following grids (pointer-based ops):\n");

	printf("   local grid number   corresponding grid id\n");
	printf("   -----------------   ---------------------\n");

	for ( i=0; i<PentaElementGetNumberOfGrids(this); i++) {
		if ( this->pg[i])
			printf("        %8d              %8d\n", i, InternalGridGetID( this->pg[i]));
	}

	printf("   element references the following:\n");

	if ( this->matice) printf("   matice entry of enum type %d\n", this->matice_enum);
	else                  printf("   matice reference is NULL\n");

	if ( this->matpar) printf("   matpar entry of enum type %d\n", this->matpar_enum);
	else                  printf("   matpar reference is NULL\n");
	
}


/*--------------------------------------------------
	PentaElementSizeOfInDoubles
  --------------------------------------------------*/

int PentaElementSizeOfInDoubles(void* vpthis)
{
	/* vpthis for polymorphic function compatibility */

	PentaElement* this = (PentaElement*)vpthis;
	int struct_doubles=0;

	struct_doubles = 
		(int)ceil((float)sizeof(*this)/(float)sizeof(double));

	#ifdef _DEBUG_
	_printf_("\n   PentaElementSizeOfInDoubles diagnostics:\n");
	_printf_("      returning %d doubles\n", struct_doubles);
	#endif

	return struct_doubles;
}

/*--------------------------------------------------
	PentaElementShrinkWrap
  --------------------------------------------------*/

void PentaElementShrinkWrap(void* to, void* vpthis)
{
	/* no "extended" data means a simple copy will do: */

	memcpy( to, vpthis, PentaElementSizeOfInDoubles(vpthis)*sizeof(double));
}


/*--------------------------------------------------
	PentaElementUnwrap
  --------------------------------------------------*/

int  PentaElementUnwrap(void* vpthis)
{
	/* though no extended data means no internal pointers to
	reestablish, re-null pointers whose values must properly come
	from a call to Configure(): */

	int i;
	PentaElement* this = (PentaElement*)vpthis;

	for ( i=0; i<numgrids; i++) this->pg[i] = NULL;
	
	this->matice = NULL;
	this->matpar = NULL;

	return 1;
}

/*--------------------------------------------------
	PentaElementMarshall
  --------------------------------------------------*/

void PentaElementMarshall( char* *pbuffer, void* vpthis, int* size)
{
	/* vpthis for polymorphic function compatibility */
	PentaElement* this = (PentaElement*)vpthis;

	char* buffer=*pbuffer;

	int i, j, size_penta=0; 
	int ig;

	int num_ints=0, num_doubles=0;


	/* first marshall the embedded Penta object: ... */

	PentaMarshall( &buffer, &this->penta, &size_penta);


	/* ... then marshall the PentaElement member data: */

	for ( i=0; i<numgrids; i++) {	
		cellmar( &this->internalgrid_indices[i], &buffer, INTEGER_FLAG);num_ints++;
		cellmar( &this->internalgrid_indicesb[i], &buffer, INTEGER_FLAG); num_ints++;
		cellmar( &this->pg[i], &buffer, INTEGER_FLAG); num_ints++;
	}

	cellmar( &this->matice_index, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->matice_enum, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->matice, &buffer, INTEGER_FLAG); num_ints++;

	cellmar( &this->matpar_index, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->matpar_enum, &buffer, INTEGER_FLAG); num_ints++;
	cellmar( &this->matpar, &buffer, INTEGER_FLAG); num_ints++;
	
	*size = size_penta + num_ints*sizeof(int) + num_doubles*sizeof(double);
	*pbuffer = buffer;

	return;
}

/*--------------------------------------------------
	PentaElementDemarshall
  --------------------------------------------------*/

void* PentaElementDemarshall( DataSet* ds, char* *pbuffer, int machine_flag)
{ 
	int i, j, ig;
	Penta* ppenta=NULL;	/* for embedded object */

	char* buffer=*pbuffer; 
	PentaElement* this=NULL;


	/* first demarshall the embedded Penta object: ... */

	ppenta = PentaDemarshall( NULL, &buffer, machine_flag);
	/* create the derived object: */
	this = NewPentaElement( ppenta);
	/* and free penta now that it's embedded in the derived object: */
	DeletePenta( (void**)&ppenta);


	/* ... then demarshall the PentaElement member data: */

	for ( i=0; i<numgrids; i++) {
		celldem( &this->internalgrid_indices[i], &buffer, INTEGER_FLAG, machine_flag);
		celldem( &this->internalgrid_indicesb[i], &buffer, INTEGER_FLAG, machine_flag);
		celldem( &this->pg[i], &buffer, INTEGER_FLAG, machine_flag);
	}

	celldem( &this->matice_index, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->matice_enum, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->matice, &buffer, INTEGER_FLAG, machine_flag);

	celldem( &this->matpar_index, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->matpar_enum, &buffer, INTEGER_FLAG, machine_flag);
	celldem( &this->matpar, &buffer, INTEGER_FLAG, machine_flag);
	
	
	/*Set pointers to NULL :*/
	for (i=0;i<numgrids;i++){
		this->pg[i]=NULL;
	}
	this->matice=NULL;
	this->matpar=NULL;
	
	if (ds) PentaElementAddToDataSet( ds, this);
	*pbuffer=buffer;  

	return (void*)this;
}

/*--------------------------------------------------
	PentaElementGetID
  --------------------------------------------------*/

int PentaElementGetID( void* vpthis )
{
	/* vpthis for polymorphic function compatibility */
	PentaElement* this = (PentaElement*)vpthis;

	return PentaGetID( &this->penta);
}


/*--------------------------------------------------
	PentaElementSetVirtualFunctions
  --------------------------------------------------*/

int PentaElementSetVirtualFunctions( void* vp)
{

	/* VirtualFunctions implemented as a "friend" class: */

	/* void* for polymorphic function compatibility */
	VirtualFunctions* vf = (VirtualFunctions*)vp;
	
	if ( VirtualFunctionsInit(vf) ) {

		/* set PentaElement-specific virtual function pointers: */

		/* general: */

		vf->Configure        = &PentaElementConfigure;
		vf->GetName          = &PentaElementGetName;

		/* element matrix generation: */
		vf->CreateKMatrix    = &PentaElementCreateKMatrix;

		/* loads vector generation: */
		vf->CreatePVector    = &PentaElementCreatePVector;

		/* velocity difference generation: */
		vf->CreateDuVector    = &PentaElementCreateDuVector;
		vf->CreateGradjVectors    = &PentaElementCreateGradjVectors;

		/* velocity misfit: */
		vf->Misfit=&PentaElementMisfit;

		return 1;
	}
	else
		return 0;
}


/*--------------------------------------------------
	PentaElementAddToDataSet
  --------------------------------------------------*/

int PentaElementAddToDataSet( DataSet* dataset, PentaElement* this)
{
	/*

	Provided as an alternate interface to DataSetAddObject.

	In the c++ world, the dataset could just add an object to its list
	using a dataset member function and a pointer to the object's base
	class.

	In the c world, however, only the object itself is capable of telling
	the dataset its type, hence the structure here; a call to a generic
	DataSetAddObject function via an object "member function."

	*/

	DataSetAddObject( dataset, PentaElementEnum(), (void*)this,
		&DeletePentaElement, &PentaElementEcho, &PentaElementSizeOfInDoubles,
		&PentaElementShrinkWrap, &PentaElementUnwrap,
		&PentaElementMarshall,
		&PentaElementGetID, &PentaElementSetVirtualFunctions);
	
	return 1;
}



/*--------------------------------------------------
    PentaElementGetName
  --------------------------------------------------*/

char* PentaElementGetName( void* vpthis)
{
	PentaElement* this = (PentaElement*)vpthis;

	#ifdef _DEBUG_
	_printf_("virtual function: PentaElementGetName\n");
	#endif

	return PentaGetName( &this->penta);
}

/*--------------------------------------------------
	PentaElementGetNumberOfGrids
  --------------------------------------------------*/

int PentaElementGetNumberOfGrids( void* vpthis )
{
	return numgrids;
}

/*--------------------------------------------------
	PentaElementConfigure
  --------------------------------------------------*/

int PentaElementConfigure( void* vpthis, int num_datasets, ...)
{
	
	/* set up data connections to all necessary, related objects: */

	PentaElement* this = (PentaElement*)vpthis;

	int	i,j, found, foundgrid, noerr=1, first=1;

	/* expected DataSet input: */

	DataSet*	pgpdt=NULL;
	DataSet*	pgpdtb=NULL;
	DataSet*	pmpt=NULL;

	/* variable argument list handling: */

	va_list args;
	DataSet* *arglist=NULL;

	#ifdef _DEBUG_
	_printf_("virtual function: PentaElementConfigure\n");
	#endif

	if (this) {

		if (!num_datasets) {

			_printf_("   %s error:\n", __func__);
			_printf_("   null input for number of datasets;\n");
			_printf_("   could not establish grid, property  and material data connections.\n");
			return 0;
		}
		else {

			/* read, save dataset pointers for data connection operations: */
			arglist = xmalloc( num_datasets*sizeof(DataSet*));
			va_start( args, num_datasets);
			for ( i=0; i<num_datasets; i++) {
				*(arglist+i) = va_arg(args,DataSet*);
			}
			va_end(args);
		}

		#ifdef _DEBUG_
		_printf_("   variable argument list statistics:\n");
		_printf_("   input number of DataSets = %d\n", num_datasets);
		_printf_("   enum types of input DataSets:");
		for ( i=0; i<num_datasets; i++)  _printf_("   %d", DataSetGetType(*(arglist+i)));
		_printf_("\n");
		#endif


		/*
			establish grid, property and material object data connections
			for this PentaElement object:
		*/

		/*Reinitialize this->pg[i]*/
		for (i=0;i<numgrids;i++){
			this->pg[i]=NULL;
		}

		/* grid connections based on DataSet of type "gpdt": */

		#ifndef _PARALLEL_
		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( GpdtEnum() == DataSetGetType(*(arglist+i)) ) {
				pgpdt = *(arglist+i);
				found = 1;
				break;
			}
		}
		if (found) {

			int	current_grid_id;
			int* pgrid_id_list = PentaGetGridIDPtr(&this->penta);

			int i;

			for ( i=0; i<PentaElementGetNumberOfGrids(this); i++) {

				if ( (current_grid_id=pgrid_id_list[i]) ) {	/* assign and test presence */

					/* if grid pointer information has been established once already
					(e.g., by a previous module or operation), grid index information
					may already be present and valid.  check to see if corresponding
					pointer information can be reestablished without an expensive
					seek operation: */

					if ( UNDEF != (int)this->internalgrid_indices[i] ) {

						/* does index still point to a valid grid? (i.e., the one with
						the same id): */

						if ( InternalGridEnum() == DataSetGetEnum( pgpdt, this->internalgrid_indices[i])
							 &&
						     current_grid_id == DataSetGetObjectID( pgpdt, this->internalgrid_indices[i]) ) {

							/* great, still valid!  since we have the index position in
							gpdt, just use it to get, save, the grid pointer: */

							this->pg[i] = DataSetGetObjectPtr( pgpdt, this->internalgrid_indices[i]);
						}
					}

					/* if we haven't yet "recovered" the InternalGrid object pointer,
					go seek it out: */

					if ( !this->pg[i] ) {

						noerr *= DataSetSeek( (void**)(this->pg+i),	/* cast because some compilers recognise as */
																	/* ptr to ptr to struct otherwise           */
									(this->internalgrid_indices+i) ,
									pgpdt, current_grid_id, 1, InternalGridEnum());
						
						if (!noerr) {
							if (first) {
								_printf_("   %s error:\n", __func__);
								first = 0;
							}
							_printf_("   could not find grid with id of %d for element %d of type \"%s\"\n",
								current_grid_id, PentaGetID(&this->penta), PentaGetName(&this->penta));
						}
					}
				}
			}
		}
		else {

			if (first) {
				_printf_("   %s error:\n", __func__);
				first=0;
			}
			_printf_("   could not find DataSet of type \"gpdt\", grid object\n");
			_printf_("   pointers not established for penta element id = %d\n",
				PentaElementGetID(this));
			noerr = 0;
		}
		#else //ifdef _PARALLEL_

		/*Configure the element when we are running on a parallel cluster: in this case, 
		 we  have partitioned the elements and the grids correspondingly. Every node has 
		 its own est, its own bgpdt (which owns the internal grids of the partition) and 
		 a common bgpdtb (b for boundaries of the partition). The boundary grids have to be 
		 dealt with. What happens is that bgpdt might not own all the grids referenced by this 
		 element, bgpdtb might own some of these grids. So we have to do a double configuration
		 sort of: 
		 */

		/*First, we need to recover pgpdt and pgpdtb from the arglist: */
		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( GpdtEnum() == DataSetGetType(*(arglist+i)) ) {
				if (found==0){
					/*This is the first GpdtEnum type dataset encountered, 
					 *ie  the internal grids: */
					pgpdt = *(arglist+i);
					found = 1;
					continue;
				}
				if (found==1){
					/*This is the second GpdtEnum type dataset encountered, 
					 *ie  the boundary grids: */
					pgpdtb = *(arglist+i);
					found = 2;
					continue;
				}
			}
		}
		
		if (found==2) {

			int	current_grid_id;
			int* pgrid_id_list = PentaGetGridIDPtr(&this->penta);

			for ( i=0; i<PentaElementGetNumberOfGrids(this); i++) {

				if ( (current_grid_id=pgrid_id_list[i]) ) {	/* assign and test presence */

					/* if grid pointer information has been established once already
					(e.g., by a previous module or operation), grid index information
					may already be present and valid.  check to see if corresponding
					pointer information can be reestablished without an expensive
					seek operation: */

					if ( UNDEF != (int)this->internalgrid_indices[i] ) {

						/* does index still point to a valid grid? (i.e., the one with
						the same id): */

						if ( InternalGridEnum() == DataSetGetEnum( pgpdt, this->internalgrid_indices[i])
							 &&
						     current_grid_id == DataSetGetObjectID( pgpdt, this->internalgrid_indices[i]) ) {

							/* great, still valid!  since we have the index position in
							gpdt, just use it to get, save, the grid pointer: */

							this->pg[i] = DataSetGetObjectPtr( pgpdt, this->internalgrid_indices[i]);
						}
					}
					if ( UNDEF != (int)this->internalgrid_indicesb[i] ) {

						/* does index still point to a valid grid? (i.e., the one with
						the same id): */

						if ( InternalGridEnum() == DataSetGetEnum( pgpdtb, this->internalgrid_indices[i])
							 &&
						     current_grid_id == DataSetGetObjectID( pgpdtb, this->internalgrid_indices[i]) ) {

							/* great, still valid!  since we have the index position in
							gpdtb, just use it to get, save, the grid pointer: */

							this->pg[i] = DataSetGetObjectPtr( pgpdtb, this->internalgrid_indices[i]);
						}
					}

					/* if we haven't yet "recovered" the InternalGrid object pointer,
					go seek it out: */

					if ( !this->pg[i] ) {

						foundgrid = DataSetSeek( (void**)(this->pg+i),	/* cast because some compilers recognise as */
																	/* ptr to ptr to struct otherwise           */
									(this->internalgrid_indices+i) ,
									pgpdt, current_grid_id, 1, InternalGridEnum());
						if(!foundgrid){
							/*We did not find this grid in pgpdt, go find it in pgpdtb:*/
							foundgrid = DataSetSeek( (void**)(this->pg+i),	/* cast because some compilers recognise as */
																	/* ptr to ptr to struct otherwise           */
									(this->internalgrid_indicesb+i) ,
									pgpdtb, current_grid_id, 1, InternalGridEnum());
						}

						if(!foundgrid){
							/*We could not find our grid in pgpdt and pgpdtb, this is not good!:*/
							noerr=0;
						}
						if (!noerr) {
							if (first) {
								_printf_("   %s error:\n", __func__);
								first = 0;
							}
							_printf_("   could not find grid with id of %d for element %d of type \"%s\"\n",
								current_grid_id, PentaGetID(&this->penta), PentaGetName(&this->penta));
						}
					}
				}
			}
		}
		else {

			if (first) {
				_printf_("   %s error:\n", __func__);
				first=0;
			}
			_printf_("   could not find DataSets of type \"gpdt\", grid object\n");
			_printf_("   pointers not established for penta element id = %d\n",
				PentaElementGetID(this));
			noerr = 0;
		}

		#endif


		/* matice property connections based on DataSet of type "mpt": */

		found=0;
		for ( i=0; i<num_datasets; i++) {
			if (*(arglist+i)==NULL)continue;
			if ( MptEnum() == DataSetGetType(*(arglist+i)) ) {
				pmpt  = *(arglist+i);
				found = 1;
				break;
			}
		}
		if (found) {

			int	current_matice_id = PentaGetMaterialID(&this->penta);

			/* as for grids, check to see if a nonzero matice index still
			points to the matice entry of interest in mpt, otherwise do
			a seek for it: */

			if ( this->matice_enum && this->matice_index ) {
			

				if ( this->matice_enum == DataSetGetEnum( pmpt, this->matice_index)
					 &&
					 current_matice_id == DataSetGetObjectID( pmpt, this->matice_index) ) {

					/* index is still valid; just get matice entry pointer directly: */

					this->matice = DataSetGetObjectPtr( pmpt, this->matice_index);
				}
			}

			/* if we haven't yet "recovered" the matice object pointer, go seek it out: */

			if ( NULL == this->matice) {

				noerr *= DataSetSeek( &this->matice, &this->matice_index,
							pmpt, current_matice_id, 1, MaticeEnum() );	/* 0 to generalise later to Pcomp, etc. ... */

				if (!noerr) {
					if (first) {
						_printf_("   %s error:\n", __func__);
						first = 0;
					}
					_printf_("   could not find matice entry with id of %d for element %d of type \"%s\"\n",
						current_matice_id, PentaGetID(&this->penta), PentaGetName(&this->penta));
				}
				else {

					/* though the answer for now is just MaticeEnum(), for generality go ahead and
					ask the question, "what type of matice has been referenced?" */

					this->matice_enum = DataSetGetEnum( *(arglist+i), this->matice_index);
				}
			}
			
			
			
			int	current_matpar_id = PentaGetMaterialParID(&this->penta);

			/* as for grids, check to see if a nonzero matpar index still
			points to the matpar entry of interest in mpt, otherwise do
			a seek for it: */

			if ( this->matpar_enum && this->matpar_index ) {
			

				if ( this->matpar_enum == DataSetGetEnum( pmpt, this->matpar_index)
					 &&
					 current_matpar_id == DataSetGetObjectID( pmpt, this->matpar_index) ) {

					/* index is still valid; just get matpar entry pointer directly: */

					this->matpar = DataSetGetObjectPtr( pmpt, this->matpar_index);
				}
			}

			/* if we haven't yet "recovered" the matpar object pointer, go seek it out: */

			if ( NULL == this->matpar) {

				noerr *= DataSetSeek( &this->matpar, &this->matpar_index,
							pmpt, current_matpar_id, 1, MatparEnum() );	/* 0 to generalise later to Pcomp, etc. ... */

				if (!noerr) {
					if (first) {
						_printf_("   %s error:\n", __func__);
						first = 0;
					}
					_printf_("   could not find matpar entry with id of %d for element %d of type \"%s\"\n",
						current_matpar_id, PentaGetID(&this->penta), PentaGetName(&this->penta));
				}
				else {

					/* though the answer for now is just MatparEnum(), for generality go ahead and
					ask the question, "what type of matpar has been referenced?" */

					this->matpar_enum = DataSetGetEnum( *(arglist+i), this->matpar_index);
				}
			}
		}
		else {

			if (first) {
				_printf_("   %s error:\n", __func__);
				first=0;
			}
			_printf_("   could not find DataSet of type \"mpt\"; matice  and matpar relationships\n");
			_printf_("   not established for penta element id = %d\n",
				PentaElementGetID(this));
			noerr = 0;
		}
	}

	return noerr;
}
/*--------------------------------------------------
    PentaElementGetGridIDPtr
  --------------------------------------------------*/

int* PentaElementGetGridIDPtr( void* vpthis)
{
	PentaElement* this = (PentaElement*)vpthis;

	#ifdef _DEBUG_
	_printf_("virtual function: PentaElementGetGridIDPtr\n");
	#endif

	return PentaGetGridIDPtr( &this->penta);
}


/*--------------------------------------------------
	PentaElementCreateKMatrix
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreateKMatrix"
int PentaElementCreateKMatrix( ElemMatrix* *pKe_gg, ElemMatrix** pKTe_gg, void* vpthis, ParameterInputs* inputs, int K_flag, int KT_flag, int analysis_type,int sub_analysis_type){


	int noerr=1;

	
	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if ((analysis_type==DiagnosticHorizAnalysisEnum()) || (analysis_type==ControlAnalysisEnum())){
		noerr=PentaElementCreateKMatrixDiagnosticHoriz( pKe_gg, vpthis, inputs);
	}
	else if(analysis_type==DiagnosticBaseVertAnalysisEnum()){
		noerr=PentaElementCreateKMatrixDiagnosticBaseVert( pKe_gg, vpthis, inputs);
	}
	else if(analysis_type==DiagnosticVertAnalysisEnum()){
		noerr=PentaElementCreateKMatrixDiagnosticVert( pKe_gg, vpthis, inputs);
	}
	else if(analysis_type==ThermalSteadyAnalysisEnum()|| (analysis_type==ThermalTransientAnalysisEnum())){
		noerr=PentaElementCreateKMatrixThermal( pKe_gg, vpthis, inputs);
	}
	else if(analysis_type==MeltingAnalysisEnum()){
		noerr=PentaElementCreateKMatrixMelting( pKe_gg, vpthis, inputs);
	}
	else if(analysis_type==PrognosticAnalysisEnum()){
		noerr=PentaElementCreateKMatrixPrognostic( pKe_gg, vpthis, inputs);
	}
	else{
		_printf_("%s%s%i%s\n",__FUNCT__," error message: analysis type",analysis_type," not supported yet");
		noerr=0;
	}
	return noerr;
}

/*--------------------------------------------------
	PentaElementCreateKMatrixDiagnosticHoriz
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreateKMatrixDiagnosticHoriz"
int PentaElementCreateKMatrixDiagnosticHoriz( ElemMatrix* *pKe_gg, void* vpthis, ParameterInputs* inputs){


	int             i,j,il,counter;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;
	Matice      * matice = NULL;
	Matpar      * matpar = NULL;

	/* local declarations */

	/* output: */
	ElemMatrix*     Ke_gg        = NULL;
	
	
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;


	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double T_bg_list[numgrids][9];
	/* 3d gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights           =  NULL;
	double* vert_gauss_weights           =  NULL;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_l1l2l3l4[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	double  gauss_weight;
	
	/* 2d gaussian point: */
	int     num_gauss2d;
	double* first_gauss_area_coord2d  =  NULL;
	double* second_gauss_area_coord2d =  NULL;
	double* third_gauss_area_coord2d  =  NULL;
	double* gauss_weights2d=NULL;
	double  gauss_l1l2l3[3];

	/*drag: */
	double  pcoeff,qcoeff;
	double  rcoeff,scoeff;
	double  velocity_x,velocity_y,velocity_mag;

	/* material data: */
	double viscosity; //viscosity
	double B_param;
	double gravity=9.8;
	double rho_ice,rho_water;
	
	/* strain rate: */
	double epsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/

	/* matrices: */
	double B[5][NDOF2*numgrids];
	double Bprime[5][NDOF2*numgrids];
	double L[2][NDOF2*numgrids];
	double D[5][5]={{ 0,0,0,0,0 },{0,0,0,0,0},{0,0,0,0,0},{0,0,0,0,0},{0,0,0,0,0}};              // material matrix, simple scalar matrix.
	double D_scalar;

	double DL[2][2]={{ 0,0 },{0,0}}; //for basal drag
	double DL_scalar;

	double Ke_gg_gaussian[NDOF2*numgrids][NDOF2*numgrids]; //stiffness matrix evaluated at the gaussian point.
	
	double Ke_gg_drag_gaussian[NDOF2*3][NDOF2*3]; //stiffness matrix contribution from drag, in 2d
	double Jdet;
	
	/*slope: */
	double  slope[NDOF2];
	double  slope_magnitude;

	/*input parameters for structural analysis (diagnostic): */
	double* thickness_param=NULL;
	double thickness_list[numgrids];
	double thickness;
	double* surface_param=NULL;
	double surface_list[numgrids];
	double* bed_param=NULL;
	double bed_list[numgrids];
	double* flow_law_param=NULL;
	double B_list[numgrids];
	double* basal_drag=NULL;
	double K_list[numgrids];
	double* velocity=NULL;
	double vxvy_list[numgrids][2];
	double* temperature_average_param=NULL;
	double  temperature_average_list[numgrids];
	double  temperature_average;
	
	//tria element parameters
	double  alpha2_list[3];
	double  alpha2;

	double MAXSLOPE=.06; // 6 %
	double MOUNTAINKEXPONENT=10;

	/*Collapsed formulation: */
	Tria*  tria=NULL;
	TriaElement* triaelement=NULL;

	/* The number of g-set dof for this element is 2*6 (vx and vy, horizontal velocities for ice, and 6 grids 
	 * per element: */
	numdof=numgrids*NDOF2;

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;
	matice=this->matice;
	matpar=this->matpar;

	/*Figure out if this pentaelem is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the stiffness matrix. */

	if ((this->penta.collapse==1) && (this->penta.onbed==0)){
		/*This element should be collapsed, but this element is not on the bedrock, therefore all its 
		 * dofs have already been frozen! Do nothing: */
		*pKe_gg=NULL;
		return noerr;
	}
	else if ((this->penta.collapse==1) && (this->penta.onbed==1)){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 *and use its CreateKMatrix functionality to return an elementary stiffness matrix: */
		tria=NewTria();
		TriaInit(tria);
            
		strcpy(tria->name,"CTRICE3");
		tria->eid=this->penta.eid;
		tria->mid=this->penta.mid;
		for(i=0;i<3;i++){
			tria->g[i]=this->penta.g[i]; //we take the first 3 grids of the penta to build the tria.
			tria->h[i]=this->penta.h[i]; 
			tria->s[i]=this->penta.s[i];
			tria->b[i]=this->penta.b[i];
			tria->k[i]=this->penta.k[i];
		}

		/*diverse:*/
		tria->theta_type=0;
		tria->theta.angle=0;
		tria->zoffs=0;

		//friction
		tria->p=this->penta.p;
		tria->q=this->penta.q;
		tria->shelf=this->penta.shelf;
		tria->friction_type=this->penta.friction_type;

		/*type of fitting? : */
		tria->fit=this->penta.fit;
		tria->meanvel=this->penta.meanvel;
		tria->epsvel=this->penta.epsvel;

		/*diverse:*/
		tria->acceleration=0;

		/*Now that the tria object is created, spawn an element out of it: */
		triaelement = NewTriaElement( tria);

		/*Transfer configuration of penta to configuration of tria: */
		for(i=0;i<3;i++){
			triaelement->internalgrid_indices[i]=this->internalgrid_indices[i];
			triaelement->internalgrid_indicesb[i]=this->internalgrid_indicesb[i];
			triaelement->pg[i]=this->pg[i];
		}
		triaelement->matice_index=this->matice_index;
		triaelement->matice_enum=this->matice_enum;
		triaelement->matice=this->matice;

		triaelement->matpar_index=this->matpar_index;
		triaelement->matpar_enum=this->matpar_enum;
		triaelement->matpar=this->matpar;


		#ifdef _DEBUGELEMENTS_
		if(my_rank==RANK && PentaElementGetID(this)==ELID){ printf("El id %i Rank %i TriaElement echo in collapsed Penta: \n",ELID,RANK); TriaElementEcho(triaelement); }
		#endif

		/*Ok, now triaelement is correctly configured, call on its method CreateKMatrix: */
		noerr=TriaElementCreateKMatrixDiagnosticHoriz( pKe_gg, (void*)triaelement, inputs);

		/*Now delete tria and triaelement: */
		DeleteTriaElement((void**)&triaelement);
		DeleteTria((void**)&tria);

		return noerr;
	}
	else{

		/*Implement standard penta element: */

		/* Allocate new stiffness element  matrix: */
		Ke_gg=NewElemMatrix(numdof);

		/*Recover material: */
		gravity=matpar->g;
		rho_ice=matpar->rho_ice;
		rho_water=matpar->rho_water;

		/*recover extra inputs from users, at current convergence iteration: */
		basal_drag=ParameterInputsRecover(inputs,"drag");
		thickness_param=ParameterInputsRecover(inputs,"thickness");
		surface_param=ParameterInputsRecover(inputs,"surface");
		bed_param=ParameterInputsRecover(inputs,"bed");
		temperature_average_param=ParameterInputsRecover(inputs,"temperature_average");
		flow_law_param=ParameterInputsRecover(inputs,"B");
		velocity=ParameterInputsRecover(inputs,"velocity"); 

		/* Get all element grid data */
		noerr = GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, this->pg, numgrids, 1 );
		if(!noerr){
			_printf_("%s%s\n",__FUNCT__," error message: could not get element grid data!");
			goto cleanup_and_return;
		}
		
		/* Build the row index vector (same as column index vector). 
		 * At the same time, get element exterior data for each grid: */
		for (i=0;i<numgrids;i++){
			/* We use the structural dof list, from which we keep only the first two dof (x for vx, and y for vy).
			 * The other degrees of freedom will be spc'd for now.*/
			structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
			for (j=0;j<NDOF2;j++){
				dof=structural_dof_list[j];
				*(Ke_gg->row_indices+i*NDOF2+j)=dof;
				if(velocity){
					vxvy_list[i][j]=*(velocity+dof);
				}
				else{
					vxvy_list[i][j]=0;
				}
			}
			dof=structural_dof_list[0];
			if(flow_law_param){
				B_list[i] = *(flow_law_param+dof);
			}
			if(basal_drag){
				K_list[i]=*(basal_drag+dof);
			}
			else{
				K_list[i]=this->penta.k[i];
			}
			if(thickness_param){
				thickness_list[i]=*(thickness_param+dof);
			}
			else{
				thickness_list[i]=this->penta.h[i];
			}
			if(surface_param){
				surface_list[i]= *(surface_param+dof);
			}
			else{
				surface_list[i]= this->penta.s[i];
			}
			if(bed_param){
				bed_list[i]=*(bed_param+dof);
			}
			else{
				bed_list[i]= this->penta.b[i];
			}

			if(temperature_average_param) temperature_average_list[i]=*(temperature_average_param+dof);
		}

		#ifdef _DEBUGELEMENTS_
		if(my_rank==RANK && PentaElementGetID(this)==ELID){ 
			printf("El id %i Rank %i PentaElement input list before gaussian loop: \n",ELID,RANK); 
			printf("   rho_ice: %g\n",rho_ice);
			printf("   gravity: %g\n",gravity);
			printf("   rho_water:%g \n",rho_water);
			printf("   Velocity: \n");
			for (i=0;i<numgrids;i++){
				printf("      grid %i  [%g,%g,%g]\n",i,vxvy_list[i][0],vxvy_list[i][1],vxvy_list[i][2]);
			}
			printf("   B [%g %g %g %g %g %g]\n",B_list[0],B_list[1],B_list[2],B_list[3],B_list[4],B_list[5]);
			printf("   K [%g %g %g %g %g %g]\n",K_list[0],K_list[1],K_list[2],K_list[3],K_list[4],K_list[5]);
			printf("   thickness [%g %g %g %g %g %g]\n",thickness_list[0],thickness_list[1],thickness_list[2],thickness_list[3],thickness_list[4],thickness_list[5]);
			printf("   surface [%g %g %g %g %g %g]\n",surface_list[0],surface_list[1],surface_list[2],surface_list[3],surface_list[4],surface_list[5]);
			printf("   bed [%g %g %g %g %g %g]\n",bed_list[0],bed_list[1],bed_list[2],bed_list[3],bed_list[4],bed_list[5]);
			printf("   temperature_average [%g %g %g %g %g %g]\n",temperature_average_list[0],temperature_average_list[1],temperature_average_list[2],temperature_average_list[3],temperature_average_list[4],temperature_average_list[5]);
		}
		#endif


		/*Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		  get tria gaussian points as well as segment gaussian points. For tria gaussian 
		  points, order of integration is 2, because we need to integrate the product tB*D*B' 
		  which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
		  points, same deal, which yields 3 gaussian points.*/

		order_area_gauss=5;
		num_vert_gauss=5;

		GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);

		#ifdef _DEBUGGAUSS_
		if(my_rank==RANK && PentaElementGetID(this)==ELID){ 
			printf("El id %i Rank %i PentaElement gauss points\n",ELID,RANK); 
			for (i=0;i<num_area_gauss;i++){
				printf("   Area Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(area_gauss_weights+i));
			}
			for (i=0;i<num_vert_gauss;i++){
				printf("   Vert Gauss coord %i: %lf Weight: %lf\n",i,*(fourth_gauss_vert_coord+i),*(vert_gauss_weights+i));
			}	
		}
		#endif

		/* Start  looping on the number of gaussian points: */
		for (ig1=0; ig1<num_area_gauss; ig1++){
			for (ig2=0; ig2<num_vert_gauss; ig2++){
			
				/*Pick up the gaussian point: */
				gauss_weight1=*(area_gauss_weights+ig1);
				gauss_weight2=*(vert_gauss_weights+ig2);
				gauss_weight=gauss_weight1*gauss_weight2;
				
				
				gauss_l1l2l3l4[0]=*(first_gauss_area_coord+ig1); 
				gauss_l1l2l3l4[1]=*(second_gauss_area_coord+ig1);
				gauss_l1l2l3l4[2]=*(third_gauss_area_coord+ig1);
				gauss_l1l2l3l4[3]=*(fourth_gauss_vert_coord+ig2);


				/*Get strain rate from velocity: */
				
				PentaElementGetStrainRate(&epsilon[0],&vxvy_list[0][0],&xyz_list[0][0],gauss_l1l2l3l4);
				#ifdef _DEBUG_ 
					_printf_("Epsilon: %lf %lf %lf %lf %lf\n",epsilon[0],epsilon[1],epsilon[2],epsilon[3],epsilon[4]);
				#endif


				//Update material if temperature is provided.
				if(temperature_average_param){
					PentaElementGetParameterValue(&temperature_average, &temperature_average_list[0],gauss_l1l2l3l4);
					B_param=Paterson(temperature_average);
					MaticeSetFlowLawParameter(this->matice,B_param);
				}
				//Update material if B is provided.
				if(flow_law_param){
					PentaElementGetParameterValue(&B_param, &B_list[0],gauss_l1l2l3l4);
					#ifdef _DEBUG_ 
						_printf_("Flow law parameters: %lf\n", B_param);
					#endif
					MaticeSetFlowLawParameter(this->matice,B_param);
				}
				

				/*Get viscosity: */
				MaticeGetViscosity3d(&viscosity, this->matice, &epsilon[0]);
				#ifdef _DEBUG_ 
					_printf_("Element id %i Viscosity: \n", PentaElementGetID(this),viscosity);
				#endif

				/*Get B and Bprime matrices: */
				PentaElementGetB(&B[0][0], &xyz_list[0][0], gauss_l1l2l3l4);
				PentaElementGetBPrime(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3l4);

				/* Get Jacobian determinant: */
				PentaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3l4);

				/*Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
				  onto this scalar matrix, so that we win some computational time: */
				D_scalar=viscosity*gauss_weight*Jdet;
				for (i=0;i<5;i++){
					D[i][i]=D_scalar;
				}
		

				/*  Do the triple product tB*D*Bprime: */
				noerr=TripleMultiply( &B[0][0],5,numgrids*NDOF2,1,
						&D[0][0],5,5,0,
						&Bprime[0][0],5,numgrids*NDOF2,0,
						&Ke_gg_gaussian[0][0],0);

				
				/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
				for( i=0; i<Ke_gg->nrows; i++){
					for (j=0;j<Ke_gg->nrows;j++){
						*(Ke_gg->terms+Ke_gg->nrows*i+j)+=Ke_gg_gaussian[i][j];
					}
				}	
			} //for (ig2=0; ig2<num_vert_gauss; ig2++)
		} //for (ig1=0; ig1<num_area_gauss; ig1++)
				
		
	
		//Deal with 2d friction at the bedrock interface
		if((this->penta.onbed) && (this->penta.shelf==0)){

			/*Build a tria element using the 3 grids of the base of the penta. Then use 
			 * the tria functionality to build a friction stiffness matrix on these 3
			 * grids: */

			tria=NewTria();
			TriaInit(tria);
				
			strcpy(tria->name,"CTRICE3");
			tria->eid=this->penta.eid;
			tria->mid=this->penta.mid;
			for(i=0;i<3;i++){
				tria->g[i]=this->penta.g[i]; //we take the first 3 grids of the penta to build the tria.
				tria->h[i]=this->penta.h[i]; 
				tria->s[i]=this->penta.s[i];
				tria->b[i]=this->penta.b[i];
				tria->k[i]=this->penta.k[i];
			}

			/*diverse:*/
			tria->theta_type=0;
			tria->theta.angle=0;
			tria->zoffs=0;

			//friction
			tria->p=this->penta.p;
			tria->q=this->penta.q;
			tria->shelf=this->penta.shelf;
			tria->friction_type=this->penta.friction_type;

			/*type of fitting? : */
			tria->fit=this->penta.fit;
			tria->meanvel=this->penta.meanvel;
			tria->epsvel=this->penta.epsvel;

			/*diverse:*/
			tria->acceleration=0;

			/*Now that the tria object is created, spawn an element out of it: */
			triaelement = NewTriaElement( tria);

			/*Transfer configuration of penta to configuration of tria: */
			for(i=0;i<3;i++){
				triaelement->internalgrid_indices[i]=this->internalgrid_indices[i];
				triaelement->internalgrid_indicesb[i]=this->internalgrid_indicesb[i];
				triaelement->pg[i]=this->pg[i];
			}
			triaelement->matice_index=this->matice_index;
			triaelement->matice_enum=this->matice_enum;
			triaelement->matice=this->matice;

			triaelement->matpar_index=this->matpar_index;
			triaelement->matpar_enum=this->matpar_enum;
			triaelement->matpar=this->matpar;


			/*Build alpha2_list used by drag stiffness matrix*/
			if (triaelement->tria.friction_type==2){

				/*Allocate friction object: */
				Friction* friction=NewFriction();
						
				/*Initialize all fields: */
				friction->element_type=xmalloc((strlen("2d")+1)*sizeof(char));
				strcpy(friction->element_type,"2d");

				friction->gravity=gravity;
				friction->rho_ice=rho_ice;
				friction->rho_water=rho_water;
				friction->K=&K_list[0];
				friction->bed=&bed_list[0];
				friction->thickness=&thickness_list[0];
				friction->velocities=&vxvy_list[0][0];
				friction->p=triaelement->tria.p;
				friction->q=triaelement->tria.q;

				/*Compute alpha2_list: */
				FrictionGetAlpha2(&alpha2_list[0],friction);

				/*Erase friction object: */
				DeleteFriction(&friction);
			}
			
			/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
			GaussTria( &num_gauss2d, &first_gauss_area_coord2d, &second_gauss_area_coord2d, &third_gauss_area_coord2d, &gauss_weights2d, 2);

			/* Start  looping on the number of gaussian points: */
			for (ig=0; ig<num_gauss2d; ig++){
				/*Pick up the gaussian point: */
				gauss_weight=*(gauss_weights2d+ig);
				gauss_l1l2l3[0]=*(first_gauss_area_coord2d+ig); 
				gauss_l1l2l3[1]=*(second_gauss_area_coord2d+ig);
				gauss_l1l2l3[2]=*(third_gauss_area_coord2d+ig);

				/* Get Jacobian determinant: */
				TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
            
				//Get L matrix if viscous basal drag present:
				TriaElementGetL(&L[0][0], &xyz_list[0][0], gauss_l1l2l3,NDOF2);

				//Get alpha2 on current gaussian point
				TriaElementGetParameterValue(&alpha2, &alpha2_list[0],gauss_l1l2l3);
				#ifdef _DEBUG_ 
					_printf_("Element id %i Alpha2: %lf\n", TriaElementGetID(this),alpha2);
				#endif

				DL_scalar=alpha2*gauss_weight*Jdet;
				for (i=0;i<2;i++){
					DL[i][i]=DL_scalar;
				}
				
				/*  Do the triple producte tL*D*L: */
				TripleMultiply( &L[0][0],2,6,1,
							&DL[0][0],2,2,0,
							&L[0][0],2,6,0,
							&Ke_gg_drag_gaussian[0][0],0);

				/*Now add the drag only on the first 3 grids, ie only the first 6 dofs: */
				for( i=0; i<6;i++){
					for (j=0;j<6;j++){
						*(Ke_gg->terms+Ke_gg->nrows*i+j)+=Ke_gg_drag_gaussian[i][j];
					}
				}
			} //for (ig=0; ig<num_gauss2d; ig++)
		} //if((this->penta.onbed) && (this->penta.shelf==0))
	} //else if ((this->tria.collapse==1) && (this->tria.onbed==1))
		
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
	xfree((void**)&first_gauss_area_coord2d);
	xfree((void**)&second_gauss_area_coord2d);
	xfree((void**)&third_gauss_area_coord2d);
	xfree((void**)&gauss_weights2d);

	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build stiffness matrix.");
		return 0;
	}
	else{
		#ifdef _DEBUG_ 
			ElemMatrixEcho(Ke_gg);
		#endif

					
		/*Assign output pointer: */
		*pKe_gg=Ke_gg;
		
		return 1;
	}
}

/*--------------------------------------------------
	PentaElementCreateKMatrixDiagnosticBaseVert
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreateKMatrixDiagnosticBaseVert"
int PentaElementCreateKMatrixDiagnosticBaseVert( ElemMatrix* *pKe_gg, void* vpthis, ParameterInputs* inputs){

	int             i,j;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;

	/* local declarations */

	/* output: */
	ElemMatrix*     Ke_gg        = NULL;
	
	int             numgrids2d=3;
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	/* grid data: */
	int    cid_list[numgrids2d];
	double xyz_list[numgrids2d][3];
	double T_bg_list[numgrids2d][9];
	
	/* 2d gaussian point: */
	int     num_gauss2d;
	double* first_gauss_area_coord2d  =  NULL;
	double* second_gauss_area_coord2d =  NULL;
	double* third_gauss_area_coord2d  =  NULL;
	double* gauss_weights2d=NULL;
	double  gauss_l1l2l3[3];
	double  gauss_weight;
	int     ig;

	/* matrices: */
	double L[1][3];
	double DL_scalar;

	double Ke_gg_gaussian[3][3]; //stiffness matrix evaluated at the gaussian point.
	double Jdet;
	/*Collapsed formulation: */
	Tria*  tria=NULL;
	TriaElement* triaelement=NULL;

	/* The number of g-set dof for this element is 1*3 (vz on bedrock grids) : */
	numdof=numgrids2d*NDOF1;

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;

	/*We are dealing with a collapsed formulation on the lower triangle of the penta, 
	  only on the bedrock.:*/

	if (this->penta.onbed==0){
		*pKe_gg=NULL;
		return noerr;
	}
	else{
		/*Build tria element at the base: */
		tria=NewTria();
		TriaInit(tria);
            
		strcpy(tria->name,"CTRICE3");
		tria->eid=this->penta.eid;
		tria->mid=this->penta.mid;
		for(i=0;i<3;i++){
			tria->g[i]=this->penta.g[i]; //we take the first 3 grids of the penta to build the tria.
			tria->h[i]=this->penta.h[i]; 
			tria->s[i]=this->penta.s[i];
			tria->b[i]=this->penta.b[i];
			tria->k[i]=this->penta.k[i];
		}

		/*diverse:*/
		tria->theta_type=0;
		tria->theta.angle=0;
		tria->zoffs=0;

		//friction
		tria->p=this->penta.p;
		tria->q=this->penta.q;
		tria->shelf=this->penta.shelf;
		tria->friction_type=this->penta.friction_type;

		/*type of fitting? : */
		tria->fit=this->penta.fit;
		tria->meanvel=this->penta.meanvel;
		tria->epsvel=this->penta.epsvel;

		/*diverse:*/
		tria->acceleration=0;

		/*Now that the tria object is created, spawn an element out of it: */
		triaelement = NewTriaElement( tria);

		/*Transfer configuration of penta to configuration of tria: */
		for(i=0;i<3;i++){
			triaelement->internalgrid_indices[i]=this->internalgrid_indices[i];
			triaelement->internalgrid_indicesb[i]=this->internalgrid_indicesb[i];
			triaelement->pg[i]=this->pg[i];
		}
		triaelement->matice_index=this->matice_index;
		triaelement->matice_enum=this->matice_enum;
		triaelement->matice=this->matice;

		triaelement->matpar_index=this->matpar_index;
		triaelement->matpar_enum=this->matpar_enum;
		triaelement->matpar=this->matpar;


		/* Allocate new stiffness element  matrix: */
		Ke_gg=NewElemMatrix(numdof);

		/* Get all element grid data */
		noerr = GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, triaelement->pg, numgrids2d, 1 );
		if(!noerr){
			_printf_("%s%s\n",__FUNCT__," error message: could not get element grid data!");
			goto cleanup_and_return;
		}
		
		/* Build the row index vector (same as column index vector). 
		 * At the same time, get element exterior data for each grid: */
		for (i=0;i<numgrids2d;i++){
			structural_dof_list= InternalGridGetStructuralDoflistPtr( triaelement->pg[i]);
			dof=structural_dof_list[2]; //computing vz
			*(Ke_gg->row_indices+i)=dof;
		}

		/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
		GaussTria( &num_gauss2d, &first_gauss_area_coord2d, &second_gauss_area_coord2d, &third_gauss_area_coord2d, &gauss_weights2d, 2);

		/* Start  looping on the number of gaussian points: */
		for (ig=0; ig<num_gauss2d; ig++){
			/*Pick up the gaussian point: */
			gauss_weight=*(gauss_weights2d+ig);
			gauss_l1l2l3[0]=*(first_gauss_area_coord2d+ig); 
			gauss_l1l2l3[1]=*(second_gauss_area_coord2d+ig);
			gauss_l1l2l3[2]=*(third_gauss_area_coord2d+ig);

			/* Get Jacobian determinant: */
			TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
			//Get L matrix if viscous basal drag present:
			TriaElementGetL(&L[0][0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

			DL_scalar=gauss_weight*Jdet;
			
			/*  Do the triple producte tL*D*L: */
			TripleMultiply( &L[0][0],1,3,1,
						&DL_scalar,1,1,0,
						&L[0][0],1,3,0,
						&Ke_gg_gaussian[0][0],0);

			/*Now add the drag only on the first 3 grids, ie only the first 6 dofs: */
			for( i=0; i<numdof;i++){
				for (j=0;j<numdof;j++){
					*(Ke_gg->terms+Ke_gg->nrows*i+j)+=Ke_gg_gaussian[i][j];
				}
			}
		} //for (ig=0; ig<num_gauss2d; ig++
		
		/*Now delete tria and triaelement: */
		DeleteTriaElement((void**)&triaelement);
		DeleteTria((void**)&tria);

	} //if (this->penta.onbed==0)
			
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord2d);
	xfree((void**)&second_gauss_area_coord2d);
	xfree((void**)&third_gauss_area_coord2d);
	xfree((void**)&gauss_weights2d);
	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build stiffness matrix.");
		return 0;
	}
	else{
		#ifdef _DEBUG_ 
			ElemMatrixEcho(Ke_gg);
		#endif

			
		/*Assign output pointer: */
		*pKe_gg=Ke_gg;
		
		return 1;
	}
}

/*--------------------------------------------------
	PentaElementCreateKMatrixPrognostic
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreateKMatrixPrognostic"
int PentaElementCreateKMatrixPrognostic( ElemMatrix* *pKe_gg, void* vpthis, ParameterInputs* inputs){

	int             i,j;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;

	/* local declarations */

	/* output: */
	ElemMatrix*     Ke_gg        = NULL;
	
	int             numgrids2d=3;
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	/* grid data: */
	int    cid_list[numgrids2d];
	double xyz_list[numgrids2d][3];
	double T_bg_list[numgrids2d][9];
	
	/* 2d gaussian point: */
	int     num_gauss2d;
	double* first_gauss_area_coord2d  =  NULL;
	double* second_gauss_area_coord2d =  NULL;
	double* third_gauss_area_coord2d  =  NULL;
	double* gauss_weights2d=NULL;
	double  gauss_l1l2l3[3];
	double  gauss_weight;
	int     ig;

	/*parameters: */
	double* velocity=NULL;
	double  vx_list[numgrids2d];
	double  vx;
	double  vy_list[numgrids2d];
	double  vy;
	double* dt_param=NULL;
	double  dt;

	/* matrices: */
	double L[1][3];
	double B[2][numgrids2d];
	double DL_scalar;
	double D[2][2];

	double Ke_gg_gaussian[3][3]; //stiffness matrix evaluated at the gaussian point.
	double Jdet;

	/* The number of g-set dof for this element is 1*3 (vz on bedrock grids) : */
	numdof=numgrids2d*NDOF1;

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;

	/*We are dealing with a collapsed formulation on the lower triangle of the penta, 
	  only on the bedrock.:*/
	if (this->penta.onbed==0){
		*pKe_gg=NULL;
		return noerr;
	}
	else{

		/* Allocate new stiffness element  matrix: */
		Ke_gg=NewElemMatrix(numdof);

		/* Get all element grid data */
		noerr = GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, this->pg, numgrids2d, 1 );
		if(!noerr){
			_printf_("%s%s\n",__FUNCT__," error message: could not get element grid data!");
			goto cleanup_and_return;
		}
	
		/*recover extra inputs from users, at current convergence iteration: */
		dt_param=ParameterInputsRecover(inputs,"dt");

		if(!dt_param){
			_printf_("%s%s\n",__FUNCT__," error message: missing dt parameter!");
			noerr=0; goto cleanup_and_return;
		}
		dt=*dt_param;

		if (this->penta.artdiff){
			velocity=ParameterInputsRecover(inputs,"velocity_average"); 
		}

		/* Build the row index vector (same as column index vector). 
		 * At the same time, get element exterior data for each grid: */
		for (i=0;i<numgrids2d;i++){
			structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
			dof=structural_dof_list[0]; //1st dof holds new thickness
			*(Ke_gg->row_indices+i)=dof;

			if(this->penta.artdiff){
				dof=structural_dof_list[0]; 
				vx_list[i]=*(velocity+dof);
				dof=structural_dof_list[0]; 
				vy_list[i]=*(velocity+dof);
			}
		}

		/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
		GaussTria( &num_gauss2d, &first_gauss_area_coord2d, &second_gauss_area_coord2d, &third_gauss_area_coord2d, &gauss_weights2d, 2);

		/* Start  looping on the number of gaussian points: */
		for (ig=0; ig<num_gauss2d; ig++){
			/*Pick up the gaussian point: */
			gauss_weight=*(gauss_weights2d+ig);
			gauss_l1l2l3[0]=*(first_gauss_area_coord2d+ig); 
			gauss_l1l2l3[1]=*(second_gauss_area_coord2d+ig);
			gauss_l1l2l3[2]=*(third_gauss_area_coord2d+ig);

			/* Get Jacobian determinant: */
			TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
			/*Get L matrix: */
			TriaElementGetL(&L[0][0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

			DL_scalar=gauss_weight*Jdet;
			
			/*  Do the triple producte tL*D*L: */
			TripleMultiply( &L[0][0],1,3,1,
						&DL_scalar,1,1,0,
						&L[0][0],1,3,0,
						&Ke_gg_gaussian[0][0],0);

			/*Now add the drag only on the first 3 grids, ie only the first 6 dofs: */
			for( i=0; i<numdof;i++){
				for (j=0;j<numdof;j++){
					*(Ke_gg->terms+Ke_gg->nrows*i+j)+=Ke_gg_gaussian[i][j];
				}
			}
		} //for (ig=0; ig<num_gauss2d; ig++
		
		/*Add artificial diffusivity if requested: */	
		if(this->penta.artdiff){
			for (ig=0; ig<num_gauss2d; ig++){
				/*Pick up the gaussian point: */
				gauss_weight=*(gauss_weights2d+ig);
				gauss_l1l2l3[0]=*(first_gauss_area_coord2d+ig); 
				gauss_l1l2l3[1]=*(second_gauss_area_coord2d+ig);
				gauss_l1l2l3[2]=*(third_gauss_area_coord2d+ig);

				/* Get Jacobian determinant: */
				TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
				
				TriaElementGetB_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
			
				/* Build the D matrix: */
				TriaElementGetParameterValue(&vx, &vx_list[0],gauss_l1l2l3);
				if(vx<0)vx=-vx;
				TriaElementGetParameterValue(&vy, &vy_list[0],gauss_l1l2l3);
				if(vy<0)vy=-vy;
				
				DL_scalar=gauss_weight*Jdet;

				D[0][0]=sqrt(2.0)*pow(3,1.0/4.0)*sqrt(Jdet)/2*vx*DL_scalar;
				D[0][1]=0;
				D[1][0]=0;
				D[1][1]=sqrt(2.0)*pow(3,1.0/4.0)*sqrt(Jdet)/2*vy*DL_scalar;
				
				/*  Do the triple producte tL*D*L: */
				TripleMultiply( &B[0][0],2,numgrids2d,1,
							&D[0][0],2,2,0,
							&B[0][0],2,numgrids2d,0,
							&Ke_gg_gaussian[0][0],0);

				/*Now add to the stiffness matrix : */
				for( i=0; i<numdof;i++){
					for (j=0;j<numdof;j++){
						*(Ke_gg->terms+Ke_gg->nrows*i+j)+=Ke_gg_gaussian[i][j];
					}
				}

			} //for (ig=0; ig<num_gauss2d; ig++)
		} //if(this->penta.artdiff)
		
	} //if (this->penta.onbed==0)
			
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord2d);
	xfree((void**)&second_gauss_area_coord2d);
	xfree((void**)&third_gauss_area_coord2d);
	xfree((void**)&gauss_weights2d);
	
	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build stiffness matrix.");
		return 0;
	}
	else{
		#ifdef _DEBUG_ 
			ElemMatrixEcho(Ke_gg);
		#endif

			
		/*Assign output pointer: */
		*pKe_gg=Ke_gg;
		
		return 1;
	}
}
/*--------------------------------------------------
	PentaElementCreateKMatrixMelting
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreateKMatrixMelting"
int PentaElementCreateKMatrixMelting( ElemMatrix* *pKe_gg, void* vpthis, ParameterInputs* inputs){

	int             i,j;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;

	/* local declarations */

	/* output: */
	ElemMatrix*     Ke_gg        = NULL;
	
	int             numgrids2d=3;
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	/* grid data: */
	int    cid_list[numgrids2d];
	double xyz_list[numgrids2d][3];
	double T_bg_list[numgrids2d][9];
	
	/* 2d gaussian point: */
	int     num_gauss2d;
	double* first_gauss_area_coord2d  =  NULL;
	double* second_gauss_area_coord2d =  NULL;
	double* third_gauss_area_coord2d  =  NULL;
	double* gauss_weights2d=NULL;
	double  gauss_l1l2l3[3];
	double  gauss_weight;
	int     ig;

	/* matrices: */
	double L[1][3];
	double DL_scalar;

	double Ke_gg_gaussian[3][3]; //stiffness matrix evaluated at the gaussian point, for first 3 grids.
	double Jdet;

	/* The number of g-set dof for this element is 1*3 (vz on bedrock grids) : */
	numdof=numgrids2d*NDOF1;

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;

	/*We are dealing with a collapsed formulation on the lower triangle of the penta, 
	  only on the bedrock.:*/

	if (this->penta.onbed==0){
		*pKe_gg=NULL;
		return noerr;
	}
	else{
		/* Allocate new stiffness element  matrix: */
		Ke_gg=NewElemMatrix(numdof);

		/* Get all element grid data */
		noerr = GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, this->pg, numgrids2d, 1 ); //only pick up the first 3
		if(!noerr){
			_printf_("%s%s\n",__FUNCT__," error message: could not get element grid data!");
			goto cleanup_and_return;
		}
		
		/* Build the row index vector (same as column index vector). 
		 * At the same time, get element exterior data for each grid: */
		for (i=0;i<numgrids2d;i++){
			structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
			dof=structural_dof_list[0]; //melting is found on the 1st dof
			*(Ke_gg->row_indices+i)=dof;
		}

		/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
		GaussTria( &num_gauss2d, &first_gauss_area_coord2d, &second_gauss_area_coord2d, &third_gauss_area_coord2d, &gauss_weights2d, 2);

		/* Start  looping on the number of gaussian points: */
		for (ig=0; ig<num_gauss2d; ig++){
			/*Pick up the gaussian point: */
			gauss_weight=*(gauss_weights2d+ig);
			gauss_l1l2l3[0]=*(first_gauss_area_coord2d+ig); 
			gauss_l1l2l3[1]=*(second_gauss_area_coord2d+ig);
			gauss_l1l2l3[2]=*(third_gauss_area_coord2d+ig);

			/* Get Jacobian determinant: */
			TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
			//Get L matrix 
			TriaElementGetL(&L[0][0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

			DL_scalar=gauss_weight*Jdet;
			
			/*Do the triple producte tL*D*L: */
			TripleMultiply( &L[0][0],1,3,1,
						&DL_scalar,1,1,0,
						&L[0][0],1,3,0,
						&Ke_gg_gaussian[0][0],0);

			/*Add gaussian contribution: */
			for( i=0; i<numdof;i++){
				for (j=0;j<numdof;j++){
					*(Ke_gg->terms+Ke_gg->nrows*i+j)+=Ke_gg_gaussian[i][j];
				}
			}
		} //for (ig=0; ig<num_gauss2d; ig++
		
	} //if (this->penta.onbed==0)
			
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord2d);
	xfree((void**)&second_gauss_area_coord2d);
	xfree((void**)&third_gauss_area_coord2d);
	xfree((void**)&gauss_weights2d);
	
	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build stiffness matrix.");
		return 0;
	}
	else{
		#ifdef _DEBUG_ 
			ElemMatrixEcho(Ke_gg);
		#endif

			
		/*Assign output pointer: */
		*pKe_gg=Ke_gg;
		
		return 1;
	}
}


/*--------------------------------------------------
	PentaElementCreateKMatrixDiagnosticVert
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreateKMatrixDiagnosticVert"
int PentaElementCreateKMatrixDiagnosticVert( ElemMatrix* *pKe_gg, void* vpthis, ParameterInputs* inputs){


	int             noerr=1;
	int             i,j;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;

	/* local declarations */

	/* output: */
	ElemMatrix*     Ke_gg        = NULL;
	
	
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;


	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double xyz_list_tria[3][3];
	double T_bg_list[numgrids][9];

	/* 3d gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights           =  NULL;
	double* vert_gauss_weights           =  NULL;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_l1l2l3l4[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	double  gauss_weight;

	/* 2d gaussian point: */
	int     num_gauss2d;
	double* first_gauss_area_coord2d  =  NULL;
	double* second_gauss_area_coord2d =  NULL;
	double* third_gauss_area_coord2d  =  NULL;
	double* gauss_weights2d=NULL;
	double  gauss_l1l2l3[3];

	/* surface normal: */
	double x4,y4,z4;
	double x5,y5,z5;
	double x6,y6,z6;
	double v46[3];
	double v56[3];
	double normal[3];
	double norm_normal;
	double nz;

	/* matrices: */
	double B[1][numgrids];
	double Bprime[1][numgrids];
	double DL_scalar;
	double L[3];

	double Ke_gg_gaussian[numgrids][numgrids]; 
	double Ke_gg_vert_gaussian[3][3]; 
	double Jdet;
	/*Collapsed formulation: */
	Tria*  tria=NULL;
	TriaElement* triaelement=NULL;

	/* The number of g-set dof for this element is 2*6 (vx and vy, horizontal velocities for ice, and 6 grids 
	 * per element: */
	numdof=numgrids*NDOF1;

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;
	
	/*Implement standard penta element: */

	/* Allocate new stiffness element  matrix: */
	Ke_gg=NewElemMatrix(numdof);

	/* Get all element grid data */
	noerr = GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, this->pg, numgrids, 1 );
	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not get element grid data!");
		goto cleanup_and_return;
	}
		
	/* Build the row index vector (same as column index vector). 
	 * At the same time, get element exterior data for each grid: */
	for (i=0;i<numgrids;i++){
		/* We use the structural dof list, from which we keep only the first two dof (x for vx, and y for vy).
		 * The other degrees of freedom will be spc'd for now.*/
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		dof=structural_dof_list[2]; //z degree of freedom
		*(Ke_gg->row_indices+i)=dof;
	}

	/*Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
	  get tria gaussian points as well as segment gaussian points. For tria gaussian 
	  points, order of integration is 2, because we need to integrate the product tB*D*B' 
	  which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
	  points, same deal, which yields 3 gaussian points.*/

	order_area_gauss=2;
	num_vert_gauss=2;

	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);
	#ifdef _DEBUG_ 
	for (i=0;i<num_area_gauss;i++){
		_printf_("Area Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(area_gauss_weights+i));
	}
	for (i=0;i<num_vert_gauss;i++){
		_printf_("Vert Gauss coord %i: %lf Weight: %lf\n",i,*(fourth_gauss_vert_coord+i),*(vert_gauss_weights+i));
	}
	#endif
	
	/* Start  looping on the number of gaussian points: */
	for (ig1=0; ig1<num_area_gauss; ig1++){
		for (ig2=0; ig2<num_vert_gauss; ig2++){
		
			/*Pick up the gaussian point: */
			gauss_weight1=*(area_gauss_weights+ig1);
			gauss_weight2=*(vert_gauss_weights+ig2);
			gauss_weight=gauss_weight1*gauss_weight2;
			
			gauss_l1l2l3l4[0]=*(first_gauss_area_coord+ig1); 
			gauss_l1l2l3l4[1]=*(second_gauss_area_coord+ig1);
			gauss_l1l2l3l4[2]=*(third_gauss_area_coord+ig1);
			gauss_l1l2l3l4[3]=*(fourth_gauss_vert_coord+ig2);

			/*Get B and Bprime matrices: */
			PentaElementGetB_vert(&B[0][0], &xyz_list[0][0], gauss_l1l2l3l4);
			PentaElementGetBPrime_vert(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3l4);

			/* Get Jacobian determinant: */
			PentaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3l4);

			DL_scalar=gauss_weight*Jdet;

			/*  Do the triple product tB*D*Bprime: */
			noerr=TripleMultiply( &B[0][0],1,numgrids,1,
					&DL_scalar,1,1,0,
					&Bprime[0][0],1,numgrids,0,
					&Ke_gg_gaussian[0][0],0);

			
			/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
			for( i=0; i<Ke_gg->nrows; i++){
				for (j=0;j<Ke_gg->nrows;j++){
					*(Ke_gg->terms+Ke_gg->nrows*i+j)+=Ke_gg_gaussian[i][j];
				}
			}	
		} //for (ig2=0; ig2<num_vert_gauss; ig2++)
	} //for (ig1=0; ig1<num_area_gauss; ig1++)

	/*Deal with upper surface boundary condition: */

	
	if(this->penta.onsurface==1){
		/*Build a tria element using the 3 grids at the upper base of the penta: */
		tria=NewTria();
		TriaInit(tria);
			
		strcpy(tria->name,"CTRICE3");
		tria->eid=this->penta.eid;
		tria->mid=this->penta.mid;
		for(i=0;i<3;i++){
			tria->g[i]=this->penta.g[3+i]; //we take the first 3 grids of the penta to build the tria.
			tria->h[i]=this->penta.h[3+i]; 
			tria->s[i]=this->penta.s[3+i];
			tria->b[i]=this->penta.b[3+i];
			tria->k[i]=this->penta.k[3+i];
		}

		/*diverse:*/
		tria->theta_type=0;
		tria->theta.angle=0;
		tria->zoffs=0;

		//friction
		tria->p=this->penta.p;
		tria->q=this->penta.q;
		tria->shelf=this->penta.shelf;
		tria->friction_type=this->penta.friction_type;

		/*type of fitting? : */
		tria->fit=this->penta.fit;
		tria->meanvel=this->penta.meanvel;
		tria->epsvel=this->penta.epsvel;

		/*diverse:*/
		tria->acceleration=0;

		/*Now that the tria object is created, spawn an element out of it: */
		triaelement = NewTriaElement( tria);

		/*Transfer configuration of penta to configuration of tria: */
		for(i=0;i<3;i++){
			triaelement->internalgrid_indices[i]=this->internalgrid_indices[3+i];
			triaelement->internalgrid_indicesb[i]=this->internalgrid_indicesb[3+i];
			triaelement->pg[i]=this->pg[3+i];
		}
		triaelement->matice_index=this->matice_index;
		triaelement->matice_enum=this->matice_enum;
		triaelement->matice=this->matice;

		triaelement->matpar_index=this->matpar_index;
		triaelement->matpar_enum=this->matpar_enum;
		triaelement->matpar=this->matpar;


		/*Build xyz_list_tria: */
		xyz_list_tria[0][0]=xyz_list[4][0];
		xyz_list_tria[0][1]=xyz_list[4][1];
		xyz_list_tria[0][2]=xyz_list[4][2];

		xyz_list_tria[1][0]=xyz_list[5][0];
		xyz_list_tria[1][1]=xyz_list[5][1];
		xyz_list_tria[1][2]=xyz_list[5][2];

		xyz_list_tria[2][0]=xyz_list[6][0];
		xyz_list_tria[2][1]=xyz_list[6][1];
		xyz_list_tria[2][2]=xyz_list[6][2];


		/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
		GaussTria( &num_gauss2d, &first_gauss_area_coord2d, &second_gauss_area_coord2d, &third_gauss_area_coord2d, &gauss_weights2d, 2);

		/*Build normal vector to the surface:*/
		
		x4=xyz_list[0][0];
		y4=xyz_list[0][1];
		z4=xyz_list[0][2];
	
		x5=xyz_list[1][0];
		y5=xyz_list[1][1];
		z5=xyz_list[1][2];
	
		x6=xyz_list[2][0];
		y6=xyz_list[2][1];
		z6=xyz_list[2][2];

		v46[0]=x4-x6;
		v46[1]=y4-y6;
		v46[2]=z4-z6;

		v56[0]=x5-x6;
		v56[1]=y5-y6;
		v56[2]=z5-z6;

		normal[0]=(y4-y6)*(z5-z6)-(z4-z6)*(y5-y6);
		normal[1]=(z4-z6)*(x5-x6)-(x4-x6)*(z5-z6);
		normal[2]=(x4-x6)*(y5-y6)-(y4-y6)*(x5-x6);

		norm_normal=sqrt(pow(normal[0],2)+pow(normal[1],2)+pow(normal[2],2));
		nz=1.0/norm_normal*normal[2];

		/* Start  looping on the number of gaussian points: */
		for (ig=0; ig<num_gauss2d; ig++){
			/*Pick up the gaussian point: */
			gauss_weight=*(gauss_weights2d+ig);
			gauss_l1l2l3[0]=*(first_gauss_area_coord2d+ig); 
			gauss_l1l2l3[1]=*(second_gauss_area_coord2d+ig);
			gauss_l1l2l3[2]=*(third_gauss_area_coord2d+ig);

			/* Get Jacobian determinant: */
			TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
			//Get L matrix if viscous basal drag present:
			TriaElementGetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

			/**********************Do not forget the sign**********************************/
			DL_scalar=- gauss_weight*Jdet*nz; 
			/******************************************************************************/
		
			/*  Do the triple producte tL*D*L: */
			TripleMultiply( L,1,3,1,
						&DL_scalar,1,1,0,
						L,1,3,0,
						&Ke_gg_vert_gaussian[0][0],0);

			/*Now add the drag only on the last 3 grids of the upper surface: */
			for( i=3; i<6;i++){
				for (j=3;j<6;j++){
					*(Ke_gg->terms+Ke_gg->nrows*i+j) =  *(Ke_gg->terms+Ke_gg->nrows*i+j)+  Ke_gg_vert_gaussian[i-3][j-3];
				}
			}
			
		} //for (ig=0; ig<num_gauss2d; ig++)
		
		/*Now delete tria and triaelement: */
		DeleteTriaElement((void**)&triaelement);
		DeleteTria((void**)&tria);
	} //if(this->penta.onsurface==1)
		
		
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
	xfree((void**)&first_gauss_area_coord2d);
	xfree((void**)&second_gauss_area_coord2d);
	xfree((void**)&third_gauss_area_coord2d);
	xfree((void**)&gauss_weights2d);
	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build stiffness matrix.");
		return 0;
	}
	else{
		#ifdef _DEBUG_ 
			ElemMatrixEcho(Ke_gg);
		#endif

			
		/*Assign output pointer: */
		*pKe_gg=Ke_gg;
		
		return 1;
	}
}


/*--------------------------------------------------
	PentaElementCreateKMatrixThermal
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreateKMatrixThermal"
int PentaElementCreateKMatrixThermal( ElemMatrix* *pKe_gg, void* vpthis, ParameterInputs* inputs){


	int             noerr=1;
	int             i,j;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;
	Matice        * matice = NULL;
	Matpar        * matpar = NULL;

	/* local declarations */

	/* output: */
	ElemMatrix*     Ke_gg        = NULL;
	
	
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;


	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double xyz_list_tria[3][3];
	double T_bg_list[numgrids][9];

	/* 3d gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights           =  NULL;
	double* vert_gauss_weights           =  NULL;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_l1l2l3l4[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	double  gauss_weight;

	/* matrices: */
	double D_scalar;
	
	double B_conduct[3][numgrids];
	double D_conduct[3][3]={{ 0,0,0 },{0,0,0},{0,0,0}}; 
	
	double B_advect[3][numgrids];
	double Bprime_advect[3][numgrids];
	double D_advect[3][3]={{ 0,0,0 },{0,0,0},{0,0,0}}; 

	double L_transient[numgrids];

	double Ke_gg_gaussian_conduct[numgrids][numgrids]; 
	double Ke_gg_gaussian_advect[numgrids][numgrids]; 
	double Ke_gg_gaussian_transient[numgrids][numgrids]; 

	/* input parameters: */
	double* velocity=NULL;
	double  vx_list[numgrids];
	double  vx;
	double  vy_list[numgrids];
	double  vy;
	double  vz_list[numgrids];
	double  vz;
	double* dt_param=NULL;
	double  dt;

	double Jdet;

	/*materials: */
	double rho_ice,heatcapacity,thermalconductivity;

	/* The number of g-set dof for this element is 1*6 (temperature on all grids)
	 * per element: */
	numdof=numgrids*NDOF1;

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;
	matpar=this->matpar;

	//recover material parameters
	rho_ice=matpar->rho_ice;
	heatcapacity=matpar->heatcapacity;
	thermalconductivity=matpar->thermalconductivity;
	
	/* Allocate new stiffness element  matrix: */
	Ke_gg=NewElemMatrix(numdof);

	/* Get all element grid data */
	noerr = GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], cid_list, this->pg, numgrids, 1 );
	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not get element grid data!");
		goto cleanup_and_return;
	}
	
	/*recover extra inputs from users, at current convergence iteration: */
	velocity=ParameterInputsRecover(inputs,"velocity"); 
	dt_param=ParameterInputsRecover(inputs,"dt");

	if((!velocity) || (!dt_param)){
		_printf_("%s%s\n",__FUNCT__," error message: missing velocity or dt parameter!");
		noerr=0; goto cleanup_and_return;
	}
	dt=*dt_param;

	/* Build the row index vector (same as column index vector). 
	 * At the same time, get element exterior data for each grid: */
	for (i=0;i<numgrids;i++){
		/* We use the structural dof list, from which we keep only the first two dof (x for vx, and y for vy).
		 * The other degrees of freedom will be spc'd for now.*/
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		dof=structural_dof_list[0]; //1st dof for temperatures
		*(Ke_gg->row_indices+i)=dof;
		
		dof=structural_dof_list[0];
		vx_list[i]=*(velocity+dof);
		dof=structural_dof_list[1];
		vy_list[i]=*(velocity+dof);
		dof=structural_dof_list[2];
		vz_list[i]=*(velocity+dof);
	}

	/*Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
	  get tria gaussian points as well as segment gaussian points. For tria gaussian 
	  points, order of integration is 2, because we need to integrate the product tB*D*B' 
	  which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
	  points, same deal, which yields 3 gaussian points.*/

	order_area_gauss=2;
	num_vert_gauss=2;

	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);
	#ifdef _DEBUG_ 
	for (i=0;i<num_area_gauss;i++){
		_printf_("Area Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(area_gauss_weights+i));
	}
	for (i=0;i<num_vert_gauss;i++){
		_printf_("Vert Gauss coord %i: %lf Weight: %lf\n",i,*(fourth_gauss_vert_coord+i),*(vert_gauss_weights+i));
	}
	#endif
	
	/* Start  looping on the number of gaussian points: */
	for (ig1=0; ig1<num_area_gauss; ig1++){
		for (ig2=0; ig2<num_vert_gauss; ig2++){
		
			/*Pick up the gaussian point: */
			gauss_weight1=*(area_gauss_weights+ig1);
			gauss_weight2=*(vert_gauss_weights+ig2);
			gauss_weight=gauss_weight1*gauss_weight2;
			
			gauss_l1l2l3l4[0]=*(first_gauss_area_coord+ig1); 
			gauss_l1l2l3l4[1]=*(second_gauss_area_coord+ig1);
			gauss_l1l2l3l4[2]=*(third_gauss_area_coord+ig1);
			gauss_l1l2l3l4[3]=*(fourth_gauss_vert_coord+ig2);

			/* Get Jacobian determinant: */
			PentaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3l4);

			/* deal with conduction: */
			
			/*Get B matrix: */
			PentaElementGetB_conduct(&B_conduct[0][0], &xyz_list[0][0], gauss_l1l2l3l4);

			/*Build the D matrix: */
			if (this->penta.thermal_steadystate==1){
				D_scalar= gauss_weight*Jdet*(thermalconductivity/(rho_ice*heatcapacity));
			}
			else{
				D_scalar= dt*gauss_weight*Jdet*(thermalconductivity/(rho_ice*heatcapacity));
			}
			for(i=0;i<3;i++){
				D_conduct[i][i]=D_scalar;
			}

			/*  Do the triple product tB*D*B': */
			noerr=TripleMultiply( &B_conduct[0][0],3,numgrids,1,
					&D_conduct[0][0],3,3,0,
					&B_conduct[0][0],3,numgrids,0,
					&Ke_gg_gaussian_conduct[0][0],0);

			/* deal with advection: */
			
			/*Get B and Bprime matrix: */
			
			PentaElementGetB_advect(&B_advect[0][0], &xyz_list[0][0], gauss_l1l2l3l4);
			PentaElementGetBprime_advect(&Bprime_advect[0][0], &xyz_list[0][0], gauss_l1l2l3l4);
					
			/*Get velocities: */
			PentaElementGetParameterValue(&vx, &vx_list[0],gauss_l1l2l3l4);
			PentaElementGetParameterValue(&vy, &vy_list[0],gauss_l1l2l3l4);
			PentaElementGetParameterValue(&vz, &vz_list[0],gauss_l1l2l3l4);

			/*Build D matrix: */
			if (this->penta.thermal_steadystate==1){
				D_scalar= gauss_weight*Jdet;
			}
			else{
				D_scalar= dt*gauss_weight*Jdet;
			}
			D_advect[0][0]=D_scalar*vx;
			D_advect[1][1]=D_scalar*vy;
			D_advect[2][2]=D_scalar*vz;

			/*  Do the triple product tB*D*B': */
			noerr=TripleMultiply( &B_advect[0][0],3,numgrids,1,
					&D_advect[0][0],3,3,0,
					&Bprime_advect[0][0],3,numgrids,0,
					&Ke_gg_gaussian_advect[0][0],0);

			/*Deal with transient: */
			if(this->penta.thermal_steadystate==1){
				/*steady state, do nothing: */
				for(i=0;i<numgrids;i++){
					for(j=0;j<numgrids;j++){
						Ke_gg_gaussian_transient[i][j]=0;
					}
				}
			}
			else{
				PentaElementGetNodalFunctions(&L_transient[0], gauss_l1l2l3l4);
				D_scalar=gauss_weight*Jdet;

				/*Do the triple product:L*D_scalar*L' */
				noerr=TripleMultiply( L_transient,1,numgrids,1,
					&D_scalar,1,1,0,
					L_transient,1,numgrids,0,
					&Ke_gg_gaussian_transient[0][0],0);
			}

			
			/* Add all the contributions to the stiffness matrix: */
			for( i=0; i<Ke_gg->nrows; i++){
				for (j=0;j<Ke_gg->nrows;j++){
					*(Ke_gg->terms+Ke_gg->nrows*i+j)+= Ke_gg_gaussian_conduct[i][j]+Ke_gg_gaussian_advect[i][j]+Ke_gg_gaussian_transient[i][j];
				}
			}


		} //for (ig2=0; ig2<num_vert_gauss; ig2++)
	} //for (ig1=0; ig1<num_area_gauss; ig1++)
		

	

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build stiffness matrix.");
		return 0;
	}
	else{
		#ifdef _DEBUG_ 
			ElemMatrixEcho(Ke_gg);
		#endif

			
		/*Assign output pointer: */
		*pKe_gg=Ke_gg;
		
		return 1;
	}
}
/*--------------------------------------------------
	PentaElementCreatePVector
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreatePVector"

int PentaElementCreatePVector( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs, int analysis_type,int sub_analysis_type){

	int noerr=1;
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if ((analysis_type==DiagnosticHorizAnalysisEnum()) || (analysis_type==ControlAnalysisEnum())){
		noerr=PentaElementCreatePVectorDiagnosticHoriz( ppe_g,vpthis,inputs);
	}
	else if(analysis_type==DiagnosticBaseVertAnalysisEnum()){
		noerr=PentaElementCreatePVectorDiagnosticBaseVert( ppe_g,vpthis,inputs);
	}
	else if(analysis_type==DiagnosticVertAnalysisEnum()){
		noerr=PentaElementCreatePVectorDiagnosticVert( ppe_g,vpthis,inputs);
	}
	else if(analysis_type==MeltingAnalysisEnum()){
		noerr=PentaElementCreatePVectorMelting( ppe_g,vpthis,inputs);
	}
	else if(analysis_type==PrognosticAnalysisEnum()){
		noerr=PentaElementCreatePVectorPrognostic( ppe_g,vpthis,inputs);
	}
	else if(analysis_type==ThermalSteadyAnalysisEnum()|| (analysis_type==ThermalTransientAnalysisEnum())){
		noerr=PentaElementCreatePVectorThermal( ppe_g,vpthis,inputs);
	}
	else{
		_printf_("%s%s%i%s\n",__FUNCT__," error message: analysis ",analysis_type," not supported yet");
		noerr=0;
	}
	return noerr;
}

/*--------------------------------------------------
	PentaElementGetStrainRate
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementGetStrainRate"
int PentaElementGetStrainRate(double* epsilon, double* velocity, double* xyz_list, double* gauss_l1l2l3l4){

	int noerr=1;
	int i;

	double B[5][NDOF2*numgrids];

	/*Get B matrix: */
	noerr=PentaElementGetB(&B[0][0], xyz_list, gauss_l1l2l3l4);
	if(!noerr)goto cleanup_and_return;

	#ifdef _DEBUG_
	_printf_("B for grid1 : [ %lf   %lf  \n",B[0][0],B[0][1]);
	_printf_("              [ %lf   %lf  \n",B[1][0],B[1][1]);
	_printf_("              [ %lf   %lf  ]\n",B[2][0],B[2][1]);
	_printf_("              [ %lf   %lf  ]\n",B[3][0],B[3][1]);
	_printf_("              [ %lf   %lf  ]\n",B[4][0],B[4][1]);
	
	_printf_("B for grid2 : [ %lf   %lf  \n",B[0][2],B[0][3]);
	_printf_("              [ %lf   %lf  \n",B[1][2],B[1][3]);
	_printf_("              [ %lf   %lf  ]\n",B[2][2],B[2][3]);
	_printf_("              [ %lf   %lf  ]\n",B[3][2],B[3][3]);
	_printf_("              [ %lf   %lf  ]\n",B[4][2],B[4][3]);
	
	_printf_("B for grid3 : [ %lf   %lf  \n", B[0][4],B[0][5]);
	_printf_("              [ %lf   %lf  \n", B[1][4],B[1][5]);
	_printf_("              [ %lf   %lf  ]\n",B[2][4],B[2][5]);
	_printf_("              [ %lf   %lf  ]\n",B[3][4],B[3][5]);
	_printf_("              [ %lf   %lf  ]\n",B[4][4],B[4][5]);
	
	_printf_("B for grid4 : [ %lf   %lf  \n", B[0][6],B[0][7]);
	_printf_("              [ %lf   %lf  \n", B[1][6],B[1][7]);
	_printf_("              [ %lf   %lf  ]\n",B[2][6],B[2][7]);
	_printf_("              [ %lf   %lf  ]\n",B[3][6],B[3][7]);
	_printf_("              [ %lf   %lf  ]\n",B[4][6],B[4][7]);
				
	_printf_("B for grid5 : [ %lf   %lf  \n", B[0][8],B[0][9]);
	_printf_("              [ %lf   %lf  \n", B[1][8],B[1][9]);
	_printf_("              [ %lf   %lf  ]\n",B[2][8],B[2][9]);
	_printf_("              [ %lf   %lf  ]\n",B[3][8],B[3][9]);
	_printf_("              [ %lf   %lf  ]\n",B[4][8],B[4][9]);

	_printf_("B for grid6 : [ %lf   %lf  \n", B[0][10],B[0][11]);
	_printf_("              [ %lf   %lf  \n", B[1][10],B[1][11]);
	_printf_("              [ %lf   %lf  ]\n",B[2][10],B[2][11]);
	_printf_("              [ %lf   %lf  ]\n",B[3][10],B[3][11]);
	_printf_("              [ %lf   %lf  ]\n",B[4][10],B[4][11]);

	for (i=0;i<numgrids;i++){
		_printf_("Velocity for grid %i %lf %lf\n",i,*(vxvy_list+2*i+0),*(vxvy_list+2*i+1));
	}
	#endif

	/*Multiply B by velocity, to get strain rate: */
	noerr=MatrixMultiply( &B[0][0],5,NDOF2*numgrids,0,
			              velocity,NDOF2*numgrids,1,0,
						  epsilon,0);
	if(!noerr)goto cleanup_and_return;


	cleanup_and_return:
	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not compute strain rate.");
		return 0;
	}
	else{
		return 1;
	}

}


/*--------------------------------------------------
	PentaElementGetParameterValue
  --------------------------------------------------*/
int PentaElementGetParameterValue(double* pvalue, double* v_list,double* gauss_l1l2l3l4){
	
	int noerr=1;
	
	double l1l2l3l4l5l6[NUM_PENTA_GRIDS];

	PentaElementGetNodalFunctions(&l1l2l3l4l5l6[0], gauss_l1l2l3l4);

	*pvalue=l1l2l3l4l5l6[0]*v_list[0]+l1l2l3l4l5l6[1]*v_list[1]+l1l2l3l4l5l6[2]*v_list[2]+l1l2l3l4l5l6[3]*v_list[3]+l1l2l3l4l5l6[4]*v_list[4]+l1l2l3l4l5l6[5]*v_list[5];

	return noerr;
}
	

/*--------------------------------------------------
	PentaElementGetNodalFunctions
  --------------------------------------------------*/
int PentaElementGetNodalFunctions(double* l1l2l3l4l5l6, double* gauss_l1l2l3l4){
	
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	l1l2l3l4l5l6[0]=gauss_l1l2l3l4[0]*(1-gauss_l1l2l3l4[3])/2.0;

	l1l2l3l4l5l6[1]=gauss_l1l2l3l4[1]*(1-gauss_l1l2l3l4[3])/2.0;
	
	l1l2l3l4l5l6[2]=gauss_l1l2l3l4[2]*(1-gauss_l1l2l3l4[3])/2.0;

	l1l2l3l4l5l6[3]=gauss_l1l2l3l4[0]*(1+gauss_l1l2l3l4[3])/2.0;

	l1l2l3l4l5l6[4]=gauss_l1l2l3l4[1]*(1+gauss_l1l2l3l4[3])/2.0;
	
	l1l2l3l4l5l6[5]=gauss_l1l2l3l4[2]*(1+gauss_l1l2l3l4[3])/2.0;

	return 1;
}


/*--------------------------------------------------
	PentaElementGetB
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementGetB"

int PentaElementGetB(double* B, double* xyz_list, double* gauss_l1l2l3l4){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*NDOF2. 
	 * For grid i, Bi can be expressed in the basic coordinate system
	 * by: 
	 *       Bi_basic=[ dh/dx          0      ]
	 *                [   0           dh/dy   ]
	 *                [ 1/2*dh/dy  1/2*dh/dx  ]
	 *                [ 1/2*dh/dz      0      ]
	 *                [  0         1/2*dh/dz  ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 5x(NDOF2*numgrids)
	 */
	
	int noerr=1;
	int i;

	double dh1dh2dh3dh4dh5dh6_basic[NDOF3][numgrids];


	/*Get dh1dh2dh3dh4dh5dh6_basic in basic coordinate system: */
	noerr=PentaElementGetNodalFunctionsDerivativesBasic(&dh1dh2dh3dh4dh5dh6_basic[0][0],xyz_list, gauss_l1l2l3l4);
	if(!noerr)goto cleanup_and_return;

	#ifdef _DEBUG_ 
	for (i=0;i<numgrids;i++){
		_printf_("Node %i  dh/dx=%lf dh/dy=%lf dh/dz=%lf\n",i,dh1dh2dh3dh4dh5dh6_basic[0][i],dh1dh2dh3dh4dh5dh6_basic[1][i],dh1dh2dh3dh4dh5dh6_basic[2][i]);
	}
	#endif

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=dh1dh2dh3dh4dh5dh6_basic[0][i]; 
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=0.0;

		*(B+NDOF2*numgrids*1+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=dh1dh2dh3dh4dh5dh6_basic[1][i];

		*(B+NDOF2*numgrids*2+NDOF2*i)=(float).5*dh1dh2dh3dh4dh5dh6_basic[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=(float).5*dh1dh2dh3dh4dh5dh6_basic[0][i]; 

		*(B+NDOF2*numgrids*3+NDOF2*i)=(float).5*dh1dh2dh3dh4dh5dh6_basic[2][i]; 
		*(B+NDOF2*numgrids*3+NDOF2*i+1)=0.0;
		
		*(B+NDOF2*numgrids*4+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*4+NDOF2*i+1)=(float).5*dh1dh2dh3dh4dh5dh6_basic[2][i]; 
	}

	
	cleanup_and_return:

	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not compute B matrix.");
				return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	PentaElementGetB_conduct
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementGetB_conduct"

int PentaElementGetB_conduct(double* B, double* xyz_list, double* gauss_l1l2l3l4){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size numgrids*NDOF
	For grid i, Bi can be expressed in the basic coordinate system
	by: 
		   Bi_basic=[dh/dx]
                   [dh/dy]
                   [dh/dz]
	where h is the interpolation function for grid i.
	*/

	int noerr=1;
	int i;

	double dh1dh2dh3dh4dh5dh6_basic[NDOF3][numgrids];


	/*Get dh1dh2dh3dh4dh5dh6_basic in basic coordinate system: */
	noerr=PentaElementGetNodalFunctionsDerivativesBasic(&dh1dh2dh3dh4dh5dh6_basic[0][0],xyz_list, gauss_l1l2l3l4);
	if(!noerr)goto cleanup_and_return;

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+numgrids*0+i)=dh1dh2dh3dh4dh5dh6_basic[0][i];
		*(B+numgrids*1+i)=dh1dh2dh3dh4dh5dh6_basic[1][i];
		*(B+numgrids*2+i)=dh1dh2dh3dh4dh5dh6_basic[2][i];
	}
	
	cleanup_and_return:

	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not compute B matrix.");
				return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	PentaElementGetB_advect
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementGetB_advect"

int PentaElementGetB_advect(double* B, double* xyz_list, double* gauss_l1l2l3l4){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size numgrids*NDOF
	For grid i, Bi can be expressed in the basic coordinate system
	by: 
		   Bi_basic=[Li]
                   [Li]
                   [Li]
	where Li is the interpolation function for grid i.

	This routine is actually the same as GetB_conduct!
	*/

	int noerr=1;
	int i;

	double l1l2l3l4l5l6[NUM_PENTA_GRIDS];

	PentaElementGetNodalFunctions(&l1l2l3l4l5l6[0], gauss_l1l2l3l4);

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+numgrids*0+i)=l1l2l3l4l5l6[i];
		*(B+numgrids*1+i)=l1l2l3l4l5l6[i];
		*(B+numgrids*2+i)=l1l2l3l4l5l6[i];
	}
	
	cleanup_and_return:

	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not compute B matrix.");
				return 0;
	}
	else{
		return 1;
	}
}
/*--------------------------------------------------
	PentaElementGetB_vert
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementGetB_vert"

int PentaElementGetB_vert(double* B, double* xyz_list, double* gauss_l1l2l3l4){


	/*	Compute B  matrix. B=[dh1/dz dh2/dz dh3/dz dh4/dz dh5/dz dh6/dz];
	where hi is the interpolation function for grid i.*/

	int noerr=1;
	int i;

	double dh1dh2dh3dh4dh5dh6_basic[NDOF3][numgrids];

	/*Get dh1dh2dh3dh4dh5dh6_basic in basic coordinate system: */
	noerr=PentaElementGetNodalFunctionsDerivativesBasic(&dh1dh2dh3dh4dh5dh6_basic[0][0],xyz_list, gauss_l1l2l3l4);
	if(!noerr)goto cleanup_and_return;

	#ifdef _DEBUG_ 
	for (i=0;i<numgrids;i++){
		_printf_("Node %i  dh/dx=%lf dh/dy=%lf dh/dz=%lf\n",i,dh1dh2dh3dh4dh5dh6_basic[0][i],dh1dh2dh3dh4dh5dh6_basic[1][i],dh1dh2dh3dh4dh5dh6_basic[2][i]);
	}
	#endif

	/*Build B: */
	for (i=0;i<numgrids;i++){
		B[i]=dh1dh2dh3dh4dh5dh6_basic[2][i];  
	}
	
	cleanup_and_return:

	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not compute B matrix.");
				return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	PentaElementGetBPrime
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementGetBPrime"

int PentaElementGetBPrime(double* B, double* xyz_list, double* gauss_l1l2l3l4){

	/*Compute B  prime matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*NDOF2. 
	 * For grid i, Bi can be expressed in the basic coordinate system
	 * by: 
	 *       Bi_basic=[ 2*dh/dx     dh/dy   ]
	 *                [   dh/dx    2*dh/dy  ]
	 *                [ dh/dy      dh/dx    ]
	 *                [ dh/dz         0     ]
	 *                [  0         dh/dz    ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 5x(NDOF2*numgrids)
	 */
	
	int noerr=1;
	int i;

	double dh1dh2dh3dh4dh5dh6_basic[NDOF3][numgrids];

	

	/*Get dh1dh2dh3dh4dh5dh6_basic in basic coordinate system: */
	noerr=PentaElementGetNodalFunctionsDerivativesBasic(&dh1dh2dh3dh4dh5dh6_basic[0][0],xyz_list, gauss_l1l2l3l4);
	if(!noerr)goto cleanup_and_return;

	#ifdef _DEBUG_ 
	for (i=0;i<numgrids;i++){
		_printf_("Node %i  dh/dx=%lf dh/dy=%lf dh/dz=%lf\n",i,dh1dh2dh3dh4dh5dh6_basic[0][i],dh1dh2dh3dh4dh5dh6_basic[1][i],dh1dh2dh3dh4dh5dh6_basic[2][i]);
	}
	#endif

	/*Build BPrime: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=2.0*dh1dh2dh3dh4dh5dh6_basic[0][i]; 
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=dh1dh2dh3dh4dh5dh6_basic[1][i];

		*(B+NDOF2*numgrids*1+NDOF2*i)=dh1dh2dh3dh4dh5dh6_basic[0][i];
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=2.0*dh1dh2dh3dh4dh5dh6_basic[1][i];

		*(B+NDOF2*numgrids*2+NDOF2*i)=dh1dh2dh3dh4dh5dh6_basic[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=dh1dh2dh3dh4dh5dh6_basic[0][i]; 

		*(B+NDOF2*numgrids*3+NDOF2*i)=dh1dh2dh3dh4dh5dh6_basic[2][i]; 
		*(B+NDOF2*numgrids*3+NDOF2*i+1)=0.0;
		
		*(B+NDOF2*numgrids*4+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*4+NDOF2*i+1)=dh1dh2dh3dh4dh5dh6_basic[2][i]; 
	}

	
	cleanup_and_return:

	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not compute B' matrix.");
				return 0;
	}
	else{
		return 1;
	}
}
/*--------------------------------------------------
	PentaElementGetBprime_advect
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementGetBprime_advect"

int PentaElementGetBprime_advect(double* B, double* xyz_list, double* gauss_l1l2l3l4){

	/*Compute B  prime matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 3*1. 
	 * For grid i, Bi can be expressed in the basic coordinate system
	 * by: 
	 *       Bi_basic=[ dh/dx]
	 *                [ dh/dy]
	 *                [ dh/dz]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 5x3
	 */
	
	int noerr=1;
	int i;

	double dh1dh2dh3dh4dh5dh6_basic[NDOF3][numgrids];


	/*Get dh1dh2dh3dh4dh5dh6_basic in basic coordinate system: */
	noerr=PentaElementGetNodalFunctionsDerivativesBasic(&dh1dh2dh3dh4dh5dh6_basic[0][0],xyz_list, gauss_l1l2l3l4);
	if(!noerr)goto cleanup_and_return;

	#ifdef _DEBUG_ 
	for (i=0;i<numgrids;i++){
		_printf_("Node %i  dh/dx=%lf dh/dy=%lf dh/dz=%lf\n",i,dh1dh2dh3dh4dh5dh6_basic[0][i],dh1dh2dh3dh4dh5dh6_basic[1][i],dh1dh2dh3dh4dh5dh6_basic[2][i]);
	}
	#endif

	/*Build BPrime: */
	for (i=0;i<numgrids;i++){
		*(B+numgrids*0+i)=dh1dh2dh3dh4dh5dh6_basic[0][i]; 
		*(B+numgrids*1+i)=dh1dh2dh3dh4dh5dh6_basic[1][i]; 
		*(B+numgrids*2+i)=dh1dh2dh3dh4dh5dh6_basic[2][i]; 
	}

	
	cleanup_and_return:

	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not compute B' matrix.");
				return 0;
	}
	else{
		return 1;
	}
}
/*--------------------------------------------------
	PentaElementGetBPrime_vert
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementGetBPrime_vert"

int PentaElementGetBPrime_vert(double* B, double* xyz_list, double* gauss_l1l2l3l4){

	// Compute Bprime  matrix. Bprime=[L1 L2 L3 L4 L5 L6] where Li is the nodal function for grid i

	int noerr=1;
	int i;
				
	PentaElementGetNodalFunctions(B, gauss_l1l2l3l4);

	cleanup_and_return:

	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not compute B' matrix.");
				return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	PentaElementGetJacobianDeterminant
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementGetJacobianDeterminant" 
int PentaElementGetJacobianDeterminant(double*  Jdet, double* xyz_list,double* gauss_l1l2l3l4){

	/*On a penta, Jacobian varies according to coordinates. We need to get the Jacobian, and take 
	 * the determinant of it: */
	double J[NDOF3][NDOF3];

	PentaElementGetJacobian(&J[0][0],xyz_list,gauss_l1l2l3l4);

	*Jdet= J[0][0]*J[1][1]*J[2][2]-J[0][0]*J[1][2]*J[2][1]-J[1][0]*J[0][1]*J[2][2]+J[1][0]*J[0][2]*J[2][1]+J[2][0]*J[0][1]*J[1][2]-J[2][0]*J[0][2]*J[1][1];
	if(*Jdet<0){
		//printf("%s%s\n",__FUNCT__," error message: negative jacobian determinant!");
	}

	return 1;
}

/*--------------------------------------------------
	PentaElementGetNodalFunctionsDerivativesBasic
  --------------------------------------------------*/
int PentaElementGetNodalFunctionsDerivativesBasic(double* dh1dh2dh3dh4dh5dh6_basic,double* xyz_list, double* gauss_l1l2l3l4){
	
	/*This routine returns the values of the nodal functions derivatives  (with respect to the basic coordinate system: */

	
	int noerr=1;
	int i;

	double dh1dh2dh3dh4dh5dh6_param[NDOF3][numgrids];
	double Jinv[NDOF3][NDOF3];

	/*Get derivative values with respect to parametric coordinate system: */
	noerr=PentaElementGetNodalFunctionsDerivativesParams(&dh1dh2dh3dh4dh5dh6_param[0][0], gauss_l1l2l3l4); 
	if(!noerr)goto cleanup_and_return;

	/*Get Jacobian invert: */
	noerr=PentaElementGetJacobianInvert(&Jinv[0][0], xyz_list, gauss_l1l2l3l4);
	if(!noerr)goto cleanup_and_return;

	/*Build dh1dh2dh3_basic: 
	 *
	 * [dhi/dx]= Jinv*[dhi/dr]
	 * [dhi/dy]       [dhi/ds]
	 * [dhi/dz]       [dhi/dn]
	 */

	for (i=0;i<numgrids;i++){
		*(dh1dh2dh3dh4dh5dh6_basic+numgrids*0+i)=Jinv[0][0]*dh1dh2dh3dh4dh5dh6_param[0][i]+Jinv[0][1]*dh1dh2dh3dh4dh5dh6_param[1][i]+Jinv[0][2]*dh1dh2dh3dh4dh5dh6_param[2][i];
		*(dh1dh2dh3dh4dh5dh6_basic+numgrids*1+i)=Jinv[1][0]*dh1dh2dh3dh4dh5dh6_param[0][i]+Jinv[1][1]*dh1dh2dh3dh4dh5dh6_param[1][i]+Jinv[1][2]*dh1dh2dh3dh4dh5dh6_param[2][i];
		*(dh1dh2dh3dh4dh5dh6_basic+numgrids*2+i)=Jinv[2][0]*dh1dh2dh3dh4dh5dh6_param[0][i]+Jinv[2][1]*dh1dh2dh3dh4dh5dh6_param[1][i]+Jinv[2][2]*dh1dh2dh3dh4dh5dh6_param[2][i];
	}

	cleanup_and_return:
	if(!noerr){
		return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	PentaElementGetJacobian
  --------------------------------------------------*/
int PentaElementGetJacobian(double* J, double* xyz_list,double* gauss_l1l2l3l4){

	int noerr=1;
	int i,j;
	
	/*The Jacobian is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double A1,A2,A3; //area coordinates
	double xi,eta,zi; //parametric coordinates

	double x1,x2,x3,x4,x5,x6;
	double y1,y2,y3,y4,y5,y6;
	double z1,z2,z3,z4,z5,z6;
	
	double sqrt3=sqrt(3.0);
	
	/*Figure out xi,eta and zi (parametric coordinates), for this gaussian point: */
	A1=gauss_l1l2l3l4[0];
	A2=gauss_l1l2l3l4[1];
	A3=gauss_l1l2l3l4[2];

	xi=A2-A1;
	eta=sqrt3*A3;
	zi=gauss_l1l2l3l4[3];

	x1=*(xyz_list+3*0+0);
	x2=*(xyz_list+3*1+0);
	x3=*(xyz_list+3*2+0);
	x4=*(xyz_list+3*3+0);
	x5=*(xyz_list+3*4+0);
	x6=*(xyz_list+3*5+0);
	
	y1=*(xyz_list+3*0+1);
	y2=*(xyz_list+3*1+1);
	y3=*(xyz_list+3*2+1);
	y4=*(xyz_list+3*3+1);
	y5=*(xyz_list+3*4+1);
	y6=*(xyz_list+3*5+1);

	z1=*(xyz_list+3*0+2);
	z2=*(xyz_list+3*1+2);
	z3=*(xyz_list+3*2+2);
	z4=*(xyz_list+3*3+2);
	z5=*(xyz_list+3*4+2);
	z6=*(xyz_list+3*5+2);


	*(J+NDOF3*0+0)=1.0/4.0*(x1-x2-x4+x5)*zi+1.0/4.0*(-x1+x2-x4+x5);
	*(J+NDOF3*1+0)=sqrt3/12.0*(x1+x2-2*x3-x4-x5+2*x6)*zi+sqrt3/12.0*(-x1-x2+2*x3-x4-x5+2*x6);
	*(J+NDOF3*2+0)=sqrt3/12.0*(x1+x2-2*x3-x4-x5+2*x6)*eta+1/4*(x1-x2-x4+x5)*xi +1.0/4.0*(-x1+x5-x2+x4);

	*(J+NDOF3*0+1)=1.0/4.0*(y1-y2-y4+y5)*zi+1.0/4.0*(-y1+y2-y4+y5);
	*(J+NDOF3*1+1)=sqrt3/12.0*(y1+y2-2*y3-y4-y5+2*y6)*zi+sqrt3/12.0*(-y1-y2+2*y3-y4-y5+2*y6);
	*(J+NDOF3*2+1)=sqrt3/12.0*(y1+y2-2*y3-y4-y5+2*y6)*eta+1.0/4.0*(y1-y2-y4+y5)*xi+1.0/4.0*(y4-y1+y5-y2);

	*(J+NDOF3*0+2)=1.0/4.0*(z1-z2-z4+z5)*zi+1.0/4.0*(-z1+z2-z4+z5);
	*(J+NDOF3*1+2)=sqrt3/12.0*(z1+z2-2*z3-z4-z5+2*z6)*zi+sqrt3/12.0*(-z1-z2+2*z3-z4-z5+2*z6);
	*(J+NDOF3*2+2)=sqrt3/12.0*(z1+z2-2*z3-z4-z5+2*z6)*eta+1.0/4.0*(z1-z2-z4+z5)*xi+1.0/4.0*(-z1+z5-z2+z4);

	#ifdef _DEBUG_
	for(i=0;i<3;i++){
		for (j=0;j<3;j++){
			printf("%lf ",*(J+NDOF3*i+j));
		}
		printf("\n");
	}
	#endif

	return noerr;
}

/*--------------------------------------------------
	PentaElementGetNodalFunctionsDerivativesParams
  --------------------------------------------------*/
int PentaElementGetNodalFunctionsDerivativesParams(double* dl1dl2dl3dl4dl5dl6,double* gauss_l1l2l3l4){
	
	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. Those values vary along xi,eta,z */

	int noerr=1;
	double A1,A2,A3,z;
	double sqrt3=sqrt(3.0);
	
	A1=gauss_l1l2l3l4[0]; //first area coordinate value. In term of xi and eta: A1=(1-xi)/2-eta/(2*sqrt(3));
	A2=gauss_l1l2l3l4[1]; //second area coordinate value In term of xi and eta: A2=(1+xi)/2-eta/(2*sqrt(3));
	A3=gauss_l1l2l3l4[2]; //third area coordinate value  In term of xi and eta: A3=y/sqrt(3);
	z=gauss_l1l2l3l4[3]; //fourth vertical coordinate value. Corresponding nodal function: (1-z)/2 and (1+z)/2


	/*First nodal function derivatives. The corresponding nodal function is N=A1*(1-z)/2. Its derivatives follow*/
	*(dl1dl2dl3dl4dl5dl6+numgrids*0+0)=-1.0/2.0*(1.0-z)/2.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*1+0)=-1.0/2.0/sqrt3*(1.0-z)/2.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*2+0)=-1.0/2.0*A1;

	/*Second nodal function: The corresponding nodal function is N=A2*(1-z)/2. Its derivatives follow*/
	*(dl1dl2dl3dl4dl5dl6+numgrids*0+1)=1.0/2.0*(1.0-z)/2.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*1+1)=-1.0/2.0/sqrt3*(1.0-z)/2.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*2+1)=-1.0/2.0*A2;

	/*Third nodal function: The corresponding nodal function is N=A3*(1-z)/2. Its derivatives follow*/
	*(dl1dl2dl3dl4dl5dl6+numgrids*0+2)=0.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*1+2)=1.0/sqrt3*(1.0-z)/2.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*2+2)=-1.0/2.0*A3;

	/*Fourth nodal function: The corresponding nodal function is N=A1*(1+z)/2. Its derivatives follow*/
	*(dl1dl2dl3dl4dl5dl6+numgrids*0+3)=-1.0/2.0*(1.0+z)/2.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*1+3)=-1.0/2.0/sqrt3*(1.0+z)/2.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*2+3)=1.0/2.0*A1;

	/*Fifth nodal function: The corresponding nodal function is N=A2*(1+z)/2. Its derivatives follow*/
	*(dl1dl2dl3dl4dl5dl6+numgrids*0+4)=1.0/2.0*(1.0+z)/2.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*1+4)=-1.0/2.0/sqrt3*(1.0+z)/2.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*2+4)=1.0/2.0*A2;

	/*Sixth nodal function: The corresponding nodal function is N=A3*(1+z)/2. Its derivatives follow*/
	*(dl1dl2dl3dl4dl5dl6+numgrids*0+5)=0.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*1+5)=1.0/sqrt3*(1.0+z)/2.0;
	*(dl1dl2dl3dl4dl5dl6+numgrids*2+5)=1.0/2.0*A3;

	return noerr;
}

/*--------------------------------------------------
	PentaElementGetJacobianInvert
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementGetJacobianInvert"
int PentaElementGetJacobianInvert(double*  Jinv, double* xyz_list,double* gauss_l1l2l3l4){

	int noerr=1;
	double Jdet;
	
	/*Call Jacobian routine to get the jacobian:*/
	noerr=PentaElementGetJacobian(Jinv, xyz_list, gauss_l1l2l3l4);
	if(!noerr)goto cleanup_and_return;

	/*Invert Jacobian matrix: */
	noerr=MatrixInverse(Jinv,NDOF3,NDOF3,NULL,0,&Jdet);
		
	cleanup_and_return:
	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not compute Jacobian matrix.");
		return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	PentaElementCreatePVectorDiagnosticHoriz
  --------------------------------------------------*/

#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreatePVectorDiagnosticHoriz"

int PentaElementCreatePVectorDiagnosticHoriz( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs){

	int             i,j;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;
	Matice        * matice = NULL;
	Matpar        * matpar = NULL;

	/* output: */
	ElemVector*     pe_g        = NULL;
	
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	
	/* parameters: */
	double  rho_ice;
	double  rho_water;
	double  gravity=9.8;
	double  plastic_stress; 
	double  slope[NDOF2];
	double  driving_stress_baseline;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights      =  NULL;
	double* vert_gauss_weights      =  NULL;
	double  gauss_l1l2l3l4[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_weight;


	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double T_bg_list[numgrids][9];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3l4l5l6[6];

	/*element vector at the gaussian points: */
	double  pe_g_gaussian[numgrids*NDOF2];

	/*input parameters for structural analysis (diagnostic): */
	double* thickness_param=NULL;
	double  thickness_list[numgrids];
	double  thickness;
	double* surface_param=NULL;
	double  surface_list[numgrids];
	double* bed_param=NULL;
	double  bed_list[numgrids];
	double* melting_param=NULL;
	double  melting_list[numgrids];
	double* basal_drag_param=NULL;
	double  K_list[numgrids];

	/*Collapsed formulation: */
	Tria*  tria=NULL;
	TriaElement* triaelement=NULL;


	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;
	matice=this->matice;
	matpar=this->matpar;

	/*Figure out if this pentaelem is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the load vector. */

	if ((this->penta.collapse==1) && (this->penta.onbed==0)){
		/*This element should be collapsed, but this element is not on the bedrock, therefore all its 
		 * dofs have already been frozen! Do nothing: */
		*ppe_g=NULL;
		return noerr;
	}
	else if ((this->penta.collapse==1) && (this->penta.onbed==1)){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 *and use its CreatePVector functionality to return an elementary stiffness matrix: */
		tria=NewTria();
		TriaInit(tria);
            
		strcpy(tria->name,"CTRICE3");
		tria->eid=this->penta.eid;
		tria->mid=this->penta.mid;
		for(i=0;i<3;i++){
			tria->g[i]=this->penta.g[i]; //we take the first 3 grids of the penta to build the tria.
			tria->h[i]=this->penta.h[i]; 
			tria->s[i]=this->penta.s[i];
			tria->b[i]=this->penta.b[i];
			tria->k[i]=this->penta.k[i];
		}

		/*diverse:*/
		tria->theta_type=0;
		tria->theta.angle=0;
		tria->zoffs=0;

		//friction
		tria->p=this->penta.p;
		tria->q=this->penta.q;
		tria->shelf=this->penta.shelf;

		/*type of fitting? : */
		tria->fit=this->penta.fit;
		tria->meanvel=this->penta.meanvel;
		tria->epsvel=this->penta.epsvel;

		/*diverse:*/
		tria->acceleration=0;

		/*Now that the tria object is created, spawn an element out of it: */
		triaelement = NewTriaElement( tria);

		/*Transfer configuration of penta to configuration of tria: */
		for(i=0;i<3;i++){
			triaelement->internalgrid_indices[i]=this->internalgrid_indices[i];
			triaelement->internalgrid_indicesb[i]=this->internalgrid_indicesb[i];
			triaelement->pg[i]=this->pg[i];
		}
		triaelement->matice_index=this->matice_index;
		triaelement->matice_enum=this->matice_enum;
		triaelement->matice=this->matice;

		triaelement->matpar_index=this->matpar_index;
		triaelement->matpar_enum=this->matpar_enum;
		triaelement->matpar=this->matpar;

		/*Ok, now triaelement is correctly configured, call on its method CreateKMatrix: */
		noerr=TriaElementCreatePVectorDiagnosticHoriz( ppe_g, (void*)triaelement, inputs);

		/*Now delete tria and triaelement: */
		DeleteTriaElement((void**)&triaelement);
		DeleteTria((void**)&tria);

		return noerr;
	}
	else{

		/*Implement standard penta element: */

		/* The number of g-set dof for this element is 2*3 (vx and vy, horizontal velocities for ice, and 3 grids per element: */
		numdof=numgrids*NDOF2;

		/* Allocate new load element  vector: */
		pe_g=NewElemVector(numdof);

		/* recover material parameters: */
		gravity=matpar->g;
		rho_ice=matpar->rho_ice;
		rho_water=matpar->rho_water;

		/* recover input parameters: */
		thickness_param=ParameterInputsRecover(inputs,"thickness");
		surface_param=ParameterInputsRecover(inputs,"surface");
		bed_param=ParameterInputsRecover(inputs,"bed");
		melting_param=ParameterInputsRecover(inputs,"melting");
		basal_drag_param=ParameterInputsRecover(inputs,"drag");

		/* Get all element grid data */
		noerr *= GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], &cid_list[0], this->pg, numgrids, 1 );

		/* Build the row index vector : */
		for (i=0;i<numgrids;i++){
			/* We use the structural dof list, from which we keep only the first two dof (x for vx, and y for vy).
			 * The other degrees of freedom will be spc'd for now.*/
			structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
			for (j=0;j<NDOF2;j++){
				*(pe_g->row_indices+i*NDOF2+j)=structural_dof_list[j];
			}
			if(thickness_param){
				dof=structural_dof_list[0];
				thickness_list[i]=*(thickness_param+dof);
			}
			else{
				thickness_list[i]=this->penta.h[i];
			}

			if(surface_param){
				dof=structural_dof_list[0];
				surface_list[i]=*(surface_param+dof);
			}
			else{
				surface_list[i]=this->penta.s[i];
			}
			if(bed_param){
				dof=structural_dof_list[0];
				bed_list[i]=*(bed_param+dof);
			}
			else{
				bed_list[i]=this->penta.b[i];
			}
			if(melting_param){
				dof=structural_dof_list[0];
				melting_list[i]=*(melting_param+dof);
			}
			else{
				melting_list[i]=this->penta.b[i];
			}
			if(basal_drag_param){
				dof=structural_dof_list[0];
				K_list[i]=*(basal_drag_param+dof);
			}
			else{
				K_list[i]=this->penta.k[i];
			}
		} //for (i=0;i<numgrids;i++)

		/*Get gaussian points and weights :*/
		order_area_gauss=2;
		num_vert_gauss=3;

		GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);
		#ifdef _DEBUG_ 
		for (i=0;i<num_area_gauss;i++){
			_printf_("Area Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(area_gauss_weights+i));
		}
		for (i=0;i<num_vert_gauss;i++){
			_printf_("Vert Gauss coord %i: %lf Weight: %lf\n",i,*(fourth_gauss_vert_coord+i),*(vert_gauss_weights+i));
		}
		#endif
	
		/* Start  looping on the number of gaussian points: */
		for (ig1=0; ig1<num_area_gauss; ig1++){
			for (ig2=0; ig2<num_vert_gauss; ig2++){
			
				/*Pick up the gaussian point: */
				gauss_weight1=*(area_gauss_weights+ig1);
				gauss_weight2=*(vert_gauss_weights+ig2);
				gauss_weight=gauss_weight1*gauss_weight2;
				
				gauss_l1l2l3l4[0]=*(first_gauss_area_coord+ig1); 
				gauss_l1l2l3l4[1]=*(second_gauss_area_coord+ig1);
				gauss_l1l2l3l4[2]=*(third_gauss_area_coord+ig1);
				gauss_l1l2l3l4[3]=*(fourth_gauss_vert_coord+ig2);
		


				/*Compute thickness at gaussian point: */
				PentaElementGetParameterValue(&thickness, &thickness_list[0],gauss_l1l2l3l4);
				#ifdef _DEBUG_ 
					printf("Thickness: %lf\n", thickness);
				#endif

				PentaElementGetParameterDerivativeValue(&slope[0], &surface_list[0],&xyz_list[0][0], gauss_l1l2l3l4);
				#ifdef _DEBUG_ 
					_printf_("Element id %i Slope: [%lf;%lf]\n", PentaElementGetID(this),slope[0],slope[1]);
				#endif

				/*In case we have plastic basal drag, compute plastic stress at gaussian point from k1, k2 and k3 fields in the 
				 * element itself: */
				if((this->penta.shelf==0) && (this->penta.friction_type==1)){
					PentaElementGetParameterValue(&plastic_stress, &K_list[0],gauss_l1l2l3l4);
					#ifdef _DEBUG_ 
						_printf_("Plastic stress: %lf\n", plastic_stress);
					#endif
				}

				/* Get Jacobian determinant: */
				PentaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3l4);
				#ifdef _DEBUG_ 
				_printf_("Element id %i Jacobian determinant: %lf\n",PentaElementGetID(this),Jdet);
				#endif
		
				 /*Get nodal functions: */
				PentaElementGetNodalFunctions(l1l2l3l4l5l6, gauss_l1l2l3l4);

				/*Compute driving stress: */
				driving_stress_baseline=rho_ice*gravity;
				#ifdef _DEBUG_ 
				_printf_("Element id %i driving_stress_baseline: %lf\n",PentaElementGetID(this),driving_stress_baseline);
				#endif

				/*Build pe_g_gaussian vector: */
				if(this->penta.friction_type==1){
					for (i=0;i<numgrids;i++){
						for (j=0;j<NDOF2;j++){
							pe_g_gaussian[i*NDOF2+j]=(-driving_stress_baseline*slope[j]-plastic_stress)*Jdet*gauss_weight*l1l2l3l4l5l6[i]; 
						}
					}
				}
				else {
					for (i=0;i<numgrids;i++){
						for (j=0;j<NDOF2;j++){
							pe_g_gaussian[i*NDOF2+j]=-driving_stress_baseline*slope[j]*Jdet*gauss_weight*l1l2l3l4l5l6[i];
						}
					}
				}

				/*Add pe_g_gaussian vector to pe_g: */
				for( i=0; i<pe_g->nrows; i++){
					*(pe_g->terms+i)+=*(pe_g_gaussian+i);
				}
			} //for (ig2=0; ig2<num_vert_gauss; ig2++)
		} //for (ig1=0; ig1<num_area_gauss; ig1++)

	} //else if ((this->penta.collapse==1) && (this->penta.onbed==1))

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);

    if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build load vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
		ElemVectorEcho(pe_g);
		#endif

		/*Assign output pointer: */
		*ppe_g=pe_g;
		return 1;
	}
}

/*--------------------------------------------------
	PentaElementCreatePVectorDiagnosticBaseVert
  --------------------------------------------------*/

#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreatePVectorDiagnosticBaseVert"

int PentaElementCreatePVectorDiagnosticBaseVert( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs){

	int             i,j;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;
	Matice        * matice = NULL;
	Matpar        * matpar = NULL;

	/* output: */
	ElemVector*     pe_g        = NULL;
	
	int             numgrids2d=3;
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	
	/* parameters: */
	double  rho_ice;
	double  rho_water;
	double  di;
	double  slope[NDOF2];
	double  dbdx,dbdy;

	/* 2d gaussian point: */
	int     num_gauss2d;
	double* first_gauss_area_coord2d  =  NULL;
	double* second_gauss_area_coord2d =  NULL;
	double* third_gauss_area_coord2d  =  NULL;
	double* gauss_weights2d=NULL;
	double  gauss_l1l2l3[3];
	double  gauss_weight;
	int     ig;

	/* grid data: */
	int    cid_list[numgrids2d];
	double xyz_list[numgrids2d][3];
	double T_bg_list[numgrids2d][9];

	/* Jacobian: */
	double Jdet;

	/* matrices: */
	double L[numgrids2d];
	double HU[numgrids2d];
	double dHU[2];
	double HV[numgrids2d];
	double dHV[2];
	double divHVel;

	/*element vector at the gaussian points: */
	double  pe_g_gaussian[numgrids2d*NDOF1];

	/*input parameters for structural analysis (diagnostic): */
	double* thickness_param=NULL;
	double  thickness_list[numgrids2d];
	double  thickness;
	double* bed_param=NULL;
	double  bed_list[numgrids2d];
	double* melting_param=NULL;
	double  melting_list[numgrids2d];
	double  melting;
	double* accumulation_param=NULL;
	double  accumulation_list[numgrids2d];
	double  accumulation;
	double* velocity_param=NULL;
	double  vx_list[numgrids2d];
	double  vy_list[numgrids2d];
	double  vx,vy;
	double* velocity_average_param=NULL;
	double  vx_average_list[numgrids2d];
	double  vy_average_list[numgrids2d];


	/*Collapsed formulation: */
	Tria*  tria=NULL;
	TriaElement* triaelement=NULL;

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;
	matice=this->matice;
	matpar=this->matpar;

	
	/*Figure out if this pentaelem is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the load vector. */

	if (this->penta.onbed==0){
		*ppe_g=NULL;
		return noerr;
	}
	else{

		/*Implement standard penta element: */

		/* The number of g-set dof for this element is 3*1 (vz for 3 base grids: */
		numdof=numgrids2d*NDOF1;

		/* Allocate new load element  vector: */
		pe_g=NewElemVector(numdof);

		/* recover material parameters: */
		rho_ice=matpar->rho_ice;
		rho_water=matpar->rho_water;
		di=rho_ice/rho_water;

		/* recover input parameters: */
		thickness_param=ParameterInputsRecover(inputs,"thickness");
		bed_param=ParameterInputsRecover(inputs,"bed");
		melting_param=ParameterInputsRecover(inputs,"melting");
		accumulation_param=ParameterInputsRecover(inputs,"accumulation");
		velocity_param=ParameterInputsRecover(inputs,"velocity");
		velocity_average_param=ParameterInputsRecover(inputs,"velocity_average");

		/* Get all element grid data */
		noerr *= GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], &cid_list[0], this->pg, numgrids2d, 1 ); //just the first 3 grids

		/* Initialize row vector and parameter lists: */
		for (i=0;i<numgrids2d;i++){

			/* Build the row index vector : */
			structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
			dof=structural_dof_list[2]; //dof for vz
			*(pe_g->row_indices+i)=dof;
			
			/* Recover vx: */
			dof=structural_dof_list[0]; //dof for vx
			if(velocity_param){
				vx_list[i]=*(velocity_param+dof);
			}
			else{
				vx_list[i]=0;
			}
			if(velocity_average_param){
				vx_average_list[i]=*(velocity_average_param+dof);
			}
			else{
				vx_average_list[i]=0;
			}
			/* Recover vy: */
			dof=structural_dof_list[1]; //dof for vy
			if(velocity_param){
				vy_list[i]=*(velocity_param+dof);
			}
			else{
				vy_list[i]=0;
			}
			if(velocity_average_param){
				vy_average_list[i]=*(velocity_average_param+dof);
			}
			else{
				vy_average_list[i]=0;
			}
			
			/*Recover other parameters on 1 dof : */
			if(thickness_param){
				dof=structural_dof_list[0]; //dof for vx, where thickness is 
				thickness_list[i]=*(thickness_param+dof);
			}
			else{
				thickness_list[i]=this->penta.h[i];
			}
			if(bed_param){
				dof=structural_dof_list[0];
				bed_list[i]=*(bed_param+dof);
			}
			else{
				bed_list[i]=this->penta.b[i];
			}
			if(melting_param){
				dof=structural_dof_list[0];
				melting_list[i]=*(melting_param+dof);
			}
			else{
				melting_list[i]=this->penta.melting[i];
			}
			if(accumulation_param){
				dof=structural_dof_list[0];
				accumulation_list[i]=*(accumulation_param+dof);
			}
			else{
				accumulation_list[i]=this->penta.accumulation[i];
			}
		} //for (i=0;i<numgrids;i++)

		/*Create Tria element out of the first 3 grids of the Pentaelement: */
		tria=NewTria();
		TriaInit(tria);
            
		strcpy(tria->name,"CTRICE3");
		tria->eid=this->penta.eid;
		tria->mid=this->penta.mid;
		for(i=0;i<3;i++){
			tria->g[i]=this->penta.g[i]; //we take the first 3 grids of the penta to build the tria.
			tria->h[i]=this->penta.h[i]; 
			tria->s[i]=this->penta.s[i];
			tria->b[i]=this->penta.b[i];
			tria->k[i]=this->penta.k[i];
		}

		/*diverse:*/
		tria->theta_type=0;
		tria->theta.angle=0;
		tria->zoffs=0;

		//friction
		tria->p=this->penta.p;
		tria->q=this->penta.q;
		tria->shelf=this->penta.shelf;

		/*type of fitting? : */
		tria->fit=this->penta.fit;
		tria->meanvel=this->penta.meanvel;
		tria->epsvel=this->penta.epsvel;

		/*diverse:*/
		tria->acceleration=0;

		/*Now that the tria object is created, spawn an element out of it: */
		triaelement = NewTriaElement( tria);

		/*Transfer configuration of penta to configuration of tria: */
		for(i=0;i<3;i++){
			triaelement->internalgrid_indices[i]=this->internalgrid_indices[i];
			triaelement->internalgrid_indicesb[i]=this->internalgrid_indicesb[i];
			triaelement->pg[i]=this->pg[i];
		}
		triaelement->matice_index=this->matice_index;
		triaelement->matice_enum=this->matice_enum;
		triaelement->matice=this->matice;

		triaelement->matpar_index=this->matpar_index;
		triaelement->matpar_enum=this->matpar_enum;
		triaelement->matpar=this->matpar;


		/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
		GaussTria( &num_gauss2d, &first_gauss_area_coord2d, &second_gauss_area_coord2d, &third_gauss_area_coord2d, &gauss_weights2d, 2);

		/*For icesheets: */
		/* Start  looping on the number of gaussian points: */
		for (ig=0; ig<num_gauss2d; ig++){

			/*Pick up the gaussian point: */
			gauss_weight=*(gauss_weights2d+ig);
			gauss_l1l2l3[0]=*(first_gauss_area_coord2d+ig); 
			gauss_l1l2l3[1]=*(second_gauss_area_coord2d+ig);
			gauss_l1l2l3[2]=*(third_gauss_area_coord2d+ig);

			/*Get melting at gaussian point: */
			TriaElementGetParameterValue(&melting, &melting_list[0],gauss_l1l2l3);

			/*Get velocity at gaussian point: */
			TriaElementGetParameterValue(&vx, &vx_list[0],gauss_l1l2l3);
			TriaElementGetParameterValue(&vy, &vy_list[0],gauss_l1l2l3);

			/*Get bed slope: */
			TriaElementGetParameterDerivativeValue(&slope[0], &bed_list[0],&xyz_list[0][0], gauss_l1l2l3);
			dbdx=slope[0];
			dbdy=slope[1];

			/* Get Jacobian determinant: */
			TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
            
			//Get L matrix if viscous basal drag present:
			TriaElementGetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

			
			/*Build gaussian vector: */
			for(i=0;i<numgrids2d;i++){
				pe_g_gaussian[i]=Jdet*gauss_weight*(vx*dbdx+vy*dbdy-melting)*L[i];
			}
			
			/*Add pe_g_gaussian vector to pe_g: */
			for( i=0; i<pe_g->nrows; i++){
				*(pe_g->terms+i)+=*(pe_g_gaussian+i);
			}
		}  //for (ig=0; ig<num_gauss2d; ig++)

		/*for iceshelves, vertical velocity at the base is determined using the hydrostatic equilibrium -> we add a corrective term 
		  to the icesheet term.:*/
		if (this->penta.shelf){

			/*build the vector thickness*velocity_average:*/
			for(i=0;i<numgrids2d;i++){
				HU[i]=thickness_list[i]*vx_average_list[i];
				HV[i]=thickness_list[i]*vy_average_list[i];
			}

			for (ig=0; ig<num_gauss2d; ig++){

				/*Pick up the gaussian point: */
				gauss_weight=*(gauss_weights2d+ig);
				gauss_l1l2l3[0]=*(first_gauss_area_coord2d+ig); 
				gauss_l1l2l3[1]=*(second_gauss_area_coord2d+ig);
				gauss_l1l2l3[2]=*(third_gauss_area_coord2d+ig);

				/*Get div(H*vel): */
				TriaElementGetParameterDerivativeValue(&dHU[0], &HU[0],&xyz_list[0][0], gauss_l1l2l3);
				TriaElementGetParameterDerivativeValue(&dHV[0], &HV[0],&xyz_list[0][0], gauss_l1l2l3);
				divHVel=dHU[0]+dHV[1];

				/*Get melting and accumulation at gaussian point: */
				TriaElementGetParameterValue(&melting, &melting_list[0],gauss_l1l2l3);
				TriaElementGetParameterValue(&accumulation, &accumulation_list[0],gauss_l1l2l3);

				/* Get Jacobian determinant: */
				TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
				
				/*Get L matrix:*/
				TriaElementGetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

				/*Build gaussian vector: */
				for(i=0;i<numgrids2d;i++){
					pe_g_gaussian[i]=Jdet*gauss_weight*(-di*(-divHVel+accumulation-melting))*L[i];
				}
				
				/*Add pe_g_gaussian vector to pe_g: */
				for( i=0; i<pe_g->nrows; i++){
					*(pe_g->terms+i)+=*(pe_g_gaussian+i);
				}
			} //for (ig=0; ig<num_gauss2d; ig++)
		} //if (this->penta.shelf)

		/*Delete Tria and TriaElement objects: */
		DeleteTriaElement((void**)&triaelement);
		DeleteTria((void**)&tria);
	} //if (this->penta.onbed==0)
	
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord2d);
	xfree((void**)&second_gauss_area_coord2d);
	xfree((void**)&third_gauss_area_coord2d);
	xfree((void**)&gauss_weights2d);

    if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build load vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
		ElemVectorEcho(pe_g);
		#endif

		/*Assign output pointer: */
		*ppe_g=pe_g;
		return 1;
	}
}

/*--------------------------------------------------
	PentaElementCreatePVectorPrognostic
  --------------------------------------------------*/

#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreatePVectorPrognostic"

int PentaElementCreatePVectorPrognostic( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs){

	int             i,j;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;

	/* output: */
	ElemVector*     pe_g        = NULL;
	
	int             numgrids2d=3;
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	/* 2d gaussian point: */
	int     num_gauss2d;
	double* first_gauss_area_coord2d  =  NULL;
	double* second_gauss_area_coord2d =  NULL;
	double* third_gauss_area_coord2d  =  NULL;
	double* gauss_weights2d=NULL;
	double  gauss_l1l2l3[3];
	double  gauss_weight;
	int     ig;

	/* grid data: */
	int    cid_list[numgrids2d];
	double xyz_list[numgrids2d][3];
	double T_bg_list[numgrids2d][9];

	/* Jacobian: */
	double Jdet;

	/* matrices: */

	/*element vector at the gaussian points: */
	double  pe_g_gaussian[numgrids2d*NDOF1];

	/*input parameters for structural analysis (diagnostic): */
	double* basal_vertical_velocity_param=NULL;
	double  wb_list[numgrids2d];
	double  wb;
	double* surface_vertical_velocity_param=NULL;
	double  ws_list[numgrids2d];
	double  ws;
	double* accumulation_param=NULL;
	double  accumulation_list[numgrids2d];
	double  accumulation;
	double*  melting_param=NULL;
	double  melting_list[numgrids2d];
	double  melting;
	double*  thickness_param=NULL;
	double  thickness_list[numgrids2d];
	double  thickness;
	double* dt_param=NULL;
	double  dt;

	/*Matrices: */
	double  L[numgrids2d];


	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;

	/*Figure out if this pentaelem is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the load vector. */

	if (this->penta.onbed==0){
		*ppe_g=NULL;
		return noerr;
	}
	else{

		/*Implement standard penta element: */

		/* The number of g-set dof for this element is 3*1 (vz for 3 base grids: */
		numdof=numgrids2d*NDOF1;

		/* Allocate new load element  vector: */
		pe_g=NewElemVector(numdof);

		/* recover input parameters: */
		accumulation_param=ParameterInputsRecover(inputs,"accumulation");
		melting_param=ParameterInputsRecover(inputs,"melting");
		basal_vertical_velocity_param=ParameterInputsRecover(inputs,"basal_vertical_velocity");
		surface_vertical_velocity_param=ParameterInputsRecover(inputs,"surface_vertical_velocity");
		thickness_param=ParameterInputsRecover(inputs,"thickness");
		dt_param=ParameterInputsRecover(inputs,"dt");

		if(!accumulation_param || !melting_param || !basal_vertical_velocity_param || !surface_vertical_velocity_param || !thickness_param || !dt_param){
			_printf_("%s%s\n",__FUNCT__," error message: missing input parameters!");
			_printf_("Here is the list of pointers: \n"); 
			_printf_("accumulation: %p\n",accumulation_param);
			_printf_("melting: %p\n",melting_param);
			_printf_("basal_vertical_velocity: %p\n",basal_vertical_velocity_param);
			_printf_("surface_vertical_velocity: %p\n",surface_vertical_velocity_param);
			_printf_("thickness: %p\n",thickness_param);
			_printf_("dt: %p\n",dt_param);
			noerr=0; goto cleanup_and_return;
		}
		dt=*dt_param;

		/* Get all element grid data */
		noerr *= GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], &cid_list[0], this->pg, numgrids2d, 1 ); //just the first 3 grids

		/* Initialize row vector and parameter lists: */
		for (i=0;i<numgrids2d;i++){

			/* Build the row index vector : */
			structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
			dof=structural_dof_list[0]; //dof for thickness
			*(pe_g->row_indices+i)=dof;
			
			/* Recover parameters on 1st dof: */
			dof=structural_dof_list[0]; 
			wb_list[i]=*(basal_vertical_velocity_param+dof);
			ws_list[i]=*(surface_vertical_velocity_param+dof);
			accumulation_list[i]=*(accumulation_param+dof);
			melting_list[i]=*(melting_param+dof);
			thickness_list[i]=*(thickness_param+dof);
		} //for (i=0;i<numgrids;i++)

		/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
		GaussTria( &num_gauss2d, &first_gauss_area_coord2d, &second_gauss_area_coord2d, &third_gauss_area_coord2d, &gauss_weights2d, 2);

		/*For icesheets: */
		/* Start  looping on the number of gaussian points: */
		for (ig=0; ig<num_gauss2d; ig++){

			/*Pick up the gaussian point: */
			gauss_weight=*(gauss_weights2d+ig);
			gauss_l1l2l3[0]=*(first_gauss_area_coord2d+ig); 
			gauss_l1l2l3[1]=*(second_gauss_area_coord2d+ig);
			gauss_l1l2l3[2]=*(third_gauss_area_coord2d+ig);

			/*Get basal_vertical_velocity at gaussian point: */
			TriaElementGetParameterValue(&wb, &wb_list[0],gauss_l1l2l3);
	
			/*Get surface_vertical_velocity at gaussian point: */
			TriaElementGetParameterValue(&ws, &ws_list[0],gauss_l1l2l3);

			/*Get melting at gaussian point: */
			TriaElementGetParameterValue(&melting, &melting_list[0],gauss_l1l2l3);

			/*Get accumulation at gaussian point: */
			TriaElementGetParameterValue(&accumulation, &accumulation_list[0],gauss_l1l2l3);

			/*Get thickness at gaussian point: */
			TriaElementGetParameterValue(&thickness, &thickness_list[0],gauss_l1l2l3);

			/* Get Jacobian determinant: */
			TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
            
			/*Get L matrix : */
			TriaElementGetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);
			
			/*Build gaussian vector: */
			for(i=0;i<numgrids2d;i++){
				pe_g_gaussian[i]=Jdet*gauss_weight*(thickness+dt*(ws-wb+accumulation-melting))*L[i];
			}
			
			/*Add pe_g_gaussian vector to pe_g: */
			for( i=0; i<pe_g->nrows; i++){
				*(pe_g->terms+i)+=*(pe_g_gaussian+i);
			}
		}  //for (ig=0; ig<num_gauss2d; ig++)
	} //if (this->penta.onbed==0)
	
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord2d);
	xfree((void**)&second_gauss_area_coord2d);
	xfree((void**)&third_gauss_area_coord2d);
	xfree((void**)&gauss_weights2d);

    if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build load vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
		ElemVectorEcho(pe_g);
		#endif

		/*Assign output pointer: */
		*ppe_g=pe_g;
		return 1;
	}
}
/*--------------------------------------------------
	PentaElementCreatePVectorMelting
  --------------------------------------------------*/

#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreatePVectorMelting"

int PentaElementCreatePVectorMelting( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs){

	int             i,j;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;
	Matice        * matice = NULL;
	Matice        * matpar = NULL;

	/* output: */
	ElemVector*     pe_g        = NULL;
	
	int             numgrids2d=3;
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	
	/* parameters: */
	double  rho_ice;
	double  rho_water;
	double  di;
	double  slope[NDOF2];
	double  dbdx,dbdy;

	/* 2d gaussian point: */
	int     num_gauss2d;
	double* first_gauss_area_coord2d  =  NULL;
	double* second_gauss_area_coord2d =  NULL;
	double* third_gauss_area_coord2d  =  NULL;
	double* gauss_weights2d=NULL;
	double  gauss_l1l2l3[3];
	double  gauss_weight;
	int     ig;

	/* grid data: */
	int    cid_list[numgrids2d];
	double xyz_list[numgrids2d][3];
	double T_bg_list[numgrids2d][9];

	/* Jacobian: */
	double Jdet;

	/* matrices: */
	double L[numgrids2d];
	double HU[numgrids2d];
	double dHU[2];
	double HV[numgrids2d];
	double dHV[2];
	double divHVel;

	/*element vector at the gaussian points: */
	double  pe_g_gaussian[numgrids2d*NDOF1];

	/*input parameters for structural analysis (diagnostic): */
	double* beta_param=NULL;
	double  beta;
	double* latentheat_param=NULL;
	double  latentheat;
	double* heatcapacity_param=NULL;
	double  heatcapacity;
	double* meltingpoint_param=NULL;
	double  meltingpoint;
	double* pressure_param=NULL;
	double  pressure_list[numgrids2d];
	double* temperature_param=NULL;
	double  temperature_list[numgrids2d];
	double* dt_param=NULL;
	double  dt;
	double* penalty_param=NULL;
	double  penalty;

	double  Tpmp_list[numgrids2d];

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;

	/*Figure out if this pentaelem is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the load vector. */

	if (this->penta.onbed==0){
		*ppe_g=NULL;
		return noerr;
	}
	else{

		/*Implement standard penta element: */

		/* The number of g-set dof for this element is 3*1 (vz for 3 base grids: */
		numdof=numgrids2d*NDOF1;

		/* Allocate new load element  vector: */
		pe_g=NewElemVector(numdof);

		/* recover input parameters: */
		beta_param=ParameterInputsRecover(inputs,"beta");
		latentheat_param=ParameterInputsRecover(inputs,"latentheat");
		heatcapacity_param=ParameterInputsRecover(inputs,"heatcapacity");
		meltingpoint_param=ParameterInputsRecover(inputs,"meltingpoint");
		pressure_param=ParameterInputsRecover(inputs,"pressure");
		temperature_param=ParameterInputsRecover(inputs,"temperature");
		dt_param=ParameterInputsRecover(inputs,"dt");
		penalty_param=ParameterInputsRecover(inputs,"penalty");
	
		if(!beta_param || !latentheat_param || !heatcapacity_param || !meltingpoint_param || !pressure_param || !temperature_param || !dt_param || !penalty_param){
			_printf_("%s%s\n",__FUNCT__," error message: missing input parameters!");
			_printf_("Here is the list of pointers: \n"); 
			_printf_("beta: %p\n",beta_param);
			_printf_("latentheat: %p\n",latentheat_param);
			_printf_("heatcapacity: %p\n",heatcapacity_param);
			_printf_("meltingpoint: %p\n",meltingpoint_param);
			_printf_("pressure: %p\n",pressure_param);
			_printf_("temperature: %p\n",temperature_param);
			_printf_("dt: %p\n",dt_param);
			_printf_("penalty: %p\n",penalty_param);
			noerr=0; goto cleanup_and_return;
		}
		beta=*beta_param;
		latentheat=*latentheat_param;
		heatcapacity=*heatcapacity_param;
		meltingpoint=*meltingpoint_param;
		dt=*dt_param;
		penalty=*penalty_param;

		/* Get all element grid data */
		noerr *= GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], &cid_list[0], this->pg, numgrids2d, 1 ); //just the first 3 grids

		/* Initialize row vector and parameter lists: */
		for (i=0;i<numgrids2d;i++){

			/* Build the row index vector : */
			structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
			dof=structural_dof_list[0]; //melting
			*(pe_g->row_indices+i)=dof;
			
			/*Recover other parameters on 1 dof : */
			temperature_list[i]=*(temperature_param+dof);
			pressure_list[i]=*(pressure_param+dof);
		} //for (i=0;i<numgrids2d;i++)

		//compute Tpmp for the 3 grids :*/
		for(i=0;i<numgrids2d;i++){
			Tpmp_list[i]=meltingpoint-beta*pressure_list[i];
		}

		/*Build gaussian vector: */
		for(i=0;i<numgrids2d;i++){
			if(temperature_list[i]<=Tpmp_list[i]){
				pe_g_gaussian[i]=0;
			}
			else{
				pe_g_gaussian[i]=heatcapacity/latentheat*penalty*(temperature_list[i]-Tpmp_list[i]);
			}
		}
		
		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<pe_g->nrows; i++){
			*(pe_g->terms+i)+=*(pe_g_gaussian+i);
		}
	} //if (this->penta.onbed==0)
	
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord2d);
	xfree((void**)&second_gauss_area_coord2d);
	xfree((void**)&third_gauss_area_coord2d);
	xfree((void**)&gauss_weights2d);

    if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build load vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
		ElemVectorEcho(pe_g);
		#endif

		/*Assign output pointer: */
		*ppe_g=pe_g;
		return 1;
	}
}
/*--------------------------------------------------
	PentaElementCreatePVectorDiagnosticVert
  --------------------------------------------------*/

#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreatePVectorDiagnosticVert"

int PentaElementCreatePVectorDiagnosticVert( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs){

	int             i,j;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;

	/* output: */
	ElemVector*     pe_g        = NULL;
	
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights      =  NULL;
	double* vert_gauss_weights      =  NULL;
	double  gauss_l1l2l3l4[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_weight;


	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double T_bg_list[numgrids][9];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3l4l5l6[6];

	/*element vector at the gaussian points: */
	double  pe_g_gaussian[numgrids];

	/*input parameters for structural analysis (diagnostic): */
	double* velocity=NULL;
	double vx_list[numgrids];
	double vy_list[numgrids];
	double du[3];
	double dv[3];
	double dudx,dvdy;
	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;

	
	/* The number of g-set dof for this element is 2*3 (vx and vy, horizontal velocities for ice, and 3 grids per element: */
	numdof=numgrids*NDOF1;

	/* Allocate new load element  vector: */
	pe_g=NewElemVector(numdof);

	/* recover input parameters: */
	velocity=ParameterInputsRecover(inputs,"velocity");

	/* Get all element grid data */
	noerr *= GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], &cid_list[0], this->pg, numgrids, 1 );

	/* Build the row index vector : */
	for (i=0;i<numgrids;i++){
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		
		//recover dofs for the load vector
		dof=structural_dof_list[2]; //vz dof
		*(pe_g->row_indices+i)=dof;
		
		//recover vx
		dof=structural_dof_list[0]; //vx dof
		if(velocity){
			vx_list[i]=*(velocity+dof);
		}
		else{
			vx_list[i]=0;
		}
		//recover vy
		dof=structural_dof_list[1]; //vy dof
		if(velocity){
			vy_list[i]=*(velocity+dof);
		}
		else{
			vy_list[i]=0;
		}
	} //for (i=0;i<numgrids;i++)

	/*Get gaussian points and weights :*/
	order_area_gauss=2;
	num_vert_gauss=2;

	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);
	#ifdef _DEBUG_ 
	for (i=0;i<num_area_gauss;i++){
		_printf_("Area Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(area_gauss_weights+i));
	}
	for (i=0;i<num_vert_gauss;i++){
		_printf_("Vert Gauss coord %i: %lf Weight: %lf\n",i,*(fourth_gauss_vert_coord+i),*(vert_gauss_weights+i));
	}
	#endif

	/* Start  looping on the number of gaussian points: */
	for (ig1=0; ig1<num_area_gauss; ig1++){
		for (ig2=0; ig2<num_vert_gauss; ig2++){
		
			/*Pick up the gaussian point: */
			gauss_weight1=*(area_gauss_weights+ig1);
			gauss_weight2=*(vert_gauss_weights+ig2);
			gauss_weight=gauss_weight1*gauss_weight2;
			
			gauss_l1l2l3l4[0]=*(first_gauss_area_coord+ig1); 
			gauss_l1l2l3l4[1]=*(second_gauss_area_coord+ig1);
			gauss_l1l2l3l4[2]=*(third_gauss_area_coord+ig1);
			gauss_l1l2l3l4[3]=*(fourth_gauss_vert_coord+ig2);
	
			/*Get velocity derivative, with respect to x and y: */
			PentaElementGetParameterDerivativeValue(&du[0], &vx_list[0],&xyz_list[0][0], gauss_l1l2l3l4);
			PentaElementGetParameterDerivativeValue(&dv[0], &vy_list[0],&xyz_list[0][0], gauss_l1l2l3l4);
			dudx=du[0];
			dvdy=dv[1];
			

			/* Get Jacobian determinant: */
			PentaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3l4);
			#ifdef _DEBUG_ 
			_printf_("Element id %i Jacobian determinant: %lf\n",PentaElementGetID(this),Jdet);
			#endif
		
			 /*Get nodal functions: */
			PentaElementGetNodalFunctions(l1l2l3l4l5l6, gauss_l1l2l3l4);

			/*Build pe_g_gaussian vector: */
			for (i=0;i<numgrids;i++){
				pe_g_gaussian[i]=(dudx+dvdy)*Jdet*gauss_weight*l1l2l3l4l5l6[i];
			}

			/*Add pe_g_gaussian vector to pe_g: */
			for( i=0; i<pe_g->nrows; i++){
				*(pe_g->terms+i)+=*(pe_g_gaussian+i);
			}
			
		} //for (ig2=0; ig2<num_vert_gauss; ig2++)
	} //for (ig1=0; ig1<num_area_gauss; ig1++)

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);

    if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build load vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
		ElemVectorEcho(pe_g);
		#endif

		/*Assign output pointer: */
		*ppe_g=pe_g;
		return 1;
	}
}


/*--------------------------------------------------
	PentaElementCreatePVectorThermal
  --------------------------------------------------*/

#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreatePVectorThermal"

int PentaElementCreatePVectorThermal( ElemVector* *ppe_g, void* vpthis, ParameterInputs* inputs){

	int             i,j;
	int             noerr=1;

	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;
	Matice        * matice = NULL;
	Matpar        * matpar = NULL;

	/* output: */
	ElemVector*     pe_g        = NULL;
	
	int             numdof;
	int*            structural_dof_list   =  NULL;
	int             dof;

	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights      =  NULL;
	double* vert_gauss_weights      =  NULL;
	double  gauss_l1l2l3l4[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_weight;

	/* 2d gaussian point: */
	int     num_gauss2d;
	double* first_gauss_area_coord2d  =  NULL;
	double* second_gauss_area_coord2d =  NULL;
	double* third_gauss_area_coord2d  =  NULL;
	double* gauss_weights2d=NULL;
	double  gauss_l1l2l3[3];

	/* grid data: */
	int    cid_list[numgrids];
	double xyz_list[numgrids][3];
	double T_bg_list[numgrids][9];

	/* Jacobian: */
	double Jdet;

	/*element vector at the gaussian points: */
	double  pe_g_gaussian_deformational[numgrids];
	double  pe_g_gaussian_transient[numgrids];
	double  pe_geo_gaussian[3];

	/*input parameters for thermal analysis: */
	double* velocity=NULL;
	double vxvyvz_list[numgrids][3];
	double vxvy_list[numgrids][2];
	double* flow_law_param=NULL;
	double  B_list[numgrids];
	double  B_param;
	double* basal_drag=NULL;
	double  K_list[numgrids];
	double* thickness_param=NULL;
	double thickness_list[numgrids];
	double thickness;
	double* temperature_param=NULL;
	double temperature_list[numgrids];
	double temperature;
	double* surface_param=NULL;
	double surface_list[numgrids];
	double* bed_param=NULL;
	double bed_list[numgrids];
	double* dt_param=NULL;
	double dt;
	double* geothermalflux_param=NULL;
	double  geothermalflux_list[numgrids];
	double  geothermalflux;

	/* material parameters: */
	double rho_water;
	double rho_ice;
	double gravity;
	double heatcapacity;
	double viscosity;

	/* strain rate: */
	double epsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double epsilon_matrix[6];
	double epsilon_e;
	double phi;
	double l1l2l3l4l5l6[numgrids];
	double l1l2l3[3];
	double alpha2_list[3];
	double vel_base_squared[3];
	double basal_friction_list[3];
	double basal_friction;
	double G_g;

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;
	matice=this->matice;
	matpar=this->matpar;

	/*Recover material: */
	gravity=matpar->g;
	rho_ice=matpar->rho_ice;
	rho_water=matpar->rho_water;
	heatcapacity=matpar->heatcapacity;

	/* The number of g-set dof for this element is 6*1 (temperature on all grids):*/
	numdof=numgrids*NDOF1;

	/* Allocate new load element  vector: */
	pe_g=NewElemVector(numdof);

	/* Get all element grid data */
	noerr *= GetElementGridData( &xyz_list[0][0], &T_bg_list[0][0], &cid_list[0], this->pg, numgrids, 1 );

	/* recover input parameters: */
	velocity=ParameterInputsRecover(inputs,"velocity"); 
	flow_law_param=ParameterInputsRecover(inputs,"B");
	basal_drag=ParameterInputsRecover(inputs,"drag");
	thickness_param=ParameterInputsRecover(inputs,"thickness");
	temperature_param=ParameterInputsRecover(inputs,"temperature");
	surface_param=ParameterInputsRecover(inputs,"surface");
	bed_param=ParameterInputsRecover(inputs,"bed");
	dt_param=ParameterInputsRecover(inputs,"dt");
	geothermalflux_param=ParameterInputsRecover(inputs,"geothermalflux");

	if((!velocity) || (!dt_param) ){
		_printf_("%s%s\n",__FUNCT__," error message: missing velocity or dt or temperature parameter!");
		noerr=0; goto cleanup_and_return;
	}
	dt=*dt_param;

	if(this->penta.thermal_steadystate==0){
		if(!temperature_param){
			_printf_("%s%s\n",__FUNCT__," error message: missing temperature for trnasient analysis!");
		}
	}

	/* Build the row index vector : */
	for (i=0;i<numgrids;i++){
		structural_dof_list= InternalGridGetStructuralDoflistPtr( this->pg[i]);
		
		//recover dofs for the load vector
		dof=structural_dof_list[0]; //temperature dof
		*(pe_g->row_indices+i)=dof;
	
		//recover vxvyvz_list: */
		for(j=0;j<NDOF3;j++){
			dof=structural_dof_list[j]; 
			if(velocity){
				vxvyvz_list[i][j]=*(velocity+dof);
			}
			else{
				vxvyvz_list[i][j]=0;
			}
		}
		//recover parameters on first dof 
		dof=structural_dof_list[0]; //temperature dof
		if(flow_law_param){
			B_list[i] = *(flow_law_param+dof);
		}
		if(basal_drag){
			K_list[i]=*(basal_drag+dof);
		}
		else{
			K_list[i]=this->penta.k[i];
		}
		if(temperature_param){
			temperature_list[i]=*(temperature_param+dof);
		}
		if(thickness_param){
			thickness_list[i]=*(thickness_param+dof);
		}
		else{
			thickness_list[i]=this->penta.h[i];
		}
		if(surface_param){
			surface_list[i]= *(surface_param+dof);
		}
		else{
			surface_list[i]= this->penta.s[i];
		}
		if(bed_param){
			bed_list[i]=*(bed_param+dof);
		}
		else{
			bed_list[i]= this->penta.b[i];
		}
		if(geothermalflux_param){
			geothermalflux_list[i]=*(geothermalflux_param+dof);
		}
		else{
			geothermalflux_list[i]= this->penta.geothermalflux[i];
		}
	} //for (i=0;i<numgrids;i++)

	/*Fill in vxvy_list for all grids: */
	for(i=0;i<numgrids;i++){
		for(j=0;j<2;j++){
			vxvy_list[i][j]=vxvyvz_list[i][j];
		}
	}

	/*Get gaussian points and weights :*/
	order_area_gauss=2;
	num_vert_gauss=2;

	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);
	#ifdef _DEBUG_ 
	for (i=0;i<num_area_gauss;i++){
		_printf_("Area Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(area_gauss_weights+i));
	}
	for (i=0;i<num_vert_gauss;i++){
		_printf_("Vert Gauss coord %i: %lf Weight: %lf\n",i,*(fourth_gauss_vert_coord+i),*(vert_gauss_weights+i));
	}
	#endif

	/* Start  looping on the number of gaussian points: */
	for (ig1=0; ig1<num_area_gauss; ig1++){
		for (ig2=0; ig2<num_vert_gauss; ig2++){
		
			/*Pick up the gaussian point: */
			gauss_weight1=*(area_gauss_weights+ig1);
			gauss_weight2=*(vert_gauss_weights+ig2);
			gauss_weight=gauss_weight1*gauss_weight2;
			
			gauss_l1l2l3l4[0]=*(first_gauss_area_coord+ig1); 
			gauss_l1l2l3l4[1]=*(second_gauss_area_coord+ig1);
			gauss_l1l2l3l4[2]=*(third_gauss_area_coord+ig1);
			gauss_l1l2l3l4[3]=*(fourth_gauss_vert_coord+ig2);

			//Update material with temperature:
			if(temperature_param){
				PentaElementGetParameterValue(&temperature, &temperature_list[0],gauss_l1l2l3l4);
				B_param=Paterson(temperature);
				MaticeSetFlowLawParameter(this->matice,B_param);
			}

			/*Build deformational heating: */
			PentaElementGetStrainRate(&epsilon[0],&vxvy_list[0][0],&xyz_list[0][0],gauss_l1l2l3l4);
			MaticeGetViscosity3d(&viscosity, this->matice, &epsilon[0]);

			//Build the  strain rate matrix: */
			epsilon_matrix[0]=epsilon[0];
			epsilon_matrix[1]=epsilon[1];
			epsilon_matrix[2]=epsilon[2];
			epsilon_matrix[3]=epsilon[3];
			epsilon_matrix[4]=epsilon[4];
			epsilon_matrix[5]=-epsilon[0]-epsilon[1]; //incompressibility!

			epsilon_e=effective_value(epsilon_matrix,6,1);

			/*compute source term: */
			phi=2*viscosity*pow(epsilon_e,2);

			/* Get Jacobian determinant: */
			PentaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3l4);
			#ifdef _DEBUG_ 
			_printf_("Element id %i Jacobian determinant: %lf\n",PentaElementGetID(this),Jdet);
			#endif
	
			/*Get nodal functions: */
			PentaElementGetNodalFunctions(l1l2l3l4l5l6, gauss_l1l2l3l4);
			
			/*Build deformation pe_g_gaussian vector: */
			if(this->penta.thermal_steadystate==1){
				for (i=0;i<numgrids;i++){
					pe_g_gaussian_deformational[i]=phi/(rho_ice*heatcapacity)*Jdet*gauss_weight*l1l2l3l4l5l6[i];
				}
			}
			else{
				for (i=0;i<numgrids;i++){
					pe_g_gaussian_deformational[i]=dt*phi/(rho_ice*heatcapacity)*Jdet*gauss_weight*l1l2l3l4l5l6[i];
				}
			}

			/*deal with transient: */
			if(this->penta.thermal_steadystate==1){
				for (i=0;i<numgrids;i++){
					pe_g_gaussian_transient[i]=0;
				}
			}
			else{
				for (i=0;i<numgrids;i++){
					pe_g_gaussian_transient[i]=temperature*Jdet*gauss_weight*l1l2l3l4l5l6[i];
				}
			}

			/*Add pe_g_gaussian vectors to pe_g: */
			for( i=0; i<pe_g->nrows; i++){
				*(pe_g->terms+i)+= *(pe_g_gaussian_deformational+i) +  *(pe_g_gaussian_transient+i);
			}
			
		} //for (ig2=0; ig2<num_vert_gauss; ig2++)
	} //for (ig1=0; ig1<num_area_gauss; ig1++)

	
	/*Geothermal flux on ice sheet base and basal friction G+tau_b.ub:*/
	if((this->penta.onbed) && (this->penta.shelf==0)){
		
		//Get the basal friction tau_b.u_b=alpha2*ub^2
		if (this->penta.friction_type==2){

			/*Allocate friction object: */
			Friction* friction=NewFriction();
					
			/*Initialize all fields: */
			friction->element_type=xmalloc((strlen("2d")+1)*sizeof(char));
			strcpy(friction->element_type,"2d");

			friction->gravity=gravity;
			friction->rho_ice=rho_ice;
			friction->rho_water=rho_water;
			friction->K=&K_list[0];
			friction->bed=&bed_list[0];
			friction->thickness=&thickness_list[0];
			friction->velocities=&vxvy_list[0][0];
			friction->p=this->penta.p;
			friction->q=this->penta.q;

			/*Compute alpha2_list: */
			FrictionGetAlpha2(&alpha2_list[0],friction);

			/*Erase friction object: */
			DeleteFriction(&friction);

			/*Get velocity magnitude and compute basal friction: */
			for(i=0;i<3;i++){
				vel_base_squared[i]=pow(vxvy_list[i][0],2)+pow(vxvy_list[i][1],2);
				basal_friction_list[i]=alpha2_list[i]*vel_base_squared[i];
			}
		} //if (this->penta.friction_type==2)

		/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
		GaussTria( &num_gauss2d, &first_gauss_area_coord2d, &second_gauss_area_coord2d, &third_gauss_area_coord2d, &gauss_weights2d, 2);

		/* Start  looping on the number of gaussian points: */
		for (ig=0; ig<num_gauss2d; ig++){
			/*Pick up the gaussian point: */
			gauss_weight=*(gauss_weights2d+ig);
			gauss_l1l2l3[0]=*(first_gauss_area_coord2d+ig); 
			gauss_l1l2l3[1]=*(second_gauss_area_coord2d+ig);
			gauss_l1l2l3[2]=*(third_gauss_area_coord2d+ig);

			/* Get Jacobian determinant: */
			TriaElementGetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
				
			/*Get geothermal flux and basal friction :*/
			TriaElementGetParameterValue(&G_g, &geothermalflux_list[0],gauss_l1l2l3);
			TriaElementGetParameterValue(&basal_friction, &basal_friction_list[0],gauss_l1l2l3);
		
			//Get nodal functions value
			TriaElementGetNodalFunctions(&l1l2l3[0], gauss_l1l2l3);

			/*Compute the elementary load vector: */
			if(this->penta.thermal_steadystate){
				for(i=0;i<3;i++){
					pe_geo_gaussian[i]=gauss_weight*Jdet*(basal_friction+G_g)/(heatcapacity*rho_ice)*l1l2l3[i];
				}
			}
			else{
				for(i=0;i<3;i++){
					pe_geo_gaussian[i]=dt*gauss_weight*Jdet*(basal_friction+G_g)/(heatcapacity*rho_ice)*l1l2l3[i];
				}
			} //if(this->penta.thermal_steadystate)

			/*Add gauss load to global load vector: */
			for( i=0; i<3; i++){ //only the first 3 grids.
				*(pe_g->terms+i)+= pe_geo_gaussian[i];
			}

		} //for (ig=0; ig<num_gauss2d; ig++)
	} //if((this->penta.onbed) && (this->penta.shelf==0))

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
	xfree((void**)&first_gauss_area_coord2d);
	xfree((void**)&second_gauss_area_coord2d);
	xfree((void**)&third_gauss_area_coord2d);
	xfree((void**)&gauss_weights2d);

    if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not build load vector.");
		return 0;
	}
	else{
		#ifdef _DEBUG_
		ElemVectorEcho(pe_g);
		#endif

		/*Assign output pointer: */
		*ppe_g=pe_g;
		return 1;
	}
}
/*--------------------------------------------------
	PentaElementGetParameterDerivativeValue
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementGetParameterDerivativeValue"
int PentaElementGetParameterDerivativeValue(double* p, double* p_list,double* xyz_list, double* gauss_l1l2l3l4){
				
	/*From grid values of parameter p (p_list[0], p_list[1], p_list[2], p_list[3], p_list[4] and p_list[4]), return parameter derivative value at gaussian point specified by gauss_l1l2l3l4:
	 *   dp/dx=p_list[0]*dh1/dx+p_list[1]*dh2/dx+p_list[2]*dh3/dx+p_list[3]*dh4/dx+p_list[4]*dh5/dx+p_list[5]*dh6/dx;
	 *   dp/dy=p_list[0]*dh1/dy+p_list[1]*dh2/dy+p_list[2]*dh3/dy+p_list[3]*dh4/dy+p_list[4]*dh5/dy+p_list[5]*dh6/dy;
	 *   dp/dz=p_list[0]*dh1/dz+p_list[1]*dh2/dz+p_list[2]*dh3/dz+p_list[3]*dh4/dz+p_list[4]*dh5/dz+p_list[5]*dh6/dz;
	 *
	 *   p is a vector of size 3x1 already allocated.
	 */
	
	int noerr=1;
	double dh1dh2dh3dh4dh5dh6_basic[NDOF3][numgrids];

	/*Get dh1dh2dh3dh4dh5dh6_basic in basic coordinate system: */
	noerr=PentaElementGetNodalFunctionsDerivativesBasic(&dh1dh2dh3dh4dh5dh6_basic[0][0],xyz_list, gauss_l1l2l3l4);
	if(!noerr)goto cleanup_and_return;
	
	*(p+0)=p_list[0]*dh1dh2dh3dh4dh5dh6_basic[0][0]+p_list[1]*dh1dh2dh3dh4dh5dh6_basic[0][1]+p_list[2]*dh1dh2dh3dh4dh5dh6_basic[0][2]+p_list[3]*dh1dh2dh3dh4dh5dh6_basic[0][3]+p_list[4]*dh1dh2dh3dh4dh5dh6_basic[0][4]+p_list[5]*dh1dh2dh3dh4dh5dh6_basic[0][5];
;
	*(p+1)=p_list[0]*dh1dh2dh3dh4dh5dh6_basic[1][0]+p_list[1]*dh1dh2dh3dh4dh5dh6_basic[1][1]+p_list[2]*dh1dh2dh3dh4dh5dh6_basic[1][2]+p_list[3]*dh1dh2dh3dh4dh5dh6_basic[1][3]+p_list[4]*dh1dh2dh3dh4dh5dh6_basic[1][4]+p_list[5]*dh1dh2dh3dh4dh5dh6_basic[1][5];

	*(p+2)=p_list[0]*dh1dh2dh3dh4dh5dh6_basic[2][0]+p_list[1]*dh1dh2dh3dh4dh5dh6_basic[2][1]+p_list[2]*dh1dh2dh3dh4dh5dh6_basic[2][2]+p_list[3]*dh1dh2dh3dh4dh5dh6_basic[2][3]+p_list[4]*dh1dh2dh3dh4dh5dh6_basic[2][4]+p_list[5]*dh1dh2dh3dh4dh5dh6_basic[2][5];

	cleanup_and_return:
	if(!noerr){
		_printf_("%s%s\n",__FUNCT__," error message: could not return parameter derivative vector.");
		return 0;
	}
	else{
		return 1;
	}
}

/*--------------------------------------------------
	PentaElementCreateDuVector
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreateDuVector"
int PentaElementCreateDuVector( ElemVector* *pdue_g, void* vpthis, double* velocity, double* obs_velocity, ParameterInputs* inputs, int analysis_type,int sub_analysis_type){

	int noerr=1;
	int i;
	
	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;

	/*surface tria element: */
	Tria*  tria=NULL;
	TriaElement* triaelement=NULL;

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;
	
	/*Bail out if this element does not touch the surface: */
	if (!this->penta.onsurface){
		*pdue_g=NULL;
	}
	else{

		/*use the TriaElementCreateDuVector routine, on a TriaElement which is made of the 3 upper grids at the surface of the ice sheet. : */
		tria=NewTria();
		TriaInit(tria);
            
		strcpy(tria->name,"CTRICE3");
		tria->eid=this->penta.eid;
		tria->mid=this->penta.mid;
		for(i=3;i<6;i++){
			tria->g[i-3]=this->penta.g[i]; //we take the last 3 grids of the penta to build the tria.
			tria->h[i-3]=this->penta.h[i]; 
			tria->s[i-3]=this->penta.s[i];
			tria->b[i-3]=this->penta.b[i];
			tria->k[i-3]=this->penta.k[i];
		}

		/*diverse:*/
		tria->theta_type=0;
		tria->theta.angle=0;
		tria->zoffs=0;

		//friction
		tria->p=this->penta.p;
		tria->q=this->penta.q;
		tria->shelf=this->penta.shelf;
		tria->friction_type=this->penta.friction_type;

		/*type of fitting? : */
		tria->fit=this->penta.fit;
		tria->meanvel=this->penta.meanvel;
		tria->epsvel=this->penta.epsvel;

		/*diverse:*/
		tria->acceleration=0;

		/*Now that the tria object is created, spawn an element out of it: */
		triaelement = NewTriaElement( tria);

		/*Transfer configuration of penta to configuration of tria: */
		for(i=3;i<6;i++){
			triaelement->internalgrid_indices[i-3]=this->internalgrid_indices[i];
			triaelement->internalgrid_indicesb[i-3]=this->internalgrid_indicesb[i];
			triaelement->pg[i-3]=this->pg[i];
		}
		
		triaelement->matice_index=this->matice_index;
		triaelement->matice_enum=this->matice_enum;
		triaelement->matice=this->matice;

		triaelement->matpar_index=this->matpar_index;
		triaelement->matpar_enum=this->matpar_enum;
		triaelement->matpar=this->matpar;


		/*Ok, now triaelement is correctly configured, call on its method CreateDuVector: */
		noerr=TriaElementCreateDuVector( pdue_g, (void*)triaelement, velocity, obs_velocity, inputs, analysis_type,sub_analysis_type);

		/*Now delete tria and triaelement: */
		DeleteTriaElement((void**)&triaelement);
		DeleteTria((void**)&tria);

	}
	return noerr;
}

/*--------------------------------------------------
	PentaElementCreateGradjVectors
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreateGradVectors"
int PentaElementCreateGradjVectors( ElemVector* *pgradje_g, void* vpthis, double* velocity, double* adjoint, ParameterInputs* inputs,int analysis_type,int sub_analysis_type,char* control_type){

	int noerr=1;
	
	if strcasecmp_eq(control_type,"drag"){
		noerr=PentaElementCreateGradjVectorsDrag( pgradje_g, vpthis, velocity, adjoint, inputs,analysis_type,sub_analysis_type);
	}
	else if strcasecmp_eq(control_type,"B"){
		noerr=PentaElementCreateGradjVectorsB( pgradje_g, vpthis, velocity, adjoint, inputs,analysis_type,sub_analysis_type);
	}
	else{
		printf("%s%s\n",__FUNCT__," error message: control type not yet supported!");
		noerr=0;
	}
	return noerr;
}

/*--------------------------------------------------
	PentaElementMisfit
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementMisfit"
int PentaElementMisfit( double* pJelem,void* vpthis, double* velocity, double* obs_velocity, ParameterInputs* inputs, int analysis_type,int sub_analysis_type){

	int noerr=1;
	int i;
	
	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;

	/*surface tria element: */
	Tria*  tria=NULL;
	TriaElement* triaelement=NULL;

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;
	
	/*Bail out if this element does not touch the surface: */
	if (!this->penta.onsurface){
		*pJelem=0;
	}
	else{

		/*use the TriaElementCreateDuVector routine, on a TriaElement which is made of the 3 upper grids at the surface of the ice sheet. : */
		tria=NewTria();
		TriaInit(tria);
            
		strcpy(tria->name,"CTRICE3");
		tria->eid=this->penta.eid;
		tria->mid=this->penta.mid;
		for(i=3;i<6;i++){
			tria->g[i-3]=this->penta.g[i]; //we take the last 3 grids of the penta to build the tria.
			tria->h[i-3]=this->penta.h[i]; 
			tria->s[i-3]=this->penta.s[i];
			tria->b[i-3]=this->penta.b[i];
			tria->k[i-3]=this->penta.k[i];
		}

		/*diverse:*/
		tria->theta_type=0;
		tria->theta.angle=0;
		tria->zoffs=0;

		//friction
		tria->p=this->penta.p;
		tria->q=this->penta.q;
		tria->shelf=this->penta.shelf;
		tria->friction_type=this->penta.friction_type;

		/*type of fitting? : */
		tria->fit=this->penta.fit;
		tria->meanvel=this->penta.meanvel;
		tria->epsvel=this->penta.epsvel;

		/*diverse:*/
		tria->acceleration=0;

		/*Now that the tria object is created, spawn an element out of it: */
		triaelement = NewTriaElement( tria);

		/*Transfer configuration of penta to configuration of tria: */
		for(i=3;i<6;i++){
			triaelement->internalgrid_indices[i-3]=this->internalgrid_indices[i];
			triaelement->internalgrid_indicesb[i-3]=this->internalgrid_indicesb[i];
			triaelement->pg[i-3]=this->pg[i];
		}
		
		triaelement->matice_index=this->matice_index;
		triaelement->matice_enum=this->matice_enum;
		triaelement->matice=this->matice;

		triaelement->matpar_index=this->matpar_index;
		triaelement->matpar_enum=this->matpar_enum;
		triaelement->matpar=this->matpar;


		/*Ok, now triaelement is correctly configured, call on its method Misfit: */
		noerr=TriaElementMisfit(pJelem,(void*)triaelement, velocity, obs_velocity, inputs, analysis_type,sub_analysis_type);

		/*Now delete tria and triaelement: */
		DeleteTriaElement((void**)&triaelement);
		DeleteTria((void**)&tria);

	}
	return noerr;
}

/*--------------------------------------------------
	PentaElementCreateGradjVectorsDrag
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreateGradVectorsDrag"
int PentaElementCreateGradjVectorsDrag( ElemVector* *pgradje_g, void* vpthis, double* velocity, double* adjoint, ParameterInputs* inputs,int analysis_type,int sub_analysis_type){

	int noerr=1;
	int i;
	
	/* vpthis for polymorphic function compatibility */
	PentaElement* this   = NULL;

	/*surface tria element: */
	Tria*  tria=NULL;
	TriaElement* triaelement=NULL;

	/*Some pointer intialization: */
	this = (PentaElement*)vpthis;
	
	/*Bail out if this element does not touch the bedrock: */
	if (!this->penta.onbed){
		*pgradje_g=NULL;
	}
	else{

		/*use the TriaElementCreateDuVector routine, on a TriaElement which is made of the 3 lower grids at the bedrock interface with the ice sheet: */
		tria=NewTria();
		TriaInit(tria);
            
		strcpy(tria->name,"CTRICE3");
		tria->eid=this->penta.eid;
		tria->mid=this->penta.mid;
		for(i=0;i<3;i++){
			tria->g[i]=this->penta.g[i]; //we take the first 3 grids of the penta to build the tria.
			tria->h[i]=this->penta.h[i]; 
			tria->s[i]=this->penta.s[i];
			tria->b[i]=this->penta.b[i];
			tria->k[i]=this->penta.k[i];
		}

		/*diverse:*/
		tria->theta_type=0;
		tria->theta.angle=0;
		tria->zoffs=0;

		//friction
		tria->p=this->penta.p;
		tria->q=this->penta.q;
		tria->shelf=this->penta.shelf;
		tria->friction_type=this->penta.friction_type;

		/*type of fitting? : */
		tria->fit=this->penta.fit;
		tria->meanvel=this->penta.meanvel;
		tria->epsvel=this->penta.epsvel;

		/*diverse:*/
		tria->acceleration=0;

		/*Now that the tria object is created, spawn an element out of it: */
		triaelement = NewTriaElement( tria);

		/*Transfer configuration of penta to configuration of tria: */
		for(i=0;i<3;i++){
			triaelement->internalgrid_indices[i]=this->internalgrid_indices[i];
			triaelement->internalgrid_indicesb[i]=this->internalgrid_indicesb[i];
			triaelement->pg[i]=this->pg[i];
		}
		
		triaelement->matice_index=this->matice_index;
		triaelement->matice_enum=this->matice_enum;
		triaelement->matice=this->matice;

		triaelement->matpar_index=this->matpar_index;
		triaelement->matpar_enum=this->matpar_enum;
		triaelement->matpar=this->matpar;


		/*Ok, now triaelement is correctly configured, call on its method CreateGradjVectorsdrag: */
		noerr=TriaElementCreateGradjVectorsDrag( pgradje_g, (void*)triaelement,velocity, adjoint, inputs,analysis_type,sub_analysis_type);

		/*Now delete tria and triaelement: */
		DeleteTriaElement((void**)&triaelement);
		DeleteTria((void**)&tria);

	}
	return noerr;
}


/*--------------------------------------------------
	PentaElementCreateGradjVectorsB
  --------------------------------------------------*/
#undef __FUNCT__ 
#define __FUNCT__ "PentaElementCreateGradjVectorsB"
int PentaElementCreateGradjVectorsB( ElemVector* *pgradje_g, void* vpthis, double* velocity, double* adjoint, ParameterInputs* inputs,int analysis_type,int sub_analysis_type){

	printf("Not supported yet!\n");
	return 0;
}




#undef NDOF1 
#undef NDOF2 
#undef NDOF3 
#undef numgrids 
