/*!\file Patch.c
 * \brief: implementation of the Patch object
 */

/*Include files: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "./objects.h"
#include "../Container/Container.h"
#include "../io/io.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../include/include.h"
/*}}}*/

/*Object constructors and destructors:*/
/*FUNCTION Patch::Patch() default constructor {{{1*/
Patch::Patch(){
	this->numrows=0;
	this->numcols=0;
	this->maxvertices=0;
	this->maxnodes=0;
	this->values=NULL;
}
/*}}}*/
/*FUNCTION Patch::Patch(int numrows, int maxvertices, int maxnodes){{{1*/
Patch::Patch(int in_numrows, int in_maxvertices, int in_maxnodes){

	int i,j;
	this->numrows=in_numrows;
	this->maxvertices=in_maxvertices;
	this->maxnodes=in_maxnodes;
    
	this->numcols=1  //enum_type
		+1           //step
		+1           //time 
		+1           //element id
		+1           //interpolation type
		+maxvertices //vertices
		+maxnodes;   //nodes

	//Allocate values and fill with NaN:
	if (this->numcols*this->numrows==0){
		this->values=NULL;
	}
	else{
		this->values=(double*)xmalloc(this->numcols*this->numrows*sizeof(double));
		for(i=0;i<this->numrows;i++){
			for(j=0;j<this->numcols;j++){
				this->values[i*this->numcols+j]=NAN;
			}
		}
	}

}
/*}}}*/
/*FUNCTION Patch::~Patch(){{{1*/
Patch::~Patch(){
	xfree((void**)&values);
}
/*}}}*/

/*Object methods*/
/*FUNCTION Patch::fillelementinfo{{{1*/
void Patch::fillelementinfo(int count, int element_id, int* vertices_ids, int num_vertices){

	int i;
	double* row=NULL;

	/*point to the start of the row: */
	row=this->values+count*this->numcols;

	/*Let's remember what is on a row: 
	 enum_type step time element_id interpolation vertices_ids nodal_values
	 */
	row[3]=element_id;
	for(i=0;i<num_vertices;i++){
		row[5+i]=vertices_ids[i];
	}

}
/*}}}*/
/*FUNCTION Patch::fillresultinfo{{{1*/
void Patch::fillresultinfo(int count,int enum_type,int step, double time, int interpolation, double* nodal_values, int num_nodes){

	int i;
	double* row=NULL;

	/*point to the start of the row: */
	row=this->values+count*this->numcols;

	/*Let's remember what is on a row: 
	 enum_type step time element_id interpolation vertices_ids nodal_values
	 */
	row[0]=enum_type;
	row[1]=(double)step;
	row[2]=time;
	row[4]=interpolation;
	for(i=0;i<num_nodes;i++){
		row[5+this->maxvertices+i]=nodal_values[i];
	}

}
/*}}}*/
/*FUNCTION Patch::MPI_Gather{{{1*/
void Patch::MPI_Gather(void){

	int         i;
	int         count;
	extern int  my_rank;
	extern int  num_procs;
	int         total_numrows;
	int         node_numrows;
	double     *total_values  = NULL;
	MPI_Status  status;

	#ifdef _SERIAL_
	return; //nothing to be done
	#endif
	
	
	/*First, figure out total number of rows combining all the cpus: */
	MPI_Reduce(&this->numrows,&total_numrows,1,MPI_INT,MPI_SUM,0,MPI_COMM_WORLD );
	MPI_Bcast(&total_numrows,1,MPI_INT,0,MPI_COMM_WORLD);

	/*Now, allocate buffer to holds all the values, on node 0: */
	if(my_rank==0)total_values=(double*)xmalloc(this->numcols*total_numrows*sizeof(double));

	/*Start by copying node 0 values onto total_values: */
	if(my_rank==0){
		count=0;
		memcpy(total_values+count,this->values,this->numcols*this->numrows*sizeof(double));
		count+=this->numrows*this->numcols;
	}

	/*Now, ask other nodes to send their values: */
	for (i=1;i<num_procs;i++){
		if (my_rank==i){ 
			MPI_Send(&this->numrows,1,MPI_INT,0,1,MPI_COMM_WORLD);   
			if (this->numrows)MPI_Send(this->values,this->numrows*this->numcols,MPI_DOUBLE,0,1,MPI_COMM_WORLD); 
		}
		if (my_rank==0){
			MPI_Recv(&node_numrows,1,MPI_INT,i,1,MPI_COMM_WORLD,&status); 
			if (node_numrows)MPI_Recv(total_values+count,node_numrows*this->numcols,MPI_DOUBLE,i,1,MPI_COMM_WORLD,&status);
			count+=node_numrows*this->numcols;
		}
	}	

	/*Now, node 0 has total_values, of size total_numrows*this->numcols. Update the fields in the patch, to reflect this new 
	 * reality. For other cpus, no point in keeping their data anymore: */
	if(my_rank==0){
		this->numrows=total_numrows;
		xfree((void**)&this->values);
		this->values=total_values;
	}
	else{
		this->numrows=0;
		xfree((void**)&this->values);
	}
}/*}}}*/
