function femmodel=control_core(femmodel)
%CONTROL_CORE - compute the core inversion
%
%   Usage:
%      femmodel=control_core(femmodel);
%

	%Preprocess models
	femmodel=stokescontrolinit(femmodel);

	%recover parameters common to all solutions
	verbose=femmodel.parameters.Verbose;
	nsteps=femmodel.parameters.NSteps;
	control_type=femmodel.parameters.ControlType;
	fit=femmodel.parameters.Fit;
	optscal=femmodel.parameters.OptScal;
	maxiter=femmodel.parameters.MaxIter;
	cm_jump=femmodel.parameters.CmJump;
	eps_cm=femmodel.parameters.EpsCm;
	tolx=femmodel.parameters.TolX;
	cm_min=femmodel.parameters.CmMin;
	cm_max=femmodel.parameters.CmMax;
	cm_gradient=femmodel.parameters.CmGradient;
	control_steady=femmodel.parameters.ControlSteady;

	%Initialise options with tolerance and maxiter
	options.TolX=femmodel.parameters.TolX;
	options.MaxIter=femmodel.parameters.MaxIter;

	%Initialize misfits with a vector of zeros
	J=zeros(nsteps,1);

	for n=1:nsteps,

		disp(sprintf('\n%s%s%s%s\n',['   control method step ' num2str(n) '/' num2str(femmodel.parameters.NSteps)]));
		[femmodel.elements,femmodel.loads]=InputUpdateFromConstant(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,fit(n),FitEnum);

		%In case we are running a steady state control method, compute new temperature field using new parameter distribution: 
		if control_steady;
			femmodel=steadystate_core(femmodel);
		end

		displaystring(verbose,'\n%s',['      computing gradJ...']);
		femmodel=gradient_core(femmodel);

		%Return gradient if asked
		if cm_gradient,
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,GradientEnum);
			return;
		end

		displaystring(verbose,'\n%s',['      optimizing along gradient direction...']);
		[search_scalar J(n)]=ControlOptimization('objectivefunctionC',0,1,options,femmodel,n,femmodel.parameters);

		displaystring('\n%s',['      updating parameter using optimized search scalar:']);
		scalar=search_scalar*optscal(n);
		femmodel.elements=InputAXPY(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,control_type,scalar,ControlParameterEnum);

		displaystring('\n%s',['      constraning the new distribution...']);
		[femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters]=InputControlConstrain(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,control_type,cm_min,cm_max);

		displaystring('\n%s',['      save new parameter...']);
		femmodel.elements=InputDuplicate(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,control_type,ControlParameterEnum);

		disp(['      value of misfit J after optimization #' num2str(n) ':' num2str(J(n))]);

		%Has convergence been reached?
		converged=controlconvergence(J,fit,eps_cm,n);
		if converged,
			break;
		end

	end

	%generate output
	displaystring(verbose,'\n%s',['      preparing final velocity solution...']);
	if control_steady,
		femmodel=steadystate_core(femmodel);
	else
		femmodel=diagnostic_core(femmodel);
	end

	%Some results not computed by diagnostic or steadystate
	femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,VxEnum);
	femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,VyEnum);
	femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,VelEnum);
	femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,GradientEnum);
	femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parametersn,AdjointxEnum);
	femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parametersn,AdjointyEnum);
	if (dim==3) femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,VzEnum);
	femmodel.results.JEnum=J;
	femmodel.results.ControlTypeEnum=EnumAsString(control_type);

end %end function
