/*
 * \file Nodes.c
 * \brief: implementation of the Nodes class, derived from DataSet class
 */

/*Headers: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./DataSet.h"
#include "../shared/shared.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"

using namespace std;
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION Nodes::Nodes(){{{1*/
Nodes::Nodes(){
	return;
}
/*}}}*/
/*FUNCTION Nodes::Nodes(int in_enum){{{1*/
Nodes::Nodes(int in_enum): DataSet(in_enum){
	//do nothing;
	return;
}
/*}}}*/
/*FUNCTION Nodes::~Nodes(){{{1*/
Nodes::~Nodes(){
	return;
}
/*}}}*/

/*Numerics*/
/*FUNCTION Nodes::DistributeDofs{{{1*/
void  Nodes::DistributeDofs(int numberofobjects,int numberofdofsperobject,int analysis_type){

	extern int num_procs;
	extern int my_rank;

	int  i;
	
	int  dofcount=0;
	int* alldofcount=NULL;
	int* truedofs=NULL;
	int* alltruedofs=NULL;

	/*Go through objects, and distribute dofs locally, from 0 to numberofdofsperobject: */
	for (i=0;i<this->Size();i++){
		Node* node=(Node*)this->GetObjectByOffset(i);

		/*Check that this node corresponds to our analysis currently being carried out: */
		if (node->InAnalysis(analysis_type)){
			node->DistributeDofs(&dofcount);
		}
	}

	/*Ok, now every object has distributed dofs, but locally, and with a dof count starting from 
	 *0. This means the dofs between all the cpus are not synchronized! We need to synchronize all 
	 *dof on all cpus, and use those to update the dofs of every object: */

	alldofcount=(int*)xmalloc(num_procs*sizeof(int));
	MPI_Gather(&dofcount,1,MPI_INT,alldofcount,1,MPI_INT,0,MPI_COMM_WORLD);
	MPI_Bcast(alldofcount,num_procs,MPI_INT,0,MPI_COMM_WORLD);

	/*Ok, now every cpu should start its own dof count at the end of the dofcount 
	 * from cpu-1. : */
	dofcount=0;
	if(my_rank==0){
		dofcount=0;
	}
	else{
		for(i=0;i<my_rank;i++){
			dofcount+=alldofcount[i];
		}
	}


	/*Ok, now every cpu knows where his dofs should start. Update the dof count: */
	for (i=0;i<this->Size();i++){

		/*Check that this node corresponds to our analysis currently being carried out: */
		Node* node=(Node*)this->GetObjectByOffset(i);
		if (node->InAnalysis(analysis_type)){
			node->OffsetDofs(dofcount);
		}

	}

	/*Finally, remember that cpus may have skipped some objects, because they were clones. For every 
	 * object that is not a clone, tell them to show their dofs, so that later on, they can get picked 
	 * up by their clones: */
	truedofs=(int*)xcalloc(numberofobjects*numberofdofsperobject,sizeof(int));
	alltruedofs=(int*)xcalloc(numberofobjects*numberofdofsperobject,sizeof(int));

	for (i=0;i<this->Size();i++){
		/*Check that this node corresponds to our analysis currently being carried out: */
		Node* node=(Node*)this->GetObjectByOffset(i);
		if (node->InAnalysis(analysis_type)){
			node->ShowTrueDofs(truedofs);
		}
	}
	
	MPI_Allreduce ( (void*)truedofs,(void*)alltruedofs,numberofobjects*numberofdofsperobject,MPI_INT,MPI_MAX,MPI_COMM_WORLD);

	/*Ok, now every cpu knows the true dofs of everyone else that is not a clone. Let the clones recover those true dofs: */
	for (i=0;i<this->Size();i++){
		/*Check that this node corresponds to our analysis currently being carried out: */
		Node* node=(Node*)this->GetObjectByOffset(i);
		if (node->InAnalysis(analysis_type)){
			node->UpdateCloneDofs(alltruedofs);
		}
	}

	/* Free ressources: */
	xfree((void**)&alldofcount);
	xfree((void**)&truedofs);
	xfree((void**)&alltruedofs);

}
/*}}}*/
/*FUNCTION Nodes::FlagClones{{{1*/
void  Nodes::FlagClones(int numberofobjects,int analysis_type){

	int i;
	extern int num_procs;

	int* ranks=NULL;
	int* minranks=NULL;

	/*Allocate ranks: */
	ranks=(int*)xmalloc(numberofobjects*sizeof(int));
	minranks=(int*)xmalloc(numberofobjects*sizeof(int));

	for(i=0;i<numberofobjects;i++)ranks[i]=num_procs; //no cpu can have rank num_procs. This is the maximum limit.

	/*Now go through all our objects and ask them to report to who they belong (which rank): */
	Ranks(ranks,analysis_type);

	/*We need to take the minimum rank for each vertex, and every cpu needs to get that result. That way, 
	 * when we start building the dof list for all vertexs, a cpu can check whether its vertex already has been 
	 * dealt with by another cpu. We take the minimum because we are going to manage dof assignment in increasing 
	 * order of cpu rank. This is also why we initialized this array to num_procs.*/
	MPI_Allreduce ( (void*)ranks,(void*)minranks,numberofobjects,MPI_INT,MPI_MIN,MPI_COMM_WORLD);

	/*Now go through all objects, and use minranks to flag which objects are cloned: */
	for(i=0;i<this->Size();i++){

		Node* node=(Node*)this->GetObjectByOffset(i);

		/*Check that this node corresponds to our analysis currently being carried out: */
		if (node->InAnalysis(analysis_type)){

			/*For this object, decide whether it is a clone: */
			node->SetClone(minranks);
		}
	}

	/*Free ressources: */
	xfree((void**)&ranks); 
	xfree((void**)&minranks);

}
/*}}}*/
/*FUNCTION Nodes::FlagNodeSets{{{1*/
void Nodes::FlagNodeSets(Vec pv_g, Vec pv_m, Vec pv_n, Vec pv_f, Vec pv_s,int analysis_type){

	int i;

	for(i=0;i<this->Size();i++){

		Node* node=(Node*)this->GetObjectByOffset(i);
			
		/*Check that this node corresponds to our analysis currently being carried out: */
		if (node->InAnalysis(analysis_type)){

			/*Plug set values intp our 4 set vectors: */
			node->CreateVecSets(pv_g,pv_m,pv_n,pv_f,pv_s);
		}
	}

	/*Assemble: */
	VecAssemblyBegin(pv_g);
	VecAssemblyEnd(pv_g);

	VecAssemblyBegin(pv_m);
	VecAssemblyEnd(pv_m);

	VecAssemblyBegin(pv_n);
	VecAssemblyEnd(pv_n);

	VecAssemblyBegin(pv_f);
	VecAssemblyEnd(pv_f);

	VecAssemblyBegin(pv_s);
	VecAssemblyEnd(pv_s);

}
/*}}}*/
/*FUNCTION Nodes::NumberOfDofs(int analysis_type){{{1*/
int   Nodes::NumberOfDofs(int analysis_type){

	int i;
	
	int   numdofs=0;
	int   allnumdofs;

	/*Now go through all nodes, and get how many dofs they own, unless they are clone nodes: */
	for(i=0;i<this->Size();i++){

		Node* node=(Node*)this->GetObjectByOffset(i);

		/*Check that this node corresponds to our analysis currently being carried out: */
		if (node->InAnalysis(analysis_type)){

			/*Ok, this object is a node, ask it to plug values into partition: */
			if (!node->IsClone()){

				numdofs+=node->GetNumberOfDofs();

			}
		}
	}

	/*Gather from all cpus: */
	MPI_Allreduce ( (void*)&numdofs,(void*)&allnumdofs,1,MPI_INT,MPI_SUM,MPI_COMM_WORLD);

	return allnumdofs;
}
/*}}}*/
/*FUNCTION Nodes::NumberOfNodes(){{{1*/
int Nodes::NumberOfNodes(void){

	int i;

	int max_sid=0;
	int sid;
	int node_max_sid;

	for(i=0;i<this->Size();i++){

		Node* node=(Node*)this->GetObjectByOffset(i);
		sid=node->Sid();
		if (sid>max_sid)max_sid=sid;
	}

	#ifdef _PARALLEL_
	MPI_Reduce (&max_sid,&node_max_sid,1,MPI_INT,MPI_MAX,0,MPI_COMM_WORLD );
	MPI_Bcast(&node_max_sid,1,MPI_INT,0,MPI_COMM_WORLD);
	max_sid=node_max_sid;
	#endif 

	/*sid starts at 0*/
	if (max_sid) max_sid++;

	/*return*/
	return max_sid;
}
/*}}}*/
/*FUNCTION Nodes::NumberOfNodes(analysis){{{1*/
int Nodes::NumberOfNodes(int analysis_type){

	int i;

	int max_sid=0;
	int sid;
	int node_max_sid;

	for(i=0;i<this->Size();i++){

		Node* node=(Node*)this->GetObjectByOffset(i);

		/*Check that this node corresponds to our analysis currently being carried out: */
		if (node->InAnalysis(analysis_type)){

			sid=node->Sid();
			if (sid>max_sid)max_sid=sid;
		}
	}

#ifdef _PARALLEL_
	MPI_Reduce (&max_sid,&node_max_sid,1,MPI_INT,MPI_MAX,0,MPI_COMM_WORLD );
	MPI_Bcast(&node_max_sid,1,MPI_INT,0,MPI_COMM_WORLD);
	max_sid=node_max_sid;
#endif 

	/*sid starts at 0*/
	max_sid++;

	/*return*/
	return max_sid;
}
/*}}}*/
/*FUNCTION Nodes::Ranks{{{1*/
void   Nodes::Ranks(int* ranks,int analysis_type){

	/*Go through nodes, and for each object, report it cpu: */

	int i;
	int rank;
	int sid;
	
	for(i=0;i<this->Size();i++){

		Node* node=(Node*)this->GetObjectByOffset(i);

		/*Check that this node corresponds to our analysis currently being carried out: */
		if (node->InAnalysis(analysis_type)){

			rank=node->MyRank();
			sid=node->Sid();

			/*Plug rank into ranks, according to id: */
			ranks[sid]=rank; 
		}
	}
	return;
}
/*}}}*/
